! f90 include file

!_______________________________________________________________________
!
   subroutine prtmt( nrow, ncol, a, ja, ia, rhs, guesol, title, key,    &
                     type, ifmt, job, iounit )

      integer, intent(in) :: nrow, ncol, ja(:), ia(:)
      real(kind=MF_DOUBLE), intent(in) :: a(:), rhs(:)
      character(len=2), intent(in) :: guesol
      character(len=72), intent(in) :: title
      character(len=8), intent(in) :: key
      character(len=3), intent(in) :: type
      integer, intent(in) :: job, iounit
      integer :: ifmt ! is modified
      !------ API end ------

!-----------------------------------------------------------------------
! Writes a matrix in Harwell-Boeing format into a file.
! assumes that the matrix is stored in COMPRESSED SPARSE COLUMN FORMAT.
! some limited functionality for right hand sides.
! Author: Youcef Saad - Date: Sept., 1989 - updated Oct. 31, 1989 to
! cope with new format.
! f90 modifications by É. Canot -- IPR/CNRS
!-----------------------------------------------------------------------
! on entry:
!---------
! nrow  = number of rows in matrix
! ncol  = number of columns in matrix
! a    = real*8 array containing the values of the matrix stored
!          columnwise
! ja   = integer array of the same length as a containing the column
!          indexes of the corresponding matrix elements of array a.
! ia   = integer array of containing the pointers to the beginning of
!        the row in arrays a and ja.
! rhs  = real*8 array containing the right-hand-side (s) and optionally
!        the associated initial guesses and/or exact solutions
!        in this order. See also guesol for details. the vector rhs will
!        be used only if job > 2 (see below). Only full storage for
!        the right hand sides is supported.
!
! guesol = a 2-character string indicating whether an initial guess
!          (1-st character) and / or the exact solution (2-nd)
!          character) is provided with the right hand side.
!     if the first character of guesol is 'G' it means that an
!          an intial guess is provided for each right-hand sides.
!          These are assumed to be appended to the right hand-sides in
!          the array rhs.
!     if the second character of guesol is 'X' it means that an
!          exact solution is provided for each right-hand side.
!          These are assumed to be appended to the right hand-sides
!          and the initial guesses (if any) in the array rhs.
!
! title  = character*72 = title of matrix test
! key    = character*8  = key of matrix
! type   = character*3  = type of matrix
!
! ifmt   = integer specifying the format chosen for the real values
!     to be output (i.e., for a, and for rhs-guess-sol if
!          applicable). The meaning of ifmt is as follows.
!    * if (ifmt < 100) then the D descriptor is used,
!           format Dd.m, in which the length (m) of the mantissa is
!           precisely the integer ifmt (and d = ifmt+7)  ->double
!    * if (ifmt > 100) then prtmt will use the
!           F- descriptor (format Fd.m) in which the length of the
!           mantissa (m) is the integer mod(ifmt,100) and the length
!           of the integer part is k=ifmt/100 (and d = k+m+2)
!      Thus  ifmt= 4   means  D10.4  +.xxxxD+ee    while
!            ifmt=104  means  F7.4   +x.xxxx
!            ifmt=205  means  F9.5   +xx.xxxxx
!      Note: formats for ja, and ia are internally computed.
!
! job  = integer to indicate whether matrix values and
!     a right-hand-side is available to be written
!          job = 1   write structure only, i.e., the arrays ja and ia.
!          job = 2   write matrix including values, i.e., a, ja, ia
!          job = 3   write matrix and one right hand side: a,ja,ia,rhs.
!     job = nrhs+2 write matrix and nrhs successive right hand sides
!     Note that there cannot be any right-hand-side if the matrix
!     has no values. Also the initial guess and exact solutions when
!     provided are for each right hand side. For example if nrhs=2
!     and guesol='GX' there are 6 vectors to write.
!
! iounit = logical unit number where to write the matrix into.
!
! on return:
!----------
! the matrix a, ja, ia will be written in output unit iounit
! in the Harwell-Boeing format. None of the inputs is modified.
!
! Notes: 1) This code attempts to pack as many elements as possible per
!        80-character line.
!        2) this code attempts to avoid as much as possible to put
!        blanks in the formats that are written in the 4-line header
!   (This is done for purely esthetical reasons since blanks
!        are ignored in format descriptors.)
!        3) sparse formats for right hand sides and guesses are not
!        supported.
!-----------------------------------------------------------------------

      character(len=16) :: ptrfmt, indfmt
      character(len=20) :: valfmt
      character(len=3) :: rhstyp
      integer :: totcrd, ptrcrd, indcrd, valcrd, rhscrd, nnz, nrhs,     &
                 len, nperli, nrwindx, ihead, i, next, iend

   !------ end of declarations -- execution starts hereafter  ------

      !--------------
      ! compute pointer format
      !--------------
      nnz = ia(ncol+1) -1
      if( nnz == 0 ) then
         return
      end if
      len = int( log10(0.1+real(nnz+1)) ) + 1
      nperli = 80/len
      ptrcrd = ncol/nperli + 1
      write(ptrfmt,"('(',i0,'I',i0,')')") nperli, len
      !----------------------------
      ! compute ROW index format
      !----------------------------
      len = int( log10(0.1+real(nrow)) ) + 1
      nperli = min0(80/len,nnz)
      indcrd = (nnz-1)/nperli+1
      write(indfmt,"('(',i0,'I',i0,')')") nperli, len
      !---------------
      ! compute values and rhs format (using the same for both)
      !---------------
      valcrd = 0
      rhscrd  = 0
      ! quit this part if no values provided.
      if( job <= 1 ) goto 20

      if( ifmt >= 100 ) then
         ihead = ifmt/100
         ifmt = ifmt-100*ihead
         len = ihead+ifmt+2
         nperli = 80/len
         write(valfmt,"('(',i0,'F',i0,'.',i0,')')") nperli, len, ifmt
      else
         len = ifmt + 7 ! for double precision
         nperli = 80/len
         write(valfmt,"('(',i0,'D',i0,'.',i0,')')") nperli, len, ifmt
      end if
      valcrd = (nnz-1)/nperli+1
      nrhs   = job -2
      if( nrhs >= 1 ) then
         i = (nrhs*nrow-1)/nperli+1
         rhscrd = i
         if( guesol(1:1) == "G" .or. guesol(1:1) == "g" ) rhscrd = rhscrd+i
         if( guesol(2:2) == "X" .or. guesol(2:2) == "x" ) rhscrd = rhscrd+i
         rhstyp = "F"//guesol
      end if
 20   continue

      totcrd = ptrcrd+indcrd+valcrd+rhscrd
      ! write 4-line or five line header
      write(iounit,"(a72,a8,/,5i14,/,a3,11x,4i14,/,2a16,2a20)")         &
            title, key, totcrd, ptrcrd, indcrd, valcrd,rhscrd, type,    &
            nrow, ncol, nnz, nrhs, ptrfmt, indfmt, valfmt, valfmt

      nrwindx = 0
      if( nrhs >= 1 ) then
         write(iounit,"(A3,11x,i14,i14)") rhstyp, nrhs, nrwindx
      end if

      write(iounit,ptrfmt) (ia (i), i = 1, ncol+1)
      write(iounit,indfmt) (ja (i), i = 1, nnz)
      if( job <= 1 ) return
      write(iounit,valfmt) (a(i), i = 1, nnz)
      if( job <= 2 ) return
      len = nrow*nrhs
      next = 1
      iend = len
      write(iounit,valfmt) (rhs(i), i = next, iend)

      ! write initial guesses if available

      if( guesol(1:1) == 'G' .or. guesol(1:1) == 'g' ) then
         next = next+len
         iend = iend+ len
         write(iounit,valfmt) (rhs(i), i = next, iend)
      end if

      ! write exact solutions if available

      if( guesol(2:2) == 'X' .or. guesol(2:2) == 'x' ) then
         next = next+len
         iend = iend+ len
         write(iounit,valfmt) (rhs(i), i = next, iend)
      end if

   end subroutine prtmt
