! f90 include file

!_______________________________________________________________________
!
   subroutine coo2csc( ncol, nnz, a, ir, jc, ao, iao, jao )

      integer,              intent(in)  :: ncol, nnz
      real(kind=MF_DOUBLE), intent(in)  :: a(:)
      integer,              intent(in)  :: ir(:), jc(:)
      real(kind=MF_DOUBLE), intent(out) :: ao(:)
      integer,              intent(out) :: iao(:), jao(:)
      !------ API end ------

      !-----------------------------------------------------------------
      !  Coordinate  to  Compressed Sparse Column
      ! (adapted from SPARSKIT2 (Mar 8 2005) 'coocsr' by É. Canot)
      ! August 4, 2006
      !-----------------------------------------------------------------
      ! Converts a matrix that is stored in coordinate format into
      ! a CSC format.
      ! No check for values of ir and jc!
      ! No check for size of ao and iao!
      !-----------------------------------------------------------------
      ! On entry:
      !---------
      ! ncol = dimension of the matrix
      ! nnz  = number of nonzero elements in matrix
      ! a,
      ! ir,
      ! jc   = real matrix in coordinate format.
      !        a(k), ir(k), jc(k) store the
      !        nnz nonzero elements of the matrix with :
      !          a(k)  = actual real value of the elements,
      !          ir(k) = its row number,
      !          jc(k) = its column  number.
      !        The order of the elements is arbitrary.
      !
      ! On return:
      !-----------
      ! ao, iao, jao = matrix in general sparse matrix format with ao
      ! containing the real values, iao containing the row indexes,
      ! and jao being the pointer to the beginning of the column,
      ! in arrays ao, iao.
      !
      ! The resulting matrix is not necessarily row sorted; it depends
      ! on the order of the entries!
      !
      ! Entries containing a null value are discarded.
      ! Warning : non-zero duplicated entries are not processed (i.e.
      !           they are left duplicated!).
      !-----------------------------------------------------------------

      integer :: j, jad, k0, k

   !------ End of declarations -- execution starts hereafter  ------

      do k = 1, ncol + 1
         jao(k) = 0
      end do
      ! Determine col-lengths.
      do k = 1, nnz
         if( a(k) /= 0.0d0 ) then
            jao(jc(k)) = jao(jc(k)) + 1
         end if
      end do
      ! Starting position of each col.
      k = 1
      do j = 1, ncol + 1
         k0 = jao(j)
         jao(j) = k
         k = k + k0
      end do
      ! Go through the structure once more. Fill in output matrix.
      do k = 1, nnz
         j = jc(k)
         jad = jao(j)
         if( a(k) /= 0.0d0 ) then
            ao(jad) = a(k)
            iao(jad) = ir(k)
            jao(j) = jad + 1
         end if
      end do
      ! Shift back jao
      do j = ncol, 1, -1
         jao(j+1) = jao(j)
      end do
      jao(1) = 1

   end subroutine coo2csc
!_______________________________________________________________________
!
   subroutine coo_symm2csc( ncol, nnz,                                  &
                            a, ir, jc, ao, iao, jao )

      integer,              intent(in)  :: ncol, nnz
      real(kind=MF_DOUBLE), intent(in)  :: a(:)
      integer,              intent(in)  :: ir(:), jc(:)
      real(kind=MF_DOUBLE), intent(out) :: ao(:)
      integer,              intent(out) :: iao(:), jao(:)
      !------ API end ------

      !-----------------------------------------------------------------
      !  Coordinate (with pattern)  to  Compressed Sparse Column
      !-----------------------------------------------------------------
      ! Converts a matrix having a "symmetric" pattern and whose either
      ! the lower or the upper part is stored in coordinate format into
      ! a CSC format.
      ! No check for values of ir and jc!
      ! No check for size of ao and iao!
      !-----------------------------------------------------------------
      ! On entry:
      !---------
      ! ncol = dimension of the matrix
      ! nnz  = number of nonzero elements in matrix
      ! a,
      ! ir,
      ! jc   = real matrix in coordinate format.
      !        a(k), ir(k), jc(k) store the
      !        nnz nonzero elements of the matrix with :
      !          a(k)  = actual real value of the elements,
      !          ir(k) = its row number,
      !          jc(k) = its column  number.
      !        The order of the elements is arbitrary.
      !
      ! On return:
      !-----------
      ! ao, iao, jao = matrix in general sparse matrix format with ao
      ! containing the real values, iao containing the row indexes,
      ! and jao being the pointer to the beginning of the column,
      ! in arrays ao, iao.
      !
      ! The resulting matrix is not necessarily row sorted; it depends
      ! on the order of the entries!
      !
      ! Entries containing a null value are discarded.
      ! Warning : non-zero duplicated entries are not processed (i.e.
      !           they are left duplicated!).
      !-----------------------------------------------------------------

      integer :: j, jad, k0, k, ik, jk

   !------ End of declarations -- execution starts hereafter  ------

      do k = 1, ncol + 1
         jao(k) = 0
      end do
      ! Determine col-lengths.
      do k = 1, nnz
         if( a(k) /= 0.0d0 ) then
            j = jc(k)
            jao(j) = jao(j) + 1
            if( ir(k) /= j ) then
               ! Out-of-diag element
               j = ir(k)
               jao(j) = jao(j) + 1
            end if
         end if
      end do
      ! Starting position of each col.
      k = 1
      do j = 1, ncol + 1
         k0 = jao(j)
         jao(j) = k
         k = k + k0
      end do
      ! Go through the structure once more. Fill in output matrix.
      do k = 1, nnz
         ik = ir(k)
         jk = jc(k)
         j = jk
         jad = jao(j)
         if( a(k) /= 0.0d0 ) then
            ao(jad) = a(k)
            iao(jad) = ik
            jao(j) = jad + 1
            ! If out-of-diag element, add it again
            if( ik /= jk ) then
               j = ik
               jad = jao(j)
               ao(jad) = a(k)
               iao(jad) = jk
               jao(j) = jad + 1
            end if
         end if
      end do
      ! Shift back jao
      do j = ncol, 1, -1
         jao(j+1) = jao(j)
      end do
      jao(1) = 1

   end subroutine coo_symm2csc
!_______________________________________________________________________
!
   subroutine coo_skew2csc( ncol, nnz,                                  &
                            a, ir, jc, ao, iao, jao )

      integer,              intent(in)  :: ncol, nnz
      real(kind=MF_DOUBLE), intent(in)  :: a(:)
      integer,              intent(in)  :: ir(:), jc(:)
      real(kind=MF_DOUBLE), intent(out) :: ao(:)
      integer,              intent(out) :: iao(:), jao(:)
      !------ API end ------

      !-----------------------------------------------------------------
      !  Coordinate (with pattern)  to  Compressed Sparse Column
      !-----------------------------------------------------------------
      ! Converts a matrix having a "skew-symmetric" pattern and whose
      ! either the lower or the upper part is stored in coordinate
      ! format into a CSC format.
      ! No check for values of ir and jc!
      ! No check for size of ao and iao!
      !-----------------------------------------------------------------
      ! On entry:
      !---------
      ! ncol = dimension of the matrix
      ! nnz  = number of nonzero elements in matrix
      ! a,
      ! ir,
      ! jc   = real matrix in coordinate format.
      !        a(k), ir(k), jc(k) store the
      !        nnz nonzero elements of the matrix with :
      !          a(k)  = actual real value of the elements,
      !          ir(k) = its row number,
      !          jc(k) = its column  number.
      !        The order of the elements is arbitrary.
      !
      ! On return:
      !-----------
      ! ao, iao, jao = matrix in general sparse matrix format with ao
      ! containing the real values, iao containing the row indexes,
      ! and jao being the pointer to the beginning of the column,
      ! in arrays ao, iao.
      !
      ! The resulting matrix is not necessarily row sorted; it depends
      ! on the order of the entries!
      !
      ! Entries containing a null value are discarded.
      ! Warning : non-zero duplicated entries are not processed (i.e.
      !           they are left duplicated!).
      !-----------------------------------------------------------------

      integer :: j, jad, k0, k, ik, jk

   !------ End of declarations -- execution starts hereafter  ------

      do k = 1, ncol + 1
         jao(k) = 0
      end do
      ! Determine col-lengths.
      do k = 1, nnz
         if( a(k) /= 0.0d0 ) then
            j = jc(k)
            jao(j) = jao(j) + 1
            ! Always out-of-diag element, add it again
            j = ir(k)
            jao(j) = jao(j) + 1
         end if
      end do
      ! Starting position of each col.
      k = 1
      do j = 1, ncol + 1
         k0 = jao(j)
         jao(j) = k
         k = k + k0
      end do
      ! Go through the structure once more. Fill in output matrix.
      do k = 1, nnz
         ik = ir(k)
         jk = jc(k)
         j = jk
         jad = jao(j)
         if( a(k) /= 0.0d0 ) then
            ao(jad) = a(k)
            iao(jad) = ik
            jao(j) = jad + 1
            ! Always out-of-diag element, add it again
            j = ik
            jad = jao(j)
            ao(jad) = -a(k)
            iao(jad) = jk
            jao(j) = jad + 1
         end if
      end do
      ! Shift back jao
      do j = ncol, 1, -1
         jao(j+1) = jao(j)
      end do
      jao(1) = 1

   end subroutine coo_skew2csc
!_______________________________________________________________________
!
   subroutine coo2csc_cmplx( ncol, nnz, a, ir, jc, ao, iao, jao )

      integer,                 intent(in)  :: ncol, nnz
      complex(kind=MF_DOUBLE), intent(in)  :: a(:)
      integer,                 intent(in)  :: ir(:), jc(:)
      complex(kind=MF_DOUBLE), intent(out) :: ao(:)
      integer,                 intent(out) :: iao(:), jao(:)
      !------ API end ------

      !-----------------------------------------------------------------
      !  Coordinate     to   Compressed Sparse Column
      ! (adapted from SPARSKIT2 (Mar 8 2005) 'coocsr' by É. Canot)
      !-----------------------------------------------------------------
      ! Complex version of 'coo2csc'
      !-----------------------------------------------------------------

      integer :: j, jad, k0, k

   !------ End of declarations -- execution starts hereafter  ------

      do k = 1, ncol + 1
         jao(k) = 0
      end do
      ! Determine col-lengths.
      do k = 1, nnz
         if( a(k) /= (0.0d0,0.0d0) ) then
            jao(jc(k)) = jao(jc(k)) + 1
         end if
      end do
      ! Starting position of each col.
      k = 1
      do j = 1, ncol + 1
         k0 = jao(j)
         jao(j) = k
         k = k + k0
      end do
      ! Go through the structure once more. Fill in output matrix.
      do k = 1, nnz
         j = jc(k)
         jad = jao(j)
         if( a(k) /= (0.0d0,0.0d0) ) then
            ao(jad) = a(k)
            iao(jad) = ir(k)
            jao(j) = jad + 1
         end if
      end do
      ! Shift back jao
      do j = ncol, 1, -1
         jao(j+1) = jao(j)
      end do
      jao(1) = 1

   end subroutine coo2csc_cmplx
!_______________________________________________________________________
!
   subroutine coo_symm2csc_cmplx( ncol, nnz,                            &
                                  a, ir, jc, ao, iao, jao )

      integer,                 intent(in)  :: ncol, nnz
      complex(kind=MF_DOUBLE), intent(in)  :: a(:)
      integer,                 intent(in)  :: ir(:), jc(:)
      complex(kind=MF_DOUBLE), intent(out) :: ao(:)
      integer,                 intent(out) :: iao(:), jao(:)
      !------ API end ------

      !-----------------------------------------------------------------
      !  Coordinate (with pattern)  to  Compressed Sparse Column
      !-----------------------------------------------------------------
      ! Complex version of 'coo_symm2csc'
      !-----------------------------------------------------------------

      integer :: j, jad, k0, k, ik, jk

   !------ End of declarations -- execution starts hereafter  ------

      do k = 1, ncol + 1
         jao(k) = 0
      end do
      ! Determine col-lengths.
      do k = 1, nnz
         if( a(k) /= (0.0d0,0.0d0) ) then
            j = jc(k)
            jao(j) = jao(j) + 1
            if( ir(k) /= j ) then
               ! Out-of-diag element
               j = ir(k)
               jao(j) = jao(j) + 1
            end if
         end if
      end do
      ! Starting position of each col.
      k = 1
      do j = 1, ncol + 1
         k0 = jao(j)
         jao(j) = k
         k = k + k0
      end do
      ! Go through the structure once more. Fill in output matrix.
      do k = 1, nnz
         ik = ir(k)
         jk = jc(k)
         j = jk
         jad = jao(j)
         if( a(k) /= (0.0d0,0.0d0) ) then
            ao(jad) = a(k)
            iao(jad) = ik
            jao(j) = jad + 1
            ! If out-of-diag element, add it again
            if( ik /= jk ) then
               j = ik
               jad = jao(j)
               ao(jad) = a(k)
               iao(jad) = jk
               jao(j) = jad + 1
            end if
         end if
      end do
      ! Shift back jao
      do j = ncol, 1, -1
         jao(j+1) = jao(j)
      end do
      jao(1) = 1

   end subroutine coo_symm2csc_cmplx
!_______________________________________________________________________
!
   subroutine coo_herm2csc_cmplx( ncol, nnz,                            &
                                  a, ir, jc, ao, iao, jao )

      integer,                 intent(in)  :: ncol, nnz
      complex(kind=MF_DOUBLE), intent(in)  :: a(:)
      integer,                 intent(in)  :: ir(:), jc(:)
      complex(kind=MF_DOUBLE), intent(out) :: ao(:)
      integer,                 intent(out) :: iao(:), jao(:)
      !------ API end ------

      !-----------------------------------------------------------------
      !  Coordinate (with pattern)  to  Compressed Sparse Column
      !-----------------------------------------------------------------
      ! Converts a complex matrix having an "hermitian" pattern and whose
      ! either the lower or the upper part is stored in coordinate format
      ! into a CSC format.
      ! No check for values of ir and jc!
      ! No check for size of ao and iao!
      !-----------------------------------------------------------------
      ! On entry:
      !---------
      ! ncol = dimension of the matrix
      ! nnz  = number of nonzero elements in matrix
      ! a,
      ! ir,
      ! jc   = real matrix in coordinate format.
      !        a(k), ir(k), jc(k) store the
      !        nnz nonzero elements of the matrix with :
      !          a(k)  = actual real value of the elements,
      !          ir(k) = its row number,
      !          jc(k) = its column  number.
      !        The order of the elements is arbitrary.
      !
      ! On return:
      !-----------
      ! ao, iao, jao = matrix in general sparse matrix format with ao
      ! containing the real values, iao containing the row indexes,
      ! and jao being the pointer to the beginning of the column,
      ! in arrays ao, iao.
      !
      ! The resulting matrix is not necessarily row sorted; it depends
      ! on the order of the entries!
      !
      ! Entries containing a null value are discarded.
      ! Warning : non-zero duplicated entries are not processed (i.e.
      !           they are left duplicated!).
      !-----------------------------------------------------------------

      integer :: j, jad, k0, k, ik, jk

   !------ End of declarations -- execution starts hereafter  ------

      do k = 1, ncol + 1
         jao(k) = 0
      end do
      ! Determine col-lengths.
      do k = 1, nnz
         if( a(k) /= (0.0d0,0.0d0) ) then
            j = jc(k)
            jao(j) = jao(j) + 1
            if( ir(k) /= j ) then
               ! Out-of-diag element
               j = ir(k)
               jao(j) = jao(j) + 1
            end if
         end if
      end do
      ! Starting position of each col.
      k = 1
      do j = 1, ncol + 1
         k0 = jao(j)
         jao(j) = k
         k = k + k0
      end do
      ! Go through the structure once more. Fill in output matrix.
      do k = 1, nnz
         ik = ir(k)
         jk = jc(k)
         j = jk
         jad = jao(j)
         if( a(k) /= (0.0d0,0.0d0) ) then
            ao(jad) = a(k)
            iao(jad) = ik
            jao(j) = jad + 1
            ! If out-of-diag element, add it again
            if( ik /= jk ) then
               j = ik
               jad = jao(j)
               ao(jad) = conjg(a(k))
               iao(jad) = jk
               jao(j) = jad + 1
            end if
         end if
      end do
      ! Shift back jao
      do j = ncol, 1, -1
         jao(j+1) = jao(j)
      end do
      jao(1) = 1

   end subroutine coo_herm2csc_cmplx
!_______________________________________________________________________
!
   subroutine coo_skew2csc_cmplx( ncol, nnz,                            &
                                  a, ir, jc, ao, iao, jao )

      integer,                 intent(in)  :: ncol, nnz
      complex(kind=MF_DOUBLE), intent(in)  :: a(:)
      integer,                 intent(in)  :: ir(:), jc(:)
      complex(kind=MF_DOUBLE), intent(out) :: ao(:)
      integer,                 intent(out) :: iao(:), jao(:)
      !------ API end ------

      !-----------------------------------------------------------------
      !  Coordinate (with pattern)  to  Compressed Sparse Column
      !-----------------------------------------------------------------
      ! Complex version of 'coo_skew2csc'
      !-----------------------------------------------------------------

      integer :: j, jad, k0, k, ik, jk

   !------ End of declarations -- execution starts hereafter  ------

      do k = 1, ncol + 1
         jao(k) = 0
      end do
      ! Determine col-lengths.
      do k = 1, nnz
         if( a(k) /= (0.0d0,0.0d0) ) then
            j = jc(k)
            jao(j) = jao(j) + 1
            ! Always out-of-diag element, add it again
            j = ir(k)
            jao(j) = jao(j) + 1
         end if
      end do
      ! Starting position of each col.
      k = 1
      do j = 1, ncol + 1
         k0 = jao(j)
         jao(j) = k
         k = k + k0
      end do
      ! Go through the structure once more. Fill in output matrix.
      do k = 1, nnz
         ik = ir(k)
         jk = jc(k)
         j = jk
         jad = jao(j)
         if( a(k) /= (0.0d0,0.0d0) ) then
            ao(jad) = a(k)
            iao(jad) = ik
            jao(j) = jad + 1
            ! Always out-of-diag element, add it again
            j = ik
            jad = jao(j)
            ao(jad) = -a(k)
            iao(jad) = jk
            jao(j) = jad + 1
         end if
      end do
      ! Shift back jao
      do j = ncol, 1, -1
         jao(j+1) = jao(j)
      end do
      jao(1) = 1

   end subroutine coo_skew2csc_cmplx
