! f90 include file

!_______________________________________________________________________
!
   subroutine colperm( ncol, a, ia, ja, ao, iao, jao, perm )

      integer,              intent(in) :: ncol, ia(:), ja(:), perm(:)
      real(kind=MF_DOUBLE), intent(in) :: a(:)
      integer,              intent(out) :: iao(:), jao(:)
      real(kind=MF_DOUBLE), intent(out) :: ao(:)
      !-----------------------------------------------------------------
      ! This subroutine permutes the columns of a matrix in CSC format.
      ! It computes B = A P  where P is a permutation matrix.
      ! The permutation P is defined through the array perm: the jth col
      ! of B is the perm(j)^th col of A.
      ! In Matlab-like syntax, it computes: B(:,:) = A(:,p)
      ! -- Édouard Canot -- Feb. 2012
      !-----------------------------------------------------------------
      ! on entry:
      !----------
      ! ncol  = nb of columns of the matrix
      ! a, ia, ja = input matrix in CSC format
      ! perm  = integer array of length ncol containing the permutation
      !         arrays for the columns.
      !
      !------------
      ! on return:
      !------------
      ! ao, iao, jao = output matrix in CSC format
      !-----------------------------------------------------------------

      integer :: i, j, jj, k, ko

   !------ end of declarations -- execution starts hereafter  ------

      ! determine lengths for output matrix.
      do j = 1, ncol
         i = perm(j)
         jao(j+1) = ja(i+1) - ja(i)
      end do

      ! get pointers from lengths
      jao(1) = 1
      do j = 1, ncol
         jao(j+1) = jao(j+1) + jao(j)
      end do

      ! copying
      do j = 1, ncol

         ! old col jj = perm(j)  -- new col = j -- ko = new pointer
         jj = perm(j)
         ko = jao(j)
         do k = ja(jj), ja(jj+1) - 1
            iao(ko) = ia(k)
            ao(ko) = a(k)
            ko = ko + 1
         end do
      end do

   end subroutine colperm
!_______________________________________________________________________
!
   subroutine colpinv( ncol, a, ia, ja, ao, iao, jao, perm )

      integer,              intent(in) :: ncol, ia(:), ja(:), perm(:)
      real(kind=MF_DOUBLE), intent(in) :: a(:)
      integer,              intent(out) :: iao(:), jao(:)
      real(kind=MF_DOUBLE), intent(out) :: ao(:)
      !-----------------------------------------------------------------
      ! This subroutine permutes the columns of a matrix in CSC format.
      ! It computes B = A P  where P is a permutation matrix.
      ! The permutation P is defined through the array perm: for each j,
      ! perm(j) represents the destination col number of col number j.
      ! In Matlab-like syntax, it computes B, such that B(:,p) = A(:,:)
      ! -- Édouard Canot -- Jan. 2012
      ! (from rperm of SPARSKIT)
      !-----------------------------------------------------------------
      ! on entry:
      !----------
      ! ncol  = nb of columns of the matrix
      ! a, ia, ja = input matrix in CSC format
      ! perm  = integer array of length ncol containing the permutation
      !         arrays for the columns.
      !
      !------------
      ! on return:
      !------------
      ! ao, iao, jao = output matrix in CSC format
      !-----------------------------------------------------------------

      integer :: i, j, jj, k, ko

   !------ end of declarations -- execution starts hereafter  ------

      ! determine lengths for output matrix.
      do j = 1, ncol
         i = perm(j)
         jao(i+1) = ja(j+1) - ja(j)
      end do

      ! get pointers from lengths
      jao(1) = 1
      do j = 1, ncol
         jao(j+1) = jao(j+1) + jao(j)
      end do

      ! copying
      do jj = 1, ncol

         ! old col = jj  -- new col = perm(jj) -- ko = new pointer
         ko = jao(perm(jj))
         do k = ja(jj), ja(jj+1) - 1
            iao(ko) = ia(k)
            ao(ko) = a(k)
            ko = ko + 1
         end do
      end do

   end subroutine colpinv
!_______________________________________________________________________
!
   subroutine colperm2( ncol, a, ia, ja, perm )

      integer,              intent(in) :: ncol, perm(:)
      integer,              intent(in out) :: ia(:), ja(:)
      real(kind=MF_DOUBLE), intent(in out) :: a(:)
      !-----------------------------------------------------------------
      ! This subroutine applies an 'in-place' permutation to the columns
      ! of a matrix in CSC format.
      ! It computes B = A P  where P is a permutation matrix.
      ! The permutation P is defined through the array perm: the jth col
      ! of B is the perm(j)^th col of A.
      ! In Matlab-like syntax, it computes: B(:,:) = A(:,p)
      ! -- Édouard Canot -- Feb. 2012
      !-----------------------------------------------------------------
      ! on entry:
      !----------
      ! ncol  = nb of columns of the matrix
      ! a, ia, ja = input matrix in CSC format
      ! perm  = integer array of length ncol containing the permutation
      !         arrays for the columns.
      !
      !------------
      ! on return:
      !------------
      ! a, ia, ja = output matrix in CSC format
      !-----------------------------------------------------------------

      integer :: i, j, jj, k, ko, nnz
      integer, allocatable :: jao(:), pinv(:)

   !------ end of declarations -- execution starts hereafter  ------

      allocate( jao(ncol+1) )

      ! determine lengths for output matrix.
      do j = 1, ncol
         i = perm(j)
         jao(j+1) = ja(i+1) - ja(i)
      end do

      ! get pointers from lengths
      jao(1) = 1
      do j = 1, ncol
         jao(j+1) = jao(j+1) + jao(j)
      end do

      ! determine the inverse permutation for the whole vectors ia(:)
      ! and a(:)
      nnz = ja(ncol+1) - 1
      allocate( pinv(nnz) )
      do j = 1, ncol
         ! old col j, = perm(j) -- new col = j
         jj = perm(j)
         ko = jao(j)
         do k = ja(jj), ja(jj+1) - 1
            pinv(k) = ko
            ko = ko + 1
         end do
      end do

      ! apply pinv(:) to 'ia' and 'a'
      call ivpinv( nnz, ia, pinv )
      call dvpinv( nnz, a, pinv )

      ! copy jao into ja
      do j = 1, ncol + 1
         ja(j) = jao(j)
      end do

   end subroutine colperm2
!_______________________________________________________________________
!
   subroutine colpinv2( ncol, a, ia, ja, perm )

      integer,              intent(in) :: ncol, perm(:)
      integer,              intent(in out) :: ia(:), ja(:)
      real(kind=MF_DOUBLE), intent(in out) :: a(:)
      !-----------------------------------------------------------------
      ! This subroutine applies an 'in-place' permutation to the columns
      ! of a matrix in CSC format.
      ! It computes B = A P  where P is a permutation matrix.
      ! The permutation P is defined through the array perm: for each j,
      ! perm(j) represents the destination col number of col number j.
      ! In Matlab-like syntax, it computes B, such that B(:,p) = A(:,:)
      ! -- Édouard Canot -- Feb. 2012
      ! (from rperm of SPARSKIT)
      !-----------------------------------------------------------------
      ! on entry:
      !----------
      ! ncol  = nb of columns of the matrix
      ! a, ia, ja = input matrix in CSC format
      ! perm  = integer array of length ncol containing the permutation
      !         arrays for the columns.
      !
      !------------
      ! on return:
      !------------
      ! a, ia, ja = output matrix in CSC format
      !-----------------------------------------------------------------

      integer :: i, j, k, ko, nnz
      integer, allocatable :: jao(:), pinv(:)

   !------ end of declarations -- execution starts hereafter  ------

      allocate( jao(ncol+1) )

      ! determine lengths for output matrix.
      do j = 1, ncol
         i = perm(j)
         jao(i+1) = ja(j+1) - ja(j)
      end do

      ! get pointers from lengths
      jao(1) = 1
      do j = 1, ncol
         jao(j+1) = jao(j+1) + jao(j)
      end do

      ! determine the inverse permutation for the whole vectors ia(:)
      ! and a(:)
      nnz = ja(ncol+1) - 1
      allocate( pinv(nnz) )
      do j = 1, ncol
         ! old col = j  -- new col = perm(j)
         ko = jao(perm(j))
         do k = ja(j), ja(j+1) - 1
            pinv(k) = ko
            ko = ko + 1
         end do
      end do

      ! apply pinv(:) to 'ia' and 'a'
      call ivpinv( nnz, ia, pinv )
      call dvpinv( nnz, a, pinv )

      ! copy jao into ja
      do j = 1, ncol + 1
         ja(j) = jao(j)
      end do

   end subroutine colpinv2
!_______________________________________________________________________
!
   subroutine colperm_cmplx( ncol, a, ia, ja, ao, iao, jao, perm )

      integer,                 intent(in) :: ncol, ia(:), ja(:), perm(:)
      complex(kind=MF_DOUBLE), intent(in) :: a(:)
      integer,                 intent(out) :: iao(:), jao(:)
      complex(kind=MF_DOUBLE), intent(out) :: ao(:)
      !-----------------------------------------------------------------
      ! complex version of 'colperm'
      !-----------------------------------------------------------------

      integer :: i, j, jj, k, ko

   !------ end of declarations -- execution starts hereafter  ------

      ! determine lengths for output matrix.
      do j = 1, ncol
         i = perm(j)
         jao(j+1) = ja(i+1) - ja(i)
      end do

      ! get pointers from lengths
      jao(1) = 1
      do j = 1, ncol
         jao(j+1) = jao(j+1) + jao(j)
      end do

      ! copying
      do j = 1, ncol

         ! old col jj = perm(j)  -- new col = j -- ko = new pointer
         jj = perm(j)
         ko = jao(j)
         do k = ja(jj), ja(jj+1) - 1
            iao(ko) = ia(k)
            ao(ko) = a(k)
            ko = ko + 1
         end do
      end do

   end subroutine colperm_cmplx
!_______________________________________________________________________
!
   subroutine colpinv_cmplx( ncol, a, ia, ja, ao, iao, jao, perm )

      integer,                 intent(in) :: ncol, ia(:), ja(:), perm(:)
      complex(kind=MF_DOUBLE), intent(in) :: a(:)
      integer,                 intent(out) :: iao(:), jao(:)
      complex(kind=MF_DOUBLE), intent(out) :: ao(:)
      !-----------------------------------------------------------------
      ! complex version of 'colpinv'
      !-----------------------------------------------------------------

      integer :: i, j, jj, k, ko

   !------ end of declarations -- execution starts hereafter  ------

      ! determine lengths for output matrix.
      do j = 1, ncol
         i = perm(j)
         jao(i+1) = ja(j+1) - ja(j)
      end do

      ! get pointers from lengths
      jao(1) = 1
      do j = 1, ncol
         jao(j+1) = jao(j+1) + jao(j)
      end do

      ! copying
      do jj = 1, ncol

         ! old col = jj  -- new col = perm(jj) -- ko = new pointer
         ko = jao(perm(jj))
         do k = ja(jj), ja(jj+1) - 1
            iao(ko) = ia(k)
            ao(ko) = a(k)
            ko = ko + 1
         end do
      end do

   end subroutine colpinv_cmplx
!_______________________________________________________________________
!
   subroutine colperm2_cmplx( ncol, a, ia, ja, perm )

      integer,                 intent(in) :: ncol, perm(:)
      integer,                 intent(in out) :: ia(:), ja(:)
      complex(kind=MF_DOUBLE), intent(in out) :: a(:)
      !-----------------------------------------------------------------
      ! complex version of 'colpinv2'
      !-----------------------------------------------------------------

      integer :: i, j, jj, k, ko, nnz
      integer, allocatable :: jao(:), pinv(:)

   !------ end of declarations -- execution starts hereafter  ------

      allocate( jao(ncol+1) )

      ! determine lengths for output matrix.
      do j = 1, ncol
         i = perm(j)
         jao(j+1) = ja(i+1) - ja(i)
      end do

      ! get pointers from lengths
      jao(1) = 1
      do j = 1, ncol
         jao(j+1) = jao(j+1) + jao(j)
      end do

      ! determine the inverse permutation for the whole vectors ia(:)
      ! and a(:)
      nnz = ja(ncol+1) - 1
      allocate( pinv(nnz) )
      do j = 1, ncol
         ! old col j, = perm(j) -- new col = j
         jj = perm(j)
         ko = jao(j)
         do k = ja(jj), ja(jj+1) - 1
            pinv(k) = ko
            ko = ko + 1
         end do
      end do

      ! apply pinv(:) to 'ia' and 'a'
      call ivpinv( nnz, ia, pinv )
      call zvpinv( nnz, a, pinv )

      ! copy jao into ja
      do j = 1, ncol + 1
         ja(j) = jao(j)
      end do

   end subroutine colperm2_cmplx
!_______________________________________________________________________
!
   subroutine colpinv2_cmplx( ncol, a, ia, ja, perm )

      integer,                 intent(in) :: ncol, perm(:)
      integer,                 intent(in out) :: ia(:), ja(:)
      complex(kind=MF_DOUBLE), intent(in out) :: a(:)
      !-----------------------------------------------------------------
      ! complex version of 'colpinv2'
      !-----------------------------------------------------------------

      integer :: i, j, k, ko, nnz
      integer, allocatable :: jao(:), pinv(:)

   !------ end of declarations -- execution starts hereafter  ------

      allocate( jao(ncol+1) )

      ! determine lengths for output matrix.
      do j = 1, ncol
         i = perm(j)
         jao(i+1) = ja(j+1) - ja(j)
      end do

      ! get pointers from lengths
      jao(1) = 1
      do j = 1, ncol
         jao(j+1) = jao(j+1) + jao(j)
      end do

      ! determine the inverse permutation for the whole vectors ia(:)
      ! and a(:)
      nnz = ja(ncol+1) - 1
      allocate( pinv(nnz) )
      do j = 1, ncol
         ! old col = j  -- new col = perm(j)
         ko = jao(perm(j))
         do k = ja(j), ja(j+1) - 1
            pinv(k) = ko
            ko = ko + 1
         end do
      end do

      ! apply pinv(:) to 'ia' and 'a'
      call ivpinv( nnz, ia, pinv )
      call zvpinv( nnz, a, pinv )

      ! copy jao into ja
      do j = 1, ncol + 1
         ja(j) = jao(j)
      end do

   end subroutine colpinv2_cmplx

