! f90 include file

!_______________________________________________________________________
!
   function mfSpDiags_vec( m, n, v, d ) result( out )

      integer, intent(in) :: m, n
      type(mfArray) :: v
      integer, intent(in), optional :: d

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! create a sparse matrix whose diagonal 'd' is made
      ! by the vector (mfArray) v.
      !
      ! v may have a size greater to those of diagonal 'd'; in such a
      ! case, the latter elements are never referenced.
      !
      ! v may be also a scalar; in such a case, this scalar is used to
      ! to fill the whole diagonal.

      integer :: nzmax, nrow, ncol
      integer :: j, k, ld, dim, d0
      logical :: v_is_scalar
      real(kind=MF_DOUBLE) :: rr
      complex(kind=MF_DOUBLE) :: cc

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v )

      if( v%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfSpDiags", "W",                           &
                            "'v' is empty!" )
         go to 99
      end if

      if( v%data_type /= MF_DT_DBLE .and.                               &
          v%data_type /= MF_DT_CMPLX )    then
         call PrintMessage( "mfSpDiags", "W",                           &
                            "'v' must be dense real or complex!" )
         go to 99
      end if

      nrow = m
      ncol = n

      if( present(d) ) then
         d0 = d
      else
         d0 = 0
      end if

      if( d0 < -(nrow-1) .or. ncol-1 < d0 ) then
         call PrintMessage( "mfSpDiags", "W",                           &
                            "'d' is out of range!" )
         go to 99
      end if

      if( v%shape(1) == 1 ) then
         dim = 2
      else if( v%shape(2) == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfSpDiags", "E",                           &
                            "mfArray 'v' must be a vector or a scalar!" )
         go to 99
      end if

      ! computes 'ld' : length of diag. number 'd0'
      ld = diag_length( nrow, ncol, d0 )

      if( v%shape(1) == 1 .and. v%shape(2) == 1 .and. ld > 1 ) then
         v_is_scalar = .true.
      else
         v_is_scalar = .false.
         if( v%shape(1)*v%shape(2) < ld ) then
            call PrintMessage( "mfSpDiags", "E",                        &
                               "bad dim for vector 'v'!" )
            go to 99
         end if
      end if

      if( v_is_scalar ) then
         if( v%data_type == MF_DT_DBLE ) then
            rr = v%double(1,1)
            if( rr == 0.0d0 ) then
               out = mfSpAlloc(nrow,ncol,nzmax=ld)
               call PrintMessage( "mfSpDiags", "I",                     &
                                  "creating a null matrix because diagonal is null!" )
               go to 89
            end if
         else if( v%data_type == MF_DT_CMPLX ) then
            cc = v%cmplx(1,1)
            if( cc == (0.0d0,0.0d0) ) then
               out = mfSpAlloc(nrow,ncol,nzmax=ld,kind="complex")
               call PrintMessage( "mfSpDiags", "I",                     &
                                  "creating a null matrix because diagonal is null!" )
               go to 89
            end if
         end if
      end if

      nzmax = ld

      out%shape = [ nrow, ncol ]

      allocate( out%i(nzmax) )

      allocate( out%j(ncol+1) )

      if( v%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_SP_DBLE
         allocate( out%a(nzmax) )

         if( d0 <= 0 ) then

            out%j(1) = 1
            if( v_is_scalar ) then
               do j = 1, ld
                  out%i(j) = j - d0
                  out%a(j) = rr
                  out%j(j+1) = out%j(j) + 1
               end do
            else ! v is a vector
               k = 0
               do j = 1, ld
                  if( dim == 1 ) then
                     rr = v%double(j,1)
                  else
                     rr = v%double(1,j)
                  end if
                  if( rr == 0.0d0 ) then
                     out%j(j+1) = out%j(j)
                  else
                     k = k + 1
                     out%i(k) = j - d0
                     out%a(k) = rr
                     out%j(j+1) = out%j(j) + 1
                  end if
               end do
            end if
            if( ld < ncol ) then
               do j = ld+1, ncol
                  out%j(j+1) = out%j(j)
               end do
            end if

         else ! d0 > 0

            out%j(1) = 1
            do j = 1, d0
               out%j(j+1) = out%j(j)
            end do
            if( v_is_scalar ) then
               do j = d0+1, d0+ld
                  out%i(j-d0) = j - d0
                  out%a(j-d0) = rr
                  out%j(j+1) = out%j(j) + 1
               end do
            else ! v is a vector
               k = 0
               do j = d0+1, d0+ld
                  if( dim == 1 ) then
                     rr = v%double(j-d0,1)
                  else
                     rr = v%double(1,j-d0)
                  end if
                  if( rr == 0.0d0 ) then
                     out%j(j+1) = out%j(j)
                  else
                     k = k + 1
                     out%i(k) = j - d0
                     out%a(k) = rr
                     out%j(j+1) = out%j(j) + 1
                  end if
               end do
            end if
            if( d0+ld < ncol ) then
               do j = d0+ld+1, ncol
                  out%j(j+1) = out%j(j)
               end do
            end if

         end if

      else if( v%data_type == MF_DT_CMPLX ) then

         out%data_type = MF_DT_SP_CMPLX
         allocate( out%z(nzmax) )

         if( d0 <= 0 ) then

            out%j(1) = 1
            if( v_is_scalar ) then
               do j = 1, ld
                  out%i(j) = j - d0
                  out%z(j) = cc
                  out%j(j+1) = out%j(j) + 1
               end do
            else ! v is a vector
               k = 0
               do j = 1, ld
                  if( dim == 1 ) then
                     cc = v%cmplx(j,1)
                  else
                     cc = v%cmplx(1,j)
                  end if
                  if( cc == (0.0d0,0.0d0) ) then
                     out%j(j+1) = out%j(j)
                  else
                     k = k + 1
                     out%i(k) = j - d0
                     out%z(k) = cc
                     out%j(j+1) = out%j(j) + 1
                  end if
               end do
            end if
            if( ld < ncol ) then
               do j = ld+1, ncol
                  out%j(j+1) = out%j(j)
               end do
            end if

         else ! d0 > 0

            out%j(1) = 1
            do j = 1, d0
               out%j(j+1) = out%j(j)
            end do
            if( v_is_scalar ) then
               do j = d0+1, d0+ld
                  out%i(j-d0) = j - d0
                  out%z(j-d0) = cc
                  out%j(j+1) = out%j(j) + 1
               end do
            else ! v is a vector
               k = 0
               do j = d0+1, d0+ld
                  if( dim == 1 ) then
                     cc = v%cmplx(j-d0,1)
                  else
                     cc = v%cmplx(1,j-d0)
                  end if
                  if( cc == (0.0d0,0.0d0) ) then
                     out%j(j+1) = out%j(j)
                  else
                     k = k + 1
                     out%i(k) = j - d0
                     out%z(k) = cc
                     out%j(j+1) = out%j(j) + 1
                  end if
               end do
            end if
            if( d0+ld < ncol ) then
               do j = d0+ld+1, ncol
                  out%j(j+1) = out%j(j)
               end do
            end if

         end if

      end if

 89   continue

      if( d0 >= 0 ) then
         out%prop%triu = TRUE
      end if
      if( d0 <= 0 ) then
         out%prop%tril = TRUE
      end if
      out%row_sorted = TRUE

      if( m == n ) then
         ! square matrix case
         if( d0 == 0 ) then
            out%prop%symm = TRUE
         else
            ! matrix is not symmetric, unless all values of 'v' are
            ! null; in such a case, out becomes an empty sparse matrix.
            if( out%j(ncol+1)-1 == 0 ) then
               out%prop%symm = TRUE
            else
               out%prop%symm = FALSE
            end if
         end if
      else
         out%prop%symm = FALSE
      end if

      if( mf_phys_units ) then
         out%units(:) = v%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v )
      call msAutoRelease( v )

#endif
   end function mfSpDiags_vec
!_______________________________________________________________________
!
   function mfSpDiags_vec_one_arg( v ) result( out )

      type(mfArray) :: v

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Create a square sparse matrix whose main diagonal 'd' is made
      ! by the vector (mfArray) v.
      !
      ! The dimensions of the output matrix is deduced from the length
      ! of the vector v.
      !
      ! If v is a scalar (pathological case) then out = v.

      integer :: n, nzmax, nrow, ncol
      integer :: j, k, ld, dim, d0
      real(kind=MF_DOUBLE) :: rr
      complex(kind=MF_DOUBLE) :: cc

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v )

      if( v%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfSpDiags", "W",                           &
                            "'v' is empty!" )
         go to 99
      end if

      if( v%data_type /= MF_DT_DBLE .and.                               &
          v%data_type /= MF_DT_CMPLX )    then
         call PrintMessage( "mfSpDiags", "W",                           &
                            "'v' must be dense real or complex!" )
         go to 99
      end if

      n = size(v)
      nrow = n
      ncol = n

      if( v%shape(1) == 1 ) then
         dim = 2
      else if( v%shape(2) == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfSpDiags", "E",                           &
                            "mfArray 'v' must be a vector or a scalar!" )
         go to 99
      end if

      ! computes 'ld' : length of diag. number 'd0'
      ld = n

      nzmax = ld

      out%shape = [ nrow, ncol ]

      allocate( out%i(nzmax) )

      allocate( out%j(ncol+1) )

      if( v%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_SP_DBLE
         allocate( out%a(nzmax) )

         out%j(1) = 1
         k = 0
         do j = 1, ld
            if( dim == 1 ) then
               rr = v%double(j,1)
            else
               rr = v%double(1,j)
            end if
            if( rr == 0.0d0 ) then
               out%j(j+1) = out%j(j)
            else
               k = k + 1
               out%i(k) = j
               out%a(k) = rr
               out%j(j+1) = out%j(j) + 1
            end if
         end do
         if( ld < ncol ) then
            do j = ld+1, ncol
               out%j(j+1) = out%j(j)
            end do
         end if

      else if( v%data_type == MF_DT_CMPLX ) then

         out%data_type = MF_DT_SP_CMPLX
         allocate( out%z(nzmax) )

         out%j(1) = 1
         k = 0
         do j = 1, ld
            if( dim == 1 ) then
               cc = v%cmplx(j,1)
            else
               cc = v%cmplx(1,j)
            end if
            if( cc == (0.0d0,0.0d0) ) then
               out%j(j+1) = out%j(j)
            else
               k = k + 1
               out%i(k) = j
               out%z(k) = cc
               out%j(j+1) = out%j(j) + 1
            end if
         end do
         if( ld < ncol ) then
            do j = ld+1, ncol
               out%j(j+1) = out%j(j)
            end do
         end if

      end if

 89   continue

      out%prop%triu = TRUE
      out%prop%tril = TRUE
      out%row_sorted = TRUE

      out%prop%symm = TRUE

      if( mf_phys_units ) then
         out%units(:) = v%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( v )
      call msAutoRelease( v )

#endif
   end function mfSpDiags_vec_one_arg
!_______________________________________________________________________
!
   function mfSpDiags_mat( m, n, A, d ) result( out )

      integer, intent(in) :: m, n
      type(mfArray) :: A
      integer, intent(in) :: d(:)

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! create a sparse matrix whose diagonals referenced by the
      ! indexes of vector 'd' are made by the columns of the mfArray A.
      !
      ! the vector 'd' must contain strictly increasing indexes;
      ! it can be also contain only one index.
      !
      ! A has of course a rectangular format; besides, some elements of
      ! A are not used.
      !
      ! storing convention for 'A': the whole first line becomes the
      ! fist column of 'out'
      !
      ! A may be also a row vector; in such a case, each column of the
      ! new sparse matrix will be composed by a unique value from A.

      integer :: nzmax, nrow, ncol
      integer :: ndiag, i, j, idiag, d0, k, nnz
      logical :: A_is_one_row
      real(kind=MF_DOUBLE) :: rr
      complex(kind=MF_DOUBLE) :: cc

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfSpDiags", "W",                           &
                            "'A' is empty!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfSpDiags", "W",                           &
                            "'A' must be dense array!" )
         go to 99
      end if

      if( .not. mfIsNumeric(A) ) then
         call PrintMessage( "mfSpDiags", "W",                           &
                            "'A' must be numeric array!" )
         go to 99
      end if

      ndiag = size(d)
      do i = 2, ndiag
         if( d(i) <= d(i-1) ) then
            call PrintMessage( "mfSpDiags", "E",                        &
                               "elements of 'd' must be in increasing order!" )
            go to 99
         end if
      end do

      nrow = m
      ncol = n

      if( d(1) < -(nrow-1) .or. ncol-1 < d(ndiag) ) then
         call PrintMessage( "mfSpDiags", "W",                           &
                            "some element(s) of 'd' are out of range!" )
         go to 99
      end if

      if( A%shape(1) == 1 .and. A%shape(2) == 1 ) then
         call PrintMessage( "mfSpDiags", "E",                           &
                            "mfArray 'A' cannot be a scalar!" )
         go to 99
      end if

      if( ndiag /= A%shape(2) ) then
         call PrintMessage( "mfSpDiags", "E",                           &
                            "bad conforming arguments!",               &
                            "[size(d) must match nb of col of A]" )
         go to 99
      end if

      if( A%shape(1) == 1 ) then
         A_is_one_row = .true.
      else
         A_is_one_row = .false.
      end if

      ! surestimation
      nzmax = min( ndiag*min(nrow,ncol), nrow*ncol )

      out%shape = [ nrow, ncol ]

      allocate( out%i(nzmax) )

      allocate( out%j(ncol+1) )

      if( A%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_SP_DBLE
         allocate( out%a(nzmax) )

         if( A_is_one_row ) then

            out%j(1) = 1
            k = 0
            do j = 1, ncol
               out%j(j+1) = out%j(j)
               do idiag = ndiag, 1, -1
                  d0 = d(idiag)
                  if( 1 <= j-d0 .and. j-d0 <= nrow ) then
                     rr = A%double(1,idiag)
                     if( rr /= 0.0d0 ) then
                        k = k + 1
                        out%i(k) = j - d0
                        out%a(k) = rr
                        out%j(j+1) = out%j(j+1) + 1
                     end if
                  end if
               end do
            end do

         else ! A is a matrix

            out%j(1) = 1
            k = 0
            do j = 1, ncol
               out%j(j+1) = out%j(j)
               do idiag = ndiag, 1, -1
                  d0 = d(idiag)
                  if( 1 <= j-d0 .and. j-d0 <= nrow ) then
                     rr = A%double(j,idiag)
                     if( rr /= 0.0d0 ) then
                        k = k + 1
                        out%i(k) = j - d0
                        out%a(k) = rr
                        out%j(j+1) = out%j(j+1) + 1
                     end if
                  end if
               end do
            end do

         end if

      else if( A%data_type == MF_DT_CMPLX ) then

         out%data_type = MF_DT_SP_CMPLX
         allocate( out%z(nzmax) )

         if( A_is_one_row ) then

            out%j(1) = 1
            k = 0
            do j = 1, ncol
               out%j(j+1) = out%j(j)
               do idiag = ndiag, 1, -1
                  d0 = d(idiag)
                  if( 1 <= j-d0 .and. j-d0 <= nrow ) then
                     cc = A%cmplx(1,idiag)
                     if( cc /= (0.0d0,0.0d0) ) then
                        k = k + 1
                        out%i(k) = j - d0
                        out%z(k) = cc
                        out%j(j+1) = out%j(j+1) + 1
                     end if
                  end if
               end do
            end do

         else ! A is a matrix

            out%j(1) = 1
            k = 0
            do j = 1, ncol
               out%j(j+1) = out%j(j)
               do idiag = ndiag, 1, -1
                  d0 = d(idiag)
                  if( 1 <= j-d0 .and. j-d0 <= nrow ) then
                     cc = A%cmplx(j,idiag)
                     if( cc /= (0.0d0,0.0d0) ) then
                        k = k + 1
                        out%i(k) = j - d0
                        out%z(k) = cc
                        out%j(j+1) = out%j(j+1) + 1
                     end if
                  end if
               end do
            end do

         end if

      end if

      nnz = out%j(ncol+1) - 1
      if( real(nzmax)/real(nnz) > 2.0 ) then
         call PrintMessage( "mfSpDiags", "I",                           &
                            "nzmax has been overestimated: nzmax > 2*nnz!", &
                            "[hint: you can use 'msSpReAlloc' to repack it]" )
      end if

      out%row_sorted = TRUE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfSpDiags_mat
