! f90 include file

!_______________________________________________________________________
!
   function mfSpImport_real( i, j, d,                                   &
                             m, n, nzmax, format, duplicated_entries,   &
                             pattern )                                  &
   result( out )

      integer,              intent(in)           :: i(:), j(:)
      real(kind=MF_DOUBLE), intent(in)           :: d(:)
      integer,              intent(in), optional :: m, n, nzmax
      character(len=*),     intent(in), optional :: format
      character(len=*),     intent(in), optional :: duplicated_entries
      character(len=*),     intent(in), optional :: pattern

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! import a sparse structure from usual f90 arrays
      !
      ! supported format: COO: (i,j,val) [default]
      !                   CSC: Compact Sparse Column
      !                   CSR: Compact Sparse Row
      !
      ! duplicated_entries: "added"       [default] (as in Matlab)
      !                     "ignored"
      !                     "replaced"
      !
      ! possible pattern: "symm"
      !                   "skew"
      !                   "herm" treated as "symm"

      ! for the COO format, the vector d may have a null size; in
      ! such a case, only the sparse structure is built.

      ! 'row sorting' is always applied (whatever the value of the
      ! global variable MF_SP_AUTO_ROW_SORTED), because the processing
      ! of duplicated entries requires sorting.

      integer :: nnz, nrow, ncol
      integer :: k, out_nnz, n_diag, n_upper, n_lower
      integer :: i_max, j_max
      character(len=3) :: format0
      character(len=8) :: dupl_entries
      character(len=4) :: pattern0
      character(len=5) :: part0 ! "lower" of "upper" (auto-detected)
      logical :: guess

      character(len=*), parameter :: ROUTINE_NAME = "mfSpImport"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      if( present(format) ) then
         format0 = to_upper(format)
      else
         format0 = "COO"
      end if

      if( present(duplicated_entries) ) then
         if( .not. MF_SP_AUTO_ROW_SORTED ) then
            ! user chose not to sort columns' entries, but, the row
            ! sorting in each column is always required.
            call PrintMessage( ROUTINE_NAME, "I",                       &
                               "you chose, via the 'msSetAutoRowSorted' routine,", &
                               "not to sort columns' entries. However, 'duplicated_entries'", &
                               "requires sorting; so temporarily activating 'row sorting'" )
         end if
         dupl_entries = to_lower(duplicated_entries)
         if( dupl_entries /= "ignored" .and.                            &
             dupl_entries /= "added"   .and.                            &
             dupl_entries /= "replaced" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "duplicated_entries must one of:",       &
                               "'ignored', 'added' or 'replaced'" )
            return
         end if
      else
         dupl_entries = "added"
      end if

      if( present(pattern) ) then
         pattern0 = to_upper(pattern)
         if( pattern0 == "HERM" ) then
!### TODO: print info msg?
            pattern0 = "SYMM"
         else
            if( pattern0 /= "SYMM" .and. pattern0 /= "SKEW" ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "pattern must one of:",               &
                                  '"SYMM" or "SKEW"' )
               return
            end if
         end if
      else
         pattern0 = ""
      end if

      select case( format0 )

         case( "COO" )

            ! i : row index   [size: exactly nnz]
            ! j : col index   [size: exactly nnz]
            ! d : value       [size: exactly nnz]

            if( minval(i) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'i' indexes must be greater than 1!" )
               return
            end if
            if( minval(j) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'j' indexes must be greater than 1!" )
               return
            end if

            nnz = size(i)
            if( size(j) /= nnz ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure in COO format,", &
                                  "arrays 'i' and 'j' must have same length!" )
               return
            end if

            ! array 'd' may have a null size...
            if( size(d) /= 0 .and. size(d) /= nnz ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure in COO format,", &
                                  "arrays 'i' and 'd' must have same length!" )
               return
            end if

if( size(d) == 0 .and. pattern0 /= "" ) then
   call PrintMessage( ROUTINE_NAME, "E",                                &
                      "when importing a sparse structure in COO format,", &
                      "'pattern' (optional) argument is not yet available!" )
   return
end if

!### NEW
            if( pattern0 == "" ) then
               out_nnz = nnz
               part0 = ""
            else ! "SYMM" or "SKEW"
               ! detect if lower or upper, and raise an error in case of
               ! unconsistency!
               if( pattern0 == "SYMM" ) then
                  guess = .false.
                  n_diag = 0
                  n_lower = 0
                  n_upper = 0
                  do k = 1, nnz
                     if( .not. guess ) then
                        if( i(k) == j(k) ) then
                           n_diag = n_diag + 1
                        else ! i(k) /= j(k)
                           if( i(k) < j(k) ) then
                              n_upper = n_upper + 1
                              part0 = "UPPER"
                           else ! i(k) > j(k)
                              n_lower = n_lower + 1
                              part0 = "LOWER"
                           end if
                           guess = .true.
                        end if
                     else
                        ! part0 is known and can be checked
                        if( i(k) == j(k) ) then
                           n_diag = n_diag + 1
                        else ! i(k) /= j(k)
                           if( part0 == "UPPER" ) then
                              if( i(k) < j(k) ) then
                                 n_upper = n_upper + 1
                              else
                                 call PrintMessage( ROUTINE_NAME, "E",  &
                                                    "when pattern is provided, indices 'i' and 'j' must be consistent!", &
                                                    "(here, we found elements located both in the lower and upper part", &
                                                    " of the matrix)" )
                                 return
                              end if
                           else ! part0 = "LOWER"
                              if( i(k) > j(k) ) then
                                 n_lower = n_lower + 1
                              else
                                 call PrintMessage( ROUTINE_NAME, "E",  &
                                                    "when pattern is provided, indices 'i' and 'j' must be consistent!", &
                                                    "(here, we found elements located both in the lower and upper part", &
                                                    " of the matrix)" )
                                 return
                              end if
                           end if
                        end if
                     end if
                  end do
                  if( .not. guess ) then
                     ! this is a strictly diagonal matrix...
                     part0 = "UPPER"
                  end if
                  if( part0 == "UPPER" ) then
                     out_nnz = n_diag + 2*n_upper
                  else ! part0 = "LOWER"
                     out_nnz = n_diag + 2*n_lower
                  end if
               else ! pattern0 = "SKEW
                  guess = .false.
                  n_lower = 0
                  n_upper = 0
                  do k = 1, nnz
                     if( .not. guess ) then
                        if( i(k) == j(k) ) then
                           call PrintMessage( ROUTINE_NAME, "E",        &
                                              "a skew-symmetric matrix cannot have any diagonal element!" )
                           return
                        end if
                        if( i(k) < j(k) ) then
                           n_upper = n_upper + 1
                           part0 = "UPPER"
                        else ! i(k) > j(k)
                           n_lower = n_lower + 1
                           part0 = "LOWER"
                        end if
                        guess = .true.
                     else
                        ! part0 is known and can be checked
                        if( i(k) == j(k) ) then
                           call PrintMessage( ROUTINE_NAME, "E",        &
                                              "a skew-symmetric matrix cannot have any diagonal element!" )
                           return
                        end if
                        if( part0 == "UPPER" ) then
                           if( i(k) < j(k) ) then
                              n_upper = n_upper + 1
                           else
                              call PrintMessage( ROUTINE_NAME, "E",     &
                                                 "when pattern is provided, indices 'i' and 'j' must be consistent!", &
                                                 "(here, we found an element located in the lower part of the matrix,", &
                                                 " whereas other element(s) have been found in the upper part)" )
                              return
                           end if
                        else ! part0 = "LOWER"
                           if( i(k) > j(k) ) then
                              n_lower = n_lower + 1
                           else
                              call PrintMessage( ROUTINE_NAME, "E",     &
                                                 "when pattern is provided, indices 'i' and 'j' must be consistent!", &
                                                 "(here, we found an element located in the upper part of the matrix,", &
                                                 " whereas other element(s) have been found in the lower part)" )
                              return
                           end if
                        end if
                     end if
                  end do
                  if( .not. guess ) then
                     print "(/,A)", "[Muesli:] " // trim(ROUTINE_NAME) // ": internal error."
                     print "(A)", "          import.inc: line 261: guess should be TRUE!"
                     print "(A,/)", "          -> you can report this bug. mailto: edouard.canot@univ-rennes.fr"
                     stop
                  end if
                  if( part0 == "UPPER" ) then
                     out_nnz = 2*n_upper
                  else ! part0 = "LOWER"
                     out_nnz = 2*n_lower
                  end if
               end if
            end if

            i_max = maxval(i)
            j_max = maxval(j)

            out%data_type = MF_DT_SP_DBLE
            if( present(m) ) then
               nrow = m
               if( pattern0 == "" ) then
                  if( nrow < i_max ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "importing a sparse structure in COO format:", &
                                        "some row indexes are out of range!" )
                     return
                  end if
               else
                  if( nrow < max( i_max, j_max ) ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "importing a sparse structure in COO format:", &
                                        "some row indexes are out of range!" )
                     return
                  end if
               end if
            else
               if( pattern0 == "" ) then
                  nrow = i_max
               else
                  nrow = max( i_max, j_max )
               end if
            end if
            if( present(n) ) then
               ncol = n
               if( pattern0 == "" ) then
                  if( ncol < j_max ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "importing a sparse structure in COO format:", &
                                        "some col indexes are out of range!" )
                     return
                  end if
               else
                  if( ncol < max( i_max, j_max ) ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "importing a sparse structure in COO format:", &
                                        "some col indexes are out of range!" )
                     return
                  end if
               end if
            else
               if( pattern0 == "" ) then
                  ncol = j_max
               else
                  ncol = max( i_max, j_max )
               end if
            end if

            if( pattern0 /= "" ) then
               if( nrow /= ncol ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "using 'pattern' when importing a sparse structure in COO format:", &
                                     "matrix must be square!",          &
                                     "(bad value for optional arg. 'm' or 'n'?)" )
                  return
               end if
            end if

            out%shape = [ nrow, ncol ]

            if( present(nzmax) ) then
               allocate( out%a(nzmax) )
               allocate( out%i(nzmax) )
            else
               allocate( out%a(out_nnz) )
               allocate( out%i(out_nnz) )
            end if
            allocate( out%j(ncol+1) )

            if( size(d) /= 0 ) then
               if( pattern0 == "SYMM" ) then
                  call coo_symm2csc( ncol, nnz, d, i, j, out%a, out%i, out%j )
               else if( pattern0 == "SKEW" ) then
                  call coo_skew2csc( ncol, nnz, d, i, j, out%a, out%i, out%j )
               else
                  call coo2csc( ncol, nnz, d, i, j, out%a, out%i, out%j )
               end if
               call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )
            else ! size(d) == 0
               ! out%a will be assigned later on... under the user
               ! responsability.
               if( pattern0 == "SYMM" ) then
stop "coo_symm2csc_struct_only: not yet implemented"
!!                  call coo_symm2csc_struct_only( ncol, nnz, d, i, j,    &
!!                                                 out%i, out%j )
               else if( pattern0 == "SKEW" ) then
stop "coo_skew2csc_struct_only: not yet implemented"
!!                  call coo_skew2csc_struct_only( ncol, nnz, d, i, j,    &
!!                                                 out%i, out%j )
               else
                  call coo2csc_struct_only( ncol, nnz, i, j, out%i, out%j )
               end if
               out%a(:) = MF_NAN
               call process_dupl_entries_struct_only( out, ROUTINE_NAME )
            end if

         case( "CSC" ) ! Compact Sparse Column

if( pattern0 /= "" ) then
   call PrintMessage( ROUTINE_NAME, "E",                                &
                      "'pattern' (optional) argument is currently available only for COO format!" )
   return
end if

            ! i : row index   [size: greater or equal nnz]
            ! j : pointer     [size: exactly ncol+1]      convention: j(1)=1
            ! d : value       [size: greater or equal nnz]
            !
            ! (i,j) must be strictly positive indexes

            nnz = j(size(j)) - 1
            if( minval(i(1:nnz)) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'i' indexes must be greater than 1!" )
               return
            end if
            if( minval(j) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'j' indexes must be greater than 1!" )
               return
            end if

            if( present(m) ) then
               nrow = m
               if( m < maxval(i(1:nnz)) ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when importing a sparse structure in CSC format,", &
                                     "bad value for argument 'm'!" )
                  return
               end if
            else
               nrow = maxval(i(1:nnz))
            end if
            if( present(n) ) then
               ncol = n
               if( ncol+1 > size(j) ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when importing a sparse structure in CSC format,", &
                                     "bad value for argument 'n'!" )
                  return
               end if
            else
               ncol = size(j) - 1
            end if
            out%data_type = MF_DT_SP_DBLE
            out%shape = [ nrow, ncol ]

            if( j(1) /= 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure in CSC format,", &
                                  "j(1) must be equal to 1!" )
               return
            end if
            nnz = j(ncol+1) - 1

            if( present(nzmax) ) then
               allocate( out%a(nzmax) )
               allocate( out%i(nzmax) )
            else
               allocate( out%a(nnz) )
               allocate( out%i(nnz) )
            end if
            allocate( out%j(ncol+1) )

            out%i(1:nnz) = i(1:nnz)
            out%j        = j(1:ncol+1)

            if( size(d) /= 0 ) then
               out%a(1:nnz) = d(1:nnz)
               call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )
            else ! size(d) == 0
               ! out%a will be assigned later on... under the user
               ! responsability.
               out%a(1:nnz) = MF_NAN
               call process_dupl_entries_struct_only( out, ROUTINE_NAME )
            end if

         case( "CSR" ) ! Compact Sparse Row

if( pattern0 /= "" ) then
   call PrintMessage( ROUTINE_NAME, "E",                                &
                      "'pattern' (optional) argument is currently available only for COO format!" )
   return
end if

            ! i : pointer     [size: exactly nrow+1]      convention: i(1)=1
            ! j : col index   [size: greater or equal nnz]
            ! d : value       [size: greater or equal nnz]

            nnz = i(size(i)) - 1
            if( minval(i) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'i' indexes must be greater than 1!" )
               return
            end if
            if( minval(j(1:nnz)) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'j' indexes must be greater than 1!" )
               return
            end if

            if( present(m) ) then
               nrow = m
               if( nrow+1 < size(i) ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when importing a sparse structure in CSR format,", &
                                     "bad value for argument 'm'!" )
                  return
               end if
            else
               nrow = size(i) - 1
            end if
            if( present(n) ) then
               ncol = n
               if( n < maxval(j(1:nnz)) ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when importing a sparse structure in CSR format,", &
                                     "bad value for argument 'n'!" )
                  return
               end if
            else
               ncol = maxval(j(1:nnz))
            end if
            out%data_type = MF_DT_SP_DBLE
            out%shape = [ nrow, ncol ]

            if( i(1) /= 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure in CSR format,", &
                                  "i(1) must be equal to 1!" )
               return
            end if
            nnz = i(nrow+1) - 1

            ! the sparse matrix OUT has the CSC storage!
            if( present(nzmax) ) then
               allocate( out%a(nzmax) )
               allocate( out%i(nzmax) )
            else
               allocate( out%a(nnz) )
               allocate( out%i(nnz) )
            end if
            allocate( out%j(ncol+1) )

            if( size(d) /= 0 ) then
               call csc_transp( ncol, nrow, d, j(1:nnz), i,             &
                                out%a, out%i, out%j )
               call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )
            else ! size(d) == 0
               ! out%a will be assigned later on... under the user
               ! responsability.
               call csc_transp_struct_only( ncol, nrow, j(1:nnz), i,    &
                                            out%i, out%j )
               out%a(:) = MF_NAN
               call process_dupl_entries_struct_only( out, ROUTINE_NAME )
            end if

         case default

            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "unknown specified format!" )
            return

      end select

      out%status_temporary = .true.

      call mf_restore_fpe( )

#endif
   end function mfSpImport_real
!_______________________________________________________________________
!
   function mfSpImport_cmplx( i, j, z,                                  &
                              m, n, nzmax, format, duplicated_entries,  &
                              pattern )                                 &
   result( out )

      integer,                 intent(in)           :: i(:), j(:)
      complex(kind=MF_DOUBLE), intent(in)           :: z(:)
      integer,                 intent(in), optional :: m, n, nzmax
      character(len=*),        intent(in), optional :: format
      character(len=*),        intent(in), optional :: duplicated_entries
      character(len=*),     intent(in), optional :: pattern

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! import a sparse structure from usual f90 arrays
      !
      ! supported format : COO : (i,j,val) [default]
      !                    CSC : Compact Sparse Column
      !                    CSR : Compact Sparse Row
      !
      ! duplicated_entries : "added"       [default] (as in Matlab)
      !                      "ignored"
      !                      "replaced"
      !
      ! possible pattern: "symm"
      !                   "skew"
      !                   "herm"

      ! for the COO format, the vector z may have a null size; in
      ! such a case, only the sparse structure is built.

      ! 'row sorting' is always applied (whatever the value of the
      ! global variable MF_SP_AUTO_ROW_SORTED), because the processing
      ! of duplicated entries requires sorting.

      integer :: nnz, nrow, ncol
      integer :: k, out_nnz, n_diag, n_upper, n_lower
      integer :: i_max, j_max
      character(len=3) :: format0
      character(len=8) :: dupl_entries
      character(len=4) :: pattern0
      character(len=5) :: part0 ! "lower" of "upper" (auto-detected)
      logical :: guess

      character(len=*), parameter :: ROUTINE_NAME = "mfSpImport"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      if( present(format) ) then
         format0 = to_upper(format)
      else
         format0 = "COO"
      end if

      if( present(duplicated_entries) ) then
         if( .not. MF_SP_AUTO_ROW_SORTED ) then
            ! user chose not to sort columns' entries, but, the row
            ! sorting in each column is always required.
            call PrintMessage( ROUTINE_NAME, "I",                       &
                               "you chose, via the 'msSetAutoRowSorted' routine,", &
                               "not to sort columns' entries. However, 'duplicated_entries'", &
                               "requires sorting; so temporarily activating 'row sorting'" )
         end if
         dupl_entries = to_lower(duplicated_entries)
         if( dupl_entries /= "ignored" .and.                            &
             dupl_entries /= "added"   .and.                            &
             dupl_entries /= "replaced" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "duplicated_entries must one of:",       &
                               "'ignored', 'added' or 'replaced'" )
            return
         end if
      else
         dupl_entries = "added"
      end if

      if( present(pattern) ) then
         pattern0 = to_upper(pattern)
         if( pattern0 /= "SYMM" .and. pattern0 /= "SKEW" .and.          &
             pattern0 /= "HERM" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "pattern must one of:",                  &
                               '"SYMM", "SKEW" or "HERM"' )
            return
         end if
      else
         pattern0 = ""
      end if

      select case( format0 )

         case( "COO" )

            ! i : row index   [size: exactly nnz]
            ! j : col index   [size: exactly nnz]
            ! d : value       [size: exactly nnz]

            if( minval(i) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'i' indexes must be greater than 1!" )
               return
            end if
            if( minval(j) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'j' indexes must be greater than 1!" )
               return
            end if

            nnz = size(i)
            if( size(j) /= nnz ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure in COO format,", &
                                  "arrays 'i' and 'j' must have same length!" )
               return
            end if

            ! array 'z' may have a null size...
            if( size(z) /= 0 .and. size(z) /= nnz ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure in COO format,", &
                                  "arrays 'i' and 'z' must have same length!" )
               return
            end if

if( size(z) == 0 .and. pattern0 /= "" ) then
   call PrintMessage( ROUTINE_NAME, "E",                                &
                      "when importing a sparse structure in COO format,", &
                      "'pattern' (optional) argument is not yet available!" )
   return
end if

!### NEW
            if( pattern0 == "" ) then
               out_nnz = nnz
               part0 = ""
            else ! "SYMM", "SKEW" or "HERM"
               ! detect if lower or upper, and raise an error in case of
               ! unconsistency!
               if( pattern0 == "SYMM" .or. pattern0 == "HERM" ) then
                  guess = .false.
                  n_diag = 0
                  n_lower = 0
                  n_upper = 0
                  do k = 1, nnz
                     if( .not. guess ) then
                        if( i(k) == j(k) ) then
                           n_diag = n_diag + 1
                        else ! i(k) /= j(k)
                           if( i(k) < j(k) ) then
                              n_upper = n_upper + 1
                              part0 = "UPPER"
                           else ! i(k) > j(k)
                              n_lower = n_lower + 1
                              part0 = "LOWER"
                           end if
                           guess = .true.
                        end if
                     else
                        ! part0 is known and can be checked
                        if( i(k) == j(k) ) then
                           n_diag = n_diag + 1
                        else ! i(k) /= j(k)
                           if( part0 == "UPPER" ) then
                              if( i(k) < j(k) ) then
                                 n_upper = n_upper + 1
                              else
                                 call PrintMessage( ROUTINE_NAME, "E",  &
                                                    "when pattern is provided, indices 'i' and 'j' must be consistent!", &
                                                    "(here, we found elements located both in the lower and upper part", &
                                                    " of the matrix)" )
                                 return
                              end if
                           else ! part0 = "LOWER"
                              if( i(k) > j(k) ) then
                                 n_lower = n_lower + 1
                              else
                                 call PrintMessage( ROUTINE_NAME, "E",  &
                                                    "when pattern is provided, indices 'i' and 'j' must be consistent!", &
                                                    "(here, we found elements located both in the lower and upper part", &
                                                    " of the matrix)" )
                                 return
                              end if
                           end if
                        end if
                     end if
                  end do
                  if( .not. guess ) then
                     ! this is a strictly diagonal matrix...
                     part0 = "UPPER"
                  end if
                  if( part0 == "UPPER" ) then
                     out_nnz = n_diag + 2*n_upper
                  else ! part0 = "LOWER"
                     out_nnz = n_diag + 2*n_lower
                  end if
               else ! pattern0 = "SKEW
                  guess = .false.
                  n_lower = 0
                  n_upper = 0
                  do k = 1, nnz
                     if( .not. guess ) then
                        if( i(k) == j(k) ) then
                           call PrintMessage( ROUTINE_NAME, "E",        &
                                              "a skew-symmetric matrix cannot have any diagonal element!" )
                           return
                        end if
                        if( i(k) < j(k) ) then
                           n_upper = n_upper + 1
                           part0 = "UPPER"
                        else ! i(k) > j(k)
                           n_lower = n_lower + 1
                           part0 = "LOWER"
                        end if
                        guess = .true.
                     else
                        ! part0 is known and can be checked
                        if( i(k) == j(k) ) then
                           call PrintMessage( ROUTINE_NAME, "E",        &
                                              "a skew-symmetric matrix cannot have any diagonal element!" )
                           return
                        end if
                        if( part0 == "UPPER" ) then
                           if( i(k) < j(k) ) then
                              n_upper = n_upper + 1
                           else
                              call PrintMessage( ROUTINE_NAME, "E",     &
                                                 "when pattern is provided, indices 'i' and 'j' must be consistent!", &
                                                 "(here, we found an element located in the lower part of the matrix,", &
                                                 " whereas other element(s) have been found in the upper part)" )
                              return
                           end if
                        else ! part0 = "LOWER"
                           if( i(k) > j(k) ) then
                              n_lower = n_lower + 1
                           else
                              call PrintMessage( ROUTINE_NAME, "E",     &
                                                 "when pattern is provided, indices 'i' and 'j' must be consistent!", &
                                                 "(here, we found an element located in the upper part of the matrix,", &
                                                 " whereas other element(s) have been found in the lower part)" )
                              return
                           end if
                        end if
                     end if
                  end do
                  if( .not. guess ) then
                     print "(/,A)", "[Muesli:] " // trim(ROUTINE_NAME) // ": internal error."
                     print "(A)", "          import.inc: line 818: guess should be TRUE!"
                     print "(A,/)", "          -> you can report this bug. mailto: edouard.canot@univ-rennes.fr"
                     stop
                  end if
                  if( part0 == "UPPER" ) then
                     out_nnz = 2*n_upper
                  else ! part0 = "LOWER"
                     out_nnz = 2*n_lower
                  end if
               end if
            end if

            i_max = maxval(i)
            j_max = maxval(j)

            out%data_type = MF_DT_SP_CMPLX
            if( present(m) ) then
               nrow = m
               if( pattern0 == "" ) then
                  if( nrow < i_max ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "importing a sparse structure in COO format:", &
                                        "some row indexes are out of range!" )
                     return
                  end if
               else
                  if( nrow < max( i_max, j_max ) ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "importing a sparse structure in COO format:", &
                                        "some row indexes are out of range!" )
                     return
                  end if
               end if
            else
               if( pattern0 == "" ) then
                  nrow = i_max
               else
                  nrow = max( i_max, j_max )
               end if
            end if
            if( present(n) ) then
               ncol = n
               if( pattern0 == "" ) then
                  if( ncol < j_max ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "importing a sparse structure in COO format:", &
                                        "some col indexes are out of range!" )
                     return
                  end if
               else
                  if( ncol < max( i_max, j_max ) ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "importing a sparse structure in COO format:", &
                                        "some col indexes are out of range!" )
                     return
                  end if
               end if
            else
               if( pattern0 == "" ) then
                  ncol = j_max
               else
                  ncol = max( i_max, j_max )
               end if
            end if

            if( pattern0 /= "" ) then
               if( nrow /= ncol ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "using 'pattern' when importing a sparse structure in COO format:", &
                                     "matrix must be square!",          &
                                     "(bad value for optional arg. 'm' or 'n'?)" )
                  return
               end if
            end if

            out%shape = [ nrow, ncol ]

            if( present(nzmax) ) then
               allocate( out%z(nzmax) )
               allocate( out%i(nzmax) )
            else
               allocate( out%z(out_nnz) )
               allocate( out%i(out_nnz) )
            end if
            allocate( out%j(ncol+1) )

            if( size(z) /= 0 ) then
               if( pattern0 == "SYMM" ) then
                  call coo_symm2csc_cmplx( ncol, nnz, z, i, j, out%z, out%i, out%j )
               else if( pattern0 == "SKEW" ) then
                  call coo_skew2csc_cmplx( ncol, nnz, z, i, j, out%z, out%i, out%j )
               else if( pattern0 == "HERM" ) then
                  call coo_herm2csc_cmplx( ncol, nnz, z, i, j, out%z, out%i, out%j )
               else
                  call coo2csc_cmplx( ncol, nnz, z, i, j, out%z, out%i, out%j )
               end if
               call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )
            else ! size(z) == 0
               ! out%z will be assigned later on... under the user
               ! responsability.
               if( pattern0 == "SYMM" .or. pattern0 == "HERM" ) then
stop "coo_symm2csc_struct_only: not yet implemented"
!!                  call coo_symm2csc_struct_only( ncol, nnz, d, i, j,    &
!!                                                 out%i, out%j )
               else if( pattern0 == "SKEW" ) then
stop "coo_skew2csc_struct_only: not yet implemented"
!!                  call coo_skew2csc_struct_only( ncol, nnz, d, i, j,    &
!!                                                 out%i, out%j )
               else
                  call coo2csc_struct_only( ncol, nnz, i, j, out%i, out%j )
               end if
               out%z(:) = (1.0d0,1.0d0)*MF_NAN
               call process_dupl_entries_struct_only( out, ROUTINE_NAME )
            end if

         case( "CSC" ) ! Compact Sparse Column

if( pattern0 /= "" ) then
   call PrintMessage( ROUTINE_NAME, "E",                                &
                      "'pattern' (optional) argument is currently available only for COO format!" )
   return
end if

            ! i : row index   [size: greater or equal nnz]
            ! j : pointer     [size: exactly ncol+1]      convention: j(1)=1
            ! z : value       [size: greater or equal nnz]
            !
            ! (i,j) must be strictly positive indexes

            nnz = j(size(j)) - 1
            if( minval(i(1:nnz)) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'i' indexes must be greater than 1!" )
               return
            end if
            if( minval(j) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'j' indexes must be greater than 1!" )
               return
            end if

            if( present(m) ) then
               nrow = m
               if( m < maxval(i(1:nnz)) ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when importing a sparse structure in CSC format,", &
                                     "bad value for argument 'm'!" )
                  return
               end if
            else
               nrow = maxval(i(1:nnz))
            end if
            if( present(n) ) then
               ncol = n
               if( ncol+1 > size(j) ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when importing a sparse structure in CSC format,", &
                                     "bad value for argument 'n'!" )
                  return
               end if
            else
               ncol = size(j) - 1
            end if
            out%data_type = MF_DT_SP_CMPLX
            out%shape = [ nrow, ncol ]

            if( j(1) /= 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure in CSC format,", &
                                  "j(1) must be equal to 1!" )
               return
            end if
            nnz = j(ncol+1) - 1

            if( present(nzmax) ) then
               allocate( out%z(nzmax) )
               allocate( out%i(nzmax) )
            else
               allocate( out%z(nnz) )
               allocate( out%i(nnz) )
            end if
            allocate( out%j(ncol+1) )

            out%i(1:nnz) = i(1:nnz)
            out%j        = j(1:ncol+1)

            if( size(z) /= 0 ) then
               out%z(1:nnz) = z(1:nnz)
               call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )
            else ! size(d) == 0
               ! out%z will be assigned later on... under the user
               ! responsability.
               out%z(1:nnz) = (1.0d0,1.0d0)*MF_NAN
               call process_dupl_entries_struct_only( out, ROUTINE_NAME )
            end if

         case( "CSR" ) ! Compact Sparse Row

if( pattern0 /= "" ) then
   call PrintMessage( ROUTINE_NAME, "E",                                &
                      "'pattern' (optional) argument is currently available only for COO format!" )
   return
end if

            ! i : pointer     [size: exactly nrow+1]      convention: i(1)=1
            ! j : col index   [size: greater or equal nnz]
            ! z : value       [size: greater or equal nnz]

            nnz = i(size(i)) - 1
            if( minval(i) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'i' indexes must be greater than 1!" )
               return
            end if
            if( minval(j(1:nnz)) < 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure:", &
                                  "'j' indexes must be greater than 1!" )
               return
            end if

            if( present(m) ) then
               nrow = m
               if( nrow+1 < size(i) ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when importing a sparse structure in CSR format,", &
                                     "bad value for argument 'm'!" )
                  return
               end if
            else
               nrow = size(i) - 1
            end if
            if( present(n) ) then
               ncol = n
               if( n < maxval(j(1:nnz)) ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when importing a sparse structure in CSR format,", &
                                     "bad value for argument 'n'!" )
                  return
               end if
            else
               ncol = maxval(j(1:nnz))
            end if
            out%data_type = MF_DT_SP_CMPLX
            out%shape = [ nrow, ncol ]

            if( i(1) /= 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when importing a sparse structure in CSR format,", &
                                  "i(1) must be equal to 1!" )
               return
            end if
            nnz = i(nrow+1) - 1

            ! the sparse matrix OUT has the CSC storage!
            if( present(nzmax) ) then
               allocate( out%z(nzmax) )
               allocate( out%i(nzmax) )
            else
               allocate( out%z(nnz) )
               allocate( out%i(nnz) )
            end if
            allocate( out%j(ncol+1) )

            if( size(z) /= 0 ) then
               call csc_transp_cmplx( ncol, nrow, z, j(1:nnz), i,       &
                                      out%z, out%i, out%j )
               call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )
            else ! size(d) == 0
               ! out%a will be assigned later on... under the user
               ! responsability.
               call csc_transp_struct_only( ncol, nrow, j(1:nnz), i,    &
                                            out%i, out%j )
               out%z(:) = (1.0d0,1.0d0)*MF_NAN
               call process_dupl_entries_struct_only( out, ROUTINE_NAME )
            end if

         case default

            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "unknown specified format!" )
            return

      end select

      out%status_temporary = .true.

      call mf_restore_fpe( )

#endif
   end function mfSpImport_cmplx
