! f90 include file

!_______________________________________________________________________
!
   function mfVoronoi( x_in, y_in, what, clip ) result( voronoi )

      type(mfArray)                              :: x_in, y_in
      character(len=*),     intent(in)           :: what
      real(kind=MF_DOUBLE), intent(in), optional :: clip(4)
      type(mfVoronoiStruct)                      :: voronoi
      !------ API end ------
#ifdef _DEVLP

      real(kind=MF_DOUBLE), pointer :: x_ptr_vec(:), y_ptr_vec(:)
      real(kind=MF_DOUBLE) :: x_min, x_max, y_min, y_max,               &
                              x_mid, y_mid, x_width, y_width
      real(kind=MF_DOUBLE), allocatable :: x_vertices(:), y_vertices(:)

      integer :: nb_pts, nb_edges, nb_neighbors, nb_vertices
      integer :: i, status, n, j1, j2
      integer, allocatable :: ptr_edges(:,:), perm_sites(:),            &
                              ptr_neighbors(:,:), neighbors(:),         &
                              list_edges(:)

      character(len=*), parameter :: ROUTINE_NAME = "mfVoronoi"

#include "../misc/voronoi/2d/voronoi_f90wrapper.inc"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x_in, y_in )

      if( mfIsEmpty(x_in) .or. mfIsEmpty(y_in) ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "'x_in' or 'y_in' is empty!" )
         go to 99
      end if

      if( x_in%data_type /= MF_DT_DBLE .or. y_in%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x_in%shape(1) /= 1 .and. x_in%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' must be a vector!" )
         go to 99
      end if

      if( y_in%shape(1) /= 1 .and. y_in%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y_in' must be a vector!" )
         go to 99
      end if

      if( x_in%shape(1) /= y_in%shape(1) .or. x_in%shape(2) /= y_in%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' and 'y_in' must have the same shape!" )
         go to 99
      end if

      call msPointer( x_in, x_ptr_vec, no_crc=.true. )
      call msPointer( y_in, y_ptr_vec, no_crc=.true. )

      nb_pts = size(x_ptr_vec)

      if( nb_pts < 2 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' and 'y_in' must have at least 2 values!" )
         go to 99
      end if

      if( to_lower(trim(what)) /= "vertices" .and.                      &
          to_lower(trim(what)) /= "neighbors" .and.                     &
          to_lower(trim(what)) /= "both" ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "wrong value for argument 'what'!",         &
                            '(can take only the values: "vertices", "neighbors" or "both")')
         go to 99
      end if

      ! Copy the input points coordinates into the Voronoi structure
      allocate( voronoi%n_xy(nb_pts,2) )
      do i = 1, nb_pts
         voronoi%n_xy(i,1) = x_ptr_vec(i)
         voronoi%n_xy(i,2) = y_ptr_vec(i)
      end do

      if( present(clip) ) then
         ! Get the clipping size from the last optional arg.
         x_min = clip(1) ; x_max = clip(2)
         y_min = clip(3) ; y_max = clip(4)
      else
         ! Compute the clipping size to be twice the size of the BBox of
         ! the points
         x_min = minval(x_ptr_vec) ; x_max = maxval(x_ptr_vec)
         y_min = minval(y_ptr_vec) ; y_max = maxval(y_ptr_vec)
         x_mid = ( x_min + x_max )/2.0d0 ; y_mid = ( y_min + y_max )/2.0d0
         x_width = ( x_max - x_min )/2.0d0 ; y_width = ( y_max - y_min )/2.0d0
         ! Safeguard: previous widths may be zero, even both!
         if( x_width == 0.0d0 .and. y_width == 0.0d0 ) then
            x_width = 0.5d0 ; y_width = 0.5d0
         else if( x_width == 0.0d0 ) then
            x_width = y_width
         else if( y_width == 0.0d0 ) then
            y_width = x_width
         end if

         ! Enlarge the BBox
         x_min = x_min - 2.0d0*x_width ; x_max = x_max + 2.0d0*x_width
         y_min = y_min - 2.0d0*y_width ; y_max = y_max + 2.0d0*y_width
      end if

      allocate( perm_sites(nb_pts) )

      call jcv_voronoi_prep( x_min, x_max, y_min, y_max,                &
                             nb_pts, x_ptr_vec, y_ptr_vec,              &
                             nb_edges, nb_neighbors,                    &
                             nb_vertices, perm_sites )

      if( to_lower(trim(what)) == "vertices" .or.                       &
          to_lower(trim(what)) == "both" ) then

         allocate( x_vertices(nb_vertices), y_vertices(nb_vertices) )
         allocate( ptr_edges(nb_pts,2), list_edges(nb_edges) )

         ! Pass only the address of ptr_edges, because this latter array
         ! is locally dimensionned as (:,2) but will be interpreted as
         ! an ordinary vector in the C routine jcv_voronoi_get_edges.
         status = jcv_voronoi_get_edges( ptr_edges(1,1), list_edges,    &
                                         x_vertices, y_vertices )

         ! Store the vertices
         allocate( voronoi%v_xy(nb_vertices,2) )
         do i = 1, nb_vertices
            voronoi%v_xy(i,1) = x_vertices(i)
            voronoi%v_xy(i,2) = y_vertices(i)
         end do

         ! Define the second column of ptr_edges
         do i = 1, nb_pts-1
            ptr_edges(i,2) = ptr_edges(i+1,1) - 1
         end do
         ptr_edges(nb_pts,2) = nb_edges

         ! Apply permutation to ptr_edges
         ptr_edges(perm_sites,:) = ptr_edges(:,:)

         ! Store the vertices list
         allocate( voronoi%vertices(nb_pts) )
         do i = 1, nb_pts
            j1 = ptr_edges(i,1)
            j2 = ptr_edges(i,2)
            n = j2 - j1 + 1
            allocate( voronoi%vertices(i)%list(n) )
            voronoi%vertices(i)%list(1:n) = list_edges(j1:j2)
         end do

      end if

      if( to_lower(trim(what)) == "neighbors" .or.                      &
          to_lower(trim(what)) == "both" ) then

         allocate( neighbors(nb_neighbors), ptr_neighbors(nb_pts,2) )

         ! Pass only the address of ptr_neighbors, because this latter array
         ! is locally dimensionned as (:,2) but will be interpreted as an
         ! ordinary vector in the C routine jcv_voronoi_get_neighbors.
         call jcv_voronoi_get_neighbors( ptr_neighbors(1,1), neighbors )

         ! Define the second column of ptr_neighbors
         do i = 1, nb_pts-1
            ptr_neighbors(i,2) = ptr_neighbors(i+1,1) - 1
         end do
         ptr_neighbors(nb_pts,2) = nb_neighbors

         ! Apply permutation
         ptr_neighbors(perm_sites,:) = ptr_neighbors(:,:)

         ! Store the neighbors list
         allocate( voronoi%neighbors(nb_pts) )
         do i = 1, nb_pts
            j1 = ptr_neighbors(i,1)
            j2 = ptr_neighbors(i,2)
            n = j2 - j1 + 1
            allocate( voronoi%neighbors(i)%list(n) )
            voronoi%neighbors(i)%list(1:n) = neighbors(j1:j2)
         end do

      end if

      call jcv_voronoi_free

      if( to_lower(trim(what)) == "vertices" ) then
         voronoi%init = 1
      else if( to_lower(trim(what)) == "neighbors" ) then
         voronoi%init = 2
      else ! "both"
         voronoi%init = 3
      end if

      voronoi%status_temporary = .true.

 99   continue

      call msFreePointer( x_in, x_ptr_vec )
      call msFreePointer( y_in, y_ptr_vec )

      call msFreeArgs( x_in, y_in )
      call msAutoRelease( x_in, y_in )

#endif
   end function mfVoronoi
!_______________________________________________________________________
!
   subroutine msRelease_mfVoronoiStruct( voronoi )

      type(mfVoronoiStruct) :: voronoi
      !------ API end ------

#ifdef _DEVLP
   !------ end of declarations -- execution starts hereafter  ------

      if( voronoi%init == 0 ) return

      deallocate( voronoi%n_xy )

      if( voronoi%init == 1 .or. voronoi%init == 3 ) then

         deallocate( voronoi%v_xy )

         call msRelease( voronoi%vertices )
         deallocate( voronoi%vertices )

      end if

      if( voronoi%init == 2 .or. voronoi%init == 3 ) then

         call msRelease( voronoi%neighbors )
         deallocate( voronoi%neighbors )

      end if

      voronoi%init = 0

#endif
   end subroutine msRelease_mfVoronoiStruct
!_______________________________________________________________________
!
   subroutine msPrintVoronoi( voronoi )


      type(mfVoronoiStruct) :: voronoi
      !------ API end ------

#ifdef _DEVLP

      integer :: i, nn, nv
   !------ end of declarations -- execution starts hereafter  ------

      if( voronoi%init == 0 ) then
         print *, "(Muesli:) msPrintVoronoi: argument is not initialized!"
         return
      end if

      nn = size(voronoi%n_xy,1)
      nv = size(voronoi%v_xy,1)

      print "(/,A)", "         ┏━━━━━━━━━━━━━━━━━━━┓"
      print "(A)",   "         ┃ Voronoi Structure ┃"
      print "(A,/)", "         ┗━━━━━━━━━━━━━━━━━━━┛"
      print "(A,I0)", "  nn: number of nodes    = ", nn
      print "(/,A)", "  Points coordinates:"
      do i = 1, nn
         print "(4X,I0,': ',ES11.4,2X,ES11.4)", i, voronoi%n_xy(i,:)
      end do
      if( voronoi%init == 1 .or. voronoi%init == 3 ) then
         print "(/,A,I0)","  nv: number of vertices = ", nv
         print "(/,A)", "  Vertices coordinates:"
         do i = 1, nv
            print "(4X,I0,': ',ES11.4,2X,ES11.4)", i, voronoi%v_xy(i,:)
         end do
         print "(/,A)", "  List of vertices, by node:"
         do i = 1, nn
            print "(4X,I0,': ',20(I0,2X))", i, voronoi%vertices(i)%list
         end do
      end if
      if( voronoi%init == 2 .or. voronoi%init == 3 ) then
         print "(/,A)", "  List of neighbors, by node:"
         do i = 1, nn
            print "(4X,I0,': ',20(I0,2X))", i, voronoi%neighbors(i)%list
         end do
      end if
#endif
   end subroutine msPrintVoronoi

