! f90 include file

!_______________________________________________________________________
!
   function mfSpline( x, y,                                             &
                      xi,                                               &
                      BC_type_1, BC_val_1, BC_type_2, BC_val_2,         &
                      periodic )        &
   result( y_sec )

      type(mfArray) :: x, y
      type(mfArray), optional :: xi
      integer,              optional :: BC_type_1
      real(kind=MF_DOUBLE), optional :: BC_val_1
      integer,              optional :: BC_type_2
      real(kind=MF_DOUBLE), optional :: BC_val_2
      logical             , optional :: periodic
      type(mfArray) :: y_sec
      !------ API end ------
#ifdef _DEVLP

      ! Cubic spline interpolation.

      ! if 'xi' is present:
      !   1) coefficients of the spline are computed,
      !   2) interpolation is done at 'xi' and the values are returned
      !      in 'y_sec' ('y_sec' shape is the same than the 'xi' shape)
      !
      ! else:
      !   we compute only the spline coefficients, which are returned
      !   in 'y_sec' ('y_sec' shape is the same than the 'x' shape)

      ! Optionally, B.C. of various (and mixed) types may be specified
      ! at the first and second end of the interval.
      !   BC_type_1 = 1         y'   = BC_val_1       at first end
      !             = 2         y''  = BC_val_1
      !             = 3         y''' = 0
      !
      !   BC_type_2 = 1         y'   = BC_val_2       at second end
      !             = 2         y''  = BC_val_2
      !             = 3         y''' = 0
      !
      !   periodic = .true.   for periodic boundary conditions (default: FALSE)

      ! 'x', 'y' and 'xi' may be tempo mfArrays

      real(kind=MF_DOUBLE), pointer :: x_ptr_vec(:), y_ptr_vec(:)
      real(kind=MF_DOUBLE), pointer :: xi_ptr_vec(:), y_sec_ptr_vec(:)
      real(kind=MF_DOUBLE), allocatable :: y2(:)
      integer :: n, nn, i, status
      logical :: periodic_BC

      integer :: k_beg, k_end
      real(kind=MF_DOUBLE) :: BC_beg, BC_end

      integer :: xdim

      character(len=*), parameter :: ROUTINE_NAME = "mfSpline"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y )
      if( present(xi) ) then
         call msInitArgs( xi )
      end if

      if( mfIsEmpty(x) .or. mfIsEmpty(y) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' or 'y': real arrays required!" )
         go to 99
      end if

      if( present(xi) ) then
         if( xi%data_type /= MF_DT_DBLE ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'xi': real arrays required!" )
            go to 99
         end if
      end if

      if( x%shape(1) == 1 ) then
         xdim = 2
      else if( x%shape(2) == 1 ) then
         xdim = 1
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a vector!" )
         go to 99
      end if

      if( .not. all(x%shape == y%shape) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' and 'y' must have the same shape!" )
         go to 99
      end if

      if( present(periodic) ) then
         periodic_BC = periodic
      else
         periodic_BC = .false.
      end if

      if( periodic_BC ) then
         if( present(BC_type_1) .or. present(BC_val_1) .or.             &
             present(BC_type_2) .or. present(BC_val_2) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "for a periodic spline, B.C. arguments cannot be present!" )
            go to 99
         end if
         if( x%shape(xdim) < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "At least 3 points are needed to construct a periodic spline!" )
            go to 99
         end if
      else
         if( x%shape(xdim) < 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "At least 2 points are needed to construct a spline!" )
            go to 99
         end if
      end if

      if( present(BC_type_1) ) then
         if( BC_type_1 < 1 .or. 3 < BC_type_1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "bad arg. value for 'BC_type_1'!" )
            go to 99
         end if
         k_beg = BC_type_1
         if( present(BC_val_1) ) then
            BC_beg = BC_val_1
         else
            BC_beg = 0.0d0
         end if
      else
         if( present(BC_val_1) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'BC_val_1' cannot be present without 'BC_type_1'!" )
            go to 99
         end if
         k_beg = 2
         BC_beg = 0.0d0
      end if

      if( present(BC_type_2) ) then
         if( BC_type_2 < 1 .or. 3 < BC_type_2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "bad arg. value for 'BC_type_2'!" )
            go to 99
         end if
         k_end = BC_type_2
         if( present(BC_val_2) ) then
            BC_end = BC_val_2
         else
            BC_end = 0.0d0
         end if
      else
         if( present(BC_val_2) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'BC_val_2' cannot be present without 'BC_type_2'!" )
            go to 99
         end if
         k_end = 2
         BC_end = 0.0d0
      end if

      if( xdim == 1 ) then
         x_ptr_vec => x%double(:,1)
         y_ptr_vec => y%double(:,1)
      else ! xdim = 2
         x_ptr_vec => x%double(1,:)
         y_ptr_vec => y%double(1,:)
      end if

      if( present(xi) ) then

         y_sec%data_type = MF_DT_DBLE
         y_sec%shape = xi%shape
         allocate( y_sec%double(y_sec%shape(1),y_sec%shape(2)) )

         call msPointer( y_sec, y_sec_ptr_vec, no_crc=.true. )

         n = size( x_ptr_vec )
         allocate( y2(n) )

         if( periodic_BC ) then
            call cub_spl( x_ptr_vec, y_ptr_vec, y2,                        &
                          period=.true. )
         else
            call cub_spl( x_ptr_vec, y_ptr_vec, y2,                        &
                          k_beg, BC_beg, k_end, BC_end )
         end if

         nn = size( xi%double )
         xi_ptr_vec => rank_2_to_1_real8( xi%double, nn )

         do i = 1, nn
            call int_spl( x_ptr_vec, y_ptr_vec, y2, any_t = .true.,     &
                          period=periodic_BC,                           &
                          t = xi_ptr_vec(i), y_0 = y_sec_ptr_vec(i) )
         end do

         deallocate( y2 )

         call msFreePointer( y_sec, y_sec_ptr_vec )

      else

         y_sec%data_type = MF_DT_DBLE
         y_sec%shape = x%shape
         allocate( y_sec%double(y_sec%shape(1),y_sec%shape(2)) )

         if( xdim == 1 ) then
            y_sec_ptr_vec => y_sec%double(:,1)
         else ! xdim = 2
            y_sec_ptr_vec => y_sec%double(1,:)
         end if

         if( periodic_BC ) then
            call cub_spl( x_ptr_vec, y_ptr_vec, y_sec_ptr_vec,          &
                          period=.true. )
         else
            call cub_spl( x_ptr_vec, y_ptr_vec, y_sec_ptr_vec,          &
                          k_beg, BC_beg, k_end, BC_end )
         end if

      end if

      if( mf_phys_units ) then
         if( present(xi) ) then
            ! 'y_sec' is the vector of interpolated values from 'y'
            y_sec%units(:) = y%units(:)
         else
            ! 'y_sec' is the vector of the spline coefficients;
            ! the physical dimension of these coefficients: second
            ! derivative of 'y' w.r.t. 'x'
            do i = 1, num_base_units
               call rational_mul( x%units(i), 2,                        &
                                  y_sec%units(i), status )
               if( status == 1 ) then
                  call PrintMessage( trim(ROUTINE_NAME), "E",           &
                                     "in processing physical units:",   &
                                     "integer overflow!" )
                  go to 99
               else if( status == -1 ) then
                  call PrintMessage( trim(ROUTINE_NAME), "E",           &
                                     "in processing physical units:",   &
                                     "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
                  go to 99
               end if
               call rational_sub( y%units(i), y_sec%units(i),           &
                                  y_sec%units(i), status )
               if( status == 1 ) then
                  call PrintMessage( trim(ROUTINE_NAME), "E",           &
                                     "in processing physical units:",   &
                                     "integer overflow!" )
                  go to 99
               else if( status == -1 ) then
                  call PrintMessage( trim(ROUTINE_NAME), "E",           &
                                     "in processing physical units:",   &
                                     "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
                  go to 99
               end if
            end do
         end if
      end if

      y_sec%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y )
      call msAutoRelease( x, y )
      if( present(xi) ) then
         call msFreeArgs( xi )
         call msAutoRelease( xi )
      end if

      call mf_restore_fpe( )

#endif
   end function mfSpline
!_______________________________________________________________________
!
   subroutine msSpline( out, x, y, weights,                             &
                        BC_type_1, BC_val_1, BC_type_2, BC_val_2,       &
                        periodic )

      type(mfArray)                  :: x, y, weights
      integer,              optional :: BC_type_1
      real(kind=MF_DOUBLE), optional :: BC_val_1
      integer,              optional :: BC_type_2
      real(kind=MF_DOUBLE), optional :: BC_val_2
      logical             , optional :: periodic
      type(mf_Out)                   :: out
      !------ API end ------
#ifdef _DEVLP

      ! Cubic spline smoothing.

      ! Optionally, B.C. of various (and mixed) types may be specified
      ! at the first and second end of the interval.
      !   BC_type_1 = 1         y' = BC_val_1       at first end
      !             = 2         y" = BC_val_1       at first end
      !
      !   BC_type_2 = 1         y' = BC_val_2       at second end
      !             = 2         y" = BC_val_2       at second end
      !

      ! 'weights' has usually the same shape than 'x' and 'y'; if it is
      ! a scalar, the same weight is applied for all points.
      ! Weights are always positive numbers.

      real(kind=MF_DOUBLE), pointer :: x_ptr_vec(:), y_ptr_vec(:), w_ptr_vec(:)
      real(kind=MF_DOUBLE), pointer :: y_smooth_ptr_vec(:), y_sec_ptr_vec(:)
      integer :: n, nn, i, status
      logical :: periodic_BC

      integer :: k_beg, k_end
      real(kind=MF_DOUBLE) :: BC_beg, BC_end

      integer :: xdim
      logical :: same_weight

      type(mfArray), pointer :: y_smooth, y_sec

      character(len=*), parameter :: ROUTINE_NAME = "msSpline"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y, weights )

      ! 2 out-args must be specified
      if( out%n /= 2 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "two output args required!",               &
                            "syntax is : call msSpline ( mfOut(y_smooth,y_sec), x, y, weights, ... )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, x )       .or. &
          .not. args_mfout_ok( out, y )       .or. &
          .not. args_mfout_ok( out, weights )         ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      y_smooth => out%ptr1
      y_sec => out%ptr2
      call msSilentRelease( y_smooth, y_sec )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' or 'y': real arrays required!" )
         go to 99
      end if

      if( weights%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'weights': real arrays required!" )
         go to 99
      end if

      if( x%shape(1) == 1 ) then
         xdim = 2
      else if( x%shape(2) == 1 ) then
         xdim = 1
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a vector!" )
         go to 99
      end if

      if( .not. all(x%shape == y%shape) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "msArray 'x' and 'y' must have the same shape!" )
         go to 99
      end if

      if( x%shape(xdim) == 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "At least 2 points are needed to construct the spline!" )
         go to 99
      end if

      if( .not. all(weights%double > 0.0d0) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'weights' must contains strictly positive values!" )
         go to 99
      end if
      if( mfIsScalar(weights) ) then
         same_weight = .true.
      else
         same_weight = .false.
         if( .not. all(x%shape == weights%shape) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "mfArray 'x' and 'weights' must have the same shape!" )
            go to 99
         end if
      end if

      if( present(periodic) ) then
         periodic_BC = periodic
      else
         periodic_BC = .false.
      end if

      if( periodic_BC ) then
         if( present(BC_type_1) .or. present(BC_val_1) .or.             &
             present(BC_type_2) .or. present(BC_val_2) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "for a periodic spline, B.C. arguments cannot be present!" )
            go to 99
         end if
         if( x%shape(xdim) < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "At least 3 points are needed to construct a periodic spline!" )
            go to 99
         end if
      else
         if( x%shape(xdim) < 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "At least 2 points are needed to construct a spline!" )
            go to 99
         end if
      end if

      if( present(BC_type_1) ) then
         if( BC_type_1 < 1 .or. 3 < BC_type_1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "bad arg. value for 'BC_type_1'!" )
            go to 99
         end if
         k_beg = BC_type_1
         if( present(BC_val_1) ) then
            BC_beg = BC_val_1
         else
            BC_beg = 0.0d0
         end if
      else
         if( present(BC_val_1) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'BC_val_1' cannot be present without 'BC_type_1'!" )
            go to 99
         end if
         k_beg = 2
         BC_beg = 0.0d0
      end if

      if( present(BC_type_2) ) then
         if( BC_type_2 < 1 .or. 3 < BC_type_2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "bad arg. value for 'BC_type_2'!" )
            go to 99
         end if
         k_end = BC_type_2
         if( present(BC_val_2) ) then
            BC_end = BC_val_2
         else
            BC_end = 0.0d0
         end if
      else
         if( present(BC_val_2) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'BC_val_2' cannot be present without 'BC_type_2'!" )
            go to 99
         end if
         k_end = 2
         BC_end = 0.0d0
      end if

      if( xdim == 1 ) then
         x_ptr_vec => x%double(:,1)
         y_ptr_vec => y%double(:,1)
      else ! xdim = 2
         x_ptr_vec => x%double(1,:)
         y_ptr_vec => y%double(1,:)
      end if

      if( same_weight ) then
         allocate( w_ptr_vec(size(x_ptr_vec)) )
         w_ptr_vec(:) = weights%double(1,1)
      else
         if( xdim == 1 ) then
            w_ptr_vec => weights%double(:,1)
         else ! xdim = 2
            w_ptr_vec => weights%double(1,:)
         end if
      end if

      y_smooth%data_type = MF_DT_DBLE
      y_smooth%shape = x%shape
      allocate( y_smooth%double(y_smooth%shape(1),y_smooth%shape(2)) )

      y_sec%data_type = MF_DT_DBLE
      y_sec%shape = x%shape
      allocate( y_sec%double(y_sec%shape(1),y_sec%shape(2)) )

      if( xdim == 1 ) then
         y_smooth_ptr_vec => y_smooth%double(:,1)
         y_sec_ptr_vec => y_sec%double(:,1)
      else ! xdim = 2
         y_smooth_ptr_vec => y_smooth%double(1,:)
         y_sec_ptr_vec => y_sec%double(1,:)
      end if

      if( periodic_BC ) then
         call cub_spl( x_ptr_vec, y_ptr_vec, y_sec_ptr_vec,                &
                       weight=w_ptr_vec, y_fit=y_smooth_ptr_vec,           &
                       period=periodic_BC )
      else
         call cub_spl( x_ptr_vec, y_ptr_vec, y_sec_ptr_vec,                &
                       k_beg, BC_beg, k_end, BC_end,                       &
                       weight=w_ptr_vec, y_fit=y_smooth_ptr_vec )
      end if

      if( same_weight ) then
         deallocate( w_ptr_vec )
      end if

      if( mf_phys_units ) then
         ! 'y_smooth' is the vector of smoothed 'y' values
         y_smooth%units(:) = y%units(:)
         ! 'y_sec' is the vector of the spline coefficients;
         ! the physical dimension of these coefficients: second
         ! derivative of 'y' w.r.t. 'x'
         do i = 1, num_base_units
            call rational_mul( x%units(i), 2,                           &
                               y_sec%units(i), status )
            if( status == 1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               go to 99
            end if
            call rational_sub( y%units(i), y_sec%units(i),              &
                               y_sec%units(i), status )
            if( status == 1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               go to 99
            end if
         end do
      end if

 99   continue

      call msFreeArgs( x, y, weights )
      call msAutoRelease( x, y, weights )

      call mf_restore_fpe( )

#endif
   end subroutine msSpline
