! f90 include file

!_______________________________________________________________________
!
   function mfPolyVal( p, x ) result( out )

      type(mfArray) :: p, x
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! evaluates polynomial 'p' at points inside 'x'

      ! 'p' is a vector of length k+1 whose elements are the
      ! coefficients of the polynomial in descending powers.
      !
      ! out = p(1)*x^k + p(2)*x^(k-1) + ... + p(k)*x + p(k+1)

      ! 'p' may have any physical unit, which will be inherited by 'out'
      !
      ! 'x' must be dimensionless

      integer :: idim, k_max, m, n, k, j, i, status

      character(len=*), parameter :: ROUTINE_NAME = "mfPolyVal"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( p, x )

      if( mfIsEmpty(x) ) then
         go to 99
      end if

      ! 'x' cannot be sparse
      if( mfIsSparse(x) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' cannot be sparse!" )
         go to 99
      end if

      ! 'x' must be numeric
      if( .not. mfIsNumeric(x) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be numeric!" )
         go to 99
      end if
      m = x%shape(1)
      n = x%shape(2)

      if( mfIsEmpty(p) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "cannot apply an empty polynomial!" )
         go to 99
      end if

      ! 'p' cannot be sparse
      if( mfIsSparse(p) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "polynomial 'p' cannot be sparse!" )
         go to 99
      end if

      ! 'p' must be numeric
      if( .not. mfIsNumeric(p) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "polynomial 'p' must be numeric!" )
         go to 99
      end if

      ! 'p' must be a vector
      if( p%shape(1) == 1 ) then
         idim = 2
      else if( p%shape(2) == 1 ) then
         idim = 1
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "polynomial 'p' must be a vector mfArray!" )
         go to 99
      end if
      k_max = size(p) - 1

      out%shape = [ m, n ]
      if( p%data_type == MF_DT_DBLE .and.                               &
          x%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_DBLE
         allocate( out%double(m,n) )

         if( idim == 1 ) then
            do j = 1, n
               do i = 1, m
                  out%double(i,j) = p%double(k_max+1,1)
                  do k = 1, k_max
                     out%double(i,j) = out%double(i,j) +                &
                              p%double(k_max-k+1,1)*x%double(i,j)**k
                  end do
               end do
            end do
         else
            do j = 1, n
               do i = 1, m
                  out%double(i,j) = p%double(1,k_max+1)
                  do k = 1, k_max
                     out%double(i,j) = out%double(i,j) +                &
                              p%double(1,k_max-k+1)*x%double(i,j)**k
                  end do
               end do
            end do
         end if
      else if( p%data_type == MF_DT_DBLE .and.                          &
               x%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         if( idim == 1 ) then
            do j = 1, n
               do i = 1, m
                  out%cmplx(i,j) = p%double(k_max+1,1)
                  do k = 1, k_max
                     out%cmplx(i,j) = out%cmplx(i,j) +                  &
                              p%double(k_max-k+1,1)*x%cmplx(i,j)**k
                  end do
               end do
            end do
         else
            do j = 1, n
               do i = 1, m
                  out%cmplx(i,j) = p%double(1,k_max+1)
                  do k = 1, k_max
                     out%cmplx(i,j) = out%cmplx(i,j) +                  &
                              p%double(1,k_max-k+1)*x%cmplx(i,j)**k
                  end do
               end do
            end do
         end if
      else if( p%data_type == MF_DT_CMPLX .and.                         &
               x%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         if( idim == 1 ) then
            do j = 1, n
               do i = 1, m
                  out%cmplx(i,j) = p%cmplx(k_max+1,1)
                  do k = 1, k_max
                     out%cmplx(i,j) = out%cmplx(i,j) +                  &
                              p%cmplx(k_max-k+1,1)*x%double(i,j)**k
                  end do
               end do
            end do
         else
            do j = 1, n
               do i = 1, m
                  out%cmplx(i,j) = p%cmplx(1,k_max+1)
                  do k = 1, k_max
                     out%cmplx(i,j) = out%cmplx(i,j) +                  &
                              p%cmplx(1,k_max-k+1)*x%double(i,j)**k
                  end do
               end do
            end do
         end if
      else if( p%data_type == MF_DT_CMPLX .and.                         &
               x%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         if( idim == 1 ) then
            do j = 1, n
               do i = 1, m
                  out%cmplx(i,j) = p%cmplx(k_max+1,1)
                  do k = 1, k_max
                     out%cmplx(i,j) = out%cmplx(i,j) +                  &
                              p%cmplx(k_max-k+1,1)*x%cmplx(i,j)**k
                  end do
               end do
            end do
         else
            do j = 1, n
               do i = 1, m
                  out%cmplx(i,j) = p%cmplx(1,k_max+1)
                  do k = 1, k_max
                     out%cmplx(i,j) = out%cmplx(i,j) +                  &
                              p%cmplx(1,k_max-k+1)*x%cmplx(i,j)**k
                  end do
               end do
            end do
         end if
      end if

      if( mf_phys_units ) then

         ! verifying the physical dimension
         call verif_adim( x%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical unit of 'x'",              &
                               "must be dimensionless!" )
            go to 99
         end if

         out%units(:) = p%units(:)

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( p, x )

      call msAutoRelease( p, x )

#endif
   end function mfPolyVal
