! f90 include file

!_______________________________________________________________________
!
   function mfPolyFit( x, y, n ) result( p )

      type(mfArray) :: x, y
      integer, intent(in) :: n
      type(mfArray) :: p
      !------ API end ------
#ifdef _DEVLP

      ! Fit Polynomial to real data
      !
      ! (x,y) are coordinates of data, under column vectors
      !
      ! n : degree of the fitting polynomial

      ! For low condition numbers : Normal Equations Method
      !                      else : QR decomposition

      ! 'y' may have any physical unit, which will be inherited by 'p'
      !
      ! 'x' must be dimensionless

      type(mfArray) :: M, MM
      type(mfArray) :: Q, R
      integer :: status

      character(len=*), parameter :: ROUTINE_NAME = "mfPolyFit"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, y )

      if( mfIsEmpty(x) ) then
         go to 99
      end if

      ! 'x' and 'y' must be numeric
      if( .not. mfIsNumeric(x) .or. .not. mfIsNumeric(y) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArrays 'x' and 'y' must be numeric!" )
         go to 99
      end if

      ! 'x' and 'y' must be dense
      if( mfIsSparse(x) .or. mfIsSparse(y) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArrays 'x' and 'y' must have dense storage!" )
         go to 99
      end if

      ! 'x' must be a column vector
      if( x%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' must be a column vector!" )
         go to 99
      end if

      ! 'x' and 'y' must have same shape
      if( any(x%shape /= y%shape) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' and 'y' must have the same shape!" )
         go to 99
      end if

      ! Vandermonde matrix
      call msAssign( M, mfVander( x, n+1 ) )
      call msAssign( MM, mfMul(M,M,transp=1) )

      if( mfDble(mfRCond(MM)) < MF_EPS*10. ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "Problem is nearly singular, so the QR decomposition", &
                            "will be used instead of Normal Equations method." )
         call msQR( mfOut(Q,R), M )
         call msAssign( p, mfLDiv( R, mfMul(Q,y,transp=1) ) )
         call msRelease(Q, R )
      else
         call msAssign( p, mfLDiv( MM, mfMul(M,y,transp=1) ) )
      end if

      call msRelease( M, MM )

      if( mf_phys_units ) then

         ! verifying the physical dimension
         call verif_adim( x%units, status=status )
         if( status /= 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                               "the physical unit of 'x'",              &
                               "must be dimensionless!" )
            go to 99
         end if

         p%units(:) = y%units(:)

      end if

      p%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y )

      call msAutoRelease( x, y )

#endif
   end function mfPolyFit
!_______________________________________________________________________
!
   subroutine msPolyFit( out, x, y, n )

      type(mfArray) :: x, y
      integer, intent(in) :: n
      type(mf_Out)        :: out
      !------ API end ------
#ifdef _DEVLP

      ! Fit Polynomial to real data
      !
      ! Subroutine version of 'mfPolyFit' which also returns 'normr',
      ! the norm of the residuals, and optionally 'r2' the correlation
      ! coefficient.

      type(mfArray) :: M, MM
      type(mfArray) :: Q, R
      integer :: status

      type(mfArray), pointer :: p, normr, r2

      character(len=*), parameter :: ROUTINE_NAME = "msPolyFit"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x, y )

      ! 2 or 3 out-args must be specified
      if( out%n < 2 .or. 3 < out%n ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "two or three output args required!",      &
                            "syntax is : call msPolyFit ( mfOut(p,normr[,r2]), x, y, n )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, x ) .or. &
          .not. args_mfout_ok( out, y )      ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      p => out%ptr1
      normr => out%ptr2
      call msSilentRelease( p, normr )
      if( out%n == 3 ) then
         r2 => out%ptr3
         call msSilentRelease( r2 )
      end if

      if( mfIsEmpty(x) ) then
         go to 99
      end if

      ! 'x' and 'y' must be numeric
      if( .not. mfIsNumeric(x) .or. .not. mfIsNumeric(y) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArrays 'x' and 'y' must be numeric!" )
         go to 99
      end if

      ! 'x' and 'y' must be dense
      if( mfIsSparse(x) .or. mfIsSparse(y) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArrays 'x' and 'y' must have dense storage!" )
         go to 99
      end if

      ! 'x' must be a column vector
      if( x%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' must be a column vector!" )
         go to 99
      end if

      ! 'x' and 'y' must have same shape
      if( any(x%shape /= y%shape) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "'x' and 'y' must have the same shape!" )
         go to 99
      end if

      ! Vandermonde matrix
      call msAssign( M, mfVander( x, n+1 ) )
      call msAssign( MM, mfMul(M,M,transp=1) )

      if( mfDble(mfRCond(MM)) < MF_EPS*10. ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "Problem is nearly singular, so the QR decomposition", &
                            "will be used instead of Normal Equations method." )
         call msQR( mfOut(Q,R), M )
         call msAssign( p, mfLDiv( R, mfMul(Q,y,transp=1) ) )
         call msRelease(Q, R )
      else
         call msAssign( p, mfLDiv( MM, mfMul(M,y,transp=1) ) )
      end if
      call msAssign( normr, mfNorm(mfMul(M,p)-y) )

      if( out%n == 3 ) then
         call msAssign( r2, 1.0d0 - normr**2/mfVar(y)/y%shape(1) )
      end if

      call msRelease( M, MM )

      if( mf_phys_units ) then

         ! verifying the physical dimension
         call verif_adim( x%units, status=status )
         if( status /= 0 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                               "the physical unit of 'x'",              &
                               "must be dimensionless!" )
            go to 99
         end if

         p%units(:) = y%units(:)

      end if

 99   continue

      call msFreeArgs( x, y )

      call msAutoRelease( x, y )

#endif
   end subroutine msPolyFit
