! f90 include file

!_______________________________________________________________________
!
   function mfInterp2_real( x, y, z, xi, yi, order, extrapol ) result( zi )

      type(mfArray)                              :: x, y, z
      type(mfArray)                              :: xi, yi
      integer,              intent(in), optional :: order
      real(kind=MF_DOUBLE), intent(in), optional :: extrapol
      type(mfArray)                              :: zi
      !------ API end ------
#ifdef _DEVLP

      ! x, y, z : must be matrices, representing data (z) on a
      !           rectangular 2D grid (x,y), having strictly monotonous
      !           values.
      !
      ! (xi,yi) must have the same shape (scalars, vectors, or matrices);
      ! zi will have the same shape as xi and yi.
      !
      ! if (xi,yi) is outside of the range of x and y:
      !    * zi is set to MF_NAN (if extrapol is not present)
      !    * zi is set to extrapol (if extrapol is present)
      !
      ! interpolation used:
      !    order=0 -> nearest:     1-point stencil
      !    order=1 -> bilinear:    4-point stencil (default)
      !    order=2 -> biquadratic: 9-point stencil
      !    order=3 -> bicubic:    16-point stencil

      integer :: i, j, ni, nj, n1, n2, ii, jj, ii_save, jj_save, status
      real(kind=MF_DOUBLE) :: xx, yy, zz, dx, dy
      real(kind=MF_DOUBLE) :: z1, z2, z3, z4
      real(kind=MF_DOUBLE) :: x1, x2, y1, y2, z11, z12, z21, z22
      real(kind=MF_DOUBLE) :: A, B, C, D

      integer :: i_order, i_nearest, j_nearest
      real(kind=MF_DOUBLE) :: dist2, new_dist2, XM, XP, YM, YP
      real(kind=MF_DOUBLE) :: FX0Y0, FX0YM, FX0YP,                      &
                              FXMY0, FXMYM, FXMYP,                      &
                              FXPY0, FXPYM, FXPYP
      real(kind=MF_DOUBLE) :: A1, A2, A3, A4, A5, A6, A7, A8, A9,       &
                              new_x, new_y

      character(len=*), parameter :: ROUTINE_NAME = "mfInterp2"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y, z, xi, yi )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) .or. mfIsEmpty(z) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE .or. &
          z%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x%shape(1) == 1 .or. x%shape(2) == 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a matrix!" )
         go to 99
      end if

      if( y%shape(1) == 1 .or. y%shape(2) == 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y' must be a matrix!" )
         go to 99
      end if

      if( z%shape(1) == 1 .or. z%shape(2) == 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'z' must be a matrix!" )
         go to 99
      end if

      if( x%shape(1) /= y%shape(1) .or. x%shape(1) /= z%shape(1) .or.   &
          x%shape(2) /= y%shape(2) .or. x%shape(2) /= z%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x', 'y' and 'z' must have the same shape!" )
         go to 99
      end if

      if( present(order) ) then
         i_order = order
         if( i_order < 0 .or. 3 < i_order ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                               "'order' optional arg must be 0, 1, 2 or 3!" )
            go to 99
         end if
      else
         i_order = 1
      end if

      n1 = size( x%double, 1 )
      n2 = size( x%double, 2 )

      if( i_order == 0 ) then
         if( n1 < 1 .or. n2 < 1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', 'y' and 'z'!" )
            go to 99
         end if
      else if( i_order == 1 ) then
         if( n1 < 2 .or. n2 < 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', 'y' and 'z'!" )
            go to 99
         end if
      else if( i_order == 2 ) then
         if( n1 < 3 .or. n2 < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', 'y' and 'z'!",   &
                               "(for biquadratic interpolation, 3 points are required in each direction)")
            go to 99
         end if
      else ! i_order = 3
         if( n1 < 4 .or. n2 < 4 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', 'y' and 'z'!",   &
                               "(for bicubic interpolation, 4 points are required in each direction)")
            go to 99
         end if
      end if

      ! verifying that x(:,:) has constant values along its 1st dim.
      do jj = 1, n2
         if( minval(x%double(:,jj)) /= maxval(x%double(:,jj)) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'x' data are not constant along the 1st dim.!" )
            go to 99
         end if
      end do

      ! verifying that x(:,:) has strictly monotonous values along its 2nd dim.
      dx = x%double(1,2) - x%double(1,1)
      do jj = 3, n2
         if( dx*(x%double(1,jj) - x%double(1,jj-1)) <= 0.0d0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'x' data are not in strictly monotonous order!" )
            go to 99
         end if
      end do
      ! now, we are sure that dx is not zero
      dx = sign(1.0d0,dx) ! +1.0 or -1.0

      ! verifying that y(:,:) has constant values along its 2nd dim.
      do ii = 1, n1
         if( minval(y%double(ii,:)) /= maxval(y%double(ii,:)) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'y' data are not constant along the 2nd dim.!" )
            go to 99
         end if
      end do

      ! verifying that y(:,:) has strictly monotonous values along its 1st dim.
      dy = y%double(2,1) - y%double(1,1)
      do ii = 3, n1
         if( dy*(y%double(ii,1) - y%double(ii-1,1)) <= 0.0d0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'y' data are not in strictly monotonous order!" )
            go to 99
         end if
      end do
      ! now, we are sure that dy is not zero
      dy = sign(1.0d0,dy) ! +1.0 or -1.0

      ni = size( xi%double, 1 )
      nj = size( xi%double, 2 )

      call msAssign( zi, mfZeros(ni,nj) )

      ii_save = 0
      jj_save = 0
      do j = 1, nj
         do i = 1, ni

            xx = xi%double(i,j)
            yy = yi%double(i,j)

            ! find the rectangle where (xx,yy) belongs
            if( dx*(xx - x%double(1,1)) < 0.0d0 .or.                    &
                dx*(x%double(n1,n2) - xx) < 0.0d0 .or.                  &
                dy*(yy - y%double(1,1)) < 0.0d0 .or.                    &
                dy*(y%double(n1,n2) - yy) < 0.0d0 ) then

               if( present(extrapol) ) then
                  call PrintMessage( trim(ROUTINE_NAME), "I",           &
                                     "'(xi,yi)' out of range: 'zi' set to a user-provided value" )
                  zi%double(i,j) = extrapol
               else
                  call PrintMessage( trim(ROUTINE_NAME), "W",           &
                                     "'(xi,yi)' out of range: 'zi' set to NaN" )
                  zi%double(i,j) = MF_NAN
               end if
               cycle

            else

               if( jj_save /= 0 ) then
                  ! use old position to start the search
                  if( dx*(xx - x%double(1,jj_save)) >= 0.0d0 ) then
                     do jj = jj_save, n2-1
                        if( dx*(xx - x%double(1,jj+1)) <= 0.0d0 ) then
                           exit
                        end if
                     end do
                  else
                     do jj = jj_save-1, 1, -1
                        if( dx*(xx - x%double(1,jj)) >= 0.0d0 ) then
                           exit
                        end if
                     end do
                  end if
                  if( dy*(yy - y%double(ii_save,1)) >= 0.0d0 ) then
                     do ii = ii_save, n1-1
                        if( dy*(yy - y%double(ii+1,1)) <= 0.0d0 ) then
                           exit
                        end if
                     end do
                  else
                     do ii = ii_save-1, 1, -1
                        if( dy*(yy - y%double(ii,1)) >= 0.0d0 ) then
                           exit
                        end if
                     end do
                  end if
               else
                  do jj = 1, n2-1
                     if( dx*(xx - x%double(1,jj+1)) <= 0.0d0 ) then
                        exit
                     end if
                  end do
                  do ii = 1, n1-1
                     if( dy*(yy - y%double(ii+1,1)) <= 0.0d0 ) then
                        exit
                     end if
                  end do
               end if

               ! saving for next search
               ii_save = ii
               jj_save = jj

            end if

            if( i_order == 0 ) then

               ! nearest interpolation in the rectangle
               !   ( [ii, ii+1], [jj, jj+1] )
               if( (xx-x%double(1,jj)) < (x%double(1,jj+1)-xx) ) then
                  if( (yy-y%double(ii,1)) < (y%double(ii+1,1)-yy) ) then
                     zi%double(i,j) = z%double(ii,jj)
                  else
                     zi%double(i,j) = z%double(ii+1,jj)
                  end if
               else
                  if( (yy-y%double(ii,1)) < (y%double(ii+1,1)-yy) ) then
                     zi%double(i,j) = z%double(ii,jj+1)
                  else
                     zi%double(i,j) = z%double(ii+1,jj+1)
                  end if
               end if

            else if( i_order == 1 ) then

               ! bilinear interpolation in the rectangle
               !   ( [ii, ii+1], [jj, jj+1] )
               !
               !  z(x,y) = A*x + B*y + C*x*y + D
               !
               ! avec translation de P11 à l'origine :
               x2 = x%double(1,jj+1) - x%double(1,jj)
               y2 = y%double(ii+1,1) - y%double(ii,1)
               z11 = z%double(ii,jj)
               z21 = z%double(ii,jj+1)
               z12 = z%double(ii+1,jj)
               z22 = z%double(ii+1,jj+1)

               D = z11
               A = (z21-D)/x2
               B = (z12-D)/y2
               C = (z22 - A*x2 - B*y2 - D)/(x2*y2)

               zi%double(i,j) = A*(xx-x%double(1,jj))                   &
                              + B*(yy-y%double(ii,1))                   &
                              + C*(xx-x%double(1,jj))*(yy-y%double(ii,1)) &
                              + D

            else if( i_order == 2 ) then

               ! find the nearest point to (xx,yy)
               dist2 = (xx-x%double(ii,jj))**2 + (yy-y%double(ii,jj))**2
               i_nearest = ii
               j_nearest = jj

               new_dist2 = (xx-x%double(ii+1,jj))**2 +                  &
                           (yy-y%double(ii+1,jj))**2
               if( new_dist2 < dist2 ) then
                  dist2 = new_dist2
                  i_nearest = ii + 1
                  j_nearest = jj
               end if

               new_dist2 = (xx-x%double(ii,jj+1))**2 +                  &
                           (yy-y%double(ii,jj+1))**2
               if( new_dist2 < dist2 ) then
                  dist2 = new_dist2
                  i_nearest = ii
                  j_nearest = jj + 1
               end if

               new_dist2 = (xx-x%double(ii+1,jj+1))**2 +                &
                           (yy-y%double(ii+1,jj+1))**2
               if( new_dist2 < dist2 ) then
                  i_nearest = ii + 1
                  j_nearest = jj + 1
               end if

               ii = i_nearest
               jj = j_nearest

               ! check if the central point (ii,jj) of
               ! our 9-point stencil is on the boundary
               if( ii == 1 ) then
                  ! North boundary
                  ii = 2
                  if( jj == 1 ) then
                     ! NW corner
                     jj = 2
                  else if( jj == n2 ) then
                     ! NE corner
                     jj = n2 - 1
                  end if
               else if( ii == n1 ) then
                  ! South boundary
                  ii = n1 - 1
                  if( jj == 1 ) then
                     ! SW corner
                     jj = 2
                  else if( jj == n2 ) then
                     ! SE corner
                     jj = n2 - 1
                  end if
               else
                  if( jj == 1 ) then
                     ! West boundary
                     jj = 2
                  else if( jj == n2 ) then
                     ! East boundary
                     jj = n2 - 1
                  else
                     ! interior point
                  end if
               end if

               XM = x%double(ii,jj-1) - x%double(ii,jj)
               XP = x%double(ii,jj+1) - x%double(ii,jj)
               YM = y%double(ii-1,jj) - y%double(ii,jj)
               YP = y%double(ii+1,jj) - y%double(ii,jj)
               FXMYM = z%double(ii-1,jj-1)
               FX0YM = z%double(ii-1,jj)
               FXPYM = z%double(ii-1,jj+1)
               FXMY0 = z%double(ii,  jj-1)
               FX0Y0 = z%double(ii,  jj)
               FXPY0 = z%double(ii,  jj+1)
               FXMYP = z%double(ii+1,jj-1)
               FX0YP = z%double(ii+1,jj)
               FXPYP = z%double(ii+1,jj+1)
               A1 = 1.0d0/ym/xm*(FX0Y0*xm*ym-FX0Y0*xm*yp-FX0Y0*xp*ym+FX0Y0*xp*yp &
                  + FX0YM*xm*yp-FX0YM*xp*yp-FX0YP*xm*ym+FX0YP*xp*ym+FXMY0*xp*ym &
                  - FXMY0*xp*yp+FXMYM*xp*yp-FXMYP*xp*ym-FXPY0*xm*ym+FXPY0*xm*yp &
                  - FXPYM*xm*yp+FXPYP*xm*ym)/xp/yp/(xm*ym-xm*yp-xp*ym+xp*yp)
               A2 = -(FX0Y0*xm**2*ym-FX0Y0*xm**2*yp-FX0Y0*xp**2*ym+FX0Y0*xp**2*yp &
                  + FX0YM*xm**2*yp-FX0YM*xp**2*yp-FX0YP*xm**2*ym+FX0YP*xp**2*ym &
                  + FXMY0*xp**2*ym-FXMY0*xp**2*yp+FXMYM*xp**2*yp-FXMYP*xp**2*ym &
                  - FXPY0*xm**2*ym+FXPY0*xm**2*yp-FXPYM*xm**2*yp+FXPYP*xm**2*ym) &
                  / xm/xp/ym/(xm-xp)/(ym-yp)/yp
               A3 = (FX0Y0*ym-FX0Y0*yp+FX0YM*yp-FX0YP*ym)/ym/yp/(ym-yp)
               A4 = -(FX0Y0*xm*ym**2-FX0Y0*xm*yp**2-FX0Y0*ym**2*xp+FX0Y0*yp**2*xp &
                  + FX0YM*xm*yp**2-FX0YM*yp**2*xp-FX0YP*xm*ym**2+FX0YP*ym**2*xp &
                  + FXMY0*ym**2*xp-FXMY0*yp**2*xp+FXMYM*yp**2*xp-FXMYP*ym**2*xp &
                  - FXPY0*xm*ym**2+FXPY0*xm*yp**2-FXPYM*xm*yp**2+FXPYP*xm*ym**2)&
                  / xm/ym/yp/(ym-yp)/(xm-xp)/xp
               A5 = (FX0Y0*xm**2*ym**2-FX0Y0*xm**2*yp**2-FX0Y0*xp**2*ym**2 &
                  + FX0Y0*xp**2*yp**2+FX0YM*xm**2*yp**2-FX0YM*xp**2*yp**2-FX0YP*xm**2*ym**2 &
                  + FX0YP*xp**2*ym**2+FXMY0*xp**2*ym**2-FXMY0*xp**2*yp**2+FXMYM*xp**2*yp**2 &
                  - FXMYP*xp**2*ym**2-FXPY0*xm**2*ym**2+FXPY0*xm**2*yp**2-FXPYM*xm**2*yp**2 &
                  + FXPYP*xm**2*ym**2)/xm/xp/ym/yp/(xm*ym-xm*yp-xp*ym+xp*yp)
               A6 = -(FX0Y0*ym**2-FX0Y0*yp**2+FX0YM*yp**2-FX0YP*ym**2)/ym/yp/(ym-yp)
               A7 = (FX0Y0*xm-FX0Y0*xp+FXMY0*xp-FXPY0*xm)/xm/xp/(xm-xp)
               A8 = -(FX0Y0*xm**2-FX0Y0*xp**2+FXMY0*xp**2-FXPY0*xm**2)/xm/xp/(xm-xp)
               A9 = FX0Y0
               ! interpolation biquadratic function is:
               ! a1*x^2*y^2 + a2*x*y^2 + a3*y^2 + a4*x^2*y + a5*x*y + a6*y + a7*x^2 + a8*x + a9
               new_x = xx - x%double(ii,jj)
               new_y = yy - y%double(ii,jj)
               zi%double(i,j) = a1*new_x**2*new_y**2 + a2*new_x*new_y**2 + a3*new_y**2 &
                              + a4*new_x**2*new_y + a5*new_x*new_y + a6*new_y + a7*new_x**2 &
                              + a8*new_x + a9

            else ! i_order = 3

               ! lazy implementation (not optimized!): working first in a
               ! direction, calling 4 times 'cubic_interp', then calling
               ! one time 'cubic_interp' in the other direction.

               ! (xx,yy) is in the rectangle ( [ii, ii+1], [jj, jj+1] )

               ! prepare indices
               if( ii == 1 ) then
                  ii = ii + 1
               else if( ii+1 == n1 ) then
                  ii = ii - 1
               end if
               ! indices are now [ ii-1, ii, ii+1, ii+2 ] for all cases
               if( jj == 1 ) then
                  jj = jj + 1
               else if( jj+1 == n2 ) then
                  jj = jj - 1
               end if
               ! indices are now [ jj-1, jj, jj+1, jj+2 ] for all cases

               z1 = cubic_interp( x%double(ii-1,jj-1:jj+2),             &
                                  z%double(ii-1,jj-1:jj+2),             &
                                  xx )
               z2 = cubic_interp( x%double(ii  ,jj-1:jj+2),             &
                                  z%double(ii  ,jj-1:jj+2),             &
                                  xx )
               z3 = cubic_interp( x%double(ii+1,jj-1:jj+2),             &
                                  z%double(ii+1,jj-1:jj+2),             &
                                  xx )
               z4 = cubic_interp( x%double(ii+2,jj-1:jj+2),             &
                                  z%double(ii+2,jj-1:jj+2),             &
                                  xx )
               zz = cubic_interp( y%double(ii-1:ii+2,jj),               &
                                  [z1,z2,z3,z4],                        &
                                  yy )
               zi%double(i,j) = zz

            end if

         end do
      end do

      if( mf_phys_units ) then

         ! 'x', 'y', 'xi' and 'yi' must have the same physical units
         ! verifying the physical dimension
         call verif_adim( x%units, y%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'y'",&
                               "are not consistent!" )
            go to 99
         end if
         call verif_adim( x%units, xi%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'xi'",&
                               "are not consistent!" )
            go to 99
         end if
         call verif_adim( y%units, yi%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'y' and 'yi'",&
                               "are not consistent!" )
            go to 99
         end if

         ! 'zi' contains interpolated values from 'z'
         zi%units(:) = z%units(:)

      end if

      zi%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y, z, xi, yi )
      call msAutoRelease( x, y, z, xi, yi )

      call mf_restore_fpe( )

#endif
   end function mfInterp2_real
!_______________________________________________________________________
!
   function mfInterp2_bool( x, y, z, xi, yi, order, extrapol ) result( zi )

      type(mfArray)                       :: x, y, z
      type(mfArray)                       :: xi, yi
      integer,       intent(in), optional :: order
      logical,       intent(in)           :: extrapol
      type(mfArray)                       :: zi
      !------ API end ------
#ifdef _DEVLP

      ! x, y, z : must be matrices, representing data (z) on a
      !           rectangular 2D grid (x,y), having strictly monotonous
      !           values.
      !
      ! (xi,yi) must have the same shape (scalars, vectors, or matrices);
      ! zi will have the same shape as xi and yi.
      !
      ! if (xi,yi) is outside of the range of x and y:
      !    * zi is set to MF_NAN (if extrapol is FALSE)
      !    * zi is computed by extrapolation (if extrapol is TRUE)
      !
      ! interpolation used:
      !    order=0 -> nearest:     1-point stencil
      !    order=1 -> bilinear:    4-point stencil (default)
      !    order=2 -> biquadratic: 9-point stencil
      !    order=3 -> bicubic:    16-point stencil

      integer :: i, j, ni, nj, n1, n2, ii, jj, ii_save, jj_save, status
      real(kind=MF_DOUBLE) :: xx, yy, zz, dx, dy
      real(kind=MF_DOUBLE) :: z1, z2, z3, z4
      real(kind=MF_DOUBLE) :: x1, x2, y1, y2, z11, z12, z21, z22
      real(kind=MF_DOUBLE) :: A, B, C, D

      integer :: i_order, i_nearest, j_nearest
      real(kind=MF_DOUBLE) :: dist2, new_dist2, XM, XP, YM, YP
      real(kind=MF_DOUBLE) :: FX0Y0, FX0YM, FX0YP,                      &
                              FXMY0, FXMYM, FXMYP,                      &
                              FXPY0, FXPYM, FXPYP
      real(kind=MF_DOUBLE) :: A1, A2, A3, A4, A5, A6, A7, A8, A9,       &
                              new_x, new_y

      logical :: bool_x_min, bool_x_max, bool_y_min, bool_y_max

      character(len=*), parameter :: ROUTINE_NAME = "mfInterp2"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y, z, xi, yi )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) .or. mfIsEmpty(z) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE .or. &
          z%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x%shape(1) == 1 .or. x%shape(2) == 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a matrix!" )
         go to 99
      end if

      if( y%shape(1) == 1 .or. y%shape(2) == 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y' must be a matrix!" )
         go to 99
      end if

      if( z%shape(1) == 1 .or. z%shape(2) == 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'z' must be a matrix!" )
         go to 99
      end if

      if( x%shape(1) /= y%shape(1) .or. x%shape(1) /= z%shape(1) .or.   &
          x%shape(2) /= y%shape(2) .or. x%shape(2) /= z%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x', 'y' and 'z' must have the same shape!" )
         go to 99
      end if

      if( present(order) ) then
         i_order = order
         if( i_order < 0 .or. 3 < i_order ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                               "'order' optional arg must be 0, 1, 2 or 3!" )
            go to 99
         end if
      else
         i_order = 1
      end if

      n1 = size( x%double, 1 )
      n2 = size( x%double, 2 )

      if( i_order == 0 ) then
         if( n1 < 1 .or. n2 < 1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', 'y' and 'z'!" )
            go to 99
         end if
      else if( i_order == 1 ) then
         if( n1 < 2 .or. n2 < 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', 'y' and 'z'!" )
            go to 99
         end if
      else if( i_order == 2 ) then
         if( n1 < 3 .or. n2 < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', 'y' and 'z'!",   &
                               "(for biquadratic interpolation, 3 points are required in each direction)")
            go to 99
         end if
      else ! i_order = 3
         if( n1 < 4 .or. n2 < 4 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', 'y' and 'z'!",   &
                               "(for bicubic interpolation, 4 points are required in each direction)")
            go to 99
         end if
      end if

      ! verifying that x(:,:) has constant values along its 1st dim.
      do jj = 1, n2
         if( minval(x%double(:,jj)) /= maxval(x%double(:,jj)) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'x' data are not constant along the 1st dim.!" )
            go to 99
         end if
      end do

      ! verifying that x(:,:) has strictly monotonous values along its 2nd dim.
      dx = x%double(1,2) - x%double(1,1)
      do jj = 3, n2
         if( dx*(x%double(1,jj) - x%double(1,jj-1)) <= 0.0d0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'x' data are not in strictly monotonous order!" )
            go to 99
         end if
      end do
      ! now, we are sure that dx is not zero
      dx = sign(1.0d0,dx) ! +1.0 or -1.0

      ! verifying that y(:,:) has constant values along its 2nd dim.
      do ii = 1, n1
         if( minval(y%double(ii,:)) /= maxval(y%double(ii,:)) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'y' data are not constant along the 2nd dim.!" )
            go to 99
         end if
      end do

      ! verifying that y(:,:) has strictly monotonous values along its 1st dim.
      dy = y%double(2,1) - y%double(1,1)
      do ii = 3, n1
         if( dy*(y%double(ii,1) - y%double(ii-1,1)) <= 0.0d0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'y' data are not in strictly monotonous order!" )
            go to 99
         end if
      end do
      ! now, we are sure that dy is not zero
      dy = sign(1.0d0,dy) ! +1.0 or -1.0

      ni = size( xi%double, 1 )
      nj = size( xi%double, 2 )

      call msAssign( zi, mfZeros(ni,nj) )

      ii_save = 0
      jj_save = 0
      do j = 1, nj
         do i = 1, ni

            xx = xi%double(i,j)
            yy = yi%double(i,j)

            bool_x_min = dx*(xx - x%double(1,1)) < 0.0d0
            bool_x_max = dx*(x%double(n1,n2) - xx) < 0.0d0
            bool_y_min = dy*(yy - y%double(1,1)) < 0.0d0
            bool_y_max = dy*(y%double(n1,n2) - yy) < 0.0d0
            ! find the rectangle where (xx,yy) belongs
            if( bool_x_min .or. bool_x_max .or. bool_y_min .or. bool_y_max ) then

               if( .not. extrapol ) then
                  call PrintMessage( trim(ROUTINE_NAME), "W",           &
                                     "'(xi,yi)' out of range: 'zi' set to NaN" )
                  zi%double(i,j) = MF_NAN
                  cycle
               end if

               if( bool_x_min ) then
                  jj = 1
               else if( bool_x_max ) then
                  if( order == 0 ) then
                     jj = nj - 1
                  else
                     jj = nj - order
                  end if
               end if

               if( bool_y_min ) then
                  ii = 1
               else
                  if( order == 0 ) then
                     ii = ni - 1
                  else
                     ii = ni - order
                  end if
               end if

            else

               if( jj_save /= 0 ) then
                  ! use old position
                  if( dx*(xx - x%double(1,jj_save)) >= 0.0d0 ) then
                     do jj = jj_save, n2-1
                        if( dx*(xx - x%double(1,jj+1)) <= 0.0d0 ) then
                           exit
                        end if
                     end do
                  else
                     do jj = jj_save-1, 1, -1
                        if( dx*(xx - x%double(1,jj)) >= 0.0d0 ) then
                           exit
                        end if
                     end do
                  end if
                  if( dy*(yy - y%double(ii_save,1)) >= 0.0d0 ) then
                     do ii = ii_save, n1-1
                        if( dy*(yy - y%double(ii+1,1)) <= 0.0d0 ) then
                           exit
                        end if
                     end do
                  else
                     do ii = ii_save-1, 1, -1
                        if( dy*(yy - y%double(ii,1)) >= 0.0d0 ) then
                           exit
                        end if
                     end do
                  end if
               else
                  do jj = 1, n2-1
                     if( dx*(xx - x%double(1,jj+1)) <= 0.0d0 ) then
                        exit
                     end if
                  end do
                  do ii = 1, n1-1
                     if( dy*(yy - y%double(ii+1,1)) <= 0.0d0 ) then
                        exit
                     end if
                  end do
               end if

               ! saving for next search
               ii_save = ii
               jj_save = jj

            end if

            if( i_order == 0 ) then

               ! nearest interpolation in the rectangle
               !   ( [ii, ii+1], [jj, jj+1] )
               if( (xx-x%double(1,jj)) < (x%double(1,jj+1)-xx) ) then
                  if( (yy-y%double(ii,1)) < (y%double(ii+1,1)-yy) ) then
                     zi%double(i,j) = z%double(ii,jj)
                  else
                     zi%double(i,j) = z%double(ii+1,jj)
                  end if
               else
                  if( (yy-y%double(ii,1)) < (y%double(ii+1,1)-yy) ) then
                     zi%double(i,j) = z%double(ii,jj+1)
                  else
                     zi%double(i,j) = z%double(ii+1,jj+1)
                  end if
               end if

            else if( i_order == 1 ) then

               ! bilinear interpolation in the rectangle
               !   ( [ii, ii+1], [jj, jj+1] )
               !
               !  z(x,y) = A*x + B*y + C*x*y + D
               !
               ! avec translation de P11 à l'origine :
               x2 = x%double(1,jj+1) - x%double(1,jj)
               y2 = y%double(ii+1,1) - y%double(ii,1)
               z11 = z%double(ii,jj)
               z21 = z%double(ii,jj+1)
               z12 = z%double(ii+1,jj)
               z22 = z%double(ii+1,jj+1)

               D = z11
               A = (z21-D)/x2
               B = (z12-D)/y2
               C = (z22 - A*x2 - B*y2 - D)/(x2*y2)

               zi%double(i,j) = A*(xx-x%double(1,jj))                   &
                              + B*(yy-y%double(ii,1))                   &
                              + C*(xx-x%double(1,jj))*(yy-y%double(ii,1)) &
                              + D

            else if( i_order == 2 ) then

               ! find the nearest point to (xx,yy)
               dist2 = (xx-x%double(ii,jj))**2 + (yy-y%double(ii,jj))**2
               i_nearest = ii
               j_nearest = jj

               new_dist2 = (xx-x%double(ii+1,jj))**2 +                  &
                           (yy-y%double(ii+1,jj))**2
               if( new_dist2 < dist2 ) then
                  dist2 = new_dist2
                  i_nearest = ii + 1
                  j_nearest = jj
               end if

               new_dist2 = (xx-x%double(ii,jj+1))**2 +                  &
                           (yy-y%double(ii,jj+1))**2
               if( new_dist2 < dist2 ) then
                  dist2 = new_dist2
                  i_nearest = ii
                  j_nearest = jj + 1
               end if

               new_dist2 = (xx-x%double(ii+1,jj+1))**2 +                &
                           (yy-y%double(ii+1,jj+1))**2
               if( new_dist2 < dist2 ) then
                  i_nearest = ii + 1
                  j_nearest = jj + 1
               end if

               ii = i_nearest
               jj = j_nearest

               ! check if the central point (ii,jj) of
               ! our 9-point stencil is on the boundary
               if( ii == 1 ) then
                  ! North boundary
                  ii = 2
                  if( jj == 1 ) then
                     ! NW corner
                     jj = 2
                  else if( jj == n2 ) then
                     ! NE corner
                     jj = n2 - 1
                  end if
               else if( ii == n1 ) then
                  ! South boundary
                  ii = n1 - 1
                  if( jj == 1 ) then
                     ! SW corner
                     jj = 2
                  else if( jj == n2 ) then
                     ! SE corner
                     jj = n2 - 1
                  end if
               else
                  if( jj == 1 ) then
                     ! West boundary
                     jj = 2
                  else if( jj == n2 ) then
                     ! East boundary
                     jj = n2 - 1
                  else
                     ! interior point
                  end if
               end if

               XM = x%double(ii,jj-1) - x%double(ii,jj)
               XP = x%double(ii,jj+1) - x%double(ii,jj)
               YM = y%double(ii-1,jj) - y%double(ii,jj)
               YP = y%double(ii+1,jj) - y%double(ii,jj)
               FXMYM = z%double(ii-1,jj-1)
               FX0YM = z%double(ii-1,jj)
               FXPYM = z%double(ii-1,jj+1)
               FXMY0 = z%double(ii,  jj-1)
               FX0Y0 = z%double(ii,  jj)
               FXPY0 = z%double(ii,  jj+1)
               FXMYP = z%double(ii+1,jj-1)
               FX0YP = z%double(ii+1,jj)
               FXPYP = z%double(ii+1,jj+1)
               A1 = 1.0d0/ym/xm*(FX0Y0*xm*ym-FX0Y0*xm*yp-FX0Y0*xp*ym+FX0Y0*xp*yp &
                  + FX0YM*xm*yp-FX0YM*xp*yp-FX0YP*xm*ym+FX0YP*xp*ym+FXMY0*xp*ym &
                  - FXMY0*xp*yp+FXMYM*xp*yp-FXMYP*xp*ym-FXPY0*xm*ym+FXPY0*xm*yp &
                  - FXPYM*xm*yp+FXPYP*xm*ym)/xp/yp/(xm*ym-xm*yp-xp*ym+xp*yp)
               A2 = -(FX0Y0*xm**2*ym-FX0Y0*xm**2*yp-FX0Y0*xp**2*ym+FX0Y0*xp**2*yp &
                  + FX0YM*xm**2*yp-FX0YM*xp**2*yp-FX0YP*xm**2*ym+FX0YP*xp**2*ym &
                  + FXMY0*xp**2*ym-FXMY0*xp**2*yp+FXMYM*xp**2*yp-FXMYP*xp**2*ym &
                  - FXPY0*xm**2*ym+FXPY0*xm**2*yp-FXPYM*xm**2*yp+FXPYP*xm**2*ym) &
                  / xm/xp/ym/(xm-xp)/(ym-yp)/yp
               A3 = (FX0Y0*ym-FX0Y0*yp+FX0YM*yp-FX0YP*ym)/ym/yp/(ym-yp)
               A4 = -(FX0Y0*xm*ym**2-FX0Y0*xm*yp**2-FX0Y0*ym**2*xp+FX0Y0*yp**2*xp &
                  + FX0YM*xm*yp**2-FX0YM*yp**2*xp-FX0YP*xm*ym**2+FX0YP*ym**2*xp &
                  + FXMY0*ym**2*xp-FXMY0*yp**2*xp+FXMYM*yp**2*xp-FXMYP*ym**2*xp &
                  - FXPY0*xm*ym**2+FXPY0*xm*yp**2-FXPYM*xm*yp**2+FXPYP*xm*ym**2)&
                  / xm/ym/yp/(ym-yp)/(xm-xp)/xp
               A5 = (FX0Y0*xm**2*ym**2-FX0Y0*xm**2*yp**2-FX0Y0*xp**2*ym**2 &
                  + FX0Y0*xp**2*yp**2+FX0YM*xm**2*yp**2-FX0YM*xp**2*yp**2-FX0YP*xm**2*ym**2 &
                  + FX0YP*xp**2*ym**2+FXMY0*xp**2*ym**2-FXMY0*xp**2*yp**2+FXMYM*xp**2*yp**2 &
                  - FXMYP*xp**2*ym**2-FXPY0*xm**2*ym**2+FXPY0*xm**2*yp**2-FXPYM*xm**2*yp**2 &
                  + FXPYP*xm**2*ym**2)/xm/xp/ym/yp/(xm*ym-xm*yp-xp*ym+xp*yp)
               A6 = -(FX0Y0*ym**2-FX0Y0*yp**2+FX0YM*yp**2-FX0YP*ym**2)/ym/yp/(ym-yp)
               A7 = (FX0Y0*xm-FX0Y0*xp+FXMY0*xp-FXPY0*xm)/xm/xp/(xm-xp)
               A8 = -(FX0Y0*xm**2-FX0Y0*xp**2+FXMY0*xp**2-FXPY0*xm**2)/xm/xp/(xm-xp)
               A9 = FX0Y0
               ! interpolation biquadratic function is:
               ! a1*x^2*y^2 + a2*x*y^2 + a3*y^2 + a4*x^2*y + a5*x*y + a6*y + a7*x^2 + a8*x + a9
               new_x = xx - x%double(ii,jj)
               new_y = yy - y%double(ii,jj)
               zi%double(i,j) = a1*new_x**2*new_y**2 + a2*new_x*new_y**2 + a3*new_y**2 &
                              + a4*new_x**2*new_y + a5*new_x*new_y + a6*new_y + a7*new_x**2 &
                              + a8*new_x + a9

            else ! i_order = 3

               ! lazy implementation (not optimized!): working first in a
               ! direction, calling 4 times 'cubic_interp', then calling
               ! one time 'cubic_interp' in the other direction.

               ! (xx,yy) is in the rectangle ( [ii, ii+1], [jj, jj+1] )

               ! prepare indices
               if( ii == 1 ) then
                  ii = ii + 1
               else if( ii+1 == n1 ) then
                  ii = ii - 1
               end if
               ! indices are now [ ii-1, ii, ii+1, ii+2 ] for all cases
               if( jj == 1 ) then
                  jj = jj + 1
               else if( jj+1 == n2 ) then
                  jj = jj - 1
               end if
               ! indices are now [ jj-1, jj, jj+1, jj+2 ] for all cases

               z1 = cubic_interp( x%double(ii-1,jj-1:jj+2),             &
                                  z%double(ii-1,jj-1:jj+2),             &
                                  xx )
               z2 = cubic_interp( x%double(ii  ,jj-1:jj+2),             &
                                  z%double(ii  ,jj-1:jj+2),             &
                                  xx )
               z3 = cubic_interp( x%double(ii+1,jj-1:jj+2),             &
                                  z%double(ii+1,jj-1:jj+2),             &
                                  xx )
               z4 = cubic_interp( x%double(ii+2,jj-1:jj+2),             &
                                  z%double(ii+2,jj-1:jj+2),             &
                                  xx )
               zz = cubic_interp( y%double(ii-1:ii+2,jj),               &
                                  [z1,z2,z3,z4],                        &
                                  yy )
               zi%double(i,j) = zz

            end if

         end do
      end do

      if( mf_phys_units ) then

         ! 'x', 'y', 'xi' and 'yi' must have the same physical units
         ! verifying the physical dimension
         call verif_adim( x%units, y%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'y'",&
                               "are not consistent!" )
            go to 99
         end if
         call verif_adim( x%units, xi%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'xi'",&
                               "are not consistent!" )
            go to 99
         end if
         call verif_adim( y%units, yi%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'y' and 'yi'",&
                               "are not consistent!" )
            go to 99
         end if

         ! 'zi' contains interpolated values from 'z'
         zi%units(:) = z%units(:)

      end if

      zi%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y, z, xi, yi )
      call msAutoRelease( x, y, z, xi, yi )

      call mf_restore_fpe( )

#endif
   end function mfInterp2_bool
