! f90 include file

!_______________________________________________________________________
!
   function mfInterp1_real( x, y, xi, order, extrapol ) result( yi )

      type(mfArray)                              :: x, y
      type(mfArray)                              :: xi
      integer,              intent(in), optional :: order
      real(kind=MF_DOUBLE), intent(in), optional :: extrapol
      type(mfArray)                              :: yi
      !------ API end ------
#ifdef _DEVLP

      ! x, y : must be vectors, representing data (y) on a 1D grid (x);
      !        'x' must have strictly monotonous values.
      !
      ! xi may be a scalar or a vectors;
      ! yi will have the same shape as xi.
      !
      ! if xi is outside of the range of x:
      !    * yi is set to MF_NAN (if extrapol is not present)
      !    * yi is set to extrapol (if extrapol is present)
      !
      ! interpolation used:
      !    order=0 -> nearest:   1-point stencil
      !    order=1 -> linear:    2-point stencil (default)
      !    order=2 -> quadratic: 3-point stencil
      !    order=3 -> cubic:     4-point stencil

      integer :: i, j, ni, dimi, n, dim, ii, ii_save, status
      real(kind=MF_DOUBLE) :: xx, yy, A, B, C, dx, xm, xp, new_x

      integer :: i_order

      character(len=*), parameter :: ROUTINE_NAME = "mfInterp1"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y, xi )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x%shape(1) == 1 ) then
         dim = 2
      else if( x%shape(2) == 1 ) then
         dim = 1
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a vector!" )
         go to 99
      end if

      if( present(order) ) then
         i_order = order
         if( i_order < 0 .or. 3 < i_order ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                               "'order' optional arg must be 0, 1, 2 or 3!" )
            go to 99
         end if
      else
         i_order = 1
      end if

      n = x%shape(dim)

      if( i_order == 0 ) then
         if( n < 1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', and 'y'!",      &
                               "(for nearest interpolation, 1 point is required)")
            go to 99
         end if
      else if( i_order == 1 ) then
         if( n < 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', and 'y'!",      &
                               "(for linear interpolation, 2 points are required)")
            go to 99
         end if
      else if( i_order == 2 ) then
         if( n < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', and 'y'!",      &
                               "(for quadratic interpolation, 3 points are required)")
            go to 99
         end if
      else ! i_order = 3
         if( n < 4 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', and 'y'!",      &
                               "(for cubic interpolation, 4 points are required)")
            go to 99
         end if
      end if

      if( x%shape(1) /= y%shape(1) .or. x%shape(2) /= y%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x', and 'y' must have the same shape!" )
         go to 99
      end if

      if( xi%shape(1) == 1 ) then
         dimi = 2
      else if( xi%shape(2) == 1 ) then
         dimi = 1
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'xi' must be a scalar or a vector!" )
         go to 99
      end if

      ni = xi%shape(dimi)

      ! verifying that x(:) has strictly monotonous values along 'dim'.
      if( dim == 1 ) then
         dx = x%double(2,1) - x%double(1,1)
         do i = 3, n
            if( dx*(x%double(i,1) - x%double(i-1,1)) <= 0.0d0 ) then
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "'x' data are not in strictly monotonous order!" )
               go to 99
            end if
         end do
      else ! dim == 2
         dx = x%double(1,2) - x%double(1,1)
         do j = 3, n
            if( dx*(x%double(1,j) - x%double(1,j-1)) <= 0.0d0 ) then
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "'x' data are not in strictly monotonous order!" )
               go to 99
            end if
         end do
      end if
      ! now, we are sure that dx is not zero
      dx = sign(1.0d0,dx) ! +1.0 or -1.0

      ii_save = 0

      if( dim == 1 ) then
         call msAssign( yi, mfZeros(ni,1) )
         do i = 1, ni
            ! find the interval where xi belongs
            xx = xi%double(i,1)
            if( dx*(xx - x%double(1,1)) < 0.0d0 .or.                    &
                dx*(x%double(n,1) - xx) < 0.0d0 ) then

               if( present(extrapol) ) then
                  call PrintMessage( trim(ROUTINE_NAME), "I",           &
                                     "'xi' out of range: 'yi' set to a user-provided value" )
                  yi%double(i,1) = extrapol
               else
                  call PrintMessage( trim(ROUTINE_NAME), "W",           &
                                     "'xi' out of range: 'yi' set to NaN" )
                  yi%double(i,1) = MF_NAN
               end if
               cycle

            else

               if( ii_save /= 0 ) then
                  ! use old position
                  if( dx*(xx - x%double(ii_save,1)) >= 0.0d0 ) then
                     do ii = ii_save, n-1
                        if( dx*(xx - x%double(ii+1,1)) <= 0.0d0 ) then
                           go to 10
                        end if
                     end do
                  else
                     do ii = ii_save-1, 1, -1
                        if( dx*(xx - x%double(ii,1)) >= 0.0d0 ) then
                           exit
                        end if
                     end do
                  end if
               else
                  do ii = 1, n-1
                     if( dx*(xx - x%double(ii+1,1)) <= 0.0d0 ) then
                        exit
                     end if
                  end do
               end if

 10            continue

               ! saving for next search
               ii_save = ii

            end if

            if( i_order == 0 ) then

               ! nearest interpolation in the interval [ii, ii+1]
               if( (xx-x%double(ii,1)) < (x%double(ii+1,1)-xx) ) then
                  yi%double(i,1) = y%double(ii,1)
               else
                  yi%double(i,1) = y%double(ii+1,1)
               end if

            else if( i_order == 1 ) then

               ! linear interpolation in the interval [ii, ii+1]
               !
               !  y(x) = A*x + B
               !
               A = ( y%double(ii+1,1) - y%double(ii,1) ) /              &
                   ( x%double(ii+1,1) - x%double(ii,1) )
               B = y%double(ii,1) - A*x%double(ii,1)

               yi%double(i,1) = A*xx + B

            else if( i_order == 2 ) then

               ! quadratic interpolation
               !
               !  y(x) = A*x^2 + B*x + C
               !
               if( ii == 1 ) then
                  ! using nodes [ ii, ii+1, ii+2 ]
                  xp = x%double(ii+2,1) - x%double(ii+1,1)
                  xm = x%double(ii,1) - x%double(ii+1,1)
                  C = y%double(ii+1,1)
                  A = (C*(xm-xp)-y%double(ii+2,1)*xm+y%double(ii,1)*xp) &
                      /xm /xp /(xm-xp)
                  B = -(C*(xm**2-xp**2)-y%double(ii+2,1)*xm**2+y%double(ii,1)*xp**2) &
                      /xm /xp /(xm-xp)

                  new_x = xx - x%double(ii+1,1)
               else
                  ! using nodes [ ii-1, ii, ii+1 ]
                  xp = x%double(ii+1,1) - x%double(ii,1)
                  xm = x%double(ii-1,1) - x%double(ii,1)
                  C = y%double(ii,1)
                  A = (C*(xm-xp)-y%double(ii+1,1)*xm+y%double(ii-1,1)*xp) &
                      /xm /xp /(xm-xp)
                  B = -(C*(xm**2-xp**2)-y%double(ii+1,1)*xm**2+y%double(ii-1,1)*xp**2) &
                      /xm /xp /(xm-xp)

                  new_x = xx - x%double(ii,1)
               end if
               yi%double(i,1) = A*new_x**2 + B*new_x + C

            else ! i_order = 3

               ! cubic interpolation
               !
               !  y(x) = A*x^3 + B*x^3 + C*x + D
               !
               if( ii == 1 ) then
                  ! using nodes [ ii, ii+1, ii+2, ii+3 ]
                  yy = cubic_interp( x%double(ii:ii+3,1), y%double(ii:ii+3,1), &
                                     xx )
               else if( ii+1 == n ) then
                  ! using nodes [ ii-2, ii-1, ii, ii+1 ]
                  yy = cubic_interp( x%double(ii-2:ii+1,1), y%double(ii-2:ii+1,1), &
                                     xx )
               else
                  ! using nodes [ ii-1, ii, ii+1, ii+2 ]
                  yy = cubic_interp( x%double(ii-1:ii+2,1), y%double(ii-1:ii+2,1), &
                                     xx )
               end if
               yi%double(i,1) = yy

            end if

         end do
      else ! dim == 2
         call msAssign( yi, mfZeros(1,ni) )
         do i = 1, ni
            ! find the interval where xi belongs
            xx = xi%double(1,i)
            if( dx*(xx - x%double(1,1)) < 0.0d0 .or.                    &
                dx*(x%double(1,n) - xx) < 0.0d0 ) then

               if( present(extrapol) ) then
                  call PrintMessage( trim(ROUTINE_NAME), "I",           &
                                     "'xi' out of range: 'yi' set to a user-provided value" )
                  yi%double(1,i) = extrapol
               else
                  call PrintMessage( trim(ROUTINE_NAME), "W",           &
                                     "'xi' out of range: 'yi' set to NaN" )
                  yi%double(1,i) = MF_NAN
               end if
               call PrintMessage( trim(ROUTINE_NAME), "W",              &
                                  "'xi' out of range: 'yi' set to NaN" )
               cycle

            else

               if( ii_save /= 0 ) then
                  ! use old position
                  if( dx*(xx - x%double(1,ii_save)) >= 0.0d0 ) then
                     do ii = ii_save, n-1
                        if( dx*(xx - x%double(1,ii+1)) <= 0.0d0 ) then
                           go to 20
                        end if
                     end do
                  else
                     do ii = ii_save-1, 1, -1
                        if( dx*(xx - x%double(1,ii)) >= 0.0d0 ) then
                           exit
                        end if
                     end do
                  end if
               else
                  do ii = 1, n-1
                     if( dx*(xx - x%double(1,ii+1)) <= 0.0d0 ) then
                        exit
                     end if
                  end do
               end if

 20            continue

               ! saving for next search
               ii_save = ii

            end if

            if( i_order == 0 ) then

               ! nearest interpolation in the interval [ii, ii+1]
               if( (xx-x%double(1,ii)) < (x%double(1,ii+1)-xx) ) then
                  yi%double(1,i) = y%double(1,ii)
               else
                  yi%double(1,i) = y%double(1,ii+1)
               end if

            else if( i_order == 1 ) then

               ! linear interpolation in the interval [ii, ii+1]
               !
               !  y(x) = A*x + B
               !
               A = ( y%double(1,ii+1) - y%double(1,ii) ) /              &
                   ( x%double(1,ii+1) - x%double(1,ii) )
               B = y%double(1,ii) - A*x%double(1,ii)

               yi%double(1,i) = A*xx + B

            else if( i_order == 2 ) then

               ! quadratic interpolation
               !
               !  y(x) = A*x^2 + B*x + C
               !
               if( ii == 1 ) then
                  ! using nodes [ ii, ii+1, ii+2 ]
                  xp = x%double(1,ii+2) - x%double(1,ii+1)
                  xm = x%double(1,ii) - x%double(1,ii+1)
                  C = y%double(1,ii+1)
                  A = (C*(xm-xp)-y%double(1,ii+2)*xm+y%double(1,ii)*xp) &
                      /xm /xp /(xm-xp)
                  B = -(C*(xm**2-xp**2)-y%double(1,ii+2)*xm**2+y%double(1,ii)*xp**2) &
                      /xm /xp /(xm-xp)

                  new_x = xx - x%double(1,ii+1)
               else
                  ! using nodes [ ii-1, ii, ii+1 ]
                  xp = x%double(1,ii+1) - x%double(1,ii)
                  xm = x%double(1,ii-1) - x%double(1,ii)
                  C = y%double(1,ii)
                  A = (C*(xm-xp)-y%double(1,ii+1)*xm+y%double(1,ii-1)*xp) &
                      /xm /xp /(xm-xp)
                  B = -(C*(xm**2-xp**2)-y%double(1,ii+1)*xm**2+y%double(1,ii-1)*xp**2) &
                      /xm /xp /(xm-xp)

                  new_x = xx - x%double(1,ii)
               end if
               yi%double(1,i) = A*new_x**2 + B*new_x + C

            else ! i_order = 3

               ! cubic interpolation
               !
               !  y(x) = A*x^3 + B*x^3 + C*x + D
               !
               if( ii == 1 ) then
                  ! using nodes [ ii, ii+1, ii+2, ii+3 ]
                  yy = cubic_interp( x%double(1,ii:ii+3), y%double(1,ii:ii+3), &
                                     xx )
               else if( ii+1 == n ) then
                  ! using nodes [ ii-2, ii-1, ii, ii+1 ]
                  yy = cubic_interp( x%double(1,ii-2:ii+1), y%double(1,ii-2:ii+1), &
                                     xx )
               else
                  ! using nodes [ ii-1, ii, ii+1, ii+2 ]
                  yy = cubic_interp( x%double(1,ii-1:ii+2), y%double(1,ii-1:ii+2), &
                                     xx )
               end if
               yi%double(1,i) = yy

            end if

         end do
      end if

      if( mf_phys_units ) then

         ! 'x', 'y' and 'xi' must have the same physical units
         ! verifying the physical dimension
         call verif_adim( x%units, y%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'y'",&
                               "are not consistent!" )
            go to 99
         end if
         call verif_adim( x%units, xi%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'xi'",&
                               "are not consistent!" )
            go to 99
         end if

         ! 'yi' contains interpolated values from 'y'
         yi%units(:) = y%units(:)

      end if

      yi%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y, xi )
      call msAutoRelease( x, y, xi )

      call mf_restore_fpe( )

#endif
   end function mfInterp1_real
!_______________________________________________________________________
!
   function mfInterp1_bool( x, y, xi, order, extrapol ) result( yi )

      type(mfArray)                       :: x, y
      type(mfArray)                       :: xi
      integer,       intent(in), optional :: order
      logical,       intent(in)           :: extrapol
      type(mfArray)                       :: yi
      !------ API end ------
#ifdef _DEVLP

      ! x, y : must be vectors, representing data (y) on a 1D grid (x);
      !        'x' must have strictly monotonous values.
      !
      ! xi may be a scalar or a vectors;
      ! yi will have the same shape as xi.
      !
      ! if xi is outside of the range of x:
      !    * yi is set to MF_NAN (if extrapol is FALSE)
      !    * yi is computed by extrapolation (if extrapol is TRUE)
      !
      ! interpolation used:
      !    order=0 -> nearest:   1-point stencil
      !    order=1 -> linear:    2-point stencil (default)
      !    order=2 -> quadratic: 3-point stencil
      !    order=3 -> cubic:     4-point stencil

      integer :: i, j, ni, dimi, n, dim, ii, ii_save, status
      real(kind=MF_DOUBLE) :: xx, yy, A, B, C, dx, xm, xp, new_x

      integer :: i_order

      character(len=*), parameter :: ROUTINE_NAME = "mfInterp1"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y, xi )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x%shape(1) == 1 ) then
         dim = 2
      else if( x%shape(2) == 1 ) then
         dim = 1
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a vector!" )
         go to 99
      end if

      if( present(order) ) then
         i_order = order
         if( i_order < 0 .or. 3 < i_order ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'order' optional arg must be 0, 1, 2 or 3!" )
            go to 99
         end if
      else
         i_order = 1
      end if

      n = x%shape(dim)

      if( i_order == 0 ) then
         if( n < 1 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', and 'y'!",      &
                               "(for nearest interpolation, 1 point is required)")
            go to 99
         end if
      else if( i_order == 1 ) then
         if( n < 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', and 'y'!",      &
                               "(for linear interpolation, 2 points are required)")
            go to 99
         end if
      else if( i_order == 2 ) then
         if( n < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', and 'y'!",      &
                               "(for quadratic interpolation, 3 points are required)")
            go to 99
         end if
      else ! i_order = 3
         if( n < 4 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few points in 'x', and 'y'!",      &
                               "(for cubic interpolation, 4 points are required)")
            go to 99
         end if
      end if

      if( x%shape(1) /= y%shape(1) .or. x%shape(2) /= y%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x', and 'y' must have the same shape!" )
         go to 99
      end if

      if( xi%shape(1) == 1 ) then
         dimi = 2
      else if( xi%shape(2) == 1 ) then
         dimi = 1
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'xi' must be a scalar or a vector!" )
         go to 99
      end if

      ni = xi%shape(dimi)

      ! verifying that x(:) has strictly monotonous values along 'dim'.
      if( dim == 1 ) then
         dx = x%double(2,1) - x%double(1,1)
         do i = 3, n
            if( dx*(x%double(i,1) - x%double(i-1,1)) <= 0.0d0 ) then
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "'x' data are not in strictly monotonous order!" )
               go to 99
            end if
         end do
      else ! dim == 2
         dx = x%double(1,2) - x%double(1,1)
         do j = 3, n
            if( dx*(x%double(1,j) - x%double(1,j-1)) <= 0.0d0 ) then
               call PrintMessage( trim(ROUTINE_NAME), "E",              &
                                  "'x' data are not in strictly monotonous order!" )
               go to 99
            end if
         end do
      end if
      ! now, we are sure that dx is not zero
      dx = sign(1.0d0,dx) ! +1.0 or -1.0

      ii_save = 0

      if( dim == 1 ) then
         call msAssign( yi, mfZeros(ni,1) )
         do i = 1, ni
            ! find the interval where xi belongs
            xx = xi%double(i,1)
            if( dx*(xx - x%double(1,1)) < 0.0d0 ) then

               if( extrapol ) then
                  ii = 1
               else
                  call PrintMessage( trim(ROUTINE_NAME), "W",              &
                                     "'xi' out of range: 'yi' set to NaN" )
                  yi%double(i,1) = MF_NAN
                  cycle
               end if

            else if( dx*(x%double(n,1) - xx) < 0.0d0 ) then

               if( extrapol ) then
                  if( order == 0 ) then
                     ii = n - 1
                  else
                     ii = n - order
                  end if
               else
                  call PrintMessage( trim(ROUTINE_NAME), "W",              &
                                     "'xi' out of range: 'yi' set to NaN" )
                  yi%double(i,1) = MF_NAN
                  cycle
               end if

            else

               if( ii_save /= 0 ) then
                  ! use old position
                  if( dx*(xx - x%double(ii_save,1)) >= 0.0d0 ) then
                     do ii = ii_save, n-1
                        if( dx*(xx - x%double(ii+1,1)) <= 0.0d0 ) then
                           go to 10
                        end if
                     end do
                  else
                     do ii = ii_save-1, 1, -1
                        if( dx*(xx - x%double(ii,1)) >= 0.0d0 ) then
                           exit
                        end if
                     end do
                  end if
               else
                  do ii = 1, n-1
                     if( dx*(xx - x%double(ii+1,1)) <= 0.0d0 ) then
                        exit
                     end if
                  end do
               end if

 10            continue

               ! saving for next search
               ii_save = ii

            end if

            if( i_order == 0 ) then

               ! nearest interpolation in the interval [ii, ii+1]
               if( (xx-x%double(ii,1)) < (x%double(ii+1,1)-xx) ) then
                  yi%double(i,1) = y%double(ii,1)
               else
                  yi%double(i,1) = y%double(ii+1,1)
               end if

            else if( i_order == 1 ) then

               ! linear interpolation in the interval [ii, ii+1]
               !
               !  y(x) = A*x + B
               !
               A = ( y%double(ii+1,1) - y%double(ii,1) ) /              &
                   ( x%double(ii+1,1) - x%double(ii,1) )
               B = y%double(ii,1) - A*x%double(ii,1)

               yi%double(i,1) = A*xx + B

            else if( i_order == 2 ) then

               ! quadratic interpolation
               !
               !  y(x) = A*x^2 + B*x + C
               !
               if( ii == 1 ) then
                  ! using nodes [ ii, ii+1, ii+2 ]
                  xp = x%double(ii+2,1) - x%double(ii+1,1)
                  xm = x%double(ii,1) - x%double(ii+1,1)
                  C = y%double(ii+1,1)
                  A = (C*(xm-xp)-y%double(ii+2,1)*xm+y%double(ii,1)*xp) &
                      /xm /xp /(xm-xp)
                  B = -(C*(xm**2-xp**2)-y%double(ii+2,1)*xm**2+y%double(ii,1)*xp**2) &
                      /xm /xp /(xm-xp)

                  new_x = xx - x%double(ii+1,1)
               else
                  ! using nodes [ ii-1, ii, ii+1 ]
                  xp = x%double(ii+1,1) - x%double(ii,1)
                  xm = x%double(ii-1,1) - x%double(ii,1)
                  C = y%double(ii,1)
                  A = (C*(xm-xp)-y%double(ii+1,1)*xm+y%double(ii-1,1)*xp) &
                      /xm /xp /(xm-xp)
                  B = -(C*(xm**2-xp**2)-y%double(ii+1,1)*xm**2+y%double(ii-1,1)*xp**2) &
                      /xm /xp /(xm-xp)

                  new_x = xx - x%double(ii,1)
               end if
               yi%double(i,1) = A*new_x**2 + B*new_x + C

            else ! i_order = 3

               ! cubic interpolation
               !
               !  y(x) = A*x^3 + B*x^3 + C*x + D
               !
               if( ii == 1 ) then
                  ! using nodes [ ii, ii+1, ii+2, ii+3 ]
                  yy = cubic_interp( x%double(ii:ii+3,1), y%double(ii:ii+3,1), &
                                     xx )
               else if( ii+1 == n ) then
                  ! using nodes [ ii-2, ii-1, ii, ii+1 ]
                  yy = cubic_interp( x%double(ii-2:ii+1,1), y%double(ii-2:ii+1,1), &
                                     xx )
               else
                  ! using nodes [ ii-1, ii, ii+1, ii+2 ]
                  yy = cubic_interp( x%double(ii-1:ii+2,1), y%double(ii-1:ii+2,1), &
                                     xx )
               end if
               yi%double(i,1) = yy

            end if

         end do
      else ! dim == 2
         call msAssign( yi, mfZeros(1,ni) )
         do i = 1, ni
            ! find the interval where xi belongs
            xx = xi%double(1,i)
            if( dx*(xx - x%double(1,1)) < 0.0d0 ) then

               if( extrapol ) then
                  ii = 1
               else
                  call PrintMessage( trim(ROUTINE_NAME), "W",              &
                                     "'xi' out of range: 'yi' set to NaN" )
                  yi%double(1,i) = MF_NAN
                  cycle
               end if

            else if( dx*(x%double(1,n) - xx) < 0.0d0 ) then

               if( extrapol ) then
                  ii = n - 1
               else
                  call PrintMessage( trim(ROUTINE_NAME), "W",              &
                                     "'xi' out of range: 'yi' set to NaN" )
                  yi%double(1,i) = MF_NAN
                  cycle
               end if

            else

               if( ii_save /= 0 ) then
                  ! use old position
                  if( dx*(xx - x%double(1,ii_save)) >= 0.0d0 ) then
                     do ii = ii_save, n-1
                        if( dx*(xx - x%double(1,ii+1)) <= 0.0d0 ) then
                           go to 20
                        end if
                     end do
                  else
                     do ii = ii_save-1, 1, -1
                        if( dx*(xx - x%double(1,ii)) >= 0.0d0 ) then
                           exit
                        end if
                     end do
                  end if
               else
                  do ii = 1, n-1
                     if( dx*(xx - x%double(1,ii+1)) <= 0.0d0 ) then
                        exit
                     end if
                  end do
               end if

 20            continue

               ! saving for next search
               ii_save = ii

            end if

            if( i_order == 0 ) then

               ! nearest interpolation in the interval [ii, ii+1]
               if( (xx-x%double(1,ii)) < (x%double(1,ii+1)-xx) ) then
                  yi%double(1,i) = y%double(1,ii)
               else
                  yi%double(1,i) = y%double(1,ii+1)
               end if

            else if( i_order == 1 ) then

               ! linear interpolation in the interval [ii, ii+1]
               !
               !  y(x) = A*x + B
               !
               A = ( y%double(1,ii+1) - y%double(1,ii) ) /              &
                   ( x%double(1,ii+1) - x%double(1,ii) )
               B = y%double(1,ii) - A*x%double(1,ii)

               yi%double(1,i) = A*xx + B

            else if( i_order == 2 ) then

               ! quadratic interpolation
               !
               !  y(x) = A*x^2 + B*x + C
               !
               if( ii == 1 ) then
                  ! using nodes [ ii, ii+1, ii+2 ]
                  xp = x%double(1,ii+2) - x%double(1,ii+1)
                  xm = x%double(1,ii) - x%double(1,ii+1)
                  C = y%double(1,ii+1)
                  A = (C*(xm-xp)-y%double(1,ii+2)*xm+y%double(1,ii)*xp) &
                      /xm /xp /(xm-xp)
                  B = -(C*(xm**2-xp**2)-y%double(1,ii+2)*xm**2+y%double(1,ii)*xp**2) &
                      /xm /xp /(xm-xp)

                  new_x = xx - x%double(1,ii+1)
               else
                  ! using nodes [ ii-1, ii, ii+1 ]
                  xp = x%double(1,ii+1) - x%double(1,ii)
                  xm = x%double(1,ii-1) - x%double(1,ii)
                  C = y%double(1,ii)
                  A = (C*(xm-xp)-y%double(1,ii+1)*xm+y%double(1,ii-1)*xp) &
                      /xm /xp /(xm-xp)
                  B = -(C*(xm**2-xp**2)-y%double(1,ii+1)*xm**2+y%double(1,ii-1)*xp**2) &
                      /xm /xp /(xm-xp)

                  new_x = xx - x%double(1,ii)
               end if
               yi%double(1,i) = A*new_x**2 + B*new_x + C

            else ! i_order = 3

               ! cubic interpolation
               !
               !  y(x) = A*x^3 + B*x^3 + C*x + D
               !
               if( ii == 1 ) then
                  ! using nodes [ ii, ii+1, ii+2, ii+3 ]
                  yy = cubic_interp( x%double(1,ii:ii+3), y%double(1,ii:ii+3), &
                                     xx )
               else if( ii+1 == n ) then
                  ! using nodes [ ii-2, ii-1, ii, ii+1 ]
                  yy = cubic_interp( x%double(1,ii-2:ii+1), y%double(1,ii-2:ii+1), &
                                     xx )
               else
                  ! using nodes [ ii-1, ii, ii+1, ii+2 ]
                  yy = cubic_interp( x%double(1,ii-1:ii+2), y%double(1,ii-1:ii+2), &
                                     xx )
               end if
               yi%double(1,i) = yy

            end if

         end do
      end if

      if( mf_phys_units ) then

         ! 'x', 'y' and 'xi' must have the same physical units
         ! verifying the physical dimension
         call verif_adim( x%units, y%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'y'",&
                               "are not consistent!" )
            go to 99
         end if
         call verif_adim( x%units, xi%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x' and 'xi'",&
                               "are not consistent!" )
            go to 99
         end if

         ! 'yi' contains interpolated values from 'y'
         yi%units(:) = y%units(:)

      end if

      yi%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y, xi )
      call msAutoRelease( x, y, xi )

      call mf_restore_fpe( )

#endif
   end function mfInterp1_bool
