! f90 include file

!_______________________________________________________________________
!
   function mfGridData3D( x, y, z, f, xi, yi, zi ) result( fi )

      type(mfArray) :: x, y, z, f
      type(mfArray) :: xi, yi, zi
      type(mfArray) :: fi
      !------ API end ------
#ifdef _DEVLP

      ! x, y, z, f : must be vectors, representing data (f) on an
      !              irregular grid (x,y,z).
      !
      ! (xi,yi,zi) must have the same shape (scalars, vectors, or matrices);
      ! fi will have the same shape as them.
      !
      ! fi is set to MF_NAN if (xi,yi,zi) is outside of the convex hull
      ! of the set of nodes described by (x,y,z).
      !
      ! a linear interpolation is used

      real(kind=MF_DOUBLE), pointer :: x_ptr_vec(:), y_ptr_vec(:), z_ptr_vec(:)
      real(kind=MF_DOUBLE), pointer :: f_ptr_vec(:)

      type(mfArray) :: tetra
      type(mfTetraConnect) :: tetra_connect
      real(kind=MF_DOUBLE) :: xi_prev, yi_prev, zi_prev, xlong
      real(kind=MF_DOUBLE) :: xi_curr, yi_curr, zi_curr, dist
      integer :: nb_pts, i, j, k, k_prev, ni, nj
      real(kind=MF_DOUBLE) :: x0, x1, x2, x3, y0, y1, y2, y3, z0, z1, z2, z3
      real(kind=MF_DOUBLE) :: volume, a1, a2, a3, b1, b2, b3, c1, c2, c3
      real(kind=MF_DOUBLE) :: xnew, ynew, znew, f1, f2, f3, f4


      character(len=*), parameter :: ROUTINE_NAME = "mfGridData3D"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y, z, f, xi, yi, zi )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) .or. mfIsEmpty(z) .or.         &
          mfIsEmpty(f) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE .or. &
          z%data_type /= MF_DT_DBLE .or. f%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x%shape(1) /= 1 .and. x%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a vector!" )
         go to 99
      end if

      if( y%shape(1) /= 1 .and. y%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y' must be a vector!" )
         go to 99
      end if

      if( z%shape(1) /= 1 .and. z%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'z' must be a vector!" )
         go to 99
      end if

      if( f%shape(1) /= 1 .and. f%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'f' must be a vector!" )
         go to 99
      end if

      if( x%shape(1) /= y%shape(1) .or. x%shape(1) /= z%shape(1) .or.   &
          x%shape(1) /= f%shape(1) .or. x%shape(2) /= y%shape(2) .or.   &
          x%shape(2) /= z%shape(2) .or. x%shape(2) /= f%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x', 'y', 'z' and 'f' must have the same shape!" )
         go to 99
      end if

      if( f%status_temporary ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'f' cannot be tempo!" )
      end if

      call msPointer( f, f_ptr_vec, no_crc=.true. )

      nb_pts = size(x%double)

      if( nb_pts < 4 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x', 'y', 'z' and 'f' must have at least 4 values!" )
         go to 99
      end if

      call msAssign( tetra, mfDelaunay3D( x, y, z ) )

      call msBuildTetraConnect( x, y, z, tetra, tetra_connect )

      if( xi%shape(1)/=yi%shape(1) .or. xi%shape(1)/=zi%shape(1) .or.   &
          xi%shape(2)/=yi%shape(2) .or. xi%shape(2)/=zi%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'xi', 'yi' and 'zi' must have the same shape!" )
         go to 99
      end if

      ni = size( xi%double, 1 )
      nj = size( xi%double, 2 )

      call msAssign( fi, mfZeros(ni,nj) )

      xi_prev = MF_REALMAX
      yi_prev = MF_REALMAX
      zi_prev = MF_REALMAX
      ! 'xlong' is the characteristic length of the current tetrahedron,
      ! it will be estimated inside the loops below, for each tetrahedron...
      xlong = 1.0d0

      ! k_prev is useful to avoid computing many times some coefficients
      ! related to the same triangle.
      k_prev = 0

      do j = 1, nj
         do i = 1, ni

            xi_curr = xi%double(i,j)
            yi_curr = yi%double(i,j)
            zi_curr = zi%double(i,j)
            dist = sqrt( (xi_prev-xi_curr)**2 + (yi_prev-yi_curr)**2 +  &
                         (zi_prev-zi_curr)**2  )

            ! if dist is short compared to xlong, we can start the
            ! tetra-search with the previous i_tetra
            if( dist < 3.0d0*xlong ) then
               k = tsearch_3D( tetra_connect%n_xyz, tetra_connect%tetra_n, &
                               tetra_connect%nodes_sorted_x,            &
                               tetra_connect%nodes_sorted_y,            &
                               tetra_connect%nodes_sorted_z,            &
                               tetra_connect%face_tetra, tetra_connect%tetra_f, &
                               tetra_connect%n_tetra,                   &
                               xi_curr, yi_curr, zi_curr, previous=.true. )
            else
               k = tsearch_3D( tetra_connect%n_xyz, tetra_connect%tetra_n, &
                               tetra_connect%nodes_sorted_x,            &
                               tetra_connect%nodes_sorted_y,            &
                               tetra_connect%nodes_sorted_z,            &
                               tetra_connect%face_tetra, tetra_connect%tetra_f, &
                               tetra_connect%n_tetra,                   &
                               xi_curr, yi_curr, zi_curr )
            end if

            if( k == 0 ) then
               fi%double(i,j) = MF_NAN
               cycle
            end if

            if( k /= k_prev ) then

               ! linear interpolation in this tetrahedron

               x0 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,1),1)
               y0 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,1),2)
               z0 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,1),3)

               ! put P1 at origin
               x1 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,2),1) - x0
               y1 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,2),2) - y0
               z1 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,2),3) - z0
               x2 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,3),1) - x0
               y2 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,3),2) - y0
               z2 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,3),3) - z0
               x3 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,4),1) - x0
               y3 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,4),2) - y0
               z3 = tetra_connect%n_xyz(tetra_connect%tetra_n(k,4),3) - z0

               ! compute coeff to map P2 to (1,0,0), P3 to (0,1,0) and
               ! P4 to (0,0,1)
               !
               !   X = a1*x + b1*y + c1*z
               !   Y = a2*x + b2*y + c2*z
               !   Z = a3*x + b3*y + c3*z
               !
               volume = x1*(y2*z3-y3*z2) + y1*(z2*x3-z3*x2) + z1*(x2*y3-x3*y2)
               if( volume == 0.0d0 ) then
                  call PrintMessage( trim(ROUTINE_NAME), "E",           &
                                     "volume = 0!",                    &
                                     "(incorrect tetrahedralization ?)" )
                  go to 99
               end if
               a1 = (y2*z3-y3*z2) / volume
               b1 = (z2*x3-z3*x2) / volume
               c1 = (x2*y3-x3*y2) / volume
               a2 = (y3*z1-y1*z3) / volume
               b2 = (z3*x1-z1*x3) / volume
               c2 = (x3*y1-x1*y3) / volume
               a3 = (y1*z2-y2*z1) / volume
               b3 = (z1*x2-z2*x1) / volume
               c3 = (x1*y2-x2*y1) / volume

               f1 = f_ptr_vec( tetra_connect%tetra_n(k,1) )
               f2 = f_ptr_vec( tetra_connect%tetra_n(k,2) )
               f3 = f_ptr_vec( tetra_connect%tetra_n(k,3) )
               f4 = f_ptr_vec( tetra_connect%tetra_n(k,4) )

            end if

            ! compute (Xnew,Ynew,Znew) image of (xi,yi,zi)
            xnew = a1*(xi_curr-x0) + b1*(yi_curr-y0) + c1*(zi_curr-z0)
            ynew = a2*(xi_curr-x0) + b2*(yi_curr-y0) + c2*(zi_curr-z0)
            znew = a3*(xi_curr-x0) + b3*(yi_curr-y0) + c3*(zi_curr-z0)

            ! compute linear value at (xi,yi,zi)
            fi%double(i,j) = f1 + xnew*(f2-f1) + ynew*(f3-f1) + znew*(f4-f1)

            ! updates for next values of (i,j)

            xi_prev = xi_curr
            yi_prev = yi_curr
            zi_prev = zi_curr
            ! a good estimation for 'xlong' comes from its volume
            xlong = volume**0.333d0

            k_prev = k

         end do
      end do

      call msFreePointer( f, f_ptr_vec )

      call msRelease( tetra )
      call msRelease( tetra_connect )
      call msEndDelaunay3D( )

      if( mf_phys_units ) then
         ! 'fi' contains interpolated values from 'f'
         fi%units(:) = f%units(:)
      end if

      fi%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y, z, f, xi, yi, zi )
      call msAutoRelease( x, y, z, f, xi, yi, zi )

      call mf_restore_fpe( )

#endif
   end function mfGridData3D
