! f90 include file

!_______________________________________________________________________
!
   function mfGridData( x, y, f, xi, yi ) result( fi )

      type(mfArray) :: x, y, f
      type(mfArray) :: xi, yi
      type(mfArray) :: fi
      !------ API end ------
#ifdef _DEVLP

      ! x, y, f : must be vectors, representing data (f) on an
      !           irregular grid (x,y).
      !
      ! (xi,yi) must have the same shape (scalars, vectors, or matrices);
      ! fi will have the same shape as them.
      !
      ! fi is set to MF_NAN if (xi,yi) is outside of the convex hull
      ! of the set of nodes described by (x,y).
      !
      ! a linear interpolation is used

      real(kind=MF_DOUBLE), pointer :: x_ptr_vec(:), y_ptr_vec(:)
      real(kind=MF_DOUBLE), pointer :: f_ptr_vec(:)

      type(mfArray) :: tri
      type(mfTriConnect) :: tri_connect
      real(kind=MF_DOUBLE) :: xi_prev, yi_prev, xlong
      real(kind=MF_DOUBLE) :: xi_curr, yi_curr, dist
      integer :: nb_pts, i, j, k, k_prev, ni, nj
      real(kind=MF_DOUBLE) :: x0, x1, x2, y0, y1, y2
      real(kind=MF_DOUBLE) :: surf, a1, a2, b1, b2
      real(kind=MF_DOUBLE) :: xnew, ynew, f1, f2, f3


      character(len=*), parameter :: ROUTINE_NAME = "mfGridData"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y, f, xi, yi )

      if( mfIsEmpty(x) .or. mfIsEmpty(y) .or. mfIsEmpty(f) ) then
         go to 99
      end if

      if( x%data_type /= MF_DT_DBLE .or. y%data_type /= MF_DT_DBLE .or. &
          f%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x%shape(1) /= 1 .and. x%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x' must be a vector!" )
         go to 99
      end if

      if( y%shape(1) /= 1 .and. y%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y' must be a vector!" )
         go to 99
      end if

      if( f%shape(1) /= 1 .and. f%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'f' must be a vector!" )
         go to 99
      end if

      if( x%shape(1) /= y%shape(1) .or. x%shape(1) /= f%shape(1) .or.   &
          x%shape(2) /= y%shape(2) .or. x%shape(2) /= f%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x', 'y' and 'f' must have the same shape!" )
         go to 99
      end if

      if( f%status_temporary ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'f' cannot be tempo!" )
      end if

      call msPointer( f, f_ptr_vec, no_crc=.true. )

      nb_pts = size(x%double)

      if( nb_pts < 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x', 'y' and 'f' must have at least 3 values!" )
         go to 99
      end if

      call msAssign( tri, mfDelaunay( x, y ) )

      call msBuildTriConnect( x, y, tri, tri_connect )

      if( xi%shape(1) /= yi%shape(1) .or. xi%shape(2) /= yi%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'xi' and 'yi' must have the same shape!" )
         go to 99
      end if

      ni = size( xi%double, 1 )
      nj = size( xi%double, 2 )

      call msAssign( fi, mfZeros(ni,nj) )

      xi_prev = MF_REALMAX
      yi_prev = MF_REALMAX
      ! 'xlong' is the characteristic length of the current triangle, it
      ! will be estimated inside the loops below, for each triangle...
      xlong = 1.0d0

      ! k_prev is useful to avoid computing many times some coefficients
      ! related to the same triangle.
      k_prev = 0

      do j = 1, nj
         do i = 1, ni

            xi_curr = xi%double(i,j)
            yi_curr = yi%double(i,j)
            dist = sqrt( (xi_prev-xi_curr)**2 + (yi_prev-yi_curr)**2 )

            k = tsearch( tri_connect%convex_domain, tri_connect%n_xy,   &
                         tri_connect%tri_n, tri_connect%face_tri,       &
                         tri_connect%tri_f, tri_connect%n_tri,          &
                         tri_connect%face_n, tri_connect%faces_boundary, &
                         tri_connect%faces_boundary_ptr,                &
                         xi_curr, yi_curr )

            if( k == 0 ) then
               fi%double(i,j) = MF_NAN
               cycle
            end if

            if( k /= k_prev ) then

               ! linear interpolation in this triangle

               x0 = tri_connect%n_xy(tri_connect%tri_n(k,1),1)
               y0 = tri_connect%n_xy(tri_connect%tri_n(k,1),2)

               ! put P1 at origin
               x1 = tri_connect%n_xy(tri_connect%tri_n(k,2),1) - x0
               y1 = tri_connect%n_xy(tri_connect%tri_n(k,2),2) - y0
               x2 = tri_connect%n_xy(tri_connect%tri_n(k,3),1) - x0
               y2 = tri_connect%n_xy(tri_connect%tri_n(k,3),2) - y0

               ! compute coeff to map P2 to (1,0) and P3 to (0,1)
               !
               !   X = a1*x + b1*y
               !   Y = a2*x + b2*y
               !
               surf = (x1*y2-x2*y1)
               if( surf == 0.0d0 ) then
                  call PrintMessage( trim(ROUTINE_NAME), "E",           &
                                     "surf = 0!",                       &
                                     "(incorrect triangulation ?)" )
                  go to 99
               end if
               a1 = y2 / surf
               b1 = -x2 / surf
               a2 = -y1 / surf
               b2 = x1 / surf

               f1 = f_ptr_vec( tri_connect%tri_n(k,1) )
               f2 = f_ptr_vec( tri_connect%tri_n(k,2) )
               f3 = f_ptr_vec( tri_connect%tri_n(k,3) )

            end if

            ! compute (Xnew,Ynew) image of (xi,yi)
            xnew = a1*(xi_curr-x0) + b1*(yi_curr-y0)
            ynew = a2*(xi_curr-x0) + b2*(yi_curr-y0)

            ! compute linear value at (xi,yi)
            fi%double(i,j) = f1 + xnew*(f2-f1) + ynew*(f3-f1)

            ! updates for next values of (i,j)

            xi_prev = xi_curr
            yi_prev = yi_curr
            ! a good estimation for 'xlong' comes from its surface
            xlong = sqrt( surf )

            k_prev = k

         end do
      end do

      call msFreePointer( f, f_ptr_vec )

      call msRelease( tri )
      call msRelease( tri_connect )

      if( mf_phys_units ) then
         ! 'fi' contains interpolated values from 'f'
         fi%units(:) = f%units(:)
      end if

      fi%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y, f, xi, yi )
      call msAutoRelease( x, y, f, xi, yi )

      call mf_restore_fpe( )

#endif
   end function mfGridData
