! f90 include file

!_______________________________________________________________________
!
   function mfFunFit( x, y, fun, p0, n, tol ) result( out )

      type(mfArray) :: x, y, p0

      interface
         function fun( x, p, n ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x, p(n)
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      integer,              intent(in)           :: n
      real(kind=MF_DOUBLE), intent(in), optional :: tol

      type(mfArray)                              :: out
      !------ API end ------
#ifdef _DEVLP

      ! Fit a user-supplied function to real data
      !
      ! (x,y) are coordinates of data, as column vectors of length m
      !
      ! fun : the function to be used for model the dependance y(x),
      !       parametrized by the n values of the vector p(:)

      ! at input, p0(:) is an initial guess

      ! Solved by the 'lmdif1_funfit' routine of MINPACK

      integer :: m, info, iwork(n), lwork
      real(kind=MF_DOUBLE) :: tol_1
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec(:), work(:)

      character(len=2) :: info_str

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y, p0 )

      if( mfIsEmpty(x) ) then
         go to 99
      end if

      ! 'x' and 'y' must be numeric
      if( .not. mfIsNumeric(x) .or. .not. mfIsNumeric(y) ) then
         call PrintMessage( "mfFunFit", "E",                            &
                            "mfArrays 'x' and 'y' must be numeric!" )
         go to 99
      end if

      ! 'x' and 'y' must be dense
      if( mfIsSparse(x) .or. mfIsSparse(y) ) then
         call PrintMessage( "mfFunFit", "E",                            &
                            "mfArrays 'x' and 'y' must have dense storage!" )
         go to 99
      end if

      ! 'x' must be a column vector
      if( x%shape(2) /= 1 ) then
         call PrintMessage( "mfFunFit", "E",                            &
                            "'x' must be a column vector!" )
         go to 99
      end if

      ! 'x' and 'y' must have same shape
      if( any(x%shape /= y%shape) ) then
         call PrintMessage( "mfFunFit", "E",                            &
                            "'x' and 'y' must have the same shape!" )
         go to 99
      end if

      m = x%shape(1)
      allocate( x_data(m), y_data(m) )

      x_data(:) = x%double(:,1)
      y_data(:) = y%double(:,1)

      if( present(tol) ) then
         tol_1 = tol
      else
         tol_1 = 1.0d-6
      end if

      lwork = m*n + 5*n + m
      allocate( fvec(m), work(lwork) )

      if( mfIsEmpty(p0) ) then
         call PrintMessage( "mfFunFit", "W",                            &
                            "initial guess 'p0' is empty : cannot begin the search!" )
         go to 99
      end if

      ! 'p0' must be numeric
      if( .not. mfIsNumeric(p0) ) then
         call PrintMessage( "mfFunFit", "E",                            &
                            "mfArray 'p0' must be numeric!" )
         go to 99
      end if

      ! 'p0' must be dense
      if( mfIsSparse(p0) ) then
         call PrintMessage( "mfFunFit", "E",                            &
                            "mfArray 'p0' must have dense storage!" )
         go to 99
      end if

      ! 'p0' must be a scalar or a vector, according the value of n
      if( n == 1 ) then
         if( .not. mfIsScalar(p0) ) then
            call PrintMessage( "mfFunFit", "E",                         &
                               "'p0' must be a scalar!" )
            go to 99
         end if
      else if( n > 1 ) then
         if( .not. mfIsVector(p0) ) then
            call PrintMessage( "mfFunFit", "E",                         &
                               "'p0' must be a vector!" )
            go to 99
         end if
      else
         call PrintMessage( "mfFunFit", "E",                            &
                            "'n' cannot be negative or null!" )
         go to 99
      end if

      ! initial guess
      if( n == 1 ) then
         var(1) = p0
      else ! n > 1
         var = p0
      end if

      call lmdif1_funfit( fcn_lmdif_funfit, m, n, var, fvec, tol_1,     &
                          info, iwork, work, lwork, fun )

      if( info < 1 .or. 3 < info ) then
         ! some problem occured
         write(info_str,'(I0)') info
         call PrintMessage( "mfFunFit", "E",                            &
                            "[minpack]/lmdif1_funfit routine failed.",  &
                            "for your information: info = " // info_str )
         go to 99
      end if

      out = var

      deallocate( x_data, y_data )

      if( mf_phys_units ) then

         out%units(:) = p0%units(:)

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x, y, p0 )
      call msAutoRelease( x, y, p0 )

      call mf_restore_fpe( )

#endif
   end function mfFunFit
!_______________________________________________________________________
!
   subroutine msFunFit( out, x, y, fun, p0, n, tol )

      type(mfArray) :: x, y, p0

      interface
         function fun( x, p, n ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x, p(n)
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      integer,              intent(in)           :: n
      real(kind=MF_DOUBLE), intent(in), optional :: tol

      type(mf_Out)                               :: out
      !------ API end ------
#ifdef _DEVLP

      ! Fit a user-supplied function to real data
      !
      ! (x,y) are coordinates of data, as column vectors of length m
      !
      ! fun : the function to be used for model the dependance y(x),
      !       parametrized by the n values of the vector p(:)

      ! at input, p0(:) is an initial guess

      ! Solved by the 'lmdif1_funfit' routine of MINPACK

      integer :: m, info, iwork(n), lwork
      real(kind=MF_DOUBLE) :: tol_1, mean_of_y
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec(:), work(:)

      character(len=2) :: info_str

      type(mfArray), pointer :: p, r2

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x, y, p0 )

      ! 2 out-args must be specified
      if( out%n /= 2 ) then
         call PrintMessage( "msFunFit", "E",                            &
                            "two output args required!",               &
                            "syntax is : call msFunFit( mfOut(p,r2), x, y, fun, p0, n [, tol])" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, x ) .or. &
          .not. args_mfout_ok( out, y ) .or. &
          .not. args_mfout_ok( out, p0 )     ) then
         call PrintMessage( "msFunFit", "E",                            &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      p => out%ptr1
      r2 => out%ptr2
      call msSilentRelease( p, r2 )

      if( mfIsEmpty(x) ) then
         go to 99
      end if

      ! 'x' and 'y' must be numeric
      if( .not. mfIsNumeric(x) .or. .not. mfIsNumeric(y) ) then
         call PrintMessage( "msFunFit", "E",                            &
                            "mfArrays 'x' and 'y' must be numeric!" )
         go to 99
      end if

      ! 'x' and 'y' must be dense
      if( mfIsSparse(x) .or. mfIsSparse(y) ) then
         call PrintMessage( "msFunFit", "E",                            &
                            "mfArrays 'x' and 'y' must have dense storage!" )
         go to 99
      end if

      ! 'x' must be a column vector
      if( x%shape(2) /= 1 ) then
         call PrintMessage( "msFunFit", "E",                            &
                            "'x' must be a column vector!" )
         go to 99
      end if

      ! 'x' and 'y' must have same shape
      if( any(x%shape /= y%shape) ) then
         call PrintMessage( "msFunFit", "E",                            &
                            "'x' and 'y' must have the same shape!" )
         go to 99
      end if

      m = x%shape(1)
      allocate( x_data(m), y_data(m) )

      x_data(:) = x%double(:,1)
      y_data(:) = y%double(:,1)

      if( present(tol) ) then
         tol_1 = tol
      else
         tol_1 = 1.0d-6
      end if

      lwork = m*n + 5*n + m
      allocate( fvec(m), work(lwork) )

      if( mfIsEmpty(p0) ) then
         call PrintMessage( "msFunFit", "W",                            &
                            "initial guess 'p0' is empty : cannot begin the search!" )
         go to 99
      end if

      ! 'p0' must be numeric
      if( .not. mfIsNumeric(p0) ) then
         call PrintMessage( "msFunFit", "E",                            &
                            "mfArray 'p0' must be numeric!" )
         go to 99
      end if

      ! 'p0' must be dense
      if( mfIsSparse(p0) ) then
         call PrintMessage( "msFunFit", "E",                            &
                            "mfArray 'p0' must have dense storage!" )
         go to 99
      end if

      ! 'p0' must be a scalar or a vector, according the value of n
      if( n == 1 ) then
         if( .not. mfIsScalar(p0) ) then
            call PrintMessage( "msFunFit", "E",                         &
                               "'p0' must be a scalar!" )
            go to 99
         end if
      else if( n > 1 ) then
         if( .not. mfIsVector(p0) ) then
            call PrintMessage( "msFunFit", "E",                         &
                               "'p0' must be a vector!" )
            go to 99
         end if
      else
         call PrintMessage( "msFunFit", "E",                            &
                            "'n' cannot be negative or null!" )
         go to 99
      end if

      ! initial guess
      if( n == 1 ) then
         var(1) = p0
      else ! n > 1
         var = p0
      end if

      call lmdif1_funfit( fcn_lmdif_funfit, m, n, var, fvec, tol_1,     &
                          info, iwork, work, lwork, fun )

      if( info < 1 .or. 3 < info ) then
         ! some problem occured
         write(info_str,'(I0)') info
         call PrintMessage( "msFunFit", "E",                            &
                            "[minpack]/lmdif1_funfit routine failed.",  &
                            "for your information: info = " // info_str )
         go to 99
      end if

      p = var

      ! compute correlation index, if required
      if( out%n == 2 ) then
         mean_of_y = sum(y_data(:))/m
         r2 = 1. - sum((fvec(:))**2) / sum((y_data(:)-mean_of_y)**2)
      end if

      deallocate( x_data, y_data )

      if( mf_phys_units ) then

         p%units(:) = p0%units(:)

      end if

 99   continue

      call msFreeArgs( x, y, p0 )
      call msAutoRelease( x, y, p0 )

      call mf_restore_fpe( )

#endif
   end subroutine msFunFit
