! f90 include file

!_______________________________________________________________________
!
   function mfDelaunay( x_in, y_in ) result( tri )

      type(mfArray) :: x_in, y_in
      type(mfArray) :: tri
      !------ API end ------
#ifdef _DEVLP

      real(kind=MF_DOUBLE), pointer :: x_ptr_vec(:), y_ptr_vec(:)
      integer, allocatable :: tri_n(:,:)
      integer :: status
      character(len=80) :: options

      integer :: nb_pts, nb_pts_2, nb_tri

      character(len=*), parameter :: ROUTINE_NAME = "mfDelaunay"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x_in, y_in )

      if( mfIsEmpty(x_in) .or. mfIsEmpty(y_in) ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "'x_in' or 'y_in' is empty!" )
         go to 99
      end if

      if( x_in%data_type /= MF_DT_DBLE .or. y_in%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x_in%shape(1) /= 1 .and. x_in%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' must be a vector!" )
         go to 99
      end if

      if( y_in%shape(1) /= 1 .and. y_in%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y_in' must be a vector!" )
         go to 99
      end if

      if( x_in%shape(1) /= y_in%shape(1) .or. x_in%shape(2) /= y_in%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' and 'y_in' must have the same shape!" )
         go to 99
      end if

!### This can be disable, since the 'no_crc' option is used below...
!!      if( x_in%status_temporary ) then
!!         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
!!                            "mfArray 'x_in' cannot be tempo!" )
!!      end if

!### This can be disable, since the 'no_crc' option is used below...
!!      if( y_in%status_temporary ) then
!!         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
!!                            "mfArray 'y_in' cannot be tempo!" )
!!      end if

      call msPointer( x_in, x_ptr_vec, no_crc=.true. )
      call msPointer( y_in, y_ptr_vec, no_crc=.true. )

      nb_pts = size(x_ptr_vec)

      if( nb_pts < 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' and 'y_in' must have at least 3 values!" )
         go to 99
      end if

      options = "Q" ! quiet mode for Triangle
      call triangle_delaunay( nb_pts, x_ptr_vec(1), y_ptr_vec(1),       &
                              trim(options)//char(0), nb_pts_2, nb_tri )

      if( nb_pts_2 /= nb_pts ) then
         write(STDERR,*) "(MUESLI mfDelaunay:) internal error."
         write(STDERR,*) "                     on output nb_pts_2 should be equal to nb_pts."
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      end if

      call msFreePointer( x_in, x_ptr_vec )
      call msFreePointer( y_in, y_ptr_vec )

      allocate( tri_n(nb_tri,3) )

      call triangle_get_tri_table( nb_tri, tri_n )

      call triangle_free_all( )

      tri%data_type = MF_DT_DBLE
      tri%shape = [ nb_tri, 3 ]
      allocate( tri%double(nb_tri,3) )

      tri%double(:,:) = tri_n(:,:)
      deallocate( tri_n )

      if( mf_phys_units ) then
         ! 'x_in' and 'y_in' must have the same physical units
         ! verifying the physical dimension
         call verif_adim( x_in%units, y_in%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x_in' and 'y_in'",&
                               "are not consistent!" )
            go to 99
         end if
      end if

      tri%status_temporary = .true.

 99   continue

      call msFreeArgs( x_in, y_in )
      call msAutoRelease( x_in, y_in )

#endif
   end function mfDelaunay
!_______________________________________________________________________
!
   subroutine msDelaunay_xy( out, x_in, y_in, theta_min, area_max )

      type(mfArray)                  :: x_in, y_in
      real(kind=MF_DOUBLE), optional :: theta_min, area_max
      type(mf_Out)                   :: out
      !------ API end ------
#ifdef _DEVLP

      real(kind=MF_DOUBLE), pointer :: x_ptr_vec(:), y_ptr_vec(:)
      integer, allocatable :: tri_n(:,:)
      integer :: status
      character(len=20) :: theta_min_str, area_max_str
      character(len=80) :: options

      integer :: nb_pts, nb_pts_2, nb_tri

      type(mfArray), pointer :: x, y, tri

      character(len=*), parameter :: ROUTINE_NAME = "msDelaunay"

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( x_in, y_in )

      ! 3 out-args must be specified
      if( out%n /= 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "three output args required!",              &
                            "syntax is : call msDelaunay ( mfOut(x,y,tri), x_in, y_in, ... )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, x_in ) .or.                         &
          .not. args_mfout_ok( out, y_in )      ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "output arguments cannot be tempo, or cannot share", &
                            "same memory as another input argument." )
         go to 99
      end if

      x => out%ptr1
      y => out%ptr2
      tri => out%ptr3
      call msSilentRelease( x, y, tri )

      options = "Q" ! quiet mode for Triangle

      if( present(theta_min) ) then
         if( theta_min > 28.6d0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'theta_min' cannot be larger than 28.6 deg." )
            go to 99
         end if
         write(theta_min_str,"(F6.2)") theta_min
         call check_and_remove_trailing_zero( theta_min_str )
         options = trim(options) // "q" // trim(adjustl(theta_min_str))
      else
         if( .not. present(area_max) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "among the two optional args 'theta_min' and 'area_max'", &
                               "at least one is required." )
            go to 99
         end if
      end if

      if( present(area_max) ) then
         if( area_max <= 0.0d0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'area_max' must be strictly positive." )
            go to 99
         end if
         write(area_max_str,"(F12.8)") area_max
         call check_and_remove_trailing_zero( area_max_str )
         options = trim(options) // "a" // trim(adjustl(area_max_str))
      end if

      if( mfIsEmpty(x_in) .or. mfIsEmpty(y_in) ) then
         call PrintMessage( trim(ROUTINE_NAME), "W",                    &
                            "'x_in' or 'y_in' is empty!" )
         go to 99
      end if

      if( x_in%data_type /= MF_DT_DBLE .or. y_in%data_type /= MF_DT_DBLE ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "real arrays required!" )
         go to 99
      end if

      if( x_in%shape(1) /= 1 .and. x_in%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' must be a vector!" )
         go to 99
      end if

      if( y_in%shape(1) /= 1 .and. y_in%shape(2) /= 1 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y_in' must be a vector!" )
         go to 99
      end if

      if( x_in%shape(1) /= y_in%shape(1) .or. x_in%shape(2) /= y_in%shape(2) ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' and 'y_in' must have the same shape!" )
         go to 99
      end if

      if( x_in%status_temporary ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' cannot be tempo!" )
      end if

      if( y_in%status_temporary ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'y_in' cannot be tempo!" )
      end if

      call msPointer( x_in, x_ptr_vec, no_crc=.true. )
      call msPointer( y_in, y_ptr_vec, no_crc=.true. )

      nb_pts = size(x_ptr_vec)

      if( nb_pts < 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "mfArray 'x_in' and 'y_in' must have at least 3 values!" )
         go to 99
      end if

      call triangle_delaunay( nb_pts, x_ptr_vec(1), y_ptr_vec(1),       &
                              trim(options)//char(0), nb_pts_2, nb_tri )

      call msFreePointer( x_in, x_ptr_vec )
      call msFreePointer( y_in, y_ptr_vec )

      x%data_type = MF_DT_DBLE
      x%shape = [ nb_pts_2, 1 ]
      allocate( x%double(nb_pts_2,1) )
      y%data_type = MF_DT_DBLE
      y%shape = [ nb_pts_2, 1 ]
      allocate( y%double(nb_pts_2,1) )
      call triangle_get_pts_coords( x%double(:,1), y%double(:,1) )

      allocate( tri_n(nb_tri,3) )
      call triangle_get_tri_table( nb_tri, tri_n )

      call triangle_free_all( )

      tri%data_type = MF_DT_DBLE
      tri%shape = [ nb_tri, 3 ]
      allocate( tri%double(nb_tri,3) )

      tri%double(:,:) = tri_n(:,:)
      deallocate( tri_n )

      if( mf_phys_units ) then
         ! 'x_in' and 'y_in' must have the same physical units
         ! verifying the physical dimension
         call verif_adim( x_in%units, y_in%units, status )
         if( status /= 0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "the physical dimensions of 'x_in' and 'y_in'",&
                               "are not consistent!" )
            go to 99
         end if
      end if

 99   continue

      call msFreeArgs( x_in, y_in )
      call msAutoRelease( x_in, y_in )

#endif
   end subroutine msDelaunay_xy
!_______________________________________________________________________
!
   subroutine msDelaunay_PLdomain( out, PL_domain, theta_min, area_max )

      type(mfPLdomain)               :: PL_domain
      real(kind=MF_DOUBLE), optional :: theta_min, area_max
      type(mf_Out)                   :: out
      !------ API end ------
#ifdef _DEVLP

      integer, allocatable :: tri_n(:,:)
      integer :: status
      character(len=20) :: theta_min_str, area_max_str
      character(len=80) :: options
      real(kind=MF_DOUBLE) :: dummy_vec(1)
      integer :: nb_pts, nb_seg, nb_hol, nb_pts_2, nb_tri

      type(mfArray), pointer :: x, y, tri

      character(len=*), parameter :: ROUTINE_NAME = "msDelaunay"

   !------ end of declarations -- execution starts hereafter  ------

      ! Checks data presence in the PL_domain structure
      if( allocated(PL_domain%n_xy) ) then
         if( size(PL_domain%n_xy,2) /= 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                              "bad 2nd dimension of v n_xy array!" )
            go to 99
         end if
         nb_pts = size(PL_domain%n_xy,1)
         if( nb_pts < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few nodes in PL_domain!",           &
                               "(you must have at least 3 nodes)" )
            go to 99
         end if
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "v seems empty or corrupted! (no nodes)" )
         go to 99
      end if
      if( allocated(PL_domain%edge_n) ) then
         if( size(PL_domain%edge_n,2) /= 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "bad 2nd dimension of PL_domain edge_n array!" )
            go to 99
         end if
         nb_seg = size(PL_domain%edge_n,1)
         if( nb_seg < 3 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "too few edges in PL_domain!",           &
                               "(you must have at least 3 edges)" )
            go to 99
         end if
      else
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "PL_domain seems empty or corrupted! (no edges)" )
         go to 99
      end if
      nb_hol = 0
      if( allocated(PL_domain%holes_xy) ) then
         if( size(PL_domain%holes_xy,2) /= 2 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "bad 2nd dimension of PL_domain holes_xy array!" )
            go to 99
         end if
         nb_hol = size(PL_domain%holes_xy,1)
      end if

      ! 3 out-args must be specified
      if( out%n /= 3 ) then
         call PrintMessage( trim(ROUTINE_NAME), "E",                    &
                            "three output args required!",              &
                            "syntax is : call msDelaunay ( mfOut(x,y,tri), PL_domain, ... )" )
         go to 99
      end if

      x => out%ptr1
      y => out%ptr2
      tri => out%ptr3
      call msSilentRelease( x, y, tri )

      options = "Qp" ! quiet mode for Triangle + triangulation from a PSLG

      if( present(theta_min) ) then
         if( theta_min > 28.6d0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'theta_min' cannot be larger than 28.6 deg." )
            go to 99
         end if
         write(theta_min_str,"(F6.2)") theta_min
         call check_and_remove_trailing_zero( theta_min_str )
         options = trim(options) // "q" // trim(adjustl(theta_min_str))
      else
         if( .not. present(area_max) ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "among the two optional args 'theta_min' and 'area_max'", &
                               "at least one is required." )
            go to 99
         end if
      end if

      if( present(area_max) ) then
         if( area_max <= 0.0d0 ) then
            call PrintMessage( trim(ROUTINE_NAME), "E",                 &
                               "'area_max' must be strictly positive." )
            go to 99
         end if
         write(area_max_str,"(F12.8)") area_max
         call check_and_remove_trailing_zero( area_max_str )
         options = trim(options) // "a" // trim(adjustl(area_max_str))
      end if

      if( nb_hol == 0 ) then
         ! Using 'dummy_vec' avoids a warning with Valgrind.
         call triangle_delaunay_pslg(                                   &
              nb_pts, PL_domain%n_xy(:,1), PL_domain%n_xy(:,2),         &
              nb_seg, PL_domain%edge_n(:,1), PL_domain%edge_n(:,2),     &
              nb_hol, dummy_vec, dummy_vec,                             &
              trim(options)//char(0),                                   &
              nb_pts_2, nb_tri )
      else
         call triangle_delaunay_pslg(                                   &
              nb_pts, PL_domain%n_xy(:,1), PL_domain%n_xy(:,2),         &
              nb_seg, PL_domain%edge_n(:,1), PL_domain%edge_n(:,2),     &
              nb_hol, PL_domain%holes_xy(:,1), PL_domain%holes_xy(:,2), &
              trim(options)//char(0),                                   &
              nb_pts_2, nb_tri )
      end if

      x%data_type = MF_DT_DBLE
      x%shape = [ nb_pts_2, 1 ]
      allocate( x%double(nb_pts_2,1) )
      y%data_type = MF_DT_DBLE
      y%shape = [ nb_pts_2, 1 ]
      allocate( y%double(nb_pts_2,1) )
      call triangle_get_pts_coords( x%double(:,1), y%double(:,1) )

      allocate( tri_n(nb_tri,3) )
      call triangle_get_tri_table( nb_tri, tri_n )

      call triangle_free_all( )

      tri%data_type = MF_DT_DBLE
      tri%shape = [ nb_tri, 3 ]
      allocate( tri%double(nb_tri,3) )

      tri%double(:,:) = tri_n(:,:)
      deallocate( tri_n )

 99   continue

#endif
   end subroutine msDelaunay_PLdomain
