! f90 include file

!_______________________________________________________________________
!
   function mfMul_A_B( A, B ) result( out )

      type(mfArray), intent(in) :: A, B
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! true matrix product : A*B (non commutative)
      !
      ! for the elemental operator (Fortran '*' overloading),
      ! cf. mfMul_mfArray_mfArray and Co.

      ! //-OpenMP

      ! see also: mfMul_At_B
      !           mfMul_A_Bt

      integer :: m, n, nrow, ncol, nzamub, nzmax, ierr
      integer :: i, j, status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      ! verifications...
      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "'A' is not allocated!" )
         go to 99
      end if

      if( mfIsEmpty(B) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "'B' is not allocated!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. a%data_type == MF_DT_SP_BOOL .or. &
          b%data_type == MF_DT_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "mfMul", "E",                               &
                            "args cannot be boolean mfArray!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC .or. B%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfMul", "E",                               &
                            "this function cannot be applied to a permutation!", &
                            "(use mfColPerm or mfRowPerm instead)" )
         go to 99
      end if

      if( A%shape(2) /= B%shape(1) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "'A' and 'B' must have conformant shapes." )
         go to 99
      end if

      m = A%shape(1)
      n = B%shape(2)
      out%shape = [ m, n ]
      if( A%data_type == MF_DT_SP_DBLE .and.                            &
          B%data_type == MF_DT_SP_DBLE       ) then
         ! usual case 'matrix * matrix'
         out%data_type = MF_DT_SP_DBLE
         ! nzmax choice: at least nnzamub, but preserve extra rooms
         ! from initial sparse arrays.
         call nnzamub( m, n, a%i, a%j, b%i, b%j, nzamub )
         nzmax = max( max(mfNzmax(a),mfNzmax(b)), nzamub )
         call msAssign( out, mfSpAlloc(m,n,nzmax))

         ! compute : out(m,n) = A(m,p)*B(p,n)
         call amub( m, n, A%a, A%i, A%j, B%a, B%i, B%j,                 &
                    out%a, out%i, out%j, nzmax, ierr )

         out%row_sorted = UNKNOWN

         if( ierr /= 0 ) then
            write(STDERR,*) "(MUESLI mfMul:) internal error:"
            write(STDERR,*) "                amub fails!"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
      else if( A%data_type == MF_DT_SP_CMPLX .and.                      &
               B%data_type == MF_DT_SP_CMPLX       ) then
         ! usual case 'matrix * matrix'
         out%data_type = MF_DT_SP_CMPLX
         ! nzmax choice: at least nnzamub, but preserve extra rooms
         ! from initial sparse arrays.
         call nnzamub( m, n, a%i, a%j, b%i, b%j, nzamub )
         nzmax = max( max(mfNzmax(a),mfNzmax(b)), nzamub )
         call msAssign( out, mfSpAlloc(m,n,nzmax,kind="complex"))

         ! compute : out(m,n) = A(m,p)*B(p,n)
         call amub_cmplx_cmplx( m, n, A%z, A%i, A%j, B%z, B%i, B%j,     &
                                out%z, out%i, out%j, nzmax, ierr )

         out%row_sorted = UNKNOWN

         if( ierr /= 0 ) then
            write(STDERR,*) "(MUESLI mfMul:) internal error:"
            write(STDERR,*) "                amub_cmplx_cmplx fails!"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
      else if( A%data_type == MF_DT_SP_DBLE .and.                       &
               B%data_type == MF_DT_SP_CMPLX       ) then
         ! usual case 'matrix * matrix'
         out%data_type = MF_DT_SP_CMPLX
         ! nzmax choice: at least nnzamub, but preserve extra rooms
         ! from initial sparse arrays.
         call nnzamub( m, n, a%i, a%j, b%i, b%j, nzamub )
         nzmax = max( max(mfNzmax(a),mfNzmax(b)), nzamub )
         call msAssign( out, mfSpAlloc(m,n,nzmax,kind="complex"))

         ! compute : out(m,n) = A(m,p)*B(p,n)
         call amub_real_cmplx( m, n, A%a, A%i, A%j, B%z, B%i, B%j,      &
                               out%z, out%i, out%j, nzmax, ierr )

         out%row_sorted = UNKNOWN

         if( ierr /= 0 ) then
            write(STDERR,*) "(MUESLI mfMul:) internal error:"
            write(STDERR,*) "                amub_real_cmplx fails!"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
      else if( A%data_type == MF_DT_SP_CMPLX .and.                      &
               B%data_type == MF_DT_SP_DBLE        ) then
         ! usual case 'matrix * matrix'
         out%data_type = MF_DT_SP_CMPLX
         ! nzmax choice: at least nnzamub, but preserve extra rooms
         ! from initial sparse arrays.
         call nnzamub( m, n, a%i, a%j, b%i, b%j, nzamub )
         nzmax = max( max(mfNzmax(a),mfNzmax(b)), nzamub )
         call msAssign( out, mfSpAlloc(m,n,nzmax,kind="complex"))

         ! compute : out(m,n) = A(m,p)*B(p,n)
         call amub_cmplx_real( m, n, A%z, A%i, A%j, B%a, B%i, B%j,      &
                               out%z, out%i, out%j, nzmax, ierr )

         out%row_sorted = UNKNOWN

         if( ierr /= 0 ) then
            write(STDERR,*) "(MUESLI mfMul:) internal error:"
            write(STDERR,*) "                amub_cmplx_real fails!"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
      else if( A%data_type == MF_DT_SP_DBLE .and.                       &
               B%data_type == MF_DT_DBLE          ) then
         ! computation is based on 'matrix * vector' product: if
         ! 'b' has more than one column, this product is repeated on
         ! each column of 'b'.
         out%data_type = MF_DT_DBLE
         allocate( out%double(m,n) )

         ncol = a%shape(2)
         do j = 1, n
            ! compute : out(m,1) = A(m,p)*b(p,1) for each col
            call amux( ncol, b%double(:,j), out%double(:,j),            &
                       A%a, A%i, A%j )
         end do
      else if( A%data_type == MF_DT_SP_DBLE .and.                       &
               B%data_type == MF_DT_CMPLX         ) then
         ! computation is based on 'matrix * vector' product: if
         ! 'b' has more than one column, this product is repeated on
         ! each column of 'b'.
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         ncol = a%shape(2)
         do j = 1, n
            ! compute : out(m,1) = A(m,p)*b(p,1)
            call amux_real_cmplx( ncol, b%cmplx(:,j), out%cmplx(:,j),   &
                                  A%a, A%i, A%j )
         end do
      else if( A%data_type == MF_DT_SP_CMPLX .and.                      &
               B%data_type == MF_DT_DBLE           ) then
         ! computation is based on 'matrix * vector' product: if
         ! 'b' has more than one column, this product is repeated on
         ! each column of 'b'.
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         ncol = a%shape(2)
         do j = 1, n
            ! compute : out(m,1) = A(m,p)*b(p,1)
            call amux_cmplx_real( ncol, b%double(:,j), out%cmplx(:,j),  &
                                  A%z, A%i, A%j )
         end do
      else if( A%data_type == MF_DT_SP_CMPLX .and.                      &
               B%data_type == MF_DT_CMPLX           ) then
         ! computation is based on 'matrix * vector' product: if
         ! 'b' has more than one column, this product is repeated on
         ! each column of 'b'.
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         ncol = a%shape(2)
         do j = 1, n
            ! compute : out(m,1) = A(m,p)*b(p,1)
            call amux_cmplx_cmplx( ncol, b%cmplx(:,j), out%cmplx(:,j),  &
                                   A%z, A%i, A%j )
         end do
      else if( A%data_type == MF_DT_DBLE .and.                          &
               B%data_type == MF_DT_SP_DBLE    ) then
         ! computation is based on 'vector * matrix' product: if
         ! 'a' has more than one row, this product is repeated on
         ! each row of 'a'.
         out%data_type = MF_DT_DBLE
         allocate( out%double(m,n) )

         nrow = b%shape(1)
         ncol = n
         do i = 1, m
            ! compute : out(1,n) = a(1,p)*B(p,n)
            call xmua( nrow, ncol, a%double(i,:), out%double(i,:),      &
                       B%a, B%i, B%j )
         end do
      else if( A%data_type == MF_DT_CMPLX .and.                         &
               B%data_type == MF_DT_SP_DBLE    ) then
         ! computation is based on 'vector * matrix' product: if
         ! 'a' has more than one row, this product is repeated on
         ! each row of 'a'.
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         nrow = b%shape(1)
         ncol = n
         do i = 1, m
            ! compute : out(1,n) = a(1,p)*B(p,n)
            call xmua_cmplx_real( nrow, ncol, a%cmplx(i,:), out%cmplx(i,:), &
                                  B%a, B%i, B%j )
         end do
      else if( A%data_type == MF_DT_DBLE .and.                          &
               B%data_type == MF_DT_SP_CMPLX    ) then
         ! computation is based on 'vector * matrix' product: if
         ! 'a' has more than one row, this product is repeated on
         ! each row of 'a'.
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         nrow = b%shape(1)
         ncol = n
         do i = 1, m
            ! compute : out(1,n) = a(1,p)*B(p,n)
            call xmua_real_cmplx( nrow, ncol, a%double(i,:), out%cmplx(i,:), &
                                  B%z, B%i, B%j )
         end do
      else if( A%data_type == MF_DT_CMPLX .and.                         &
               B%data_type == MF_DT_SP_CMPLX    ) then
         ! computation is based on 'vector * matrix' product: if
         ! 'a' has more than one row, this product is repeated on
         ! each row of 'a'.
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(m,n) )

         nrow = b%shape(1)
         ncol = n
         do i = 1, m
            ! compute : out(1,n) = a(1,p)*B(p,n)
            call xmua_cmplx_cmplx( nrow, ncol, a%cmplx(i,:), out%cmplx(i,:), &
                                   B%z, B%i, B%j )
         end do
      else if( A%data_type == MF_DT_DBLE .and.                          &
               B%data_type == MF_DT_DBLE       ) then
         out%data_type = MF_DT_DBLE

#if defined _INTEL_IFC
         allocate( out%double(m,n) )
#else
         ! if one of the mfArrays is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( a%status_temporary .and. all(a%shape == [m,n]) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else if( b%status_temporary .and. all(b%shape == [m,n]) ) then
            out%double => b%double
            call set_status_tempo_to_false( b )
         else
            allocate( out%double(m,n) )

         end if
#endif

#if defined _INTEL_IFC
         ! for some compilers, using BLAS is always much more efficient
         ! than using the F90 intrinsic 'matmul'
         call dgemm( "N", "N", m, n, A%shape(2), 1.0d0,                 &
                     A%double, A%shape(1), B%double, B%shape(1),        &
                     0.0d0, out%double, out%shape(1) )
#else
         out%double = matmul( a%double, b%double )
#endif

      else
         out%data_type = MF_DT_CMPLX
         if( a%data_type == MF_DT_DBLE .and. b%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(m,n) )

            out%cmplx = matmul( a%double, b%cmplx )
         else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_DBLE ) then
            allocate( out%cmplx(m,n) )

            out%cmplx = matmul( a%cmplx, b%double )
         else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_CMPLX ) then

#if defined _INTEL_IFC
            allocate( out%cmplx(m,n) )
#else
            ! if one of the mfArrays is tempo and not protected,
            ! processing is done 'in place' (avoid an allocation).
            if( a%status_temporary .and. all(a%shape == [m,n]) ) then
               out%cmplx => a%cmplx
               call set_status_tempo_to_false( a )
            else if( b%status_temporary .and. all(b%shape == [m,n]) ) then
               out%cmplx => b%cmplx
               call set_status_tempo_to_false( b )
            else
               allocate( out%cmplx(m,n) )

            end if
#endif

#if defined _INTEL_IFC
            ! for some compilers, using BLAS is always much more efficient
            ! than using the F90 intrinsic 'matmul'
            call zgemm( "N", "N", m, n, A%shape(2), (1.0d0,0.0d0),      &
                        A%cmplx, A%shape(1), B%cmplx, B%shape(1),       &
                        (0.0d0,0.0d0), out%cmplx, out%shape(1) )
#else
            out%cmplx = matmul( a%cmplx, b%cmplx )
#endif

         end if
      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_add( a%units(i), b%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfMul", "E",                         &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfMul:) internal error:"
               write(STDERR,*) "                in processing physical units."
               write(STDERR,*) "                Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%prop = op2_pattern_prop( A%prop, B%prop )

!! 89   continue

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )

      call msAutoRelease( A, B )

#endif
   end function mfMul_A_B
!_______________________________________________________________________
!
   function mfMul_transp( A, B, transp ) result( out )

      type(mfArray), intent(in) :: A, B
      integer,       intent(in) :: transp
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! transp = 1 : computes A'* B
      !
      !          2 :    "     A * B'

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      if( mfIsSparse(A) .or. mfIsSparse(B) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "using the optional arg 'transp': sparse matrices not handled!" )
         go to 99
      end if
      ! other verifications are done in the following functions

      if( transp == 1 ) then
         call Mul_At_B( A, B, out )
      else if( transp == 2 ) then
         call Mul_A_Bt( A, B, out )
      else
         call PrintMessage( "mfMul", "E",                               &
                            "'transp' must be equal to 1 or 2!" )
         go to 99
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, B )

      call msAutoRelease( A, B )

#endif
   end function mfMul_transp
!_______________________________________________________________________
!
   subroutine Mul_At_B( A, B, out )

      type(mfArray) :: A, B
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! true matrix product : A'*B (non commutative)
      !
      ! shapes are: A (p,m)
      !             B (p,n)
      !
      ! for the elemental operator (Fortran '*' overloading),
      ! cf. mfMul_mfArray_mfArray and Co.

      ! Matrices A and B must be both dense.

      ! see also: mfMul_A_B
      !             Mul_A_Bt

      integer :: m, n
      integer :: i, status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      ! verifications...
      if( mfIsEmpty(a) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "'A' is not allocated!" )
         go to 99
      end if

      if( mfIsEmpty(b) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "'B' is not allocated!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. a%data_type == MF_DT_SP_BOOL .or. &
          b%data_type == MF_DT_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "mfMul", "E",                               &
                            "args cannot be boolean mfArray!" )
         go to 99
      end if

      if( a%data_type == MF_DT_PERM_VEC .or. b%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfMul", "E",                               &
                            "this function cannot be applied to a permutation!", &
                            "(use mfColPerm or mfRowPerm instead)" )
         go to 99
      end if

      if( a%shape(1) /= b%shape(1) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "'A' and 'B' must have conformant shapes.", &
                            "(currently: product A'*B)" )
         go to 99
      end if

      m = a%shape(2)
      n = b%shape(2)
      out%shape = [ m, n ]
      if( a%data_type == MF_DT_DBLE .and.                               &
          b%data_type == MF_DT_DBLE       ) then
         out%data_type = MF_DT_DBLE
         allocate( out%double(m,n) )

         ! BLAS-3 call
         call dgemm( "T", "N", m, n, A%shape(1), 1.0d0,                 &
                     A%double, A%shape(1), B%double, B%shape(1),        &
                     0.0d0, out%double, out%shape(1) )

      else
         out%data_type = MF_DT_CMPLX
         if( a%data_type == MF_DT_DBLE .and. b%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(m,n) )

!### TODO 2:
print *, 'Mul_At_B not yet available for DBLE/CMPLX'
mf_message_displayed = .true.
call muesli_trace( pause="yes" )
stop
!!            out%cmplx = matmul( a%double, b%cmplx )
         else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_DBLE ) then
            allocate( out%cmplx(m,n) )

!### TODO 2:
print *, 'Mul_At_B not yet available for CMPLX/DBLE'
mf_message_displayed = .true.
call muesli_trace( pause="yes" )
stop
!!            out%cmplx = matmul( a%cmplx, b%double )
         else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_CMPLX ) then

            allocate( out%cmplx(m,n) )

            ! BLAS-3 call
            call zgemm( "C", "N", m, n, A%shape(1), (1.0d0,0.0d0),      &
                        A%cmplx, A%shape(1), B%cmplx, B%shape(1),       &
                        (0.0d0,0.0d0), out%cmplx, out%shape(1) )

         end if
      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_add( a%units(i), b%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfMul", "E",                         &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfMul:) internal error:"
               write(STDERR,*) "                in processing physical units."
               write(STDERR,*) "                Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%prop = op2_pattern_prop( A%prop, B%prop )

 99   continue

      call msFreeArgs( A, B )

      call msAutoRelease( A, B )

#endif
   end subroutine Mul_At_B
!_______________________________________________________________________
!
   subroutine Mul_A_Bt( A, B, out )

      type(mfArray), intent(in) :: A, B
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! true matrix product : A*B' (non commutative)
      !
      ! shapes are: A (m,p)
      !             B (n,p)
      !
      ! for the elemental operator (Fortran '*' overloading),
      ! cf. mfMul_mfArray_mfArray and Co.

      ! Matrices A and B must be both dense.

      ! see also: mfMul_A_B
      !             Mul_At_B

      integer :: m, n
      integer :: i, status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, B )

      ! verifications...
      if( mfIsEmpty(a) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "'A' is not allocated!" )
         go to 99
      end if

      if( mfIsEmpty(b) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "'B' is not allocated!" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. a%data_type == MF_DT_SP_BOOL .or. &
          b%data_type == MF_DT_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "mfMul", "E",                               &
                            "args cannot be boolean mfArray!" )
         go to 99
      end if

      if( a%data_type == MF_DT_PERM_VEC .or. b%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfMul", "E",                               &
                            "this function cannot be applied to a permutation!", &
                            "(use mfColPerm or mfRowPerm instead)" )
         go to 99
      end if

      if( a%shape(2) /= b%shape(2) ) then
         call PrintMessage( "mfMul", "E",                               &
                            "'A' and 'B' must have conformant shapes.", &
                            "(currently: product A*B')" )
         go to 99
      end if

      m = a%shape(1)
      n = b%shape(1)
      out%shape = [ m, n ]
      if( a%data_type == MF_DT_DBLE .and.                               &
          b%data_type == MF_DT_DBLE       ) then
         out%data_type = MF_DT_DBLE
         allocate( out%double(m,n) )

         ! BLAS-3
         call dgemm( "N", "T", m, n, A%shape(2), 1.0d0,                 &
                     A%double, A%shape(1), B%double, B%shape(1),        &
                     0.0d0, out%double, out%shape(1) )

      else
         out%data_type = MF_DT_CMPLX
         if( a%data_type == MF_DT_DBLE .and. b%data_type == MF_DT_CMPLX ) then
            allocate( out%cmplx(m,n) )

!### TODO 2:
print *, 'Mul_A_Bt not yet available for DBLE/CMPLX'
mf_message_displayed = .true.
call muesli_trace( pause="yes" )
stop
!!            out%cmplx = matmul( a%double, b%cmplx )
         else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_DBLE ) then
            allocate( out%cmplx(m,n) )

!### TODO 2:
print *, 'Mul_A_Bt not yet available for CMPLX/DBLE'
mf_message_displayed = .true.
call muesli_trace( pause="yes" )
stop
!!            out%cmplx = matmul( a%cmplx, b%double )
         else if( a%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_CMPLX ) then

            allocate( out%cmplx(m,n) )

            ! BLAS-3
            call zgemm( "N", "C", m, n, A%shape(2), (1.0d0,0.0d0),      &
                        A%cmplx, A%shape(1), B%cmplx, B%shape(1),       &
                        (0.0d0,0.0d0), out%cmplx, out%shape(1) )

         end if
      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_add( a%units(i), b%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfMul", "E",                         &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfMul:) internal error:"
               write(STDERR,*) "                in processing physical units."
               write(STDERR,*) "                Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%prop = op2_pattern_prop( A%prop, B%prop )

 99   continue

      call msFreeArgs( A, B )

      call msAutoRelease( A, B )

#endif
   end subroutine Mul_A_Bt
