! f90 include file

!_______________________________________________________________________
!
   function mfMul_mfArray_mfArray( a, b ) result( out )

      type(mfArray), intent(in) :: a, b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! overloading of '*' (element-wise) for mfArrays
      ! (for the true matrix product, cf. mfMul in matmul.inc)

      ! the SPARSE case is not taken into account here!

      integer :: i, status
      character(len=4) :: data_type_char

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( a, b )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsEmpty(b) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "mfArray 'B' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(a) .or. mfIsSparse(b) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "sparse matrices cannot be handled in this element-wise product!",&
                            "Did you thought 'mfMul' instead?" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. a%data_type == MF_DT_SP_BOOL .or. &
          b%data_type == MF_DT_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( a%data_type == MF_DT_PERM_VEC .or. b%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a permutation!" )
         go to 99
      end if

      if( all(a%shape == b%shape) ) then

         out%shape = a%shape
         if( a%data_type == MF_DT_DBLE ) then
            if( b%data_type == MF_DT_DBLE ) then
               out%data_type = MF_DT_DBLE

               ! if the mfArray is tempo and not protected,
               ! processing is done 'in place' (avoid an allocation).
               if( can_use_memory(A) ) then
                  out%double => a%double
                  call set_status_tempo_to_false( a )
               else if( can_use_memory(B) ) then
                  out%double => b%double
                  call set_status_tempo_to_false( b )
               else
                  allocate( out%double(out%shape(1),out%shape(2)) )

               end if
               out%double = a%double * b%double
            else if( b%data_type == MF_DT_CMPLX ) then
               out%data_type = MF_DT_CMPLX
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               out%cmplx = a%double * b%cmplx
            else
               write(data_type_char,"(I0)") B%data_type
               call PrintMessage( "operator(*)", "E",                   &
                                  "internal error?",                   &
                                  "B%data_type = " // data_type_char )
               go to 99
            end if
         else if( a%data_type == MF_DT_CMPLX ) then
            out%data_type = MF_DT_CMPLX
            if( b%data_type == MF_DT_DBLE ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               out%cmplx = a%cmplx * b%double
            else if( b%data_type == MF_DT_CMPLX ) then

               ! if the mfArray is tempo and not protected,
               ! processing is done 'in place' (avoid an allocation).
               if( can_use_memory(A) ) then
                  out%cmplx => a%cmplx
                  call set_status_tempo_to_false( a )
               else if( can_use_memory(B) ) then
                  out%cmplx => b%cmplx
                  call set_status_tempo_to_false( b )
               else
                  allocate( out%cmplx(out%shape(1),out%shape(2)) )

               end if
               out%cmplx = a%cmplx * b%cmplx

            end if
         else
            write(data_type_char,"(I0)") A%data_type
            call PrintMessage( "operator(*)", "E",                      &
                               "internal error?",                      &
                               "A%data_type = " // data_type_char )
            go to 99
         end if

      else if( all(a%shape == 1) ) then ! a is scalar

         if( a%data_type == MF_DT_DBLE ) then
            call msAssign( out, mfMul_real8_mfArray( a%double(1,1), b ))
         else if( a%data_type == MF_DT_CMPLX ) then
            call msAssign( out, mfMul_cmplx8_mfArray( a%cmplx(1,1), b ))
         else
            write(data_type_char,"(I0)") A%data_type
            call PrintMessage( "operator(*)", "E",                      &
                               "case not handled",                      &
                               "a%data_type = " // data_type_char )
            go to 99
         end if

      else if( all(b%shape == 1) ) then ! b is scalar

         if( b%data_type == MF_DT_DBLE ) then
            call msAssign( out, mfMul_mfArray_real8( a, b%double(1,1) ))
         else if( b%data_type == MF_DT_CMPLX ) then
            call msAssign( out, mfMul_mfArray_cmplx8( a, b%cmplx(1,1) ))
         else
            write(data_type_char,"(I0)") B%data_type
            call PrintMessage( "operator(*)", "E",                      &
                               "case not handled",                      &
                               "b%data_type = " // data_type_char )
            go to 99
         end if

      else

         call PrintMessage( "operator(*)", "E",                         &
                            "a and b must have the same shape" )
         go to 99

      end if

      out%prop = op2_pattern_prop( A%prop, B%prop )
      if( A%prop%symm == TRUE .and. B%prop%symm == TRUE ) then
         out%prop%symm = TRUE
      else
         out%prop%symm = FALSE
      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_add( a%units(i), b%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "operator(*)", "E",                   &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               call PrintMessage( "operator(*)", "E",                   &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               go to 99
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a, b )
      call msAutoRelease(a,b)

      call mf_restore_fpe( )

#endif
   end function mfMul_mfArray_mfArray
!_______________________________________________________________________
!
   function mfMul_mfArray_int( a, b ) result( out )

      type(mfArray), intent(in) :: a
      integer, intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! converts the 2nd arg in double and calls 'mfMul_mfArray_real8'
      !
      ! not very efficient (2 calls instead of one, and some tests are
      ! made twice) : an information message is therefore emitted

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      ! Special case of MF_END: only addition and substraction are allowed
      ! when doing arithmetic about MF_END.
      ! Cannot use mfIsEqual(a,MF_END) in the test because a previous
      ! operation on MF_END has perhaps already occured (crc modified!).
      if( all(a%shape == MF_END%shape) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this operator cannot be used with MF_END!" )
         return
      end if

      call msInitArgs( a )

      call PrintMessage( "operator(*)", "I",                            &
                         "you are using an operator which involves an integer",&
                         "argument. Converting this argument in double",&
                         "precision will lead to a slightly more efficient",&
                         "computation!" )

      call msAssign( out, mfMul_mfArray_real8( a, dble(b) ) )

      out%status_temporary = .true.

      call msFreeArgs( a )
      call msAutoRelease( a )

      call mf_restore_fpe( )

#endif
   end function mfMul_mfArray_int
!_______________________________________________________________________
!
   function mfMul_int_mfArray( a, b ) result( out )

      integer, intent(in) :: a
      type(mfArray), intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! converts the 2nd arg in double and calls 'mfMul_mfArray_real8'
      !
      ! not very efficient (2 calls instead of one, and some tests are
      ! made twice) : an information message is therefore emitted

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      ! Special case of MF_END: only addition and substraction are allowed
      ! when doing arithmetic about MF_END.
      ! Cannot use mfIsEqual(b,MF_END) in the test because a previous
      ! operation on MF_END has perhaps already occured (crc modified!).
      if( all(b%shape == MF_END%shape) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this operator cannot be used with MF_END!" )
         return
      end if

      call msInitArgs( b )

      call PrintMessage( "operator(*)", "I",                            &
                         "you are using an operator which involves an integer",&
                         "argument. Converting this argument in double",&
                         "precision will lead to a slightly more efficient",&
                         "computation!" )

      call msAssign( out, mfMul_real8_mfArray( dble(a), b ) )

      out%status_temporary = .true.

      call msFreeArgs( b )
      call msAutoRelease( b )

      call mf_restore_fpe( )

#endif
   end function mfMul_int_mfArray
!_______________________________________________________________________
!
   function mfMul_mfArray_real4( a, b ) result( out )

      type(mfArray), intent(in) :: a
      real, intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! converts the 2nd arg in double and calls 'mfMul_mfArray_real8'
      !
      ! not very efficient (2 calls instead of one, and some tests are
      ! made twice) : an information message is therefore emitted

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( a )

      call PrintMessage( "operator(*)", "I",                            &
                         "you are using an operator which involves a single",&
                         "precision argument. Converting this argument in double",&
                         "precision will lead to a slightly more efficient",&
                         "computation! Moreover, be aware of precision loss..." )

      call msAssign( out, mfMul_mfArray_real8( a, dble(b) ) )

      out%status_temporary = .true.

      call msFreeArgs( a )
      call msAutoRelease( a )

      call mf_restore_fpe( )

#endif
   end function mfMul_mfArray_real4
!_______________________________________________________________________
!
   function mfMul_real4_mfArray( a, b ) result( out )

      real, intent(in) :: a
      type(mfArray), intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! converts the 2nd arg in double and calls 'mfMul_mfArray_real8'
      !
      ! not very efficient (2 calls instead of one, and some tests are
      ! made twice) : an information message is therefore emitted

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( b )

      call PrintMessage( "operator(*)", "I",                            &
                         "you are using an operator which involves a single",&
                         "precision argument. Converting this argument in double",&
                         "precision will lead to a slightly more efficient",&
                         "computation! Moreover, be aware of precision loss..." )

      call msAssign( out, mfMul_real8_mfArray( dble(a), b ) )

      out%status_temporary = .true.

      call msFreeArgs( b )
      call msAutoRelease( b )

      call mf_restore_fpe( )

#endif
   end function mfMul_real4_mfArray
!_______________________________________________________________________
!
   function mfMul_mfArray_real8( a, b ) result( out )

      type(mfArray),        intent(in) :: a
      real(kind=MF_DOUBLE), intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: ncol, nnz
      character(len=4) :: data_type_char

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "mfArray 'a' cannot be empty" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. a%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( a%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a permutation!" )
         go to 99
      end if

      out%data_type = a%data_type
      out%shape = a%shape
      if( a%data_type == MF_DT_DBLE ) then
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => a%double
            call set_status_tempo_to_false( a )
         else
            allocate( out%double(a%shape(1),a%shape(2)) )

         end if
         out%double = a%double * b
      else if( a%data_type == MF_DT_CMPLX ) then
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(a%shape(1),a%shape(2)) )

         end if
         out%cmplx = a%cmplx * b
      else if( a%data_type == MF_DT_SP_DBLE ) then
         ncol = a%shape(2)
         nnz = a%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%a => a%a
            out%i => a%i
            out%j => a%j
            call set_status_tempo_to_false( a )
         else
            allocate( out%a(size(a%a)) )

            allocate( out%i(size(a%i)) )

            allocate( out%j(size(a%j)) )

            out%i(1:nnz) = a%i(1:nnz)
            out%j = a%j
         end if
         out%a(1:nnz) = a%a(1:nnz) * b
      else if( a%data_type == MF_DT_SP_CMPLX ) then
         ncol = a%shape(2)
         nnz = a%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%z => a%z
            out%i => a%i
            out%j => a%j
            call set_status_tempo_to_false( a )
         else
            allocate( out%z(size(a%z)) )

            allocate( out%i(size(a%i)) )

            allocate( out%j(size(a%j)) )

            out%i(1:nnz) = a%i(1:nnz)
            out%j = a%j
         end if
         out%z(1:nnz) = a%z(1:nnz) * b
      else
         write(data_type_char,"(I0)") A%data_type
         call PrintMessage( "operator(*)", "E",                         &
                            "internal error?",                         &
                            "A%data_type = " // data_type_char )
         go to 99
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         out%units(:) = a%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )
      call msAutoRelease(a)

      call mf_restore_fpe( )

#endif
   end function mfMul_mfArray_real8
!_______________________________________________________________________
!
   function mfMul_real8_mfArray( a, b ) result( out )

      real(kind=MF_DOUBLE), intent(in) :: a
      type(mfArray),        intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: ncol, nnz
      character(len=4) :: data_type_char

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( b )

      if( mfIsEmpty(b) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "mfArray 'b' cannot be empty" )
         go to 99
      end if

      if( b%data_type == MF_DT_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( b%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a permutation!" )
         go to 99
      end if

      out%data_type = b%data_type
      out%shape = b%shape
      if( b%data_type == MF_DT_DBLE ) then
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(B) ) then
            out%double => b%double
            call set_status_tempo_to_false( b )
         else
            allocate( out%double(b%shape(1),b%shape(2)) )

         end if
         out%double = a * b%double
      else if( b%data_type == MF_DT_CMPLX ) then
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(B) ) then
            out%cmplx => b%cmplx
            call set_status_tempo_to_false( b )
         else
            allocate( out%cmplx(b%shape(1),b%shape(2)) )

         end if
         out%cmplx = a * b%cmplx
      else if( b%data_type == MF_DT_SP_DBLE ) then
         ncol = b%shape(2)
         nnz = b%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(B) ) then
            out%a => b%a
            out%i => b%i
            out%j => b%j
            call set_status_tempo_to_false( b )
         else
            allocate( out%a(size(b%a)) )

            allocate( out%i(size(b%i)) )

            allocate( out%j(size(b%j)) )

            out%i(1:nnz) = b%i(1:nnz)
            out%j = b%j
         end if
         out%a(1:nnz) = a * b%a(1:nnz)
      else if( b%data_type == MF_DT_SP_CMPLX ) then
         ncol = b%shape(2)
         nnz = b%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(B) ) then
            out%z => b%z
            out%i => b%i
            out%j => b%j
            call set_status_tempo_to_false( b )
         else
            allocate( out%z(size(b%z)) )

            allocate( out%i(size(b%i)) )

            allocate( out%j(size(b%j)) )

            out%i(1:nnz) = b%i(1:nnz)
            out%j = b%j
         end if
         out%z(1:nnz) = a * b%z(1:nnz)
      else
         write(data_type_char,"(I0)") B%data_type
         call PrintMessage( "operator(*)", "E",                         &
                            "internal error?",                         &
                            "B%data_type = " // data_type_char )
         go to 99
      end if

      out%prop%tril = B%prop%tril
      out%prop%triu = B%prop%triu
      out%prop%symm = B%prop%symm

      if( mf_phys_units ) then
         out%units(:) = b%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( b )
      call msAutoRelease(b)

      call mf_restore_fpe( )

#endif
   end function mfMul_real8_mfArray
!_______________________________________________________________________
!
   function mfMul_mfArray_cmplx4( a, b ) result( out )

      type(mfArray), intent(in) :: a
      complex, intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! converts the 2nd arg in double and calls 'mfMul_mfArray_cmplx8'
      !
      ! not very efficient (2 calls instead of one, and some tests are
      ! made twice) : an information message is therefore emitted

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( a )

      call PrintMessage( "operator(*)", "I",                            &
                         "you are using an operator which involves a single",&
                         "precision argument. Converting this argument in double",&
                         "precision will lead to a slightly more efficient",&
                         "computation! Moreover, be aware of precision loss..." )

      call msAssign( out, mfMul_mfArray_cmplx8( a, cmplx(b,kind=MF_DOUBLE) ) )

      out%status_temporary = .true.

      call msFreeArgs( a )
      call msAutoRelease( a )

      call mf_restore_fpe( )

#endif
   end function mfMul_mfArray_cmplx4
!_______________________________________________________________________
!
   function mfMul_cmplx4_mfArray( a, b ) result( out )

      complex, intent(in) :: a
      type(mfArray), intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! converts the 2nd arg in double and calls 'mfMul_mfArray_cmplx8'
      !
      ! not very efficient (2 calls instead of one, and some tests are
      ! made twice) : an information message is therefore emitted

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( b )

      call PrintMessage( "operator(*)", "I",                            &
                         "you are using an operator which involves a single",&
                         "precision argument. Converting this argument in double",&
                         "precision will lead to a slightly more efficient",&
                         "computation! Moreover, be aware of precision loss..." )

      call msAssign( out, mfMul_cmplx8_mfArray( cmplx(a,kind=MF_DOUBLE), b ) )

      out%status_temporary = .true.

      call msFreeArgs( b )
      call msAutoRelease( b )

      call mf_restore_fpe( )

#endif
   end function mfMul_cmplx4_mfArray
!_______________________________________________________________________
!
   function mfMul_mfArray_cmplx8( a, b ) result( out )

      type(mfArray),           intent(in) :: a
      complex(kind=MF_DOUBLE), intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "mfArray 'a' cannot be empty" )
         go to 99
      end if

      if( a%data_type == MF_DT_BOOL .or. a%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( a%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a permutation!" )
         go to 99
      end if

      out%shape = a%shape
      if( a%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(a%shape(1),a%shape(2)) )

         out%cmplx = a%double * b
      else if( a%data_type == MF_DT_CMPLX ) then
         out%data_type = MF_DT_CMPLX
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => a%cmplx
            call set_status_tempo_to_false( a )
         else
            allocate( out%cmplx(a%shape(1),a%shape(2)) )

         end if
         out%cmplx = a%cmplx * b
      else if( a%data_type == MF_DT_SP_DBLE ) then
         out%data_type = MF_DT_SP_CMPLX
         ncol = a%shape(2)
         nnz = a%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%i => a%i
            out%j => a%j
            call set_status_tempo_to_false( a )
            allocate( out%z(size(a%a)) )

            out%z(1:nnz) = a%a(1:nnz) * b
            call manual_dealloc_vec( a%a )
         else
            allocate( out%i(size(a%i)) )

            allocate( out%j(size(a%j)) )

            allocate( out%z(size(a%a)) )

            out%z(1:nnz) = a%a(1:nnz) * b
            out%i(1:nnz) = a%i(1:nnz)
            out%j = a%j
         end if
      else if( a%data_type == MF_DT_SP_CMPLX ) then
         out%data_type = MF_DT_SP_CMPLX
         ncol = a%shape(2)
         nnz = a%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%z => a%z
            out%i => a%i
            out%j => a%j
            call set_status_tempo_to_false( a )
         else
            allocate( out%z(size(a%z)) )

            allocate( out%i(size(a%i)) )

            allocate( out%j(size(a%j)) )

            out%i(1:nnz) = a%i(1:nnz)
            out%j = a%j
         end if
         out%z(1:nnz) = a%z(1:nnz) * b
      else
         call PrintMessage( "operator(*)", "E",                         &
                            "unknown data type for 'b'" )
         go to 99
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = UNKNOWN

      if( mf_phys_units ) then
         out%units(:) = a%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )
      call msAutoRelease(a)

      call mf_restore_fpe( )

#endif
   end function mfMul_mfArray_cmplx8
!_______________________________________________________________________
!
   function mfMul_cmplx8_mfArray( a, b ) result( out )

      complex(kind=MF_DOUBLE), intent(in) :: a
      type(mfArray),           intent(in) :: b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( b )

      if( mfIsEmpty(b) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "mfArray 'b' cannot be empty" )
         go to 99
      end if

      if( b%data_type == MF_DT_BOOL .or. b%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( b%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a permutation!" )
         go to 99
      end if

      out%shape = b%shape
      if( b%data_type == MF_DT_DBLE ) then
         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx(b%shape(1),b%shape(2)) )

         out%cmplx = a * b%double
      else if( b%data_type == MF_DT_CMPLX ) then

         out%data_type = MF_DT_CMPLX
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(B) ) then
            out%cmplx => b%cmplx
            call set_status_tempo_to_false( b )
         else
            allocate( out%cmplx(b%shape(1),b%shape(2)) )

         end if
         out%cmplx = a * b%cmplx
      else if( b%data_type == MF_DT_SP_DBLE ) then
         out%data_type = MF_DT_SP_CMPLX
         ncol = b%shape(2)
         nnz = b%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(B) ) then
            out%i => b%i
            out%j => b%j
            call set_status_tempo_to_false( b )
            allocate( out%z(size(b%a)) )

            out%z(1:nnz) = a * b%a(1:nnz)
            call manual_dealloc_vec( b%a )
         else
            allocate( out%i(size(b%i)) )

            allocate( out%j(size(b%j)) )

            allocate( out%z(size(b%a)) )

            out%z(1:nnz) = a * b%a(1:nnz)
            out%i(1:nnz) = b%i(1:nnz)
            out%j = b%j
         end if
      else if( b%data_type == MF_DT_SP_CMPLX ) then
         out%data_type = MF_DT_SP_CMPLX
         ncol = b%shape(2)
         nnz = b%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(B) ) then
            out%z => b%z
            out%i => b%i
            out%j => b%j
            call set_status_tempo_to_false( b )
         else
            allocate( out%z(size(b%z)) )

            allocate( out%i(size(b%i)) )

            allocate( out%j(size(b%j)) )

            out%i(1:nnz) = b%i(1:nnz)
            out%j = b%j
         end if
         out%z(1:nnz) = a * b%z(1:nnz)
      else
         call PrintMessage( "operator(*)", "E",                         &
                            "unknown data type for 'b'" )
         go to 99
      end if

      out%prop%tril = B%prop%tril
      out%prop%triu = B%prop%triu
      out%prop%symm = UNKNOWN

      if( mf_phys_units ) then
         out%units(:) = b%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( b )
      call msAutoRelease(b)

      call mf_restore_fpe( )

#endif
   end function mfMul_cmplx8_mfArray
!_______________________________________________________________________
!
   function mfMul_mfArray_mfUnit( A, u ) result( out )

      type(mfArray), intent(in) :: A
      type(mfUnit),  intent(in) :: u
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, status, ncol, nnz
      character(len=4) :: data_type_char

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "mfArray 'A' cannot be empty" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a permutation!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double = A%double * u%value
      else if( A%data_type == MF_DT_CMPLX ) then
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         out%cmplx = A%cmplx * u%value
      else if( A%data_type == MF_DT_SP_DBLE ) then
         ncol = A%shape(2)
         nnz = A%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%a => A%a
            out%i => A%i
            out%j => A%j
            call set_status_tempo_to_false( A )
         else
            allocate( out%a(size(A%a)) )

            allocate( out%i(size(A%i)) )

            allocate( out%j(size(A%j)) )

            out%i(1:nnz) = A%i(1:nnz)
            out%j = A%j
         end if
         out%a(1:nnz) = A%a(1:nnz) * u%value
      else if( A%data_type == MF_DT_SP_CMPLX ) then
         ncol = A%shape(2)
         nnz = A%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%z => A%z
            out%i => A%i
            out%j => A%j
            call set_status_tempo_to_false( A )
         else
            allocate( out%z(size(A%z)) )

            allocate( out%i(size(A%i)) )

            allocate( out%j(size(A%j)) )

            out%i(1:nnz) = A%i(1:nnz)
            out%j = A%j
         end if
         out%z(1:nnz) = A%z(1:nnz) * u%value
      else
         write(data_type_char,"(I0)") A%data_type
         call PrintMessage( "operator(*)", "E",                         &
                            "internal error while processing phys. units?",&
                            "A%data_type = " // data_type_char )
         go to 99
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_add( A%units(i), u%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "operator(*)", "E",                   &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               call PrintMessage( "operator(*)", "E",                   &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               go to 99
            end if
         end do
      else
         call PrintMessage( "operator(*)", "W",                         &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease(A)

      call mf_restore_fpe( )

#endif
   end function mfMul_mfArray_mfUnit
!_______________________________________________________________________
!
   function mfMul_mfUnit_mfArray( u, A ) result( out )

      type(mfUnit),  intent(in) :: u
      type(mfArray), intent(in) :: A
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, status, ncol, nnz
      character(len=4) :: data_type_char

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "mfArray 'A' cannot be empty" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL .or. A%data_type == MF_DT_SP_BOOL ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a logical!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "operator(*)", "E",                         &
                            "this function cannot be applied to a permutation!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      if( A%data_type == MF_DT_DBLE ) then
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%double => A%double
            call set_status_tempo_to_false( A )
         else
            allocate( out%double(A%shape(1),A%shape(2)) )

         end if
         out%double = A%double * u%value
      else if( A%data_type == MF_DT_CMPLX ) then
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%cmplx => A%cmplx
            call set_status_tempo_to_false( A )
         else
            allocate( out%cmplx(A%shape(1),A%shape(2)) )

         end if
         out%cmplx = A%cmplx * u%value
      else if( A%data_type == MF_DT_SP_DBLE ) then
         ncol = A%shape(2)
         nnz = A%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%a => A%a
            out%i => A%i
            out%j => A%j
            call set_status_tempo_to_false( A )
         else
            allocate( out%a(size(A%a)) )

            allocate( out%i(size(A%i)) )

            allocate( out%j(size(A%j)) )

            out%i(1:nnz) = A%i(1:nnz)
            out%j = A%j
         end if
         out%a(1:nnz) = A%a(1:nnz) * u%value
      else if( A%data_type == MF_DT_SP_CMPLX ) then
         ncol = A%shape(2)
         nnz = A%j(ncol+1)-1
         ! if the mfArray is tempo and not protected,
         ! processing is done 'in place' (avoid an allocation).
         if( can_use_memory(A) ) then
            out%z => A%z
            out%i => A%i
            out%j => A%j
            call set_status_tempo_to_false( A )
         else
            allocate( out%z(size(A%z)) )

            allocate( out%i(size(A%i)) )

            allocate( out%j(size(A%j)) )

            out%i(1:nnz) = A%i(1:nnz)
            out%j = A%j
         end if
         out%z(1:nnz) = A%z(1:nnz) * u%value
      else
         write(data_type_char,"(I0)") A%data_type
         call PrintMessage( "operator(*)", "E",                         &
                            "internal error while processing phys. units?",&
                            "A%data_type = " // data_type_char )
         go to 99
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%prop%symm = A%prop%symm

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_add( A%units(i), u%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "operator(*)", "E",                   &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               call PrintMessage( "operator(*)", "E",                   &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               go to 99
            end if
         end do
      else
         call PrintMessage( "operator(*)", "W",                         &
                            "this operation involves a type(mfUnit) object:",&
                            "you should activate physical units via the", &
                            "'msUsePhysUnits' routine!" )
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease(A)

      call mf_restore_fpe( )

#endif
   end function mfMul_mfUnit_mfArray
