! f90 include file

! auxiliairy routines used by :
!          Eigs.inc
!          SVDS.inc
!
! (cf. also headers of the previous files in order to know which
!  routines are really called)
!
! liste :
!
!     mf_mat_vec()
!     mf_mat_vec_inv()
!     mf_mat_vec_inv_lu()
!     mf_mat_vec_inv_chol()
!     mf_mat_vec_inv_spchol()
!     mf_mat_vec_inv_factor()
!
!     mf_mat_vec_cmplx()
!     mf_mat_vec_inv_cmplx()
!     mf_mat_vec_inv_chol_cmplx()
!     mf_mat_vec_inv_spchol_cmplx()
!     mf_mat_vec_inv_factor_cmplx()
!
!     mf_mat_vec_2steps()
!     mf_mat_vec_2steps_inv()
!     mf_mat_vec_2steps_inv_factor()
!
!     mf_mat_vec_2steps_cmplx()
!     mf_mat_vec_2steps_inv_cmplx()
!     mf_mat_vec_2steps_inv_factor_cmplx()

!_______________________________________________________________________
!
   subroutine mf_mat_vec( A, x, y )

      type(mfArray) :: A
      real(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !------------------------------------------
      ! matrix-vector multiplication: y = A x
      !------------------------------------------
      ! calcul de y = A*x
      !
      ! real case

      integer :: ncol

   !------ end of declarations -- execution starts hereafter  ------

      if( mfIsSparse(A) ) then

         ncol = A%shape(2)
         call amux( ncol, x, y, A%a, A%i, A%j )

      else

         y(:) = matmul( A%double(:,:), x(:) )

      end if

#endif
   end subroutine mf_mat_vec
!_______________________________________________________________________
!
   subroutine mf_mat_vec_inv( A, x, y )

      type(mfArray) :: A
      real(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !--------------------------------
      ! gets solution of: A y = x
      !--------------------------------
      ! computes y = A\x
      !
      ! real case

      type(mfArray) :: mfx, mfy

   !------ end of declarations -- execution starts hereafter  ------

!### TODO ?:
print *, '(MUESLI_DEBUG mf_mat_vec_inv:) recopie !'
call msPause()
      call msEquiv(x,mfx)
      call msEquiv(y,mfy)

      call msAssign( mfy, mfLdiv( A, mfx ) )

      y(:) = mfy%double(:,1)

      call msSilentRelease( mfx, mfy )

#endif
   end subroutine mf_mat_vec_inv
!_______________________________________________________________________
!
   subroutine mf_mat_vec_inv_lu( L, U, x, y )

      type(mfArray) :: L, U
      real(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !--------------------------------
      ! gets solution of: A y = x
      !--------------------------------
      ! computes y = U\(L\x)
      !
      ! real case - dense

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(x)
      y(:) = x(:)
      call dtrsm( "left", "lower", "no transpose", "unit", n, 1,        &
                  1.0d0, L%double(1,1), n, y(1), n )
      call dtrsm( "left", "upper", "no transpose", "non-unit", n, 1,    &
                  1.0d0, U%double(1,1), n, y(1), n )

#endif
   end subroutine mf_mat_vec_inv_lu
!_______________________________________________________________________
!
   subroutine mf_mat_vec_inv_chol( U, x, y )

      type(mfArray) :: U
      real(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !--------------------------------
      ! gets solution of: A y = x
      !--------------------------------
      ! computes y = U\(U'\x)
      !
      ! real case - dense

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(x)
      y(:) = x(:)
      call dtrsm( "left", "upper", "transpose", "non-unit", n, 1,       &
                  1.0d0, U%double(1,1), n, y(1), n )
      call dtrsm( "left", "upper", "no transpose", "non-unit", n, 1,    &
                  1.0d0, U%double(1,1), n, y(1), n )

#endif
   end subroutine mf_mat_vec_inv_chol
!_______________________________________________________________________
!
   subroutine mf_mat_vec_inv_spchol( factor, x, y )

      type(mfMatFactor) :: factor
      real(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !--------------------------------
      ! gets solution of: A y = x
      !--------------------------------
      ! solves A.y = x
      !        (A factorised by sparse Cholesky -> factor)
      !
      ! real case - sparse

      integer(kind=MF_ADDRESS) :: c_addr, L_addr
      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(x)
      if( factor%data_type == MF_DT_SP_DBLE ) then
         c_addr = factor%ptr_1
         L_addr = factor%ptr_2
         call cholmod_solve_factor( n, c_addr, L_addr, x(:), y(:) )
      else
         call msPause("mf_mat_vec_inv_spchol: bad data type!")
         stop
      end if

#endif
   end subroutine mf_mat_vec_inv_spchol
!_______________________________________________________________________
!
   subroutine mf_mat_vec_inv_factor( factor, x, y )

      type(mfMatFactor) :: factor
      real(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !--------------------------------
      ! gets solution of: A y = x
      !--------------------------------
      ! dense: A = P L U
      ! computes y = U\[L\(P' x)]
      !
      ! real case

      type(mfArray), pointer :: L, U, P
      integer :: n, sys
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      integer(kind=MF_ADDRESS) :: numeric ! LU handle
      character(len=4) :: info_char

   !------ end of declarations -- execution starts hereafter  ------

      if( factor%data_type == MF_DT_DBLE ) then
         L => factor%mf_ptr_1
         U => factor%mf_ptr_2
         P => factor%mf_ptr_3
         n = size(x)
         y(:) = matmul( transpose(P%double(:,:)), x(:) )
         call dtrsm( "left", "lower", "no transpose", "unit", n, 1,     &
                     1.0d0, L%double(1,1), n, y(1), n )
         call dtrsm( "left", "upper", "no transpose", "non-unit", n, 1, &
                     1.0d0, U%double(1,1), n, y(1), n )
      else if( factor%data_type == MF_DT_SP_DBLE ) then
         ! set UMFPACK-4 default parameters
         call umf4def_d(control)
         ! print control parameters.
         ! Set control(1) to 1 to print error messages only
         control(1) = 1

         ! solve A.x=b, without iterative refinement
         sys = 0
         numeric = factor%ptr_1
         call umf4sol_d( sys, y, x, numeric, control, infos )
         if( infos(1) < 0 ) then
            write(info_char,"(I0)") nint(infos(1))
            call PrintMessage( "mfLDiv", "E",                           &
                               "in umf4sol_d: infos(1) = " // info_char )
         end if
      else
         call msPause("mf_mat_vec_inv_factor: bad data type!")
         stop
      end if

#endif
   end subroutine mf_mat_vec_inv_factor
!_______________________________________________________________________
!
   subroutine mf_mat_vec_cmplx( A, x, y )

      type(mfArray) :: A
      complex(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !------------------------------------------
      ! matrix-vector multiplication: y = A x
      !------------------------------------------
      ! computes y = A*x
      !
      ! complexe case

      integer :: ncol

   !------ end of declarations -- execution starts hereafter  ------

      if( mfIsSparse(A) ) then

         ncol = A%shape(2)
         call amux_cmplx_cmplx( ncol, x, y, A%z, A%i, A%j )

      else

         y(:) = matmul( A%cmplx(:,:), x(:) )

      end if

#endif
   end subroutine mf_mat_vec_cmplx
!_______________________________________________________________________
!
   subroutine mf_mat_vec_inv_cmplx( A, x, y )

      type(mfArray) :: A
      complex(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !--------------------------------
      ! gets solution of: A y = x
      !--------------------------------
      ! computes y = A\x
      !
      ! complexe case

      type(mfArray) :: mfx, mfy

   !------ end of declarations -- execution starts hereafter  ------

!### TODO ?:
print *, '(MUESLI_DEBUG mf_mat_vec_inv_cmplx:) recopie !'
call msPause()
      call msEquiv(x,mfx)
      call msEquiv(y,mfy)

      call msAssign( mfy, mfLdiv( A, mfx ) )

      y(:) = mfy%cmplx(:,1)

      call msSilentRelease( mfx, mfy )

#endif
   end subroutine mf_mat_vec_inv_cmplx
!_______________________________________________________________________
!
   subroutine mf_mat_vec_inv_chol_cmplx( U, x, y )

      type(mfArray) :: U
      complex(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !--------------------------------
      ! gets solution of: A y = x
      !--------------------------------
      ! computes y = U\(U'\x)
      !
      ! complexe case - dense

      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(x)
      y(:) = x(:)
      call ztrsm( "left", "upper", "conjg transpose", "non-unit", n, 1, &
                  (1.0d0,0.0d0), U%cmplx(1,1), n, y(1), n )
      call ztrsm( "left", "upper", "no transpose", "non-unit", n, 1,    &
                  (1.0d0,0.0d0), U%cmplx(1,1), n, y(1), n )

#endif
   end subroutine mf_mat_vec_inv_chol_cmplx
!_______________________________________________________________________
!
   subroutine mf_mat_vec_inv_spchol_cmplx( factor, x, y )

      type(mfMatFactor) :: factor
      complex(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !--------------------------------
      ! gets solution of: A y = x
      !--------------------------------
      ! solves A.y = x
      !        (A factorised by sparse Cholesky -> factor)
      !
      ! complexe case - sparse

      integer(kind=MF_ADDRESS) :: c_addr, L_addr
      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(x)
      if( factor%data_type == MF_DT_SP_CMPLX ) then
         c_addr = factor%ptr_1
         L_addr = factor%ptr_2
         call cholmod_cmplx_solve_factor( n, c_addr, L_addr, x(:), y(:) )
      else
         call msPause("mf_mat_vec_inv_spchol_cmplx: bad data type!")
         stop
      end if

#endif
   end subroutine mf_mat_vec_inv_spchol_cmplx
!_______________________________________________________________________
!
   subroutine mf_mat_vec_inv_factor_cmplx( factor, x, y )

      type(mfMatFactor) :: factor
      complex(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !--------------------------------
      ! gets solution of: A y = x
      !--------------------------------
      ! dense: A = P L U
      ! computes y = U\[L\(P' x)]
      !
      ! complexe case

      type(mfArray), pointer :: L, U, P
      integer :: n, sys
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      integer(kind=MF_ADDRESS) :: numeric ! LU handle
      character(len=4) :: info_char
      real(kind=MF_DOUBLE), allocatable :: ptr_b(:), ptr_x(:)
      real(kind=MF_DOUBLE), allocatable :: ptr_bz(:), ptr_xz(:)

   !------ end of declarations -- execution starts hereafter  ------

      n = size(x)
      if( factor%data_type == MF_DT_CMPLX ) then
         L => factor%mf_ptr_1
         U => factor%mf_ptr_2
         P => factor%mf_ptr_3
         y(:) = matmul( transpose(P%double(:,:)), x(:) )
         call ztrsm( "left", "lower", "no transpose", "unit", n, 1,     &
                     (1.0d0,0.0d0), L%cmplx(1,1), n, y(1), n )
         call ztrsm( "left", "upper", "no transpose", "non-unit", n, 1, &
                     (1.0d0,0.0d0), U%cmplx(1,1), n, y(1), n )
      else if( factor%data_type == MF_DT_SP_CMPLX ) then
         ! set UMFPACK-4 default parameters
         call umf4def_z(control)
         ! print control parameters.
         ! Set control(1) to 1 to print error messages only
         control(1) = 1

         ! solve A.x=b, without iterative refinement
         sys = 0
         numeric = factor%ptr_1

         allocate( ptr_x(n), ptr_xz(n) )
         allocate( ptr_b(n), ptr_bz(n) )
         ptr_b(:) = real(x(:))
         ptr_bz(:) = aimag(x(:))
         call umf4sol_z( sys, ptr_x(1), ptr_xz(1), ptr_b(1), ptr_bz(1),          &
                         numeric, control, infos)
         if( infos(1) < 0 ) then
            write(info_char,"(I0)") nint(infos(1))
            call PrintMessage( "mfLDiv", "E",                           &
                               "in umf4sol_z: infos(1) = " // info_char )
         end if

         y(:) = cmplx(ptr_x(:),ptr_xz(:),kind=MF_DOUBLE)
      else
         call msPause("mf_mat_vec_inv_factor_cmplx: bad data type!")
         stop
      end if

#endif
   end subroutine mf_mat_vec_inv_factor_cmplx
!_______________________________________________________________________
!
!***********************************************************************
!_______________________________________________________________________
!
   subroutine mf_mat_vec_2steps( A, x, y )

      type(mfArray) :: A
      real(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !---------------------------------------------
      ! matrix-vector multiplication: y = A' A x
      !---------------------------------------------
      ! computes w = A*x,
      !     then y = A'*w
      !
      ! real case (valid for A non-square)

      integer :: nrow, ncol
      real(kind=MF_DOUBLE), allocatable :: tmp(:)
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      nrow = A%shape(1)
      ncol = A%shape(2)
      allocate( tmp(nrow) )
      if( mfIsSparse(A) ) then

         ! more efficient because we don't use mfArrays (no copies)
         call amux( ncol, x, tmp, A%a, A%i, A%j )
         call xmua( nrow, ncol, tmp, y, A%a, A%i, A%j )

      else

         tmp(:) = matmul( A%double(:,:), x(:) )
         do j = 1, ncol
            y(j) = 0.0d0
            do i = 1, nrow
               y(j) = y(j) + A%double(i,j)*tmp(i)
            end do
         end do

      end if

#endif
   end subroutine mf_mat_vec_2steps
!_______________________________________________________________________
!
   subroutine mf_mat_vec_2steps_inv( A, x, y )

      type(mfArray) :: A
      real(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !-------------------------------------
      ! gets solution of: A' A y = x
      !-------------------------------------
      ! computes w = A'\x,
      !     then y = A\w
      !
      ! real case (valid for A non-square)

      type(mfArray) :: mfx, mfy, mftmp

   !------ end of declarations -- execution starts hereafter  ------

!### TODO ?:
print *, '(MUESLI_DEBUG mf_mat_vec_2steps_inv:) recopie !'
call msPause()
      call msEquiv(x,mfx)
      call msEquiv(y,mfy)

      call msAssign( mftmp, .t.mfRDiv( .t.mfx, A ) )
      call msAssign( mfy, mfLDiv( A, mftmp ) )

      y(:) = mfy%double(:,1)

      call msSilentRelease( mfx, mfy, mftmp )

#endif
   end subroutine mf_mat_vec_2steps_inv
!_______________________________________________________________________
!
   subroutine mf_mat_vec_2steps_inv_factor( factor, x, y )

      type(mfMatFactor) :: factor
      real(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !-------------------------------------
      ! gets solution of: A' A y = x
      !-------------------------------------
      ! dense: A = P L U
      !        A' = U' L' P' ; U' L' P' w = x
      !
      ! computes w = L'\(U'\x),
      !     then y = U\(L\w)                 no P
      !
      ! real case, dense, square matrix A(n,n) only

      type(mfArray), pointer :: L, U
      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      if( factor%data_type == MF_DT_DBLE ) then
         L => factor%mf_ptr_1
         if( Size(L,1) /= Size(L,2) ) then
            call msPause("mf_mat_vec_2steps_inv_factor: internal error [A non square]")
            stop
         end if
         U => factor%mf_ptr_2
         n = size(x)
         y(:) = x(:)
         call dtrsm( "left", "upper", "transpose", "non-unit", n, 1,    &
                     1.0d0, U%double(1,1), n, y(1), n )
         call dtrsm( "left", "lower", "transpose", "unit", n, 1,        &
                     1.0d0, L%double(1,1), n, y(1), n )
         call dtrsm( "left", "lower", "no transpose", "unit", n, 1,     &
                     1.0d0, L%double(1,1), n, y(1), n )
         call dtrsm( "left", "upper", "no transpose", "non-unit", n, 1, &
                     1.0d0, U%double(1,1), n, y(1), n )
      else
         call msPause("mf_mat_vec_2steps_inv_factor: bad data type!")
         stop
      end if

#endif
   end subroutine mf_mat_vec_2steps_inv_factor
!_______________________________________________________________________
!
   subroutine mf_mat_vec_2steps_cmplx( A, x, y )

      type(mfArray) :: A
      complex(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !---------------------------------------------
      ! matrix-vector multiplication: y = A' A x
      !---------------------------------------------
      ! computes w = A*x,
      !     then y = A'*w
      !
      ! complexe case (valid for A non-square)

      integer :: nrow, ncol
      complex(kind=MF_DOUBLE), allocatable :: tmp(:)
      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      nrow = A%shape(1)
      ncol = A%shape(2)
      allocate( tmp(nrow) )
      if( mfIsSparse(A) ) then

         ! more efficient because we don't use mfArrays (no copies)
         call amux_cmplx_cmplx( ncol, x, tmp, A%z, A%i, A%j )
         tmp(:) = conjg( tmp(:) )
         call xmua_cmplx_cmplx( nrow, ncol, tmp, y, A%z, A%i, A%j )
         y(:) = conjg( y(:) )

      else

         tmp(:) = matmul( A%cmplx(:,:), x(:) )
         do j = 1, ncol
            y(j) = (0.0d0,0.0d0)
            do i = 1, nrow
               y(j) = y(j) + conjg(A%cmplx(i,j))*tmp(i)
            end do
         end do

      end if

#endif
   end subroutine mf_mat_vec_2steps_cmplx
!_______________________________________________________________________
!
   subroutine mf_mat_vec_2steps_inv_cmplx( A, x, y )

      type(mfArray) :: A
      complex(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !-------------------------------------
      ! gets solution of: A' A y = x
      !-------------------------------------
      ! computes w = A'\x,
      !     then y = A\w
      !
      ! complexe case (valid for A non-square)

      type(mfArray) :: mfx, mfy, mftmp

   !------ end of declarations -- execution starts hereafter  ------

!### TODO ?:
print *, '(MUESLI_DEBUG mf_mat_vec_2steps_inv_cmplx:) recopie !'
call msPause()
      call msEquiv(x,mfx)
      call msEquiv(y,mfy)

      call msAssign( mftmp, .h.mfRDiv( .h.mfx, A ) )
      call msAssign( mfy, mfLDiv( A, mftmp ) )

      y(:) = mfy%cmplx(:,1)

      call msSilentRelease( mfx, mfy, mftmp )

#endif
   end subroutine mf_mat_vec_2steps_inv_cmplx
!_______________________________________________________________________
!
   subroutine mf_mat_vec_2steps_inv_factor_cmplx( factor, x, y )

      type(mfMatFactor) :: factor
      complex(kind=MF_DOUBLE) :: x(:), y(:)
      !------ API end ------
#ifdef _DEVLP

      !-------------------------------------
      ! gets solution of: A' A y = x
      !-------------------------------------
      ! dense: A = P L U
      !        A' = U' L' P' ; U' L' P' w = x
      !
      ! computes w = L'\(U'\x),
      !     then y = U\(L\w)                 no P
      !
      ! complexe case, dense

      type(mfArray), pointer :: L, U
      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      if( factor%data_type == MF_DT_CMPLX ) then
         L => factor%mf_ptr_1
         if( Size(L,1) /= Size(L,2) ) then
            call msPause("mf_mat_vec_2steps_inv_factor_cmplx: internal error [A non square]")
            stop
         end if
         U => factor%mf_ptr_2
         n = size(x)
         y(:) = x(:)
         call ztrsm( "left", "upper", "conjg transpose", "non-unit", n, 1, &
                     (1.0d0,0.0d0), U%cmplx(1,1), n, y(1), n )
         call ztrsm( "left", "lower", "conjg transpose", "unit", n, 1,  &
                     (1.0d0,0.0d0), L%cmplx(1,1), n, y(1), n )
         call ztrsm( "left", "lower", "no transpose", "unit", n, 1,     &
                     (1.0d0,0.0d0), L%cmplx(1,1), n, y(1), n )
         call ztrsm( "left", "upper", "no transpose", "non-unit", n, 1, &
                     (1.0d0,0.0d0), U%cmplx(1,1), n, y(1), n )
      else
         call msPause("mf_mat_vec_2steps_inv_factor_cmplx: bad data type!")
         stop
      end if

#endif
   end subroutine mf_mat_vec_2steps_inv_factor_cmplx
