! f90 include file

!_______________________________________________________________________
!
   subroutine msSchur( out, A, form )

      type(mfArray), target :: A
      character(len=*), intent(in), optional :: form
      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! for nonsymmetric matrices ('msEig' is called if A is
      ! symmetric/hermitian)
      !
      ! form = "real" [default] or "complex"

      type(mfArray), pointer :: U, T
      integer :: n, lda, info, lwork
      real(kind=MF_DOUBLE), allocatable :: work(:)
      real(kind=MF_DOUBLE), allocatable :: wr(:), wi(:)
      character(len=1) :: jobvs, sort, form0
      logical :: select, bwork(1)
      integer :: sdim
      integer :: ldvs
      real(kind=MF_DOUBLE), allocatable :: vs(:,:)
      complex(kind=MF_DOUBLE), allocatable :: cvs(:,:), w(:), cwork(:)
      character(len=3) :: info_char

      type(mfArray), pointer :: A_copy
      logical :: A_copy_is_allocated

      integer, external :: ilaenv

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! 2 out-args must be specified
      if( out%n /= 2 ) then
         call PrintMessage( "msSchur", "E",                             &
                            "two output args required!",               &
                            "syntax is : call msSchur ( mfOut(U,T), A )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msSchur", "E",                             &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      U => out%ptr1
      T => out%ptr2
      call msSilentRelease( U, T )

      if( present(form) ) then
         if( to_upper(form) == "REAL" ) then
            form0 = "R"
         else if( to_upper(form) == "COMPLEX" ) then
            form0 = "C"
         else
            call PrintMessage( "mfSchur", "E",                          &
                               "optional arg 'form' must be equal to 'real' or 'complex'!" )
            go to 99
         end if
      else
         form0 = "R"
      end if

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msSchur", "E",                             &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "msSchur", "E",                             &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msSchur", "E",                             &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msSchur", "E",                             &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "msSchur", "E",                             &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      n = A%shape(1)
      ! square matrix ?
      if( A%shape(2) /= n ) then
         call PrintMessage( "msSchur", "E",                             &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      if( mfIsTriu(A) ) then
         call PrintMessage( "msSchur", "I",                             &
                            "'A' is already upper triangular!" )
         if( A%level_protected == 1 ) then
            call msAssign( T, A )
         else
            T = A
         end if
         call msAssign( U, mfEye(n,n) )
         go to 99
      end if

      if( mfIsSymm(A) ) then
         ! A is real, symmetric OR complex, hermitian
         call PrintMessage( "msSchur", "I",                             &
                            "'A' is symmetric or hermitian...",         &
                            "-> calling 'msEig'" )
         ! -> calling 'msEig'
         call msEig( mfOut(U,T), A )
         go to 99
      end if

      U%shape = [ n, n ]
      T%shape = [ n, n ]

      ! if necessary, making copies because LAPACK overwrites A
      if( mfIsReal(A) .and. form0 == "C" ) then
         allocate( A_copy ) ! no_mem_trace !
         call msAssign( A_copy, mfComplex(A) )
         A_copy_is_allocated = .true.
      else
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if
      end if

      if( A_copy%data_type == MF_DT_DBLE ) then

         allocate( U%double( U%shape(1), U%shape(2) ) )

         allocate( T%double( T%shape(1), T%shape(2) ) )

         lda = n
         jobvs = 'V' ! Schur vectors are computed
         ldvs = n
         sort = 'N' ! Eigenvalues are not sorted

         ! get optimal work-array size
         allocate( work(1) )

         allocate( wr(n), wi(n) )

         allocate( vs(ldvs,n) )

         call dgees( jobvs, sort, select, n, A_copy%double(1,1), lda, sdim, &
                     wr(1), wi(1), vs(1,1), ldvs, work(1), -1, bwork, info )
         lwork = int( work(1) )
         if( info /= 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "msSchur", "W",                          &
                               "in determining the optimal work-array size by LAPACK:", &
                               "info = " // info_char )
         end if
         if( lwork < 3*n ) then
            call PrintMessage( "msSchur", "I",                          &
                               "setting work-array size to 3*n" )
            lwork = 3*n
         end if
         deallocate( work )

         allocate( work(lwork) )

         ! General system -- real general matrix : DGEES
         call dgees( jobvs, sort, select, n, A_copy%double(1,1), lda, sdim, &
                     wr(1), wi(1), vs(1,1), ldvs, work(1), lwork, bwork, info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "msSchur", "W",                          &
                               "in computing Schur form of matrix A by LAPACK:", &
                               "info = " // info_char,                  &
                               "(as a side effect, U and T will be empty)" )
            call msSilentRelease( U, T )
         else
            T%data_type = MF_DT_DBLE
            T%double(:,:) = A_copy%double(:,:)
            U%data_type = MF_DT_DBLE
            U%double(:,:) = vs(:,:)
         end if

      else if( A_copy%data_type == MF_DT_CMPLX ) then

         allocate( U%cmplx( U%shape(1), U%shape(2) ) )

         allocate( T%cmplx( T%shape(1), T%shape(2) ) )

         lda = n
         jobvs = 'V' ! Schur vectors are not computed
         ldvs = n
         sort = 'N' ! Eigenvalues are not sorted

         ! get optimal work-array size
         allocate( cwork(1) )

         allocate( work(n) )

         allocate( w(n) )

         allocate( cvs(ldvs,n) )

         call zgees( jobvs, sort, select, n, A_copy%cmplx(1,1), lda, sdim, &
                     w(1), cvs(1,1), ldvs, cwork(1), -1, work(1), bwork, info )
         lwork = int( cwork(1) )
         if( info /= 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "msSchur", "W",                          &
                               "in determining the optimal work-array size by LAPACK:", &
                               "info = " // info_char )
         end if
         if( lwork < 2*n ) then
            call PrintMessage( "msSchur", "I",                          &
                               "setting work-array size to 3*n" )
            lwork = 2*n
         end if
         deallocate( cwork )

         allocate( cwork(lwork) )

         ! General system -- complex general matrix : ZGEES
         call zgees( jobvs, sort, select, n, A_copy%cmplx(1,1), lda, sdim, &
                     w(1), cvs(1,1), ldvs, cwork(1), lwork, work(1), bwork, info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "msSchur", "W",                          &
                               "in computing Schur form of matrix A by LAPACK:", &
                               "info = " // info_char,                  &
                               "(as a side effect, U and T will be empty)" )
            call msSilentRelease( U, T )
         else
            T%data_type = MF_DT_CMPLX
            T%cmplx(:,:) = A_copy%cmplx(:,:)
            T%prop%triu = TRUE
            U%data_type = MF_DT_CMPLX
            U%cmplx(:,:) = cvs(:,:)
         end if

      end if

      if( A_copy_is_allocated ) then
         call msSilentRelease( A_copy )
         deallocate( A_copy ) ! no_mem_trace !
      end if

      if( mf_phys_units ) then
         T%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end subroutine msSchur
!_______________________________________________________________________
!
   function mfSchur( A, form ) result( out )

      type(mfArray), target :: A
      character(len=*), intent(in), optional :: form
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! for nonsymmetric matrices ('mfEig' is called if A is
      ! symmetric/hermitian)
      !
      ! form = "real" [default] or "complex"

      integer :: n, lda, info, lwork
      real(kind=MF_DOUBLE), allocatable :: work(:)
      real(kind=MF_DOUBLE), allocatable :: wr(:), wi(:)
      character(len=1) :: jobvs, sort, form0
      logical :: select, bwork(1)
      integer :: sdim
      integer, parameter :: ldvs = 1
      real(kind=MF_DOUBLE), allocatable :: vs(:,:)
      complex(kind=MF_DOUBLE), allocatable :: cvs(:,:), w(:), cwork(:)
      character(len=3) :: info_char

      type(mfArray), pointer :: A_copy
      logical :: A_copy_is_allocated

      integer, external :: ilaenv

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( present(form) ) then
         if( to_upper(form) == "REAL" ) then
            form0 = "R"
         else if( to_upper(form) == "COMPLEX" ) then
            form0 = "C"
         else
            call PrintMessage( "mfSchur", "E",                          &
                               "optional arg 'form' must be equal to 'real' or 'complex'!" )
            go to 99
         end if
      else
         form0 = "R"
      end if

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfSchur", "E",                             &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfSchur", "E",                             &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfSchur", "E",                             &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfSchur", "E",                             &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "mfSchur", "E",                             &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      n = A%shape(1)
      ! square matrix ?
      if( A%shape(2) /= n ) then
         call PrintMessage( "mfSchur", "E",                             &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      if( mfIsTriu(A) ) then
         call PrintMessage( "mfSchur", "I",                             &
                            "'A' is already upper triangular!" )
         if( A%level_protected == 1 ) then
            call msAssign( out, A )
         else
            out = A
         end if
         out%status_temporary = .true.
         go to 99
      end if

      if( mfIsSymm(A) ) then
         ! A is real, symmetric OR complex, hermitian
         call PrintMessage( "mfSchur", "I",                             &
                            "'A' is symmetric or hermitian...",         &
                            "-> calling 'mfEig'" )
         ! -> calling 'mfEig'
         call msAssign( out, mfEig(A) )
         out%status_temporary = .true.
         go to 99
      end if

      out%shape = [ n, n ]

      ! if necessary, making copies because LAPACK overwrites A
      if( mfIsReal(A) .and. form0 == "C" ) then
         allocate( A_copy ) ! no_mem_trace !
         call msAssign( A_copy, mfComplex(A) )
         A_copy_is_allocated = .true.
      else
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if
      end if

      if( A_copy%data_type == MF_DT_DBLE ) then

         allocate( out%double( out%shape(1), out%shape(2) ) )

         lda = n
         jobvs = 'N' ! Schur vectors are not computed
         sort = 'N' ! Eigenvalues are not sorted

         ! get optimal work-array size
         allocate( work(1) )

         allocate( wr(n), wi(n) )

         allocate( vs(ldvs,n) )

         call dgees( jobvs, sort, select, n, A_copy%double(1,1), lda, sdim, &
                     wr(1), wi(1), vs(1,1), ldvs, work(1), -1, bwork, info )
         lwork = int( work(1) )
         if( info /= 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "mfSchur", "W",                          &
                               "in determining the optimal work-array size by LAPACK:", &
                               "info = " // info_char )
         end if
         if( lwork < 3*n ) then
            call PrintMessage( "mfSchur", "I",                          &
                               "setting work-array size to 3*n" )
            lwork = 3*n
         end if
         deallocate( work )

         allocate( work(lwork) )

         ! General system -- real general matrix : DGEES
         call dgees( jobvs, sort, select, n, A_copy%double(1,1), lda, sdim, &
                     wr(1), wi(1), vs(1,1), ldvs, work(1), lwork, bwork, info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "mfSchur", "W",                          &
                               "in computing Schur form of matrix A by LAPACK:", &
                               "info = " // info_char,                  &
                               "(as a side effect, output will be empty)" )
            call msSilentRelease( out )
         else
            out%data_type = MF_DT_DBLE
            out%double(:,:) = A_copy%double(:,:)
         end if

      else if( A_copy%data_type == MF_DT_CMPLX ) then

         allocate( out%cmplx( out%shape(1), out%shape(2) ) )

         lda = n
         jobvs = 'N' ! Schur vectors are not computed
         sort = 'N' ! Eigenvalues are not sorted

         ! get optimal work-array size
         allocate( cwork(1) )

         allocate( work(n) )

         allocate( w(n) )

         allocate( cvs(ldvs,n) )

         call zgees( jobvs, sort, select, n, A_copy%cmplx(1,1), lda, sdim, &
                     w(1), cvs(1,1), ldvs, cwork(1), -1, work(1), bwork, info )
         lwork = int( cwork(1) )
         if( info /= 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "mfSchur", "W",                          &
                               "in determining the optimal work-array size by LAPACK:", &
                               "info = " // info_char )
         end if
         if( lwork < 2*n ) then
            call PrintMessage( "mfSchur", "I",                          &
                               "setting work-array size to 3*n" )
            lwork = 2*n
         end if
         deallocate( cwork )

         allocate( cwork(lwork) )

         ! General system -- complex general matrix : ZGEES
         call zgees( jobvs, sort, select, n, A_copy%cmplx(1,1), lda, sdim, &
                     w(1), cvs(1,1), ldvs, cwork(1), lwork, work(1), bwork, info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "mfSchur", "W",                          &
                               "in computing Schur form of matrix A by LAPACK:", &
                               "info = " // info_char,                  &
                               "(as a side effect, output will be empty)" )
            call msSilentRelease( out )
         else
            out%data_type = MF_DT_CMPLX
            out%cmplx(:,:) = A_copy%cmplx(:,:)
            out%prop%triu = TRUE
         end if

      end if

      if( A_copy_is_allocated ) then
         call msSilentRelease( A_copy )
         deallocate( A_copy ) ! no_mem_trace !
      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfSchur
