! f90 include file

! currently, uses only:
!     mf_mat_vec_inv_chol()
!     mf_mat_vec_inv_chol_cmplx()
!     mf_mat_vec_inv_spchol()
!     mf_mat_vec_inv_spchol_cmplx()
!
!     mf_mat_vec_2steps()
!     mf_mat_vec_2steps_cmplx()
!     mf_mat_vec_2steps_inv_factor()
!     mf_mat_vec_2steps_inv_factor_cmplx()

!_______________________________________________________________________
!
   function mfSVDS_which( A, k, which, tol, ncv ) result( out )

      type(mfArray) :: A
      integer, intent(in) :: k
      character(len=2), intent(in), optional :: which
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      integer, intent(in), optional :: ncv

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! if A(m,n) is such that m >= n :
      !   'mfSVDS_which_worker' is called with the same arg A
      ! else, it is called with A'
      !
      ! A may be : real/complex, sparse/dense
      !
      ! which = 'LM' : Largest Magnitude
      !         'SM' : Smallest Magnitude
      !         'BE' : Both Ends (algebraic sort, only for real matrices)

      character(len=2) :: which0
      real(kind=MF_DOUBLE) :: tol0
      integer :: ncv0
      integer :: nev, m, n, i, nnz
      logical :: present_ncv, nz_found
      character(len=*), parameter :: name = "mfSVDS"


   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfSVDS", "W",                              &
                            "'A' is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfSVDS", "E",                              &
                            "'A' cannot be boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfSVDS", "E",                              &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( k < 1 ) then
         call PrintMessage( "mfSVDS", "W",                              &
                            "k < 1; output will be empty!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)

      ! if the matrix is sparse and null, but with a number of zero
      ! entry(ies), then the ARPACK routine 'dsaupd' will fail:
      !   * some compilers complains that the ARPACK routine 'dsaupd'
      !     got a null vector ('-9' error on exit);
      !   * others return a vector of NaNs!
      if( mfIsSparse(A) ) then
         ! this check should be very quick, because we exit as soon as a
         ! non-zero entry is found, which should be the quasi majority of
         ! the cases!
         nnz = A%j(n+1) - 1
         nz_found = .false.
         if( mfIsReal(A) ) then
            do i = 1, nnz
               if( A%a(i) /= 0.0d0 ) then
                  nz_found = .true.
                  exit
               end if
            end do
         else ! complex
            do i = 1, nnz
               if( A%z(i) /= (0.0d0,0.0d0) ) then
                  nz_found = .true.
                  exit
               end if
            end do
         end if
         if( .not. nz_found ) then
            ! matrix is null: quick return with a null vector.
            out = mfZeros(k,1)
            out%status_temporary = .true.
            go to 99
         end if
      end if

      if( present(which) ) then
         which0 = to_upper(which)
         if( which0 == "BE" ) then
            if( .not. mfIsReal(A) ) then
               call PrintMessage( "mfSVDS", "E",                        &
                                  "'which' arg may be equal to 'BE' only for real matrices!" )
               go to 99
            end if
         else if( which0 /= "LM" .and. which0 /= "SM" ) then
            call PrintMessage( "mfSVDS", "E",                           &
                               "'which' arg must be equal to 'LM', 'SM' or 'BE'!" )
            go to 99
         end if
      else
         which0 = "LM" ! Default : Largest Eigenvalues
      end if

      ! relative accuracy of the Ritz values
      if( present(tol) ) then
         tol0 = tol
      else
         tol0 = MF_EPS
      end if

      ! allocating for exactly 'k' values
      ! if 'nev' < 'k', remainers will be set to NaN
      out%data_type = MF_DT_DBLE
      out%shape = [ k, 1 ]
      allocate( out%double(k,1) )

      if( m >= n  ) then

         if( k > n-2 ) then
            call PrintMessage( "mfSVDS", "W",                           &
                               "the ARPACK routine used allows only",   &
                               "the computation of n-2 max.",           &
                               "(n being the column number of A)" )
            nev = n-2
         else
            nev = k
         end if

         ! how many Arnoldi vectors are generated
         if( present(ncv) ) then
            present_ncv = .true.
            ncv0 = ncv
            if( ncv0 < 2*nev+1 ) then
               call PrintMessage( "mfSVDS", "I",                        &
                                  "ARPACK: ncv < 2*k+1",                &
                                  "(2*k+1 is the recommended value from the ARPACK User Guide)", &
                                  "-> setting : ncv = 2*k+1" )
               ncv0 = 2*nev+1
            end if
         else
            present_ncv = .false.
            ncv0 = 2*nev+1 ! recommended value from User Guide
         end if
         if( ncv0 > n ) then
            call PrintMessage( "mfSVDS", "I",                           &
                               "ARPACK: ncv > n",                       &
                               "It seems that you are working with a small matrix, or", &
                               "that a great number of eigs have been requested...", &
                               "(ARPACK works well when computing few eigenvalues", &
                               "of a large matrix)",                    &
                               "-> setting : ncv = n" )
            ncv0 = n
         end if

         call SVDS_which_worker( A, nev, k, which0, tol0, ncv0,         &
                                 present_ncv, name, out )

      else

         if( k > m-2 ) then
            call PrintMessage( "mfSVDS", "W",                           &
                               "the ARPACK routine used allows only",   &
                               "the computation of m-2 max.",           &
                               "(m being the row number of A)" )
            nev = m-2
         else
            nev = k
         end if

         ! how many Arnoldi vectors are generated
         if( present(ncv) ) then
            present_ncv = .true.
            ncv0 = ncv
            if( ncv0 < 2*nev+1 ) then
               call PrintMessage( "mfSVDS", "I",                        &
                                  "ARPACK: ncv < 2*k+1",                &
                                  "(2*k+1 is the recommended value from the ARPACK User Guide)", &
                                  "-> setting : ncv = 2*k+1" )
               ncv0 = 2*nev+1
            end if
         else
            present_ncv = .false.
            ncv0 = 2*nev+1 ! recommended value from User Guide
         end if
         if( ncv0 > m ) then
            call PrintMessage( "mfSVDS", "I",                           &
                               "ARPACK: ncv > m",                       &
                               "It seems that you are working with a small matrix, or", &
                               "that a great number of eigs have been requested...", &
                               "(ARPACK works well when computing few eigenvalues", &
                               "of a large matrix)",                    &
                               "-> setting : ncv = m" )
            ncv0 = m
         end if

         if( mfIsComplex(A) ) then
            call SVDS_which_worker( .h.A, nev, k, which0, tol0, ncv0,   &
                                    present_ncv, name, out )
         else ! real case
            call SVDS_which_worker( .t.A, nev, k, which0, tol0, ncv0,   &
                                    present_ncv, name, out )
         end if

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfSVDS_which
!_______________________________________________________________________
!
   subroutine msSVDS_which( out, A, k, which, tol, ncv )

      type(mfArray), target :: A
      integer, intent(in) :: k
      character(len=2), intent(in), optional :: which
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      integer, intent(in), optional :: ncv

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! if A(m,n) is such that m >= n :
      !   'msSVDS_which_worker' is called with the same arg A
      ! else, it is called with A'
      !
      ! A may be : real/complex, sparse/dense
      !
      ! which = 'LM' : Largest Magnitude
      !         'SM' : Smallest Magnitude
      !         'BE' : Both Ends (algebraic sort, only for real matrices)

      character(len=2) :: which0
      real(kind=MF_DOUBLE) :: tol0
      integer :: ncv0
      integer :: nev, m, n, i, nnz
      type(mfArray), pointer :: S, flag
      logical :: present_ncv, nz_found
      character(len=*), parameter :: name = "msSVDS"


   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      ! 2 out-args must be specified
      if( out%n /= 2 ) then
         call PrintMessage( "msSVDS", "E",                              &
                            "two output args required!",               &
                            "syntax is: call msSVDS ( mfOut(S,flag), A, ... )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, A ) ) then
         call PrintMessage( "msSVDS", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      S => out%ptr1
      call msSilentRelease( S )
      flag => out%ptr2
      call msSilentRelease( flag )
      flag = .false.

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "msSVDS", "W",                              &
                            "'A' is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msSVDS", "E",                              &
                            "'A' cannot be boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msSVDS", "E",                              &
                            "'A' cannot be a permutation vector!" )
         go to 99
      end if

      if( k < 1 ) then
         call PrintMessage( "msSVDS", "W",                              &
                            "k < 1; output will be empty!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)

      ! if the matrix is sparse and null, but with a number of zero
      ! entry(ies), then the ARPACK routine 'dsaupd' will fail:
      !   * some compilers complains that the ARPACK routine 'dsaupd'
      !     got a null vector ('-9' error on exit);
      !   * others return a vector of NaNs!
      if( mfIsSparse(A) ) then
         ! this check should be very quick, because we exit as soon as a
         ! non-zero entry is found, which should be the quasi majority of
         ! the cases!
         nnz = A%j(n+1) - 1
         nz_found = .false.
         if( mfIsReal(A) ) then
            do i = 1, nnz
               if( A%a(i) /= 0.0d0 ) then
                  nz_found = .true.
                  exit
               end if
            end do
         else ! complex
            do i = 1, nnz
               if( A%z(i) /= (0.0d0,0.0d0) ) then
                  nz_found = .true.
                  exit
               end if
            end do
         end if
         if( .not. nz_found ) then
            ! matrix is null: quick return with a null vector.
            S = mfZeros(k,1)
            go to 99
         end if
      end if

      if( present(which) ) then
         which0 = to_upper(which)
         if( which0 == "BE" ) then
            if( .not. mfIsReal(A) ) then
               call PrintMessage( "msSVDS", "E",                        &
                                  "'which' arg may be equal to 'BE' only for real matrices!" )
               go to 99
            end if
         else if( which0 /= "LM" .and. which0 /= "SM" ) then
            call PrintMessage( "msSVDS", "E",                           &
                               "'which' arg must be equal to 'LM', 'SM' or 'BE'!" )
            go to 99
         end if
      else
         which0 = "LM" ! Default : Largest Eigenvalues
      end if

      ! relative accuracy of the Ritz values
      if( present(tol) ) then
         tol0 = tol
      else
         tol0 = MF_EPS
      end if

      ! allocating for exactly 'k' values
      ! if 'nev' < 'k', remainers will be set to NaN
      S%data_type = MF_DT_DBLE
      S%shape = [ k, 1 ]
      allocate( S%double(k,1) )

      if( m >= n  ) then

         if( k > n-2 ) then
            call PrintMessage( "msSVDS", "W",                           &
                               "the ARPACK routine used allows only",   &
                               "the computation of n-2 max.",           &
                               "(n being the column number of A)" )
            nev = n-2
         else
            nev = k
         end if

         ! how many Arnoldi vectors are generated
         if( present(ncv) ) then
            present_ncv = .true.
            ncv0 = ncv
            if( ncv0 < 2*nev+1 ) then
               call PrintMessage( "msSVDS", "I",                        &
                                  "ARPACK: ncv < 2*k+1",                &
                                  "(2*k+1 is the recommended value from the ARPACK User Guide)", &
                                  "-> setting : ncv = 2*k+1" )
               ncv0 = 2*nev+1
            end if
         else
            present_ncv = .false.
            ncv0 = 2*nev+1 ! recommended value from User Guide
         end if
         if( ncv0 > n ) then
            call PrintMessage( "msSVDS", "I",                           &
                               "ARPACK: ncv > n",                       &
                               "It seems that you are working with a small matrix, or", &
                               "that a great number of eigs have been requested...", &
                               "(ARPACK works well when computing few eigenvalues", &
                               "of a large matrix)",                    &
                               "-> setting : ncv = n" )
            ncv0 = n
         end if

         call SVDS_which_worker( A, nev, k, which0, tol0, ncv0,         &
                                 present_ncv, name, S, flag )

      else

         if( k > m-2 ) then
            call PrintMessage( "msSVDS", "W",                           &
                               "the ARPACK routine used allows only",   &
                               "the computation of m-2 max.",           &
                               "(m being the row number of A)" )
            nev = m-2
         else
            nev = k
         end if

         ! how many Arnoldi vectors are generated
         if( present(ncv) ) then
            present_ncv = .true.
            ncv0 = ncv
            if( ncv0 < 2*nev+1 ) then
               call PrintMessage( "msSVDS", "I",                        &
                                  "ARPACK: ncv < 2*k+1",                &
                                  "(2*k+1 is the recommended value from the ARPACK User Guide)", &
                                  "-> setting : ncv = 2*k+1" )
               ncv0 = 2*nev+1
            end if
         else
            present_ncv = .false.
            ncv0 = 2*nev+1 ! recommended value from User Guide
         end if
         if( ncv0 > m ) then
            call PrintMessage( "msSVDS", "I",                           &
                               "ARPACK: ncv > m",                       &
                               "It seems that you are working with a small matrix, or",&
                               "that a great number of eigs have been requested...",&
                               "(ARPACK works well when computing few eigenvalues", &
                               "of a large matrix)",                    &
                               "-> setting : ncv = m" )
            ncv0 = m
         end if

         if( mfIsComplex(A) ) then
            call SVDS_which_worker( .h.A, nev, k, which0, tol0, ncv0,   &
                                    present_ncv, name, S, flag )
         else
            call SVDS_which_worker( .t.A, nev, k, which0, tol0, ncv0,   &
                                    present_ncv, name, S, flag )
         end if

      end if

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end subroutine msSVDS_which
!_______________________________________________________________________
!
   subroutine SVDS_which_worker( A, nev, k, which0, tol0, ncv0,         &
                                 present_ncv, name, S, flag )

      type(mfArray), target :: A
      integer, intent(in) :: nev, k
      character(len=2), intent(in) :: which0
      real(kind=MF_DOUBLE), intent(in) :: tol0
      integer :: ncv0
      logical, intent(in) :: present_ncv
      character(len=*), intent(in) :: name
      type(mfArray) :: S
      type(mfArray), optional :: flag
      !------ API end ------
#ifdef _DEVLP

      ! equivalent of 'SVDS' of MATLAB (simplified version)
      !
      ! compute a vector containing 'k' eigenvalues of A'*A
      ! which are the squared singular values of A; then takes the
      ! square root of these values.
      !
      ! if flag is present :
      !   returns a convergence flag (boolean mfArray):
      !     flag = TRUE  all eigenvalues have converged
      !     flag = FALSE on the contrairy
      ! else : print a warning message

      ! A may be : real/complex, sparse/dense
      !
      ! A doesn't need to be square since A'*A is always square,
      ! but we must have: m >= n.
      !
      ! A'*A is symm. and pos. def.
      !
      ! shape(A) = (m,n)  ->  shape(A'*A) = (n,n)
      !
      ! because of the method used (working with A'*A),
      ! a scaling must be applied to A if : norm_1(A) < 10*sqrt(eps)

      integer :: m, n
      logical :: finished

      ! declarations for ARPACK
      integer :: ido, ldv, lworkl, info, ldz, i, nconv
      real(kind=MF_DOUBLE), allocatable :: resid(:), v(:,:)
      real(kind=MF_DOUBLE), allocatable :: workd(:), workl(:)
      real(kind=MF_DOUBLE), allocatable :: d(:)
      real(kind=MF_DOUBLE), allocatable :: rwork(:)
      integer :: iparam(11), ipntr(14)
      logical, allocatable :: select(:)
      real(kind=MF_DOUBLE) :: z(1,1), sigma
      character(len=2) :: which1
      complex(kind=MF_DOUBLE), allocatable :: zresid(:), zv(:,:), zd(:)
      complex(kind=MF_DOUBLE), allocatable :: zworkd(:), zworkl(:), zworkev(:)
      integer :: ldzv, lzworkl, ldzz
      complex(kind=MF_DOUBLE) :: zz(1,1), zsigma
      character(len=6) :: info_char
      character(len=12) :: method

      real(kind=MF_DOUBLE) :: scaling
      logical :: scaled
      type(mfArray), pointer :: B => null() ! A_scaled
      type(mfArray) :: ATA, U
      type(mfMatFactor) :: factor

      ! new sorting of eigenvalues
      type(mfArray) :: mf_dummy, mf_ind
      integer, allocatable :: ind(:)
      character(len=10) :: mode_sort

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      m = A%shape(1)
      n = A%shape(2)

      if( m < n  ) then
         write(STDERR,*) "(MUESLI SVDS_which_worker:) internal error"
         write(STDERR,*) "                            works only for m >= n!"
         write(STDERR,*) "[RETURN] to resume..."
         mf_message_displayed = .true.
         call muesli_trace( pause="yes" )
         stop
      end if

      if( which0 == "SM" ) then
         method = "OP = inv(A)"
         which1 = "LM"
      else
         method = "OP = A"
         which1 = which0
      end if

      scaling = mfNorm(A,1)
      if( scaling < 10.0d0*sqrt(MF_EPS) ) then
         scaled = .true.
         allocate(B) ! no_mem_trace !
         call msAssign( B, A / scaling )
      else
         scaled = .false.
         B => A
      end if

      ! hereafter, we must work with B

      if( which0 == "SM" ) then
         if( mfIsReal(A) ) then
            if( mfIsSparse(A) ) then
               call msAssign( ATA, mfMul(.t.B,B) )
            else
               call msAssign( ATA, mfMul(B,B,transp=1) )
            end if
         else ! complex
            if( mfIsSparse(A) ) then
               call msAssign( ATA, mfMul(.h.B,B) )
            else
               call msAssign( ATA, mfMul(B,B,transp=1) )
            end if
         end if
         ! Warn if matrix is close to singular
         ! RCond is valid only for square matrices
         if( mfDble(mfSqrt(mfRCond(ATA))) <= MF_EPS*10.0d0 ) then
            call PrintMessage( name, "W",                               &
                               "matrix 'A' is close to singular",       &
                               "Computation can be very inaccurate!" )
            method = "OP = A"
            which1 = "SM"
         end if
      end if

      ! using ARPACK-2

      ! here, we are sure that the eigenvalues are real, since the
      ! matrix A'*A is symmetric (real or complex).

      if( present_ncv ) then

         ! computation is done with ncv0, which is not changed
         ! (only one try is made)

         if( .not. mfIsComplex(B) ) then

            ! real (sparse or dense)

            if( method == "OP = inv(A)" ) then
               if( mfIsSparse(B) ) then
                  if( mfIsEmpty(ATA) ) then
                     call msAssign( ATA, mfMul(.t.B,B) ) ! symm. pos. def.
                  end if
                  call msChol_mfMatFactor( ATA, factor )
               else ! dense
                  if( m == n ) then
                     call msLU_mfMatFactor( B, factor )
                  else ! non-square
                     if( mfIsEmpty(ATA) ) then
                        call msAssign( ATA, mfMul(B,B,transp=1) ) ! symm. pos. def.
                     end if
                        call msAssign( U, mfChol(ATA) )
                  end if
               end if
            end if

            !-------------------------------------
            ! preparation
            !-------------------------------------
            ido = 0 ! first call to the reverse communication interface
            allocate( resid(n) ) ! residual vector
            allocate( v(n,ncv0) ) ! set of Arnoldi basis vectors
            ldv = n
            allocate( workd(3*n) ) ! distributed array for reverse comm.
            lworkl = ncv0**2 + 8*ncv0
            allocate( workl(lworkl) ) ! private workspace

            !---------------------------------------
            ! computes eigenvalues only
            !---------------------------------------
            iparam(:) = 0
            iparam(1) = 1 ! method for selecting the implicit shifts
            iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
            if( method == "OP = A" ) then
               iparam(7) = 1 ! type of eigenproblem : OP = A
            else if( method == "OP = inv(A)" ) then
               iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
            end if
            info = 0 ! a randomly initial residual vector
            finished = .false.
            do while( .not. finished )
               call dsaupd( ido,                                        &
                            "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, info )
               if( ido == 1 .or. ido == -1 ) then
                  if( method == "OP = A" ) then
                     call mf_mat_vec_2steps( B, workd(ipntr(1):ipntr(1)+n-1), &
                                                workd(ipntr(2):ipntr(2)+n-1) )
                  else if( method == "OP = inv(A)" ) then
                     if( mfIsSparse(B) ) then
                        call mf_mat_vec_inv_spchol( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                            workd(ipntr(2):ipntr(2)+n-1) )
                     else
                        if( m == n ) then
                           call mf_mat_vec_2steps_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                                      workd(ipntr(2):ipntr(2)+n-1) )
                        else
                           call mf_mat_vec_inv_chol( U, workd(ipntr(1):ipntr(1)+n-1), &
                                                        workd(ipntr(2):ipntr(2)+n-1) )
                        end if
                     end if
                  end if
               else if( ido == 99 ) then
                  finished = .true.
               end if
            end do

            if( factor%data_type /= MF_DT_EMPTY ) then
               call msFreeMatFactor( factor )
            end if
            if( .not. mfIsEmpty(ATA) ) then
               call msSilentRelease( ATA )
            end if

            ! check for some errors
            if( info < 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( name, "E",                            &
                                  "'dsaupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               go to 99
            end if

            ! check for convergence
            nconv = iparam(5)
            if( present(flag) ) then
               if( nconv >= nev ) then
                  flag = .true.
               end if
            else
               if( nconv /= nev ) then
                  call PrintMessage( name, "W",                         &
                                     "ARPACK: nconv /= nev",            &
                                     "Number of converged Ritz values is not equal", &
                                     "to the number of requested eigenvalues.", &
                                     "(try with a greater value of ncv)" )
                  go to 99
               end if
            end if

            !-------------------------------------
            ! no fatal errors occurred
            ! post processing
            !-------------------------------------
            ! .false. : don't want the Ritz vectors
            ! 'A' : compute all the NEV Ritz values
            allocate( select(ncv0) ) ! for the present call : workspace
            allocate( d(nev) ) ! vector of the Ritz values
            ldz = 1
            sigma = 0.0d0
            call dseupd( .false., "A", select(1), d(1), z, ldz, sigma,  &
                         "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                         iparam, ipntr, workd(1), workl(1), lworkl, info )

            if( info == -14 ) then
               call PrintMessage( name, "I",                            &
                                  "ARPACK:",                            &
                                  "dsaupd didn't find any eigenvalues to sufficient accuracy", &
                                  "(try with a greater value of ncv)" )
               if( present(flag) ) then
                  flag = .false.
               end if
               go to 99
            end if

            ! check for some errors
            if( info /= 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( name, "W",                            &
                                  "'dseupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               if( present(flag) ) then
                  flag = .false.
               end if
               go to 99
            end if

            ! sorting eigenvalues
            if( which0 == "LM" ) then
               mode_sort = "descending"
            else
               mode_sort = "ascending"
            end if
            call msSort( mfOut(mf_dummy,mf_ind), mf(d), mode_sort )
            allocate( ind(nev) )
            ind = mf_ind
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               S%double(i,1) = d(ind(i))
            end do
            do i = min(nev,nconv)+1, k
               S%double(i,1) = MF_NAN
            end do

         else

            ! complex (sparse or dense)

            if( method == "OP = inv(A)" ) then
               if( mfIsSparse(B) ) then
                  if( mfIsEmpty(ATA) ) then
                     call msAssign( ATA, mfMul(.h.B,B) ) ! symm. pos. def.
                  end if
                  call msChol_mfMatFactor( ATA, factor )
               else ! dense
                  if( m == n ) then
                     call msLU_mfMatFactor( B, factor )
                  else ! non-square
                     if( mfIsEmpty(ATA) ) then
                        call msAssign( ATA, mfMul(.h.B,B) ) ! symm. pos. def.
                     end if
                        call msAssign( U, mfChol(ATA) )
                  end if
               end if
            end if

            !-------------------------------------
            ! preparation
            !-------------------------------------
            ido = 0 ! first call to the reverse communication interface
            allocate( zresid(n) ) ! residual vector
            allocate( zv(n,ncv0) ) ! set of Arnoldi basis vectors
            ldzv = n
            allocate( zworkd(3*n) ) ! distributed array for reverse comm.
            lzworkl = 3*ncv0**2 + 5*ncv0
            allocate( zworkl(lzworkl) ) ! private workspace
            allocate( rwork(ncv0) ) ! private workspace

            !---------------------------------------
            ! computes eigenvalues only
            !---------------------------------------
            iparam(:) = 0
            iparam(1) = 1 ! method for selecting the implicit shifts
            iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
            if( method == "OP = A" ) then
               iparam(7) = 1 ! type of eigenproblem : OP = A
            else if( method == "OP = inv(A)" ) then
               iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
            end if
            info = 0 ! a randomly initial residual vector
            finished = .false.
            do while( .not. finished )
               call znaupd( ido,                                        &
                            "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                            iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                            info )
               if( ido == 1 .or. ido == -1 ) then
                  if( method == "OP = A" ) then
                     call mf_mat_vec_2steps_cmplx( B, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                      zworkd(ipntr(2):ipntr(2)+n-1) )
                  else if( method == "OP = inv(A)" ) then
                     if( mfIsSparse(B) ) then
                        call mf_mat_vec_inv_spchol_cmplx( factor, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                                  zworkd(ipntr(2):ipntr(2)+n-1) )
                     else
                        if( m == n ) then
                           call mf_mat_vec_2steps_inv_factor_cmplx( factor, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                                            zworkd(ipntr(2):ipntr(2)+n-1) )
                        else
                           call mf_mat_vec_inv_chol_cmplx( U, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                              zworkd(ipntr(2):ipntr(2)+n-1) )
                        end if
                     end if
                  end if
               else if( ido == 99 ) then
                  finished = .true.
               end if
            end do

            if( factor%data_type /= MF_DT_EMPTY ) then
               call msFreeMatFactor( factor )
            end if
            if( .not. mfIsEmpty(ATA) ) then
               call msSilentRelease( ATA )
            end if

            ! check for some errors
            if( info < 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( name, "E",                            &
                                  "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               go to 99
            end if

            ! check for convergence
            nconv = iparam(5)
            if( present(flag) ) then
               if( nconv >= nev ) then
                  flag = .true.
               end if
            else
               if( nconv /= nev ) then
                  call PrintMessage( name, "W",                         &
                                     "ARPACK: nconv /= nev",            &
                                     "Number of converged Ritz values is not equal", &
                                     "to the number of requested eigenvalues.", &
                                     "(try with a greater value of ncv)" )
                  go to 99
               end if
            end if

            !-------------------------------------
            ! no fatal errors occurred
            ! post processing
            !-------------------------------------
            ! .false. : don't want the Ritz vectors
            ! 'A' : compute all the NEV Ritz values
            allocate( select(ncv0) ) ! for the present call : workspace
            allocate( zd(nev+1) ) ! vector of the Ritz values
            ! init. to avoid error detection with valgrind; moreover,
            ! NaN values are useful because they will be ignored by the sort.
            zd(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
            ldzz = 1
            allocate( zworkev(2*ncv0) ) ! private workspace
            zsigma = (0.0d0,0.0d0)
            call zneupd( .false., "A", select(1), zd(1), zz, ldzz, zsigma, zworkev(1), &
                         "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                         iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                         info )

            if( info == -14 ) then
               call PrintMessage( name, "I",                            &
                                  "ARPACK:",                            &
                                  "znaupd didn't find any eigenvalues to sufficient accuracy",                           &
                                  "(try with a greater value of ncv)" )
               if( present(flag) ) then
                  flag = .false.
               end if
               go to 99
            end if

            ! check for some errors
            if( info /= 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( name, "E",                            &
                                  "'zneupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               if( present(flag) ) then
                  flag = .false.
               end if
               go to 99
            end if

            ! sorting eigenvalues
            if( which0 == "LM" ) then
               mode_sort = "descending"
            else
               mode_sort = "ascending"
            end if
            call msSort( mfOut(mf_dummy,mf_ind), mf(real(zd)), mode_sort )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               S%double(i,1) = sqrt( real(zd(ind(i))) )
            end do
            do i = min(nev,nconv)+1, k
               S%double(i,1) = MF_NAN
            end do

         end if

      else ! .not. present(ncv)

         ! ncv chosen automatically
         ! computation starts with ncv0, and this value is increased
         ! until convergence of all singular values...

         if( .not. mfIsComplex(B) ) then

            ! real (sparse or dense)

            allocate( resid(n) ) ! residual vector
            ldv = n
            allocate( workd(3*n) ) ! distributed array for reverse comm.
            info = 0 ! a randomly initial residual vector

            if( method == "OP = inv(A)" ) then
               if( mfIsSparse(B) ) then
                  if( mfIsEmpty(ATA) ) then
                     call msAssign( ATA, mfMul(.t.B,B) ) ! symm. pos. def.
                  end if
                  call msChol_mfMatFactor( ATA, factor )
               else ! dense
                  if( m == n ) then
                     call msLU_mfMatFactor( B, factor )
                  else ! non-square
                     if( mfIsEmpty(ATA) ) then
                        call msAssign( ATA, mfMul(B,B,transp=1) ) ! symm. pos. def.
                     end if
                        call msAssign( U, mfChol(ATA) )
                  end if
               end if
            end if

            do

               !-------------------------------------
               ! preparation
               !-------------------------------------
               ido = 0 ! first call to the reverse communication interface
               allocate( v(n,ncv0) ) ! set of Arnoldi basis vectors
               lworkl = ncv0**2 + 8*ncv0
               allocate( workl(lworkl) ) ! private workspace

               !---------------------------------------
               ! computes eigenvalues only
               !---------------------------------------
               iparam(:) = 0
               iparam(1) = 1 ! method for selecting the implicit shifts
               iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
               if( method == "OP = A" ) then
                  iparam(7) = 1 ! type of eigenproblem : OP = A
               else if( method == "OP = inv(A)" ) then
                  iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
               end if
               finished = .false.
               do while( .not. finished )
                  call dsaupd( ido,                                     &
                               "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                               iparam, ipntr, workd(1), workl(1), lworkl, info )
                  if( ido == 1 .or. ido == -1 ) then
                     if( method == "OP = A" ) then
                        call mf_mat_vec_2steps( B, workd(ipntr(1):ipntr(1)+n-1), &
                                                   workd(ipntr(2):ipntr(2)+n-1) )
                     else if( method == "OP = inv(A)" ) then
                        if( mfIsSparse(B) ) then
                           call mf_mat_vec_inv_spchol( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                               workd(ipntr(2):ipntr(2)+n-1) )
                        else
                           if( m == n ) then
                              call mf_mat_vec_2steps_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                                         workd(ipntr(2):ipntr(2)+n-1) )
                           else
                              call mf_mat_vec_inv_chol( U, workd(ipntr(1):ipntr(1)+n-1), &
                                                           workd(ipntr(2):ipntr(2)+n-1) )
                           end if
                        end if
                     end if
                  else if( ido == 99 ) then
                     finished = .true.
                  end if
               end do

               ! check for some errors
               if( info < 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( name, "E",                         &
                                     "'dsaupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               ! check for convergence
               nconv = iparam(5)
               if( nconv >= nev ) then
                  if( present(flag) ) then
                     flag = .true.
                  end if
                  exit
               end if

               ! we must increase ncv0
               if( ncv0 == n ) then
                  call PrintMessage( name, "I",                         &
                                     "'dsaupd' (ARPACK) cannot compute eigenvalues", &
                                     "(even after increase of ncv)" )
                  if( present(flag) ) then
                     flag = .false.
                  end if
                  go to 99
               end if
               ncv0 = 2*ncv0
               if( ncv0 > n ) then
                  ncv0 = n
               end if
               deallocate( v )
               deallocate( workl )

               info = 1 ! re-uses the residual vector

            end do

            if( factor%data_type /= MF_DT_EMPTY ) then
               call msFreeMatFactor( factor )
            end if
            if( .not. mfIsEmpty(ATA) ) then
               call msSilentRelease( ATA )
            end if
            if( .not. mfIsEmpty(U) ) then
               call msSilentRelease( U )
            end if

            !-------------------------------------
            ! no fatal errors occurred
            ! post processing
            !-------------------------------------
            ! .false. : don't want the Ritz vectors
            ! 'A' : compute all the NEV Ritz values
            allocate( select(ncv0) ) ! for the present call : workspace
            allocate( d(nev) ) ! vector of the Ritz values
            ldz = 1
            sigma = 0.0d0
            call dseupd( .false., "A", select(1), d(1), z, ldz, sigma,  &
                         "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                         iparam, ipntr, workd(1), workl(1), lworkl, info )

            if( info == -14 ) then
               call PrintMessage( name, "I",                            &
                                  "ARPACK:",                            &
                                  "dsaupd didn't find any eigenvalues to sufficient accuracy" )
               if( present(flag) ) then
                  flag = .false.
               end if
               go to 99
            end if

            ! check for some errors
            if( info /= 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( name, "W",                            &
                                  "'dseupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               if( present(flag) ) then
                  flag = .false.
               end if
               go to 99
            end if

            ! sorting eigenvalues
            if( which0 == "LM" ) then
               mode_sort = "descending"
            else
               mode_sort = "ascending"
            end if
            call msSort( mfOut(mf_dummy,mf_ind), mf(d), mode_sort )
            allocate( ind(nev) )
            ind = mf_ind
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               S%double(i,1) = sqrt( d(ind(i)) )
            end do
            do i = min(nev,nconv)+1, k
               S%double(i,1) = MF_NAN
            end do

         else

            ! complex (sparse or dense)

            allocate( zresid(n) ) ! residual vector
            ldzv = n
            allocate( zworkd(3*n) ) ! distributed array for reverse comm.
            info = 0 ! a randomly initial residual vector

            if( method == "OP = inv(A)" ) then
               if( mfIsSparse(B) ) then
                  if( mfIsEmpty(ATA) ) then
                     call msAssign( ATA, mfMul(.h.B,B) ) ! symm. pos. def.
                  end if
                  call msChol_mfMatFactor( ATA, factor )
               else ! dense
                  if( m == n ) then
                     call msLU_mfMatFactor( B, factor )
                  else ! non-square
                     if( mfIsEmpty(ATA) ) then
                        call msAssign( ATA, mfMul(.h.B,B) ) ! symm. pos. def.
                     end if
                        call msAssign( U, mfChol(ATA) )
                  end if
               end if
            end if

            do

               !-------------------------------------
               ! preparation
               !-------------------------------------
               ido = 0 ! first call to the reverse communication interface
               allocate( zv(n,ncv0) ) ! set of Arnoldi basis vectors
               lzworkl = 3*ncv0**2 + 5*ncv0
               allocate( zworkl(lzworkl) ) ! private workspace
               allocate( rwork(ncv0) ) ! private workspace

               !---------------------------------------
               ! computes eigenvalues only
               !---------------------------------------
               iparam(:) = 0
               iparam(1) = 1 ! method for selecting the implicit shifts
               iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
               if( method == "OP = A" ) then
                  iparam(7) = 1 ! type of eigenproblem : OP = A
               else if( method == "OP = inv(A)" ) then
                  iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
               end if
               finished = .false.
               do while( .not. finished )
                  call znaupd( ido,                                     &
                               "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                               iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                               info )
                  if( ido == 1 .or. ido == -1 ) then
                     if( method == "OP = A" ) then
                        call mf_mat_vec_2steps_cmplx( B, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                         zworkd(ipntr(2):ipntr(2)+n-1) )
                     else if( method == "OP = inv(A)" ) then
                        if( mfIsSparse(B) ) then
                           call mf_mat_vec_inv_spchol_cmplx( factor, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                                     zworkd(ipntr(2):ipntr(2)+n-1) )
                        else
                           if( m == n ) then
                              call mf_mat_vec_2steps_inv_factor_cmplx( factor, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                                               zworkd(ipntr(2):ipntr(2)+n-1) )
                           else
                              call mf_mat_vec_inv_chol_cmplx( U, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                                 zworkd(ipntr(2):ipntr(2)+n-1) )
                           end if
                        end if
                     end if
                  else if( ido == 99 ) then
                     finished = .true.
                  end if
               end do

               ! check for some errors
               if( info < 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( name, "E",                         &
                                     "'znaupd' (ARPACK) cannot compute eigenvalues",&
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               ! check for convergence
               nconv = iparam(5)
               if( nconv >= nev ) then
                  if( present(flag) ) then
                     flag = .true.
                  end if
                  exit
               end if

               ! we must increase ncv0
               if( ncv0 == n ) then
                  call PrintMessage( name, "W",                         &
                                     "'znaupd' (ARPACK) cannot compute eigenvalues",&
                                     "(even after increase of ncv)" )
                  if( present(flag) ) then
                     flag = .false.
                  end if
                  go to 99
               end if
               ncv0 = 2*ncv0
               if( ncv0 > n ) then
                  ncv0 = n
               end if
               deallocate( zv )
               deallocate( zworkl )
               deallocate( rwork )

               info = 1 ! re-uses the residual vector

            end do

            if( factor%data_type /= MF_DT_EMPTY ) then
               call msFreeMatFactor( factor )
            end if
            if( .not. mfIsEmpty(ATA) ) then
               call msSilentRelease( ATA )
            end if
            if( .not. mfIsEmpty(U) ) then
               call msSilentRelease( U )
            end if

            !-------------------------------------
            ! no fatal errors occurred
            ! post processing
            !-------------------------------------
            ! .false. : don't want the Ritz vectors
            ! 'A' : compute all the NEV Ritz values
            allocate( select(ncv0) ) ! for the present call : workspace
            allocate( zd(nev+1) ) ! vector of the Ritz values
            ! init. to avoid error detection with valgrind; moreover,
            ! NaN values are useful because they will be ignored by the sort.
            zd(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
            ldzz = 1
            allocate( zworkev(2*ncv0) ) ! private workspace
            zsigma = (0.0d0,0.0d0)
            call zneupd( .false., "A", select(1), zd(1), zz, ldzz, zsigma, zworkev(1), &
                         "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                         iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                         info )

            if( info == -14 ) then
               call PrintMessage( name, "I",                            &
                                  "ARPACK:",                            &
                                  "znaupd didn't find any eigenvalues to sufficient accuracy" )
               if( present(flag) ) then
                  flag = .false.
               end if
               go to 99
            end if

            ! check for some errors
            if( info /= 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( name, "W",                            &
                                  "'zneupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               if( present(flag) ) then
                  flag = .false.
               end if
               go to 99
            end if

            ! sorting eigenvalues
            if( which0 == "LM" ) then
               mode_sort = "descending"
            else
               mode_sort = "ascending"
            end if
            call msSort( mfOut(mf_dummy,mf_ind), mf(real(zd)), mode_sort )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               S%double(i,1) = sqrt( real(zd(ind(i))) )
            end do
            do i = min(nev,nconv)+1, k
               S%double(i,1) = MF_NAN
            end do

         end if

      end if

      if( scaled ) then
         call msRelease( B )
         deallocate( B ) ! no_mem_trace !
         call msAssign( S, S * scaling )
      end if

      if( mf_phys_units ) then
         S%units(:) = B%units(:)
      end if

 99   continue

      call msSilentRelease( mf_dummy, mf_ind )

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end subroutine SVDS_which_worker
