! f90 include file

!_______________________________________________________________________
!
   function mfRDiv_A( b, A ) result( out )

      type(mfArray), intent(in) :: A, b
      type(mfArray)             :: out
      !------ API end ------
#ifdef _DEVLP

      ! routine used for the interface of operator(.xi.)
      ! => it must have exactly two arguments.

      ! solve linear system of the form : x A = b
      ! i.e. compute : b / A

      ! equivalence with :
      !
      ! ( x A )' = b'
      !
      !    A' x' = b'           ->      At x' = out

      integer :: i, m, n, nrhs, lda, ldb, info, rank, lwork, nb
      real(kind=MF_DOUBLE), allocatable :: work(:)
      complex(kind=MF_DOUBLE), allocatable :: cwork(:)
      integer, allocatable :: ipiv(:)
      type(mfArray) :: A_copy, At, R
      character(len=4) :: info_char
      integer :: mf_message_level_save

      ! declarations for UMFPACK
      integer(kind=MF_ADDRESS) :: numeric, symbolic ! LU handle
      integer :: nnz, sys
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      integer, allocatable :: ptr_Aj(:), ptr_Ai(:)
      real(kind=MF_DOUBLE), pointer :: ptr_b(:) => null(),              &
                                       ptr_x(:) => null()
      real(kind=MF_DOUBLE), pointer :: ptr_bz(:) => null(),             &
                                       ptr_xz(:) => null()
      real(kind=MF_DOUBLE), allocatable :: Ar(:), Az(:)

      integer :: status, nnz2

   !------ end of declarations -- execution starts hereafter  ------

      if( A%prop%tril == TRUE .and. A%prop%triu == TRUE ) then

         call msAssign( out, mfRDiv_triang( b, A, pattern="diag" ) )
         out%status_temporary = .true.
         return

      else if( A%prop%tril == TRUE ) then

         call msAssign( out, mfRDiv_triang( b, A, pattern="tril" ) )
         out%status_temporary = .true.
         return

      else if( A%prop%triu == TRUE ) then

         call msAssign( out, mfRDiv_triang( b, A, pattern="triu" ) )
         out%status_temporary = .true.
         return

      end if

      call msInitArgs( A, b )

      ! special case for division of two scalars
      if( mfIsScalar(A) .and. mfIsScalar(b) ) then
         out = b / A
         out%status_temporary = .true.
         go to 99
      end if

      ! verifications...
      if( A%data_type == MF_DT_EMPTY .or.                               &
          b%data_type == MF_DT_EMPTY      ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'A' or 'b' is not allocated!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)
      nrhs = b%shape(1)

      if( b%shape(2) /= n ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'b' must be conformant to 'A'!" )
         go to 99
      end if

      ! SPARSE cases first -- jump to label 89 after
      if( A%data_type == MF_DT_SP_DBLE .and.                            &
          b%data_type == MF_DT_DBLE          ) then

         if( nrhs /= 1 ) then
            call PrintMessage( "mfRDiv", "E",                           &
                               "for a sparse system, only NRHS=1 is valid!", &
                               "(however, you can use the mfLDiv form,", &
                               " using transposition. See documentation)" )
            go to 99
         end if

         nnz = A%j(n+1) - 1

         out%data_type = MF_DT_DBLE
         out%shape = [ 1, m ]
         allocate( out%double(1,m) )

         if( m == n ) then ! real square system

            if( mfIsSymm(A) ) then
               if( A%prop%posd == FALSE ) then
                  go to 12
               else
                  go to 11
               end if
            else
               go to 12
            end if

 11         continue ! symmetric & positive definite
            ! uses CHOLMOD solver (C)
            ! when A is symmetric : A' == A
            call msAssign( A_copy, mfTriu(A) )
            nnz2 = mfNnz(A_copy)
            call cholmod_solve_f90( n, nnz2, A_copy%j, A_copy%i, A_copy%a,  &
                                    b%double(1,:), out%double(1,:), status )

            if( status == 0 ) then
               if( A%prop%posd == UNKNOWN ) then
                  ! cannot use the following statement, because
                  ! A has intent(in)!
                  ! A%prop%posd = TRUE
                  call change_pos_def( A, TRUE )
               end if
               go to 89
            else
               if( A%prop%posd == TRUE ) then
                  ! internal error
                  write(STDERR,*) "(MUESLI mfRDiv:) Internal Error:"
                  write(STDERR,*) "                 mfArray A was tagged as 'Pos. Def.'"
                  write(STDERR,*) "                 but Cholesky factorization failed!"
                  mf_message_displayed = .true.
                  call muesli_trace( pause="yes" )
                  stop
               end if
               call PrintMessage( "mfRDiv", "I",                        &
                                  "Cholesky fails, using LU instead!" )
               ! cannot use the following statement, because
               ! A has intent(in)!
               ! A%prop%posd = FALSE
               call change_pos_def( A, FALSE )
               go to 12
            end if

 12         continue ! unsymmetric
            ! uses UMFPACK-4 solver (C)
            ! sparse matrices must be row-sorted !
            if( A%row_sorted /= TRUE ) then
               call msRowSort(A)
            end if

            ! set UMFPACK-4 default parameters
            call umf4def_d(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! convert from 1-based to 0-based
            allocate( ptr_Aj(n+1) )

            ptr_Aj(1:n+1) = A%j(1:n+1) - 1
            allocate( ptr_Ai(nnz) )

            ptr_Ai(1:nnz) = A%i(1:nnz) - 1
            ! pre-order and symbolic analysis
            call umf4sym_d( m, n, ptr_Aj, ptr_Ai, A%a, symbolic,        &
                            control, infos )
            ! check umf4sym error condition
            if( infos(1) < 0 ) then
               write(STDERR,*) "(MUESLI mfRDiv:) Error occurred in umf4sym_d: ", infos(1)
               if( infos(1) == -8 ) then ! -8 : UMFPACK-version dependant !
                  ! UMFPACK_ERROR_invalid_matrix (-8)
                  ! (the following description has been adapted to fortran-like 1-based
                  !  indexes !)
                  write(STDERR,*) "Number of entries in the matrix is negative, Ap(1) is not 1,"
                  write(STDERR,*) "a column has a negative number of entries, a row index is out of"
                  write(STDERR,*) "bounds, or the columns of input matrix were jumbled (unsorted"
                  write(STDERR,*) "columns or duplicate entries)."
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               go to 99
            end if

            ! numeric factorization
            call umf4num_d( n, ptr_Aj, ptr_Ai, A%a, symbolic, numeric,  &
                            control, infos )

            ! check umf4num error condition
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in umf4num_d: infos(1) = " // info_char )
               go to 99
            end if

            ! solve A'.x=b, without iterative refinement
            sys = 1
            call umf4sol_d( sys, out%double(:,1), b%double(:,1), numeric, &
                            control, infos )
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in umf4sol_d: infos(1) = " // info_char )
               go to 99
            end if

            ! free the symbolic analysis
            call umf4fsym_d(symbolic)
            ! free the numeric factorization
            call umf4fnum_d(numeric)
            ! No LU factors (symbolic or numeric) are in memory at this point.
            go to 89

         else if( n > m ) then ! over-determined system

            ! call the 'spqr_solve_lsq' C routine (SuiteSparseQR)
            ! sparse matrices must be row-sorted
            if( A%row_sorted /= TRUE ) then
               call msRowSort(A)
            end if

! il faudra transposer A dans SPQR, car il n'y a pas (encore ?) de
! routines permettant de travailler directement sur A'
            call spqr_tsolve_lsq( m, n, nnz, A%j, A%i, A%a,             &
                                  b%double(:,1),                        &
                                  out%double(:,1), status )

            ! check for an error
            if( status /= 0 ) then
               call PrintMessage( "mfRDiv", "E",                        &
                                  "spqr_tsolve_lsq failed!" )
               go to 99
            end if

            go to 89

         else ! n < m : under-determined system

            ! call the 'spqr_solve_min2norm' C routine (SuiteSparseQR)
            ! sparse matrices must be row-sorted
            if( A%row_sorted /= TRUE ) then
               call msRowSort(A)
            end if

! il faudra transposer A dans SPQR, car il n'y a pas (encore ?) de
! routines permettant de travailler directement sur A'
            call spqr_tsolve_min2norm( m, n, nnz, A%j, A%i, A%a,        &
                                       b%double(:,1),                   &
                                       out%double(:,1), status )

            ! check for an error
            if( status /= 0 ) then
               call PrintMessage( "mfRDiv", "E",                        &
                                  "spqr_tsolve_min2norm failed!" )
               go to 99
            end if

            go to 89

         end if

      else if( A%data_type == MF_DT_SP_CMPLX .and.                      &
               b%data_type == MF_DT_CMPLX          ) then

         if( nrhs /= 1 ) then
            call PrintMessage( "mfRDiv", "E",                           &
                               "for a sparse system, only NRHS=1 is valid!" )
            go to 99
         end if

         nnz = A%j(n+1) - 1

         out%data_type = MF_DT_CMPLX
         out%shape = [ 1, m ]
         allocate( out%cmplx(1,m) )

         if( m == n ) then ! complex square system

            if( mfIsSymm(A) ) then
               if( A%prop%posd == FALSE ) then
                  go to 22
               else
                  go to 21
               end if
            else
               go to 22
            end if

 21         continue
            ! uses CHOLMOD solver (C)
            ! when A is hermitian : A' == conj(A)
            call msAssign( A_copy, mfTriu(A) )
            nnz2 = mfNnz(A_copy)
            allocate( cwork(n) )
            cwork(:) = conjg(b%cmplx(1,:))
            call cholmod_cmplx_solve( n, nnz2, A_copy%j, A_copy%i,      &
                                      A_copy%z, cwork,                  &
                                      out%cmplx(1,:), status )

            if( status == 0 ) then
               out%cmplx(1,:) = conjg(out%cmplx(1,:))
               if( A%prop%posd == UNKNOWN ) then
                  ! cannot use the following statement, because
                  ! A has intent(in)!
                  ! A%prop%posd = TRUE
                  call change_pos_def( A, TRUE )
               end if
               go to 89
            else
               if( A%prop%posd == TRUE ) then
                  ! internal error
                  write(STDERR,*) "(MUESLI mfRDiv:) Internal Error:"
                  write(STDERR,*) "                 mfArray A was tagged as 'Pos. Def.'"
                  write(STDERR,*) "                 but Cholesky factorization failed!"
                  mf_message_displayed = .true.
                  call muesli_trace( pause="yes" )
                  stop
               end if
               call PrintMessage( "mfRDiv", "I",                        &
                                  "Cholesky fails, using LU instead!" )
               ! cannot use the following statement, because
               ! A has intent(in)!
               ! A%prop%posd = FALSE
               call change_pos_def( A, FALSE )
               go to 22
            end if

 22         continue
            ! uses UMFPACK-4 solver (C)
            ! sparse matrices must be row-sorted !
            if( A%row_sorted /= TRUE ) then
               call msRowSort(A)
            end if

            ! set UMFPACK-4 default parameters
            call umf4def_z(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! convert from 1-based to 0-based
            allocate( ptr_Aj(n+1) )
            ptr_Aj(1:n+1) = A%j(1:n+1) - 1
            allocate( ptr_Ai(nnz) )
            ptr_Ai(1:nnz) = A%i(1:nnz) - 1
            allocate( Ar(nnz) )
            Ar(:) = real(A%z(1:nnz))
            allocate( Az(nnz) )
            ! WARNING : in UMFPACK, the solver (sys=1 -> A'.x=b)
            !           makes a conjg transpose (.h. complex).
            !           Then, we must return the conjuguate
            Az(:) = -aimag(A%z(1:nnz))
            ! pre-order and symbolic analysis
            call umf4sym_z( m, n, ptr_Aj, ptr_Ai, Ar, Az, symbolic,     &
                            control, infos )
            ! check umf4sym error condition
            if( infos(1) < 0 ) then
               write(STDERR,*) "(MUESLI mfRDiv:) Error occurred in umf4sym_z: ", infos(1)
               if( infos(1) == -8 ) then ! -8 : UMFPACK-version dependant !
                  ! UMFPACK_ERROR_invalid_matrix (-8)
                  ! (the following description has been adapted to fortran-like 1-based
                  !  indexes !)
                  write(STDERR,*) "Number of entries in the matrix is negative, Ap(1) is not 1,"
                  write(STDERR,*) "a column has a negative number of entries, a row index is out of"
                  write(STDERR,*) "bounds, or the columns of input matrix were jumbled (unsorted"
                  write(STDERR,*) "columns or duplicate entries)."
               end if
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               go to 99
            end if
            ! numeric factorization
            call umf4num_z( n, ptr_Aj, ptr_Ai, Ar, Az, symbolic, numeric, &
                            control, infos )

            ! check umf4num error condition
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in umf4num_z: infos(1) = " // info_char )
               go to 99
            end if
            ! solve A'.x=b, without iterative refinement
            sys = 1

            allocate( ptr_x(m), ptr_xz(m) )
            allocate( ptr_b(n), ptr_bz(n) )
            ptr_b(:) = real(b%cmplx(1,:))
            ptr_bz(:) = aimag(b%cmplx(1,:))
            call umf4sol_z(sys, ptr_x, ptr_xz, ptr_b, ptr_bz,           &
                           numeric, control, infos)
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in umf4sol_z: infos(1) = " // info_char )
               go to 99
            end if

            out%cmplx(1,:) = cmplx(ptr_x(:),ptr_xz(:),kind=MF_DOUBLE)
            deallocate( ptr_x, ptr_xz, ptr_b, ptr_bz )

            ! free the symbolic analysis
            call umf4fsym_z(symbolic)
            ! free the numeric factorization
            call umf4fnum_z(numeric)

            ! No LU factors (symbolic or numeric) are in memory at this point.
            go to 89

         else if( n > m ) then ! over-determined system

            call PrintMessage( "mfRDiv", "E",                           &
                               "over-determined system : not possible!", &
                               "(no complex routine available to do this job)" )
            go to 99

         else ! n < m : under-determined system

            call PrintMessage( "mfRDiv", "E",                           &
                               "under-determined system : not possible!", &
                               "(no complex routine available to do this job)" )
            go to 99

         end if

      else

         if( mfIsSparse(A) ) then
            call PrintMessage( "mfRDiv", "E",                           &
                               "A [sparse], b [vector] should be both real or complex!" )
         end if

      end if

      ! dense structures now

      rank = mfInt( mfRank(A) )
      if( rank == min(m,n) ) then

         ! full-rank case
         if( m == n ) then ! square system

            if( A%data_type == MF_DT_DBLE .and.                         &
                b%data_type == MF_DT_DBLE ) then
               ! making copies because LAPACK overwrites A and b
               call msAssign( At, .t. A)
               call msAssign( out, .t. b)

               n = At%shape(1)
               nrhs = out%shape(2)
               lda = At%shape(1)
               ldb = out%shape(1)
               allocate( ipiv(n) )

               if( mfIsSymm(A) ) then
                  if( A%prop%posd == FALSE ) then
                     go to 32
                  else
                     go to 31
                  end if
               else
                  go to 33
               end if

 31            continue ! symmetric & positive definite
               ! Symmetric Pos. Def. -- real square matrix : DPOSV
               call dposv( 'U', n, nrhs, At%double(1,1), lda,           &
                           out%double(1,1), ldb, info )
               call msAssign( out, .t. out)
               if( info == 0 ) then
                  if( A%prop%posd == UNKNOWN ) then
                     ! cannot use the following statement, because
                     ! A has intent(in)!
                     ! A%prop%posd = TRUE
                     call change_pos_def( A, TRUE )
                  end if
                  go to 89
               else
                  if( A%prop%posd == TRUE ) then
                     ! internal error
                     write(STDERR,*) "(MUESLI mfRDiv:) Internal Error:"
                     write(STDERR,*) "                 mfArray A was tagged as 'Pos. Def.'"
                     write(STDERR,*) "                 but Cholesky factorization failed!"
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     stop
                  end if
                  call PrintMessage( "mfRDiv", "I",                     &
                                     "Cholesky fails, using LDLT instead!" )
                  ! cannot use the following statement, because
                  ! A has intent(in)!
                  ! A%prop%posd = FALSE
                  call change_pos_def( A, FALSE )
                  ! copie again A, because At has been destroyed
                  call msAssign( At, .t. A)
                  go to 32
               end if

 32            continue ! symmetric indefinite
               ! Symmetric system -- real square matrix : DSYSV

               ! determine optimal blocksize
               allocate( work(1) )
               call dsysv( 'U', n, nrhs, At%double(1,1), lda, ipiv(1),  &
                           out%double(1,1), ldb, work(1), -1, info )
               lwork = n*int( work(1) )
               deallocate( work )
               allocate( work(lwork) )

               call dsysv( 'U', n, nrhs, At%double(1,1), lda, ipiv(1),  &
                           out%double(1,1), ldb, work(1), lwork, info )
               call msAssign( out, .t. out)
               if( info /= 0 ) then
                  ! solution is not ok -- light warning
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfRDiv", "W",                     &
                                     "in solving the system by LAPACK:", &
                                     "info = " // info_char,            &
                                     "as a side effect, solution will be set to NaN" )
                  out%double(:,:) = MF_NAN
               end if
               go to 89

 33            continue

               ! General system -- real square matrix : DGESV
               call dgesv( n, nrhs, At%double(1,1), lda, ipiv(1), out%double(1,1), ldb, info )
               call msAssign( out, .t. out)
               if( info /= 0 ) then
                  ! solution is not ok -- light warning
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfRDiv", "W",                     &
                                     "in solving the system by LAPACK:", &
                                     "info = " // info_char,            &
                                     "as a side effect, solution will be set to NaN" )
                  out%double(:,:) = MF_NAN
               end if
               go to 89

            else ! dense complex case

               mf_message_level_save = mf_message_level
               mf_message_level = 0

               ! making copies because LAPACK overwrites A and b
               call msAssign( At, mfComplex(.t. A))
               call msAssign( out, mfComplex(.t. b))

               n = At%shape(1)
               nrhs = out%shape(2)
               lda = At%shape(1)
               ldb = out%shape(1)
               allocate( ipiv(n) )

               if( mfIsSymm(A) ) then ! is hermitian ?
                  if( A%prop%posd == FALSE ) then
                     go to 42
                  else
                     go to 41
                  end if
               else
                  go to 43
               end if

 41            continue ! hermitian & positive definite
               ! Hermitian Pos. Def. -- complex square matrix : ZPOSV
               call zposv( 'U', n, nrhs, At%cmplx(1,1), lda,            &
                           out%cmplx(1,1), ldb, info )
               call msAssign( out, .t. out)
               if( info == 0 ) then
                  if( A%prop%posd == UNKNOWN ) then
                     ! cannot use the following statement, because
                     ! A has intent(in)!
                     ! A%prop%posd = TRUE
                     call change_pos_def( A, TRUE )
                  end if
                  mf_message_level = mf_message_level_save
                  go to 89
               else
                  if( A%prop%posd == TRUE ) then
                     ! internal error
                     write(STDERR,*) "(MUESLI mfRDiv:) Internal Error:"
                     write(STDERR,*) "                 mfArray A was tagged as 'Pos. Def.'"
                     write(STDERR,*) "                 but Cholesky factorization failed!"
                     mf_message_displayed = .true.
                     call muesli_trace( pause="yes" )
                     stop
                  end if
                  call PrintMessage( "mfRDiv", "I",                     &
                                     "Cholesky fails, using LDLT instead!" )
                  ! cannot use the following statement, because
                  ! A has intent(in)!
                  ! A%prop%posd = FALSE
                  call change_pos_def( A, FALSE )
                  ! copy again A, because A_copy has been destroyed
                  call msAssign( At, mfComplex(.t. A))
                  go to 42
               end if

 42            continue ! hermitian indefinite
               ! Hermitian system -- complex square matrix : ZHESV

               ! determine optimal blocksize
               allocate( cwork(1) )
               call zhesv( 'U', n, nrhs, At%cmplx(1,1), lda, ipiv(1),   &
                           out%cmplx(1,1), ldb, cwork(1), -1, info )
               lwork = n*int( cwork(1) )
               deallocate( cwork )
               allocate( cwork(lwork) )

               call zhesv( 'U', n, nrhs, At%cmplx(1,1), lda, ipiv(1),   &
                           out%cmplx(1,1), ldb, cwork(1), lwork, info )
               call msAssign( out, .t. out)
               if( info /= 0 ) then
                  ! solution is not ok -- light warning
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfRDiv", "W",                     &
                                     "in solving the system by LAPACK:", &
                                     "info = " // info_char,            &
                                     "as a side effect, solution will be set to NaN" )
                  out%cmplx(:,:) = MF_NAN*(1.0d0,1.0d0)
               end if
               mf_message_level = mf_message_level_save
               go to 89

 43            continue
               ! General system -- complex square matrix : ZGESV
               call zgesv( n, nrhs, At%cmplx(1,1), lda, ipiv(1), out%cmplx(1,1), ldb, info )
               call msAssign( out, .t. out)
               if( info /= 0 ) then
                  ! solution is not ok -- light warning
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfRDiv", "W",                     &
                                     "in solving the system by LAPACK:", &
                                     "info = " // info_char,            &
                                     "as a side effect, solution will be set to NaN" )
                  out%cmplx(:,:) = MF_NAN*(1.0d0,1.0d0)
               end if
               mf_message_level = mf_message_level_save
               go to 89

            end if

         else ! over- or under-determined system

            if( A%data_type == MF_DT_DBLE .and. b%data_type == MF_DT_DBLE ) then

               ! making copies because LAPACK overwrites A and b
               call msAssign( At, .t. A)
               call msAssign( out, .t. b)

               m = At%shape(1)
               n = At%shape(2)
               nrhs = out%shape(2)
               lda = At%shape(1)
               ldb = out%shape(1)

               ! special case under-determined: the vector 'b' must be
               ! extended to 'n' (in order to get the vector solution!)
               if( m < n ) then
                  call msAssign( out, out .vc. mfZeros(n-m,1))
                  ldb = size(out%double,1)
               end if

               ! optimal blocksize for DGELS (from DGELS and ILAENV)
               nb = 32
               lwork = max(1,min(m,n)+max(min(m,n),nrhs)*nb)
               allocate( work(lwork) )

               ! General system -- real general matrix : DGELS
               call dgels( 'no trans', m, n, nrhs, At%double(1,1), lda, &
                           out%double(1,1), ldb, work(1), lwork, info )
               if( info /= 0 ) then
                  ! solution is not ok -- light warning
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfRDiv", "W",                     &
                                     "in solving the system by LAPACK:", &
                                     "info = " // info_char,            &
                                     "as a side effect, solution will be set to NaN" )
                  out%double(:,:) = MF_NAN
               end if
               ! special case over-determined: the vector 'b' must be
               ! restricted to 'n'
               if( m > n ) then
                  call msAssign( out, mfGet( out, [(i,i=1,n)], MF_COLON ))
               end if

               call msAssign( out, .t. out)

            else if( A%data_type == MF_DT_CMPLX .and. b%data_type == MF_DT_CMPLX ) then

               mf_message_level_save = mf_message_level
               mf_message_level = 0

               ! making copies because LAPACK overwrites A and b
               call msAssign( At, mfComplex(.t. A))
               call msAssign( out, mfComplex(.t. b))

               m = At%shape(1)
               n = At%shape(2)
               nrhs = out%shape(2)
               lda = At%shape(1)
               ldb = out%shape(1)

               ! special case under-determined: the vector 'b' must be
               ! extended to 'n' (in order to get the vector solution!)
               if( m < n ) then
                  call msAssign( out, out .vc. mfZeros(n-m,1)*MF_I)
                  ldb = size(out%cmplx,1)
               end if

               lda = m
               ! optimal blocksize for ZGELS (from ZGELS and ILAENV)
               nb = 32
               lwork = max(1,min(m,n)+max(min(m,n),nrhs)*nb)
               allocate( cwork(lwork) )

               ! General system -- real general matrix : DGELS
               call zgels( 'no trans', m, n, nrhs, At%cmplx(1,1), lda,  &
                           out%cmplx(1,1), ldb, cwork(1), lwork, info )
               if( info /= 0 ) then
                  ! solution is not ok -- light warning
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfRDiv", "W",                     &
                                     "in solving the system by LAPACK:", &
                                     "info = " // info_char,            &
                                     "as a side effect, solution will be set to NaN" )
                  out%cmplx(:,:) = MF_NAN*(1.0d0,1.0d0)
               end if
               ! special case over-determined: the vector 'b' must be
               ! restricted to 'n'
               if( m > n ) then
                  call msAssign( out, mfGet( out, [(i,i=1,n)], MF_COLON ))
               end if

               call msAssign( out, .t. out)

               mf_message_level = mf_message_level_save

            else
               if( mfIsDense(A) .and. mfIsSparse(b) ) then
                  if( mfIsVector(A) .and. mfIsMatrix(b) ) then
                     call PrintMessage( "mfRDiv", "W",                  &
                                        "I detected a strange case in your call...", &
                                        "Did you thought solving x*A=b and calling mfRDiv(A,b), which is not", &
                                        "a correct call? I suspect that the two arguments have been swapped...", &
                                        " -> see the documentation about the 'mfRDiv' routine.")
                  end if
               end if
               call PrintMessage( "mfRDiv", "E",                        &
                                  "Wrong arg. types?",                  &
                                  "(for a non-square system, both arg. must be real or complex!)" )

            end if

         end if

      else ! rank deficient

         call PrintMessage( "mfRDiv", "W",                              &
                            "A is not inversible!",                    &
                            "Moore-Penrose pseudo-inverse will be used to compute", &
                            "the solution." )
         ! A(m,n) => A_inv(n,m)
         ! ok whatever (m,n)
         call msAssign( out, mfMul(b,mfPseudoInv(A)) )

      end if

 89   continue ! normal termination

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( b%units(i), A%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfRDiv:) internal error: in processing physical units:"
               write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue ! some error occured

      call msSilentRelease( A_copy, At, R )

      call msFreeArgs( A, b )

      call msAutoRelease( A, b )

#endif
   end function mfRDiv_A
!_______________________________________________________________________
!
   function mfRDiv_triang( b, A, pattern ) result( out )

      type(mfArray)    :: A, b
      character(len=*) :: pattern

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! internal routine

      ! * 'pattern' must be : "triu", "tril" or "diag"
      !     solve triangular square system of the form : x A = b
      !     i.e. compute : x = b / A

      ! A may have a dense (LAPACK called) or sparse storage.
      ! Handles mixed data type

      integer :: i, j, n, ldb, nrhs
      integer :: status
      character(len=10) :: char_case
      type(mfArray) :: tmp, d
      integer :: mf_message_level_save

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A, b )

      if( pattern /= "triu" .and. pattern /= "tril"                     &
                            .and. pattern /= "diag" ) then
         call PrintMessage( "mfRDiv_triang", "internal error",          &
                            "'option' must be equal to 'triu', 'tril' or 'diag'!" )
         go to 99

      end if

      if( mfIsSparse(b) ) then
         call PrintMessage( "mfRDiv (triangular system)", "E",          &
                            "sparse triangular solving supports only dense RHS!" )
         go to 99
      end if

      ! verifications...
      if( A%data_type == MF_DT_EMPTY .or.                               &
          b%data_type == MF_DT_EMPTY      ) then
         call PrintMessage( "mfRDiv (triangular system)", "E",          &
                            "'A' or 'b' is not allocated!" )
         go to 99
      end if

      n = A%shape(1)
      if( A%shape(2) /= n ) then
         call PrintMessage( "mfRDiv (triangular system)", "E",          &
                            "'A' must be square!" )
         go to 99
      end if

      nrhs = b%shape(1)
      ldb = b%shape(2)

      if( ldb /= n ) then
         call PrintMessage( "mfRDiv (triangular system)", "E",          &
                            "'b' has a wrong size!" )
         go to 99
      end if

      mf_message_level_save = mf_message_level
      mf_message_level = 0
      ! prepare for mixte data type
      if( mfIsReal(A) .and. mfIsReal(b) ) then
         char_case = "real"
         if( mfIsDense(A) ) then
            out = b
         end if
      else if( mfIsComplex(A) .and. mfIsComplex(b) ) then
         char_case = "complex"
         out = b
      else if( mfIsComplex(A) .and. mfIsReal(b) ) then
         char_case = "complex"
         call msAssign( out, mfComplex(b) )
      else if( mfIsReal(A) .and. mfIsComplex(b) ) then
         char_case = "complex2"
         if( mfIsDense(A) ) then
            out = b
         end if
      end if
      mf_message_level = mf_message_level_save

      select case( char_case )
      case( "real" )

         if( pattern == "tril" ) then

            if( mfIsSparse(A) ) then

               out%data_type = MF_DT_DBLE
               out%shape = [ nrhs, n ]
               allocate( out%double(nrhs,n) )

               ! trilsol accepts only row sorted matrices
               if( .not. mfIsRowSorted(A) ) then
                  ! sorting (whatever MF_SP_AUTO_ROW_SORTED)
                  call row_sort( n, n, A%a, A%i, A%j )
                  A%row_sorted = TRUE
               end if
               if( nrhs == 1 ) then
                  call trilsol_transp( n, out%double(1,:), b%double(1,:), &
                                       A%a, A%j, A%i )
               else
                  call trilsol_transp_nrhs( n, out%double(:,:), b%double(:,:), &
                                            A%a, A%j, A%i )
               end if

            else

               ! Lower triangular system -- real square matrix : DTRSM
               ! assuming that A is "tril", but non-unit on diagonal
               call dtrsm( 'left', 'lower', 'transpose', 'non-unit', n, nrhs, &
                           1.0d0, A%double(1,1), n, out%double(1,1), ldb )

            end if

         else if( pattern == "triu" ) then

            if( mfIsSparse(A) ) then

               out%data_type = MF_DT_DBLE
               out%shape = [ nrhs, n ]
               allocate( out%double(nrhs,n) )

               ! triusol accepts only row sorted matrices
               if( .not. mfIsRowSorted(A) ) then
                  ! sorting (whatever MF_SP_AUTO_ROW_SORTED)
                  call row_sort( n, n, A%a, A%i, A%j )
                  A%row_sorted = TRUE
               end if
               if( nrhs == 1 ) then
                  call triusol_transp( n, out%double(1,:), b%double(1,:), &
                                       A%a, A%j, A%i )
               else
                  call triusol_transp_nrhs( n, out%double(:,:), b%double(:,:), &
                                            A%a, A%j, A%i )
               end if

            else

               ! Upper triangular system -- real square matrix : DTRSM
               ! assuming that A is "triu", but non-unit on diagonal
               call dtrsm( 'left', 'upper', 'transpose', 'non-unit', n, nrhs, &
                           1.0d0, A%double(1,1), n, out%double(1,1), ldb )

            end if

         else ! "diag"

            if( mfIsSparse(A) ) then

               out%data_type = MF_DT_DBLE
               out%shape = [ nrhs, n ]
               allocate( out%double(nrhs,n) )

               call msAssign( d, mfDiag(A) ) ! row vector
               do j = 1, n
                  do i = 1, nrhs
                     out%double(i,j) = b%double(i,j)/d%double(1,j)
                  end do
               end do

            else

               do j = 1, n
                  do i = 1, nrhs
                     out%double(i,j) = out%double(i,j)/A%double(j,j)
                  end do
               end do

            end if

         end if

      case( "complex" )

         if( pattern == "tril" ) then

            if( mfIsSparse(A) ) then

               ! trilsol accepts only row sorted matrices
               if( .not. mfIsRowSorted(A) ) then
                  ! sorting (whatever MF_SP_AUTO_ROW_SORTED)
                  call row_sort_cmplx( n, n, A%z, A%i, A%j )
                  A%row_sorted = TRUE
               end if
               ! trilsol accepts same object (with same address)
               ! for 'out' and 'b'
               if( nrhs == 1 ) then
                  call trilsol_transp_c_c( n, out%cmplx(1,:), out%cmplx(1,:), &
                                           A%z, A%j, A%i )
               else
                  call trilsol_transp_nrhs_c_c( n, out%cmplx(:,:), out%cmplx(:,:), &
                                                A%z, A%j, A%i )
               end if

            else

               ! Lower triangular system -- real square matrix : ZTRSM
               ! assuming that A is "tril", but non-unit on diagonal
               call ztrsm( 'left', 'lower', 'transpose', 'non-unit', n, nrhs, &
                           (1.0d0,0.0d0), A%cmplx(1,1), n, out%cmplx(1,1), ldb )

            end if

         else if( pattern == "triu" ) then

            if( mfIsSparse(A) ) then

               ! triusol accepts only row sorted matrices
               if( .not. mfIsRowSorted(A) ) then
                  ! sorting (whatever MF_SP_AUTO_ROW_SORTED)
                  call row_sort_cmplx( n, n, A%z, A%i, A%j )
                  A%row_sorted = TRUE
               end if
               ! trilsol accepts same object (with same address)
               ! for 'out' and 'b'
               if( nrhs == 1 ) then
                  call triusol_transp_c_c( n, out%cmplx(1,:), out%cmplx(1,:), &
                                           A%z, A%j, A%i )
               else
                  call triusol_transp_nrhs_c_c( n, out%cmplx(:,:), out%cmplx(:,:), &
                                                A%z, A%j, A%i )
               end if

            else

               ! Upper triangular system -- real square matrix : ZTRSM
               ! assuming that A is "triu", but non-unit on diagonal
               call ztrsm( 'left', 'upper', 'transpose', 'non-unit', n, nrhs, &
                           (1.0d0,0.0d0), A%cmplx(1,1), n, out%cmplx(1,1), ldb )

            end if

         else ! "diag"

            if( mfIsSparse(A) ) then

               call msAssign( d, mfDiag(A) ) ! row vector
               do j = 1, n
                  do i = 1, nrhs
                     out%cmplx(i,j) = out%cmplx(i,j)/d%cmplx(1,j)
                  end do
               end do

            else

               do j = 1, n
                  do i = 1, nrhs
                     out%cmplx(i,j) = out%cmplx(i,j)/A%cmplx(j,j)
                  end do
               end do

            end if

         end if

      case( "complex2" )

         call msAssign( tmp, mfComplex(A) )
         if( pattern == "tril" ) then

            if( mfIsSparse(A) ) then

               out%data_type = MF_DT_CMPLX
               out%shape = [ nrhs, n ]
               allocate( out%cmplx(nrhs,n) )

               ! trilsol accepts only row sorted matrices
               if( .not. mfIsRowSorted(A) ) then
                  ! sorting (whatever MF_SP_AUTO_ROW_SORTED)
                  call row_sort( n, n, A%a, A%i, A%j )
                  A%row_sorted = TRUE
               end if
               if( nrhs == 1 ) then
                  call trilsol_transp_r_c( n, out%cmplx(1,:), b%cmplx(1,:), &
                                           A%a, A%j, A%i )
               else
                  call trilsol_transp_nrhs_r_c( n, out%cmplx(:,:), b%cmplx(:,:), &
                                                A%a, A%j, A%i )
               end if

            else

               ! Lower triangular system -- real square matrix : ZTRSM
               ! assuming that A is "tril", but non-unit on diagonal
               call ztrsm( 'left', 'lower', 'transpose', 'non-unit', n, nrhs, &
                           (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )

            end if

         else if( pattern == "triu" ) then

            if( mfIsSparse(A) ) then

               out%data_type = MF_DT_CMPLX
               out%shape = [ nrhs, n ]
               allocate( out%cmplx(nrhs,n) )

               ! triusol accepts only row sorted matrices
               if( .not. mfIsRowSorted(A) ) then
                  ! sorting (whatever MF_SP_AUTO_ROW_SORTED)
                  call row_sort( n, n, A%a, A%i, A%j )
                  A%row_sorted = TRUE
               end if
               if( nrhs == 1 ) then
                  call triusol_transp_r_c( n, out%cmplx(1,:), b%cmplx(1,:), &
                                           A%a, A%j, A%i )
               else
                  call triusol_transp_nrhs_r_c( n, out%cmplx(:,:), b%cmplx(:,:), &
                                                A%a, A%j, A%i )
               end if

            else

               ! Upper triangular system -- real square matrix : ZTRSM
               ! assuming that A is "triu", but non-unit on diagonal
               call ztrsm( 'left', 'upper', 'transpose', 'non-unit', n, nrhs, &
                           (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )

            end if

         else ! "diag"

            if( mfIsSparse(A) ) then

               out%data_type = MF_DT_CMPLX
               out%shape = [ nrhs, n ]
               allocate( out%cmplx(nrhs,n) )

               call msAssign( d, mfDiag(A) ) ! row vector
               do j = 1, n
                  do i = 1, nrhs
                     out%cmplx(i,j) = b%cmplx(i,j)/d%double(1,j)
                  end do
               end do

            else

               do j = 1, n
                  do i = 1, nrhs
                     out%cmplx(i,j) = out%cmplx(i,j)/A%double(j,j)
                  end do
               end do

            end if

         end if

         call msSilentRelease( tmp )

      end select

      call msSilentRelease( d )

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( b%units(i), A%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfRDiv (triangular system)", "E",    &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfRDiv_triang:) internal error: in processing physical units:"
               write(STDERR,*) "                        Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A, b )

      call msAutoRelease( A, b )

#endif
   end function mfRDiv_triang
!_______________________________________________________________________
!
   function mfRDiv_SPD( b, U ) result( out )

      type(mfArray) :: U, b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! internal routine

      ! Symm. Pos. Def. case : U is the Cholesky factor

      !   solve linear system of the form : x U' U = b
      !     i.e. compute : y = b / U        [ <=> y' = U' \ b' ]
      !             then : x = y / U'       [ <=> x' = U  \ y' ]

      ! U must have a dense storage. (LAPACK called)
      ! handles mixed data type

      integer :: i, n, ldb, nrhs
      integer :: status
      character(len=10) :: char_case
      type(mfArray) :: tmp

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( U, b )

      if( mfIsSparse(U) .or. mfIsSparse(b) ) then
         call PrintMessage( "mfRDiv (Symm. Pos. Def.)", "E",            &
                            "optional arg 'option' is valid only for dense matrices!" )
         go to 99
      end if

      ! verifications...
      if( U%data_type == MF_DT_EMPTY .or.                               &
          b%data_type == MF_DT_EMPTY      ) then
         call PrintMessage( "mfRDiv (Symm. Pos. Def.)", "E",            &
                            "'U' or 'b' is not allocated!" )
         go to 99
      end if

      n = U%shape(1)
      if( U%shape(2) /= n ) then
         call PrintMessage( "mfRDiv (Symm. Pos. Def.)", "E",            &
                            "'U' must be square!" )
         go to 99
      end if

      nrhs = b%shape(1)
      ldb = b%shape(2)

      if( ldb /= n ) then
         call PrintMessage( "mfRDiv (Symm. Pos. Def.)", "E",            &
                            "'b' has a wrong size!" )
         go to 99
      end if

      ! copying: out = b'

      ! prepare for mixte data type
      if( U%data_type == MF_DT_DBLE .and.                               &
          b%data_type == MF_DT_DBLE       ) then
         char_case = "real"
         call msAssign( out, .t. b )
      else if( U%data_type == MF_DT_CMPLX .and.                         &
               b%data_type == MF_DT_CMPLX       ) then
         char_case = "complex"
         call msAssign( out, .h. b )
      else if( U%data_type == MF_DT_CMPLX .and.                         &
               b%data_type == MF_DT_DBLE       ) then

         char_case = "complex"
         call msAssign( out, .t. mfComplex(b) )
      else if( U%data_type == MF_DT_DBLE .and.                          &
               b%data_type == MF_DT_CMPLX       ) then
         char_case = "complex2"
         call msAssign( out, .h. b )
      end if

      select case( char_case )
      case( "real" )

         ! solve U' y = out, overwriting out with y
         call dtrsm( 'left', 'upper', 'transpose', 'non-unit', n, nrhs, &
                     1.0d0, U%double(1,1), n, out%double(1,1), ldb )
         ! solve U x = out, overwriting out with x
         call dtrsm( 'left', 'upper', 'no transpose', 'non-unit', n, nrhs, &
                     1.0d0, U%double(1,1), n, out%double(1,1), ldb )

         ! last transposing again out.
         call msAssign( out, .t. out )

      case( "complex" )

         ! solve U' y = out, overwriting out with y
         call ztrsm( 'left', 'upper', 'conjg transpose', 'non-unit', n, nrhs, &
                     (1.0d0,0.0d0), U%cmplx(1,1), n, out%cmplx(1,1), ldb )
         ! solve U x = out, overwriting out with x
         call ztrsm( 'left', 'upper', 'no transpose', 'non-unit', n, nrhs, &
                     (1.0d0,0.0d0), U%cmplx(1,1), n, out%cmplx(1,1), ldb )

         ! last transposing again out.
         call msAssign( out, .h. out )

      case( "complex2" )

         call msAssign( tmp, mfComplex(U) )

         ! solve U' y = out, overwriting out with y
         call ztrsm( 'left', 'upper', 'conjg transpose', 'non-unit', n, nrhs, &
                     (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )
         ! solve U x = out, overwriting out with x
         call msAssign( tmp, mfComplex(U) )
         call ztrsm( 'left', 'upper', 'no transpose', 'non-unit', n, nrhs, &
                     (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )

         call msSilentRelease( tmp )

         ! last transposing again out.
         call msAssign( out, .h. out )

      end select

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( b%units(i), U%units(i),                  &
                               out%units(i), status )
            call rational_sub( out%units(i), U%units(i),                &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfRDiv (Symm. Pos. Def.)", "E",      &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfRDiv_SPD:) internal error: in processing physical units:"
               write(STDERR,*) "                     Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( U, b )

      call msAutoRelease( U, b )

#endif
   end function mfRDiv_SPD
