! f90 include file

!_______________________________________________________________________
!
   function mfRDiv_two_mf( b, A, form ) result( out )

      type(mfArray)              :: A, b
      character(len=*), optional :: form

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! driver routine for the case where two mfArray are in the
      ! argument's list.

      !    form : "cholesky"

      ! routines called:
      !   mfRDiv_A( b, A )
      !   mfRDiv_SPD( b, U )

   !------ end of declarations -- execution starts hereafter  ------

      if( present(form) ) then
         if( to_lower(form) /= "cholesky" ) then
            call PrintMessage( "mfRDiv", "E",                           &
                               "optional argument 'form' must be equal to 'cholesky'!" )
         end if
            call msAssign( out, mfRDiv_SPD( b, A ) )
      else ! form not present
         call msAssign( out, mfRDiv_A( b, A ) )
      end if

      out%status_temporary = .true.

#endif
   end function mfRDiv_two_mf
!_______________________________________________________________________
!
   function mfRDiv_three_mf( b, L, U ) result( out )

      type(mfArray) :: L, U, b

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! solve linear system of the form : x L U = b
      ! i.e. compute : y = b / U        ( <=> U' \ b )
      !         then : x = y / L        ( <=> L' \ y )

      ! handles only the SPARSE case (UMFPack)
      !
      ! the 'numeric' C structure of UMFPack contains all necessary
      ! data (i.e., L, U, P, Q, R)
      !
      ! L and U are passed as argument, but actually are not used !
      !
      ! L%umf4_ptr_numeric is a pointer to the previous C structure

      ! declarations for UMFPACK
      integer(kind=MF_ADDRESS) :: numeric ! LU handle
      integer :: sys
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      real(kind=MF_DOUBLE), pointer :: ptr_b(:) => null(),              &
                                       ptr_x(:) => null()
      real(kind=MF_DOUBLE), pointer :: ptr_bz(:) => null(),             &
                                       ptr_xz(:) => null()

      integer :: i, n, ldb, nrhs
      character(len=4) :: info_char
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( L, U, b )

      ! verifications...
      if( L%data_type == MF_DT_EMPTY .or.                               &
          U%data_type == MF_DT_EMPTY .or.                               &
          b%data_type == MF_DT_EMPTY      ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'L', 'U' or 'b' is not allocated!" )
         go to 99
      end if

      n = L%shape(1)
      if( L%shape(2) /= n ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'L' must be square!" )
         go to 99
      end if

      if( any(U%shape /= [n,n]) ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'U' must have same size than 'L'!" )
         go to 99
      end if

      nrhs = b%shape(1)
      ldb = b%shape(2)

      if( ldb /= n ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'b' has a wrong size!" )
         go to 99
      end if

      if( L%data_type == MF_DT_SP_DBLE .and.                            &
          b%data_type == MF_DT_DBLE          ) then

         if( nrhs /= 1 ) then
            call PrintMessage( "mfRDiv", "E",                           &
                               "for a sparse system, only NRHS=1 is valid!" )
            go to 99
         end if

         if( associated( L%umf4_ptr_numeric ) ) then
            out%data_type = MF_DT_DBLE
            out%shape = [ 1, n ]
            allocate( out%double(1,n) )

            ! set UMFPACK-4 default parameters
            call umf4def_d(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! solve A'.x=b, without iterative refinement
            sys = 1
            numeric = L%umf4_ptr_numeric
            call umf4sol_d( sys, out%double(:,1), b%double(:,1), numeric, &
                            control, infos )
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in umf4sol_d: infos(1) = " // info_char )
               go to 99
            end if

         else
            call PrintMessage( "mfRDiv", "E",                           &
                               "'L' and 'U' factors passed to this routine must have", &
                               "been obtained with the 'msLU' routine!" )
            go to 99
         end if

      else if( L%data_type == MF_DT_SP_CMPLX .and.                      &
               b%data_type == MF_DT_CMPLX          ) then

         if( nrhs /= 1 ) then
            call PrintMessage( "mfRDiv", "E",                           &
                               "for a sparse system, only NRHS=1 is valid!" )
            go to 99
         end if

         if( associated( L%umf4_ptr_numeric ) ) then
            out%data_type = MF_DT_CMPLX
            out%shape = [ 1, n ]
            allocate( out%cmplx(1,n) )

            ! set UMFPACK-4 default parameters
            call umf4def_z(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! solve A'.x=b, without iterative refinement
            sys = 1
            numeric = L%umf4_ptr_numeric

            allocate( ptr_x(n), ptr_xz(n) )
            allocate( ptr_b(n), ptr_bz(n) )
            ! taking the conjugate of b ?
            ptr_b(:)  =   real(b%cmplx(1,:))
            ptr_bz(:) = -aimag(b%cmplx(1,:))
            call umf4sol_z( sys, ptr_x, ptr_xz, ptr_b, ptr_bz,          &
                            numeric, control, infos )
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in umf4sol_d: infos(1) = " // info_char )
               go to 99
            end if

            out%cmplx(1,:) = cmplx(ptr_x(:),ptr_xz(:),kind=MF_DOUBLE)
            deallocate( ptr_x, ptr_xz, ptr_b, ptr_bz )

         else
            call PrintMessage( "mfRDiv", "E",                           &
                               "'L' and 'U' factors passed to this routine must have", &
                               "been obtained with the 'msLU' routine!" )
            go to 99
         end if

      else

         call PrintMessage( "mfRDiv", "E",                              &
                            "bad data type!" )
         go to 99

      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( b%units(i), L%units(i),                  &
                               out%units(i), status )
            call rational_sub( out%units(i), U%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfRDiv:) internal error: in processing physical units:"
               write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( L, U, b )

      call msAutoRelease( L, U, b )

#endif
   end function mfRDiv_three_mf
!_______________________________________________________________________
!
   function mfRDiv_four_mf( b, L, U, p ) result( out )

      type(mfArray) :: L, U, p, b
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! solve linear system : x A = b     where      A = p L U
      ! i.e. compute : y = b / U         ( <=> U' \ b )
      !         then : z = y / L         ( <=> L' \ y )
      !     and last : x = z P'

      ! handles only the DENSE case (LAPACK)
      ! handles mixed data type

      ! p is a integer permutation vector (from 2.6.0)

      integer :: i, n, ldb, nrhs
      integer :: status
      character(len=10) :: char_case
      type(mfArray) :: tmp

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( L, U, p, b )

      ! verifications...
      if( L%data_type == MF_DT_EMPTY .or.                               &
          U%data_type == MF_DT_EMPTY .or.                               &
          p%data_type == MF_DT_EMPTY .or.                               &
          b%data_type == MF_DT_EMPTY      ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'L', 'U', 'p' or 'b' is not allocated!" )
         go to 99
      end if

      n = L%shape(1)
      if( L%shape(2) /= n ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'L' must be square!" )
         go to 99
      end if

      if( any(U%shape /= [n,n]) ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'U' must have same size than 'L'!" )
         go to 99
      end if

      if( P%shape(1) /= n ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "permutation 'p' must have same size than 'L'!" )
         go to 99
      end if

      nrhs = b%shape(1)
      ldb = b%shape(2)

      if( ldb /= n ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'b' has a wrong size!" )
         go to 99
      end if

      ! prepare for mixte data type
      if( L%data_type == MF_DT_DBLE .and.                               &
          b%data_type == MF_DT_DBLE       ) then
         char_case = "real"
         call msAssign( out, .t. b )
      else if( L%data_type == MF_DT_CMPLX .and.                         &
               b%data_type == MF_DT_CMPLX       ) then
         char_case = "complex"
         call msAssign( out, .h. b )
      else if( L%data_type == MF_DT_CMPLX .and.                         &
               b%data_type == MF_DT_DBLE       ) then

         call msAssign( out, .h. mfComplex(b) )
         char_case = "complex"
      else if( L%data_type == MF_DT_DBLE .and.                          &
               b%data_type == MF_DT_CMPLX       ) then
         char_case = "complex2"
         call msAssign( out, .h. b )
      end if

      select case( char_case )
      case( "real" )
         ! solve U' y = out, overwriting out with y
         call dtrsm( 'left', 'upper', 'transpose', 'non-unit', n, nrhs, &
                     1.0d0, U%double(1,1), n, out%double(1,1), ldb )
         ! solve L' z = out, overwriting out with z
         call dtrsm( 'left', 'lower', 'transpose', 'unit', n, nrhs,     &
                     1.0d0, L%double(1,1), n, out%double(1,1), ldb )
      case( "complex" )
         ! solve U' y = out, overwriting out with y
         call ztrsm( 'left', 'upper', 'conjg transpose', 'non-unit', n, nrhs, &
                     (1.0d0,0.0d0), U%cmplx(1,1), n, out%cmplx(1,1), ldb )
         ! solve L' z = out, overwriting out with z
         call ztrsm( 'left', 'lower', 'conjg transpose', 'unit', n, nrhs,     &
                     (1.0d0,0.0d0), L%cmplx(1,1), n, out%cmplx(1,1), ldb )
      case( "complex2" )
         call msAssign( tmp, mfComplex(L) )
         ! solve U' y = out, overwriting out with y
         call ztrsm( 'left', 'upper', 'conjg transpose', 'non-unit', n, nrhs, &
                     (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )
         ! solve L' z = out, overwriting out with z
         call msAssign( tmp, mfComplex(L) )
         call ztrsm( 'left', 'lower', 'conjg transpose', 'unit', n, nrhs,     &
                     (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )
         call msSilentRelease( tmp )
      end select

      ! apply row interchanges
      call msAssign( out, .t.mfRowPerm( out, mfInvPerm(P) ) )

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( b%units(i), L%units(i),                  &
                               out%units(i), status )
            call rational_sub( out%units(i), U%units(i),                &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfRDiv:) internal error: in processing physical units:"
               write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( L, U, p, b )

      call msAutoRelease( L, U, p, b )

#endif
   end function mfRDiv_four_mf
!_______________________________________________________________________
!
   function mfRDiv_mfMatFactor( b, factor ) result( out )

      type(mfMatFactor) :: factor
      type(mfArray)     :: b

      type(mfArray)     :: out
      !------ API end ------
#ifdef _DEVLP

      ! solve linear system of the form : A x = b
      !
      ! sparse matrix A is provided via its factors;
      ! mfArray b may have multiple columns and, for the SPD case
      ! (i.e. using CHOLMOD), may be sparse.

      ! handles only the SPARSE case (SuiteSparse)
      !
      ! 'factor' contains C structure, either from LU factorization
      ! (i.e., L, U, P, Q, R from UMFPack), either from Cholesky
      ! factorization (i.e., L and other from CHOLMOD)

      ! declarations for UMFPACK
      integer(kind=MF_ADDRESS) :: numeric ! LU handle
      integer :: sys
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      real(kind=MF_DOUBLE), pointer :: ptr_b(:) => null(),              &
                                       ptr_x(:) => null()
      real(kind=MF_DOUBLE), pointer :: ptr_bz(:) => null(),             &
                                       ptr_xz(:) => null()
      integer(kind=MF_UF_LONG), allocatable :: Wi(:)
      real(kind=MF_DOUBLE), allocatable :: W(:)
      complex(kind=MF_DOUBLE), allocatable :: ctmp(:,:)

      integer :: i, n, ldb, nrhs, bnz, xnz
      character(len=4) :: info_char
      integer(kind=MF_ADDRESS) :: c_addr, L_addr, out_addr
      integer :: status
      type(mfArray) :: btmp

   !------ end of declarations -- execution starts hereafter  ------

      ! verifications...
      if( .not. associated(factor%ptr_1) ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'factor' is not allocated!",              &
                            "(this factor must have been obtained via the 'mfLU' routine!" )
         go to 99
      end if

      if( b%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'b' is not allocated!" )
         go to 99
      end if

      n = factor%order
      nrhs = b%shape(1)
      ldb  = b%shape(2)

      if( ldb /= n ) then
         call PrintMessage( "mfRDiv", "E",                              &
                            "'b' has a wrong size!" )
         go to 99
      end if

      if( factor%data_type == MF_DT_SP_DBLE .and.                       &
          b%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_DBLE
         out%shape = [ nrhs, n ]
         allocate( out%double(nrhs,n) )

         select case( factor%package )
         case( 1 ) ! UMFPACK

            ! set UMFPACK-4 default parameters
            call umf4def_d(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! solve A'.x=b, without iterative refinement
            sys = 1
            numeric = factor%ptr_1

            allocate( Wi(n), W(n) )
            do i = 1, nrhs
               call umf4wsol_d( sys, out%double(i,:), b%double(i,:), numeric, &
                                control, infos, Wi, W )
               if( infos(1) < 0 ) then
                  write(info_char,"(I0)") nint(infos(1))
                  call PrintMessage( "mfRDiv", "E",                     &
                                     "in umf4wsol_d: infos(1) = " // info_char )
                  go to 99
               end if
            end do

         case( 2, 4 ) ! CHOLMOD

            ! here A is (at least) symmetric,
            ! hence: A.x=b or A'.x=b is the same problem
            ! solve L.D.L'.x=b, without iterative refinement
            c_addr = factor%ptr_1
            L_addr = factor%ptr_2
            do i = 1, nrhs
               call cholmod_solve_factor( n, c_addr, L_addr, b%double(i,:), &
                                          out%double(i,:) )
            end do
!!            else
!!call msPause('not sure that it is correct...')
!!               call cholmod_solve_factor_nRHS( n, c_addr, L_addr, nrhs, b%double(:,:), &
!!                                               out%double(:,:) )
!!            end if

         case default
            write(STDERR,*) "(MUESLI mfRDiv:) internal error: bad package number in factor"
            write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop

         end select

      else if( factor%data_type == MF_DT_SP_DBLE .and.                  &
               b%data_type == MF_DT_SP_DBLE ) then

         out%data_type = MF_DT_SP_DBLE

         select case( factor%package )
         case( 1 ) ! UMFPACK

!### TODO 2:
stop 'not yet available'

         case( 2, 4 ) ! CHOLMOD

            ! here A is (at least) symmetric,
            ! hence: A.x=b or A'.x=b is the same problem
            ! solve L.L'.x=b, without iterative refinement
            c_addr = factor%ptr_1
            L_addr = factor%ptr_2
            bnz = b%j(n+1) - 1
            call msAssign( btmp, .t. b )
            call cholmod_spsol_fact_nRHS_prep( n, c_addr, L_addr, nrhs, &
                                               bnz, btmp%j, btmp%i, btmp%a, &
                                               out_addr, xnz )

            call msSilentRelease( btmp )
            out%shape = [ n, nrhs ]
            allocate( out%j(nrhs+1) )

            allocate( out%i(xnz) )

            allocate( out%a(xnz) )

            ! now, can get sparse solution 'out' ...
            call cholmod_get_sol_nrhs( c_addr, out_addr, n, nrhs, xnz,  &
                                       out%j, out%i, out%a )

            ! All matrices returned by SuiteSparse are "row sorted"
            out%row_sorted = TRUE
            call msAssign( out, .t. out)

         case default
            write(STDERR,*) "(MUESLI mfRDiv:) internal error: bad package number in factor"
            write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop

         end select

      else if( factor%data_type == MF_DT_SP_CMPLX .and.                 &
               b%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_CMPLX
         out%shape = [ nrhs, n ]
         allocate( out%cmplx(nrhs,n) )

         select case( factor%package )
         case( 1 ) ! UMFPACK

            ! set UMFPACK-4 default parameters
            call umf4def_z(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! solve A'.x=b, without iterative refinement
            sys = 1
            numeric = factor%ptr_1

            allocate( ptr_x(n), ptr_xz(n) )
            allocate( ptr_b(n), ptr_bz(n) )

            ptr_bz(:) = 0.0d0
            allocate( Wi(n), W(4*n) )
            do i = 1, nrhs
               ptr_b(:) = b%double(i,:)
               call umf4wsol_z( sys, ptr_x, ptr_xz, ptr_b, ptr_bz,      &
                                numeric, control, infos, Wi, W )
               if( infos(1) < 0 ) then
                  write(info_char,"(I0)") nint(infos(1))
                  call PrintMessage( "mfRDiv", "E",                     &
                                     "in umf4wsol_z: infos(1) = " // info_char )
                  go to 99
               end if
               ! taking conjuguate
               out%cmplx(i,:) = cmplx(ptr_x(:),-ptr_xz(:),kind=MF_DOUBLE)
            end do

            deallocate( ptr_x, ptr_xz, ptr_b, ptr_bz )

         case( 2, 4 ) ! CHOLMOD

            ! here A is (at least) hermitian,
            ! hence: A.x=b or A'.x=b is the same problem
            ! solve L.D.L'.x=b, without iterative refinement
            c_addr = factor%ptr_1
            L_addr = factor%ptr_2
            allocate( ctmp(nrhs,n) )
            ctmp(:,:) = b%double(:,:)
            do i = 1, nrhs
               call cholmod_cmplx_solve_factor( n, c_addr, L_addr, ctmp(i,:), &
                                                out%cmplx(i,:) )
            end do
            ! taking conjuguate
            out%cmplx(:,:) = conjg( out%cmplx(:,:) )
!!            else
!!call msPause('not sure that it is correct...')
!!               call cholmod_cmplx_solve_factor_nRHS( n, c_addr, L_addr, nrhs, ctmp(:,:), &
!!                                                     out%cmplx(:,:) )
!!            end if

         case default
            write(STDERR,*) "(MUESLI mfRDiv:) internal error: bad package number in factor"
            write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop

         end select

      else if( factor%data_type == MF_DT_SP_CMPLX .and.                 &
               b%data_type == MF_DT_CMPLX ) then

         out%data_type = MF_DT_CMPLX
         out%shape = [ nrhs, n ]
         allocate( out%cmplx(nrhs,n) )

         select case( factor%package )
         case( 1 ) ! UMFPACK

            ! set UMFPACK-4 default parameters
            call umf4def_z(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! solve A'.x=b, without iterative refinement
            sys = 1
            numeric = factor%ptr_1

            allocate( ptr_x(n), ptr_xz(n) )
            allocate( ptr_b(n), ptr_bz(n) )

            allocate( Wi(n), W(4*n) )
            do i = 1, nrhs
               ptr_b(:)  =   real(b%cmplx(i,:))
               ptr_bz(:) = -aimag(b%cmplx(i,:))
               call umf4wsol_z( sys, ptr_x, ptr_xz, ptr_b, ptr_bz,      &
                                numeric, control, infos, Wi, W )
               if( infos(1) < 0 ) then
                  write(info_char,"(I0)") nint(infos(1))
                  call PrintMessage( "mfRDiv", "E",                     &
                                     "in umf4wsol_z: infos(1) = " // info_char )
                  go to 99
               end if
               out%cmplx(i,:) = cmplx(ptr_x(:),ptr_xz(:),kind=MF_DOUBLE)
            end do

            deallocate( ptr_x, ptr_xz, ptr_b, ptr_bz )

         case( 2, 4 ) ! CHOLMOD

            ! here A is (at least) hermitian,
            ! hence: A.x=b or A'.x=b is the same problem
            ! solve L.D.L'.x=b, without iterative refinement
            c_addr = factor%ptr_1
            L_addr = factor%ptr_2
            allocate( ctmp(nrhs,n) )
            ctmp(:,:) = conjg(b%cmplx(:,:))
            do i = 1, nrhs
               call cholmod_cmplx_solve_factor( n, c_addr, L_addr, ctmp(i,:), &
                                                out%cmplx(i,:) )
            end do
!!            else
!!call msPause('not sure that it is correct...')
!!               call cholmod_cmplx_solve_factor_nRHS( n, c_addr, L_addr, nrhs, b%cmplx(:,:), &
!!                                                     out%cmplx(:,:) )
!!            end if

         case default
            write(STDERR,*) "(MUESLI mfRDiv:) internal error: bad package number in factor"
            write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop

         end select

      else

         call PrintMessage( "mfRDiv", "E",                              &
                            "bad data type!" )
         go to 99

      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( b%units(i), factor%units(i),             &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfRDiv", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfRDiv:) internal error: in processing physical units:"
               write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msAutoRelease( b )

#endif
   end function mfRDiv_mfMatFactor
