! f90 include file

!_______________________________________________________________________
!
   function mfRcond( A ) result( out )

      type(mfArray), target :: A
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! usually (as for Matlab), RCond is valid only for square matrices.
      !
      ! in the current implementation, if A is a non-square matrix,
      ! the product A'*A (which is square) is computed first; then
      ! the square root of the RCond for this product is returned.

      integer :: n, lda, info, linfo
      integer, allocatable :: ipiv(:), iwork(:)
      character(len=1) :: lnorm
      type(mfArray) :: B_copy
      real(kind=MF_DOUBLE) :: lanorm, rcond
      real(kind=MF_DOUBLE), allocatable :: work(:)
      complex(kind=MF_DOUBLE), allocatable :: cwork(:)
      type(mfArray) :: tmp_norm

      ! declarations for UMFPACK
      integer(kind=MF_ADDRESS) :: numeric, symbolic ! LU handle
      integer :: m, nnz, nz_udiag, status
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      integer, allocatable :: ptr_Aj(:), ptr_Ai(:)
      real(kind=MF_DOUBLE), allocatable :: udiag(:), udiagi(:)
      complex(kind=MF_DOUBLE), allocatable :: udiagz(:)
      real(kind=MF_DOUBLE), allocatable :: Ar(:), Az(:)
      character(len=4) :: info_char
      integer :: mf_message_level_save

      ! declarations for CHOLMOD
      integer(kind=MF_ADDRESS) :: c_addr, LL_addr, S_addr
      integer :: lnz, nnz2

      type(mfArray), pointer :: B => null()

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfRcond", "E",                             &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfRcond", "E",                             &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfRcond", "E",                             &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "mfRcond", "E",                             &
                            "mfArray 'a' should be a matrix!" )
         go to 99
      end if

      ! square matrix ?
      if( A%shape(1) == A%shape(2) ) then
         B => A
      else if( A%shape(1) <= A%shape(2) ) then
         allocate(B) ! no_mem_trace !
         if( mfIsReal(A) ) then
            if( mfIsSparse(A) ) then
               call msAssign( B, mfMul(A,.t.A) )
            else
               call msAssign( B, mfMul(A,A,transp=2) )
            end if
         else ! complex
            if( mfIsSparse(A) ) then
               call msAssign( B, mfMul(A,.h.A) )
            else
               call msAssign( B, mfMul(A,A,transp=2) )
            end if
         end if
      else
         allocate(B) ! no_mem_trace !
         if( mfIsReal(A) ) then
            if( mfIsSparse(A) ) then
               call msAssign( B, mfMul(.t.A,A) )
            else
               call msAssign( B, mfMul(A,A,transp=1) )
            end if
         else ! complex
            if( mfIsSparse(A) ) then
               call msAssign( B, mfMul(.h.A,A) )
            else
               call msAssign( B, mfMul(A,A,transp=1) )
            end if
         end if
      end if

      ! hereafter, we must work with B

      if( B%data_type == MF_DT_DBLE ) then

         ! making copies because LAPACK overwrites B
         mf_message_level_save = mf_message_level
         mf_message_level = 0
         B_copy = B
         mf_message_level = mf_message_level_save

         ! General system -- real square matrix : DGETRF
         n = B_copy%shape(1)
         lda = B_copy%shape(1)
         allocate( ipiv(n) )

         call dgetrf( n, n, B_copy%double(1,1), lda, ipiv(1), info )
         if( info /= 0 ) then
            ! solution is not ok : condition number is infinite
            out = 0.0d0
         else
            lnorm = "1" ! as in MATLAB
            call msAssign( tmp_norm, mfNorm( B, 1 ))
            lanorm = tmp_norm%double(1,1)
            call msSilentRelease( tmp_norm )
            allocate( work(4*n) )

            allocate( iwork(n) )

            call dgecon( lnorm, n, B_copy%double(1,1), lda, lanorm,     &
                         rcond, work(1), iwork(1), linfo )
            out = rcond
         end if

         call msSilentRelease( B_copy )

      else if( B%data_type == MF_DT_CMPLX ) then

         ! making copies because LAPACK overwrites B
         mf_message_level_save = mf_message_level
         mf_message_level = 0
         B_copy = B
         mf_message_level = mf_message_level_save

         ! General system -- complex square matrix : ZGETRF
         n = B_copy%shape(1)
         lda = B_copy%shape(1)
         allocate( ipiv(n) )

         call zgetrf( n, n, B_copy%cmplx(1,1), lda, ipiv(1), info )
         if( info /= 0 ) then
            ! solution is not ok : condition number is infinite
            out = 0.0d0
         else
            lnorm = "1" ! as in MATLAB
            call msAssign( tmp_norm, mfNorm( B, 1 ))
            lanorm = tmp_norm%double(1,1)
            call msSilentRelease( tmp_norm )
            allocate( cwork(2*n) )

            allocate( work(2*n) )

            call zgecon( lnorm, n, B_copy%cmplx(1,1), lda, lanorm,      &
                         rcond, cwork(1), work(1), linfo )
            out = rcond
         end if

         call msSilentRelease( B_copy )

      else if( B%data_type == MF_DT_SP_DBLE ) then

         ! LU or LDLT factorization in order to get the diagonal.
         ! -> ESTIMATION of RCond !

         m = B%shape(1)
         n = B%shape(2)

         nnz = B%j(n+1) - 1

         if( m == n .and. mfIsSymm(B) ) then
            goto 11
         else
            goto 21
         end if

 11      continue

            ! L*D*L' factorization, but get only the diagonal
            call msAssign( B_copy, mfTriu(B) )
            nnz2 = mfNnz(B_copy)
            call cholmod_ldlt_prep( n, nnz2, B_copy%j, B_copy%i, B_copy%a, &
                                    c_addr, LL_addr, S_addr, lnz )

            call msSilentRelease( B_copy )

            if( lnz == 0 ) then
            ! in the case where cholmod_ldlt_prep failed, it has cleaned
            ! memory.
               call PrintMessage( "mfRCond", "I",                       &
                                  "CHOLMOD (LDLT): failed",             &
                                  "using the LU decomposition instead", &
                                  "(as for unsymmetric matrices)" )
               go to 21
            end if

            allocate( udiag(n) )

            ! getting the diagonal (this routine will properly clean
            ! memory)
            call cholmod_get_diag( c_addr, LL_addr, S_addr, n, lnz, udiag )

            out = minval( abs(udiag) ) / maxval( abs(udiag) )

            goto 31

 21      continue

            ! sparse matrices must be row-sorted !
            if( B%row_sorted /= TRUE ) then
               call msRowSort(B)
            end if

            ! set UMFPACK-4 default parameters
            call umf4def_d(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! convert from 1-based to 0-based
            allocate( ptr_Aj(n+1) )

            ptr_Aj(1:n+1) = B%j(1:n+1) - 1
            allocate( ptr_Ai(nnz) )

            ptr_Ai(1:nnz) = B%i(1:nnz) - 1
            ! pre-order and symbolic analysis
            call umf4sym_d( m, n, ptr_Aj, ptr_Ai, B%a, symbolic,        &
                            control, infos )
            ! check umf4sym error condition
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfRCond", "E",                       &
                                  "in umf4sym_d: infos(1) = " // info_char )
               go to 99
            end if

            ! numeric factorization
            call umf4num_d( n, ptr_Aj, ptr_Ai, B%a, symbolic, numeric,  &
                            control, infos )
            ! check umf4num error condition
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfRCond", "E",                       &
                                  "in umf4num_d: infos(1) = " // info_char )
               go to 99
            end if

            ! free the symbolic analysis
            call umf4fsym_d(symbolic)

            ! get the NNZ of U_diag
            call umf4nzudiag_d( nz_udiag, numeric, status )

            ! check umf4nzudiag error condition
            if( status /= 0 ) then
               call PrintMessage( "mfRCond", "E",                       &
                                  "in umf4nzudiag_d!" )
               go to 99
            end if

            if( nz_udiag < min(m,n) ) then
               out = 0.0d0
            else
               allocate( udiag(nz_udiag) )

               call umf4getudiag_d( udiag(1), numeric, status )

               ! check umf4getudiag error condition
               if( status /= 0 ) then
                  call PrintMessage( "mfRCond", "E",                    &
                                     "in umf4getudiag_d!" )
                  go to 99
               end if

               out = minval( abs(udiag) ) / maxval( abs(udiag) )
            end if

            ! free the numeric factorization
            call umf4fnum_d(numeric)

 31      continue

      else if( B%data_type == MF_DT_SP_CMPLX ) then

         ! LU factorization in order to get the diagonal.
         ! -> ESTIMATION of RCond !

         m = B%shape(1)
         n = B%shape(2)

         nnz = B%j(n+1) - 1

         ! sparse matrices must be row-sorted !
         if( B%row_sorted /= TRUE ) then
            call msRowSort(B)
         end if

         ! set UMFPACK-4 default parameters
         call umf4def_d(control)
         ! print control parameters.
         ! Set control(1) to 1 to print error messages only
         control(1) = 1

         ! convert from 1-based to 0-based
         allocate( ptr_Aj(n+1) )

         ptr_Aj(1:n+1) = B%j(1:n+1) - 1
         allocate( ptr_Ai(nnz) )

         ptr_Ai(1:nnz) = B%i(1:nnz) - 1

         allocate( Ar(nnz) )

         Ar(:) = real(B%z(1:nnz))
         allocate( Az(nnz) )

         Az(:) = aimag(B%z(1:nnz))
         ! pre-order and symbolic analysis
         call umf4sym_z( m, n, ptr_Aj(1), ptr_Ai(1), Ar(1), Az(1), symbolic, &
                         control, infos )
         ! check umf4sym error condition
         if( infos(1) < 0 ) then
            write(info_char,"(I0)") nint(infos(1))
            call PrintMessage( "mfRCond", "E",                          &
                               "in umf4sym_z: infos(1) = " // info_char )
            go to 99
         end if

         ! numeric factorization
         call umf4num_z( n, ptr_Aj(1), ptr_Ai(1), Ar(1), Az(1), symbolic, numeric, &
                         control, infos )
         ! check umf4num error condition
         if( infos(1) < 0 ) then
            write(info_char,"(I0)") nint(infos(1))
            call PrintMessage( "mfRCond", "E",                          &
                               "in umf4num_z: infos(1) = " // info_char )
            go to 99
         end if

         ! free the symbolic analysis
         call umf4fsym_z(symbolic)

         ! get the NNZ of U_diag
         call umf4nzudiag_z( nz_udiag, numeric, status )

         ! check umf4nzudiag error condition
         if( status /= 0 ) then
            call PrintMessage( "mfRCond", "E",                          &
                               "in umf4nzudiag_d!" )
            go to 99
         end if

         if( nz_udiag < min(m,n) ) then
            out = 0.0d0
         else
            allocate( udiag(nz_udiag) )

            allocate( udiagi(nz_udiag) )

            call umf4getudiag_z( udiag(1), udiagi(1), numeric, status )

            ! check umf4getudiag error condition
            if( status /= 0 ) then
               call PrintMessage( "mfRCond", "E",                       &
                                  "in umf4getudiag_z!" )
               go to 99
            end if

            ! free the numeric factorization
            call umf4fnum_z(numeric)

            allocate( udiagz(nz_udiag) )

            udiagz(:) = cmplx( udiag, udiagi,kind=MF_DOUBLE )
            out = minval( abs(udiagz) ) / maxval( abs(udiagz) )
         end if

      else

         call PrintMessage( "mfRcond", "E",                             &
                            "bad data type for 'A'!" )
         go to 99

      end if

      if( A%shape(1) /= A%shape(2) ) then
         call msRelease( B )
         deallocate( B ) ! no_mem_trace !
         call msAssign( out, mfSqrt(out) )
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfRcond
