! f90 include file

!_______________________________________________________________________
!
   function mfQR( A ) result( R )

      type(mfArray), target :: A
      type(mfArray) :: R
      !------ API end ------
#ifdef _DEVLP

      ! Q-less factorization of A.
      ! The returned matrix R is of size: n x n

      integer :: i, m, n, lda, info, lwork, nb
      real(kind=MF_DOUBLE), allocatable :: tau(:), work(:)
      complex(kind=MF_DOUBLE), allocatable :: ctau(:), cwork(:)
      type(mfArray), pointer :: A_copy
      character(len=3) :: info_char
      logical :: A_copy_is_allocated

      ! SPQR variables
      integer(kind=MF_ADDRESS) :: c_addr, R_addr
      integer :: status, nnz, rnz
      integer(kind=MF_LONG_INT) :: total_size, n1
      character(len=19) :: n1_char, n2_char

      integer, external :: ilaenv

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfQR", "E",                                &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)

      if( mfIsSparse(A) ) then
         nnz = A%j(n+1) - 1
         if( A%data_type == MF_DT_SP_DBLE ) then
            if( nnz /= 0 ) then
               if( .not. all(mfIsFinite(mf(A%a))) ) then
                  call PrintMessage( "mfQR", "E",                       &
                                    "mfArray 'A' must not contain Inf or NaN values!" )
                  go to 99
               end if
            end if
         else
            call PrintMessage( "mfQR", "E",                             &
                               "sparse complex case not yet handled!" )
            go to 99
         end if
      else
         if( .not. all(mfIsFinite(A)) ) then
            call PrintMessage( "mfQR", "E",                             &
                               "mfArray 'A' must not contain Inf or NaN values!" )
            go to 99
         end if
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfQR", "E",                                &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfQR", "E",                                &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_DBLE ) then

         ! making copies because LAPACK overwrites A
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if

         lda = m
         allocate( tau(min(m,n)) )

         ! optimal blocksize for DGEQRF (from ILAENV)
         nb = 32
         lwork = n*nb
         allocate( work(lwork) )

         ! General system -- real general matrix : DGEQRF
         call dgeqrf( m, n, A_copy%double(1,1), lda, tau(1), work(1), lwork, info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "mfQR", "W",                             &
                               "in QR-factorizing matrix A by LAPACK:", &
                               "info = " // info_char,                  &
                               "(as a side effect, R will be empty)" )
            call msSilentRelease( R )
         else
            if( m >= n ) then
               call msAssign( R, mfGet( A_copy, [ (i, i = 1, n) ], MF_COLON ))
               call msAssign( R, mfTriu( R ))
            else
               call msAssign( R, mfTriu( A_copy ))
            end if
         end if

         if( A_copy_is_allocated ) then
            call msSilentRelease( A_copy )
            deallocate( A_copy ) ! no_mem_trace !
         end if

      else if( A%data_type == MF_DT_CMPLX ) then

         ! making copies because LAPACK overwrites A
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if

         lda = m
         allocate( ctau(min(m,n)) )

         ! optimal blocksize for ZGEQRF (from ILAENV)
         nb = 32
         lwork = n*nb
         allocate( cwork(lwork) )

         ! General system -- complex general matrix : ZGEQRF
         call zgeqrf( m, n, A_copy%cmplx(1,1), lda, ctau(1), cwork(1), lwork, info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "mfQR", "W",                             &
                               "in QR-factorizing matrix A by LAPACK:", &
                               "info = " // info_char,                  &
                               "(as a side effect, R will be empty)" )
            call msSilentRelease( R )
         else
            if( m >= n ) then
               call msAssign( R, mfGet( A_copy, [ (i, i = 1, n) ], MF_COLON ))
               call msAssign( R, mfTriu( R ))
            else
               call msAssign( R, mfTriu( A_copy ))
            end if
         end if

         if( A_copy_is_allocated ) then
            call msSilentRelease( A_copy )
            deallocate( A_copy ) ! no_mem_trace !
         end if

      else if( a%data_type == MF_DT_SP_DBLE ) then

         nnz = A%j(n+1) - 1

         if( nnz == 0 ) then

            ! quick return if it is an all-zero matrix
            call msAssign( R, mfSpAlloc(n,n) )

         else

            ! calling the 'SuiteSparseQR' C++ routine (SuiteSparse)

            ! sparse matrices must be row-sorted !
            if( A%row_sorted /= TRUE ) then
               call msRowSort(A)
            end if

            ! do a Q-less factorization of A and returns, among other
            ! addresses, the size of the R factor.
            call spqr_q_less_prep( m, n, nnz, A%j, A%i, A%a,            &
                                   c_addr, R_addr, rnz, status )
            ! check for an error
            if( status /= 0 ) then
               call PrintMessage( "mfQR", "E",                          &
                                  "spqr_q_less_prep failed!" )
               ! no more information: SPQR should never fail!
               ! (except out-of-memory)
               go to 99
            end if

            ! check for validity: we must have nnz(R) >= 0
            if( rnz < 0 ) then
               write(n1_char,*) rnz
               write(STDERR,*) "(MUESLI mfQR:) internal error:"
               write(STDERR,*) "               spqr_q_less_prep failed to return a valid R nnz!"
               write(STDERR,*) "               nnz(R) >= 0 must hold, but: nnz(R) = ", trim(adjustl(n1_char))
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               stop
            end if

            ! check for validity: we must have nnz(R) <= n*n
            !   (i.e. less than the max number of elements)
            n1 = n
            total_size = n1*n1
            if( rnz > total_size ) then ! rnz > n*n
               write(n1_char,*) rnz
               write(n2_char,*) total_size
               write(STDERR,*) "(MUESLI mfQR:) internal error:"
               write(STDERR,*) "               spqr_q_less_prep failed to return a valid R nnz!"
               write(STDERR,*) "               nnz(R) <= n*n must hold, but: nnz(R) = ", trim(adjustl(n1_char))
               write(STDERR,*) "               whereas n*n = ", trim(adjustl(n2_char))
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               stop
            end if

            ! allocating space for minimum square (n,n) matrix R
            call msAssign( R, mfSpAlloc(n,n,rnz) )

            ! get the R factor.
            ! (freeing of unused obj are also done)
            call spqr_get_R( c_addr, R_addr, n, rnz,                    &
                             R%j , R%i , R%a  )

            ! All matrices returned by SuiteSparse are "row sorted"
            R%row_sorted = TRUE

         end if

      else

         call PrintMessage( "mfQR", "E",                                &
                            "currently, complex sparse matrices are not handled!" )
         go to 99

      end if

      R%prop%triu = TRUE
      R%prop%symm = FALSE

      if( mf_phys_units ) then
         R%units(:) = A%units(:)
      end if

      R%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfQR
!_______________________________________________________________________
!
   subroutine msQR( out, A, tol, fullQ )

      type(mfArray),        target   :: A
      real(kind=MF_DOUBLE), optional :: tol
      logical,              optional :: fullQ
      type(mf_Out)                   :: out
      !------ API end ------
#ifdef _DEVLP

      ! A : dense or sparse matrix (driver routine)
      !
      ! if called with two to four (out) mfArrays, and an mfArray A
      ! (dense or sparse):
      !   returns Q, R factors and, optionally, the permutation P and
      !   the numerical RANK;
      !   in the case where the RANK is requested (RRQR), tolerance
      !   can be specified (machine EPS is the default).
      !
      ! if called with one (out) mfMatFactor, one to three (out)
      ! mfArray, and a sparse matrix A:
      !   idem as the previous case except that the Q factor is under
      !   the Householder form.
      !
      ! Not documented (for the moment): [used in dae_cic_dense]
      ! for a dense matrix, the 'economic' form is always used, i.e.
      ! a square R matrix (upper part) and the N first columns of Q.
      ! The use of fullQ=.true. allows the user to get the full Q and R
      ! factors.
      ! Currently only available for the specific call:
      !         call msQR( mfOut(Q,R,p,rank), A )

      ! from 2.6.0, the permutation is returned as a PERM_VEC mfArray

      type(mfArray), pointer :: Q, R, p, RANK
      type(mfMatFactor), pointer :: Qhouse
      integer :: n, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! valid cases
      !   two, three or four output mfArrays
      if( out%factor_n == 0 ) then
         if( out%n < 2 .or. 4 < out%n  ) then
            call PrintMessage( "msQR", "E",                             &
                               "bad argument call!",                   &
                               "syntax is: call msQR ( mfOut(Q,R[,p,RANK]), A[,tol] )", &
                               "           [Q: mfArray]" )
            go to 99
         end if
      else  if( out%factor_n == 1 ) then
         if( out%n < 1 .or. 3 < out%n  ) then
            call PrintMessage( "msQR", "E",                             &
                               "bad argument call!",                   &
                               "syntax is: call msQR ( mfOut(Qhouse,R[,p,RANK]), A[,tol] )", &
                               "           [Qhouse: mfMatFactor]" )
            go to 99
         end if
      else
         call PrintMessage( "msQR", "E",                                &
                            "bad argument call!",                      &
                            "  (only the first out arg may be of type mfMatFactor)" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msQR", "E",                                &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msQR", "E",                                &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         n = A%shape(2)
         nnz = A%j(n+1) - 1
         if( A%data_type == MF_DT_SP_DBLE ) then
            if( nnz /= 0 ) then
               if( .not. all(mfIsFinite(mf(A%a))) ) then
                  call PrintMessage( "msQR", "E",                       &
                                    "mfArray 'A' must not contain Inf or NaN values!" )
                  go to 99
               end if
            end if
         else
            call PrintMessage( "msQR", "E",                             &
                               "sparse complex case not yet handled!" )
            go to 99
         end if
      else
         if( .not. all(mfIsFinite(A)) ) then
            call PrintMessage( "msQR", "E",                             &
                               "mfArray 'A' must not contain Inf or NaN values!" )
            go to 99
         end if
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msQR", "E",                                &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msQR", "E",                                &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( present(tol) ) then
         if( out%n /= 4 ) then
            call PrintMessage( "msQR", "W",                             &
                               "'tol' is used only when RANK is requested!" )
         end if
         if( tol <= 0.0d0 ) then
            call PrintMessage( "msQR", "E",                             &
                               "'tol' must be strictly positive!" )
            go to 99
         end if
      end if

      if( out%factor_n == 0 ) then ! Q in standard matrix form

         Q => out%ptr1
         R => out%ptr2
         call msSilentRelease( Q, R )
         if( out%n == 2 ) then
            call msQR_mfArray( A, Q, R )
         else if( out%n == 3 ) then
            p => out%ptr3
            call msSilentRelease( p )
            call msQR_mfArray( A, Q, R, p )
         else ! out%n == 4
            p => out%ptr3
            call msSilentRelease( p )
            RANK => out%ptr4
            call msSilentRelease( RANK )
            if( present(tol) ) then
               call msQR_mfArray( A, Q, R, p, RANK, tol )
            else
if( present(fullQ) ) then
   if( fullQ ) then
      call msQR_mfArray( A, Q, R, p, RANK, full=fullQ )
   else
      call msQR_mfArray( A, Q, R, p, RANK )
   end if
else
               call msQR_mfArray( A, Q, R, p, RANK )
end if
            end if
         end if

      else                         ! Q in Householder form

         ! A must be sparse
         if( .not. mfIsSparse(A) ) then
            call PrintMessage( "msQR", "E",                             &
                               "currently, when using an mfMatFactor,", &
                               "the mfArray 'A' must be sparse!" )
            go to 99
         end if

         Qhouse => out%factor_ptr1
         call msFreeMatFactor( Qhouse )
         R => out%ptr1
         call msSilentRelease( R )
         if( out%n == 1 ) then
            call msQR_mfMatFactor( A, Qhouse, R )
         else if( out%n == 2 ) then
            p => out%ptr2
            call msSilentRelease( p )
            call msQR_mfMatFactor( A, Qhouse, R, p )
         else ! out%n == 3
            p => out%ptr2
            call msSilentRelease( p )
            RANK => out%ptr3
            call msSilentRelease( RANK )
            if( present(tol) ) then
               call msQR_mfMatFactor( A, Qhouse, R, p, RANK, tol )
            else
               call msQR_mfMatFactor( A, Qhouse, R, p, RANK )
            end if
         end if

      end if

      R%prop%triu = TRUE

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msQR
!_______________________________________________________________________
!
   subroutine msQR_mfArray( A, Q, R, p, RANK, tol, full )

      type(mfArray),        target                :: A
      type(mfArray)                               :: Q, R
      type(mfArray),        intent(out), optional :: p, RANK
      real(kind=MF_DOUBLE), intent(in),  optional :: tol
      logical,              intent(in),  optional :: full
      !------ API end ------
#ifdef _DEVLP

      ! this routine is only for internal use: an ordinary user
      ! should always call 'msQR' which makes many tests on arguments.

      ! if m > n, always returns the economy-size Q and R.

      ! when computing rank, the default tolerance is consistent
      ! between the dense and the sparse versions:
      !  20*(m+n)*eps*sqrt(max(diag(A'*A)))
      ! (found in the MATLAB help of SPQR)

      ! the permutation 'p' is always returned as a vector (from 2.6.0)

      integer :: i, j, m, n, lda, info, lwork, nb, k
      real(kind=MF_DOUBLE), allocatable :: tau(:), work(:)
      complex(kind=MF_DOUBLE), allocatable :: ctau(:), cwork(:)
      type(mfArray), pointer :: A_copy
      character(len=3) :: info_char
      logical :: A_copy_is_allocated
      ! ACM_782 routine (DGEQPX)
      integer :: job, ldc, irank
      real(kind=MF_DOUBLE) :: ircond, orcond, svlues(4)
      type(mfArray) :: C
      real(kind=MF_DOUBLE) :: etol, max_diag_spqr, diag_spqr

      ! SPQR variables
      integer(kind=MF_ADDRESS) :: c_addr, Q_addr, R_addr, p_addr
      integer :: econ, A_rank
      integer :: ordering, status, nnz, qnz, rnz

      integer(kind=MF_LONG_INT) :: total_size, n1, n2
      character(len=19) :: n1_char, n2_char

      integer, external :: ilaenv
      real(kind=MF_DOUBLE), external :: ddot

      logical :: fullQ
      integer :: imax_R, jmax_Q

   !------ end of declarations -- execution starts hereafter  ------

      if( present(full) ) then
         if( full ) then
            fullQ = .true.
         else
            fullQ = .false.
         end if
      else
         fullQ = .false.
      end if

      call msInitArgs( A )

      m = A%shape(1)
      n = A%shape(2)

      if( fullQ ) then

         ! full size
         Q%shape = [ m, m ]
         R%shape = [ m, n ]

      else

         ! economy size
         Q%shape = [ m, min(m,n) ]
         R%shape = [ min(m,n), n ]

      end if

      if( mfIsSparse(A) ) then

         nnz = A%j(n+1) - 1

         if( nnz == 0 ) then

            ! quick return if it is an all-zero matrix
            if( m > n ) then
               ! least-square problem: always return 'economy size'
               call msAssign( Q, mfSpEye(m,n) )
               call msAssign( R, mfSpAlloc(n,n) )
            else
               call msAssign( Q, mfSpEye(m,m) )
               call msAssign( R, mfSpAlloc(m,n) )
            end if
            if( present(p) ) then ! QR with column pivoting
               call msAssign( p, mfPerm([(i,i=1,n)]) )
            end if
            if( present(RANK) ) then
               RANK = 0.0d0
            end if

         else

            ! calling the 'SuiteSparseQR' C++ routine (SuiteSparse)

            ! sparse matrices must be row-sorted !
            if( A%row_sorted /= TRUE ) then
               call msRowSort(A)
            end if

            econ = min(m,n)

            if( present(p) ) then ! QR with column pivoting
               ordering = MF_CHOLMOD_AMD !
            else ! simpler QR (no column pivoting)
               ordering = MF_CHOLMOD_NATURAL ! no ordering (identity)
            end if

            if( present(tol) ) then
               etol = tol
            else
               max_diag_spqr = max_diag_At_A(n,A%a,A%i,A%j)
               etol = 20*(m+n)*MF_EPS*sqrt(max_diag_spqr)
            end if

            if( mfIsReal(A) ) then

               ! do a QR factorization of A and returns, among other
               ! addresses, the size of the factors Q and R.
               call spqr_qr_prep( m, n, nnz, A%j, A%i, A%a,             &
                                  ordering, etol, econ,                 &
                                  c_addr, Q_addr, qnz, R_addr, rnz, p_addr, &
                                  A_rank, status )

               ! check for an error
               if( status /= 0 ) then
                  call PrintMessage( "msQR", "E",                       &
                                     "spqr_qr_prep failed!" )
                  go to 99
               end if

               ! check for validity: we must have nnz(Q) >= min(m,n)
               !   (i.e. at least one element in each col)
               if( qnz < min(m,n) ) then
                  write(n1_char,*) qnz
                  write(n2_char,*) min(m,n)
                  write(STDERR,*) "(MUESLI msQR:) internal error:"
                  write(STDERR,*) "               spqr_qr_prep failed to return a valid Q nnz!"
                  write(STDERR,*) "               nnz(Q) >= min(m,n) must hold, but: nnz(Q) = ", trim(adjustl(n1_char))
                  write(STDERR,*) "               whereas min(m,n) = ", trim(adjustl(n2_char))
                  mf_message_displayed = .true.
                  call muesli_trace( pause ="yes" )
                  stop
               end if

               ! check for validity: we must have nnz(Q) <= m*min(m,n)
               !   (i.e. less than the max number of elements)
               n1 = m
               n2 = min(m,n)
               total_size = n1*n2
               if( qnz > total_size ) then ! qnz > m*min(m,n)
                  write(n1_char,*) qnz
                  write(n2_char,*) total_size
                  write(STDERR,*) "(MUESLI msQR:) internal error:"
                  write(STDERR,*) "               spqr_qr_prep failed to return a valid Q nnz!"
                  write(STDERR,*) "               nnz(Q) <= m*min(m,n) must hold, but: nnz(Q) = ", trim(adjustl(n1_char))
                  write(STDERR,*) "               whereas m*min(m,n) = ", trim(adjustl(n2_char))
                  mf_message_displayed = .true.
                  call muesli_trace( pause ="yes" )
                  stop
               end if

               ! allocating space for matrix Q
               call msAssign( Q, mfSpAlloc(Q%shape(1),Q%shape(2),qnz) )

               ! check for validity: we must have nnz(R) >= 0
               if( rnz < 0 ) then
                  write(n1_char,*) rnz
                  write(STDERR,*) "(MUESLI msQR:) internal error:"
                  write(STDERR,*) "               spqr_q_less_prep failed to return a valid R nnz!"
                  write(STDERR,*) "               nnz(R) >= 0 must hold, but: nnz(R) = ", trim(adjustl(n1_char))
                  mf_message_displayed = .true.
                  call muesli_trace( pause ="yes" )
                  stop
               end if

               ! check for validity: we must have nnz(R) <= min(m,n)*n
               !   (i.e. less than the max number of elements)
               n1 = min(m,n)
               n2 = n
               total_size = n1*n2
               if( rnz > total_size ) then ! rnz > min(m,n)*n
                  write(n1_char,*) rnz
                  write(n2_char,*) total_size
                  write(STDERR,*) "(MUESLI msQR:) internal error:"
                  write(STDERR,*) "               spqr_qr_prep failed to return a valid R nnz!"
                  write(STDERR,*) "               nnz(R) <= min(m,n)*n must hold, but: nnz(R) = ", trim(adjustl(n1_char))
                  write(STDERR,*) "               whereas min(m,n)*n = ", trim(adjustl(n2_char))
                  mf_message_displayed = .true.
                  call muesli_trace( pause ="yes" )
                  stop
               end if

               ! allocating space for matrix R (rnz == 0 should be ok,
               ! but it must have at least one element for the R%a(1)
               ! below, which will be passed as argument to spqr_get_Q_R_p)
               if( rnz == 0 ) rnz = 1
               call msAssign( R, mfSpAlloc(R%shape(1),R%shape(2),rnz) )

               if( present(p) ) then ! QR with column pivoting
                  p%data_type = MF_DT_PERM_VEC
                  p%shape = [ n, 1 ]
                  allocate( p%i(n) )

                  ! get the factors Q,R and permutation p.
                  ! (freeing of unused obj are also done)
                  call spqr_get_Q_R_p( c_addr, Q_addr, min(m,n), qnz,   &
                                               R_addr, n, rnz,          &
                                               p_addr, n,               &
                                       Q%j, Q%i, Q%a,                   &
                                       R%j, R%i, R%a, p%i )

               else ! simpler QR (no column pivoting)

                  ! get the factors Q and R. No need to get p.
                  ! (freeing of unused obj are also done)
                  call spqr_get_Q_R( c_addr, Q_addr, min(m,n), qnz,     &
                                             R_addr, n, rnz,            &
                                     Q%j, Q%i, Q%a, R%j, R%i, R%a )

               end if

            else ! A Complex

               call PrintMessage( "msQR", "E",                          &
                                  "currently, complex sparse matrices are not handled!" )
               go to 99

            end if

            if( present(RANK) ) then
               RANK = dble(A_rank)
            end if

            ! All matrices returned by SuiteSparse are "row sorted"
            Q%row_sorted = TRUE
            R%row_sorted = TRUE

         end if

      else ! A is dense

         ! making copies because LAPACK (or ACM 782) overwrites A
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if

         if( present(p) ) then ! QR with column pivoting

            p%data_type = MF_DT_PERM_VEC
            p%shape = [ n, 1 ]
            allocate( p%i(n) )

            if( A%data_type == MF_DT_DBLE ) then

               Q%data_type = MF_DT_DBLE
               allocate( Q%double( Q%shape(1), Q%shape(2) ) )

               R%data_type = MF_DT_DBLE
               allocate( R%double( R%shape(1), R%shape(2) ) )

               lda = m
               if( present(RANK) ) then ! Rank Revealing (using ACM 782)
                  job = 3 ! in order to get Q
                  ! make initial C : identity matrix
                  call msAssign( C, mfEye(m,m) )
                  k = m
                  ldc = m
                  if( present(tol) ) then
                     etol = tol
                  else
                     max_diag_spqr = 0.0d0
                     do j = 1, n
                        diag_spqr = ddot(m, A%double(:,j), 1,           &
                                            A%double(:,j), 1 )
                        if( diag_spqr > max_diag_spqr ) then
                           max_diag_spqr = diag_spqr
                        end if
                     end do
                     etol = 20*(m+n)*MF_EPS*sqrt(max_diag_spqr)
                  end if
                  ! threshold for rank computing
                  ircond = etol
                  nb = 32
                  lwork = 2*min(m,n) + nb**2 + nb*max(k,n)
                  allocate( work(lwork) )

                  ! ACM 782 routine
                  call dgeqpx( job, m, n, k, A_copy%double(1,1), lda,   &
                               C%double(1,1), ldc, p%i(1), ircond,      &
                               orcond, irank, svlues(1), work(1), lwork, &
                               info )
                  if( info /= 0 ) then
                     ! solution is not ok -- light warning
                     write(info_char,"(I0)") info
                     call PrintMessage( "msQR", "E",                    &
                                        "in QR-factorizing matrix A by ACM-782:", &
                                        "info = " // info_char,         &
                                        "(as a side effect, Q, R and RANK will be empty)" )
                     call msSilentRelease( Q, R )
                     p%i(:) = [ (i,i=1,n) ]
                  else
                     jmax_Q = Q%shape(2)
                     imax_R = R%shape(1)
#if defined _INTEL_IFC & defined _64_BITS
! bug
                     do j = 1, jmax_Q
                        Q%double(:,j) = C%double(:,j)
                     end do
                     do j = 1, R%shape(2)
                        R%double(:,j) = A_copy%double(1:imax_R,j)
                     end do
#else
                     Q%double(:,:) = C%double(:,1:jmax_Q)
                     R%double(:,:) = A_copy%double(1:imax_R,:)
#endif
                     RANK = irank
                  end if
                  call msSilentRelease( C )

               else ! not Rank Revealing (using LAPACK)

                  allocate( tau(min(m,n)) )

                  ! optimal blocksize for DGEQP3 (from DGEQP3 and ILAENV)
                  nb = 32
                  lwork = 2*n+(n+1)*nb
                  allocate( work(lwork) )

                  ! General system -- real general matrix : DGEQP3
                  ! as p%i = JPVT is also an 'in' argument, by default we must
                  ! initialize it to zero (without any special consideration
                  ! about the 'free' columns)
                  p%i(:) = 0
                  call dgeqp3( m, n, A_copy%double(1,1), lda, p%i(1), tau(1), &
                               work(1), lwork, info )
                  if( info /= 0 ) then
                     ! solution is not ok -- light warning
                     write(info_char,"(I0)") info
                     call PrintMessage( "msQR", "E",                    &
                                        "in QR-factorizing matrix A by LAPACK:", &
                                        "info = " // info_char,         &
                                        "(as a side effect, Q and R will be empty)" )
                     call msSilentRelease( Q, R )
                     p%i(:) = [ (i,i=1,n) ]
                  else
                     if( m >= n ) then
                        call msAssign( R, mfGet( A_copy, [ (i, i = 1, n) ], MF_COLON ))
                        call msAssign( R, mftriu( R ) )
                        k = n
                        call dorgqr( m, n, k, A_copy%double(1,1), lda, tau(1), work(1), lwork, info )
                        if( info /= 0 ) then
                           ! solution is not ok -- light warning
                           write(info_char,"(I0)") info
                           call PrintMessage( "msQR", "E",              &
                                              "in building Q matrix A by LAPACK:", &
                                              "info = " // info_char,   &
                                              "(as a side effect, Q will be empty)" )
                           call msSilentRelease( Q )
                           p%i(:) = [ (i,i=1,n) ]
                        else
                           Q%data_type = MF_DT_DBLE
                           Q%double(:,:) = A_copy%double(:,:)
                        end if
                     else ! m < n
                        call msAssign( R, mftriu( A_copy ))
                        k = m
                        call dorgqr( m, m, k, A_copy%double(1,1), lda, tau(1), work(1), lwork, info )
                        if( info /= 0 ) then
                           ! solution is not ok -- light warning
                           write(info_char,"(I0)") info
                           call PrintMessage( "msQR", "E",              &
                                              "in building Q matrix A by LAPACK:", &
                                              "info = " // info_char,   &
                                              "(as a side effect, Q will be empty)" )
                           call msSilentRelease( Q )
                           p%i(:) = [ (i,i=1,n) ]
                        else
                           call msAssign( Q, mfGet( A_copy, MF_COLON, [ (i, i = 1, m) ] ))
                        end if
                     end if
                  end if
               end if

            else if( A%data_type == MF_DT_CMPLX ) then

               allocate( Q%cmplx( Q%shape(1), Q%shape(2) ) )

               allocate( R%cmplx( R%shape(1), R%shape(2) ) )

               lda = m
               allocate( ctau(min(m,n)) )

               allocate( work(2*n) )

               ! optimal blocksize for ZGEQP3 (from ZGEQP3 and ILAENV)
               nb = 32
               lwork = (n+1)*nb
               allocate( cwork(lwork) )

               ! General system -- complex general matrix : ZGEQP3
               ! as p%i = JPVT is also an 'in' argument, by default we must
               ! initialize it to zero (without any special consideration
               ! about the 'free' columns)
               p%i(:) = 0
               call zgeqp3( m, n, A_copy%cmplx(1,1), lda, p%i(1), ctau(1), &
                            cwork(1), lwork, work(1), info )
               if( info /= 0 ) then
                  ! solution is not ok -- light warning
                  write(info_char,"(I0)") info
                  call PrintMessage( "msQR", "E",                       &
                                     "in QR-factorizing matrix A by LAPACK:", &
                                     "info = " // info_char,            &
                                     "(as a side effect, Q and R will be empty)" )
                  call msSilentRelease( Q, R )
                  p%i(:) = [ (i,i=1,n) ]
               else
                  if( m >= n ) then
                     call msAssign( R, mfGet( A_copy, [ (i, i = 1, n) ], MF_COLON ))
                     call msAssign( R, mftriu( R ) )
                     k = n
                     call zungqr( m, n, k, A_copy%cmplx(1,1), lda, ctau(1), cwork(1), lwork, info )
                     if( info /= 0 ) then
                        ! solution is not ok -- light warning
                        write(info_char,"(I0)") info
                        call PrintMessage( "msQR", "E",                 &
                                           "in building Q matrix A by LAPACK:", &
                                           "info = " // info_char,      &
                                           "(as a side effect, Q will be empty)" )
                        call msSilentRelease( Q )
                        p%i(:) = [ (i,i=1,n) ]
                     else
                        Q%data_type = MF_DT_CMPLX
                        Q%cmplx(:,:) = A_copy%cmplx(:,:)
                     end if
                  else ! m < n
                     call msAssign( R, mftriu( A_copy ))
                     k = m
                     call zungqr( m, m, k, A_copy%cmplx(1,1), lda, ctau(1), cwork(1), lwork, info )
                     if( info /= 0 ) then
                        ! solution is not ok -- light warning
                        write(info_char,"(I0)") info
                        call PrintMessage( "msQR", "E",                 &
                                           "in building Q matrix A by LAPACK:", &
                                           "info = " // info_char,      &
                                           "(as a side effect, Q will be empty)" )
                        call msSilentRelease( Q )
                        p%i(:) = [ (i,i=1,n) ]
                     else
                        call msAssign( Q, mfGet( A_copy, MF_COLON, [ (i, i = 1, m) ] ))
                     end if
                  end if
               end if

            end if

         else ! simpler QR (no column pivoting)

            if( A%data_type == MF_DT_DBLE ) then

               allocate( Q%double( Q%shape(1), Q%shape(2) ) )

               allocate( R%double( R%shape(1), R%shape(2) ) )

               lda = m
               allocate( tau(min(m,n)) )

               ! optimal blocksize for DGEQRF (from ILAENV)
               nb = 32
               lwork = n*nb
               allocate( work(lwork) )

               ! General system -- real general matrix : DGEQRF
               call dgeqrf( m, n, A_copy%double(1,1), lda, tau(1), work(1), lwork, info )
               if( info /= 0 ) then
                  ! solution is not ok -- light warning
                  write(info_char,"(I0)") info
                  call PrintMessage( "msQR", "E",                       &
                                     "in QR-factorizing matrix A by LAPACK:", &
                                     "info = " // info_char,            &
                                     "(as a side effect, Q and R will be empty)" )
                  call msSilentRelease( Q, R )
               else
                  if( m >= n ) then
                     call msAssign( R, mfGet( A_copy, [ (i, i = 1, n) ], MF_COLON ))
                     call msAssign( R, mftriu( R ))
                     k = n
                     call dorgqr( m, n, k, A_copy%double(1,1), lda, tau(1), work(1), lwork, info )
                     if( info /= 0 ) then
                        ! solution is not ok -- light warning
                        write(info_char,"(I0)") info
                        call PrintMessage( "msQR", "E",                 &
                                           "in building Q matrix A by LAPACK:", &
                                           "info = " // info_char,      &
                                           "(as a side effect, Q will be empty)" )
                        call msSilentRelease( Q )
                     else
                        Q%data_type = MF_DT_DBLE
                        Q%double(:,:) = A_copy%double(:,:)
                     end if
                  else
                     call msAssign( R, mftriu( A_copy ))
                     k = m
                     call dorgqr( m, m, k, A_copy%double(1,1), lda, tau(1), work(1), lwork, info )
                     if( info /= 0 ) then
                        ! solution is not ok -- light warning
                        write(info_char,"(I0)") info
                        call PrintMessage( "msQR", "E",                 &
                                           "in building Q matrix A by LAPACK:", &
                                           "info = " // info_char,      &
                                           "(as a side effect, Q will be empty)" )
                        call msSilentRelease( Q )
                     else
                        call msAssign( Q, mfGet( A_copy, MF_COLON, [ (i, i = 1, m) ] ))
                     end if
                  end if
               end if

            else if( A%data_type == MF_DT_CMPLX ) then

               allocate( Q%cmplx( Q%shape(1), Q%shape(2) ) )

               allocate( R%cmplx( R%shape(1), R%shape(2) ) )

               lda = m
               allocate( ctau(min(m,n)) )

               ! optimal blocksize for ZGEQRF (from ILAENV)
               nb = 32
               lwork = n*nb
               allocate( cwork(lwork) )

               ! General system -- complex general matrix : ZGEQRF
               call zgeqrf( m, n, A_copy%cmplx(1,1), lda, ctau(1), cwork(1), lwork, info )
               if( info /= 0 ) then
                  ! solution is not ok -- light warning
                  write(info_char,"(I0)") info
                  call PrintMessage( "msQR", "E",                       &
                                     "in QR-factorizing matrix A by LAPACK:", &
                                     "info = " // info_char,            &
                                     "(as a side effect, Q and R will be empty)" )
                  call msSilentRelease( Q, R )
               else
                  if( m >= n ) then
                     call msAssign( R, mfGet( A_copy, [ (i, i = 1, n) ], MF_COLON ))
                     call msAssign( R, mftriu( R ))
                     k = n
                     call zungqr( m, n, k, A_copy%cmplx(1,1), lda, ctau(1), cwork(1), lwork, info )
                     if( info /= 0 ) then
                        ! solution is not ok -- light warning
                        write(info_char,"(I0)") info
                        call PrintMessage( "msQR", "E",                 &
                                           "in building Q matrix A by LAPACK:", &
                                           "info = " // info_char,      &
                                           "(as a side effect, Q will be empty)" )
                        call msSilentRelease( Q )
                     else
                        Q%data_type = MF_DT_CMPLX
                        Q%cmplx(:,:) = A_copy%cmplx(:,:)
                     end if
                  else
                     call msAssign( R, mftriu( A_copy ))
                     k = m
                     call zungqr( m, m, k, A_copy%cmplx(1,1), lda, ctau(1), cwork(1), lwork, info )
                     if( info /= 0 ) then
                        ! solution is not ok -- light warning
                        write(info_char,"(I0)") info
                        call PrintMessage( "msQR", "E",                 &
                                           "in building Q matrix A by LAPACK:", &
                                           "info = " // info_char,      &
                                           "(as a side effect, Q will be empty)" )
                        call msSilentRelease( Q )
                     else
                        call msAssign( Q, mfGet( A_copy, MF_COLON, [ (i, i = 1, m) ] ))
                     end if
                  end if
               end if

            end if

         end if

         if( A_copy_is_allocated ) then
            call msSilentRelease( A_copy )
            deallocate( A_copy ) ! no_mem_trace !
         end if

      end if

      R%prop%symm = FALSE

      if( mf_phys_units ) then
         R%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msQR_mfArray
!_______________________________________________________________________
!
   subroutine msQR_mfMatFactor( A, Qhouse, R, p, RANK, tol )

      type(mfArray), target   :: A
      type(mfMatFactor)       :: Qhouse
      type(mfArray)           :: R
      type(mfArray), optional :: p, RANK
      real(kind=MF_DOUBLE), optional :: tol
      !------ API end ------
#ifdef _DEVLP

      ! this routine is only for internal use: an ordinary user
      ! should always call 'msQR' which makes many tests on arguments.

      ! full QR factorization of a sparse matrice 'A',
      ! using SuiteSparse/SPQR:
      !     [H,HTau,HPinv] is stored in the mfMatFactor 'Qhouse'
      !      R             is returned in the mfArray 'R'
      !
      ! if arg p is present, a columns permutation of A, chosen in
      ! such a way to get better sparsity of Q and R, is also returned.

      integer :: m, n
      real(kind=MF_DOUBLE) :: etol

      ! SPQR variables
      integer(kind=MF_ADDRESS) :: c_addr, H_addr, HTau_addr, HPinv_addr, &
                                  R_addr, p_addr
      integer :: econ, A_rank
      integer :: ordering, status, nnz, rnz
      integer(kind=MF_LONG_INT) :: total_size, n1, n2
      character(len=19) :: n1_char, n2_char

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "msQR", "E",                                &
                            "currently, when using an mfMatFactor,",    &
                            "the mfArray 'A' must be real!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)

      econ = m ! standard (no economy size for Householder form)

      R%shape = [ econ, n ]

      if( present(tol) ) then
         etol = tol
      else
         etol = MF_EPS
      end if

      ! call the 'SuiteSparseQR' C++ routine (SuiteSparse)

      ! sparse matrices must be row-sorted !
      if( A%row_sorted /= TRUE ) then
         call msRowSort(A)
      end if

      nnz = A%j(n+1) - 1

      if( present(p) ) then ! QR with column pivoting
         ordering = MF_CHOLMOD_AMD !
      else ! simpler QR (no column pivoting)
         ordering = MF_CHOLMOD_NATURAL ! no ordering (identity)
      end if

      ! do a QR factorization of A and returns, among other
      ! addresses and size of the factors Q (Householder form) and R.
      call spqhr_qr_prep( m, n, nnz, A%j, A%i, A%a,                     &
                          ordering, etol, econ,                         &
                          c_addr, H_addr, HTau_addr, HPinv_addr,        &
                          R_addr, rnz, p_addr, A_rank, status )

      ! check for an error
      if( status /= 0 ) then
         call PrintMessage( "msQR", "E",                                &
                            "spqr_qhr_prep failed!" )
         ! no more information: SPQR should never fail!
         ! (except out-of-memory)
         go to 99
      end if

      ! check for validity: we must have nnz(R) >= 0
      if( rnz < 0 ) then
         write(n1_char,*) rnz
         write(STDERR,*) "(MUESLI mfQR:) internal error:"
         write(STDERR,*) "               spqr_q_less_prep failed to return a valid R nnz!"
         write(STDERR,*) "               nnz(R) >= 0 must hold, but: nnz(R) = ", trim(adjustl(n1_char))
         mf_message_displayed = .true.
         call muesli_trace( pause ="yes" )
         stop
      end if

      ! check for validity: we must have nnz(R) <= m*n
      !   (i.e. less than the max number of elements)
      n1 = m
      n2 = n
      total_size = n1*n2
      if( rnz > total_size ) then ! rnz > m*n
         write(n1_char,*) rnz
         write(n2_char,*) total_size
         write(STDERR,*) "(MUESLI mfQR:) internal error:"
         write(STDERR,*) "               spqr_q_less_prep failed to return a valid R nnz!"
         write(STDERR,*) "               nnz(R) <= m*n must hold, but: nnz(R) = ", trim(adjustl(n1_char))
         write(STDERR,*) "               whereas m*n = ", trim(adjustl(n2_char))
         mf_message_displayed = .true.
         call muesli_trace( pause ="yes" )
         stop
      end if

      call msFreeMatFactor( Qhouse )
      Qhouse%package = 3
      Qhouse%shape = [ m, m ]
      allocate( Qhouse%ptr_1 )
      Qhouse%ptr_1 = c_addr
      allocate( Qhouse%ptr_2 )
      Qhouse%ptr_2 = H_addr
      allocate( Qhouse%ptr_3 )
      Qhouse%ptr_3 = HTau_addr
      allocate( Qhouse%ptr_4 )
      Qhouse%ptr_4 = HPinv_addr

      ! allocating space for matrix R
      call msAssign( R, mfSpAlloc(R%shape(1),R%shape(2),rnz) )

      if( present(p) ) then ! QR with column pivoting
         p%data_type = MF_DT_PERM_VEC
         p%shape = [ n, 1 ]
         allocate( P%i(n) )

         ! get the factor R and permutation P
         ! (freeing of unused obj are also done)
         call spqhr_get_R_P( c_addr, R_addr, n, rnz, P_addr, n,         &
                             R%j, R%i, R%a, P%i )

      else ! simpler QR (no column pivoting)

         ! get the factor R. No need to get P.
         ! (freeing of unused obj are also done)
         call spqr_get_sparse_entries( c_addr, R_addr, n, rnz,          &
                                       R%j, R%i, R%a )

      end if

      R%prop%symm = FALSE

      ! All matrices returned by SuiteSparse are "row sorted"
      R%row_sorted = TRUE

      if( mf_phys_units ) then
         R%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msQR_mfMatFactor
!_______________________________________________________________________
!
   function mfQleft( Qhouse, A ) result( out )

      type(mfMatFactor)     :: Qhouse
      type(mfArray), target :: A
      type(mfArray)         :: out
      !------ API end ------
#ifdef _DEVLP

      ! apply Householder vectors on the left, i.e. computes:
      !
      !    out = Q'*A
      !
      ! where 'Q' is stored (as Householder vectors) in an mfMatFactor
      ! structure.

      ! 'A' may be a vector (dense only) or a matrix (dense or sparse)
      !
      ! 'out' has the same type as 'A'

      ! SPQR variables
      integer(kind=MF_ADDRESS) :: c_addr, H_addr, HTau_addr, HPinv_addr, &
                                  out_addr
      integer :: m, nrow_A, ncol_A, status, nnz, out_nz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( Qhouse%package /= 3 ) then
         call PrintMessage( "mfQleft", "E",                             &
                            "bad type for first arg!",                 &
                            "(must be an orthogonal matrix stored in Householder form)" )
         go to 99
      end if

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfQleft", "E",                             &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfQleft", "E",                             &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      m = Qhouse%shape(1)
      !   Qhouse%shape(2) contains no farther information

      nrow_A = A%shape(1)
      ncol_A = A%shape(2)

      if( nrow_A /= m ) then
         call PrintMessage( "mfQleft", "E",                             &
                            "bad shape for 'A'!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfQleft", "E",                             &
                            "currently, 'A' must be real!" )
         go to 99
      end if

      c_addr     = Qhouse%ptr_1
      H_addr     = Qhouse%ptr_2
      HTau_addr  = Qhouse%ptr_3
      HPinv_addr = Qhouse%ptr_4

      out%shape = A%shape

      if( mfIsSparse(A) ) then

         ! call the 'SuiteSparseQR' C++ routine (SuiteSparse)

         ! sparse matrices must be row-sorted !
         if( A%row_sorted /= TRUE ) then
            call msRowSort(A)
         end if

         nnz = A%j(ncol_A+1) - 1

         call spqhr_mleft_sparse_prep(                                  &
                 c_addr, H_addr, HTau_addr, HPinv_addr,                 &
                 nrow_A, ncol_A, nnz, A%j, A%i, A%a,                    &
                 out_addr, out_nz, status )

         ! check for an error
         if( status /= 0 ) then
            call PrintMessage( "mfQleft", "E",                          &
                               "spqhr_mleft_sparse_prep failed!" )
            go to 99
         end if

         ! check for a valid out nnz
         if( out_nz == -1 ) then
            call PrintMessage( "mfQleft", "E",                          &
                               "spqhr_mleft_sparse_prep failed to return a valid out nnz!" )
            go to 99
         end if

         ! allocating space for matrix out
         call msAssign( out, mfSpAlloc(out%shape(1),out%shape(2),out_nz) )

         ! get all entries of 'out'
         call spqr_get_sparse_entries( c_addr, out_addr, ncol_A, out_nz, &
                                       out%j, out%i, out%a )

         ! All matrices returned by SuiteSparse are "row sorted"
         out%row_sorted = TRUE

      else ! A : dense matrix

         out%data_type = MF_DT_DBLE
         allocate( out%double(out%shape(1),out%shape(2)) )

         call spqhr_mleft_dense( c_addr, H_addr, HTau_addr, HPinv_addr, &
                                 A%double(:,1), nrow_A, ncol_A, nrow_A, &
                                 out%double(:,1), status )

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfQleft
!_______________________________________________________________________
!
   function mfQright( A, Qhouse ) result( out )

      type(mfArray), target :: A
      type(mfMatFactor)     :: Qhouse
      type(mfArray)         :: out
      !------ API end ------
#ifdef _DEVLP

      ! apply Householder vectors on the right, i.e. computes:
      !
      !    out = A*Q
      !
      ! where 'Q' is stored (as Householder vectors) in an mfMatFactor
      ! structure.

      ! 'A' may be a vector (dense only) or a matrix (dense or sparse)
      !
      ! 'out' has the same type as 'A'

      ! SPQR variables
      integer(kind=MF_ADDRESS) :: c_addr, H_addr, HTau_addr, HPinv_addr, &
                                  out_addr
      integer :: m, nrow_A, ncol_A, status, nnz, out_nz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfQright", "E",                            &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfQright", "E",                            &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( Qhouse%package /= 3 ) then
         call PrintMessage( "mfQright", "E",                            &
                            "bad type for second arg!",                &
                            "(must be an orthogonal matrix stored in Householder form)" )
         go to 99
      end if

      m = Qhouse%shape(1)
      !   Qhouse%shape(2) contains no farther information

      nrow_A = A%shape(1)
      ncol_A = A%shape(2)

      if( ncol_A /= m ) then
         call PrintMessage( "mfQright", "E",                            &
                            "bad shape for 'A'!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfQright", "E",                            &
                            "'A' must be real!" )
         go to 99
      end if

      c_addr     = Qhouse%ptr_1
      H_addr     = Qhouse%ptr_2
      HTau_addr  = Qhouse%ptr_3
      HPinv_addr = Qhouse%ptr_4

      out%shape = A%shape

      if( mfIsSparse(A) ) then

         ! call the 'SuiteSparseQR' C++ routine (SuiteSparse)

         ! sparse matrices must be row-sorted !
         if( A%row_sorted /= TRUE ) then
            call msRowSort(A)
         end if

         nnz = A%j(ncol_A+1) - 1

         call spqhr_mright_sparse_prep(                                 &
                 c_addr, H_addr, HTau_addr, HPinv_addr,                 &
                 nrow_A, ncol_A, nnz, A%j, A%i, A%a,                    &
                 out_addr, out_nz, status )

         ! check for an error
         if( status /= 0 ) then
            call PrintMessage( "mfQright", "E",                         &
                               "spqhr_mright_sparse_prep failed!" )
            go to 99
         end if

         ! check for a valid out nnz
         if( out_nz == -1 ) then
            call PrintMessage( "mfQright", "E",                         &
                               "spqhr_mright_sparse_prep failed to return a valid out nnz!" )
            go to 99
         end if

         ! allocating space for matrix out
         call msAssign( out, mfSpAlloc(out%shape(1),out%shape(2),out_nz) )

         ! get all entries of 'out'
         call spqr_get_sparse_entries( c_addr, out_addr, ncol_A, out_nz, &
                                       out%j, out%i, out%a )

         ! All matrices returned by SuiteSparse are "row sorted"
         out%row_sorted = TRUE

      else ! A : dense matrix

         out%data_type = MF_DT_DBLE
         allocate( out%double(out%shape(1),out%shape(2)) )

         call spqhr_mright_dense( c_addr, H_addr, HTau_addr, HPinv_addr, &
                                  A%double(:,1), nrow_A, ncol_A, nrow_A, &
                                  out%double(:,1), status )

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfQright
