! f90 include file

!_______________________________________________________________________
!
   function mfPowm_integer( A, n ) result( out )

      type(mfArray) :: A
      integer, intent(in) :: n
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Matrix power (integer exponent)

      ! For a DENSE square matrix A:
      !
      !   - for small values of n (2, 3 or 4), direct matrix
      !     multiplication is used (with optimization of the number of
      !     operations).
      !
      !   - for larger values, eigenvalues and eigenvectors of A are
      !     computed; if V, the matrix composed with all column
      !     eigenvectors, is full rank, then A is diagonalizable and
      !     we can write:
      !
      !     A = V*D*V^(-1)    =>    A^n = V*D.^n*V^(-1)
      !
      ! On the contrary, the SCHUR decomposition is used and direct
      ! matrix multiplication (with optimization of the number of
      ! operations) is applied to the quasi-triangular matrix.
      !
      ! For a SPARSE square matrix A, direct matrix multiplication is
      ! used (with optimization of the number of operations).

      type(mfArray) :: V, D, U, T
      integer :: i, p, status
      integer :: nrow, ncol, nnz

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      nrow = A%shape(1)
      ncol = A%shape(2)

      if( nrow <= 1 .or. ncol <= 1 ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      ! square matrix ?
      if( nrow /= ncol ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      if( n < 0 ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "'n' must be positive!" )
         go to 99
      end if

      if( n == 0 ) then
         call msAssign( out, mfEye(nrow) ) ! returns identity matrix
         out%status_temporary = .true.
         go to 99
      end if

      if( n == 1 ) then
         out = A ! returns A
         out%status_temporary = .true.
         go to 99
      end if

      if( mfIsSparse(A) ) then

         nnz = A%j(ncol+1) - 1
         if( nnz == 0 ) then
            out = A ! returns A
            out%status_temporary = .true.
            go to 99
         end if

         go to 10

      else ! dense

         if( n <= 4 ) then
            go to 10
         else
            go to 20
         end if

      end if

 10   continue

      ! optimized matrix multiplication
      call powm_optim_mult( out, A, n )
      go to 89

 20   continue

      ! diagonalization of A
      call msEig( mfOut(V,D), A )
      p = mfRank(V,sqrt(MF_EPS))
      if( p /= A%shape(1) ) then
         go to 30 ! using SCHUR decomposition
      end if
      call PrintMessage( "mfPowm", "I",                                 &
                         "diagonalization is used" )
      if( mfIsReal(D) ) then
         do i = 1, D%shape(1)
            D%double(i,i) = D%double(i,i)**n
         end do
      else ! complex
         do i = 1, D%shape(1)
            D%cmplx(i,i) = D%cmplx(i,i)**n
         end do
      end if
      call msAssign( out, mfMul(mfMul(V,D),mfInv(V)) )

      go to 89

 30   continue

      ! SCHUR decomposition
      call msSchur( mfOut(U,T), A )
      call PrintMessage( "mfPowm", "I",                                 &
                         "SCHUR decomposition is used" )
      call powm_optim_mult( out, T, n )
      call msAssign( out, mfMul(mfMul(U,out),.t.U) )
      call msSilentRelease( U, T )

 89   continue

      call msSilentRelease( V, D )

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfPowm", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfPowm_integer
!_______________________________________________________________________
!
   function mfPowm_real( A, r ) result( out )

      type(mfArray) :: A
      real(kind=MF_DOUBLE), intent(in) :: r
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Matrix power (real exponent)

      ! for a DENSE square matrix A only:
      !
      ! for r == n (positive integer), mfPowm_integer is called
      !
      ! for r /= -1, eigenvalues and eigenvectors of A are computed;
      ! if V, the matrix composed with all column eigenvectors,
      ! is full rank, then A is diagonalizable and we can
      ! write:
      !
      !     A = V*D*V^(-1)    =>    A^r = V*D.^r*V^(-1)
      !
      ! on the contrary, the current routine fails.

      type(mfArray) :: V, D
      integer :: i, p, status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( r == dble(nint(r)) ) then
         call msAssign( out, mfPowm_integer(A,nint(r)) )
         out%status_temporary = .true.
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      ! square matrix ?
      if( A%shape(1) /= A%shape(2) ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      if( r == -1.0d0 ) then
         call msAssign( out, mfInv(A) ) ! returns inverse of A
         out%status_temporary = .true.
         go to 99
      end if

      ! diagonalization of A
      call msEig( mfOut(V,D), A )
      p = mfRank(V,sqrt(MF_EPS))
      if( p /= A%shape(1) ) then
         call PrintMessage( "mfPowm", "E",                              &
                            "the routine fails because A is defective!" )
         go to 99
      end if
      if( mfIsReal(D) ) then
         do i = 1, D%shape(1)
            D%double(i,i) = D%double(i,i)**r
         end do
      else ! complex
         do i = 1, D%shape(1)
            D%cmplx(i,i) = D%cmplx(i,i)**r
         end do
      end if
      call msAssign( out, mfMul(mfMul(V,D),mfInv(V)) )
      call msSilentRelease( V, D )

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfPowm", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfPowm_real
!_______________________________________________________________________
!
   subroutine powm_optim_mult( out, A, n )

      type(mfArray) :: out, A
      integer, intent(in) :: n
      !------ API end ------
#ifdef _DEVLP

      type(mfArray) :: f
      integer :: p

      ! optimized matrix multiplication

   !------ end of declarations -- execution starts hereafter  ------

      if( A%level_protected == 1 ) then
         call msAssign( out, A )
      else
         out = A
      end if
      ! decompose n in prime factors
      f = n
      call msAssign( f, mfFactor(f) )
      do while( .not. mfIsEmpty(f) )
         p = mfGet( f, 1 )
         call msSet( MF_EMPTY, f, 1 )
         select case( p )
            case( 2 )
               call msAssign( out, mfMul(out,out) )
            case( 3 )
               call msAssign( out, mfMul(mfMul(out,out),out) )
            case( 5 )
               call msAssign( out, mfMul(mfMul(mfMul(mfMul(out,out),out),out),out) )
            case( 7 )
               call msAssign( out, mfMul(mfMul(mfMul(mfMul(mfMul(mfMul(out,out), &
                                         out),out),out),out),out) )
            case default
               write(STDERR,*) "(MUESLI powm_optim_mult:) Internal Error"
               write(STDERR,*) "                          unexpected value: n = ", n
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
         end select
      end do
      call msSilentRelease( f )

#endif
   end subroutine powm_optim_mult
