! f90 include file

!_______________________________________________________________________
!
   subroutine msLU( out, A )

      type(mfArray), target      :: A
      type(mf_Out)               :: out
      !------ API end ------
#ifdef _DEVLP

      ! A : dense or sparse matrix (driver routine)
      !
      ! if called with (out) mfArrays:
      !   returns the LU factors and (optionally), the permutations
      !   and the scaling
      !
      ! if called with one (out) mfMatFactor and a sparse matrix A:
      !   returns the LU factors in one variable

      logical :: case_1, case_2, case_3
      type(mfArray), pointer :: L, U, P, Q, R
      type(mfMatFactor), pointer :: factor
      logical :: L_U_ok

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! valid cases
      if( mfIsSparse(A) ) then
         ! 2 or 5 out-args must be specified
         case_1 = ( out%n == 2 .or. out%n == 5 )                        &
                  .and.                                                 &
                  ( out%factor_n == 0 )
         ! one output mfMatFactor
         case_2 = ( out%n == 0 )                                        &
                  .and.                                                 &
                  ( out%factor_n == 1 )
         case_3 = .false.
         if( .not. case_1 .and. .not. case_2 ) then
            call PrintMessage( "msLU", "E",                             &
                               "(sparse case) bad argument call!",      &
                               "syntax is: call msLU ( mfOut(L,U[,P,Q,R]), A ) [mfArrays]", &
                               "       or: call msLU ( mfOut(factors), A )     [mfMatFactor]" )
            go to 99
         end if
      else ! dense matrix
         case_1 = .false.
         case_2 = .false.
         ! 2 or 3 out-args must be specified
         case_3 = ( out%n == 2 .or. out%n == 3 )                        &
                  .and.                                                 &
                  ( out%factor_n == 0 )
         if( .not. case_3 ) then
            call PrintMessage( "msLU", "E",                             &
                               "(dense case) bad argument call!",       &
                               "syntax is: call msLU ( mfOut(L,U[,P]), A ) [mfArrays]" )
            go to 99
         end if
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, A ) ) then
         call PrintMessage( "msLU", "E",                                &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msLU", "E",                                &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msLU", "E",                                &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msLU", "E",                                &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( case_1 ) then ! two or five output mfArrays (sparse matrix)

         L => out%ptr1
         U => out%ptr2
         call msSilentRelease( L, U )
         if( out%n == 2 ) then
            call msLU_mfArray( A, L, U )
         else ! out%n == 5
            P => out%ptr3
            Q => out%ptr4
            R => out%ptr5
            call msSilentRelease( P, Q, R )
            call msLU_mfArray( A, L, U, P, Q, R )
            R%prop%tril = TRUE
            R%prop%triu = TRUE
            R%prop%symm = TRUE
         end if
         L_U_ok = .true.

      else if ( case_2 ) then ! one output mfMatFactor (sparse matrix)

         factor => out%factor_ptr1
         call msFreeMatFactor( factor )
         call msLU_mfMatFactor( A, factor )
         L_U_ok = .false.

      else ! two or three output mfArrays (dense matrix)

         L => out%ptr1
         U => out%ptr2
         call msSilentRelease( L, U )
         if( out%n == 2 ) then
            call msLU_mfArray( A, L, U )
         else ! out%n == 3
            P => out%ptr3
            call msSilentRelease( P )
            call msLU_mfArray( A, L, U, P )
         end if
         L_U_ok = .true.

      end if

      if( L_U_ok ) then
         L%prop%tril = TRUE
         L%prop%symm = FALSE
         U%prop%triu = TRUE
         U%prop%symm = FALSE
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msLU
!_______________________________________________________________________
!
   subroutine msLU_mfArray( A, L, U, p, q, r )

      type(mfArray), target   :: A
      type(mfArray)           :: L, U
      type(mfArray), optional :: p, q, r
      !------ API end ------
#ifdef _DEVLP

      ! for a sparse matrix, 'msLU' stores in L%umf4_ptr_numeric
      ! a pointer to a complete C structure, containing all
      ! necessary data (i.e., L, U, p, q, r)
      !
      ! cf. also: mfLDiv_three_args

      ! permutations {p, q, r} are returned as column vectors (from 2.6.0)

      integer :: i, j, m, n, lda, info, i_tmp
      integer, allocatable :: ipiv(:)
      real(kind=MF_DOUBLE), allocatable :: vec_tmp(:)
      complex(kind=MF_DOUBLE), allocatable :: zvec_tmp(:)
      type(mfArray), pointer :: A_copy
      character(len=3) :: info_char
      integer :: status
      logical :: A_copy_is_allocated
      integer :: mf_message_level_save

      ! declarations for UMFPACK
      integer(kind=MF_ADDRESS) :: numeric, symbolic ! LU handle
      integer :: nnz, lnz, unz
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      integer, allocatable :: ptr_Aj(:), ptr_Ai(:)
      real(kind=MF_DOUBLE), allocatable :: Ar(:), Az(:),                &
                                           Lr(:), Lz(:), Ur(:), Uz(:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      m = A%shape(1)
      n = A%shape(2)

      L%shape = [ m, min(m,n) ]
      U%shape = [ min(m,n), n ]

      if( mfIsSymm(A) ) then
         if( mfIsPosDef(A) ) then
            call PrintMessage( "msLU", "W",                             &
                               "matrix 'A' is Symm. and Pos. Def.!",    &
                               "You should use a Cholesky factorization instead." )
         else
            call PrintMessage( "msLU", "I",                             &
                               "matrix 'A' is symmetric.",              &
                               "You could use an L*D*L' factorization instead, but", &
                               "be aware that it ca fail (a non-singular matrix is necessary", &
                               "but not sufficient)." )
         end if
      end if

      if( A%data_type == MF_DT_DBLE ) then

         ! making copies because LAPACK overwrites A
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if

         allocate( L%double( L%shape(1), L%shape(2) ) )

         allocate( U%double( U%shape(1), U%shape(2) ) )

         ! General system -- real matrix : DGETRF
         lda = m
         allocate( ipiv(min(m,n)) )

         call dgetrf( m, n, A_copy%double(1,1), lda, ipiv(1), info )

         if( info < 0 ) then
            ! call was not ok -- the i-th argument had an illegal value
            write(STDERR,*) "(MUESLI msLU:) internal error:"
            write(STDERR,*) "               arg number: ", info
            write(STDERR,*) "               had an illegal value"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         else if( info > 0 ) then
            ! solution is ok -- but info is written because matrix
            !                   may be singular
            write(info_char,"(I0)") info
            call PrintMessage( "msLU", "I",                             &
                               "in LU-factorizing matrix A by LAPACK:", &
                               "The element U(i,i), with i = " // info_char // " is exactly zero.", &
                               "(The factorization has been completed, but the factor U is exactly singular, ", &
                               " and division by zero will occur if it is used to solve a system  of equations.)" )
         end if

         if( m < n ) then
            call msAssign( L, mfGet( mfTril( A_copy, -1 ), MF_COLON,    &
                                     [(i,i=1,min(m,n))] ) )
         else
            call msAssign( L, mfTril( A_copy, -1 ) )
         end if
         do i = 1, min(m,n)
            L%double(i,i) = 1.0d0
         end do
         if( m > n ) then
            call msAssign( U, mfGet( mfTriu( A_copy ), [(i,i=1,min(m,n))], &
                                     MF_COLON ) )
         else
            call msAssign( U, mfTriu( A_copy ) )
         end if

         if( present(p) ) then
            ! returning a column vector permutation
            p%shape = [ m, 1 ]
            p%data_type = MF_DT_PERM_VEC
            allocate( p%i( m ) )

            ! IPIV :
            ! The pivot indexes that define the permutation matrix P;
            ! row i of the matrix was interchanged with row IPIV(i).
            p%i = [ (i, i=1,m) ]
            ! transformed sequence
            do i = 1, min(m,n)
               j = ipiv(i)
               if( j /= i ) then
                  ! exchange indexes (i,j) in p%i
                  i_tmp = p%i(i)
                  p%i(i) = p%i(j)
                  p%i(j) = i_tmp
               end if
            end do
         else
            ! permutation matrix P is not requested, so we must
            ! return a modified L factor so that: L*U = A
            ! (hence we return P*L instead of L)
            allocate( vec_tmp(min(m,n)) )
            do i = min(m,n), 1, -1
               j = ipiv(i)
               if( j /= i ) then
                  ! exchange lines (i,j) in L
                  vec_tmp(:) = L%double(i,:)
                  L%double(i,:) = L%double(j,:)
                  L%double(j,:) = vec_tmp(:)
               end if
            end do
         end if

         if( A_copy_is_allocated ) then
            call msSilentRelease( A_copy )
            deallocate( A_copy ) ! no_mem_trace !
         end if

      else if( A%data_type == MF_DT_CMPLX ) then

         ! making copies because LAPACK overwrites A
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if

         allocate( L%cmplx( L%shape(1), L%shape(2) ) )

         allocate( U%cmplx( U%shape(1), U%shape(2) ) )

         ! General system -- complex matrix : ZGETRF
         lda = m
         allocate( ipiv(min(m,n)) )

         call zgetrf( m, n, A_copy%cmplx(1,1), lda, ipiv(1), info )

         if( info < 0 ) then
            ! call was not ok -- the i-th argument had an illegal value
            write(STDERR,*) "(MUESLI msLU:) internal error:"
            write(STDERR,*) "               arg number: ", info
            write(STDERR,*) "               had an illegal value"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         else if( info > 0 ) then
            ! solution is ok -- but info is written because matrix
            !                   may be singular
            write(info_char,"(I0)") info
            call PrintMessage( "msLU", "I",                             &
                               "in LU-factorizing matrix A by LAPACK:", &
                               "The element U(i,i), with i = " // info_char // " is exactly zero.", &
                               "(The factorization has been completed, but the factor U is exactly singular, ", &
                               " and division by zero will occur if it is used to solve a system  of equations.)" )
         end if

         if( m < n ) then
            call msAssign( L, mfGet( mfTril( A_copy, -1 ), MF_COLON,    &
                                     [(i,i=1,min(m,n))] ) )
         else
            call msAssign( L, mfTril( A_copy, -1 ) )
         end if
         do i = 1, min(m,n)
            L%cmplx(i,i) = 1.0d0
         end do
         if( m > n ) then
            call msAssign( U, mfGet( mfTriu( A_copy ), [(i,i=1,min(m,n))], &
                                     MF_COLON ) )
         else
            call msAssign( U, mfTriu( A_copy ) )
         end if

         if( present(p) ) then
            ! returning a column vector permutation
            p%shape = [ m, 1 ]
            p%data_type = MF_DT_PERM_VEC
            allocate( p%i( m ) )

            ! IPIV :
            ! The pivot indexes that define the permutation matrix P;
            ! row i of the matrix was interchanged with row IPIV(i).
            p%i = [ (i, i=1,m) ]
            ! transformed sequence
            do i = 1, min(m,n)
               j = ipiv(i)
               if( j /= i ) then
                  ! exchange indexes (i,j) in p%i
                  i_tmp = p%i(i)
                  p%i(i) = p%i(j)
                  p%i(j) = i_tmp
               end if
            end do
         else
            ! permutation matrix P is not requested, so we must
            ! return a modified L factor so that: L*U = A
            ! (hence we return P*L instead of L)
            allocate( zvec_tmp(min(m,n)) )
            do i = min(m,n), 1, -1
               j = ipiv(i)
               if( j /= i ) then
                  ! exchange lines (i,j) in L
                  zvec_tmp(:) = L%cmplx(i,:)
                  L%cmplx(i,:) = L%cmplx(j,:)
                  L%cmplx(j,:) = zvec_tmp(:)
               end if
            end do
         end if

         if( A_copy_is_allocated ) then
            call msSilentRelease( A_copy )
            deallocate( A_copy ) ! no_mem_trace !
         end if

      else if( A%data_type == MF_DT_SP_DBLE ) then

         ! ok for m /= n

         ! uses UMFPACK-4 (C)
         ! sparse matrices must be row-sorted !
         if( A%row_sorted /= TRUE ) then
            call msRowSort(A)
         end if

         ! set UMFPACK-4 default parameters
         call umf4def_d(control)
         ! print control parameters.
         ! Set control(1) to 1 to print error messages only
         control(1) = 1

         ! convert from 1-based to 0-based
         allocate( ptr_Aj(n+1) )
         ptr_Aj(1:n+1) = A%j(1:n+1) - 1
         nnz = A%j(n+1) - 1
         allocate( ptr_Ai(nnz) )
         ptr_Ai(1:nnz) = A%i(1:nnz) - 1

         ! pre-order and symbolic analysis
         call umf4sym_d( m, n, ptr_Aj, ptr_Ai, A%a, symbolic,           &
                         control, infos )
         ! check umf4sym error condition
         if( infos(1) < 0 ) then
            write(STDERR,*) "(MUESLI msLU:) Error occurred in umf4sym_d: ", infos(1)
            if( infos(1) == -8 ) then ! -8 : UMFPACK-version dependant !
               ! UMFPACK_ERROR_invalid_matrix (-8)
               ! (the following description has been adapted to fortran-like 1-based
               !  indexes !)
               write(STDERR,*) "Number of entries in the matrix is negative, Ap(1) is not 1,"
               write(STDERR,*) "a column has a negative number of entries, a row index is out of"
               write(STDERR,*) "bounds, or the columns of input matrix were jumbled (unsorted"
               write(STDERR,*) "columns or duplicate entries)."
            end if
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            go to 99
         end if

         ! numeric factorization
         call umf4num_d( n, ptr_Aj, ptr_Ai, A%a, symbolic, numeric,     &
                         control, infos )
         ! check umf4num error condition
         if( infos(1) < 0 ) then
            write(info_char,"(I0)") nint(infos(1))
            call PrintMessage( "msLU", "E",                             &
                               "in umf4num_d: infos(1) = " // info_char )
            go to 99
         end if

         deallocate( ptr_Aj )
         deallocate( ptr_Ai )

         ! free the symbolic analysis
         call umf4fsym_d(symbolic)

         ! get NNZ of L and U factors
         call umf4nzlu_d( lnz, unz, numeric, status )
         ! check error condition
         if( status /= 0 ) then
            call PrintMessage( "msLU", "E",                             &
                               "in umf4nzlu_d!" )
            go to 99
         end if

         ! L will be transposed later on...
         call msAssign( L, mfSpAlloc( min(m,n), m, nzmax=lnz ) )
         call msAssign( U, mfSpAlloc( min(m,n), n, nzmax=unz ) )
         if( present(p) .and. present(q) .and. present(r) ) then
            p%data_type = MF_DT_PERM_VEC
            p%shape = [ m, 1 ]
            allocate( p%i(m) )

            q%data_type = MF_DT_PERM_VEC
            q%shape = [ n, 1 ]
            allocate( q%i(n) )

            r%data_type = MF_DT_DBLE
            r%shape = [ m, 1 ]
            allocate( r%double(m,1) )

            ! L.U = P'.R.A.Q

            ! get the L, U, p, q, r factors
            call umf4getlupqr_d( m, n, lnz, unz,                        &
                                 L%j, L%i, L%a, U%j, U%i, U%a,          &
                                 p%i, q%i, r%double, numeric, status )
            ! check error condition
            if( status /= 0 ) then
               call PrintMessage( "msLU", "E",                          &
                                  "in umf4getlupqr_d!" )
               go to 99
            end if

            ! convert from 0-based to 1-based
            p%i(:) = p%i(:) + 1
            q%i(:) = q%i(:) + 1

            ! now, returns p_inv and not p (because it's a row permutation)
!### ???
!!            call msAssign( P, .t. P )

         else ! L and U only

            ! get only the L, U factors
            call umf4getlu_d( m, n, lnz, unz,                           &
                              L%j, L%i, L%a, U%j, U%i, U%a,             &
                              numeric, status )
            ! check error condition
            if( status /= 0 ) then
               call PrintMessage( "msLU", "E",                          &
                                  "in umf4getlu_d!" )
               go to 99
            end if

         end if

         ! All matrices returned by SuiteSparse are "row sorted"
         L%row_sorted = TRUE
         U%row_sorted = TRUE

         ! convert from 0-based to 1-based
         L%i(:) = L%i(:) + 1
         L%j(:) = L%j(:) + 1
         U%i(:) = U%i(:) + 1
         U%j(:) = U%j(:) + 1

         ! be careful : L is returned in CSR format !
         call msAssign( L, .t. L )

         ! only L factor contains a pointer to the 'numeric' struct
         !
         ! this UMFPACK struct will be freed in 'Implem_msRelease'
         ! and 'Implem_msSilentRelease'
         allocate( L%umf4_ptr_numeric )

         L%umf4_ptr_numeric = numeric

      else if( A%data_type == MF_DT_SP_CMPLX ) then

         ! ok for m /= n

         ! uses UMFPACK-4 (C)
         ! sparse matrices must be row-sorted !
         if( A%row_sorted /= TRUE ) then
            call msRowSort(A)
         end if

         ! set UMFPACK-4 default parameters
         call umf4def_z(control)
         ! print control parameters.
         ! Set control(1) to 1 to print error messages only
         control(1) = 1

         ! convert from 1-based to 0-based
         allocate( ptr_Aj(n+1) )
         ptr_Aj(1:n+1) = A%j(1:n+1) - 1
         nnz = A%j(n+1) - 1
         allocate( ptr_Ai(nnz) )
         ptr_Ai(1:nnz) = A%i(1:nnz) - 1
         allocate( Ar(nnz) )
         Ar(:) = real(A%z(1:nnz))
         allocate( Az(nnz) )
         Az(:) = aimag(A%z(1:nnz))

         ! pre-order and symbolic analysis
         call umf4sym_z( m, n, ptr_Aj, ptr_Ai, Ar, Az, symbolic,        &
                         control, infos )
         ! check umf4sym error condition
         if( infos(1) < 0 ) then
            write(STDERR,*) "(MUESLI msLU:) Error occurred in umf4sym_z: ", infos(1)
            if( infos(1) == -8 ) then ! -8 : UMFPACK-version dependant !
               ! UMFPACK_ERROR_invalid_matrix (-8)
               ! (the following description has been adapted to fortran-like 1-based
               !  indexes !)
               write(STDERR,*) "Number of entries in the matrix is negative, Ap(1) is not 1,"
               write(STDERR,*) "a column has a negative number of entries, a row index is out of"
               write(STDERR,*) "bounds, or the columns of input matrix were jumbled (unsorted"
               write(STDERR,*) "columns or duplicate entries)."
            end if
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            go to 99
         end if

         ! numeric factorization
         call umf4num_z( n, ptr_Aj, ptr_Ai, Ar, Az, symbolic, numeric,  &
                         control, infos )
         ! check umf4num error condition
         if( infos(1) < 0 ) then
            write(info_char,"(I0)") nint(infos(1))
            call PrintMessage( "msLU", "E",                             &
                               "in umf4num_z: infos(1) = " // info_char )
            go to 99
         end if

         deallocate( ptr_Aj, ptr_Ai )
         deallocate( Ar, Az )

         ! free the symbolic analysis
         call umf4fsym_z(symbolic)

         ! get NNZ of L and U factors
         call umf4nzlu_z( lnz, unz, numeric, status )
         ! check error condition
         if( status /= 0 ) then
            call PrintMessage( "msLU", "E",                             &
                               "in umf4nzlu_z!" )
            go to 99
         end if

         ! L will be transposed later on...
         call msAssign( L, mfSpAlloc( min(m,n), m, nzmax=lnz, kind="complex" ) )
         call msAssign( U, mfSpAlloc( min(m,n), n, nzmax=unz, kind="complex" ) )
         allocate( Lr(lnz), Lz(lnz) )
         allocate( Ur(unz), Uz(unz) )
         if( present(p) .and. present(q) .and. present(r) ) then
            p%data_type = MF_DT_PERM_VEC
            p%shape = [ m, 1 ]
            allocate( p%i(m) )

            q%data_type = MF_DT_PERM_VEC
            q%shape = [ n, 1 ]
            allocate( q%i(n) )

            r%data_type = MF_DT_DBLE
            r%shape = [ m, 1 ]
            allocate( r%double(m,1) )

            ! L.U = P'.R.A.Q

            ! get the L, U, p, q, r factors
            call umf4getlupqr_z( m, n, lnz, unz,                        &
                                 L%j, L%i, Lr, Lz, U%j, U%i, Ur, Uz,    &
                                 p%i, q%i, r%double, numeric, status )
            ! check error condition
            if( status /= 0 ) then
               call PrintMessage( "msLU", "E",                          &
                                  "in umf4getlupqr_z!" )
               go to 99
            end if

            ! convert from 0-based to 1-based
            p%i(:) = p%i(:) + 1
            q%i(:) = q%i(:) + 1

            ! now returns p_inv and not p (because it's a row permutation)
!!            call msAssign( P, .t. P )

         else ! L and U only

            ! get only the L, U factors
            call umf4getlu_z( m, n, lnz, unz,                           &
                              L%j, L%i, Lr, Lz, U%j, U%i, Ur, Uz,       &
                              numeric, status )
            ! check error condition
            if( status /= 0 ) then
               call PrintMessage( "msLU", "E",                          &
                                  "in umf4getlu_z!" )
               go to 99
            end if

         end if

         ! All matrices returned by SuiteSparse are "row sorted"
         L%row_sorted = TRUE
         U%row_sorted = TRUE

         L%z = cmplx(Lr,Lz,kind=MF_DOUBLE)
         U%z = cmplx(Ur,Uz,kind=MF_DOUBLE)

         deallocate( Lr, Lz, Ur, Uz )

         ! convert from 0-based to 1-based
         L%i(:) = L%i(:) + 1
         L%j(:) = L%j(:) + 1
         U%i(:) = U%i(:) + 1
         U%j(:) = U%j(:) + 1

         ! be careful : L is returned in CSR format !
         ! not .h.
         mf_message_level_save = mf_message_level
         mf_message_level = 0
         call msAssign( L, .t. L )
         mf_message_level = mf_message_level_save

         ! only L factor contains a pointer to the 'numeric' struct
         !
         ! this UMFPACK struct will be freed in 'Implem_msRelease'
         ! and 'Implem_msSilentRelease'
         allocate( L%umf4_ptr_numeric )

         L%umf4_ptr_numeric = numeric

      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_div( a%units(i), 2,                           &
                               L%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "msLU", "E",                          &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI msLU:) internal error: in processing physical units:"
               write(STDERR,*) "               Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
            U%units(i) = L%units(i)
         end do
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msLU_mfArray
!_______________________________________________________________________
!
   subroutine msLU_mfMatFactor( A, factor )

      type(mfArray), target :: A
      type(mfMatFactor)     :: factor
      !------ API end ------
#ifdef _DEVLP

      ! for a dense matrix, 'msLU' returns (via an mfMatFactor)
      ! the pointers to L, U, P.
      !
      ! for a sparse matrix, 'msLU' returns (via an mfMatFactor)
      ! the pointer to a complete C structure, containing all
      ! necessary data (i.e., L, U, P, Q, R)
      !
      ! this mfMatFactor is intended to be used in few routines,
      ! like 'mfLDiv', or the internal ones in 'mat_vec.inc'
      ! (reverse communication for ARPACK)

      type(mfArray), pointer :: L, U, P
      integer :: i, j, lda, info, i_tmp
      type(mfArray), pointer :: A_copy
      integer, allocatable :: ipiv(:), ipiv_new(:)

      ! declarations for UMFPACK
      integer :: m, n, nnz
      integer(kind=MF_ADDRESS) :: numeric, symbolic ! LU handle
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      integer, allocatable :: ptr_Aj(:), ptr_Ai(:)
      real(kind=MF_DOUBLE), allocatable :: Ar(:), Az(:)
      character(len=3) :: info_char
      logical :: A_copy_is_allocated

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsSymm(A) ) then
         if( mfIsPosDef(A) ) then
            call PrintMessage( "msLU", "W",                             &
                               "matrix 'A' is Symm. and Pos. Def.!",    &
                               "You should use a Cholesky factorization instead.", &
                               "[cf. the 'mfChol'/'msChol' routines]" )
         end if
      end if

      m = A%shape(1)
      n = A%shape(2)

      call msFreeMatFactor( factor )

      if( A%data_type == MF_DT_DBLE ) then

         factor%data_type = MF_DT_DBLE

         allocate( factor%mf_ptr_1 )

         L => factor%mf_ptr_1
         L%shape = [ m, min(m,n) ]
         allocate( L%double( L%shape(1), L%shape(2) ) )

         allocate( factor%mf_ptr_2 )

         U => factor%mf_ptr_2
         U%shape = [ min(m,n), n ]
         allocate( U%double( U%shape(1), U%shape(2) ) )

         allocate( factor%mf_ptr_3 )

         P => factor%mf_ptr_3
         P%shape = [ m, m ]
         P%data_type = MF_DT_DBLE ! P is always real
         allocate( P%double( P%shape(1), P%shape(2) ) )

         ! making copies because LAPACK overwrites A
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if

         if( mfIsSymm(A) ) then
            if( .not. mfIsPosDef(A) ) then
               call PrintMessage( "msLU", "I",                          &
                                  "matrix 'A' is symmetric.",           &
                                  "You could use an L*D*L' factorization instead, but", &
                                  "be aware that it ca fail (a non-singular matrix is necessary", &
                                  "but not sufficient)." )
            end if
         end if

         ! General system -- real matrix : DGETRF
         lda = m
         allocate( ipiv(min(m,n)), ipiv_new(m) )

         call dgetrf( m, n, A_copy%double(1,1), lda, ipiv(1), info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "msLU", "W",                             &
                               "in LU-factorizing matrix A by LAPACK:", &
                               "info = " // info_char,                  &
                               "(as a side effect, L and U will be empty)" )
            call msSilentRelease( L, U )
            call msAssign( P, mfEye(m) )
         else
            if( m < n ) then
               call msAssign( L, mfGet( mfTril( A_copy, -1 ), MF_COLON, &
                                        [(i,i=1,min(m,n))] ) )
            else
               call msAssign( L, mfTril( A_copy, -1 ))
            end if
            do i = 1, min(m,n)
               L%double(i,i) = 1.0d0
            end do
            if( m > n ) then
               call msAssign( U, mfGet( mfTriu( A_copy ), [(i,i=1,min(m,n))], &
                                        MF_COLON ) )
            else
               call msAssign( U, mfTriu( A_copy ))
            end if

            ! IPIV :
            ! The pivot indexes that define the permutation matrix P;
            ! row i of the matrix was interchanged with row IPIV(i).
            ipiv_new = [ (i, i=1,m) ]
            ! transformed sequence
            do i = 1, min(m,n)
               j = ipiv(i)
               if( j /= i ) then
                  ! exchange indexes (i,j) in ipiv_new
                  i_tmp = ipiv_new(i)
                  ipiv_new(i) = ipiv_new(j)
                  ipiv_new(j) = i_tmp
               end if
            end do
            P%double(:,:) = 0.0d0
            do j = 1, m
               P%double(ipiv_new(j),j) = 1.0d0
            end do
         end if

         if( A_copy_is_allocated ) then
            call msSilentRelease( A_copy )
            deallocate( A_copy ) ! no_mem_trace !
         end if

      else if( A%data_type == MF_DT_CMPLX ) then

         factor%data_type = MF_DT_CMPLX

         allocate( factor%mf_ptr_1 )
         L => factor%mf_ptr_1
         L%shape = [ m, min(m,n) ]
         allocate( L%cmplx( L%shape(1), L%shape(2) ) )

         allocate( factor%mf_ptr_2 )
         U => factor%mf_ptr_2
         U%shape = [ min(m,n), n ]
         allocate( U%cmplx( U%shape(1), U%shape(2) ) )

         allocate( factor%mf_ptr_3 )
         P => factor%mf_ptr_3
         P%shape = [ m, m ]
         P%data_type = MF_DT_DBLE ! P is always real
         allocate( P%double( P%shape(1), P%shape(2) ) )

         ! making copies because LAPACK overwrites A
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if

         ! General system -- complex matrix : ZGETRF
         lda = m
         allocate( ipiv(min(m,n)), ipiv_new(m) )

         call zgetrf( m, n, A_copy%cmplx(1,1), lda, ipiv(1), info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "msLU", "W",                             &
                               "in LU-factorizing matrix A by LAPACK:", &
                               "info = " // info_char,                  &
                               "(as a side effect, L and U will be empty)" )
            call msSilentRelease( L, U )
            call msAssign( P, mfEye(m) )
         else
            if( m < n ) then
               call msAssign( L, mfGet( mfTril( A_copy, -1 ), MF_COLON, &
                                        [(i,i=1,min(m,n))] ) )
            else
               call msAssign( L, mfTril( A_copy, -1 ) )
            end if
            do i = 1, min(m,n)
               L%cmplx(i,i) = 1.0d0
            end do
            if( m > n ) then
               call msAssign( U, mfGet( mfTriu( A_copy ), [(i,i=1,min(m,n))], &
                                        MF_COLON ) )
            else
               call msAssign( U, mfTriu( A_copy ) )
            end if

            ! IPIV :
            ! The pivot indexes that define the permutation matrix P;
            ! row i of the matrix was interchanged with row IPIV(i).
            ipiv_new = [ (i, i=1,m) ]
            ! transformed sequence
            do i = 1, min(m,n)
               j = ipiv(i)
               if( j /= i ) then
                  ! exchange indexes (i,j) in ipiv_new
                  i_tmp = ipiv_new(i)
                  ipiv_new(i) = ipiv_new(j)
                  ipiv_new(j) = i_tmp
               end if
            end do
            P%double(:,:) = 0.0d0
            do j = 1, m
               P%double(ipiv_new(j),j) = 1.0d0
            end do
         end if

         if( A_copy_is_allocated ) then
            call msSilentRelease( A_copy )
            deallocate( A_copy ) ! no_mem_trace !
         end if

      else if( A%data_type == MF_DT_SP_DBLE ) then

         ! ok for m /= n

         ! uses UMFPACK-4 (C)
         ! sparse matrices must be row-sorted !
         if( A%row_sorted /= TRUE ) then
            call msRowSort(A)
         end if

         ! set UMFPACK-4 default parameters
         call umf4def_d(control)
         ! print control parameters.
         ! Set control(1) to 1 to print error messages only
         control(1) = 1

         ! convert from 1-based to 0-based
         allocate( ptr_Aj(n+1) )
         ptr_Aj(1:n+1) = A%j(1:n+1) - 1
         nnz = A%j(n+1) - 1
         allocate( ptr_Ai(nnz) )
         ptr_Ai(1:nnz) = A%i(1:nnz) - 1

         ! pre-order and symbolic analysis
         call umf4sym_d( m, n, ptr_Aj, ptr_Ai, A%a, symbolic,           &
                         control, infos )
         ! check umf4sym error condition
         if( infos(1) < 0 ) then
            write(STDERR,*) "(MUESLI msLU:) Error occurred in umf4sym_d: ", infos(1)
            if( infos(1) == -8 ) then ! -8 : UMFPACK-version dependant !
               ! UMFPACK_ERROR_invalid_matrix (-8)
               ! (the following description has been adapted to fortran-like 1-based
               !  indexes !)
               write(STDERR,*) "Number of entries in the matrix is negative, Ap(1) is not 1,"
               write(STDERR,*) "a column has a negative number of entries, a row index is out of"
               write(STDERR,*) "bounds, or the columns of input matrix were jumbled (unsorted"
               write(STDERR,*) "columns or duplicate entries)."
            end if
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            go to 99
         end if

         ! numeric factorization
         call umf4num_d( n, ptr_Aj, ptr_Ai, A%a, symbolic, numeric,     &
                         control, infos )
         ! check umf4num error condition
         if( infos(1) < 0 ) then
            write(info_char,"(I0)") nint(infos(1))
            call PrintMessage( "msLU", "E",                             &
                               "in umf4num_d: infos(1) = " // info_char )
            go to 99
         end if
         ! check umf4num warning for singular matrix
         if( infos(UMFPACK_STATUS) == UMFPACK_WARNING_singular_matrix ) then
            UMF4NUM_SINGULAR_MATRIX = 1
         else
            UMF4NUM_SINGULAR_MATRIX = 0
         end if

         deallocate( ptr_Aj )
         deallocate( ptr_Ai )

         ! free the symbolic analysis
         call umf4fsym_d(symbolic)

         ! storing information

         factor%package = 1 ! 1 for UMFPACK
         factor%data_type = MF_DT_SP_DBLE
         factor%order = m

         ! the UMFPACK struct 'numeric' will be freed only
         ! in 'msFreeMatFactor' (under the user responsibility)
         allocate( factor%ptr_1 )

         factor%ptr_1 = numeric

      else if( A%data_type == MF_DT_SP_CMPLX ) then

         ! ok for m /= n

         ! uses UMFPACK-4 (C)
         ! sparse matrices must be row-sorted !
         if( A%row_sorted /= TRUE ) then
            call msRowSort(A)
         end if

         ! set UMFPACK-4 default parameters
         call umf4def_z(control)
         ! print control parameters.
         ! Set control(1) to 1 to print error messages only
         control(1) = 1

         ! convert from 1-based to 0-based
         allocate( ptr_Aj(n+1) )
         ptr_Aj(1:n+1) = A%j(1:n+1) - 1
         nnz = A%j(n+1) - 1
         allocate( ptr_Ai(nnz) )
         ptr_Ai(1:nnz) = A%i(1:nnz) - 1
         allocate( Ar(nnz) )
         Ar(:) = real(A%z(1:nnz))
         allocate( Az(nnz) )
         Az(:) = aimag(A%z(1:nnz))

         ! pre-order and symbolic analysis
         call umf4sym_z( m, n, ptr_Aj, ptr_Ai, Ar, Az, symbolic,        &
                         control, infos )
         ! check umf4sym error condition
         if( infos(1) < 0 ) then
            write(STDERR,*) "(MUESLI msLU:) Error occurred in umf4sym_z: ", infos(1)
            if( infos(1) == -8 ) then ! -8 : UMFPACK-version dependant !
               ! UMFPACK_ERROR_invalid_matrix (-8)
               ! (the following description has been adapted to fortran-like 1-based
               !  indexes !)
               write(STDERR,*) "Number of entries in the matrix is negative, Ap(1) is not 1,"
               write(STDERR,*) "a column has a negative number of entries, a row index is out of"
               write(STDERR,*) "bounds, or the columns of input matrix were jumbled (unsorted"
               write(STDERR,*) "columns or duplicate entries)."
            end if
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            go to 99
         end if

         ! numeric factorization
         call umf4num_z( n, ptr_Aj, ptr_Ai, Ar, Az, symbolic, numeric,  &
                         control, infos )
         ! check umf4num error condition
         if( infos(1) < 0 ) then
            write(info_char,"(I0)") nint(infos(1))
            call PrintMessage( "msLU", "E",                             &
                               "in umf4num_z: infos(1) = " // info_char )
            go to 99
         end if
         ! check umf4num warning for singular matrix
         if( infos(UMFPACK_STATUS) == UMFPACK_WARNING_singular_matrix ) then
            UMF4NUM_SINGULAR_MATRIX = 1
         else
            UMF4NUM_SINGULAR_MATRIX = 0
         end if

         deallocate( ptr_Aj, ptr_Ai )
         deallocate( Ar, Az )

         ! free the symbolic analysis
         call umf4fsym_z(symbolic)

         ! storing information

         factor%package = 1 ! 1 for UMFPACK
         factor%data_type = MF_DT_SP_CMPLX
         factor%order = m

         ! the UMFPACK struct 'numeric' will be freed only
         ! in 'msFreeMatFactor' (under the user responsibility)
         allocate( factor%ptr_1 )

         factor%ptr_1 = numeric

      else

         call PrintMessage( "msLU", "E",                                &
                            "bad data type for 'A'!" )
         go to 99

      end if

      if( mf_phys_units ) then
         factor%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msLU_mfMatFactor
