! f90 include file

!_______________________________________________________________________
!
   function mfLDiv_two_mf( A, b, form ) result( out )

      type(mfArray)              :: A, b
      character(len=*), optional :: form

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! driver routine for the case where two mfArray are in the
      ! argument's list.

      !    form : "cholesky"

      ! routines called:
      !   mfLDiv_A( A, b )
      !   mfLDiv_SPD( U, b )

   !------ end of declarations -- execution starts hereafter  ------

      if( present(form) ) then
         if( to_lower(form) /= "cholesky" ) then
            call PrintMessage( "mfLDiv", "E",                           &
                               "optional argument 'form' must be equal to 'cholesky'!" )
         end if
         call msAssign( out, mfLDiv_SPD( A, b ) )
      else ! form not present
         call msAssign( out, mfLDiv_A( A, b ) )
      end if

      out%status_temporary = .true.

#endif
   end function mfLDiv_two_mf
!_______________________________________________________________________
!
   function mfLDiv_three_mf( L, U, b ) result( out )

      type(mfArray) :: L, U, b

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! solve linear system of the form : L U x = b
      ! i.e. compute : y = L \ b
      !         then : x = U \ y

      ! handles only the SPARSE case (UMFPack)
      !
      ! the 'numeric' C structure of UMFPack contains all necessary
      ! data (i.e., L, U, P, Q, R)
      !
      ! L and U are passed as argument, but actually are not used !
      !
      ! L%umf4_ptr_numeric is a pointer to the previous C structure

      ! declarations for UMFPACK
      integer(kind=MF_ADDRESS) :: numeric ! LU handle
      integer :: sys
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      real(kind=MF_DOUBLE), pointer :: ptr_b(:) => null(),              &
                                       ptr_x(:) => null()
      real(kind=MF_DOUBLE), pointer :: ptr_bz(:) => null(),             &
                                       ptr_xz(:) => null()

      integer :: i, n, ldb, nrhs
      character(len=4) :: info_char
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( L, U, b )

      ! verifications...
      if( L%data_type == MF_DT_EMPTY .or.                               &
          U%data_type == MF_DT_EMPTY .or.                               &
          b%data_type == MF_DT_EMPTY      ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'L', 'U' or 'b' is not allocated!" )
         go to 99
      end if

      n = L%shape(1)
      if( L%shape(2) /= n ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'L' must be square!" )
         go to 99
      end if

      if( any(U%shape /= [n,n]) ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'U' must have same size than 'L'!" )
         go to 99
      end if

      ldb = b%shape(1)
      nrhs = b%shape(2)

      if( ldb /= n ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'b' has a wrong size!" )
         go to 99
      end if

      if( L%data_type == MF_DT_SP_DBLE .and.                            &
          b%data_type == MF_DT_DBLE          ) then

         if( nrhs /= 1 ) then
            call PrintMessage( "mfLDiv", "E",                           &
                               "for a sparse system, only NRHS=1 is valid!" )
            go to 99
         end if

         if( associated( L%umf4_ptr_numeric ) ) then
            out%data_type = MF_DT_DBLE
            out%shape = [ n, 1 ]
            allocate( out%double(n,1) )

            ! set UMFPACK-4 default parameters
            call umf4def_d(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! solve A.x=b, without iterative refinement
            sys = 0
            numeric = L%umf4_ptr_numeric
            call umf4sol_d( sys, out%double(:,1), b%double(:,1), numeric, &
                            control, infos )
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfLDiv", "E",                        &
                                  "in umf4sol_d: infos(1) = " // info_char )
               go to 99
            end if

         else
            call PrintMessage( "mfLDiv", "E",                           &
                               "'L' and 'U' factors passed to this routine must have", &
                               "been obtained with the 'msLU' routine!" )
            go to 99
         end if

      else if( L%data_type == MF_DT_SP_CMPLX .and.                      &
               b%data_type == MF_DT_CMPLX          ) then

         if( nrhs /= 1 ) then
            call PrintMessage( "mfLDiv", "E",                           &
                               "for a sparse system, only NRHS=1 is valid!" )
            go to 99
         end if

         if( associated( L%umf4_ptr_numeric ) ) then
            out%data_type = MF_DT_CMPLX
            out%shape = [ n, 1 ]
            allocate( out%cmplx(n,1) )

            ! set UMFPACK-4 default parameters
            call umf4def_z(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            ! solve A.x=b, without iterative refinement
            sys = 0
            numeric = L%umf4_ptr_numeric

            allocate( ptr_x(n), ptr_xz(n) )

            allocate( ptr_b(n), ptr_bz(n) )

            ptr_b(:)  =   real(b%cmplx(:,1))
            ptr_bz(:) =  aimag(b%cmplx(:,1))
            call umf4sol_z( sys, ptr_x, ptr_xz, ptr_b, ptr_bz,          &
                            numeric, control, infos )
            if( infos(1) < 0 ) then
               write(info_char,"(I0)") nint(infos(1))
               call PrintMessage( "mfLDiv", "E",                        &
                                  "in umf4sol_d: infos(1) = " // info_char )
               go to 99
            end if

            out%cmplx(:,1) = cmplx(ptr_x(:),ptr_xz(:),kind=MF_DOUBLE)
            deallocate( ptr_x, ptr_xz, ptr_b, ptr_bz )

         else
            call PrintMessage( "mfLDiv", "E",                           &
                               "'L' and 'U' factors passed to this routine must have", &
                               "been obtained with the 'msLU' routine!" )
            go to 99
         end if

      else

         call PrintMessage( "mfLDiv", "E",                              &
                            "bad data type!" )
         go to 99

      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( b%units(i), L%units(i),                  &
                               out%units(i), status )
            call rational_sub( out%units(i), U%units(i),                &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfLDiv", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfLDiv:) internal error: in processing physical units:"
               write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( L, U, b )

      call msAutoRelease( L, U, b )

#endif
   end function mfLDiv_three_mf
!_______________________________________________________________________
!
   function mfLDiv_four_mf( L, U, p, b, option ) result( out )

      type(mfArray)              :: L, U, p, b
      character(len=*), optional :: option

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! solve linear system : A x = b     where      A = p L U
      ! i.e. compute : y = L \ (p' b)
      !         then : x = U \ y
      !
      !                  or : A' x = b    (if option="transp")
      ! i.e. compute : y = U' \ b
      !         then : z = L' \ y
      !     and last : x = p z

      ! handles only the DENSE case (LAPACK)
      ! handles mixed data type

      ! p is a integer permutation vector (from 2.6.0)

      integer :: i, n, ldb, nrhs
      integer :: status
      character(len=10) :: char_case
      type(mfArray) :: tmp
      logical :: transpose

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( L, U, p, b )

      if( present(option) ) then
         if( option == "transp" ) then
            transpose = .true.
         else
            call PrintMessage( "mfLDiv", "E",                           &
                               "option arg must be equal to ""transp""!" )
            go to 99
         end if
      else
         transpose = .false.
      end if

      ! verifications...
      if( L%data_type == MF_DT_EMPTY .or.                               &
          U%data_type == MF_DT_EMPTY .or.                               &
          p%data_type == MF_DT_EMPTY .or.                               &
          b%data_type == MF_DT_EMPTY      ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'L', 'U', 'p' or 'b' is not allocated!" )
         go to 99
      end if

      n = L%shape(1)
      if( L%shape(2) /= n ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'L' must be square!" )
         go to 99
      end if

      if( any(U%shape /= [n,n]) ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'U' must have same size than 'L'!" )
         go to 99
      end if

      if( p%shape(1) /= n ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "permutation 'p' must have same size than 'L'!" )
         go to 99
      end if

      ldb = b%shape(1)
      nrhs = b%shape(2)

      if( ldb /= n ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'b' has a wrong size!" )
         go to 99
      end if

      if( transpose ) then
         if( B%level_protected == 1 ) then
            call msAssign( out, B )
         else
            out = B
         end if
      else
         ! apply row interchanges to 'b'
         call msAssign( out, mfRowPerm( b, p ) )
      end if

      ! prepare for mixte data type
      if( L%data_type == MF_DT_DBLE .and.                               &
          b%data_type == MF_DT_DBLE       ) then
         char_case = "real"
      else if( L%data_type == MF_DT_CMPLX .and.                         &
               b%data_type == MF_DT_CMPLX       ) then
         char_case = "complex"
      else if( L%data_type == MF_DT_CMPLX .and.                         &
               b%data_type == MF_DT_DBLE       ) then

         call msAssign( out, mfComplex(out) )
         char_case = "complex"
      else if( L%data_type == MF_DT_DBLE .and.                          &
               b%data_type == MF_DT_CMPLX       ) then
         char_case = "complex2"
      end if

      select case( char_case )
      case( "real" )
         if( transpose ) then
            ! solve U' y = out, overwriting out with y
            call dtrsm( 'left', 'upper', 'transpose', 'non-unit', n, nrhs, &
                        1.0d0, U%double(1,1), n, out%double(1,1), ldb )
            ! solve L' z = out, overwriting out with z
            call dtrsm( 'left', 'lower', 'transpose', 'unit', n, nrhs,     &
                        1.0d0, L%double(1,1), n, out%double(1,1), ldb )
         else
            ! solve L y = out, overwriting out with y
            call dtrsm( 'left', 'lower', 'no transpose', 'unit', n, nrhs,  &
                        1.0d0, L%double(1,1), n, out%double(1,1), ldb )
            ! solve U x = out, overwriting out with x
            call dtrsm( 'left', 'upper', 'no transpose', 'non-unit', n, nrhs, &
                        1.0d0, U%double(1,1), n, out%double(1,1), ldb )
         end if
      case( "complex" )
         if( transpose ) then
            ! solve U' y = out, overwriting out with y
            call ztrsm( 'left', 'upper', 'conjg transpose', 'non-unit', n, nrhs, &
                        (1.0d0,0.0d0), U%cmplx(1,1), n, out%cmplx(1,1), ldb )
            ! solve L' z = out, overwriting out with z
            call ztrsm( 'left', 'lower', 'conjg transpose', 'unit', n, nrhs,     &
                        (1.0d0,0.0d0), L%cmplx(1,1), n, out%cmplx(1,1), ldb )
         else
            ! solve L y = out, overwriting out with y
            call ztrsm( 'left', 'lower', 'no transpose', 'unit', n, nrhs,  &
                        (1.0d0,0.0d0), L%cmplx(1,1), n, out%cmplx(1,1), ldb )
            ! solve U x = out, overwriting out with x
            call ztrsm( 'left', 'upper', 'no transpose', 'non-unit', n, nrhs, &
                        (1.0d0,0.0d0), U%cmplx(1,1), n, out%cmplx(1,1), ldb )
         end if
      case( "complex2" )
         call msAssign( tmp, mfComplex(L) )
         if( transpose ) then
            ! solve U' y = out, overwriting out with y
            call ztrsm( 'left', 'upper', 'conjg transpose', 'non-unit', n, nrhs, &
                        (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )
            ! solve L' z = out, overwriting out with z
            call msAssign( tmp, mfComplex(L) )
            call ztrsm( 'left', 'lower', 'conjg transpose', 'unit', n, nrhs,     &
                        (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )
         else
            ! solve L y = out, overwriting out with y
            call ztrsm( 'left', 'lower', 'no transpose', 'unit', n, nrhs,  &
                        (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )
            ! solve U x = out, overwriting out with x
            call msAssign( tmp, mfComplex(U) )
            call ztrsm( 'left', 'upper', 'no transpose', 'non-unit', n, nrhs, &
                        (1.0d0,0.0d0), tmp%cmplx(1,1), n, out%cmplx(1,1), ldb )
         end if
         call msSilentRelease( tmp )
      end select

      if( transpose ) then
         ! apply row interchanges
         call msAssign( out, mfRowPerm( out, mfInvPerm(p) ) )
      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( b%units(i), L%units(i),                  &
                               out%units(i), status )
            call rational_sub( out%units(i), U%units(i),                &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfLDiv", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfLDiv:) internal error: in processing physical units:"
               write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( L, U, p, b )

      call msAutoRelease( L, U, p, b )

#endif
   end function mfLDiv_four_mf
!_______________________________________________________________________
!
   function mfLDiv_mfMatFactor( factor, b, option ) result( out )

      type(mfMatFactor)          :: factor
      type(mfArray)              :: b
      character(len=*), optional :: option

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! solve linear system of the form : A x = b
      !    or, if option is present and
      !               equal to "transp" : A' x = b
      !
      ! sparse matrix A is provided via its factors;
      ! mfArray b may have multiple columns and, for the SPD case
      ! (i.e. using CHOLMOD), may be sparse.

      ! handles only the SPARSE case (SuiteSparse)
      !
      ! 'factor' contains C structure, either from LU factorization
      ! (i.e., L, U, P, Q, R from UMFPack), either from Cholesky
      ! factorization (i.e., L and other from CHOLMOD)

      ! declarations for UMFPACK
      integer(kind=MF_ADDRESS) :: numeric ! LU handle
      integer :: sys
      real(kind=MF_DOUBLE) :: control(20), infos(90)
      real(kind=MF_DOUBLE), pointer :: ptr_b(:) => null(),              &
                                       ptr_x(:) => null()
      real(kind=MF_DOUBLE), pointer :: ptr_bz(:) => null(),             &
                                       ptr_xz(:) => null()
      integer(kind=MF_UF_LONG), allocatable :: Wi(:)
      real(kind=MF_DOUBLE), allocatable :: W(:)
      complex(kind=MF_DOUBLE), allocatable :: ctmp(:,:)

      integer :: i, n, ldb, nrhs, bnz, xnz
      character(len=4) :: info_char
      integer(kind=MF_ADDRESS) :: c_addr, L_addr, out_addr
      integer :: status
      logical :: transposed

   !------ end of declarations -- execution starts hereafter  ------

      ! verifications...
      if( .not. associated(factor%ptr_1) ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'factor' is not allocated!",              &
                            "(this factor must have been obtained via the 'mfLU' routine!" )
         go to 99
      end if

      if( b%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'b' is not allocated!" )
         go to 99
      end if

      n = factor%order
      ldb  = b%shape(1)
      nrhs = b%shape(2)

      if( ldb /= n ) then
         call PrintMessage( "mfLDiv", "E",                              &
                            "'b' has a wrong size!" )
         go to 99
      end if

      if( present(option) ) then
         if( option /= "transp" ) then
            call PrintMessage( "mfLDiv", "E",                           &
                               "option arg must be equal to ""transp""!" )
            go to 99
         end if
         transposed = .true.
      else
         transposed = .false.
      end if

      if( factor%data_type == MF_DT_SP_DBLE .and.                       &
          b%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_DBLE
         out%shape = [ n, nrhs ]
         allocate( out%double(n,nrhs) )

         select case( factor%package )
         case( 1 ) ! UMFPACK

            ! set UMFPACK-4 default parameters
            call umf4def_d(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            if( transposed ) then
               ! solve A'.x=b, without iterative refinement
               sys = 1
            else
               ! solve A.x=b, without iterative refinement
               sys = 0
            end if
            numeric = factor%ptr_1

            allocate( Wi(n), W(n) )

            do i = 1, nrhs
               call umf4wsol_d( sys, out%double(:,i), b%double(:,i), numeric, &
                                control, infos, Wi, W )
               if( infos(1) < 0 ) then
                  write(info_char,"(I0)") nint(infos(1))
                  call PrintMessage( "mfLDiv", "E",                     &
                                     "in umf4wsol_d: infos(1) = " // info_char )
                  go to 99
               end if
            end do

         case( 2, 4 ) ! CHOLMOD

            ! here A is (at least) symmetric,
            ! hence: A.x=b or A'.x=b is the same problem
            ! solve L.L'.x=b, without iterative refinement
            c_addr = factor%ptr_1
            L_addr = factor%ptr_2
            if( nrhs == 1 ) then
               call cholmod_solve_factor( n, c_addr, L_addr, b%double(:,1), &
                                          out%double(:,1) )
            else
               call cholmod_solve_factor_nRHS( n, c_addr, L_addr, nrhs, b%double(:,:), &
                                               out%double(:,:) )
            end if

         case default
            write(STDERR,*) "(MUESLI mfLDiv:) internal error: bad package number in factor"
            write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop

         end select

      else if( factor%data_type == MF_DT_SP_DBLE .and.                  &
               b%data_type == MF_DT_SP_DBLE ) then

         out%data_type = MF_DT_SP_DBLE
         out%shape = [ n, nrhs ]

         select case( factor%package )
         case( 1 ) ! UMFPACK

!### TODO 2:
stop 'not yet available'

         case( 2, 4 ) ! CHOLMOD

            ! here A is (at least) symmetric,
            ! hence: A.x=b or A'.x=b is the same problem
            ! solve L.L'.x=b, without iterative refinement
            c_addr = factor%ptr_1
            L_addr = factor%ptr_2
            bnz = b%j(nrhs+1) - 1
            call cholmod_spsol_fact_nRHS_prep( n, c_addr, L_addr,       &
                                               nrhs, bnz, b%j, b%i, b%a, &
                                               out_addr, xnz )
            allocate( out%j(nrhs+1) )

            allocate( out%i(xnz) )

            allocate( out%a(xnz) )

            ! now, can get sparse solution 'out' ...
            call cholmod_get_sol_nrhs( c_addr, out_addr, n, nrhs, xnz,  &
                                       out%j, out%i, out%a )

            ! All matrices returned by SuiteSparse are "row sorted"
            out%row_sorted = TRUE

         case default
            write(STDERR,*) "(MUESLI mfLDiv:) internal error: bad package number in factor"
            write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop

         end select

      else if( factor%data_type == MF_DT_SP_CMPLX .and.                 &
               b%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_CMPLX
         out%shape = [ n, nrhs ]
         allocate( out%cmplx(n,nrhs) )

         select case( factor%package )
         case( 1 ) ! UMFPACK

            ! set UMFPACK-4 default parameters
            call umf4def_z(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            if( transposed ) then
               ! solve A'.x=b, without iterative refinement
               sys = 1
            else
               ! solve A.x=b, without iterative refinement
               sys = 0
            end if
            numeric = factor%ptr_1

            allocate( ptr_x(n), ptr_xz(n) )

            allocate( ptr_b(n), ptr_bz(n) )

            ptr_bz(:) = 0.0d0
            allocate( Wi(n), W(4*n) )

            do i = 1, nrhs
               ptr_b(:) = b%double(:,i)
               call umf4wsol_z( sys, ptr_x, ptr_xz, ptr_b, ptr_bz,      &
                                numeric, control, infos, Wi, W )
               if( infos(1) < 0 ) then
                  write(info_char,"(I0)") nint(infos(1))
                  call PrintMessage( "mfLDiv", "E",                     &
                                     "in umf4wsol_z: infos(1) = " // info_char )
                  go to 99
               end if
               out%cmplx(:,i) = cmplx(ptr_x(:),ptr_xz(:),kind=MF_DOUBLE)
            end do

            deallocate( ptr_x, ptr_xz, ptr_b, ptr_bz )

         case( 2, 4 ) ! CHOLMOD

            ! here A is (at least) hermitian,
            ! hence: A.x=b or A'.x=b is the same problem
            ! solve L.L'.x=b, without iterative refinement
            c_addr = factor%ptr_1
            L_addr = factor%ptr_2
            allocate( ctmp(n,nrhs) )

            ctmp(:,:) = b%double(:,:)
            if( nrhs == 1 ) then
               call cholmod_cmplx_solve_factor( n, c_addr, L_addr, ctmp(:,1), &
                                                out%cmplx(:,1) )
            else
               call cholmod_cmplx_solve_factor_nRHS( n, c_addr, L_addr, nrhs, ctmp(:,:), &
                                                     out%cmplx(:,:) )
            end if

         case default
            write(STDERR,*) "(MUESLI mfLDiv:) internal error: bad package number in factor"
            write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop

         end select

      else if( factor%data_type == MF_DT_SP_CMPLX .and.                 &
               b%data_type == MF_DT_CMPLX ) then

         out%data_type = MF_DT_CMPLX
         out%shape = [ n, nrhs ]
         allocate( out%cmplx(n,nrhs) )

         select case( factor%package )
         case( 1 ) ! UMFPACK

            ! set UMFPACK-4 default parameters
            call umf4def_z(control)
            ! print control parameters.
            ! Set control(1) to 1 to print error messages only
            control(1) = 1

            if( transposed ) then
               ! solve A'.x=b, without iterative refinement
               sys = 1
            else
               ! solve A.x=b, without iterative refinement
               sys = 0
            end if
            numeric = factor%ptr_1

            allocate( ptr_x(n), ptr_xz(n) )

            allocate( ptr_b(n), ptr_bz(n) )

            allocate( Wi(n), W(4*n) )

            do i = 1, nrhs
               ptr_b(:)  =   real(b%cmplx(:,i))
               ptr_bz(:) =  aimag(b%cmplx(:,i))
               call umf4wsol_z( sys, ptr_x, ptr_xz, ptr_b, ptr_bz,      &
                                numeric, control, infos, Wi, W )
               if( infos(1) < 0 ) then
                  write(info_char,"(I0)") nint(infos(1))
                  call PrintMessage( "mfLDiv", "E",                     &
                                     "in umf4wsol_z: infos(1) = " // info_char )
                  go to 99
               end if
               out%cmplx(:,i) = cmplx(ptr_x(:),ptr_xz(:),kind=MF_DOUBLE)
            end do

            deallocate( ptr_x, ptr_xz, ptr_b, ptr_bz )

         case( 2, 4 ) ! CHOLMOD

            ! here A is (at least) hermitian,
            ! hence: A.x=b or A'.x=b is the same problem
            ! solve L.L'.x=b, without iterative refinement
            c_addr = factor%ptr_1
            L_addr = factor%ptr_2
            if( nrhs == 1 ) then
               call cholmod_cmplx_solve_factor( n, c_addr, L_addr, b%cmplx(:,1), &
                                                out%cmplx(:,1) )
            else
               call cholmod_cmplx_solve_factor_nRHS( n, c_addr, L_addr, nrhs, b%cmplx(:,:), &
                                                     out%cmplx(:,:) )
            end if

         case default
            write(STDERR,*) "(MUESLI mfLDiv:) internal error: bad package number in factor"
            write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
            mf_message_displayed = .true.
            call muesli_trace( pause="yes" )
            stop

         end select

      else

         call PrintMessage( "mfLDiv", "E",                              &
                            "bad data type!" )
         go to 99

      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( b%units(i), factor%units(i),             &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfLDiv", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfLDiv:) internal error: in processing physical units:"
               write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msAutoRelease( b )

#endif
   end function mfLDiv_mfMatFactor
