! f90 include file

!_______________________________________________________________________
!
   subroutine msFunm_intr( out, A, fun )

      type(mfArray) :: A
      character(len=*) :: fun
      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! for intrinsic functions

      ! Parlett's method.  See Golub and VanLoan (1983), p. 384.
      ! (from Matlab-6.5.2 : funm.m)

      type(mfArray), pointer :: F, E

      type(mfArray) :: Q, T
      integer :: i, n, p, l, j
      integer, allocatable :: k(:)
      real(kind=MF_DOUBLE) :: dmin, esterr
      complex(kind=MF_DOUBLE) :: s, d
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! 2 out-args must be specified
      if( out%n /= 2 ) then
         call PrintMessage( "msFunm", "E",                              &
                            "two output args required!",               &
                            "syntax is : call msFunm ( mfOut(F,E), A, fun )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msFunm", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      F => out%ptr1
      E => out%ptr2
      call msSilentRelease( F, E )

      if( len_trim(fun) == 0 ) then
         call PrintMessage( "msFunm", "E",                              &
                            "function 'fun' is not defined!" )
         go to 99
      end if

      ! special cases for which predefined functions exist
      select case( trim(fun) )
         case( 'exp' )
            call PrintMessage( "msFunm", "I",                           &
                               "when 'fun=exp', the mfExpm is called;", &
                               "as a consequence, the error estimation is not available!" )
            call msAssign( F, mfExpm( A ))
            call msSilentRelease( E )      ! estim. error is not available
            return
      end select

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msFunm", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "msFunm", "E",                              &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msFunm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msFunm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "msFunm", "E",                              &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      n = A%shape(1)
      ! square matrix ?
      if( A%shape(2) /= n ) then
         call PrintMessage( "msFunm", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      call msSchur( mfOut(Q,T), A )
      call rsf2csf( Q, T )
!### TODO 2: strange: why this second form is less efficient?
!!call msSchur( mfOut(Q,T), A, form="complex" )

      ! a priori complexe !
      call msAssign( F, mfZeros(n,n)*MF_I )

      ! warning : in Fortran, functions
      !           {tan,log10,acos,asin,atan,cosh,sinh,tanh}
      !           are not defined for complex values
      select case( trim(fun) )
         case( 'acos' )
            do i = 1, n
               F%cmplx(i,i) = mf_gsl_complex_arccos( T%cmplx(i,i) )
            end do
         case( 'asin' )
            do i = 1, n
               F%cmplx(i,i) = mf_gsl_complex_arcsin( T%cmplx(i,i) )
            end do
         case( 'atan' )
            do i = 1, n
               F%cmplx(i,i) = mf_gsl_complex_arctan( T%cmplx(i,i) )
            end do
         case( 'cos' )
            do i = 1, n
               F%cmplx(i,i) = cos( T%cmplx(i,i) )
            end do
         case( 'cosh' )
            do i = 1, n
               F%cmplx(i,i) = mf_gsl_complex_cosh( T%cmplx(i,i) )
            end do
         case( 'log' )
            do i = 1, n
               F%cmplx(i,i) = log( T%cmplx(i,i) )
            end do
         case( 'log10' )
            do i = 1, n
               F%cmplx(i,i) = log( T%cmplx(i,i) ) / MF_LN_10
            end do
         case( 'sin' )
            do i = 1, n
               F%cmplx(i,i) = sin( T%cmplx(i,i) )
            end do
         case( 'sinh' )
            do i = 1, n
               F%cmplx(i,i) = mf_gsl_complex_sinh( T%cmplx(i,i) )
            end do
         case( 'sqrt' )
            do i = 1, n
               F%cmplx(i,i) = sqrt( T%cmplx(i,i) )
            end do
         case( 'tan' )
            do i = 1, n
               F%cmplx(i,i) = mf_gsl_complex_tan( T%cmplx(i,i) )
            end do
         case( 'tanh' )
            do i = 1, n
               F%cmplx(i,i) = mf_gsl_complex_tanh( T%cmplx(i,i) )
            end do
         case default
            call PrintMessage( "msFunm", "E",                           &
                               "function:" // trim(fun),                &
                               "is not an intrinsic!",                 &
                               "(or it is not implemented.)" )
            call msSilentRelease(F)
            go to 99
      end select

      dmin = abs( T%cmplx(1,1) )
      do p = 1, n-1
         do i = 1, n-p
            j = i + p
            s = T%cmplx(i,j)*(F%cmplx(j,j)-F%cmplx(i,i))
            if( p > 1 ) then
               allocate( k(j-1-(i+1)+1) )

               k = [ (l+i, l = 1, j-1-(i+1)+1) ]
               s = s + mfCmplx( mfMul( mfGet(T,[i],k), mfGet(F,k,[j]) ) ) &
                     - mfCmplx( mfMul( mfGet(F,[i],k), mfGet(T,k,[j]) ) )
               deallocate( k )

            end if
            d = T%cmplx(j,j) - T%cmplx(i,i)
            if( d /= (0.0d0,0.0d0) ) then
               s = s / d
            end if
            F%cmplx(i,j) = s
            dmin = min( dmin, abs(d) )
         end do
      end do

      call msAssign( F, mfMul( Q, Mfmul(F,.h.Q) ))

#if defined _GNU_GFC
! Bug ? gfortran-4.8.1 a du mal à désallouer proprement certains variables
!       (détecté par Valgrind)
      if( mfIsReal(A) ) then
         if( mfDble(mfNorm(mfImag(F),1)) <= 10*n*MF_EPS*1000 ) then
            call msAssign( F, mfReal(F) )
         end if
      end if
#else
      if( mfIsReal(A) .and.                                             &
         mfDble(mfNorm(mfImag(F),1)) <= 10*n*MF_EPS*1000 ) then
         call msAssign( F, mfReal(F) )
      end if
#endif

      if( dmin == 0 ) then
         dmin = MF_EPS
      end if

      call msAssign( Q, mfTriu(T,1))

      esterr = mfDble( mfNorm(Q) )
      esterr = min(1.0d0,max(MF_EPS,(MF_EPS*dmin)*esterr))

      if( Any( .not. mfIsFinite(F) ) ) then
         esterr = MF_INF
      end if

      call msAssign( E, mf( esterr ))

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "msFunm", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!",               &
                               "(because the exact function used cannot be detected)" )
            go to 99
         end if
      end if

 99   continue

      call msSilentRelease( Q, T )

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msFunm_intr
!_______________________________________________________________________
!
   subroutine msFunm_user( out, A, fun )

      type(mfArray) :: A

      interface
         function fun(z1) result(z2)
            import :: MF_DOUBLE
            complex(kind=MF_DOUBLE) :: z1, z2
         end function fun
      end interface

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! for user functions

      ! Parlett's method.  See Golub and VanLoan (1983), p. 384.
      ! (from Matlab-6.5.2 : funm.m)

      type(mfArray), pointer :: F, E

      type(mfArray) :: Q, T
      integer :: i, n, p, l, j
      integer, allocatable :: k(:)
      real(kind=MF_DOUBLE) :: dmin, esterr
      complex(kind=MF_DOUBLE) :: s, d
      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! 2 out-args must be specified
      if( out%n /= 2 ) then
         call PrintMessage( "msFunm", "E",                              &
                            "two output args required!",               &
                            "syntax is : call msFunm ( mfOut(F,E), A, fun )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msFunm", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      F => out%ptr1
      E => out%ptr2
      call msSilentRelease( F, E )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msFunm", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "msFunm", "E",                              &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msFunm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msFunm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "msFunm", "E",                              &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      n = A%shape(1)
      ! square matrix ?
      if( A%shape(2) /= n ) then
         call PrintMessage( "msFunm", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      call msSchur( mfOut(Q,T), A )
      call rsf2csf( Q, T )
!### TODO 2: strange: why this second form is less efficient?
!!call msSchur( mfOut(Q,T), A, form="complex" )

      ! a priori complexe !
      call msAssign( F, mfZeros(n,n)*MF_I)
      do i = 1, n
         F%cmplx(i,i) = fun( T%cmplx(i,i) )
      end do

      dmin = abs( T%cmplx(1,1) )
      do p = 1, n-1
         do i = 1, n-p
            j = i + p
            s = T%cmplx(i,j)*(F%cmplx(j,j)-F%cmplx(i,i))
            if( p > 1 ) then
               allocate( k(j-1-(i+1)+1) )

               k = [ (l+i, l = 1, j-1-(i+1)+1) ]
               s = s + mfCmplx( mfMul( mfGet(T,[i],k), mfGet(F,k,[j]) ) )  &
                     - mfCmplx( mfMul( mfGet(F,[i],k), mfGet(T,k,[j]) ) )
               deallocate( k )

            end if
            d = T%cmplx(j,j) - T%cmplx(i,i)
            if( d /= (0.0d0,0.0d0) ) then
               s = s / d
            end if
            F%cmplx(i,j) = s
            dmin = min( dmin, abs(d) )
         end do
      end do

      call msAssign( F, mfMul( Q, Mfmul(F,.h.Q) ))

#if defined _GNU_GFC
! Bug ? gfortran-4.8.1 a du mal à désallouer proprement certains variables
!       (détecté par Valgrind)
      if( mfIsReal(A) ) then
         if( All( mfNorm(mfImag(F),1) <= 10*n*MF_EPS*mfNorm(F,1) ) ) then
            call msAssign( F, mfReal(F) )
         end if
      end if
#else
      if( mfIsReal(A) .and.                                             &
          All( mfNorm(mfImag(F),1) <= 10*n*MF_EPS*mfNorm(F,1) ) ) then
         call msAssign( F, mfReal(F) )
      end if
#endif

      if( dmin == 0 ) then
         dmin = MF_EPS
      end if

      call msAssign( Q, mfTriu(T,1))

      esterr = mfDble( mfNorm(Q))
      esterr = min(1.0d0,max(MF_EPS,(MF_EPS*dmin)*esterr))

      if( Any( .not. mfIsFinite(F) ) ) then
         esterr = MF_INF
      end if

      call msAssign( E, mf( esterr ))

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "msFunm", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!",               &
                               "(because the exact function used cannot be detected)" )
            go to 99
         end if
      end if

 99   continue

      call msSilentRelease( Q, T )

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msFunm_user
!_______________________________________________________________________
!
   function mfFunm_intr( A, fun ) result ( out )

      type(mfArray) :: A
      character(len=*) :: fun
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! for intrinsic functions

      ! Parlett's method.  See Golub and VanLoan (1983), p. 384.
      ! (from Matlab-6.5.2 : funm.m)

      type(mfArray) :: Q, T
      integer :: i, n, p, l, j
      integer, allocatable :: k(:)
      real(kind=MF_DOUBLE) :: dmin, esterr
      complex(kind=MF_DOUBLE) :: s, d
      integer :: status
      character(len=12) :: esterr_char

   !------ end of declarations -- execution starts hereafter  ------

      if( len_trim(fun) == 0 ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "function 'fun' is not defined!" )
         go to 99
      end if

      ! special cases for which predefined functions exist
      select case( trim(fun) )
         case( 'exp' )
            call msAssign( out, mfExpm( A ))
            out%status_temporary = .true.
            return
      end select

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "msFunm", "E",                              &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      n = A%shape(1)
      ! square matrix ?
      if( A%shape(2) /= n ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      call msSchur( mfOut(Q,T), A )
      call rsf2csf( Q, T )
!### TODO 2: strange: why this second form is less efficient?
!!call msSchur( mfOut(Q,T), A, form="complex" )

      ! a priori complexe !
      call msAssign( out, mfZeros(n,n)*MF_I )

      ! warning : in Fortran, functions
      !           {tan,log10,acos,asin,atan,cosh,sinh,tanh}
      !           are not defined for complex values
      select case( trim(fun) )
         case( 'acos' )
            do i = 1, n
               out%cmplx(i,i) = mf_gsl_complex_arccos( T%cmplx(i,i) )
            end do
         case( 'asin' )
            do i = 1, n
               out%cmplx(i,i) = mf_gsl_complex_arcsin( T%cmplx(i,i) )
            end do
         case( 'atan' )
            do i = 1, n
               out%cmplx(i,i) = mf_gsl_complex_arctan( T%cmplx(i,i) )
            end do
         case( 'cos' )
            do i = 1, n
               out%cmplx(i,i) = cos( T%cmplx(i,i) )
            end do
         case( 'cosh' )
            do i = 1, n
               out%cmplx(i,i) = mf_gsl_complex_cosh( T%cmplx(i,i) )
            end do
         case( 'log' )
            do i = 1, n
               out%cmplx(i,i) = log( T%cmplx(i,i) )
            end do
         case( 'log10' )
            do i = 1, n
               out%cmplx(i,i) = log( T%cmplx(i,i) ) / MF_LN_10
            end do
         case( 'sin' )
            do i = 1, n
               out%cmplx(i,i) = sin( T%cmplx(i,i) )
            end do
         case( 'sinh' )
            do i = 1, n
               out%cmplx(i,i) = mf_gsl_complex_sinh( T%cmplx(i,i) )
            end do
         case( 'sqrt' )
            do i = 1, n
               out%cmplx(i,i) = sqrt( T%cmplx(i,i) )
            end do
         case( 'tan' )
            do i = 1, n
               out%cmplx(i,i) = mf_gsl_complex_tan( T%cmplx(i,i) )
            end do
         case( 'tanh' )
            do i = 1, n
               out%cmplx(i,i) = mf_gsl_complex_tanh( T%cmplx(i,i) )
            end do
         case default
            call PrintMessage( "mfFunm", "E",                           &
                               "function:" // trim(fun),                &
                               "is not an intrinsic!",                 &
                               "(or it is not implemented.)" )
            call msSilentRelease(out)
            go to 99
      end select

      dmin = abs( T%cmplx(1,1) )
      do p = 1, n-1
         do i = 1, n-p
            j = i + p
            s = T%cmplx(i,j)*(out%cmplx(j,j)-out%cmplx(i,i))
            if( p > 1 ) then
               allocate( k(j-1-(i+1)+1) )

               k = [ (l+i, l = 1, j-1-(i+1)+1) ]
               s = s + mfCmplx( mfMul( mfGet(T,[i],k), mfGet(out,k,[j]) ) )  &
                     - mfCmplx( mfMul( mfGet(out,[i],k), mfGet(T,k,[j]) ) )
               deallocate( k )

            end if
            d = T%cmplx(j,j) - T%cmplx(i,i)
            if( d /= (0.0d0,0.0d0) ) then
               s = s / d
            end if
            out%cmplx(i,j) = s
            dmin = min( dmin, abs(d) )
         end do
      end do

      call msAssign( out, mfMul( Q, Mfmul(out,.h.Q) ))

#if defined _GNU_GFC
! Bug ? gfortran-4.8.1 a du mal à désallouer proprement certains variables
!       (détecté par Valgrind)
      if( mfIsReal(A) ) then
         if( All( mfNorm(mfImag(out),1) <= 10*n*MF_EPS*mfNorm(out,1) ) ) then
            call msAssign( out, mfReal(out) )
         endif
      end if
#else
      if( mfIsReal(A) .and.                                             &
          All( mfNorm(mfImag(out),1) <= 10*n*MF_EPS*mfNorm(out,1) ) ) then
         call msAssign( out, mfReal(out) )
      end if
#endif

      if( dmin == 0 ) then
         dmin = MF_EPS
      end if

      call msAssign( Q, mfTriu(T,1))

      esterr = mfDble( mfNorm(Q))
      esterr = min(1.0d0,max(MF_EPS,(MF_EPS*dmin)*esterr))

      if( Any( .not. mfIsFinite(out) ) ) then
         esterr = MF_INF
      end if

      if( esterr > 1.0d+3*MF_EPS ) then
         write(esterr_char,"(E10.3)") esterr
         call PrintMessage( "mfFunm", "W",                              &
                            "Result may be inaccurate!",               &
                            "Estim. Error = " // esterr_char )
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfFunm", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!",               &
                               "(because the exact function used cannot be detected)" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%status_temporary = .true.

 99   continue

      call msSilentRelease( Q, T )

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfFunm_intr
!_______________________________________________________________________
!
   function mfFunm_user( A, fun ) result ( out )

      type(mfArray) :: A

      interface
         function fun(z1) result(z2)
            import :: MF_DOUBLE
            complex(kind=MF_DOUBLE) :: z1, z2
         end function fun
      end interface

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! for user functions

      ! Parlett's method.  See Golub and VanLoan (1983), p. 384.
      ! (from Matlab-6.5.2 : funm.m)

      type(mfArray) :: Q, T
      integer :: i, n, p, l, j
      integer, allocatable :: k(:)
      real(kind=MF_DOUBLE) :: dmin, esterr
      complex(kind=MF_DOUBLE) :: s, d
      integer :: status
      character(len=12) :: esterr_char

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      n = A%shape(1)
      ! square matrix ?
      if( A%shape(2) /= n ) then
         call PrintMessage( "mfFunm", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      call msSchur( mfOut(Q,T), A )
      call rsf2csf( Q, T )
!### TODO 2: strange: why this second form is less efficient?
!!call msSchur( mfOut(Q,T), A, form="complex" )

      ! a priori complexe !
      call msAssign( out, mfZeros(n,n)*MF_I )
      do i = 1, n
         out%cmplx(i,i) = fun( T%cmplx(i,i) )
      end do

      dmin = abs( T%cmplx(1,1) )
      do p = 1, n-1
         do i = 1, n-p
            j = i + p
            s = T%cmplx(i,j)*(out%cmplx(j,j)-out%cmplx(i,i))
            if( p > 1 ) then
               allocate( k(j-1-(i+1)+1) )

               k = [ (l+i, l = 1, j-1-(i+1)+1) ]
               s = s + mfCmplx( mfMul( mfGet(T,[i],k), mfGet(out,k,[j]) ) )  &
                     - mfCmplx( mfMul( mfGet(out,[i],k), mfGet(T,k,[j]) ) )
               deallocate( k )

            end if
            d = T%cmplx(j,j) - T%cmplx(i,i)
            if( d /= (0.0d0,0.0d0) ) then
               s = s / d
            end if
            out%cmplx(i,j) = s
            dmin = min( dmin, abs(d) )
         end do
      end do

      call msAssign( out, mfMul( Q, Mfmul(out,.h.Q) ))

#if defined _GNU_GFC
! Bug ? gfortran-4.8.1 a du mal à désallouer proprement certains variables
!       (détecté par Valgrind)
      if( mfIsReal(A) ) then
         if( All( mfNorm(mfImag(out),1) <= 10*n*MF_EPS*mfNorm(out,1) ) ) then
            call msAssign( out, mfReal(out) )
         end if
      end if
#else
      if( mfIsReal(A) .and.                                             &
          All( mfNorm(mfImag(out),1) <= 10*n*MF_EPS*mfNorm(out,1) ) ) then
         call msAssign( out, mfReal(out) )
      end if
#endif

      if( dmin == 0 ) then
         dmin = MF_EPS
      end if

      call msAssign( Q, mfTriu(T,1))

      esterr = mfDble( mfNorm(Q))
      esterr = min(1.0d0,max(MF_EPS,(MF_EPS*dmin)*esterr))

      if( Any( .not. mfIsFinite(out) ) ) then
         esterr = MF_INF
      end if

      if( esterr > 1.0d+3*MF_EPS ) then
         write(esterr_char,"(E10.3)") esterr
         call PrintMessage( "mfFunm", "W",                              &
                            "Result may be inaccurate!",               &
                            "Estim. Error = " // esterr_char )
      end if

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfFunm", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!",               &
                               "(because the exact function used cannot be detected)" )
            go to 99
         end if
      end if

      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%status_temporary = .true.

 99   continue

      call msSilentRelease( Q, T )

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfFunm_user
!_______________________________________________________________________
!
   subroutine rsf2csf( U, T )

      type(mfArray) :: U, T
      !------ API end ------
#ifdef _DEVLP

      ! Real Schur Form to Complex Schur Form.
      ! (from Matlab-7.5 :rsf2csf.m)
      !
      ! U and T are modified !

      ! This version has been very optimized by working directly
      ! on internal arrays of mfArrays.
      ! (got a speed-up factor from 2 to 3)
      ! Old version was before muesli-2.0.0_2008_12_01

      integer :: n, m
      integer :: k(2)
      real(kind=MF_DOUBLE) :: r, mu_imag_1, a11, a12, a21, a22, c2
      complex(kind=MF_DOUBLE) :: c, s, G(2,2), GH(2,2)

   !------ end of declarations -- execution starts hereafter  ------

      call msAssign( T, mfComplex(T))
      call msAssign( U, mfComplex(U))
      ! hereafter, T and U are dense, complex mfArrays.
      n = Size(T,2)

      do m = n, 2, -1
         if( T%cmplx(m,m-1) /= (0.0d0,0.0d0) ) then
            ! from Maple
            a11 = dble( T%cmplx(m-1,m-1) )
            a12 = dble( T%cmplx(m-1,m) )
            a21 = dble( T%cmplx(m,m-1) )
            a22 = dble( T%cmplx(m,m) )
            ! the following expression must be negative...
            c2 = (a22-a11)**2 + 4*a21*a12
            c2 = sqrt( -c2 )
            mu_imag_1 = c2/2.0d0
            k = [ m-1, m ]
            r = hypot( mu_imag_1, real(T%cmplx(m,m-1)) )
            c = cmplx( 0.0d0, mu_imag_1/r, kind=MF_DOUBLE )
            s = T%cmplx(m,m-1)/r
            G(1,1) = conjg(c)
            G(1,2) = s
            G(2,1) = -s
            G(2,2) = c
            GH(1,1) = c
            GH(1,2) = -s
            GH(2,1) = s
            GH(2,2) = conjg(c)

            T%cmplx(m-1:m,m-1:n) = matmul( G(:,:), T%cmplx(m-1:m,m-1:n) )
            T%cmplx(1:m,m-1:m) = matmul( T%cmplx(1:m,m-1:m), GH(:,:) )
            U%cmplx(:,m-1:m) = matmul( U%cmplx(:,m-1:m), GH(:,:) )
            T%cmplx(m,m-1) = (0.0d0,0.0d0)
         end if
      end do

#endif
   end subroutine rsf2csf
