! f90 include file

!_______________________________________________________________________
!
   function mfExpm( A ) result( out )

      type(mfArray), target :: A
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Matrix exponential via Padé approximation.
      ! (from Matlab-6.5.2 : expm.m)

      type(mfArray), pointer :: A_copy
      type(mfArray) :: f, e
      type(mfArray) :: X, D, cX
      integer :: k, s, q
      logical :: p, A_copy_is_allocated
      real(kind=MF_DOUBLE) :: c

      integer :: status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfExpm", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfExpm", "E",                              &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfExpm", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfExpm", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( A%shape(1) <= 1 .or. A%shape(2) <= 1 ) then
         call PrintMessage( "mfExpm", "E",                              &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      ! square matrix ?
      if( A%shape(1) /= A%shape(2) ) then
         call PrintMessage( "mfExpm", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      ! Scale A by power of 2 so that its norm is < 1/2
      call msLog2(mfOut(f,e),mfNorm(A,"inf"))
      s = max( 0, int(mfDble(e))+1 )
      if( s == 0 ) then
         if( A%status_temporary .and. (A%level_protected==1) ) then
            A_copy => A
            A_copy_is_allocated = .false.
         else
            allocate( A_copy ) ! no_mem_trace !
            A_copy = A
            A_copy_is_allocated = .true.
         end if
      else
         allocate( A_copy ) ! no_mem_trace !
         call msAssign( A_copy, A / 2.0d0**s )
         A_copy_is_allocated = .true.
      end if

      ! Padé approximation for exp(A)
      X = A_copy
      c = 0.5d0
      call msAssign( out, mfEye( Size(A_copy,1), Size(A_copy,2) ) + c*A_copy)
      call msAssign( D, mfEye( Size(A_copy,1), Size(A_copy,2) ) - c*A_copy)
      q = 6
      p = .true.
      do k = 2, q
         c = c*(q-k+1)/(k*(2*q-k+1))
         call msAssign( X, mfMul(A_copy,X))
         call msAssign( cX, c*X)
         call msAssign( out, out + cX)
         if( p ) then
            call msAssign( D, D + cX)
         else
            call msAssign( D, D - cX)
         end if
         p = .not. p
      end do
      call msAssign( out, mfLDiv(D,out))

      ! Undo scaling by repeated squaring
      do k = 1, s
         call msAssign( out, mfMul(out,out))
      end do

      if( mf_phys_units ) then
         ! verifying the physical dimension
         call verif_adim( a%units, status=status )
         if( status /= 0 ) then
            call PrintMessage( "mfExpm", "E",                           &
                               "the physical unit of the mfArray",      &
                               "must be dimensionless!" )
            go to 99
         end if
      end if

      ! don't apply the 'symmetry' property: in theory, the exponential
      ! of a symmetric matrix is also symmetric, but the numerical method
      ! used leads to a small unsymmetric component...
      out%prop%tril = A%prop%tril
      out%prop%triu = A%prop%triu
      out%status_temporary = .true.

      call msSilentRelease( f, e, X, D, cX )
      if( A_copy_is_allocated ) then
         call msSilentRelease( A_copy )
         deallocate( A_copy ) ! no_mem_trace !
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfExpm
