! f90 include file

! currently, uses only:
!     mf_mat_vec()
!     mf_mat_vec_cmplx()
!     mf_mat_vec_inv_factor()
!     mf_mat_vec_inv_factor_cmplx()
!     mf_mat_vec_inv_chol()
!     mf_mat_vec_inv_chol_cmplx()

!_______________________________________________________________________
!
   function mfEigs_which( A, k, which, tol, ncv ) result( out )

      type(mfArray)                              :: A
      integer,              intent(in)           :: k
      character(len=2),     intent(in), optional :: which
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      integer,              intent(in), optional :: ncv

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! equivalent of 'eigs' in MATLAB (simplified version)
      !
      ! returns a vector containing 'k' eigenvalues of A.
      !
      ! A may be : real/complex, sparse/dense
      !
      ! which = 'LM' : Largest Magnitude   | (all types of matrices)
      !         'SM' : Smallest Magnitude  |
      !         ----------------------------------------------------------
      !         'LA' : Largest Algebraic   |
      !         'SA' : Smallest Algebraic  | (only for real, symmetric)
      !         'BE' : Both End            |
      !         ---------------------------
      !         'LR' : Largest Real part   |
      !         'SR' : Smallest Real part  | (not real, or not symmetric)
      !         'LI' : Largest Imag part   |
      !         'SI' : Smallest Imag part  |

      integer :: n, ncol, nnz
      logical :: finished

      ! declarations for ARPACK
      integer :: ido, nev, nev0, ncv0, ldv, lworkl, info, ldz, i, nconv
      real(kind=MF_DOUBLE) :: tol0
      real(kind=MF_DOUBLE), allocatable :: resid(:), v(:,:)
      real(kind=MF_DOUBLE), allocatable :: workd(:), workl(:), workev(:)
      real(kind=MF_DOUBLE), allocatable :: dr(:), di(:), d(:)
      real(kind=MF_DOUBLE), allocatable :: rwork(:)
      integer :: iparam(11), ipntr(14)
      logical, allocatable :: select(:)
      real(kind=MF_DOUBLE) :: z(1,1), sigmar, sigmai, sigma
      character(len=2) :: which0, which1
      complex(kind=MF_DOUBLE), allocatable :: zresid(:), zv(:,:), zd(:)
      complex(kind=MF_DOUBLE), allocatable :: zworkd(:), zworkl(:), zworkev(:)
      integer :: ldzv, lzworkl, ldzz
      complex(kind=MF_DOUBLE) :: zz(1,1), zsigma
      character(len=6) :: info_char
      character(len=12) :: method

      type(mfMatFactor) :: factor
      type(mfArray) :: U
      logical :: A_is_SPD

      ! new sorting of eigenvalues
      type(mfArray) :: mf_dummy, mf_ind, mf_tmp
      integer, allocatable :: ind(:)
      character(len=10) :: mode_sort

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfEigs", "W",                              &
                            "'A' is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfEigs", "E",                              &
                            "'A' cannot be boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfEigs", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      n = A%shape(1)

      ! square matrix ?
      if( n /= A%shape(2) ) then
         call PrintMessage( "mfEigs", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      if( k < 2 ) then
         call PrintMessage( "mfEigs", "E",                              &
                            "k must be greater or equal 2!" )
         go to 99
      else if( n-2 < k ) then
         call PrintMessage( "mfEigs", "W",                              &
                            "the ARPACK routine used allows only the computation", &
                            "of n-2 max eigenvalues/vectors!",         &
                            "(use mfEig to retrieve all eigenvalues/vectors)" )
         nev = n-2
      else
         nev = k
      end if

      if( mfIsSparse(A) ) then
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         if( nnz == 0 ) then
            call msAssign( out, mfZeros(k,1) )
            out%status_temporary = .true.
            go to 99
         end if
      end if

      if( present(which) ) then
         which0 = to_upper(which)
      else
         which0 = "LM" ! Default : Largest Eigenvalues
      end if

      ! check 'which' argument
      if( which0 == "LM" .or. which0 == "SM" .or.                       &
          which0 == "LA" .or. which0 == "SA" .or.                       &
          which0 == "BE" .or.                                           &
          which0 == "LR" .or. which0 == "SR" .or.                       &
          which0 == "LI" .or. which0 == "SI" ) then
         ! do nothing
      else
         call PrintMessage( "mfEigs", "E",                              &
                            "'which' arg must be equal to 'LM', 'SM', 'LA', 'SA', 'BE',", &
                            "'LR', 'SR', 'LI' or 'SI'",                 &
                            "(see documentation)" )
         go to 99
      end if

      if( mfIsSymm(A) ) then

         if( mfIsReal(A) ) then

            ! real, symmetric matrices : 'LR', 'SR', 'LI' or 'SI' are not valid
            if( which0 == "LR" .or. which0 == "SR" .or.                 &
                which0 == "LI" .or. which0 == "SI" ) then
               call PrintMessage( "mfEigs", "E",                        &
                                  "'which' arg may be equal to 'LR', 'SR', 'LI' or 'SI'", &
                                  " only for complex or unsymmetric real matrices!" )
               go to 99
            end if

         else

            ! complex, symmetric matrices, then eigenvalues are real.
            ! It is not useful to use the imaginary part for sorting
            if( which0 == "LI" .or. which0 == "SI" ) then
               call PrintMessage( "mfEigs", "W",                        &
                                  "The matrix A is symmetric, then the eigenvalues are real.", &
                                  "So, it is not useful to use the imaginary part for sorting" )
            end if

            ! not real matrices : 'LA', 'SA', 'BE' are not valid
            if( which0 == "LA" .or. which0 == "SA" .or.                 &
                which0 == "BE" ) then
               call PrintMessage( "mfEigs", "E",                        &
                                  "'which' arg may be equal to 'LA', 'SA' or 'BE'", &
                                  " only for real, symmetric matrices!" )
               go to 99
            end if

         end if

      else

         ! not symmetric matrices : 'LA', 'SA', 'BE' are not valid
         if( which0 == "LA" .or. which0 == "SA" .or.                    &
             which0 == "BE" ) then
            call PrintMessage( "mfEigs", "E",                           &
                               "'which' arg may be equal to 'LA', 'SA' or 'BE'", &
                               " only for real, symmetric matrices!" )
            go to 99
         end if

      end if

      ! choice of OP

!### TODO 2: (i)   pourquoi ne pas choisir un seuil à sqrt(eps) ?
!            (ii)  test de même nature pour "SA" ?
!            (iii) pourquoi ne pas tester si A est SPD ?
      if( which0 == "SM" ) then
         ! Warn if matrix is close to singular
         if( mfDble(mfRCond(A)) <= MF_EPS*10.0d0 ) then
            call PrintMessage( "mfEigs", "W",                           &
                               "matrix 'A' is close to singular",       &
                               "Computation can be very inaccurate!" )
            method = "OP = A"
            which1 = "SM"
         else
            method = "OP = inv(A)"
            which1 = "LM"
         end if
      else
         method = "OP = A"
         which1 = which0
      end if

      ! relative accuracy of the Ritz values
      if( present(tol) ) then
         tol0 = tol
      else
         tol0 = MF_EPS
      end if

      ! how many Arnoldi vectors are generated
      if( present(ncv) ) then
         ncv0 = ncv
      else
         ncv0 = 2*nev+1 ! recommended value from User Guide
      end if

      if( ncv0 < 2*nev+1 ) then
         call PrintMessage( "mfEigs", "W",                              &
                            "ARPACK: ncv < 2*k+1",                      &
                            "(2*k+1 is the recommended value from the ARPACK User Guide)", &
                            "-> setting : ncv = 2*k+1" )
         ncv0 = 2*nev+1
      end if

      if( ncv0 > n ) then
         if( present(ncv) ) then
            call PrintMessage( "mfEigs", "W",                           &
                               "ARPACK: ncv > n",                       &
                               "It seems that you are working with a small matrix, or", &
                               "that a great number of eigs have been requested...", &
                               "(ARPACK works well when computing few eigenvalues", &
                               "of a large matrix)",                    &
                               "-> setting : ncv = n" )
         end if
         ncv0 = n
      end if

      ! using ARPACK-2

      if( present(ncv) ) then

         if( A%data_type == MF_DT_DBLE .or.                             &
             A%data_type == MF_DT_SP_DBLE ) then

            ! real matrix

            if( mfIsSymm(A) ) then

               ! symmetric case, eigenvalues are all real

               if( method == "OP = inv(A)" ) then
                  if( mfIsPosDef(A) ) then
                     A_is_SPD = .true.
                     call msAssign(U, mfChol(A) )
                  else
                     A_is_SPD = .false.
                     call msLU_mfMatFactor( A, factor )
                  end if
               end if

               !-------------------------------------
               ! preparation
               !-------------------------------------
               ido = 0 ! first call to the reverse communication interface
               allocate( resid(n) ) ! residual vector
               allocate( v(n,ncv0) ) ! set of Arnoldi basis vectors
               ldv = n
               allocate( workd(3*n) ) ! distributed array for reverse comm.
               lworkl = ncv0**2 + 8*ncv0
               allocate( workl(lworkl) ) ! private workspace

               !-------------------------------------
               ! computes eigenvalues and eigenvectors
               !-------------------------------------
               iparam(:) = 0
               iparam(1) = 1 ! method for selecting the implicit shifts
               iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
               if( method == "OP = A" ) then
                  iparam(7) = 1 ! type of eigenproblem : OP = A
               else if( method == "OP = inv(A)" ) then
                  iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
               end if
               info = 0 ! a randomly initial residual vector
               finished = .false.
               do while( .not. finished )
                  call dsaupd( ido,                                     &
                               "I", n, which1, nev, tol0, resid, ncv0, v, ldv, &
                               iparam, ipntr, workd, workl, lworkl, info )
                  if( ido == 1 .or. ido == -1 ) then
                     if( method == "OP = A" ) then
                        call mf_mat_vec( A, workd(ipntr(1):ipntr(1)+n-1), &
                                            workd(ipntr(2):ipntr(2)+n-1) )
                     else if( method == "OP = inv(A)" ) then
                        if( A_is_SPD ) then
                           call mf_mat_vec_inv_chol( U, workd(ipntr(1):ipntr(1)+n-1), &
                                                        workd(ipntr(2):ipntr(2)+n-1) )
                        else
                           call mf_mat_vec_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                               workd(ipntr(2):ipntr(2)+n-1) )
                        end if
                     end if
                  else if( ido == 99 ) then
                     finished = .true.
                  end if
               end do

               if( method == "OP = inv(A)" ) then
                  call msFreeMatFactor( factor )
               end if
               if( .not. mfIsEmpty(U) ) then
                  call msSilentRelease( U )
               end if

               ! check for some errors
               if( info < 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfEigs", "E",                     &
                                     "'dsaupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               ! check for convergence
               nconv = iparam(5)
               if( nconv /= nev ) then
                  call PrintMessage( "mfEigs", "W",                     &
                                     "ARPACK: nconv /= nev",            &
                                     "Number of converged Ritz values is not equal", &
                                     "to the number of requested eigenvalues." )
               end if

               !-------------------------------------
               ! no fatal errors occurred
               ! post processing
               !-------------------------------------
               ! .false. : don't want the Ritz vectors
               ! 'A' : compute all the NEV Ritz values
               allocate( select(ncv0) ) ! for the present call : workspace
               allocate( d(nev) ) ! vector of the Ritz values
               ldz = 1
               sigma = 0.0d0
               call dseupd( .false., "A", select, d, z, ldz, sigma,     &
                            "I", n, which1, nev, tol0, resid, ncv0, v, ldv, &
                            iparam, ipntr, workd, workl, lworkl, info )

               if( info == -14 ) then
                  call PrintMessage( "mfEigs", "E",                     &
                                     "ARPACK:",                         &
                                     "dsaupd didn't find any eigenvalues to sufficient accuracy", &
                                     "-> try to increase ncv" )
                  go to 99
               end if

               ! check for some errors
               if( info /= 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfEigs", "E",                     &
                                     "'dseupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               out%data_type = MF_DT_DBLE
               out%shape = [ nev, 1 ]
               allocate( out%double(nev,1) )

               ! non-converged eigenvalues will contain NaN values
               out%double(:,:) = MF_NAN

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LA" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  call msAssign( mf_tmp , mf(abs(d)) )
               else
                  call msEquiv( d, mf_tmp )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev) )
               ind = mf_ind
               do i = 1, min(nev,nconv) ! limited to the converged ritz values!
                  out%double(i,1) = d(ind(i))
               end do

            else

               ! non-symmetric case, eigenvalues are complex

               if( method == "OP = inv(A)" ) then
                  call msLU_mfMatFactor( A, factor )
               end if

               !-------------------------------------
               ! preparation
               !-------------------------------------
               ido = 0 ! first call to the reverse communication interface
               allocate( resid(n) ) ! residual vector
               allocate( v(n,ncv0) ) ! set of Arnoldi basis vectors
               ldv = n
               allocate( workd(3*n) ) ! distributed array for reverse comm.
               lworkl = 3*ncv0**2 + 6*ncv0
               allocate( workl(lworkl) ) ! private workspace

               !-------------------------------------
               ! calculs valeurs et vecteurs propres
               !-------------------------------------
               iparam(:) = 0
               iparam(1) = 1 ! method for selecting the implicit shifts
               iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
               if( method == "OP = A" ) then
                  iparam(7) = 1 ! type of eigenproblem : OP = A
               else if( method == "OP = inv(A)" ) then
                  iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
               end if
               info = 0 ! a randomly initial residual vector
               finished = .false.
               do while( .not. finished )
                  call dnaupd( ido,                                     &
                               "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                               iparam, ipntr, workd(1), workl(1), lworkl, info )
                  if( ido == 1 .or. ido == -1 ) then
                     if( method == "OP = A" ) then
                        call mf_mat_vec( A, workd(ipntr(1):ipntr(1)+n-1), &
                                            workd(ipntr(2):ipntr(2)+n-1) )
                  else if( method == "OP = inv(A)" ) then
                        call mf_mat_vec_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                            workd(ipntr(2):ipntr(2)+n-1) )
                  end if
                  else if( ido == 99 ) then
                     finished = .true.
                  end if
               end do

               if( method == "OP = inv(A)" ) then
                  call msFreeMatFactor( factor )
               end if

               ! check for some errors
               if( info < 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfEigs", "E",                     &
                                     "'dnaupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               ! check for convergence
               nconv = iparam(5)
               if( nconv /= nev ) then
                  call PrintMessage( "mfEigs", "W",                     &
                                     "ARPACK: nconv /= nev",            &
                                     "Number of converged Ritz values is not equal", &
                                     "to the number of requested eigenvalues." )
               end if

               !-------------------------------------
               ! no fatal errors occurred
               ! post processing
               !-------------------------------------
               ! .false. : don't want the Ritz vectors
               ! 'A' : compute all the NEV Ritz values
               allocate( select(ncv0) ) ! for the present call : workspace
               allocate( dr(nev+1) ) ! real part of the Ritz values
               allocate( di(nev+1) ) ! imag part of the Ritz values
               ! init. to avoid error detection with valgrind; moreover,
               ! NaN values are useful because they will be ignored by the sort.
               dr(:) = MF_NAN
               di(:) = MF_NAN
               ldz = 1
               allocate( workev(3*ncv0) ) ! private workspace
               sigmar = 0.0d0
               sigmai = 0.0d0
               nev0 = nev
               call dneupd( .false., "A", select(1), dr(1), di(1), z, ldz, &
                            sigmar, sigmai, workev(1),                  &
                            "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, info )
               nev = nev0

               if( info == -14 ) then
                  call PrintMessage( "mfEigs", "E",                     &
                                     "ARPACK:",                         &
                                     "dnaupd didn't find any eigenvalues to sufficient accuracy", &
                                     "-> try to increase ncv" )
                  go to 99
               end if

               ! check for some errors
               if( info /= 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfEigs", "E",                     &
                                     "'dneupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               out%data_type = MF_DT_CMPLX
               out%shape = [ nev, 1 ]
               allocate( out%cmplx(nev,1) )

               ! non-converged eigenvalues will contain NaN values
               out%cmplx(:,:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(dr**2+di**2) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msEquiv( dr, mf_tmp )
               else
                  ! magnitude imaginary part
                  call msAssign( mf_tmp, mf(abs(di)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, min(nev,nconv) ! limited to the converged ritz values!
                  out%cmplx(i,1) = cmplx( dr(ind(i)), di(ind(i)), kind=MF_DOUBLE )
               end do

            end if

         else if( A%data_type == MF_DT_CMPLX .or.                       &
                  A%data_type == MF_DT_SP_CMPLX ) then

            ! complex matrix

            ! for complex matrices, ARPACK deals only with one driver,
            ! and doesn't take care whether A is symmetric or not.

            if( method == "OP = inv(A)" ) then
               if( mfIsSymm(A) ) then
                  if( mfIsPosDef(A) ) then
                     A_is_SPD = .true.
                  else
                     A_is_SPD = .false.
                  end if
               else
                  A_is_SPD = .false.
               end if
               if( A_is_SPD ) then
                  call msAssign(U, mfChol(A) )
               else
                  call msLU_mfMatFactor( A, factor )
               end if
            end if

            !-------------------------------------
            ! preparation
            !-------------------------------------
            ido = 0 ! first call to the reverse communication interface
            allocate( zresid(n) ) ! residual vector
            allocate( zv(n,ncv0) ) ! set of Arnoldi basis vectors
            ldzv = n
            allocate( zworkd(3*n) ) ! distributed array for reverse comm.
            lzworkl = 3*ncv0**2 + 5*ncv0
            allocate( zworkl(lzworkl) ) ! private workspace
            allocate( rwork(ncv0) ) ! private workspace

            !-------------------------------------
            ! computes eigenvalues and eigenvectors
            !-------------------------------------
            iparam(:) = 0
            iparam(1) = 1 ! method for selecting the implicit shifts
            iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
            if( method == "OP = A" ) then
               iparam(7) = 1 ! type of eigenproblem : OP = A
            else if( method == "OP = inv(A)" ) then
               iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
            end if
            info = 0 ! a randomly initial residual vector
            finished = .false.
            do while( .not. finished )
               call znaupd( ido,                                        &
                            "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                            iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                            info )
               if( ido == 1 .or. ido == -1 ) then
                  if( method == "OP = A" ) then
                     call mf_mat_vec_cmplx( A, zworkd(ipntr(1):ipntr(1)+n-1), &
                                               zworkd(ipntr(2):ipntr(2)+n-1) )
                  else if( method == "OP = inv(A)" ) then
                     if( A_is_SPD ) then
                        call mf_mat_vec_inv_chol_cmplx( U, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                           zworkd(ipntr(2):ipntr(2)+n-1) )
                     else
                        call mf_mat_vec_inv_factor_cmplx( factor, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                                  zworkd(ipntr(2):ipntr(2)+n-1) )
                     end if
                  end if
               else if( ido == 99 ) then
                  finished = .true.
               end if
            end do

            if( method == "OP = inv(A)" ) then
               call msFreeMatFactor( factor )
            end if
            if( .not. mfIsEmpty(U) ) then
               call msSilentRelease( U )
            end if

            ! check for some errors
            if( info < 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( "mfEigs", "E",                        &
                                  "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               go to 99
            end if

            ! check for convergence
            nconv = iparam(5)
            if( nconv /= nev ) then
               call PrintMessage( "mfEigs", "W",                        &
                                  "ARPACK: nconv /= nev",               &
                                  "Number of converged Ritz values is not equal", &
                                  "to the number of requested eigenvalues." )
            end if

            !-------------------------------------
            ! no fatal errors occurred
            ! post processing
            !-------------------------------------
            ! .false. : don't want the Ritz vectors
            ! 'A' : compute all the NEV Ritz values
            allocate( select(ncv0) ) ! for the present call : workspace
            allocate( zd(nev+1) ) ! Ritz values
            ! init. to avoid error detection with valgrind; moreover,
            ! NaN values are useful because they will be ignored by the sort.
            zd(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
            ldzz = 1
            allocate( zworkev(2*ncv0) ) ! private workspace
            zsigma = (0.0d0,0.0d0)
            call zneupd( .false., "A", select(1), zd(1), zz, ldzz, zsigma, zworkev(1), &
                         "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                         iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                         info )

            if( info == -14 ) then
               call PrintMessage( "mfEigs", "W",                        &
                                  "ARPACK:",                            &
                                  "znaupd didn't find any eigenvalues to sufficient accuracy", &
                                  "-> try to increase ncv" )
               go to 99
            end if

            ! check for some errors
            if( info /= 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( "mfEigs", "E",                        &
                                  "'zneupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               go to 99
            end if

            if( mfIsSymm(A) ) then

               out%data_type = MF_DT_DBLE
               out%shape = [ nev, 1 ]
               allocate( out%double(nev,1) )

               ! non-converged eigenvalues will contain NaN values
               out%double(:,:) = MF_NAN

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(zd)) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msAssign( mf_tmp, mf(real(zd)) )
               else
                  ! algebraic imaginary part
                  call msAssign( mf_tmp, mf(aimag(zd)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, min(nev,nconv) ! limited to the converged ritz values!
                  out%double(i,1) = real( zd(ind(i)) )
               end do

            else

               out%data_type = MF_DT_CMPLX
               out%shape = [ nev, 1 ]
               allocate( out%cmplx(nev,1) )

               ! non-converged eigenvalues will contain NaN values
               out%cmplx(:,:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(zd)) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msAssign( mf_tmp, mf(real(zd)) )
               else
                  ! algebraic imaginary part
                  call msAssign( mf_tmp, mf(aimag(zd)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, min(nev,nconv) ! limited to the converged ritz values!
                  out%cmplx(i,1) = zd(ind(i))
               end do

            end if

         end if

      else ! .not. present(ncv)

         ! ncv chosen automatically
         ! computation starts with ncv0, and this value is increased
         ! until convergence of all singular values...

         if( A%data_type == MF_DT_DBLE .or.                             &
             A%data_type == MF_DT_SP_DBLE ) then

            ! real matrix

            if( mfIsSymm(A) ) then

               ! symmetric case, eigenvalues are all real

               allocate( resid(n) ) ! residual vector
               ldv = n
               allocate( workd(3*n) ) ! distributed array for reverse comm.
               info = 0 ! a randomly initial residual vector

               if( method == "OP = inv(A)" ) then
                  if( mfIsPosDef(A) ) then
                     A_is_SPD = .true.
                     call msAssign( U, mfChol(A) )
                  else
                     A_is_SPD = .false.
                     call msLU_mfMatFactor( A, factor )
                  end if
               end if

               do

                  !-------------------------------------
                  ! preparation
                  !-------------------------------------
                  ido = 0 ! first call to the reverse communication interface
                  allocate( v(n,ncv0) ) ! set of Arnoldi basis vectors
                  lworkl = ncv0**2 + 8*ncv0
                  allocate( workl(lworkl) ) ! private workspace

                  !-------------------------------------
                  ! computes eigenvalues and eigenvectors
                  !-------------------------------------
                  iparam(:) = 0
                  iparam(1) = 1 ! method for selecting the implicit shifts
                  iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
                  if( method == "OP = A" ) then
                     iparam(7) = 1 ! type of eigenproblem : OP = A
                  else if( method == "OP = inv(A)" ) then
                     iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
                  end if
                  finished = .false.
                  do while( .not. finished )
                     call dsaupd( ido,                                  &
                                  "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                                  iparam, ipntr, workd(1), workl(1), lworkl, info )
                     if( ido == 1 .or. ido == -1 ) then
                        if( method == "OP = A" ) then
                           call mf_mat_vec( A, workd(ipntr(1):ipntr(1)+n-1), &
                                               workd(ipntr(2):ipntr(2)+n-1) )
                        else if( method == "OP = inv(A)" ) then
                           if( A_is_SPD ) then
                              call mf_mat_vec_inv_chol( U, workd(ipntr(1):ipntr(1)+n-1), &
                                                           workd(ipntr(2):ipntr(2)+n-1) )
                           else
                              call mf_mat_vec_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                                  workd(ipntr(2):ipntr(2)+n-1) )
                           end if
                        end if
                     else if( ido == 99 ) then
                        finished = .true.
                     end if
                  end do

                  ! check for some errors
                  if( info < 0 ) then
                     write(info_char,"(I0)") info
                     call PrintMessage( "mfEigs", "E",                  &
                                        "'dsaupd' (ARPACK) cannot compute eigenvalues", &
                                        "Error returned, info = " // info_char, &
                                        "Check the ARPACK documentation" )
                     go to 99
                  end if

                  ! check for convergence
                  nconv = iparam(5)
                  if( nconv >= nev ) then
                     exit
                  end if

                  ! we cannot increase further ncv0
                  if( ncv0 == n ) then
                     call PrintMessage( "mfEigs", "E",                  &
                                        "'dsaupd' (ARPACK) cannot compute eigenvalues", &
                                        "(even after increase of ncv to its maximum value)" )
                     go to 99
                  end if
                  ncv0 = 2*ncv0
                  if( ncv0 > n ) then
                     ncv0 = n
                  end if
                  deallocate( v )
                  deallocate( workl )

                  info = 1 ! re-uses the residual vector

               end do

               if( method == "OP = inv(A)" ) then
                  call msFreeMatFactor( factor )
               end if
               if( .not. mfIsEmpty(U) ) then
                  call msSilentRelease( U )
               end if

               !-------------------------------------
               ! no fatal errors occurred
               ! post processing
               !-------------------------------------
               ! .false. : don't want the Ritz vectors
               ! 'A' : compute all the NEV Ritz values
               allocate( select(ncv0) ) ! for the present call : workspace
               allocate( d(nev) ) ! vector of the Ritz values
               ldz = 1
               sigma = 0.0d0
               call dseupd( .false., "A", select(1), d(1), z(1,1), ldz, sigma, &
                            "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, info )

               if( info == -14 ) then
                  call PrintMessage( "mfEigs", "E",                     &
                                     "ARPACK:",                         &
                                     "dsaupd didn't find any eigenvalues to sufficient accuracy" )
                  go to 99
               end if

               ! check for some errors
               if( info /= 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfEigs", "E",                     &
                                     "'dseupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               out%data_type = MF_DT_DBLE
               out%shape = [ nev, 1 ]
               allocate( out%double(nev,1) )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LA" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(d)) )
               else
                  ! algebraic
                  call msEquiv( d, mf_tmp )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev) )
               ind = mf_ind
               do i = 1, nev
                  out%double(i,1) = d(ind(i))
               end do

            else

               ! non-symmetric case, eigenvalues are complex

               allocate( resid(n) ) ! residual vector
               ldv = n
               allocate( workd(3*n) ) ! distributed array for reverse comm.
               info = 0 ! a randomly initial residual vector

               if( method == "OP = inv(A)" ) then
                  call msLU_mfMatFactor( A, factor )
               end if

               do

                  !-------------------------------------
                  ! preparation
                  !-------------------------------------
                  ido = 0 ! first call to the reverse communication interface
                  allocate( v(n,ncv0) ) ! set of Arnoldi basis vectors
                  lworkl = 3*ncv0**2 + 6*ncv0
                  allocate( workl(lworkl) ) ! private workspace

                  !-------------------------------------
                  ! calculs valeurs et vecteurs propres
                  !-------------------------------------
                  iparam(:) = 0
                  iparam(1) = 1 ! method for selecting the implicit shifts
                  iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
                  if( method == "OP = A" ) then
                     iparam(7) = 1 ! type of eigenproblem : OP = A
                  else if( method == "OP = inv(A)" ) then
                     iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
                  end if
                  finished = .false.
                  do while( .not. finished )
                     call dnaupd( ido,                                  &
                                  "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                                  iparam, ipntr, workd(1), workl(1), lworkl, info )
                     if( ido == 1 .or. ido == -1 ) then
                        if( method == "OP = A" ) then
                           call mf_mat_vec( A, workd(ipntr(1):ipntr(1)+n-1), &
                                               workd(ipntr(2):ipntr(2)+n-1) )
                        else if( method == "OP = inv(A)" ) then
                           call mf_mat_vec_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                               workd(ipntr(2):ipntr(2)+n-1) )
                        end if
                     else if( ido == 99 ) then
                        finished = .true.
                     end if
                  end do

                  ! check for some errors
                  if( info < 0 ) then
                     write(info_char,"(I0)") info
                     call PrintMessage( "mfEigs", "E",                  &
                                        "'dnaupd' (ARPACK) cannot compute eigenvalues", &
                                        "Error returned, info = " // info_char, &
                                        "Check the ARPACK documentation" )
                     go to 99
                  end if

                  ! check for convergence
                  nconv = iparam(5)
                  if( nconv >= nev ) then
                     exit
                  end if

                  ! we cannot increase further ncv0
                  if( ncv0 == n ) then
                     call PrintMessage( "mfEigs", "E",                  &
                                        "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                        "(even after increase of ncv to its maximum value)" )
                     go to 99
                  end if
                  ncv0 = 2*ncv0
                  if( ncv0 > n ) then
                     ncv0 = n
                  end if
                  deallocate( v )
                  deallocate( workl )

                  info = 1 ! re-uses the residual vector

               end do

               if( method == "OP = inv(A)" ) then
                  call msFreeMatFactor( factor )
               end if

               !-------------------------------------
               ! no fatal errors occurred
               ! post processing
               !-------------------------------------
               ! .false. : don't want the Ritz vectors
               ! 'A' : compute all the NEV Ritz values
               allocate( select(ncv0) ) ! for the present call : workspace
               allocate( dr(nev+1) ) ! real part of the Ritz values
               allocate( di(nev+1) ) ! imag part of the Ritz values
               ! init. to avoid error detection with valgrind; moreover,
               ! NaN values are useful because they will be ignored by the sort.
               dr(:) = MF_NAN
               di(:) = MF_NAN
               ldz = 1
               allocate( workev(3*ncv0) ) ! private workspace
               sigmar = 0.0d0
               sigmai = 0.0d0
               nev0 = nev
               call dneupd( .false., "A", select(1), dr(1), di(1), z, ldz, &
                            sigmar, sigmai, workev(1),                  &
                            "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, info )
               nev = nev0

               if( info == -14 ) then
                  call PrintMessage( "mfEigs", "W",                     &
                                     "ARPACK:",                         &
                                     "dnaupd didn't find any eigenvalues to sufficient accuracy" )
                  go to 99
               end if

               ! check for some errors
               if( info /= 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfEigs", "E",                     &
                                     "'dneupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               out%data_type = MF_DT_CMPLX
               out%shape = [ nev, 1 ]
               allocate( out%cmplx(nev,1) )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(dr**2+di**2) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msEquiv( dr, mf_tmp )
               else
                  ! magnitude imaginary part
                  call msAssign( mf_tmp, mf(abs(di)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, nev
                  out%cmplx(i,1) = cmplx( dr(ind(i)), di(ind(i)), kind=MF_DOUBLE )
               end do

            end if

         else if( A%data_type == MF_DT_CMPLX .or.                       &
                  A%data_type == MF_DT_SP_CMPLX ) then

            ! complex matrix

            ! for complex matrices, ARPACK deals only with one driver,
            ! and doesn't take care whether A is symmetric or not.

            allocate( zresid(n) ) ! residual vector
            ldzv = n
            allocate( zworkd(3*n) ) ! distributed array for reverse comm.
            info = 0 ! a randomly initial residual vector

            if( method == "OP = inv(A)" ) then
               if( mfIsSymm(A) ) then
                  if( mfIsPosDef(A) ) then
                     A_is_SPD = .true.
                  else
                     A_is_SPD = .false.
                  end if
               else
                  A_is_SPD = .false.
               end if
               if( A_is_SPD ) then
                  call msAssign(U, mfChol(A) )
               else
                  call msLU_mfMatFactor( A, factor )
               end if
            end if

            do

               !-------------------------------------
               ! preparation
               !-------------------------------------
               ido = 0 ! first call to the reverse communication interface
               allocate( zv(n,ncv0) ) ! set of Arnoldi basis vectors
               lzworkl = 3*ncv0**2 + 5*ncv0
               allocate( zworkl(lzworkl) ) ! private workspace
               allocate( rwork(ncv0) ) ! private workspace

               !-------------------------------------
               ! computes eigenvalues and eigenvectors
               !-------------------------------------
               iparam(:) = 0
               iparam(1) = 1 ! method for selecting the implicit shifts
               iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
               if( method == "OP = A" ) then
                  iparam(7) = 1 ! type of eigenproblem : OP = A
               else if( method == "OP = inv(A)" ) then
                  iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
               end if
               finished = .false.
               do while( .not. finished )
                  call znaupd( ido,                                     &
                               "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                               iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                               info )
                  if( ido == 1 .or. ido == -1 ) then
                     if( method == "OP = A" ) then
                        call mf_mat_vec_cmplx( A, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                  zworkd(ipntr(2):ipntr(2)+n-1) )
                     else if( method == "OP = inv(A)" ) then
                        if( A_is_SPD ) then
                           call mf_mat_vec_inv_chol_cmplx( U, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                              zworkd(ipntr(2):ipntr(2)+n-1) )
                        else
                           call mf_mat_vec_inv_factor_cmplx( factor, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                                     zworkd(ipntr(2):ipntr(2)+n-1) )
                        end if
                     end if
                  else if( ido == 99 ) then
                     finished = .true.
                  end if
               end do

               ! check for some errors
               if( info < 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "mfEigs", "E",                     &
                                     "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               ! check for convergence
               nconv = iparam(5)
               if( nconv >= nev ) then
                  exit
               end if

               ! we cannot increase further ncv0
               if( ncv0 == n ) then
                  call PrintMessage( "mfEigs", "E",                     &
                                     "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                     "(even after increase of ncv to its maximum value)" )
                  go to 99
               end if
               ncv0 = 2*ncv0
               if( ncv0 > n ) then
                  ncv0 = n
               end if
               deallocate( zv )
               deallocate( zworkl )
               deallocate( rwork )

               info = 1 ! re-uses the residual vector

            end do

            if( method == "OP = inv(A)" ) then
               call msFreeMatFactor( factor )
            end if
            if( .not. mfIsEmpty(U) ) then
               call msSilentRelease( U )
            end if

            !-------------------------------------
            ! no fatal errors occurred
            ! post processing
            !-------------------------------------
            ! .false. : don't want the Ritz vectors
            ! 'A' : compute all the NEV Ritz values
            allocate( select(ncv0) ) ! for the present call : workspace
            allocate( zd(nev+1) ) ! Ritz values
            ! init. to avoid error detection with valgrind; moreover,
            ! NaN values are useful because they will be ignored by the sort.
            zd(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
            ldzz = 1
            allocate( zworkev(2*ncv0) ) ! private workspace
            zsigma = (0.0d0,0.0d0)
            call zneupd( .false., "A", select(1), zd(1), zz, ldzz, zsigma, zworkev(1), &
                         "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                         iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                         info )

            if( info == -14 ) then
               call PrintMessage( "mfEigs", "W",                        &
                                  "ARPACK:",                            &
                                  "znaupd didn't find any eigenvalues to sufficient accuracy" )
               go to 99
            end if

            ! check for some errors
            if( info /= 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( "mfEigs", "E",                        &
                                  "'zneupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               go to 99
            end if

            if( mfIsSymm(A) ) then

               out%data_type = MF_DT_DBLE
               out%shape = [ nev, 1 ]
               allocate( out%double(nev,1) )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(zd)) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msAssign( mf_tmp, mf(real(zd)) )
               else
                  ! algebraic imaginary part
                  call msAssign( mf_tmp, mf(aimag(zd)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, nev
                  out%double(i,1) = real( zd(ind(i)) )
               end do

            else

               out%data_type = MF_DT_CMPLX
               out%shape = [ nev, 1 ]
               allocate( out%cmplx(nev,1) )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(zd)) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msAssign( mf_tmp, mf(real(zd)) )
               else
                  ! algebraic imaginary part
                  call msAssign( mf_tmp, mf(aimag(zd)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, nev
                  out%cmplx(i,1) = zd(ind(i))
               end do

            end if

         end if

      end if

      out%status_temporary = .true.

 99   continue

      call msSilentRelease( mf_dummy, mf_ind, mf_tmp )

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfEigs_which
!_______________________________________________________________________
!
   subroutine msEigs_which( out, A, k, which, tol, ncv )

      type(mfArray)                              :: A
      integer,              intent(in)           :: k
      character(len=2),     intent(in), optional :: which
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      integer,              intent(in), optional :: ncv

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! equivalent of 'eigs' in MATLAB (simplified version)
      !
      ! returns a vector containing 'k' eigenvalues of A, and a matrix
      ! whose columns are the corresponding eigenvectors.
      !
      ! A may be : real/complex, sparse/dense
      !
      ! which = 'LM' : Largest Magnitude   | (all types of matrices)
      !         'SM' : Smallest Magnitude  |
      !         ----------------------------------------------------------
      !         'LA' : Largest Algebraic   |
      !         'SA' : Smallest Algebraic  | (only for real, symmetric)
      !         'BE' : Both End            |
      !         ---------------------------
      !         'LR' : Largest Real part   |
      !         'SR' : Smallest Real part  | (not real, or not symmetric)
      !         'LI' : Largest Imag part   |
      !         'SI' : Smallest Imag part  |
      !
      ! Optionally returns a convergence flag (boolean mfArray):
      !     flag = TRUE if all eigenvalues have converged
      !     flag = FALSE else

      type(mfArray), pointer :: V, D, flag
      integer :: n, ncol, nnz
      logical :: finished

      ! declarations for ARPACK
      integer :: ido, nev, nev0, ncv0, ldv, lworkl, info, ldz, i, nconv
      real(kind=MF_DOUBLE) :: tol0
      real(kind=MF_DOUBLE), allocatable :: resid(:), vv(:,:)
      real(kind=MF_DOUBLE), allocatable :: workd(:), workl(:), workev(:)
      real(kind=MF_DOUBLE), allocatable :: dr(:), di(:), dd(:)
      real(kind=MF_DOUBLE), allocatable :: rwork(:), z(:,:)
      integer :: iparam(11), ipntr(14)
      logical, allocatable :: select(:)
      real(kind=MF_DOUBLE) :: sigmar, sigmai, sigma
      character(len=2) :: which0, which1
      complex(kind=MF_DOUBLE), allocatable :: zresid(:), zv(:,:), zd(:), zz(:,:)
      complex(kind=MF_DOUBLE), allocatable :: zworkd(:), zworkl(:), zworkev(:)
      integer :: ldzv, lzworkl, ldzz
      complex(kind=MF_DOUBLE) :: zsigma
      character(len=6) :: info_char
      character(len=12) :: method

      type(mfMatFactor) :: factor
      type(mfArray) :: U
      logical :: A_is_SPD

      ! new sorting of eigenvalues
      type(mfArray) :: mf_dummy, mf_ind, mf_tmp
      integer, allocatable :: ind(:), cjg(:)
      character(len=10) :: mode_sort
      integer :: j, j_shift, pair_count
      logical :: found

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      ! 3 out-args must be specified
      if( out%n /= 3 ) then
         call PrintMessage( "msEigs", "E",                              &
                            "three output args required!",             &
                            "syntax is: call msEigs ( mfOut(V,D,flag), A, ... )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msEigs", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      V => out%ptr1
      D => out%ptr2
      call msSilentRelease( V, D )
      flag => out%ptr3
      call msSilentRelease( flag )
      flag = .false.

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "msEigs", "W",                              &
                            "'A' is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msEigs", "E",                              &
                            "'A' cannot be boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msEigs", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      n = A%shape(1)

      ! square matrix ?
      if( A%shape(1) /= A%shape(2) ) then
         call PrintMessage( "msEigs", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      if( k < 2 ) then
         call PrintMessage( "msEigs", "E",                              &
                            "k must be greater or equal 2!" )
         go to 99
      else if( n-2 < k ) then
         call PrintMessage( "msEigs", "W",                              &
                            "the ARPACK routine used allows only the computation", &
                            "of n-2 max eigenvalues/vectors!",         &
                            "(use mfEig to retrieve all eigenvalues/vectors)" )
         nev = n-2
      else
         nev = k
      end if

      if( mfIsSparse(A) ) then
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         if( nnz == 0 ) then
            call msAssign( V, mfEye(n,k) )
            call msAssign( D, mfZeros(k,1) )
            flag = .true.
            go to 99
         end if
      end if

      if( present(which) ) then
         which0 = to_upper(which)
      else
         which0 = "LM" ! Default : Largest Eigenvalues
      end if

      ! check 'which' argument
      if( which0 == "LM" .or. which0 == "SM" .or.                       &
          which0 == "LA" .or. which0 == "SA" .or.                       &
          which0 == "BE" .or.                                           &
          which0 == "LR" .or. which0 == "SR" .or.                       &
          which0 == "LI" .or. which0 == "SI" ) then
         ! do nothing
      else
         call PrintMessage( "msEigs", "E",                              &
                            "'which' arg must be equal to 'LM', 'SM', 'LA', 'SA', 'BE',", &
                            "'LR', 'SR', 'LI' or 'SI'",                 &
                            "(see documentation)" )
         go to 99
      end if

      if( mfIsSymm(A) ) then

         if( mfIsReal(A) ) then

            ! real, symmetric matrices : 'LR', 'SR', 'LI' or 'SI' are not valid
            if( which0 == "LR" .or. which0 == "SR" .or.                 &
                which0 == "LI" .or. which0 == "SI" ) then
               call PrintMessage( "msEigs", "E",                        &
                                  "'which' arg may be equal to 'LR', 'SR', 'LI' or 'SI'", &
                                  " only for complex or unsymmetric real matrices!" )
               go to 99
            end if

         else

            ! complex, symmetric matrices, then eigenvalues are real.
            ! It is not useful to use the imaginary part for sorting
            if( which0 == "LI" .or. which0 == "SI" ) then
               call PrintMessage( "msEigs", "W",                        &
                                  "The matrix A is symmetric, then the eigenvalues are real.", &
                                  "So, it is not useful to use the imaginary part for sorting" )
            end if

            ! not real matrices : 'LA', 'SA', 'BE' are not valid
            if( which0 == "LA" .or. which0 == "SA" .or.                 &
                which0 == "BE" ) then
               call PrintMessage( "msEigs", "E",                        &
                                  "'which' arg may be equal to 'LA', 'SA' or 'BE'", &
                                  " only for real, symmetric matrices!" )
               go to 99
            end if

         end if

      else

         ! not symmetric matrices : 'LA', 'SA', 'BE' are not valid
         if( which0 == "LA" .or. which0 == "SA" .or.                    &
             which0 == "BE" ) then
            call PrintMessage( "msEigs", "E",                           &
                               "'which' arg may be equal to 'LA', 'SA' or 'BE'", &
                               " only for real, symmetric matrices!" )
            go to 99
         end if

      end if

      ! choice of OP

!### TODO 2: (i)   pourquoi ne pas choisir un seuil à sqrt(eps) ?
!            (ii)  test de même nature pour "SA" ?
!            (iii) pourquoi ne pas tester si A est SPD ?
      if( which0 == "SM" ) then
         ! Warn if matrix is close to singular
         if( mfDble(mfRCond(A)) <= MF_EPS*10.0d0 ) then
            call PrintMessage( "msEigs", "W",                           &
                               "matrix 'A' is close to singular",       &
                               "Computation can be very inaccurate!" )
            method = "OP = A"
            which1 = "SM"
         else
            method = "OP = inv(A)"
            which1 = "LM"
         end if
      else
         method = "OP = A"
         which1 = which0
      end if

      ! relative accuracy of the Ritz values
      if( present(tol) ) then
         tol0 = tol
      else
         tol0 = MF_EPS
      end if

      ! how many Arnoldi vectors are generated
      if( present(ncv) ) then
         ncv0 = ncv
      else
         ncv0 = 2*nev+1 ! recommended value from User Guide
      end if

      if( ncv0 < 2*nev+1 ) then
         call PrintMessage( "msEigs", "W",                              &
                            "ARPACK: ncv < 2*k+1",                      &
                            "(2*k+1 is the recommended value from the ARPACK User Guide)", &
                            "-> setting : ncv = 2*k+1" )
         ncv0 = 2*nev+1
      end if

      if( ncv0 > n ) then
         if( present(ncv) ) then
            call PrintMessage( "msEigs", "W",                           &
                               "ARPACK: ncv > n",                       &
                               "It seems that you are working with a small matrix, or", &
                               "that a great number of eigs have been requested...", &
                               "(ARPACK works well when computing few eigenvalues", &
                               "of a large matrix)",                    &
                               "-> setting : ncv = n" )
         end if
         ncv0 = n
      end if

      ! using ARPACK-2

      if( present(ncv) ) then

         if( A%data_type == MF_DT_DBLE .or.                             &
             A%data_type == MF_DT_SP_DBLE ) then

            ! real matrix

            if( mfIsSymm(A) ) then

               ! symmetric case, eigenvalues are all real

               if( method == "OP = inv(A)" ) then
                  if( mfIsPosDef(A) ) then
                     A_is_SPD = .true.
                     call msAssign(U, mfChol(A) )
                  else
                     A_is_SPD = .false.
                     call msLU_mfMatFactor( A, factor )
                  end if
               end if

               !-------------------------------------
               ! preparation
               !-------------------------------------
               ido = 0 ! first call to the reverse communication interface
               allocate( resid(n) ) ! residual vector
               allocate( vv(n,ncv0) ) ! set of Arnoldi basis vectors
               ldv = n
               allocate( workd(3*n) ) ! distributed array for reverse comm.
               lworkl = ncv0**2 + 8*ncv0
               allocate( workl(lworkl) ) ! private workspace

               !-------------------------------------
               ! computes eigenvalues and eigenvectors
               !-------------------------------------
               iparam(:) = 0
               iparam(1) = 1 ! method for selecting the implicit shifts
               iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
               if( method == "OP = A" ) then
                  iparam(7) = 1 ! type of eigenproblem : OP = A
               else if( method == "OP = inv(A)" ) then
                  iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
               end if
               info = 0 ! a randomly initial residual vector
               finished = .false.
               do while( .not. finished )
                  call dsaupd( ido,                                     &
                               "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                               iparam, ipntr, workd(1), workl(1), lworkl, info )
                  if( ido == 1 .or. ido == -1 ) then
                     if( method == "OP = A" ) then
                        call mf_mat_vec( A, workd(ipntr(1):ipntr(1)+n-1), &
                                            workd(ipntr(2):ipntr(2)+n-1) )
                     else if( method == "OP = inv(A)" ) then
                        if( A_is_SPD ) then
                           call mf_mat_vec_inv_chol( U, workd(ipntr(1):ipntr(1)+n-1), &
                                                        workd(ipntr(2):ipntr(2)+n-1) )
                        else
                           call mf_mat_vec_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                               workd(ipntr(2):ipntr(2)+n-1) )
                        end if
                     end if
                  else if( ido == 99 ) then
                     finished = .true.
                  end if
               end do

               if( method == "OP = inv(A)" ) then
                  call msFreeMatFactor( factor )
               end if
               if( .not. mfIsEmpty(U) ) then
                  call msSilentRelease( U )
               end if

               ! check for some errors
               if( info < 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "msEigs", "E",                     &
                                     "'dsaupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               ! check for convergence
               nconv = iparam(5)
               if( nconv >= nev ) then
                  flag = .true.
               end if

               !-------------------------------------
               ! no fatal errors occurred
               ! post processing
               !-------------------------------------
               ! .true. : compute Ritz vectors
               ! 'A' : compute all the NEV Ritz values
               allocate( select(ncv0) ) ! for the present call : workspace
               allocate( dd(nev) ) ! vector of the Ritz values
               ldz = n
               allocate( z(n,nev) )
               sigma = 0.0d0
               call dseupd( .true., "A", select(1), dd(1), z(1,1), ldz, sigma, &
                            "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, info )

               if( info == -14 ) then
                  call PrintMessage( "msEigs", "W",                     &
                                     "ARPACK:",                         &
                                     "dsaupd didn't find any eigenvalues to sufficient accuracy", &
                                     "-> try to increase ncv" )
                  flag = .false.
                  go to 99
               end if

               ! check for some errors
               if( info /= 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "msEigs", "E",                     &
                                     "'dseupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  flag = .false.
                  go to 99
               end if

               D%data_type = MF_DT_DBLE
               D%shape = [ nev, 1 ]
               allocate( D%double(nev,1) )

               V%data_type = MF_DT_DBLE
               V%shape = [ n, nev ]
               allocate( V%double(n,nev) )

               ! non-converged eigenvalues will contain NaN values, and
               ! the corresponding eigenvectors will also contain NaNs.
               D%double(:,:) = MF_NAN
               V%double(:,:) = MF_NAN

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LA" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  call msAssign( mf_tmp , mf(abs(dd)) )
               else
                  call msEquiv( dd, mf_tmp )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev) )
               ind = mf_ind
               do i = 1, min(nev,nconv) ! limited to the converged ritz values!
                  D%double(i,1) = dd(ind(i))
                  V%double(:,i) = z(:,ind(i))
               end do

            else

               ! non-symmetric case, eigenvalues are complex

               if( method == "OP = inv(A)" ) then
                  call msLU_mfMatFactor( A, factor )
               end if

               !-------------------------------------
               ! preparation
               !-------------------------------------
               ido = 0 ! first call to the reverse communication interface
               allocate( resid(n) ) ! residual vector
               allocate( vv(n,ncv0) ) ! set of Arnoldi basis vectors
               ldv = n
               allocate( workd(3*n) ) ! distributed array for reverse comm.
               lworkl = 3*ncv0**2 + 6*ncv0
               allocate( workl(lworkl) ) ! private workspace

               !-------------------------------------
               ! computes eigenvalues and eigenvectors
               !-------------------------------------
               iparam(:) = 0
               iparam(1) = 1 ! method for selecting the implicit shifts
               iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
               if( method == "OP = A" ) then
                  iparam(7) = 1 ! type of eigenproblem : OP = A
               else if( method == "OP = inv(A)" ) then
                  iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
               end if
               info = 0 ! a randomly initial residual vector
               finished = .false.
               do while( .not. finished )
                  call dnaupd( ido,                                     &
                               "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                               iparam, ipntr, workd(1), workl(1), lworkl, info )
                  if( ido == 1 .or. ido == -1 ) then
                     if( method == "OP = A" ) then
                        call mf_mat_vec( A, workd(ipntr(1):ipntr(1)+n-1), &
                                            workd(ipntr(2):ipntr(2)+n-1) )
                     else if( method == "OP = inv(A)" ) then
                        call mf_mat_vec_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                            workd(ipntr(2):ipntr(2)+n-1) )
                     end if
                  else if( ido == 99 ) then
                     finished = .true.
                  end if
               end do

               if( method == "OP = inv(A)" ) then
                  call msFreeMatFactor( factor )
               end if

               ! check for some errors
               if( info < 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "msEigs", "E",                     &
                                     "'dnaupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               ! check for convergence
               nconv = iparam(5)
               if( nconv >= nev ) then
                  flag = .true.
               end if

               !-------------------------------------
               ! no fatal errors occurred
               ! post processing
               !-------------------------------------
               ! .true. : compute Ritz vectors
               ! 'A' : compute all the NEV Ritz values
               allocate( select(ncv0) ) ! for the present call : workspace
               allocate( dr(nev+1) ) ! real part of the Ritz values
               allocate( di(nev+1) ) ! imag part of the Ritz values
               ! init. to avoid error detection with valgrind; moreover,
               ! initialization of dr (only) to NaN values is useful because
               ! they will be ignored by the sort. di must be initialized to
               ! zero because it will be used as test to build cjg(:).
               dr(:) = MF_NAN
               di(:) = 0.0d0
               ldz = n
               allocate( z(n,nev+1) )
               allocate( workev(3*ncv0) ) ! private workspace
               sigmar = 0.0d0
               sigmai = 0.0d0
               nev0 = nev
               call dneupd( .true., "A", select(1), dr(1), di(1), z(1,1), ldz, &
                            sigmar, sigmai, workev(1),                  &
                            "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, info )
               nev = nev0

               if( info == -14 ) then
                  call PrintMessage( "msEigs", "W",                     &
                                     "ARPACK:",                         &
                                     "dnaupd didn't find any eigenvalues to sufficient accuracy" )
                  flag = .false.
                  go to 99
               end if

               ! check for some errors
               if( info /= 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "msEigs", "E",                     &
                                     "'dneupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  flag = .false.
                  go to 99
               end if

               D%data_type = MF_DT_CMPLX
               D%shape = [ nev, 1 ]
               allocate( D%cmplx(nev,1) )

               V%data_type = MF_DT_CMPLX
               V%shape = [ n, nev ]
               allocate( V%cmplx(n,nev) )

               ! non-converged eigenvalues will contain NaN values, and
               ! the corresponding eigenvectors will also contain NaNs.
               D%cmplx(:,:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
               V%cmplx(:,:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(dr**2+di**2) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msEquiv( dr, mf_tmp )
               else
                  ! magnitude imaginary part
                  call msAssign( mf_tmp, mf(abs(di)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1), cjg(nev+1) )
               ind = mf_ind
               ! construct a table for indicating how are stored the eigenvects
               !    0 : is purely real
               !   +k : is the real part of the k-th eigenvects pair
               !   -k : is the imag  "            "      "       "
               i = 1
               pair_count = 0
               do while( i <= nev+1 )
                  if( di(i) == 0.0d0 ) then
                     ! real eigenvalue
                     cjg(i) = 0
                     i = i + 1
                  else
                     ! complex eigenvalue (conjugate by pair)
                     pair_count = pair_count + 1
                     cjg(i)   = +pair_count
                     cjg(i+1) = -pair_count
                     i = i + 2
                  end if
               end do
               do i = 1, min(nev,nconv) ! limited to the converged ritz values!
                  j = ind(i) ! j is ranged in [1,nev+1]
                  D%cmplx(i,1) = cmplx( dr(j), di(j), kind=MF_DOUBLE )
                  ! Warning: not so easy to get the eigenvectors, which have
                  !          been stored in a compact way (both real and imag)
                  !          in the real array Z...
                  if( cjg(j) == 0 ) then
                     V%cmplx(:,i) = z(:,j)
                  else
                     ! in the di(:) array, where is the conjugate of
                     ! the complex eigenvalue ( dr(j), di(j) )?
                     ! -> is it just before or just after?
                     found = .false.
                     if( j == 1 ) then
                        if( cjg(j) == -cjg(j+1) ) then
                           j_shift = +1
                           found = .true.
                        end if
                     else if( j == nev+1 ) then
                        if( cjg(j) == -cjg(j-1) ) then
                           j_shift = -1
                           found = .true.
                        end if
                     else ! 1 < j < nev+1
                        if( cjg(j) == -cjg(j+1) ) then
                           j_shift = +1
                           found = .true.
                        else if( cjg(j) == -cjg(j-1) ) then
                           j_shift = -1
                           found = .true.
                        end if
                     end if
                     if( .not. found ) then
                        write(STDERR,*) "(MUESLI msEigs:) internal error 1:"
                        write(STDERR,*) "                 cannot find an eigenvalues pair!"
                        write(STDERR,*) "                 (in msEigs_which after dneupd call)"
                        mf_message_displayed = .true.
                        call muesli_trace( pause ="yes" )
                        stop
                     end if
                     if( cjg(j) > 0 ) then
                        V%cmplx(:,i) = cmplx( z(:,j), z(:,j+j_shift),   &
                                              kind=MF_DOUBLE )
                     else
                        V%cmplx(:,i) = cmplx( z(:,j+j_shift), -z(:,j),  &
                                              kind=MF_DOUBLE )
                     end if
                  end if
               end do

            end if

         else if( A%data_type == MF_DT_CMPLX .or.                       &
                  A%data_type == MF_DT_SP_CMPLX ) then

            ! complex matrix

            ! for complex matrices, ARPACK deals only with one driver,
            ! and doesn't take care whether A is symmetric or not.

            if( method == "OP = inv(A)" ) then
               if( mfIsSymm(A) ) then
                  if( mfIsPosDef(A) ) then
                     A_is_SPD = .true.
                  else
                     A_is_SPD = .false.
                  end if
               else
                  A_is_SPD = .false.
               end if
               if( A_is_SPD ) then
                  call msAssign(U, mfChol(A) )
               else
                  call msLU_mfMatFactor( A, factor )
               end if
            end if

            !-------------------------------------
            ! preparation
            !-------------------------------------
            ido = 0 ! first call to the reverse communication interface
            allocate( zresid(n) ) ! residual vector
            allocate( zv(n,ncv0) ) ! set of Arnoldi basis vectors
            ldzv = n
            allocate( zworkd(3*n) ) ! distributed array for reverse comm.
            lzworkl = 3*ncv0**2 + 5*ncv0
            allocate( zworkl(lzworkl) ) ! private workspace
            allocate( rwork(ncv0) ) ! private workspace

            !-------------------------------------
            ! computes eigenvalues and eigenvectors
            !-------------------------------------
            iparam(:) = 0
            iparam(1) = 1 ! method for selecting the implicit shifts
            iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
            if( method == "OP = A" ) then
               iparam(7) = 1 ! type of eigenproblem : OP = A
            else if( method == "OP = inv(A)" ) then
               iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
            end if
            info = 0 ! a randomly initial residual vector
            finished = .false.
            do while( .not. finished )
               call znaupd( ido,                                        &
                            "I", n, which0, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                            iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                            info )
               if( ido == 1 .or. ido == -1 ) then
                  if( method == "OP = A" ) then
                     call mf_mat_vec_cmplx( A, zworkd(ipntr(1):ipntr(1)+n-1), &
                                               zworkd(ipntr(2):ipntr(2)+n-1) )
                  else if( method == "OP = inv(A)" ) then
                     if( A_is_SPD ) then
                        call mf_mat_vec_inv_chol_cmplx( U, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                           zworkd(ipntr(2):ipntr(2)+n-1) )
                     else
                        call mf_mat_vec_inv_factor_cmplx( factor, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                                  zworkd(ipntr(2):ipntr(2)+n-1) )
                     end if
                  end if
               else if( ido == 99 ) then
                  finished = .true.
               end if
            end do

            if( method == "OP = inv(A)" ) then
               call msFreeMatFactor( factor )
            end if
            if( .not. mfIsEmpty(U) ) then
               call msSilentRelease( U )
            end if

            ! check for some errors
            if( info < 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( "msEigs", "E",                        &
                                  "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               go to 99
            end if

            ! check for convergence
            nconv = iparam(5)
            if( nconv >= nev ) then
               flag = .true.
            end if

            !-------------------------------------
            ! no fatal errors occurred
            ! post processing
            !-------------------------------------
            ! .true. : compute Ritz vectors
            ! 'A' : compute all the NEV Ritz values
            allocate( select(ncv0) ) ! for the present call : workspace
            allocate( zd(nev+1) ) ! Ritz values
            ! init. to avoid error detection with valgrind; moreover,
            ! NaN values are useful car they will be ignored by the sort.
            zd(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
            ldzz = n
            allocate( zz(n,nev) )
            allocate( zworkev(2*ncv0) ) ! private workspace
            zsigma = (0.0d0,0.0d0)
            call zneupd( .true., "A", select(1), zd(1), zz, ldzz, zsigma, zworkev(1), &
                         "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                         iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                         info )

            if( info == -14 ) then
               call PrintMessage( "msEigs", "W",                        &
                                  "ARPACK:",                            &
                                  "znaupd didn't find any eigenvalues to sufficient accuracy" )
               flag = .false.
               go to 99
            end if

            ! check for some errors
            if( info /= 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( "msEigs", "E",                        &
                                  "'zneupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               flag = .false.
               go to 99
            end if

            if( mfIsSymm(A) ) then

               D%data_type = MF_DT_DBLE
               D%shape = [ nev, 1 ]
               allocate( D%double(nev,1) )

               ! non-converged eigenvalues will contain NaN values, and
               ! the corresponding eigenvectors will also contain NaNs.
               D%double(:,:) = MF_NAN

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(zd)) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msAssign( mf_tmp, mf(real(zd)) )
               else
                  ! algebraic imaginary part
                  call msAssign( mf_tmp, mf(aimag(zd)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, min(nev,nconv) ! limited to the converged ritz values!
                  D%double(i,1) = real( zd(ind(i)) )
               end do

            else

               D%data_type = MF_DT_CMPLX
               D%shape = [ nev, 1 ]
               allocate( D%cmplx(nev,1) )

               ! non-converged eigenvalues will contain NaN values, and
               ! the corresponding eigenvectors will also contain NaNs.
               D%cmplx(:,:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(zd)) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msAssign( mf_tmp, mf(real(zd)) )
               else
                  ! algebraic imaginary part
                  call msAssign( mf_tmp, mf(aimag(zd)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, min(nev,nconv) ! limited to the converged ritz values!
                  D%cmplx(i,1) = zd(ind(i))
               end do

            end if

            V%data_type = MF_DT_CMPLX
            V%shape = [ n, nev ]
            allocate( V%cmplx(n,nev) )

            ! non-converged eigenvalues will contain NaN values, and
            ! the corresponding eigenvectors will also contain NaNs.
            V%cmplx(:,:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )

            ! change the ouput order
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               V%cmplx(:,i) = zz(:,ind(i))
            end do

         end if

      else ! .not. present(ncv)

         ! ncv chosen automatically
         ! computation starts with ncv0, and this value is increased
         ! until convergence of all singular values...

         if( A%data_type == MF_DT_DBLE .or.                             &
             A%data_type == MF_DT_SP_DBLE ) then

            ! real matrix

            if( mfIsSymm(A) ) then

               ! symmetric case, eigenvalues are all real

               allocate( resid(n) ) ! residual vector
               ldv = n
               allocate( workd(3*n) ) ! distributed array for reverse comm.
               info = 0 ! a randomly initial residual vector

               if( method == "OP = inv(A)" ) then
                  if( mfIsPosDef(A) ) then
                     A_is_SPD = .true.
                     call msAssign( U, mfChol(A) )
                  else
                     A_is_SPD = .false.
                     call msLU_mfMatFactor( A, factor )
                  end if
               end if

               do

                  !-------------------------------------
                  ! preparation
                  !-------------------------------------
                  ido = 0 ! first call to the reverse communication interface
                  allocate( vv(n,ncv0) ) ! set of Arnoldi basis vectors
                  lworkl = ncv0**2 + 8*ncv0
                  allocate( workl(lworkl) ) ! private workspace

                  !-------------------------------------
                  ! computes eigenvalues and eigenvectors
                  !-------------------------------------
                  iparam(:) = 0
                  iparam(1) = 1 ! method for selecting the implicit shifts
                  iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
                  if( method == "OP = A" ) then
                     iparam(7) = 1 ! type of eigenproblem : OP = A
                  else if( method == "OP = inv(A)" ) then
                     iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
                  end if
                  finished = .false.
                  do while( .not. finished )
                     call dsaupd( ido,                                  &
                                  "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                                  iparam, ipntr, workd(1), workl(1), lworkl, info )
                     if( ido == 1 .or. ido == -1 ) then
                        if( method == "OP = A" ) then
                           call mf_mat_vec( A, workd(ipntr(1):ipntr(1)+n-1), &
                                               workd(ipntr(2):ipntr(2)+n-1) )
                        else if( method == "OP = inv(A)" ) then
                           if( A_is_SPD ) then
                              call mf_mat_vec_inv_chol( U, workd(ipntr(1):ipntr(1)+n-1), &
                                                           workd(ipntr(2):ipntr(2)+n-1) )
                           else
                              call mf_mat_vec_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                                  workd(ipntr(2):ipntr(2)+n-1) )
                           end if
                        end if
                     else if( ido == 99 ) then
                        finished = .true.
                     end if
                  end do

                  ! check for some errors
                  if( info < 0 ) then
                     write(info_char,"(I0)") info
                     call PrintMessage( "msEigs", "E",                  &
                                        "'dsaupd' (ARPACK) cannot compute eigenvalues", &
                                        "Error returned, info = " // info_char, &
                                        "Check the ARPACK documentation" )
                     go to 99
                  end if

                  ! check for convergence
                  nconv = iparam(5)
                  if( nconv >= nev ) then
                     flag = .true.
                     exit
                  end if

                  ! we cannot increase further ncv0
                  if( ncv0 == n ) then
                     call PrintMessage( "msEigs", "W",                  &
                                        "'dsaupd' (ARPACK) cannot compute eigenvalues", &
                                        "(even after increase of ncv to its maximum value)" )
                     flag = .false.
                     go to 99
                  end if
                  ncv0 = 2*ncv0
                  if( ncv0 > n ) then
                     ncv0 = n
                  end if
                  deallocate( vv )
                  deallocate( workl )

                  info = 1 ! re-uses the residual vector

               end do

               if( method == "OP = inv(A)" ) then
                  call msFreeMatFactor( factor )
               end if
               if( .not. mfIsEmpty(U) ) then
                  call msSilentRelease( U )
               end if

               !-------------------------------------
               ! no fatal errors occurred
               ! post processing
               !-------------------------------------
               ! .true. : compute Ritz vectors
               ! 'A' : compute all the NEV Ritz values
               allocate( select(ncv0) ) ! for the present call : workspace
               allocate( dd(nev) ) ! vector of the Ritz values
               ldz = n
               allocate( z(n,nev) )
               sigma = 0.0d0
               call dseupd( .true., "A", select(1), dd(1), z(1,1), ldz, sigma, &
                            "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, info )

               if( info == -14 ) then
                  call PrintMessage( "msEigs", "W",                     &
                                     "ARPACK:",                         &
                                     "dsaupd didn't find any eigenvalues to sufficient accuracy" )
                  flag = .false.
                  go to 99
               end if

               ! check for some errors
               if( info /= 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "msEigs", "E",                     &
                                     "'dseupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  flag = .false.
                  go to 99
               end if

               D%data_type = MF_DT_DBLE
               D%shape = [ nev, 1 ]
               allocate( D%double(nev,1) )

               V%data_type = MF_DT_DBLE
               V%shape = [ n, nev ]
               allocate( V%double(n,nev) )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LA" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(dd)) )
               else
                  ! algebraic
                  call msEquiv( dd, mf_tmp )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev) )
               ind = mf_ind
               do i = 1, nev
                  D%double(i,1) = dd(ind(i))
                  V%double(:,i) = z(:,ind(i))
               end do

            else

               ! non-symmetric case, eigenvalues are complex

               allocate( resid(n) ) ! residual vector
               ldv = n
               allocate( workd(3*n) ) ! distributed array for reverse comm.
               info = 0 ! a randomly initial residual vector

               if( method == "OP = inv(A)" ) then
                  call msLU_mfMatFactor( A, factor )
               end if

               do

                  !-------------------------------------
                  ! preparation
                  !-------------------------------------
                  ido = 0 ! first call to the reverse communication interface
                  allocate( vv(n,ncv0) ) ! set of Arnoldi basis vectors
                  lworkl = 3*ncv0**2 + 6*ncv0
                  allocate( workl(lworkl) ) ! private workspace

                  !-------------------------------------
                  ! computes eigenvalues and eigenvectors
                  !-------------------------------------
                  iparam(:) = 0
                  iparam(1) = 1 ! method for selecting the implicit shifts
                  iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
                  if( method == "OP = A" ) then
                     iparam(7) = 1 ! type of eigenproblem : OP = A
                  else if( method == "OP = inv(A)" ) then
                     iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
                  end if
                  finished = .false.
                  do while( .not. finished )
                     call dnaupd( ido,                                  &
                                  "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                                  iparam, ipntr, workd(1), workl(1), lworkl, info )
                     if( ido == 1 .or. ido == -1 ) then
                        if( method == "OP = A" ) then
                           call mf_mat_vec( A, workd(ipntr(1):ipntr(1)+n-1), &
                                               workd(ipntr(2):ipntr(2)+n-1) )
                        else if( method == "OP = inv(A)" ) then
                           call mf_mat_vec_inv_factor( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                               workd(ipntr(2):ipntr(2)+n-1) )
                        end if
                     else if( ido == 99 ) then
                        finished = .true.
                     end if
                  end do

                  ! check for some errors
                  if( info < 0 ) then
                     write(info_char,"(I0)") info
                     call PrintMessage( "msEigs", "E",                  &
                                        "'dnaupd' (ARPACK) cannot compute eigenvalues", &
                                        "Error returned, info = " // info_char, &
                                        "Check the ARPACK documentation" )
                     go to 99
                  end if

                  ! check for convergence
                  nconv = iparam(5)
                  if( nconv >= nev ) then
                     flag = .true.
                     exit
                  end if

                  ! we cannot increase further ncv0
                  if( ncv0 == n ) then
                     call PrintMessage( "msEigs", "E",                  &
                                        "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                        "(even after increase of ncv to its maximum value)" )
                     flag = .false.
                     go to 99
                  end if
                  ncv0 = 2*ncv0
                  if( ncv0 > n ) then
                     ncv0 = n
                  end if
                  deallocate( vv )
                  deallocate( workl )

                  info = 1 ! re-uses the residual vector

               end do

               if( method == "OP = inv(A)" ) then
                  call msFreeMatFactor( factor )
               end if

               !-------------------------------------
               ! no fatal errors occurred
               ! post processing
               !-------------------------------------
               ! .true. : compute Ritz vectors
               ! 'A' : compute all the NEV Ritz values
               allocate( select(ncv0) ) ! for the present call : workspace
               allocate( dr(nev+1) ) ! real part of the Ritz values
               allocate( di(nev+1) ) ! imag part of the Ritz values
               ! init. to avoid error detection with valgrind; moreover,
               ! initialization of dr (only) to NaN values is useful because
               ! they will be ignored by the sort. di must be initialized to
               ! zero because it will be used as test to build cjg(:).
               dr(:) = MF_NAN
               di(:) = 0.0d0
               ldz = n
               allocate( z(n,nev+1) )
               allocate( workev(3*ncv0) ) ! private workspace
               sigmar = 0.0d0
               sigmai = 0.0d0
               nev0 = nev
               call dneupd( .true., "A", select(1), dr(1), di(1), z(1,1), ldz, &
                            sigmar, sigmai, workev(1),                  &
                            "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, info )
               nev = nev0

               if( info == -14 ) then
                  call PrintMessage( "msEigs", "W",                     &
                                     "ARPACK:",                         &
                                     "dnaupd didn't find any eigenvalues to sufficient accuracy" )
                  flag = .false.
                  go to 99
               end if

               ! check for some errors
               if( info /= 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "msEigs", "E",                     &
                                     "'dneupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  flag = .false.
                  go to 99
               end if

               D%data_type = MF_DT_CMPLX
               D%shape = [ nev, 1 ]
               allocate( D%cmplx(nev,1) )

               V%data_type = MF_DT_CMPLX
               V%shape = [ n, nev ]
               allocate( V%cmplx(n,nev) )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(dr**2+di**2) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msEquiv( dr, mf_tmp )
               else
                  ! magnitude imaginary part
                  call msAssign( mf_tmp, mf(abs(di)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1), cjg(nev+1) )
               ind = mf_ind
               ! construct a table for indicating how are stored the eigenvects
               !    0 : is purely real
               !   +k : is the real part of the k-th eigenvects pair
               !   -k : is the imag  "            "      "       "
               i = 1
               pair_count = 0
               do while( i <= nev+1 )
                  if( di(i) == 0.0d0 ) then
                     ! real eigenvalue
                     cjg(i) = 0
                     i = i + 1
                  else
                     ! complex eigenvalue (conjugate by pair)
                     pair_count = pair_count + 1
                     cjg(i)   = +pair_count
                     cjg(i+1) = -pair_count
                     i = i + 2
                  end if
               end do
               do i = 1, nev
                  j = ind(i) ! j is ranged in [1,nev+1]
                  D%cmplx(i,1) = cmplx( dr(j), di(j), kind=MF_DOUBLE )
                  ! Warning: not so easy to get the eigenvectors, which have
                  !          been stored in a compact way (both real and imag)
                  !          in the real array Z...
                  if( cjg(j) == 0 ) then
                     V%cmplx(:,i) = z(:,j)
                  else
                     ! in the di(:) array, where is the conjugate of
                     ! the complex eigenvalue ( dr(j), di(j) )?
                     ! -> is it just before or just after?
                     found = .false.
                     if( j == 1 ) then
                        if( cjg(j) == -cjg(j+1) ) then
                           j_shift = +1
                           found = .true.
                        end if
                     else if( j == nev+1 ) then
                        if( cjg(j) == -cjg(j-1) ) then
                           j_shift = -1
                           found = .true.
                        end if
                     else ! 1 < j < nev+1
                        if( cjg(j) == -cjg(j+1) ) then
                           j_shift = +1
                           found = .true.
                        else if( cjg(j) == -cjg(j-1) ) then
                           j_shift = -1
                           found = .true.
                        end if
                     end if
                     if( .not. found ) then
                        write(STDERR,*) "(MUESLI msEigs:) internal error 2:"
                        write(STDERR,*) "                 cannot find an eigenvalues pair!"
                        write(STDERR,*) "                 (in msEigs_which after dneupd call)"
                        mf_message_displayed = .true.
                        call muesli_trace( pause ="yes" )
                        stop
                     end if
                     if( cjg(j) > 0 ) then
                        V%cmplx(:,i) = cmplx( z(:,j), z(:,j+j_shift),   &
                                              kind=MF_DOUBLE )
                     else
                        V%cmplx(:,i) = cmplx( z(:,j+j_shift), -z(:,j),  &
                                              kind=MF_DOUBLE )
                     end if
                  end if
               end do

            end if

         else if( A%data_type == MF_DT_CMPLX .or.                       &
                  A%data_type == MF_DT_SP_CMPLX ) then

            ! complex matrix

            ! for complex matrices, ARPACK deals only with one driver,
            ! and doesn't take care whether A is symmetric or not.

            allocate( zresid(n) ) ! residual vector
            ldzv = n
            allocate( zworkd(3*n) ) ! distributed array for reverse comm.
            info = 0 ! a randomly initial residual vector

            if( method == "OP = inv(A)" ) then
               if( mfIsSymm(A) ) then
                  if( mfIsPosDef(A) ) then
                     A_is_SPD = .true.
                  else
                     A_is_SPD = .false.
                  end if
               else
                  A_is_SPD = .false.
               end if
               if( A_is_SPD ) then
                  call msAssign(U, mfChol(A) )
               else
                  call msLU_mfMatFactor( A, factor )
               end if
            end if

            do

               !-------------------------------------
               ! preparation
               !-------------------------------------
               ido = 0 ! first call to the reverse communication interface
               allocate( zv(n,ncv0) ) ! set of Arnoldi basis vectors
               lzworkl = 3*ncv0**2 + 5*ncv0
               allocate( zworkl(lzworkl) ) ! private workspace
               allocate( rwork(ncv0) ) ! private workspace

               !-------------------------------------
               ! computes eigenvalues and eigenvectors
               !-------------------------------------
               iparam(:) = 0
               iparam(1) = 1 ! method for selecting the implicit shifts
               iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
               if( method == "OP = A" ) then
                  iparam(7) = 1 ! type of eigenproblem : OP = A
               else if( method == "OP = inv(A)" ) then
                  iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
               end if
               finished = .false.
               do while( .not. finished )
                  call znaupd( ido,                                     &
                               "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                               iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                               info )
                  if( ido == 1 .or. ido == -1 ) then
                     if( method == "OP = A" ) then
                        call mf_mat_vec_cmplx( A, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                  zworkd(ipntr(2):ipntr(2)+n-1) )
                     else if( method == "OP = inv(A)" ) then
                        if( A_is_SPD ) then
                           call mf_mat_vec_inv_chol_cmplx( U, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                              zworkd(ipntr(2):ipntr(2)+n-1) )
                        else
                           call mf_mat_vec_inv_factor_cmplx( factor, zworkd(ipntr(1):ipntr(1)+n-1), &
                                                                     zworkd(ipntr(2):ipntr(2)+n-1) )
                        end if
                     end if
                  else if( ido == 99 ) then
                     finished = .true.
                  end if
               end do

               ! check for some errors
               if( info < 0 ) then
                  write(info_char,"(I0)") info
                  call PrintMessage( "msEigs", "E",                     &
                                     "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                     "Error returned, info = " // info_char, &
                                     "Check the ARPACK documentation" )
                  go to 99
               end if

               ! check for convergence
               nconv = iparam(5)
               if( nconv >= nev ) then
                  flag = .true.
                  exit
               end if

               ! we cannot increase further ncv0
               if( ncv0 == n ) then
                  call PrintMessage( "msEigs", "E",                     &
                                     "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                     "(even after increase of ncv to its maximum value)" )
                  flag = .false.
                  go to 99
               end if
               ncv0 = 2*ncv0
               if( ncv0 > n ) then
                  ncv0 = n
               end if
               deallocate( zv )
               deallocate( zworkl )
               deallocate( rwork )

               info = 1 ! re-uses the residual vector

            end do

            if( method == "OP = inv(A)" ) then
               call msFreeMatFactor( factor )
            end if
            if( .not. mfIsEmpty(U) ) then
               call msSilentRelease( U )
            end if

            !-------------------------------------
            ! no fatal errors occurred
            ! post processing
            !-------------------------------------
            ! .true. : compute Ritz vectors
            ! 'A' : compute all the NEV Ritz values
            allocate( select(ncv0) ) ! for the present call : workspace
            allocate( zd(nev+1) ) ! Ritz values
            ! init. to avoid error detection with valgrind; moreover,
            ! NaN values are useful car they will be ignored by the sort.
            zd(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
            ldzz = n
            allocate( zz(n,nev) )
            allocate( zworkev(2*ncv0) ) ! private workspace
            zsigma = (0.0d0,0.0d0)
            call zneupd( .true., "A", select(1), zd(1), zz(1,1), ldzz, zsigma, zworkev(1), &
                         "I", n, which1, nev, tol0, zresid(1), ncv0, zv(1,1), ldzv, &
                         iparam, ipntr, zworkd(1), zworkl(1), lzworkl, rwork(1), &
                         info )

            if( info == -14 ) then
               call PrintMessage( "msEigs", "W",                        &
                                  "ARPACK:",                            &
                                  "znaupd didn't find any eigenvalues to sufficient accuracy" )
               flag = .false.
               go to 99
            end if

            ! check for some errors
            if( info /= 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( "msEigs", "E",                        &
                                  "'zneupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               flag = .false.
               go to 99
            end if

            if( mfIsSymm(A) ) then

               D%data_type = MF_DT_DBLE
               D%shape = [ nev, 1 ]
               allocate( D%double(nev,1) )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(zd)) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msAssign( mf_tmp, mf(real(zd)) )
               else
                  ! algebraic imaginary part
                  call msAssign( mf_tmp, mf(aimag(zd)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, nev
                  D%double(i,1) = real( zd(ind(i)) )
               end do

            else

               D%data_type = MF_DT_CMPLX
               D%shape = [ nev, 1 ]
               allocate( D%cmplx(nev,1) )

               ! sorting eigenvalues
               if( which0 == "LM" .or. which0 == "LR" .or. which0 == "LI" ) then
                  mode_sort = "descending"
               else
                  mode_sort = "ascending"
               end if
               if( which0 == "LM" .or. which0 == "SM" ) then
                  ! magnitude
                  call msAssign( mf_tmp, mf(abs(zd)) )
               else if( which0 == "LR" .or. which0 == "SR" ) then
                  ! algebraic real part
                  call msAssign( mf_tmp, mf(real(zd)) )
               else
                  ! algebraic imaginary part
                  call msAssign( mf_tmp, mf(aimag(zd)) )
               end if
               call msSort( mfOut(mf_dummy,mf_ind), mf_tmp, mode_sort )
               allocate( ind(nev+1) )
               ind = mf_ind
               do i = 1, nev
                  D%cmplx(i,1) = zd(ind(i))
               end do

            end if

            V%data_type = MF_DT_CMPLX
            V%shape = [ n, nev ]
            allocate( V%cmplx(n,nev) )

            ! change the ouput order
            do i = 1, nev
               V%cmplx(:,i) = zz(:,ind(i))
            end do

         end if

      end if

 99   continue

      call msSilentRelease( mf_dummy, mf_ind, mf_tmp )

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end subroutine msEigs_which
