! f90 include file

! currently, uses only:
!     mf_mat_vec_cmplx()
!     mf_mat_vec_inv_factor_cmplx()

!_______________________________________________________________________
!
   function mfEigs_sigma_real( A, k, sigma, tol, ncv ) result( out )

      type(mfArray)                              :: A
      integer,              intent(in)           :: k
      real(kind=MF_DOUBLE), intent(in)           :: sigma
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      integer,              intent(in), optional :: ncv

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      complex(kind=MF_DOUBLE) :: csigma

      csigma = sigma
      call msAssign( out, mfEigs_sigma_cmplx( A, k, csigma, tol, ncv ) )

      out%status_temporary = .true.

#endif
   end function mfEigs_sigma_real
!_______________________________________________________________________
!
   function mfEigs_sigma_cmplx( A, k, sigma, tol, ncv ) result( out )

      type(mfArray)                                 :: A
      integer,                 intent(in)           :: k
      complex(kind=MF_DOUBLE), intent(in)           :: sigma
      real(kind=MF_DOUBLE),    intent(in), optional :: tol
      integer,                 intent(in), optional :: ncv

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! equivalent of 'eigs' in MATLAB (simplified version)
      !
      ! returns a vector containing 'k' eigenvalues of A.
      !
      ! A may be : real/complex, sparse/dense

      integer :: n, ncol, nnz
      logical :: finished

      ! declarations for ARPACK
      integer :: ido, nev, ncv0, ldv, lworkl, info, ldz, i, nconv
      real(kind=MF_DOUBLE) :: tol0
      complex(kind=MF_DOUBLE), allocatable :: resid(:), v(:,:), d(:)
      complex(kind=MF_DOUBLE), allocatable :: workd(:), workl(:), workev(:)
      real(kind=MF_DOUBLE), allocatable :: rwork(:)
      integer :: iparam(11), ipntr(14)
      logical, allocatable :: select(:)
      complex(kind=MF_DOUBLE) :: z(1,1)
      type(mfArray) :: B
      character(len=6) :: info_char
      character(len=12) :: method
      type(mfMatFactor) :: factor
      character(len=2) :: which1

      ! new sorting of eigenvalues
      type(mfArray) :: mf_dummy, mf_ind
      integer, allocatable :: ind(:)

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "mfEigs", "W",                              &
                            "'A' is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfEigs", "E",                              &
                            "'A' cannot be boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfEigs", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( k < 1 ) then
         call PrintMessage( "mfEigs", "W",                              &
                            "k < 1; output will be empty!" )
         go to 99
      end if

      ! square matrix ?
      if( A%shape(1) /= A%shape(2) ) then
         call PrintMessage( "mfEigs", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      n = A%shape(1)
      if( mfIsSparse(A) ) then
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         if( nnz == 0 ) then

            call msAssign( out, mfZeros(k,1) )
            out%status_temporary = .true.
            go to 99
         end if
      end if

      if( k > n-2 ) then
         call PrintMessage( "mfEigs", "W",                              &
                            "the ARPACK routine used allows only the computation", &
                            "of n-2 max eigenvalues/vectors!",         &
                            "(use mfEig to retrieve all eigenvalues/vectors)" )
         nev = n-2
      else
         nev = k
      end if

      ! relative accuracy of the Ritz values
      if( present(tol) ) then
         tol0 = tol
      else
         tol0 = MF_EPS
      end if

      ! how many Arnoldi vectors are generated
      if( present(ncv) ) then
         ncv0 = ncv
      else
         ncv0 = 2*nev+1 ! recommended value from User Guide
      end if

      if( ncv0 < 2*nev+1 ) then
         call PrintMessage( "mfEigs", "W",                              &
                            "ARPACK: ncv < 2*k+1",                      &
                            "(2*k+1 is the recommended value from the ARPACK User Guide)", &
                            "-> setting : ncv = 2*k+1" )
         ncv0 = 2*nev+1
      end if

      if( ncv0 > n ) then
         if( present(ncv) ) then
            call PrintMessage( "mfEigs", "W",                           &
                               "ARPACK: ncv > n",                       &
                               "It seems that you are working with a small matrix, or", &
                               "that a great number of eigs have been requested...", &
                               "(ARPACK works well when computing few eigenvalues", &
                               "of a large matrix)",                    &
                               "-> setting : ncv = n" )
         end if
         ncv0 = n
      end if

      ! Warn if matrix A-sigma*I is close to singular
      if( mfIsSparse(A) ) then
         ! B should be sparse, as A
         call msAssign( B, A-sigma*mfSpEye(n) )
      else
         ! B should be dense, as A
         call msAssign( B, A-sigma*mfEye(n) )
      end if

      method = "OP = inv(A)"
      which1 = "LM"

      if( mfDble(mfRCond(B)) <= MF_EPS*10.0d0 ) then
         call PrintMessage( "mfEigs", "W",                              &
                            "sigma is closed to an eigenvalue of 'A'",  &
                            "Computation can be very inaccurate!" )
         method = "OP = A"
         which1 = "SM"
      end if

      ! using ARPACK-2

      if( present(ncv) ) then

         ! for complex matrices, ARPACK deals only with one driver,
         ! and doesn't take care whether A is symmetric or not.

         if( method == "OP = inv(A)" ) then
            call msLU_mfMatFactor( B, factor )
         end if

         !-------------------------------------
         ! preparation
         !-------------------------------------
         ido = 0 ! first call to the reverse communication interface
         allocate( resid(n) ) ! residual vector
         allocate( v(n,ncv0) ) ! set of Arnoldi basis vectors
         ldv = n
         allocate( workd(3*n) ) ! distributed array for reverse comm.
         lworkl = 3*ncv0**2 + 5*ncv0
         allocate( workl(lworkl) ) ! private workspace
         allocate( rwork(ncv0) ) ! private workspace

         !-------------------------------------
         ! computes eigenvalues and eigenvectors
         !-------------------------------------
         iparam(:) = 0
         iparam(1) = 1 ! method for selecting the implicit shifts
         ! 'LM' : NEV eigenvalues of smallest magnitude about sigma
         !        largest for OP=inv(A) -> smallest for OP=A
         iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
         if( method == "OP = A" ) then
            iparam(7) = 1 ! type of eigenproblem : OP = A
         else if( method == "OP = inv(A)" ) then
            iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
         end if
         info = 0 ! a randomly initial residual vector
         finished = .false.
         do while( .not. finished )
            call znaupd( ido,                                           &
                         "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                         iparam, ipntr, workd(1), workl(1), lworkl, rwork(1), &
                         info )
            if( ido == 1 .or. ido == -1 ) then
               if( method == "OP = A" ) then
                  call mf_mat_vec_cmplx( B, workd(ipntr(1):ipntr(1)+n-1), &
                                            workd(ipntr(2):ipntr(2)+n-1) )
               else if( method == "OP = inv(A)" ) then
                  call mf_mat_vec_inv_factor_cmplx( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                            workd(ipntr(2):ipntr(2)+n-1) )
               end if
            else if( ido == 99 ) then
               finished = .true.
            end if
         end do

         if( method == "OP = inv(A)" ) then
            call msFreeMatFactor( factor )
         end if

         ! check for some errors
         if( info < 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "mfEigs", "E",                           &
                               "'znaupd' (ARPACK) cannot compute eigenvalues", &
                               "Error returned, info = " // info_char,  &
                               "Check the ARPACK documentation" )
            go to 99
         end if

         ! check for convergence
         nconv = iparam(5)
         if( nconv /= nev ) then
            call PrintMessage( "mfEigs", "W",                           &
                               "ARPACK: nconv /= nev",                  &
                               "Number of converged Ritz values is not equal", &
                               "to the number of requested eigenvalues." )
         end if

         !-------------------------------------
         ! no fatal errors occurred
         ! post processing
         !-------------------------------------
         ! .false. : don't want the Ritz vectors
         ! 'A' : compute all the NEV Ritz values
         allocate( select(ncv0) ) ! for the present call : workspace
         allocate( d(nev+1) ) ! Ritz values
         ! init. to avoid error detection with valgrind; moreover,
         ! NaN values are useful because they will be ignored by the sort.
         d(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
         ldz = 1
         allocate( workev(2*ncv0) ) ! private workspace
         call zneupd( .false., "A", select(1), d(1), z(1,1), ldz, sigma, workev(1), &
                      "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), &
                      ldv, iparam, ipntr, workd(1), workl(1), lworkl, rwork(1), &
                      info )

         if( info == -14 ) then
            call PrintMessage( "mfEigs", "W",                           &
                               "ARPACK:",                               &
                                  "znaupd didn't find any eigenvalues to sufficient accuracy", &
                                  "-> try to increase ncv" )
            go to 99
         end if

         call msSilentRelease( B )

         ! check for some errors
         if( info /= 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "mfEigs", "E",                           &
                               "'zneupd' (ARPACK) cannot compute eigenvalues", &
                               "Error returned, info = " // info_char,  &
                               "Check the ARPACK documentation" )
            go to 99
         end if

         if( mfIsSymm(A) ) then

            out%data_type = MF_DT_DBLE
            out%shape = [ nev, 1 ]
            allocate( out%double(nev,1) )

            ! non-converged eigenvalues will contain NaN values, and
            ! the corresponding eigenvectors will also contain NaNs.
            out%double(:,:) = MF_NAN

            ! sorting eigenvalues by ascending module
            call msSort( mfOut(mf_dummy,mf_ind), mf(d) )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               out%double(i,1) = real( d(ind(i)) )
            end do

         else

            out%data_type = MF_DT_CMPLX
            out%shape = [ nev, 1 ]
            allocate( out%cmplx(nev,1) )

            ! non-converged eigenvalues will contain NaN values, and
            ! the corresponding eigenvectors will also contain NaNs.
            out%cmplx(:,:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )

            ! sorting eigenvalues by ascending module
            call msSort( mfOut(mf_dummy,mf_ind), mf(d) )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               out%cmplx(i,1) = d(ind(i))
            end do

         end if

      else ! .not. present(ncv)

         ! for complex matrices, ARPACK deals only with one driver,
         ! and doesn't take care whether A is symmetric or not.

         allocate( resid(n) ) ! residual vector
         ldv = n
         allocate( workd(3*n) ) ! distributed array for reverse comm.
         info = 0 ! a randomly initial residual vector

         if( method == "OP = inv(A)" ) then
            call msLU_mfMatFactor( B, factor )
         end if

         do

            !-------------------------------------
            ! preparation
            !-------------------------------------
            ido = 0 ! first call to the reverse communication interface
            allocate( v(n,ncv0) ) ! set of Arnoldi basis vectors
            lworkl = 3*ncv0**2 + 5*ncv0
            allocate( workl(lworkl) ) ! private workspace
            allocate( rwork(ncv0) ) ! private workspace

            !-------------------------------------
            ! computes eigenvalues and eigenvectors
            !-------------------------------------
            iparam(:) = 0
            iparam(1) = 1 ! method for selecting the implicit shifts
            ! 'LM' : NEV eigenvalues of smallest magnitude about sigma
            !        largest for OP=inv(A) -> smallest for OP=A
            iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
            if( method == "OP = A" ) then
               iparam(7) = 1 ! type of eigenproblem : OP = A
            else if( method == "OP = inv(A)" ) then
               iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
            end if
            finished = .false.
            do while( .not. finished )
               call znaupd( ido,                                        &
                            "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, rwork(1), &
                            info )
               if( ido == 1 .or. ido == -1 ) then
                  if( method == "OP = A" ) then
                     call mf_mat_vec_cmplx( B, workd(ipntr(1):ipntr(1)+n-1), &
                                               workd(ipntr(2):ipntr(2)+n-1) )
                  else if( method == "OP = inv(A)" ) then
                     call mf_mat_vec_inv_factor_cmplx( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                               workd(ipntr(2):ipntr(2)+n-1) )
                  end if
               else if( ido == 99 ) then
                  finished = .true.
               end if
            end do

            ! check for some errors
            if( info < 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( "mfEigs", "E",                        &
                                  "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               go to 99
            end if

            ! check for convergence
            nconv = iparam(5)
            if( nconv >= nev ) then
               exit
            end if

            ! we must increase ncv0
            if( ncv0 == n ) then
               call PrintMessage( "mfEigs", "E",                        &
                                  "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                  "(even after increase of ncv to its maximum value)" )
               go to 99
            end if
            ncv0 = 2*ncv0
            if( ncv0 > n ) then
               ncv0 = n
            end if
            deallocate( v )
            deallocate( workl )
            deallocate( rwork )

            info = 1 ! re-uses the residual vector

         end do

         if( method == "OP = inv(A)" ) then
            call msFreeMatFactor( factor )
         end if

         !-------------------------------------
         ! no fatal errors occurred
         ! post processing
         !-------------------------------------
         ! .false. : don't want the Ritz vectors
         ! 'A' : compute all the NEV Ritz values
         allocate( select(ncv0) ) ! for the present call : workspace
         allocate( d(nev+1) ) ! Ritz values
         ! init. to avoid error detection with valgrind; moreover,
         ! NaN values are useful because they will be ignored by the sort.
         d(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
         ldz = 1
         allocate( workev(2*ncv0) ) ! private workspace
         call zneupd( .false., "A", select(1), d(1), z(1,1), ldz, sigma, workev(1), &
                      "I", n, which1, nev, tol0, resid(1), ncv0, v(1,1), &
                      ldv, iparam, ipntr, workd(1), workl(1), lworkl, rwork(1), &
                      info )

         if( info == -14 ) then
            call PrintMessage( "mfEigs", "W",                           &
                               "ARPACK:",                               &
                               "znaupd didn't find any eigenvalues to sufficient accuracy" )
            go to 99
         end if

         call msSilentRelease( B )

         ! check for some errors
         if( info /= 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "mfEigs", "E",                           &
                               "'zneupd' (ARPACK) cannot compute eigenvalues", &
                               "Error returned, info = " // info_char,  &
                               "Check the ARPACK documentation" )
            go to 99
         end if

         if( mfIsSymm(A) ) then

            out%data_type = MF_DT_DBLE
            out%shape = [ nev, 1 ]
            allocate( out%double(nev,1) )

            ! sorting eigenvalues by ascending module
            call msSort( mfOut(mf_dummy,mf_ind), mf(d) )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, nev
               out%double(i,1) = real( d(ind(i)) )
            end do

         else

            out%data_type = MF_DT_CMPLX
            out%shape = [ nev, 1 ]
            allocate( out%cmplx(nev,1) )

            ! sorting eigenvalues by ascending module
            call msSort( mfOut(mf_dummy,mf_ind), mf(d) )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, nev
               out%cmplx(i,1) = d(ind(i))
            end do

         end if

      end if

      out%status_temporary = .true.

 99   continue

      call msSilentRelease( mf_dummy, mf_ind )

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end function mfEigs_sigma_cmplx
!_______________________________________________________________________
!
   subroutine msEigs_sigma_real( out, A, k, sigma, tol, ncv )

      type(mfArray)                              :: A
      integer,              intent(in)           :: k
      real(kind=MF_DOUBLE), intent(in)           :: sigma
      real(kind=MF_DOUBLE), intent(in), optional :: tol
      integer,              intent(in), optional :: ncv

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      type(mfArray), pointer :: V, D, flag
      complex(kind=MF_DOUBLE) :: csigma

      ! 3 out-args must be specified
      if( out%n /= 3 ) then
         call PrintMessage( "msEigs", "E",                              &
                            "three output args required!",             &
                            "syntax is: call msEigs ( mfOut(V,D,flag), A, ... )" )
         return
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msEigs", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         return
      end if

      V => out%ptr1
      D => out%ptr2
      call msSilentRelease( V, D )
      flag => out%ptr3
      call msSilentRelease( flag )

      csigma = sigma
      call msEigs_sigma_cmplx( mfout(V, D, flag), A, k, csigma, tol, ncv )

#endif
   end subroutine msEigs_sigma_real
!_______________________________________________________________________
!
   subroutine msEigs_sigma_cmplx( out, A, k, sigma, tol, ncv )

      type(mfArray)                                 :: A
      integer,                 intent(in)           :: k
      complex(kind=MF_DOUBLE), intent(in)           :: sigma
      real(kind=MF_DOUBLE),    intent(in), optional :: tol
      integer,                 intent(in), optional :: ncv

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! equivalent of 'eigs' in MATLAB (simplified version)
      !
      ! returns a vector containing 'k' eigenvalues of A, and a matrix
      ! whose columns are the corresponding eigenvectors.
      !
      ! optionally returns a convergence flag (boolean mfArray):
      !     flag = TRUE if all eigenvalues have converged
      !     flag = FALSE else

      ! A may be : real/complex, sparse/dense

      type(mfArray), pointer :: V, D, flag
      integer :: n, ncol, nnz
      logical :: finished, real_eigenvects

      ! declarations for ARPACK
      integer :: ido, nev, ncv0, ldv, lworkl, info, ldz, i, nconv
      real(kind=MF_DOUBLE) :: tol0, norm
      complex(kind=MF_DOUBLE), allocatable :: resid(:), vv(:,:), dd(:)
      complex(kind=MF_DOUBLE), allocatable :: workd(:), workl(:), workev(:)
      real(kind=MF_DOUBLE), allocatable :: rwork(:)
      complex(kind=MF_DOUBLE), allocatable :: z(:,:)
      integer :: iparam(11), ipntr(14)
      logical, allocatable :: select(:)
      type(mfArray) :: B
      character(len=6) :: info_char
      character(len=12) :: method
      type(mfMatFactor) :: factor
      character(len=2) :: which1

      ! new sorting of eigenvalues
      type(mfArray) :: mf_dummy, mf_ind
      integer, allocatable :: ind(:)

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( A )

      ! check if A is real, symmetric : in such a case, eigenvectors
      ! are real, and we must take the real part of the matrix V, then
      ! normalise its columns.
      real_eigenvects = .false.
      if( mfIsReal(A) ) then
         if( mfIsSymm(A) ) then
            real_eigenvects = .true.
         end if
      end if

      ! 3 out-args must be specified
      if( out%n /= 3 ) then
         call PrintMessage( "msEigs", "E",                              &
                            "three output args required!",             &
                            "syntax is: call msEigs ( mfOut(V,D,flag), A, ... )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msEigs", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      V => out%ptr1
      D => out%ptr2
      call msSilentRelease( V, D )
      flag => out%ptr3
      call msSilentRelease( flag )
      flag = .false.

      if( A%data_type == MF_DT_EMPTY ) then
         call PrintMessage( "msEigs", "W",                              &
                            "'A' is empty!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msEigs", "E",                              &
                            "'A' cannot be boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msEigs", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( k < 1 ) then
         call PrintMessage( "msEigs", "W",                              &
                            "k < 1; output will be empty!" )
         go to 99
      end if

      ! square matrix ?
      if( A%shape(1) /= A%shape(2) ) then
         call PrintMessage( "msEigs", "E",                              &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      n = A%shape(1)
      if( mfIsSparse(A) ) then
         ncol = A%shape(2)
         nnz = A%j(ncol+1) - 1
         if( nnz == 0 ) then
            call msAssign( V, mfEye(n,k) )
            call msAssign( D, mfZeros(k,1) )
            flag = .true.
            go to 99
         end if
      end if

      if( k > n-2 ) then
         call PrintMessage( "msEigs", "W",                              &
                            "the ARPACK routine used allows only the computation", &
                            "of n-2 max eigenvalues/vectors!",         &
                            "(use msEig to retrieve all eigenvalues/vectors)" )
         nev = n-2
      else
         nev = k
      end if

      ! relative accuracy of the Ritz values
      if( present(tol) ) then
         tol0 = tol
      else
         tol0 = MF_EPS
      end if

      ! how many Arnoldi vectors are generated
      if( present(ncv) ) then
         ncv0 = ncv
      else
         ncv0 = 2*nev+1 ! recommended value from User Guide
      end if

      if( ncv0 < 2*nev+1 ) then
         call PrintMessage( "msEigs", "W",                              &
                            "ARPACK: ncv < 2*k+1",                      &
                            "(2*k+1 is the recommended value from the ARPACK User Guide)", &
                            "-> setting : ncv = 2*k+1" )
         ncv0 = 2*nev+1
      end if

      if( ncv0 > n ) then
         if( present(ncv) ) then
            call PrintMessage( "msEigs", "W",                           &
                               "ARPACK: ncv > n",                       &
                               "It seems that you are working with a small matrix, or", &
                               "that a great number of eigs have been requested...", &
                               "(ARPACK works well when computing few eigenvalues", &
                               "of a large matrix)",                    &
                               "-> setting : ncv = n" )
         end if
         ncv0 = n
      end if

      ! Warn if matrix A-sigma*I is close to singular
      if( mfIsSparse(A) ) then
         ! B should be sparse, as A
         call msAssign( B, A-sigma*mfSpEye(n) )
      else
         ! B should be dense, as A
         call msAssign( B, A-sigma*mfEye(n) )
      end if

      method = "OP = inv(A)"
      which1 = "LM"

      if( mfDble(mfRCond(B)) <= MF_EPS*10.0d0 ) then
         call PrintMessage( "msEigs", "W",                              &
                            "sigma is closed to an eigenvalue of 'A'",  &
                            "Computation can be very inaccurate!" )
         method = "OP = A"
         which1 = "SM"
      end if

      ! using ARPACK-2

      if( present(ncv) ) then

         ! for complex matrices, ARPACK deals only with one driver,
         ! and doesn't take care whether A is symmetric or not.

         if( method == "OP = inv(A)" ) then
            call msLU_mfMatFactor( B, factor )
         end if

         !-------------------------------------
         ! preparation
         !-------------------------------------
         ido = 0 ! first call to the reverse communication interface
         allocate( resid(n) ) ! residual vector
         allocate( vv(n,ncv0) ) ! set of Arnoldi basis vectors
         ldv = n
         allocate( workd(3*n) ) ! distributed array for reverse comm.
         lworkl = 3*ncv0**2 + 5*ncv0
         allocate( workl(lworkl) ) ! private workspace
         allocate( rwork(ncv0) ) ! private workspace

         !-------------------------------------
         ! computes eigenvalues and eigenvectors
         !-------------------------------------
         iparam(:) = 0
         iparam(1) = 1 ! method for selecting the implicit shifts
         ! 'LM' : NEV eigenvalues of smallest magnitude about sigma
         !        largest for OP=inv(A) -> smallest for OP=A
         iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
         if( method == "OP = A" ) then
            iparam(7) = 1 ! type of eigenproblem : OP = A
         else if( method == "OP = inv(A)" ) then
            iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
         end if
         info = 0 ! a randomly initial residual vector
         finished = .false.
         do while( .not. finished )
            call znaupd( ido,                                           &
                         "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                         iparam, ipntr, workd(1), workl(1), lworkl, rwork(1), &
                         info )
            if( ido == 1 .or. ido == -1 ) then
               if( method == "OP = A" ) then
                  call mf_mat_vec_cmplx( B, workd(ipntr(1):ipntr(1)+n-1), &
                                            workd(ipntr(2):ipntr(2)+n-1) )
               else if( method == "OP = inv(A)" ) then
                  call mf_mat_vec_inv_factor_cmplx( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                            workd(ipntr(2):ipntr(2)+n-1) )
               end if
            else if( ido == 99 ) then
               finished = .true.
            end if
         end do

         if( method == "OP = inv(A)" ) then
            call msFreeMatFactor( factor )
         end if


         ! check for some errors
         if( info < 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "msEigs", "E",                           &
                               "'znaupd' (ARPACK) cannot compute eigenvalues", &
                               "Error returned, info = " // info_char,  &
                               "Check the ARPACK documentation" )
            go to 99
         end if

         ! check for convergence
         nconv = iparam(5)
         if( nconv >= nev ) then
            flag = .true.
         end if

         !-------------------------------------
         ! no fatal errors occurred
         ! post processing
         !-------------------------------------
         ! .true. : don't want the Ritz vectors
         ! 'A' : compute all the NEV Ritz values
         allocate( select(ncv0) ) ! for the present call : workspace
         allocate( dd(nev+1) ) ! Ritz values
         ! init. to avoid error detection with valgrind; moreover,
         ! NaN values are useful car they will be ignored by the sort.
         dd(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
         ldz = n
         allocate( z(n,nev) )
         allocate( workev(2*ncv0) ) ! private workspace
         call zneupd( .true., "A", select(1), dd(1), z(1,1), ldz, sigma, workev(1), &
                      "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), &
                      ldv, iparam, ipntr, workd(1), workl(1), lworkl, rwork(1), &
                      info )

         if( info == -14 ) then
            call PrintMessage( "msEigs", "W",                           &
                               "ARPACK:",                               &
                               "znaupd didn't find any eigenvalues to sufficient accuracy" )
            flag = .false.
            go to 99
         end if

         call msSilentRelease( B )

         ! check for some errors
         if( info /= 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "msEigs", "E",                           &
                               "'zneupd' (ARPACK) cannot compute eigenvalues", &
                               "Error returned, info = " // info_char,  &
                               "Check the ARPACK documentation" )
            flag = .false.
            go to 99
         end if

         if( mfIsSymm(A) ) then

            D%data_type = MF_DT_DBLE
            D%shape = [ nev, 1 ]
            allocate( D%double(nev,1) )

            ! non-converged eigenvalues will contain NaN values, and
            ! the corresponding eigenvectors will also contain NaNs.
            D%double(:,:) = MF_NAN

            ! sorting eigenvalues by ascending module
            call msSort( mfOut(mf_dummy,mf_ind), mf(dd) )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               D%double(i,1) = real( dd(ind(i)) )
            end do

         else

            D%data_type = MF_DT_CMPLX
            D%shape = [ nev, 1 ]
            allocate( D%cmplx(nev,1) )

            ! non-converged eigenvalues will contain NaN values, and
            ! the corresponding eigenvectors will also contain NaNs.
            D%cmplx(:,:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )

            ! sorting eigenvalues by ascending module
            call msSort( mfOut(mf_dummy,mf_ind), mf(dd) )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               D%cmplx(i,1) = dd(ind(i))
            end do

         end if

         if( real_eigenvects ) then
            V%data_type = MF_DT_DBLE
            V%shape = [ n, nev ]
            allocate( V%double(n,nev) )

            ! non-converged eigenvalues will contain NaN values, and
            ! the corresponding eigenvectors will also contain NaNs.
            V%double(:,:) = MF_NAN

            ! change the ouput order and normalize
            deallocate( rwork )
            allocate( rwork(n) )
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               rwork(:) = real(z(:,ind(i)))
               norm = mfNorm( mf(rwork) )
               if( norm /= 0.0d0 ) then
                  V%double(:,i) = rwork(:)/norm
               else
                  V%double(:,i) = 0.0d0
               end if
            end do
         else
            V%data_type = MF_DT_CMPLX
            V%shape = [ n, nev ]
            allocate( V%cmplx(n,nev) )

            ! non-converged eigenvalues will contain NaN values, and
            ! the corresponding eigenvectors will also contain NaNs.
            V%cmplx(:,:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )

            ! change the output order
            do i = 1, min(nev,nconv) ! limited to the converged ritz values!
               V%cmplx(:,i) = z(:,ind(i))
            end do
         end if

      else ! .not. present(ncv)

         ! for complex matrices, ARPACK deals only with one driver,
         ! and doesn't take care whether A is symmetric or not.

         allocate( resid(n) ) ! residual vector
         ldv = n
         allocate( workd(3*n) ) ! distributed array for reverse comm.
         info = 0 ! a randomly initial residual vector

         if( method == "OP = inv(A)" ) then
            call msLU_mfMatFactor( B, factor )
         end if

         do

            !-------------------------------------
            ! preparation
            !-------------------------------------
            ido = 0 ! first call to the reverse communication interface
            allocate( vv(n,ncv0) ) ! set of Arnoldi basis vectors
            lworkl = 3*ncv0**2 + 5*ncv0
            allocate( workl(lworkl) ) ! private workspace
            allocate( rwork(ncv0) ) ! private workspace

            !-------------------------------------
            ! computes eigenvalues and eigenvectors
            !-------------------------------------
            iparam(:) = 0
            iparam(1) = 1 ! method for selecting the implicit shifts
            ! 'LM' : NEV eigenvalues of smallest magnitude about sigma
            !        largest for OP=inv(A) -> smallest for OP=A
            iparam(3) = MF_ARNOLDI_ITER_MAX ! maximum number of Arnoldi update iterations
            if( method == "OP = A" ) then
               iparam(7) = 1 ! type of eigenproblem : OP = A
            else if( method == "OP = inv(A)" ) then
               iparam(7) = 3 ! type of eigenproblem : OP = inv(A)
            end if
            finished = .false.
            do while( .not. finished )
               call znaupd( ido,                                        &
                            "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), ldv, &
                            iparam, ipntr, workd(1), workl(1), lworkl, rwork(1), &
                            info )
               if( ido == 1 .or. ido == -1 ) then
                  if( method == "OP = A" ) then
                     call mf_mat_vec_cmplx( B, workd(ipntr(1):ipntr(1)+n-1), &
                                               workd(ipntr(2):ipntr(2)+n-1) )
                  else if( method == "OP = inv(A)" ) then
                     call mf_mat_vec_inv_factor_cmplx( factor, workd(ipntr(1):ipntr(1)+n-1), &
                                                               workd(ipntr(2):ipntr(2)+n-1) )
                  end if
               else if( ido == 99 ) then
                  finished = .true.
               end if
            end do

            ! check for some errors
            if( info < 0 ) then
               write(info_char,"(I0)") info
               call PrintMessage( "msEigs", "E",                        &
                                  "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                  "Error returned, info = " // info_char, &
                                  "Check the ARPACK documentation" )
               go to 99
            end if

            ! check for convergence
            nconv = iparam(5)
            if( nconv >= nev ) then
               flag = .true.
               exit
            end if

            ! we must increase ncv0
            if( ncv0 == n ) then
               call PrintMessage( "msEigs", "W",                        &
                                  "'znaupd' (ARPACK) cannot compute eigenvalues", &
                                  "(even after increase of ncv to its maximum value)" )
               flag = .false.
               go to 99
            end if
            ncv0 = 2*ncv0
            if( ncv0 > n ) then
               ncv0 = n
            end if
            deallocate( vv )
            deallocate( workl )
            deallocate( rwork )

            info = 1 ! re-uses the residual vector

         end do

         if( method == "OP = inv(A)" ) then
            call msFreeMatFactor( factor )
         end if

         !-------------------------------------
         ! no fatal errors occurred
         ! post processing
         !-------------------------------------
         ! .true. : don't want the Ritz vectors
         ! 'A' : compute all the NEV Ritz values
         allocate( select(ncv0) ) ! for the present call : workspace
         allocate( dd(nev+1) ) ! Ritz values
         ! init. to avoid error detection with valgrind; moreover,
         ! NaN values are useful car they will be ignored by the sort.
         dd(:) = cmplx( MF_NAN, MF_NAN, kind=MF_DOUBLE )
         ldz = n
         allocate( z(n,nev) )
         allocate( workev(2*ncv0) ) ! private workspace
         call zneupd( .true., "A", select(1), dd(1), z(1,1), ldz, sigma, workev(1), &
                      "I", n, which1, nev, tol0, resid(1), ncv0, vv(1,1), &
                      ldv, iparam, ipntr, workd(1), workl(1), lworkl, rwork(1), &
                      info )

         if( info == -14 ) then
            call PrintMessage( "msEigs", "W",                           &
                               "ARPACK:",                               &
                               "znaupd didn't find any eigenvalues to sufficient accuracy" )
            flag = .false.
            go to 99
         end if

         call msSilentRelease( B )

         ! check for some errors
         if( info /= 0 ) then
            write(info_char,"(I0)") info
            call PrintMessage( "msEigs", "E",                           &
                               "'zneupd' (ARPACK) cannot compute eigenvalues", &
                               "Error returned, info = " // info_char,  &
                               "Check the ARPACK documentation" )
            flag = .false.
            go to 99
         end if

         if( mfIsSymm(A) ) then

            D%data_type = MF_DT_DBLE
            D%shape = [ nev, 1 ]
            allocate( D%double(nev,1) )

            ! sorting eigenvalues by ascending module
            call msSort( mfOut(mf_dummy,mf_ind), mf(dd) )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, nev
               D%double(i,1) = real( dd(ind(i)) )
            end do

         else

            D%data_type = MF_DT_CMPLX
            D%shape = [ nev, 1 ]
            allocate( D%cmplx(nev,1) )

            ! sorting eigenvalues by ascending module
            call msSort( mfOut(mf_dummy,mf_ind), mf(dd) )
            allocate( ind(nev+1) )
            ind = mf_ind
            do i = 1, nev
               D%cmplx(i,1) = dd(ind(i))
            end do

         end if

         if( real_eigenvects ) then
            V%data_type = MF_DT_DBLE
            V%shape = [ n, nev ]
            allocate( V%double(n,nev) )

            ! change the ouput order and normalize
            deallocate( rwork )
            allocate( rwork(n) )
            do i = 1, nev
               rwork(:) = real(z(:,ind(i)))
               norm = mfNorm( mf(rwork) )
               if( norm /= 0.0d0 ) then
                  V%double(:,i) = rwork(:)/norm
               else
                  V%double(:,i) = 0.0d0
               end if
            end do
         else
            V%data_type = MF_DT_CMPLX
            V%shape = [ n, nev ]
            allocate( V%cmplx(n,nev) )

            ! change the output order
            do i = 1, nev
               V%cmplx(:,i) = z(:,ind(i))
            end do
         end if

      end if

 99   continue

      call msSilentRelease( mf_dummy, mf_ind )

      call msFreeArgs( A )
      call msAutoRelease( A )

      call mf_restore_fpe( )

#endif
   end subroutine msEigs_sigma_cmplx
