! f90 include file

!_______________________________________________________________________
!
   function mfChol( A ) result( out )

      type(mfArray), target :: A
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! A : dense matrix
      !
      ! returns a upper triangular matrix such that A = U'*U
      !
      ! A must be symmetric and positive definite

      integer :: n, lda, info
      type(mfArray), pointer :: A_copy
      character(len=1) :: luplo
      character(len=4) :: info_char
      logical :: A_copy_is_allocated

      integer :: i, status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfChol", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfChol", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfChol", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfChol", "E",                              &
                            "mfArray 'A' should be dense!",            &
                            "please use 'msChol' for sparse matrices" )
         go to 99
      end if

      ! symmetric matrix ? (and also square)
      if( .not. mfIsSymm(A) ) then
         if( mfIsReal(A) ) then
            call PrintMessage( "mfChol", "E",                           &
                               "'A' must be a symmetric matrix!" )
         else ! complex
            call PrintMessage( "mfChol", "E",                           &
                               "'A' must be an hermitian matrix!" )
         end if
            go to 99
      end if

      ! positive definite ?
      if( A%prop%posd == FALSE ) then
         call PrintMessage( "mfChol", "E",                              &
                            "'A' must be positive definite!" )
         go to 99
      end if

      n = A%shape(1)

      ! making copies because LAPACK overwrites A
      if( A%status_temporary .and. (A%level_protected==1) ) then
         A_copy => A
         A_copy_is_allocated = .false.
      else
         allocate( A_copy ) ! no_mem_trace !
         A_copy = A
         A_copy_is_allocated = .true.
      end if

      if( A%data_type == MF_DT_DBLE ) then

         ! Symmetric Positive Definite system
         ! real square matrix : DPOTRF
         luplo = "U"
         lda = n
         call dpotrf( luplo, n, a_copy%double(1,1), lda, info )
         if( info /= 0 ) then
            ! solution is not ok
            write(info_char,"(I0)") info
            call PrintMessage( "mfChol", "E",                           &
                               "in Cholesky-factorization of A by LAPACK;",&
                               "info = " // info_char,                  &
                               "(perhaps A is not positive definite)",  &
                               "as a side effect, output will be empty, and 'A'",&
                               "will be tagged as 'not pos. def.'." )
            A%prop%posd = FALSE
            go to 99
         else
            call msAssign( out, mfTriu( a_copy ))
         end if

      else if( A%data_type == MF_DT_CMPLX ) then

         ! Hermitian Positive Definite system
         ! complex square matrix : ZPOTRF
         luplo = "U"
         lda = n
         call zpotrf( luplo, n, a_copy%cmplx(1,1), lda, info )
         if( info /= 0 ) then
            ! solution is not ok
            write(info_char,"(I0)") info
            call PrintMessage( "mfChol", "E",                           &
                               "in Cholesky-factorization of A by LAPACK;",&
                               "info = " // info_char,                  &
                               "(perhaps A is not positive definite)",  &
                               "as a side effect, output will be empty." )
            go to 99
         else
            call msAssign( out, mfTriu( a_copy ))
         end if

      end if

      if( A%prop%posd == UNKNOWN ) then
         ! ok, side effect, but it should be more efficient
         A%prop%posd = TRUE
      end if

      if( A_copy_is_allocated ) then
         call msSilentRelease( A_copy )
         deallocate( A_copy ) ! no_mem_trace !
      end if

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_div( a%units(i), 2,                           &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfChol", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI mfChol:) internal error: in processing physical units:"
               write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

      out%prop%triu = TRUE
      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfChol
!_______________________________________________________________________
!
   subroutine msChol( out, A )

      type(mfArray) :: A
      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! A : sparse matrix (driver routine)
      !
      ! if called with two (out) mfArrays:
      !   returns a lower triangular matrix such that L*L' = P'*A*P
      !
      ! if called with one (out) mfMatFactor:
      !   returns the pointer to a complete C structure, containing
      !   all necessary data (i.e. L, P)
      !
      ! A must be symmetric and positive definite

      ! the permutation p is returned as a vector (from 2.6.0)

      type(mfArray), pointer :: L, p
      type(mfMatFactor), pointer :: factor

      integer :: n, nnz
      logical :: case_1, case_2

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! valid cases
      !   two output mfArrays
      case_1 = ( out%n == 2 )                                           &
               .and.                                                    &
               ( out%factor_n == 0 )
      !   one output mfMatFactor
      case_2 = ( out%n == 0 )                                           &
               .and.                                                    &
               ( out%factor_n == 1 )
      if( .not. case_1 .and. .not. case_2 ) then
         call PrintMessage( "msChol", "E",                              &
                            "(sparse case) bad argument call!",         &
                            "syntax is: call msChol ( mfOut(L,p), A )     [mfArrays]", &
                            "       or: call msChol ( mfOut(factors), A ) [mfMatFactor]" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, A ) ) then
         call PrintMessage( "msChol", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msChol", "E",                              &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msChol", "E",                              &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msChol", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( a%shape(1) <= 1 .or. a%shape(2) <= 1 ) then
         call PrintMessage( "msChol", "E",                              &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      if( .not. mfIsSparse(A) ) then
         call PrintMessage( "msChol", "E",                              &
                            "mfArray 'A' should be sparse!",           &
                            "please use 'mfChol' for dense matrices" )
         go to 99
      end if

      n = A%shape(2)
      nnz = A%j(n+1) - 1
      if( nnz == 0 ) then
         ! avoid testing properties of a non allocated matrix
         A%prop%symm = TRUE
         A%prop%posd = FALSE
      end if

      ! symmetric matrix ? (and also square)
      if( .not. mfIsSymm(A) ) then
         call PrintMessage( "msChol", "E",                              &
                            "'A' must be a symmetric matrix!" )
         go to 99
      end if

      ! positive definite ?
      if( A%prop%posd == FALSE ) then
         call PrintMessage( "msChol", "E",                              &
                            "'A' must be positive definite!" )
         go to 99
      end if

      if( case_1 ) then ! two output mfArrays

         L => out%ptr1
         p => out%ptr2
         call msSilentRelease( L, p )
         call msChol_mfArray( A, L, p )

         L%prop%tril = TRUE

      else ! case_2: one output mfMatFactor

         factor => out%factor_ptr1
         call msFreeMatFactor( factor )
         call msChol_mfMatFactor( A, factor )

      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msChol
!_______________________________________________________________________
!
   subroutine msCholSpSymb( factor, A )

      type(mfMatFactor) :: factor
      type(mfArray) :: A
      !------ API end ------
#ifdef _DEVLP

      ! 'msCholSpSymb' makes only the 'symbolic' factorization of the
      ! sparse matrix A.
      !
      ! A (sparse) must be symmetric and positive definite

      integer :: n
      type(mfArray) :: A_copy

      ! CHOLMOD declaration
      integer(kind=MF_ADDRESS) :: c_addr, LL_addr, AA_addr
      integer :: nnz2

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( .not. mfIsSparse(A) ) then
         call PrintMessage( "msCholSpSymb", "E",                        &
                            "mfArray 'A' must be sparse!",             &
                            "please use 'ms/mfChol' for dense matrices" )
         go to 99
      end if

      ! symmetric matrix ? (and also square)
      if( .not. mfIsSymm(A) ) then
         call PrintMessage( "msCholSpSymb", "E",                        &
                            "'A' must be a symmetric matrix!" )
         go to 99
      end if

      if( A%prop%posd == FALSE ) then
         call PrintMessage( "msCholSpSymb", "E",                        &
                            "'A' must be positive definite!" )
         go to 99
      end if

      call msFreeMatFactor( factor )

      n = A%shape(1)

      if( A%data_type == MF_DT_SP_DBLE ) then

         call msAssign( A_copy, mfTriu(A) )
         nnz2 = mfNnz(A_copy)
         call cholmod_llt_symb( n, nnz2, A_copy%j, A_copy%i,            &
                                c_addr, LL_addr, AA_addr )

         ! storing information

         factor%package = 4 ! 4 for CHOLMOD special
         factor%data_type = MF_DT_SP_DBLE
         factor%order = n

         ! the CHOLMOD struct 'LL' will be freed only
         ! in 'msFreeMatFactor' (under the user responsibility)
         allocate( factor%ptr_1 )

         allocate( factor%ptr_2 )

         allocate( factor%ptr_3 )

         factor%ptr_1 = c_addr
         factor%ptr_2 = LL_addr
         factor%ptr_3 = AA_addr

      else if( A%data_type == MF_DT_SP_CMPLX ) then

         call msAssign( A_copy, mfTriu(A) )
         nnz2 = mfNnz(A_copy)
         call cholmod_cmplx_llt_symb( n, nnz2, A_copy%j, A_copy%i,      &
                                      c_addr, LL_addr, AA_addr )

         ! storing information

         factor%package = 4 ! 4 for CHOLMOD special
         factor%data_type = MF_DT_SP_CMPLX
         factor%order = n

         ! the CHOLMOD struct 'LL' will be freed only
         ! in 'msFreeMatFactor' (under the user responsibility)
         allocate( factor%ptr_1 )

         allocate( factor%ptr_2 )

         allocate( factor%ptr_3 )

         factor%ptr_1 = c_addr
         factor%ptr_2 = LL_addr
         factor%ptr_3 = AA_addr

      else

         call PrintMessage( "msCholSpSymb", "E",                        &
                            "bad data type for 'A'!" )
         go to 99

      end if

      call msSilentRelease( A_copy )

      if( mf_phys_units ) then
         factor%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msCholSpSymb
!_______________________________________________________________________
!
   subroutine msCholSpNum( factor, A )

      type(mfMatFactor) :: factor
      type(mfArray) :: A
      !------ API end ------
#ifdef _DEVLP

      ! 'msCholSpNum' makes only the 'numeric' factorization of the
      ! sparse matrix A.
      !
      ! A (sparse) must be symmetric and positive definite

      integer :: n, status
      type(mfArray) :: A_copy

      ! CHOLMOD declaration
      integer(kind=MF_ADDRESS) :: c_addr, LL_addr, AA_addr
      integer :: nnz2

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( .not. mfIsSparse(A) ) then
         call PrintMessage( "msCholSpNum", "E",                         &
                            "mfArray 'A' must be sparse!",             &
                            "please use 'ms/mfChol' for dense matrices" )
         go to 99
      end if

      ! symmetric matrix ? (and also square)
      if( .not. mfIsSymm(A) ) then
         call PrintMessage( "msCholSpNum", "E",                         &
                            "'A' must be a symmetric matrix!" )
         go to 99
      end if

      if( A%prop%posd == FALSE ) then
         call PrintMessage( "msCholSpNum", "E",                         &
                            "'A' must be positive definite!" )
         go to 99
      end if

      ! factor%package must be 4
      if( factor%package /= 4 ) then
         call PrintMessage( "msCholSpNum", "E",                         &
                            "'factor' must come from 'msCholSpSymb'!" )
         go to 99
      end if

      n = A%shape(1)

      if( A%data_type == MF_DT_SP_DBLE ) then

         call msAssign( A_copy, mfTriu(A) )
         nnz2 = mfNnz(A_copy)
         c_addr = factor%ptr_1
         LL_addr = factor%ptr_2
         AA_addr = factor%ptr_3
         call cholmod_llt_num( n, nnz2, A_copy%a, c_addr, LL_addr, AA_addr, &
                               status )

         ! check whether factorization is ok
         if( status /= 0 ) then
            call PrintMessage( "msCholSpNum", "E",                      &
                               "in Cholesky-factorization of A by CHOLMOD:",&
                               "A is not positive definite!",           &
                               "(as a side effect, output will be left empty)" )
            go to 99
         end if

      else if( A%data_type == MF_DT_SP_CMPLX ) then

         call msAssign( A_copy, mfTriu(A) )
         nnz2 = mfNnz(A_copy)
         c_addr = factor%ptr_1
         LL_addr = factor%ptr_2
         AA_addr = factor%ptr_3
         call cholmod_cmplx_llt_num( n, nnz2, A_copy%z, c_addr, LL_addr, AA_addr, &
                                     status )

         ! check whether factorization is ok
         if( status /= 0 ) then
            call PrintMessage( "msCholSpNum", "E",                      &
                               "in Cholesky-factorization of A by CHOLMOD:",&
                               "A is not positive definite!",           &
                               "(as a side effect, output will be left empty)" )
            go to 99
         end if

      else

         call PrintMessage( "msCholSpNum", "E",                         &
                            "bad data type for 'A'!" )
         go to 99

      end if

      if( A%prop%posd == UNKNOWN ) then
         ! ok, side effect, but it should be more efficient
         A%prop%posd = TRUE
      end if

      call msSilentRelease( A_copy )

      if( mf_phys_units ) then
         factor%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msCholSpNum
!_______________________________________________________________________
!
   subroutine msChol_mfArray( A, L, p )

      type(mfArray) :: A
      type(mfArray) :: L, p
      !------ API end ------
#ifdef _DEVLP

      ! this routine is only for internal use: an ordinary user
      ! should always call 'msChol' which makes many tests on arguments.

      ! A : sparse matrix
      !
      ! returns a lower triangular matrix and a permutation, such that:
      !                    L*L' = p'*A*p
      !
      ! A must be symmetric and positive definite
      !
      ! the permutation p is returned as a column vector (from 2.6.0)

      integer :: n
      type(mfArray) :: A_copy

      integer :: i, status

      ! CHOLMOD declaration
      integer(kind=MF_ADDRESS) :: c_addr, LL_addr, S_addr
      integer :: nnz2, lnz
      real(kind=MF_DOUBLE), allocatable :: tmp_r(:), tmp_i(:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      n = A%shape(1)

      if( A%data_type == MF_DT_SP_DBLE ) then

         call msAssign( A_copy, mfTriu(A) )
         nnz2 = mfNnz(A_copy)
         call cholmod_llt_prep( n, nnz2, A_copy%j, A_copy%i, A_copy%a,  &
                                c_addr, LL_addr, S_addr, lnz )

         ! check whether factorization is ok
         if( lnz == 0 ) then
            call PrintMessage( "msChol", "E",                           &
                               "in Cholesky-factorization of A by CHOLMOD:",&
                               "A is not positive definite!",           &
                               "(as a side effect, output will be left empty)" )
            go to 99
         end if

         call msAssign( L, mfSpAlloc( n, n, nzmax=lnz ) )

         p%data_type = MF_DT_PERM_VEC
         p%shape = [ n, 1 ]
         allocate( p%i(n) )

         ! getting the lower factor L and the permutation applied to
         ! the columns of A
         call cholmod_get_factor( c_addr, LL_addr, S_addr, n, lnz,      &
                                  L%j, L%i, L%a, p%i )

      else if( A%data_type == MF_DT_SP_CMPLX ) then

         call msAssign( A_copy, mfTriu(A) )
         nnz2 = mfNnz(A_copy)
         call cholmod_cmplx_llt_prep( n, nnz2, A_copy%j, A_copy%i,      &
                                      A_copy%z,                         &
                                      c_addr, LL_addr, S_addr, lnz )

         ! check whether factorization is ok
         if( lnz == 0 ) then
            call PrintMessage( "msChol", "E",                           &
                               "in Cholesky-factorization of A by CHOLMOD:",&
                               "A is not positive definite!",           &
                               "(as a side effect, output will be left empty)" )
            go to 99
         end if

         call msAssign( L, mfSpAlloc( n, n, nzmax=lnz, kind="complex" ) )

         p%data_type = MF_DT_PERM_VEC
         p%shape = [ n, 1 ]
         allocate( P%i(n) )

         allocate( tmp_r(lnz), tmp_i(lnz) )
         ! getting the lower factor L and the permutation applied to
         ! the columns of A
         call cholmod_cmplx_get_factor( c_addr, LL_addr, S_addr, n, lnz, &
                                        L%j, L%i, L%z, p%i )

      end if

      ! All matrices returned by SuiteSparse are "row sorted"
      L%row_sorted = TRUE

      if( A%prop%posd == UNKNOWN ) then
         ! ok, side effect, but it should be more efficient
         A%prop%posd = TRUE
      end if

      call msSilentRelease( A_copy )

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_div( a%units(i), 2,                           &
                               L%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "msChol", "E",                        &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               write(STDERR,*) "(MUESLI msChol:) internal error: in processing physical units:"
               write(STDERR,*) "                 Please report this bug to: Edouard.Canot@univ-rennes.fr"
               mf_message_displayed = .true.
               call muesli_trace( pause="yes" )
               stop
            end if
         end do
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msChol_mfArray
!_______________________________________________________________________
!
   subroutine msChol_mfMatFactor( A, factor )

      type(mfArray) :: A
      type(mfMatFactor) :: factor
      !------ API end ------
#ifdef _DEVLP

      ! this routine is only for internal use: an ordinary user
      ! should always call 'msChol' which makes many tests on arguments.

      ! A : sparse matrix
      !
      ! 'msChol' returns (via an mfMatFactor) the pointer to a complete
      ! C structure, containing all necessary data (i.e. L, P)
      !
      ! A must be symmetric and positive definite

      integer :: n, status
      type(mfArray) :: A_copy

      ! CHOLMOD declaration
      integer(kind=MF_ADDRESS) :: c_addr, LL_addr
      integer :: nnz2

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      n = A%shape(1)

      if( A%data_type == MF_DT_SP_DBLE ) then

         call msAssign( A_copy, mfTriu(A) )
         nnz2 = mfNnz(A_copy)
         call cholmod_llt_prep2( n, nnz2, A_copy%j, A_copy%i, A_copy%a, &
                                 c_addr, LL_addr, status )

         ! check whether factorization is ok
         if( status /= 0 ) then
            call PrintMessage( "msChol", "E",                           &
                               "in Cholesky-factorization of A by CHOLMOD:",&
                               "A is not positive definite!",           &
                               "(as a side effect, output will be left empty)" )
            go to 99
         end if

         ! storing information

         factor%package = 2 ! 2 for CHOLMOD
         factor%data_type = MF_DT_SP_DBLE
         factor%order = n

         ! the CHOLMOD struct 'LL' will be freed only
         ! in 'msFreeMatFactor' (under the user responsibility)
         allocate( factor%ptr_1 )

         allocate( factor%ptr_2 )

         factor%ptr_1 = c_addr
         factor%ptr_2 = LL_addr

      else if( A%data_type == MF_DT_SP_CMPLX ) then

         call msAssign( A_copy, mfTriu(A) )
         nnz2 = mfNnz(A_copy)
         call cholmod_cmplx_llt_prep2( n, nnz2, A_copy%j, A_copy%i,     &
                                       A_copy%z,                        &
                                       c_addr, LL_addr, status )

         ! check whether factorization is ok
         if( status /= 0 ) then
            call PrintMessage( "msChol", "E",                           &
                               "in Cholesky-factorization of A by CHOLMOD:",&
                               "A is not positive definite!",           &
                               "(as a side effect, output will be left empty)" )
            go to 99
         end if

         ! storing information

         factor%package = 2 ! 2 for CHOLMOD
         factor%data_type = MF_DT_SP_CMPLX
         factor%order = n

         ! the CHOLMOD struct 'LL' will be freed only
         ! in 'msFreeMatFactor' (under the user responsibility)
         allocate( factor%ptr_1 )

         allocate( factor%ptr_2 )

         factor%ptr_1 = c_addr
         factor%ptr_2 = LL_addr

      else

         call PrintMessage( "msChol", "E",                              &
                            "bad data type for 'A'!" )
         go to 99

      end if

      if( A%prop%posd == UNKNOWN ) then
         ! ok, side effect, but it should be more efficient
         A%prop%posd = TRUE
      end if

      call msSilentRelease( A_copy )

      if( mf_phys_units ) then
         factor%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msChol_mfMatFactor
