! f90 include file

!_______________________________________________________________________
!
   subroutine msBalance( out, A )

      type(mfArray), target :: A
      type(mf_Out)          :: out
      !------ API end ------
#ifdef _DEVLP

      ! returns the balanced matrix B
      !
      ! also returns the transformation matrix T such that : T*B = A*T

      type(mfArray), pointer :: T, B
      integer :: i, n, lda, info, ilo, ihi
      type(mfArray), pointer :: A_copy
      double precision, allocatable :: scale(:)
      character :: job
      character(len=3) :: info_char
      logical :: A_copy_is_allocated

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! 2 out-args must be specified
      if( out%n /= 2 ) then
         call PrintMessage( "msBalance", "E",                           &
                            "two output args required!",               &
                            "syntax is : call msBalance ( mfOut(T,B), A )" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msBalance", "E",                           &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      T => out%ptr1
      B => out%ptr2
      call msSilentRelease( T, B )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "msBalance", "E",                           &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "msBalance", "E",                           &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "msBalance", "E",                           &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "msBalance", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( a%shape(1) <= 1 .or. a%shape(2) <= 1 ) then
         call PrintMessage( "msBalance", "E",                           &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      ! square matrix ?
      if( A%shape(1) /= A%shape(2) ) then
         call PrintMessage( "msBalance", "E",                           &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      T%shape = a%shape
      B%shape = a%shape

      T%data_type = MF_DT_DBLE ! T is always real
      allocate( T%double( T%shape(1), T%shape(2) ) )

      ! making copies because LAPACK overwrites A
      if( A%status_temporary .and. (A%level_protected==1) ) then
         A_copy => A
         A_copy_is_allocated = .false.
      else
         allocate( A_copy ) ! no_mem_trace !
         A_copy = A
         A_copy_is_allocated = .true.
      end if

      if( a%data_type == MF_DT_DBLE ) then

         B%data_type = MF_DT_DBLE
         allocate( B%double( B%shape(1), B%shape(2) ) )

         ! General system -- real square matrix : DGEBAL
         ! only scaling, no permutation
         job = "S"
         n = A_copy%shape(1)
         lda = A_copy%shape(1)
         allocate( scale(n) )

         call dgebal( job, n, A_copy%double(1,1), lda, ilo, ihi, scale(1), info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "msBalance", "W",                        &
                               "in balancing matrix A by LAPACK:",      &
                               "info = " // info_char,                  &
                               "(as a side effect, T and B will be empty)" )
            call msSilentRelease( T, B )
         else
            B%double(:,:) = A_copy%double
            T%double(:,:) = 0.0d0
            do i = 1, n
               T%double(i,i) = scale(i)
            end do
         end if

      else if( a%data_type == MF_DT_CMPLX ) then

         B%data_type = MF_DT_CMPLX
         allocate( B%cmplx( B%shape(1), B%shape(2) ) )

         ! General system -- complex square matrix : ZGEBAL
         ! only scaling, no permutation
         job = "S"
         n = A_copy%shape(1)
         lda = A_copy%shape(1)
         allocate( scale(n) )

         call zgebal( job, n, A_copy%cmplx(1,1), lda, ilo, ihi, scale(1), info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "msBalance", "W",                        &
                               "in balancing matrix A by LAPACK:",      &
                               "info = " // info_char,                  &
                               "(as a side effect, T and B will be empty)" )
            call msSilentRelease( T, B )
         else
            B%cmplx(:,:) = A_copy%cmplx
            T%double(:,:) = 0.0d0
            do i = 1, n
               T%double(i,i) = scale(i)
            end do
         end if

      end if

      if( A_copy_is_allocated ) then
         call msSilentRelease( A_copy )
         deallocate( A_copy ) ! no_mem_trace !
      end if

      ! matrices properties : T is diagonal
      T%prop%tril = TRUE
      T%prop%triu = TRUE

      if( mf_phys_units ) then
         B%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msBalance
!_______________________________________________________________________
!
   function mfBalance( A ) result( out )

      type(mfArray), target :: A
      type(mfArray)         :: out
      !------ API end ------
#ifdef _DEVLP

      ! returns the balanced matrix

      integer :: n, lda, info, ilo, ihi
      type(mfArray), pointer :: A_copy
      double precision, allocatable :: scale(:)
      character :: job
      character(len=3) :: info_char
      logical :: A_copy_is_allocated

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         call PrintMessage( "mfBalance", "E",                           &
                            "mfArray 'A' is not allocated!" )
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfBalance", "E",                           &
                            "sparse matrices not yet handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfBalance", "E",                           &
                            "cannot be applied to a boolean!" )
         go to 99
      end if

      if( A%data_type == MF_DT_PERM_VEC ) then
         call PrintMessage( "mfBalance", "E",                           &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      if( a%shape(1) <= 1 .or. a%shape(2) <= 1 ) then
         call PrintMessage( "mfBalance", "E",                           &
                            "mfArray 'A' should be a matrix!" )
         go to 99
      end if

      ! square matrix ?
      if( A%shape(1) /= A%shape(2) ) then
         call PrintMessage( "mfBalance", "E",                           &
                            "'A' must be a square matrix!" )
         go to 99
      end if

      out%shape = a%shape

      ! making copies because LAPACK overwrites A
      if( A%status_temporary .and. (A%level_protected==1) ) then
         A_copy => A
         A_copy_is_allocated = .false.
      else
         allocate( A_copy ) ! no_mem_trace !
         A_copy = A
         A_copy_is_allocated = .true.
      end if

      if( a%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_DBLE
         allocate( out%double( out%shape(1), out%shape(2) ) )

         ! General system -- real square matrix : DGEBAL
         ! only scaling, no permutation
         job = "S"
         n = A_copy%shape(1)
         lda = A_copy%shape(1)
         allocate( scale(n) )

         call dgebal( job, n, A_copy%double(1,1), lda, ilo, ihi, scale(1), info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "mfBalance", "W",                        &
                               "in balancing matrix A by LAPACK:",      &
                               "info = " // info_char,                  &
                               "(as a side effect, result will be empty)" )
         else
            out%double(:,:) = A_copy%double
         end if

      else if( a%data_type == MF_DT_CMPLX ) then

         out%data_type = MF_DT_CMPLX
         allocate( out%cmplx( out%shape(1), out%shape(2) ) )

         ! General system -- complex square matrix : ZGEBAL
         ! only scaling, no permutation
         job = "S"
         n = A_copy%shape(1)
         lda = A_copy%shape(1)
         allocate( scale(n) )

         call zgebal( job, n, A_copy%cmplx(1,1), lda, ilo, ihi, scale(1), info )
         if( info /= 0 ) then
            ! solution is not ok -- light warning
            write(info_char,"(I0)") info
            call PrintMessage( "mfBalance", "W",                        &
                               "in balancing matrix A by LAPACK:",      &
                               "info = " // info_char,                  &
                               "(as a side effect, result will be empty)" )
         else
            out%cmplx(:,:) = A_copy%cmplx
         end if

      end if

      if( A_copy_is_allocated ) then
         call msSilentRelease( A_copy )
         deallocate( A_copy ) ! no_mem_trace !
      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfBalance
