! f90 include file

! Internally, routine 'DDASSL' of the SLATEC library is used.
! UMFPACK is also used when Jacobian is sparse.

!_______________________________________________________________________
!
   subroutine msDaeSolve_JacUserSP( out, resid, t_span, y_0, yp_0,      &
                                    options, jac, sparse )

      ! subroutine resid( t, y, yprime, delta, flag )
      ! subroutine jac( t, y, yprime, cj, nrow, pd, ipd, jpd, nnz )
      type(mfArray) :: t_span, y_0, yp_0
      type(mf_DE_Options), target, optional :: options
      logical, intent(in) :: sparse

      type(mf_Out) :: out

      interface
         subroutine resid( t, y, yprime, delta, flag )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
            real(kind=MF_DOUBLE), intent(out)    :: delta(*)
            integer,              intent(in out) :: flag
         end subroutine resid
      end interface

      interface
         subroutine jac( t, y, yprime, cj, nrow, job, pd, ipd, jpd, nnz )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*), cj
            integer,              intent(in)     :: nrow, job
            real(kind=MF_DOUBLE), intent(out)    :: pd(*)
            integer,              intent(out)    :: ipd(*), jpd(*)
            integer,              intent(in out) :: nnz
            ! the CSC representation of the matrix is as follows:
            !   pd(1:nnz)    : values of the matrix entries
            !  ipd(1:nnz)    : contains the row indexes
            !  jpd(1:ncol+1) : the pointer to the beginning of the
            !                  columns, in arrays pd,ipd.
            !                (here, the matrix is square, the ncol=nrow)
            !
            ! when job=0, only the value of nnz must be returned
            !
            ! {pd,ipd} must contain all diagonal terms, even if they
            ! are null. (a test is made in SLATEC/DDASSL.F)
            !
            ! moreover, row indexes must be sorted in ascending order
            ! (constraint from UMFPACK)
         end subroutine jac
      end interface
      !------ API end ------
#ifdef _DEVLP

      ! The logical 'sparse' must be in the argument list in order to
      ! differentiate this routine from 'msDaeSolve_JacUser', because
      ! 'msDaeSolve' is generic.

      ! method  : Backward Differentiation Formula  [variable order]
      !           (with Jacobian provided by the user)
      !           implemented in DDASSL (SLATEC)
      !
      ! here: the Jacobian matrix is sparse

      ! if Consistent Initial Conditions must be found, it uses the
      ! (exact) method implemented in DECIC of MATLAB:
      ! call of 'dae_cic_sparse'

      type(mfArray), pointer :: tol, y_ind_out, non_neg, y0_ind, yp0_ind
      type(mfArray), target :: empty
      integer :: dummy_check_jac = 0, dummy_print_check_jac = 0
      logical :: IC_known

      real(kind=MF_DOUBLE) :: t, tout

      ! The following ones are declared in the mod_ddebd2 module:
      !   yy(:), yyp(:), rwork(:), iwork(:)

      integer :: neq, info(15), lrw, liw, idid, n_t_out, i
      real(kind=MF_DOUBLE), allocatable :: rtol(:), atol(:)
      integer, allocatable :: nonneg(:), yindout(:)
      type(mfArray), pointer :: y, status, tolout, yp, solve_log,       &
                                init_log, t_log, order_log
      logical :: interval_mode, whole_y_out, continuation, first_call,  &
                 de_cic_must_be_called
      integer :: maxord, flag, stat, n_y_out
      integer(kind=MF_ADDRESS), allocatable :: set_args_addr(:)
      integer, allocatable :: perm_args(:)
      character(len=16), allocatable :: args_name(:)
      integer :: args_status, i1, i2, alloc_status
      integer :: idid_save

      ! other global variables are stored in the MOD_DDEBD2 module
      type(mfArray), pointer :: monitor_y_ind, monitor_yp_ind
      integer :: n_monitor_y = -1, n_monitor_yp = -1
      logical :: monitor_y, monitor_yp, reuse_spjac_struct
      integer :: recl
      character(len=20) :: n_monitor_y_str, n_monitor_yp_str
      character(len=80) :: format

      ! communication with some SLATEC routines
      real(kind=MF_DOUBLE) :: dt_min, dt_max
      integer :: nb_step, nb_resid
      common /slatec_daesolve_1/ dt_min, dt_max, nb_step, nb_resid

      integer :: nb_jac, nb_solv
      common /slatec_daesolve_2/ nb_jac, nb_solv

      real(kind=MF_DOUBLE) :: cic_cpu_time_0
      integer :: cic_nb_resid_0, cic_nb_jac_0, cic_nb_solv_0
      common /dae_cic_1/ cic_cpu_time_0, cic_nb_resid_0,                &
                         cic_nb_jac_0, cic_nb_solv_0

      integer ddaini_info
      common /slatec_ddaini_status/ ddaini_info

      ! internal print progression
      logical :: print_progress, disp_times
      character(len=80) :: string
      real(kind=MF_DOUBLE) :: t_0, t_end, pc_fact, left_time
      integer :: clock_rate, clock_init, clock,                         &
                 hrs_1, min_1, sec_1
      common /slatec_daesolve_progress/ print_progress, disp_times,     &
             t_0, t_end, pc_fact, clock_rate, clock_init

      real(kind=MF_DOUBLE) :: ddajac_rcond_min
      logical :: save_sing_jac, ddajac_investig, ddajac_rat_basis
      common /slatec_ddajac_status/ ddajac_rcond_min, save_sing_jac,    &
                                    ddajac_investig, ddajac_rat_basis

      character(len=*), parameter :: ROUTINE_NAME = "msDaeSolve"

      integer :: istat, i_max, i_named_group, i_group_eqn
      character(len=10) :: i_max_str, i_group_eqn_str
      real(kind=MF_DOUBLE) :: diff, diff_max


   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      idid_save = 0

      call msInitArgs( t_span, y_0, yp_0 )

      if( present(options) ) then
         continuation       = options%continuation
         tol               => options%tol
         y_ind_out         => options%y_ind_out
         non_neg           => options%non_neg
         y0_ind            => options%y0_ind
         yp0_ind           => options%yp0_ind
         IC_known           = options%IC_known
         print_progress     = options%print_progress
         disp_times         = options%disp_times
         ddajac_investig    = options%jac_investig
         spjac_const_struct = options%spjac_const_struct
         reuse_spjac_struct = options%reuse_spjac_struct
         ddajac_rat_basis   = options%rational_null_basis
         ddajac_rcond_min   = min(options%jac_rcond_min,1.0d0)
         save_sing_jac      = options%save_sing_jac
         monitor_y_ind     => options%monitor_y_ind
         monitor_yp_ind    => options%monitor_yp_ind
         monitor_pause      = options%monitor_pause
         if( allocated(options%named_eqn) ) then
            call check_named_groups( options%named_eqn, istat )
            if( istat == 0 ) then
               NAMED_EQN_PRESENCE = .true.
               NAMED_EQN_PTR => options%named_eqn
            else
               NAMED_EQN_PRESENCE = .false.
               call PrintMessage( ROUTINE_NAME, "W",                    &
                                  "named groups of equations are not valid!", &
                                  " (overlap of indices)",              &
                                  "-> this feature is deactivated." )
            end if
         end if
         if( allocated(options%named_var) ) then
            call check_named_groups( options%named_var, istat )
            if( istat == 0 ) then
               NAMED_VAR_PRESENCE = .true.
               NAMED_VAR_PTR => options%named_var
            else
               NAMED_VAR_PRESENCE = .false.
               call PrintMessage( ROUTINE_NAME, "W",                    &
                                  "named groups of variables are not valid!", &
                                  " (overlap of indices)",              &
                                  "-> this feature is deactivated." )
            end if
         end if
         jac_symm_pos_def   = options%jac_symm_pos_def ! in 'MOD_DDEBD2' module
      else
         ! the following value must match those declared by default
         ! in mod_funfun.F90
         continuation       = .false.
         tol               => empty
         y_ind_out         => empty
         non_neg           => empty
         y0_ind            => empty
         yp0_ind           => empty
         IC_known           = .true.
         print_progress     = .false.
         disp_times         = .false.
         ddajac_investig    = .false.
         spjac_const_struct = .true.
         reuse_spjac_struct = .false.
         ddajac_rat_basis   = .false.
         ddajac_rcond_min   = 0.0d0
         save_sing_jac      = .false.
         monitor_y_ind     => empty
         monitor_yp_ind    => empty
         monitor_pause      = .false.
         jac_symm_pos_def   = .false. ! in 'MOD_DDEBD2' module
      end if

      ! 2 to 8 out-args must be specified
      if( out%n < 2 .or. 8 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to eight output args required!",      &
                            "syntax is: call msDaeSolve( mfOut(<out_args>), resid, t_span, y_0, yp_0[, options, jac, sparse] )", &
                            "    with <out_args> equal to {y,status[,tolout,yp,solve_log,init_log,t_log,order_log]}", &
                            "",                                         &
                            "Moreover, the output arg 'status' has not been set, because it may be not present!" )
         go to 99
      end if

      y => out%ptr1
      call msSilentRelease( y )
      status => out%ptr2
      call msSilentRelease( status )
      if( out%arg_present(3) ) then
         tolout => out%ptr3
         call msSilentRelease( tolout )
      else
         tolout => null()
      end if
      if( out%arg_present(4) ) then
         yp => out%ptr4
         call msSilentRelease( yp )
      else
         yp => null()
      end if
      if( out%arg_present(5) ) then
         solve_log => out%ptr5
         call msSilentRelease( solve_log )
      else
         solve_log => null()
      end if
      if( out%arg_present(6) ) then
         init_log => out%ptr6
         call msSilentRelease( init_log )
      else
         init_log => null()
      end if
      if( out%arg_present(7) ) then
         t_log => out%ptr7
         call msSilentRelease( t_log )
         save_times = .true.
      else
         t_log => null()
         save_times = .false.
      end if
      if( out%arg_present(8) ) then
         order_log => out%ptr8
         call msSilentRelease( order_log )
         save_orders = .true.
      else
         order_log => null()
         save_orders = .false.
      end if

      first_call = .not. continuation

      ! verifying that all 11 arguments are really different
      allocate( set_args_addr(11), args_name(11), perm_args(11) )
      perm_args(:) = [ (i, i = 1, 11) ]
      ! 1: getting address of 8 output args
      !    and storing them in 'set_args_addr' from position 1
      call get_address_of_out_args( set_args_addr, 1, out, 8  )
      args_name(1) = "y"
      args_name(2) = "status"
      args_name(3) = "tolout"
      args_name(4) = "yp"
      args_name(5) = "solve_log"
      args_name(6) = "init_log"
      args_name(7) = "t_log"
      args_name(8) = "order_log"
      ! 2: getting address of 3 input args
      !    and storing them in 'set_args_addr' from position 9
      call get_address_of_in_args( set_args_addr, 9,                    &
                                   t_span, y_0, yp_0 )
      args_name(9) = "t_span"
      args_name(10) = "y_0"
      args_name(11) = "yp_0"
      ! verifying that all adresses are different
      call quick_sort( "asc", set_args_addr, perm_args )
      call no_duplicates( set_args_addr, args_status, i1, i2 )
      if( args_status /= 0 ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "Found that args '" // trim(args_name(perm_args(i1))) // "' and ", &
                            "'" // trim(args_name(perm_args(i2))) // "' are identical,", &
                            "or at least point to the same address!" )
         ! -33 is the code for: invalid argument
         status = -33
         go to 99
      end if

      if( first_call ) then
         if( y_0%data_type /= MF_DT_DBLE ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'y_0' must be real!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if

         if( y_0%shape(1) /= 1 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'y_0' must be a row vector!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if

         neq = y_0%shape(2)
      else
         ! no need to check y_0: on continuation, it is not referenced.
         neq = IWORK(18)
      end if

      ! Integrator options: all-zero is the default.
      info(:) = 0

      if( first_call ) then
         if( IC_known ) then
            if( yp_0%data_type /= MF_DT_DBLE ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'yp_0' must be real!" )
               ! -33 is the code for: invalid argument
               status = -33
               go to 99
            end if

            if( any( shape(yp_0) /= shape(y_0) ) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'yp_0' must have the same shape than 'y_0'!" )
               ! -33 is the code for: invalid argument
               status = -33
               go to 99
            end if
            ! info(11) = 0 ! y_0 and yp_0 are supposed to be consistent
            !               (a check will be performed later)
            de_cic_must_be_called = .false.
         else
            de_cic_must_be_called = .true.
         end if
      end if

      ! initialisation for DAE_CIC stats
      cic_nb_resid_0 = 0
      cic_nb_jac_0   = 0
      cic_nb_solv_0  = 0
      cic_cpu_time_0 = 0.0d0

      if( .not. mfIsEmpty( non_neg ) ) then

         if( .not. mfIsVector( non_neg) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "optional arg 'non_neg' must be a vector mfArray!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
         info(10) = size(non_neg)
         if( info(10) > neq ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "size of arg 'non_neg' cannot be greater than", &
                               "the number of equations!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
         allocate( nonneg(size(non_neg)) )
         nonneg(:) = non_neg
         if( any(nonneg(:)<1) .or. any(neq<nonneg(:)) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "all values in 'non_neg' must be ranged in", &
                               "[1,NEQ]!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if

      end if

      if( .not. mfIsEmpty(y_ind_out) ) then

         if( y_ind_out%shape(1) /= 1 .and. y_ind_out%shape(2) /= 1 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "optional arg 'y_ind_out' must be a vector mfArray!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
         whole_y_out = .false.
         n_y_out = size(y_ind_out)
         if( n_y_out == neq ) then
            whole_y_out = .true.
         else if( n_y_out > neq ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "size of arg 'y_ind_out' cannot be greater than", &
                               "the number of equations!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
         allocate( yindout(n_y_out) )
         yindout(:) = y_ind_out
         if( any(yindout(:)<1) .or. any(neq<yindout(:)) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "all values in 'y_ind_out' must be ranged in [1,NEQ]!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if

      else
         whole_y_out = .true.
         n_y_out = neq
      end if

      if( .not. mfIsEmpty(monitor_y_ind) ) then

         if( monitor_y_ind%shape(1) /= 1 .and. monitor_y_ind%shape(2) /= 1 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "optional arg 'monitor_y_ind' must be a vector mfArray!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
         n_monitor_y = size(monitor_y_ind)
         allocate( monitoryind(n_monitor_y) )
         monitoryind(:) = monitor_y_ind
         if( any(monitoryind(:)<1) .or. any(neq<monitoryind(:)) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "all values in 'monitor_y_ind' must be ranged in [1,NEQ]!", &
                               "(NEQ is the number of equations)" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
         monitor_y = .true.
         call find_unit( unit_monitor_y )
         ! define RECL (important for large number of columns).
         ! 30 is chosen because it seems that Fortran use at most 30 chars
         ! to write a double precision real number in the case where no format
         ! is specified.
#if defined _INTEL_IFC
! BUG: INTEL-ifort (release 10.1 and other versions...)
!      See also the first use of this in the 'msSaveAscii()' routine.
         recl = (30 + 1)* (n_monitor_y+1) ! (+1 for INTEL-ifort, bug)
#else
         recl = 30 * (n_monitor_y+1)
#endif
         ! when n_monitor_y is small, take care that we have to write
         ! some comments at the beginning of the file (see below)
         recl = max( recl, 80 )
         open( unit_monitor_y, recl=recl, file="daesolve_y.out" )
         write( unit_monitor_y, "(A)" )                                 &
            "% output of DaeSolve at internal time steps"
         write( unit_monitor_y, "(A)" )                                 &
            "%   first column contains the time t"
         write( unit_monitor_y, "(A)" )                                 &
            "%   other columns contain values of y(:) at the following indexes:"
         write( n_monitor_y_str, "(I0)" ) n_monitor_y
         format = "( '%', 4X, " // trim(n_monitor_y_str) // "(1X,I0) )"
         write( unit_monitor_y, format )                                &
            monitoryind(:)
         format_monitor_y = "(ES14.7, " // trim(n_monitor_y_str) // "(2X,ES14.7) )"

      else
         monitor_y = .false.
      end if

      if( .not. mfIsEmpty(monitor_yp_ind) ) then

         if( monitor_yp_ind%shape(1) /= 1 .and. monitor_yp_ind%shape(2) /= 1 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "optional arg 'monitor_yp_ind' must be a vector mfArray!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
         n_monitor_yp = size(monitor_yp_ind)
         allocate( monitorypind(n_monitor_yp) )
         monitorypind(:) = monitor_yp_ind
         if( any(monitorypind(:)<1) .or. any(neq<monitorypind(:)) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "all values in 'monitor_yp_ind' must be ranged in [1,NEQ]!", &
                               "(NEQ is the number of equations)" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
         monitor_yp = .true.
         call find_unit( unit_monitor_yp )
         ! define RECL (important for large number of columns).
         ! 30 is chosen because it seems that Fortran use at most 30 chars
         ! to write a double precision real number in the case where no format
         ! is specified.
#if defined _INTEL_IFC
! BUG: INTEL-ifort (release 10.1 and other versions...)
!      See also the first use of this in the 'msSaveAscii()' routine.
         recl = (30 + 1)* (n_monitor_yp+1) ! (+1 for INTEL-ifort, bug)
#else
         recl = 30 * (n_monitor_yp+1)
#endif
         ! when n_monitor_y is small, take care that we have to write
         ! some comments at the beginning of the file (see below)
         recl = max( recl, 80 )
         open( unit_monitor_yp, recl=recl, file="daesolve_yp.out" )
         write( unit_monitor_yp, "(A)" )                                &
            "% output of DaeSolve at internal time steps"
         write( unit_monitor_yp, "(A)" )                                &
            "%   first column contains the time t"
         write( unit_monitor_yp, "(A)" )                                &
            "%   other columns contain values of yp(:) at the following indexes:"
         write( n_monitor_yp_str, "(I0)" ) n_monitor_yp
         format = "( '%', 4X, " // trim(n_monitor_yp_str) // "(1X,I0) )"
         write( unit_monitor_yp, format )                               &
            monitorypind(:)
         format_monitor_yp = "(ES14.7, " // trim(n_monitor_yp_str) // "(2X,ES14.7) )"

      else
         monitor_yp = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         call process_tol_for_daesolve( tol, rtol, atol, info, neq,     &
                                        ROUTINE_NAME, stat )
         if( stat /= 0 ) then
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
      else
         allocate( rtol(1), atol(1) )
         rtol(1) = DAE_rtol_def
         atol(1) = DAE_atol_def
      end if

      if( t_span%data_type /= MF_DT_DBLE ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'t_span' must be real!" )
         ! -33 is the code for: invalid argument
         status = -33
         go to 99
      end if

      if( t_span%shape(2) /= 1 ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'t_span' must be a column vector!" )
         ! -33 is the code for: invalid argument
         status = -33
         go to 99
      end if

      n_t_out = size(t_span)
      t_end = mfGet( t_span, n_t_out )
      if( first_call ) then
         t_0 = mfGet( t_span, 1 )
         if( n_t_out == 2 ) then
            interval_mode = .false.
         else if( n_t_out > 2 ) then
            interval_mode = .true.
         else
            call PrintMessage( ROUTINE_NAME, "E",                          &
                              "'t_span' vector must have at least two components!" )
            ! -33 is the code for: invalid argument
            status = -33
            go to 99
         end if
      else ! continuation
         ! in this case t_0 is the final time of the last integration
         t_0 = rwork(10)
         if( n_t_out == 1 ) then
            interval_mode = .false.
         else if( n_t_out > 1 ) then
            interval_mode = .true.
         end if
      end if

      ! error handling in SLATEC: we accept that XERRMSG prints
      ! some information, but not that it stops.
      call xsetf(1)

      t = t_0

      if( first_call ) then
         ! first call for this problem: the integrator will be completely
         ! initialized
         info(1) = 0
         if( allocated(yy) ) then
            if( size(yy) /= neq ) then
               deallocate( yy )
               allocate( yy(neq) )
            end if
         else
            allocate( yy(neq) )
         end if
         yy(:) = y_0%double(1,:)
         if( allocated(yyp) ) then
            if( size(yyp) /= neq ) then
               deallocate( yyp )
               allocate( yyp(neq) )
            end if
         else
            allocate( yyp(neq) )
         end if
         if( mfIsEmpty( yp_0 ) ) then
            yyp(:) = 0.0d0 ! first guess for finding root
         else
            yyp(:) = yp_0%double(1,:)
         end if
      else
         info(1) = 1
         ! when integration continue, 't', 'y' and 'yp' should not be altered
         if( .not. allocated(yy) ) then
            allocate( yy(neq) )
         end if
         if( .not. allocated(yyp) ) then
            allocate( yyp(neq) )
         end if
      end if

      ! info(3) = 0 ! no need to get intermediate results
      ! info(4) = 0 ! integration can be carried out beyond the end
      info(5) = 1 ! jacobian computed by a user routine
      ! info(6) = 0 ! jacobian matrix is not banded
      info(12) = 1 ! jacobian matrix is sparse

      maxord = 5 ! default value

      liw = 20 + neq
      if( first_call ) then
         if( allocated(iwork) ) then
            if( size(iwork) /= liw ) then
               deallocate( iwork )
               allocate( iwork(liw) )
            end if
         else
            allocate( iwork(liw) )
         end if
#ifdef _DEVLP
         ! Avoid unitialized values in Valgrind...
         ! (because not all values are used)
         iwork(:) = 0
#endif
      end if

      lrw = 40 + (maxord+4)*neq
      if( first_call ) then
         if( allocated(rwork) ) then
            if( size(rwork) /= lrw ) then
               deallocate( rwork )
               allocate( rwork(lrw) )
            end if
         else
            allocate( rwork(lrw) )
         end if
#ifdef _DEVLP
         ! Avoid unitialized values in Valgrind...
         ! (because not all values are used)
         rwork(:) = 0.0d0
#endif
      end if

      if( .not. reuse_spjac_struct ) then
        jac_ipd_crc = 0
        first_call_to_sp_lu = .true.
      end if

      if( first_call ) then
         if( de_cic_must_be_called ) then
            call dae_cic_sparse( t_0, yy, y0_ind, yyp, yp0_ind,         &
                                 resid, jac, flag )
            if( flag /= 0 ) then
               select case( flag )
                  case( -1 )
                     write(STDERR,*) "-> pb in 'resid' call"
                  case( -2 )
                     ! a comment is already present in dae_cic_sparse
                     ! (bad number of prescribed components)
                  case( -3 )
                     write(STDERR,*) "-> DAE index perhaps greater than one"
                  case( -4 )
                     write(STDERR,*) "-> convergence not reached"
                  case( -5 )
                     write(STDERR,*) "-> illegal call"
                  case( -6, -8 )
                     write(STDERR,*) "-> pb in 'jac' call"
                  case( -7 )
                     write(STDERR,*) "-> pb in 'resid' call"
                  case default
                     write(STDERR,*) "(MUESLI msDaeSolve:) internal error:"
                     write(STDERR,*) "                     bad value for returned flag from dae_cic_sparse!"
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="yes" )
                     stop
               end select
               call PrintMessage( ROUTINE_NAME, "W",                    &
                                  "failure in computing consistent initial conditions!", &
                                  "(see diagnostic above)",             &
                                  no_pause=.true. )
               status = -102
               go to 99
            end if
         end if

         if( IC_known ) then
            ! check of validity of initial conditions: compute the
            ! residual for {t_0,y_0,yp_0} and compare it against the
            ! provided tolerance.
            ! we use the test:
            !     abs(delta(i)) <= rtol(i)*abs(y_0(i)) + atol(i)
            flag = 0 ! IRES in DDASSL
            call resid( t_0, yy, yyp, rwork, flag )
            if( flag /= 0 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "cannot check initial conditions!",   &
                                  "during the 'resid' call, flag /= 0", &
                                  no_pause=.true. )
               status = -100
               go to 99
            end if
            flag = 0
            if( info(2) == 1 ) then
               ! 'rtol' and 'atol' are both vectors
               diff_max = 0.0d0
               do i = 1, neq
                  diff = abs(rwork(i)) - (rtol(i)*abs(yy(i)) + atol(i))
                  if( diff > 0.0d0 ) then
                     if( diff > diff_max ) then
                        i_max = i
                        diff_max = diff
                     end if
                  end if
               end do
               if( diff_max > 0.0d0 ) flag = 1
            else
               ! 'rtol' and 'atol' are both scalars
               diff_max = 0.0d0
               do i = 1, neq
                  diff = abs(rwork(i)) - (rtol(1)*abs(yy(i)) + atol(1))
                  if( diff > 0.0d0 ) then
                     if( diff > diff_max ) then
                        i_max = i
                        diff_max = diff
                     end if
                  end if
               end do
               if( diff_max > 0.0d0 ) flag = 1
            end if
            if( flag /= 0 ) then
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i_max, i_named_group, &
                                                          i_group_eqn )
                  write(i_group_eqn_str,"(I0)") i_group_eqn
                  call PrintMessage( ROUTINE_NAME, "W",                 &
                        "initial conditions are not consistent!",       &
                        "Greatest discrepancy is at named equation ",   &
                        "  '" // trim(NAMED_EQN_PTR(i_named_group)%name) // "'", &
                        "  at equation number: " // trim(i_group_eqn_str), &
                        no_pause=.true. )
               else
                  write(i_max_str,"(I0)") i_max
                  call PrintMessage( ROUTINE_NAME, "W",                 &
                        "initial conditions are not consistent!",       &
                        "Greatest discrepancy is at equation number " // trim(i_max_str), &
                        no_pause=.true. )
               end if
               ! -101 is the code for: initial conditions not consistent
               status = -101
               if( out%arg_present(3) ) then
                  tolout = abs(rwork(:))
               end if
               go to 99
            end if
         end if

      end if

      y%data_type = MF_DT_DBLE
      if( out%arg_present(4) ) then
         yp%data_type = MF_DT_DBLE
      end if
      if( print_progress ) then
         ! initialize progression (comes from 'msPrepProgress')
         pc_fact = 100.0d0/( t_end - t_0 ) ! should be finite
         if( disp_times ) then
            write(STDOUT,"(A)")                                         &
            "................... [Progress status from DaeSolve] ...................."
            call system_clock( count_rate=clock_rate, count=clock_init )
            string = "  0.0 % (time left =   0h  0m  0s -- estim. remain. time =    h   m   s)"
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_on_term_no_adv( trim(string) // char(0) )
#else
            write(STDOUT,"(A)",advance="no") trim(string)
            call msFlush(STDOUT)
#endif
         else
            write(STDOUT,"(A)")                                         &
            "... [Progress status from DaeSolve] ..."
            string = "  0.0 %"
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_on_term_no_adv( trim(string) // char(0) )
#else
            write(STDOUT,"(A)",advance="no") trim(string)
            call msFlush(STDOUT)
#endif
         end if
      end if

      if( monitor_y ) then
         write( unit_monitor_y, format_monitor_y ) 0.0d0, yy(monitoryind)
         call flush( unit_monitor_y )
      end if
      if( monitor_yp ) then
         write( unit_monitor_yp, format_monitor_yp ) 0.0d0, yyp(monitoryind)
         call flush( unit_monitor_yp )
      end if
      if( monitor_pause ) then
         print *
         call msPause( "[MUESLI] DAE solver: monitoring pause" )
      end if

      if( interval_mode ) then

         ! copy initial values
         y%shape = [ n_t_out, n_y_out ]
         allocate( y%double(y%shape(1),y%shape(2)), stat=alloc_status )
         if( alloc_status /= 0 ) then
            write(STDERR,"( 1X,A)") "(MUESLI msDaeSolve:) Error: Cannot allocate the matrix 'y' to store"
            write(STDERR,"(22X,A)") "the solution at all intermediate outputs."
            write(STDERR,"(22X,A)") "For information, the shape is:"
            write(STDERR,"(22X,A,I0)") "  nb of intermediate times = ", n_t_out
            write(STDERR,"(22X,A,I0)") "  total mesh size = ", n_y_out
            write(STDERR,"(22X,A)") "A solution is to decrease either the number of interme-"
            write(STDERR,"(22X,A)") "diate times, i.e. the length of the 't_span' vector, or"
            write(STDERR,"(22X,A)") "the total mesh size."
            mf_message_displayed = .true.
            call muesli_trace( pause ="no" )
            status = -20
            go to 99
         end if
         ! if integration fails, data will be NaN
         y%double(:,:) = MF_NAN
         if( first_call ) then
            if( whole_y_out ) then
               y%double(1,:) = y_0%double(1,:)
            else
               y%double(1,:) = y_0%double(1,yindout)
            end if
         end if

         if( out%arg_present(4) ) then
            yp%shape = [ n_t_out, n_y_out ]
            allocate( yp%double(yp%shape(1),yp%shape(2)), stat=alloc_status )
            if( alloc_status /= 0 ) then
               write(STDERR,"( 1X,A)") "(MUESLI msDaeSolve:) Error: Cannot allocate the matrix 'yp' to store the time-"
               write(STDERR,"(22X,A)") "derivative of the solution at all intermediate outputs."
               write(STDERR,"(22X,A)") "For information, the shape is:"
               write(STDERR,"(22X,A,I0)") "  nb of intermediate times = ", n_t_out
               write(STDERR,"(22X,A,I0)") "  total mesh size = ", n_y_out
               write(STDERR,"(22X,A)") "A solution is to decrease either the number of interme-"
               write(STDERR,"(22X,A)") "diate times, i.e. the length of the 't_span' vector, or"
               write(STDERR,"(22X,A)") "the total mesh size."
               write(STDERR,"(22X,A)") "Another solution is to avoid the output of 'yp'."
               mf_message_displayed = .true.
               call muesli_trace( pause ="no" )
               status = -20
               go to 99
            end if
            ! if integration fails, data will be NaN
            yp%double(:,:) = MF_NAN
            if( first_call ) then
               if( IC_known ) then
                  if( whole_y_out ) then
                     yp%double(1,:) = yp_0%double(1,:)
                  else
                     yp%double(1,:) = yp_0%double(1,yindout)
                  end if
!### TODO 2: faut-il un test supplémentaire, comme dans msDaeSolve_JacUser ?
               else
                  ! I.C. computed
                  if( whole_y_out ) then
                     yp%double(1,:) = yyp
                  else
                     yp%double(1,:) = yyp(yindout)
                  end if
               end if
            end if
         end if

         if( first_call ) then
            i = 2
         else
            i = 1
         end if
         tout = mfGet( t_span, i )
         idid = -1 ! integration is not yet finished, of course
         do ! possible 'cycle'

            if( idid > 1 ) then
               tout = mfGet( t_span, i )
            else
               idid = 0
            end if

            if( tout /= t ) then ! avoid an error in ddassl
               call ddassl( resid, neq, t, yy(1), yyp(1), tout, info(1), rtol, atol, &
                            idid, rwork(1), lrw, iwork(1), liw,         &
                            dummy_dae_jac, jac, nonneg,                 &
                            dummy_check_jac, dummy_print_check_jac )

               if( idid < 0 ) then
                  ! abnormal return
                  if( idid == -33 ) then
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI msDaeSolve:) internal error:"
                     write(STDERR,*) "                     invalid input for a SLATEC routine"
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="yes" )
                     stop
                  else if( idid == -1 ) then
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "A large amount of work has been expended (about 500 steps)" )
                     ! let's continue, this was not an error
                     info(1) = 1
                     idid_save = -1
                     cycle
                  else if( idid == -8 ) then
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI msDaeSolve:) Error: the DAE solver encountered"
                     write(STDERR,*) "                     the following problem:"
                     write(STDERR,*) "                     'Jacobian is singular during the integration phase'"
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="no" )
                     status = idid
                     go to 99
                  else if( idid == -10 ) then
                     ! flag set to -1 in 'resid' (illegal condition)
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "Illegal condition in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'Illegal condition' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, i )
                     call msSet( MF_NAN, t_span, i+1 .to. MF_END )
                     ! no cycle, return to the calling routine...
                  else if( idid == -11 ) then
                     ! flag set to -2 in 'resid' (emergency exit!)
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "Emergency exit in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'Emergency exit' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, i )
                     call msSet( MF_NAN, t_span, i+1 .to. MF_END )
                     ! no cycle, return to the calling routine...
                  else if( idid == -12 ) then
                     ! flag set to +3 in 'resid' (end condition!)
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "End condition in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'End condition' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, i )
                     call msSet( MF_NAN, t_span, i+1 .to. MF_END )
                     ! no cycle, return to the calling routine...
                  else if( idid == -13 ) then
                     ! a message has already been displayed by DDASSL
                     go to 99
                  else if( idid == -14 ) then
                     ! a message has already been displayed by DDASTP
                     go to 99
                  else if( idid == -15 ) then
                     ! a message has already been displayed by DDASTP
                     go to 99
                  else if( idid == -102 ) then
                     ! find the actual reason of fail...
                     if( ddaini_info == -1 ) then
                        call PrintMessage( ROUTINE_NAME, "W",           &
                                           "the DAE solver wasn't able to start:", &
                                           "return flag in 'resid' was not zero.", &
                                           no_pause=.true. )
                        status = -100
                     else if( ddaini_info == -2 ) then
                        call PrintMessage( ROUTINE_NAME, "W",           &
                                           "the DAE solver wasn't able to compute the Jacobian matrix:", &
                                           "return flag in 'ddajac' was not zero.", &
                                           no_pause=.true. )
                        status = -102
                     else
                        call PrintMessage( ROUTINE_NAME, "W",           &
                                           "the DAE solver encountered the following problem:", &
                                           "failed to compute the initial vector yp_0", &
                                           no_pause=.true. )
                        status = idid
                     end if
                     go to 99
                  else
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI msDaeSolve:) error status: ", idid
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "the DAE solver encountered some problem!", &
                                        "[please examine the status value and the ref. manual]" )
                     status = idid
                     go to 99
                  end if
               end if
            else
               idid = 3
            end if

            if( whole_y_out ) then
               y%double(i,:) = yy(:)
            else
               y%double(i,:) = yy(yindout)
            end if

            if( out%arg_present(4) ) then
               if( whole_y_out ) then
                  yp%double(i,:) = yyp(:)
               else
                  yp%double(i,:) = yyp(yindout)
               end if
            end if

            if( -12 <= idid .and. idid <= -10 ) exit

            i = i + 1
            if( i > n_t_out ) exit

            info(1) = 1

         end do

      else ! .not. interval_mode

         y%shape = [ 1, n_y_out ]
         allocate( y%double(y%shape(1),y%shape(2)), stat=alloc_status )
         if( alloc_status /= 0 ) then
            write(STDERR,"( 1X,A)") "(MUESLI msDaeSolve:) Error: Cannot allocate the vector 'y' to store"
            write(STDERR,"(22X,A)") "the solution."
            write(STDERR,"(22X,A)") "For information, the length of 'y' is:"
            write(STDERR,"(22X,A,I0)") "  total mesh size = ", n_y_out
            write(STDERR,"(22X,A)") "A solution is to decrease the total mesh size."
            mf_message_displayed = .true.
            call muesli_trace( pause ="no" )
            status = -20
            go to 99
         end if
         ! if integration fails, data will be NaN
         y%double(:,:) = MF_NAN

         if( out%arg_present(4) ) then
            yp%shape = [ 1, n_y_out ]
            allocate( yp%double(yp%shape(1),y%shape(2)), stat=alloc_status )
            if( alloc_status /= 0 ) then
               write(STDERR,"( 1X,A)") "(MUESLI msDaeSolve:) Error: Cannot allocate the vector 'yp' to store"
               write(STDERR,"(22X,A)") "the time-derivative of the solution."
               write(STDERR,"(22X,A)") "For information, the length of 'y' is:"
               write(STDERR,"(22X,A,I0)") "  total mesh size = ", n_y_out
               write(STDERR,"(22X,A)") "A solution is to decrease the total mesh size."
               write(STDERR,"(22X,A)") "Another solution is to avoid the output of 'yp'."
               mf_message_displayed = .true.
               call muesli_trace( pause ="no" )
               status = -20
               go to 99
            end if
            ! if integration fails, data will be NaN
            yp%double(:,:) = MF_NAN
            if( IC_known ) then
               if( whole_y_out ) then
                  yp%double(1,:) = yp_0%double(1,:)
               else
                  yp%double(1,:) = yp_0%double(1,yindout)
               end if
            end if
         end if

         tout = t_end
         do ! possible 'cycle'

            if( tout /= t ) then ! avoid an error in ddassl
               call ddassl( resid, neq, t, yy(1), yyp(1), tout, info(1), rtol, atol, &
                            idid, rwork(1), lrw, iwork(1), liw,         &
                            dummy_dae_jac, jac, nonneg,                 &
                            dummy_check_jac, dummy_print_check_jac )

               if( idid < 0 ) then
                  ! abnormal return
                  if( idid == -33 ) then
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI msDaeSolve:) internal error:"
                     write(STDERR,*) "                     invalid input for a SLATEC routine"
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="yes" )
                     stop
                  else if( idid == -1 ) then
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "A large amount of work has been expended (about 500 steps)" )
                     ! let's continue, this was not an error
                     info(1) = 1
                     idid_save = -1
                     cycle
                  else if( idid == -8 ) then
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI msDaeSolve:) Error: the DAE solver encountered"
                     write(STDERR,*) "                     the following problem:"
                     write(STDERR,*) "'Jacobian is singular during the integration phase'"
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="no" )
                     status = idid
                     go to 99
                  else if( idid == -10 ) then
                     ! flag set to -1 in 'resid' (illegal condition)
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "Illegal condition in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'Illegal condition' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, 2 )
                     ! no cycle, return to the calling routine...
                  else if( idid == -11 ) then
                     ! flag set to -2 in 'resid' (emergency exit!)
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "Emergency exit in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'Emergency exit' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, 2 )
                     ! no cycle, return to the calling routine...
                  else if( idid == -12 ) then
                     ! flag set to +3 in 'resid' (end condition!)
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "End condition in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'End condition' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, 2 )
                     ! no cycle, return to the calling routine...
                  else if( idid == -13 ) then
                     ! a message has already been displayed by DDASSL
                     go to 99
                  else if( idid == -14 ) then
                     ! a message has already been displayed by DDASTP
                     go to 99
                  else if( idid == -15 ) then
                     ! a message has already been displayed by DDASTP
                     go to 99
                  else if( idid == -102 ) then
                     ! find the actual reason of fail...
                     if( ddaini_info == -1 ) then
                        call PrintMessage( ROUTINE_NAME, "W",           &
                                           "the DAE solver wasn't able to start:", &
                                           "return flag in 'resid' was not zero.", &
                                           no_pause=.true. )
                        status = -100
                     else if( ddaini_info == -2 ) then
                        call PrintMessage( ROUTINE_NAME, "W",           &
                                           "the DAE solver wasn't able to compute the Jacobian matrix:", &
                                           "return flag in 'ddajac' was not zero.", &
                                           no_pause=.true. )
                        status = -102
                     else
                        call PrintMessage( ROUTINE_NAME, "W",           &
                                           "the DAE solver encountered the following problem:", &
                                           "failed to compute the initial vector yp_0", &
                                           no_pause=.true. )
                        status = idid
                     end if
                     go to 99
                  else
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI msDaeSolve:) error status: ", idid
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "the DAE solver encountered some problem!", &
                                        "[please examine the status value and the ref. manual]" )
                     status = idid
                     go to 99
                  end if
               end if
            else
               idid = 3
            end if

            exit

         end do

         if( whole_y_out ) then
            y%double(1,:) = yy(:)
         else
            y%double(1,:) = yy(yindout)
         end if
         if( out%arg_present(4) ) then
            if( whole_y_out ) then
               yp%double(1,:) = yyp(:)
            else
               yp%double(1,:) = yyp(yindout)
            end if
         end if

      end if

      ! when IC are not known, the array 'yp_init_ddassl' is allocated
      ! by the solver DDASSL, so it must be deallocated for a future use
      ! in the same main program...
      if( allocated(yp_init_ddassl) ) then
         deallocate(yp_init_ddassl)
      end if

      if( monitor_y ) then
         close( unit_monitor_y )
         unit_monitor_y = -1
         deallocate( monitoryind )
         n_monitor_y = -1
         format_monitor_y = ""
      end if
      if( monitor_yp ) then
         close( unit_monitor_yp )
         unit_monitor_yp = -1
         deallocate( monitorypind )
         n_monitor_yp = -1
         format_monitor_yp = ""
      end if

      if( print_progress ) then
         ! summarize progression (comes from 'msPostProgress')
         if( disp_times ) then
            call system_clock( count=clock )
            left_time = dble(clock-clock_init)/clock_rate
            call sec_2_hms( left_time, hrs_1, min_1, sec_1 )
            write(string,"(A,I3,A,2(I2,A),A40)") "done. (time left = ", &
                  hrs_1, "h ", min_1, "m ", sec_1, "s)", " "
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_back_on_term( string(1:72) // char(0) )
#else
            call go_home_on_term()
            write(STDOUT,"(A)") string(1:72)
#endif
            write(STDOUT,"(A)")                                         &
            "........................................................................"
         else
            string = "done.  "
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_back_on_term( string(1:7) // char(0) )
#else
            call go_home_on_term()
            write(STDOUT,"(A)") string(1:7)
#endif
            write(STDOUT,"(A)") "......................................."
         end if
      end if

      if( out%arg_present(3) ) then
         if( info(2) == 0 ) then ! 'rtol' and 'atol' are both scalars
            tolout = [ rtol(1), atol(1) ]
         else ! 'rtol' and 'atol' are both vectors
            tolout = (.t.mf(rtol(1:neq))) .hc. (.t.mf(atol(1:neq)))
         end if
      end if

      if( out%arg_present(5) ) then
         solve_log = [ dble(nb_step), dt_min, dt_max, dble(nb_resid),   &
                       dble(nb_jac), dble(nb_solv) ]
      end if

      if( out%arg_present(6) ) then
         ! sparse Jacobian matrix : DAE_CIC has been called
         init_log = [ dble(cic_nb_resid_0), dble(cic_nb_jac_0),         &
                      dble(cic_nb_solv_0), cic_cpu_time_0 ]
      end if

      ! even in case of integration error, t_log is returned if required
 99   continue

      if( out%arg_present(7) ) then
         ! In case where integration duration is zero, the array
         ! 'times_solve' has not been allocated...
         if( allocated(times_solve) ) then
            t_log = times_solve(1:n_times)
            deallocate( times_solve )
            ! FIX of T_LOG: after stopping condition the T_LOG vector must be
            ! fixed...
            ! (added a test on 't_log' because in case of fail during the
            !  integration process, this vector may contain only one value!)
            if( size(t_log) >= 2 ) then
               if( all( mfGet(t_log,MF_END) <= mfGet(t_log,MF_END-1) ) ) then
                  call msSet( MF_EMPTY, t_log, MF_END-1 )
               end if
            end if
         end if
      end if

      if( out%arg_present(8) ) then
         ! In case where integration duration is zero, the array
         ! 'orders_solve' has not been allocated...
         if( allocated(orders_solve) ) then
            order_log = orders_solve(1:n_orders)
            deallocate( orders_solve )
         end if
      end if

      if( mfIsEmpty(status) ) then
         status = idid
      end if

      ! special cases
      if( idid == 3 ) then
         status = 0
      else if( idid == -12 ) then
         status = 12
      end if
      if( idid_save == -1 ) then
         status = 4
      end if

      call msFreeArgs( t_span, y_0, yp_0 )
      call msAutoRelease( t_span, y_0, yp_0 )

      call mf_restore_fpe( )

#endif
   end subroutine msDaeSolve_JacUserSP
