! f90 include file

! Internally, routine 'DDASSL' of the SLATEC library is used.

!_______________________________________________________________________
!
   function mfDaeSolve_JacUser( resid, t_span, y_0, yp_0,               &
                                options, jac )                          &
   result( out )

      ! subroutine resid( t, y, yprime, delta, flag )
      ! subroutine jac( t, y, yprime, jacobian, cj, nrow )
      type(mfArray) :: t_span, y_0, yp_0
      type(mf_DE_Options), target, optional :: options

      type(mfArray) :: out

      interface
         subroutine resid( t, y, yprime, delta, flag )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
            real(kind=MF_DOUBLE), intent(out)    :: delta(*)
            integer,              intent(in out) :: flag
         end subroutine resid
      end interface

      interface
         subroutine jac( t, y, yprime, jacobian, cj, nrow )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)  :: t, y(*), yprime(*), cj
            integer,              intent(in)  :: nrow
            real(kind=MF_DOUBLE), intent(out) :: jacobian(nrow,*)
         end subroutine jac
      end interface
      !------ API end ------
#ifdef _DEVLP

      ! method  : Backward Differentiation Formula  [variable order]
      !           (with Jacobian provided by the user)
      !           implemented in DDASSL (SLATEC)
      !
      ! here: the Jacobian matrix is dense (full or banded)

      ! if Consistent Initial Conditions must be found, it uses
      ! the (exact) method implemented in DECIC of MATLAB:
      !     call of 'dae_cic_dense' or 'dae_cic_band'

      type(mfArray), pointer :: tol, y_ind_out, non_neg, band, y0_ind, yp0_ind
      type(mfArray), target  :: empty
      integer :: check_jac, print_check_jac
      logical :: IC_known

      real(kind=MF_DOUBLE) :: t, tout

      ! The following ones are declared in the mod_ddebd2 module:
      !   yy(:), yyp(:), rwork(:), iwork(:)

      integer :: neq, info(15), lrw, liw, idid, n_t_out, i
      real(kind=MF_DOUBLE), allocatable :: rtol(:), atol(:)
      integer, allocatable :: nonneg(:), yindout(:)
      logical :: interval_mode, whole_y_out, continuation, first_call,  &
                 de_cic_must_be_called
      integer :: maxord, flag, stat, n_y_out, alloc_status

      ! other global variables are stored in the MOD_DDEBD2 module
      type(mfArray), pointer :: monitor_y_ind, monitor_yp_ind
      integer :: n_monitor_y = -1, n_monitor_yp = -1
      logical :: monitor_y, monitor_yp
      integer :: recl
      character(len=20) :: n_monitor_y_str, n_monitor_yp_str
      character(len=80) :: format

      ! internal print progression
      logical :: print_progress, disp_times
      character(len=80) :: string
      real(kind=MF_DOUBLE) :: t_0, t_end, pc_fact, left_time
      integer :: clock_rate, clock_init, clock,                         &
                 hrs_1, min_1, sec_1
      common /slatec_daesolve_progress/ print_progress, disp_times,     &
             t_0, t_end, pc_fact, clock_rate, clock_init

      real(kind=MF_DOUBLE) :: ddajac_rcond_min
      logical :: save_sing_jac, ddajac_investig, ddajac_rat_basis
      common /slatec_ddajac_status/ ddajac_rcond_min, save_sing_jac,    &
                                    ddajac_investig, ddajac_rat_basis

      character(len=*), parameter :: ROUTINE_NAME = "mfDaeSolve"

      integer :: istat, i_max, i_named_group, i_group_eqn
      character(len=10) :: i_max_str, i_group_eqn_str
      real(kind=MF_DOUBLE) :: diff, diff_max

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( t_span, y_0, yp_0 )

      if( present(options) ) then
         continuation     = options%continuation
         tol             => options%tol
         y_ind_out       => options%y_ind_out
         non_neg         => options%non_neg
         band            => options%band
         y0_ind          => options%y0_ind
         yp0_ind         => options%yp0_ind
         IC_known         = options%IC_known
         print_progress   = options%print_progress
         disp_times       = options%disp_times
         ddajac_investig  = options%jac_investig
         ddajac_rat_basis = options%rational_null_basis
         ddajac_rcond_min = min(options%jac_rcond_min,1.0d0)
         save_sing_jac    = options%save_sing_jac
         monitor_y_ind   => options%monitor_y_ind
         monitor_yp_ind  => options%monitor_yp_ind
         monitor_pause    = options%monitor_pause
         check_jac        = options%check_jac
         print_check_jac  = options%print_check_jac
         if( allocated(options%named_eqn) ) then
            call check_named_groups( options%named_eqn, istat )
            if( istat == 0 ) then
               NAMED_EQN_PRESENCE = .true.
               NAMED_EQN_PTR => options%named_eqn
            else
               NAMED_EQN_PRESENCE = .false.
               call PrintMessage( ROUTINE_NAME, "W",                    &
                                  "named groups of equations are not valid!", &
                                  " (overlap of indices)",              &
                                  "-> this feature is deactivated." )
            end if
         end if
         if( allocated(options%named_var) ) then
            call check_named_groups( options%named_var, istat )
            if( istat == 0 ) then
               NAMED_VAR_PRESENCE = .true.
               NAMED_VAR_PTR => options%named_var
            else
               NAMED_VAR_PRESENCE = .false.
               call PrintMessage( ROUTINE_NAME, "W",                    &
                                  "named groups of variables are not valid!", &
                                  " (overlap of indices)",              &
                                  "-> this feature is deactivated." )
            end if
         end if
         jac_symm_pos_def   = options%jac_symm_pos_def ! in 'MOD_DDEBD2' module
      else
         ! the following value must match those declared by default
         ! in mod_funfun.F90
         continuation     = .false.
         tol             => empty
         y_ind_out       => empty
         non_neg         => empty
         band            => empty
         y0_ind          => empty
         yp0_ind         => empty
         IC_known         = .true.
         print_progress   = .false.
         disp_times       = .false.
         ddajac_investig  = .false.
         ddajac_rat_basis = .false.
         ddajac_rcond_min = 0.0d0
         save_sing_jac    = .false.
         monitor_y_ind   => empty
         monitor_yp_ind  => empty
         monitor_pause    = .false.
         check_jac        = 0
         print_check_jac  = 0
         jac_symm_pos_def   = .false. ! in 'MOD_DDEBD2' module
      end if

      if( .not. MF_NUMERICAL_CHECK ) then
         if( check_jac > 0 ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "jacobian check cannot be done in the release mode!", &
                               "(it has been disabled)" )
         end if

         if( print_check_jac > 0 ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "jacobian check cannot be done in the release mode!", &
                               "(printing of the result of check has been disabled)" )
         end if
      end if

      first_call = .not. continuation

      if( first_call ) then
         if( y_0%data_type /= MF_DT_DBLE ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'y_0' must be real!" )
            go to 99
         end if

         if( y_0%shape(1) /= 1 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'y_0' must be a row vector!" )
            go to 99
         end if

         neq = y_0%shape(2)
      else
         ! no need to check y_0: on continuation, it is not referenced.
         neq = IWORK(18)
      end if

      ! Integrator options: all-zero is the default.
      info(:) = 0

      if( first_call ) then
         if( IC_known ) then
            if( yp_0%data_type /= MF_DT_DBLE ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'yp_0' must be real!" )
               go to 99
            end if

            if( any( shape(yp_0) /= shape(y_0) ) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'yp_0' must have the same shape than 'y_0'!" )
               go to 99
            end if
            ! info(11) = 0 ! y_0 and yp_0 are supposed to be consistent
            !               (a check will be performed later)
            de_cic_must_be_called = .false.
         else
            de_cic_must_be_called = .true.
         end if
      end if

      if( .not. mfIsEmpty( non_neg ) ) then

         if( .not. mfIsVector( non_neg) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "optional arg 'non_neg' must be a vector mfArray!" )
            go to 99
         end if
         info(10) = size(non_neg)
         if( info(10) > neq ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "size of arg 'non_neg' cannot be greater than", &
                               "the number of equations!" )
            go to 99
         end if
         allocate( nonneg(size(non_neg)) )
         nonneg(:) = non_neg
         if( any(nonneg(:)<1) .or. any(neq<nonneg(:)) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "all values in 'non_neg' must be ranged in", &
                               "[1,NEQ]!" )
            go to 99
         end if

      end if

      if( .not. mfIsEmpty(y_ind_out) ) then

         if( y_ind_out%shape(1) /= 1 .and. y_ind_out%shape(2) /= 1 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "optional arg 'y_ind_out' must be a vector mfArray!" )
            go to 99
         end if
         whole_y_out = .false.
         n_y_out = size(y_ind_out)
         if( n_y_out == neq ) then
            whole_y_out = .true.
         else if( n_y_out > neq ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "size of arg 'y_ind_out' cannot be greater than", &
                               "the number of equations!" )
            go to 99
         end if
         allocate( yindout(n_y_out) )
         yindout(:) = y_ind_out
         if( any(yindout(:)<1) .or. any(neq<yindout(:)) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "all values in 'y_ind_out' must be ranged in [1,NEQ]!" )
            go to 99
         end if

      else
         whole_y_out = .true.
         n_y_out = neq
      end if

      if( .not. mfIsEmpty(monitor_y_ind) ) then

         if( monitor_y_ind%shape(1) /= 1 .and. monitor_y_ind%shape(2) /= 1 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "optional arg 'monitor_y_ind' must be a vector mfArray!" )
            go to 99
         end if
         n_monitor_y = size(monitor_y_ind)
         allocate( monitoryind(n_monitor_y) )
         monitoryind(:) = monitor_y_ind
         if( any(monitoryind(:)<1) .or. any(neq<monitoryind(:)) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "all values in 'monitor_y_ind' must be ranged in [1,NEQ]!", &
                               "(NEQ is the number of equations)" )
            go to 99
         end if
         monitor_y = .true.
         call find_unit( unit_monitor_y )
         ! define RECL (important for large number of columns).
         ! 30 is chosen because it seems that Fortran use at most 30 chars
         ! to write a double precision real number in the case where no format
         ! is specified.
#if defined _INTEL_IFC
! BUG: INTEL-ifort (release 10.1 and other versions...)
!      See also the first use of this in the 'msSaveAscii()' routine.
         recl = (30 + 1)* (n_monitor_y+1) ! (+1 for INTEL-ifort, bug)
#else
         recl = 30 * (n_monitor_y+1)
#endif
         ! when n_monitor_y is small, take care that we have to write
         ! some comments at the beginning of the file (see below)
         recl = max( recl, 80 )
         open( unit_monitor_y, recl=recl, file="daesolve_y.out" )
         write( unit_monitor_y, "(A)" )                                 &
            "% output of DaeSolve at internal time steps"
         write( unit_monitor_y, "(A)" )                                 &
            "%   first column contains the time t"
         write( unit_monitor_y, "(A)" )                                 &
            "%   other columns contain values of y(:) at the following indexes:"
         write( n_monitor_y_str, "(I0)" ) n_monitor_y
         format = "( '%', 4X, " // trim(n_monitor_y_str) // "(1X,I0) )"
         write( unit_monitor_y, format )                                &
            monitoryind(:)
         format_monitor_y = "(ES14.7, " // trim(n_monitor_y_str) // "(2X,ES14.7) )"

      else
         monitor_y = .false.
      end if

      if( .not. mfIsEmpty(monitor_yp_ind) ) then

         if( monitor_yp_ind%shape(1) /= 1 .and. monitor_yp_ind%shape(2) /= 1 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "optional arg 'monitor_yp_ind' must be a vector mfArray!" )
            go to 99
         end if
         n_monitor_yp = size(monitor_yp_ind)
         allocate( monitorypind(n_monitor_yp) )
         monitorypind(:) = monitor_yp_ind
         if( any(monitorypind(:)<1) .or. any(neq<monitorypind(:)) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "all values in 'monitor_yp_ind' must be ranged in [1,NEQ]!", &
                               "(NEQ is the number of equations)" )
            go to 99
         end if
         monitor_yp = .true.
         call find_unit( unit_monitor_yp )
         ! define RECL (important for large number of columns).
         ! 30 is chosen because it seems that Fortran use at most 30 chars
         ! to write a double precision real number in the case where no format
         ! is specified.
#if defined _INTEL_IFC
! BUG: INTEL-ifort (release 10.1 and other versions...)
!      See also the first use of this in the 'msSaveAscii()' routine.
         recl = (30 + 1)* (n_monitor_yp+1) ! (+1 for INTEL-ifort, bug)
#else
         recl = 30 * (n_monitor_yp+1)
#endif
         ! when n_monitor_y is small, take care that we have to write
         ! some comments at the beginning of the file (see below)
         recl = max( recl, 80 )
         open( unit_monitor_yp, recl=recl, file="daesolve_yp.out" )
         write( unit_monitor_yp, "(A)" )                                &
            "% output of DaeSolve at internal time steps"
         write( unit_monitor_yp, "(A)" )                                &
            "%   first column contains the time t"
         write( unit_monitor_yp, "(A)" )                                &
            "%   other columns contain values of yp(:) at the following indexes:"
         write( n_monitor_yp_str, "(I0)" ) n_monitor_yp
         format = "( '%', 4X, " // trim(n_monitor_yp_str) // "(1X,I0) )"
         write( unit_monitor_yp, format )                               &
            monitorypind(:)
         format_monitor_yp = "(ES14.7, " // trim(n_monitor_yp_str) // "(2X,ES14.7) )"

      else
         monitor_yp = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         call process_tol_for_daesolve( tol, rtol, atol, info, neq,     &
                                        ROUTINE_NAME, stat )
         if( stat /= 0 ) go to 99
      else
         allocate( rtol(1), atol(1) )
         rtol(1) = DAE_rtol_def
         atol(1) = DAE_atol_def
      end if

      if( t_span%data_type /= MF_DT_DBLE ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'t_span' must be real!" )
         go to 99
      end if

      if( t_span%shape(2) /= 1 ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'t_span' must be a column vector!" )
         go to 99
      end if

      n_t_out = size(t_span)
      t_end = mfGet( t_span, n_t_out )
      if( first_call ) then
         t_0 = mfGet( t_span, 1 )
         if( n_t_out == 2 ) then
            interval_mode = .false.
         else if( n_t_out > 2 ) then
            interval_mode = .true.
         else
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'t_span' vector must have at least two components!" )
            go to 99
         end if
      else ! continuation
         ! in this case t_0 is the final time of the last integration
         t_0 = rwork(10)
         if( n_t_out == 1 ) then
            interval_mode = .false.
         else if( n_t_out > 1 ) then
            interval_mode = .true.
         end if
      end if

      ! error handling in SLATEC: we accept that XERRMSG prints
      ! some information, but not that it stops.
      call xsetf(1)

      t = t_0

      if( first_call ) then
         ! first call for this problem: the integrator will be completely
         ! initialized
         info(1) = 0
         if( allocated(yy) ) then
            if( size(yy) /= neq ) then
               deallocate( yy )
               allocate( yy(neq) )
            end if
         else
            allocate( yy(neq) )
         end if
         yy(:) = y_0%double(1,:)
         if( allocated(yyp) ) then
            if( size(yyp) /= neq ) then
               deallocate( yyp )
               allocate( yyp(neq) )
            end if
         else
            allocate( yyp(neq) )
         end if
         if( mfIsEmpty( yp_0 ) ) then
            yyp(:) = 0.0d0 ! first guess for finding root
         else
            yyp(:) = yp_0%double(1,:)
         end if
      else
         info(1) = 1
         ! when integration continue, 't', 'y' and 'yp' should not be altered
         if( .not. allocated(yy) ) then
            allocate( yy(neq) )
         end if
         if( .not. allocated(yyp) ) then
            allocate( yyp(neq) )
         end if
      end if

      ! info(3) = 0 ! no need to get intermediate results
      ! info(4) = 0 ! integration can be carried out beyond the end
      info(5) = 1 ! jacobian computed by a user routine
      ! info(12) = 0 ! jacobian matrix is not sparse

      maxord = 5 ! default value

      liw = 20 + neq
      if( first_call ) then
         if( allocated(iwork) ) then
            if( size(iwork) /= liw ) then
               deallocate( iwork )
               allocate( iwork(liw) )
            end if
         else
            allocate( iwork(liw) )
         end if
#ifdef _DEVLP
         ! Avoid unitialized values in Valgrind...
         ! (because not all values are used)
         iwork(:) = 0
#endif
      end if

      if( .not. mfIsEmpty(band) ) then
         if( .not. ( mfIsReal(band) .and. mfIsVector(band) .and.        &
                     size(band) == 2 ) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'band' must be a real vector of size 2!" )
            go to 99
         end if
         iwork(1) = mfGet( band, 1 )
         iwork(2) = mfGet( band, 2 )
         lrw = 40 + (maxord+4)*neq + (2*iwork(1)+iwork(2)+1)*neq
         info(6) = 1 ! jacobian matrix has a banded structure
      else
         lrw = 40 + (maxord+4)*neq + neq**2
         ! info(6) = 0 ! full jacobian matrix (not banded)
      end if

      if( first_call ) then
         if( allocated(rwork) ) then
            if( size(rwork) /= lrw ) then
               deallocate( rwork )
               allocate( rwork(lrw) )
            end if
         else
            allocate( rwork(lrw) )
         end if
#ifdef _DEVLP
         ! Avoid unitialized values in Valgrind...
         ! (because not all values are used)
         rwork(:) = 0.0d0
#endif

         if( de_cic_must_be_called ) then
            if( mfIsEmpty( band ) ) then
               call dae_cic_dense( t_0, yy, y0_ind, yyp, yp0_ind,       &
                                   resid, jac, flag,                    &
                                   check_jac, print_check_jac )
               string = "dae_cic_dense"
            else ! banded jacobian
               call dae_cic_band( t_0, yy, y0_ind, yyp, yp0_ind,        &
                                 resid, jac, band, flag )
               string = "dae_cic_band"
            end if
            if( flag /= 0 ) then
               select case( flag )
                  case( -1 )
                     write(STDERR,*) "-> pb in 'resid' call"
                  case( -2 )
                     ! a comment is already present in dae_cic_dense/dae_cic_band
                     ! (bad number of prescribed components)
                  case( -3 )
                     write(STDERR,*) "-> DAE index perhaps greater than one"
                  case( -4 )
                     write(STDERR,*) "-> convergence not reached"
                  case( -5 )
                     write(STDERR,*) "-> illegal call"
                  case( -7 )
                     write(STDERR,*) "-> pb in 'resid' call"
                  case( -8 )
                     write(STDERR,*) "-> pb in 'jac' call"
                  case default
                     write(STDERR,*) "(MUESLI mfDaeSolve:) internal error:"
                     write(STDERR,*) "        bad value for returned flag from " // trim(string)
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="yes" )
                     stop
               end select
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "failure in computing consistent initial conditions!", &
                                  "(see diagnostic above)" )
               go to 99
            end if
         end if

         if( IC_known ) then
            ! check of consistency of initial conditions: compute the
            ! residual for {t_0,y_0,yp_0} and compare it against the
            ! provided tolerance.
            ! we use the test:
            !     abs(delta(i)) <= rtol(i)*abs(y_0(i)) + atol(i)
            flag = 0 ! IRES in DDASSL
            call resid( t_0, yy, yyp, rwork, flag )
            if( flag /= 0 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "cannot check initial conditions!",   &
                                  "during the 'resid' call, flag /= 0" )
               go to 99
            end if
            flag = 0
            if( info(2) == 1 ) then
               ! 'rtol' and 'atol' are both vectors
               diff_max = 0.0d0
               do i = 1, neq
                  diff = abs(rwork(i)) - (rtol(i)*abs(yy(i)) + atol(i))
                  if( diff > 0.0d0 ) then
                     if( diff > diff_max ) then
                        i_max = i
                        diff_max = diff
                     end if
                  end if
               end do
               if( diff_max > 0.0d0 ) flag = 1
            else
               ! 'rtol' and 'atol' are both scalars
               diff_max = 0.0d0
               do i = 1, neq
                  diff = abs(rwork(i)) - (rtol(1)*abs(yy(i)) + atol(1))
                  if( diff > 0.0d0 ) then
                     if( diff > diff_max ) then
                        i_max = i
                        diff_max = diff
                     end if
                  end if
               end do
               if( diff_max > 0.0d0 ) flag = 1
            end if
            if( flag /= 0 ) then
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i_max, i_named_group, &
                                                          i_group_eqn )
                  write(i_group_eqn_str,"(I0)") i_group_eqn
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                        "initial conditions are not consistent!",       &
                        "Greatest discrepancy is at named equation ",   &
                        "  '" // trim(NAMED_EQN_PTR(i_named_group)%name) // "'", &
                        "  at equation number: " // trim(i_group_eqn_str), &
                        no_pause=.true. )
               else
                  write(i_max_str,"(I0)") i_max
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                        "initial conditions are not consistent!",       &
                        "Greatest discrepancy is at equation number " // trim(i_max_str), &
                        no_pause=.true. )
               end if
               go to 99
            end if
         end if

      end if

      out%data_type = MF_DT_DBLE
      if( print_progress ) then
         ! initialize progression (comes from 'msPrepProgress')
         pc_fact = 100.0d0/( t_end - t_0 ) ! should be finite
         if( disp_times ) then
            write(STDOUT,"(A)")                                         &
            "................... [Progress status from DaeSolve] ...................."
            call system_clock( count_rate=clock_rate, count=clock_init )
            string = "  0.0 % (time left =   0h  0m  0s -- estim. remain. time =    h   m   s)"
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_on_term_no_adv( trim(string) // char(0) )
#else
            write(STDOUT,"(A)",advance="no") trim(string)
            call msFlush(STDOUT)
#endif
         else
            write(STDOUT,"(A)")                                         &
            "... [Progress status from DaeSolve] ..."
            string = "  0.0 %"
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_on_term_no_adv( trim(string) // char(0) )
#else
            write(STDOUT,"(A)",advance="no") trim(string)
            call msFlush(STDOUT)
#endif
         end if
      end if

      if( monitor_y ) then
         write( unit_monitor_y, format_monitor_y ) 0.0d0, yy(monitoryind)
         call flush( unit_monitor_y )
      end if
      if( monitor_yp ) then
         write( unit_monitor_yp, format_monitor_yp ) 0.0d0, yyp(monitoryind)
         call flush( unit_monitor_yp )
      end if
      if( monitor_pause ) then
         print *
         call msPause( "[MUESLI] DAE solver: monitoring pause" )
      end if

      if( interval_mode ) then

         ! copy initial values
         out%shape = [ n_t_out, n_y_out ]
         allocate( out%double(out%shape(1),out%shape(2)), stat=alloc_status )
         if( alloc_status /= 0 ) then
            write(STDERR,"( 1X,A)") "(MUESLI mfDaeSolve:) Error: Cannot allocate the matrix 'y' to store"
            write(STDERR,"(22X,A)") "the solution at all intermediate outputs."
            write(STDERR,"(22X,A)") "For information, the shape is:"
            write(STDERR,"(22X,A,I0)") "  nb of intermediate times = ", n_t_out
            write(STDERR,"(22X,A,I0)") "  total mesh size = ", n_y_out
            write(STDERR,"(22X,A)") "A solution is to decrease either the number of interme-"
            write(STDERR,"(22X,A)") "diate times, i.e. the length of the 't_span' vector, or"
            write(STDERR,"(22X,A)") "the total mesh size."
            stop
         end if

         ! if integration fails, data will be NaN
         out%double(:,:) = MF_NAN
         if( first_call ) then
            if( whole_y_out ) then
               out%double(1,:) = y_0%double(1,:)
            else
               out%double(1,:) = y_0%double(1,yindout)
            end if
         end if

         if( first_call ) then
            i = 2
         else
            i = 1
         end if
         tout = mfGet( t_span, i )
         idid = -1 ! integration is not yet finished, of course
         do ! possible 'cycle'

            if( idid > 1 ) then
               tout = mfGet( t_span, i )
            else
               idid = 0
            end if

            if( tout /= t ) then ! avoid an error in ddassl
               call ddassl( resid, neq, t, yy(1), yyp(1), tout, info(1), rtol, atol, &
                            idid, rwork(1), lrw, iwork(1), liw,         &
                            jac, dummy_dae_spjac, nonneg,               &
                            check_jac, print_check_jac )

               if( idid < 0 ) then
                  ! abnormal return
                  if( idid == -33 ) then
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI mfDaeSolve:) internal error:"
                     write(STDERR,*) "                     invalid input for a SLATEC routine"
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="yes" )
                     stop
                  else if( idid == -1 ) then
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "A large amount of work has been expended (about 500 steps)" )
                     ! let's continue, this was not an error
                     info(1) = 1
                     cycle
                  else if( idid == -8 ) then
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI mfDaeSolve:) Error: the DAE solver encountered"
                     write(STDERR,*) "                     the following problem:"
                     write(STDERR,*) "                     'Jacobian is singular during the integration phase'"
                     call process_singular_jacobian( ROUTINE_NAME )
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="yes" )
                     go to 99
                  else if( idid == -10 ) then
                     ! flag set to -1 in 'resid' (illegal condition)
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "Illegal condition in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'Illegal condition' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, i )
                     call msSet( MF_NAN, t_span, i+1 .to. MF_END )
                     ! no cycle, return to the calling routine...
                  else if( idid == -11 ) then
                     ! flag set to -2 in 'resid' (emergency exit!)
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "Emergency exit in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'Emergency exit' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, i )
                     call msSet( MF_NAN, t_span, i+1 .to. MF_END )
                     ! no cycle, return to the calling routine...
                  else if( idid == -12 ) then
                     ! flag set to +3 in 'resid' (end condition!)
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "End condition in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'End condition' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, i )
                     call msSet( MF_NAN, t_span, i+1 .to. MF_END )
                     ! no cycle, return to the calling routine...
                  else if( idid == -13 ) then
                     ! a message has already been displayed by DDASSL
                     go to 99
                  else if( idid == -14 ) then
                     ! a message has already been displayed by DDASTP
                     go to 99
                  else if( idid == -15 ) then
                     ! a message has already been displayed by DDASTP
                     go to 99
                  else if( idid == -102 ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "the DAE solver encountered the following problem:", &
                                        "failed to compute the initial vector yp_0" )
                     go to 99
                  else
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI mfDaeSolve:) error status: ", idid
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "the DAE solver encountered some problem!", &
                                        "[please examine the status value and the ref. manual]" )
                     go to 99
                  end if
               end if
            else
               idid = 3
            end if

            if( whole_y_out ) then
               out%double(i,:) = yy(:)
            else
               out%double(i,:) = yy(yindout)
            end if

            if( -12 <= idid .and. idid <= -10 ) exit

            i = i + 1
            if( i > n_t_out ) exit

            info(1) = 1

         end do

      else ! .not. interval_mode

         out%shape = [ 1, n_y_out ]
         allocate( out%double(out%shape(1),out%shape(2)), stat=alloc_status )
         if( alloc_status /= 0 ) then
            write(STDERR,"( 1X,A)") "(MUESLI msDaeSolve:) Error: Cannot allocate the vector 'y' to store"
            write(STDERR,"(22X,A)") "the solution."
            write(STDERR,"(22X,A)") "For information, the length of 'y' is:"
            write(STDERR,"(22X,A,I0)") "  total mesh size = ", n_y_out
            write(STDERR,"(22X,A)") "A solution is to decrease the total mesh size."
            stop
         end if

         out%double(:,:) = MF_NAN

         tout = t_end
         do ! possible 'cycle'

            if( tout /= t ) then ! avoid an error in ddassl
               call ddassl( resid, neq, t, yy(1), yyp(1), tout, info(1), rtol, atol, &
                            idid, rwork(1), lrw, iwork(1), liw,         &
                            jac, dummy_dae_spjac, nonneg,               &
                            check_jac, print_check_jac )

               if( idid < 0 ) then
                  ! abnormal return
                  if( idid == -33 ) then
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI mfDaeSolve:) internal error:"
                     write(STDERR,*) "                     invalid input for a SLATEC routine"
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="yes" )
                     stop
                  else if( idid == -1 ) then
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "A large amount of work has been expended (about 500 steps)" )
                     ! let's continue, this was not an error
                     info(1) = 1
                     cycle
                  else if( idid == -8 ) then
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI mfDaeSolve:) Error: the DAE solver encountered"
                     write(STDERR,*) "                     the following problem:"
                     write(STDERR,*) "                     'Jacobian is singular during the integration phase'"
                     call process_singular_jacobian( ROUTINE_NAME )
                     mf_message_displayed = .true.
                     call muesli_trace( pause ="yes" )
                     go to 99
                  else if( idid == -10 ) then
                     ! flag set to -1 in 'resid' (illegal condition)
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "Illegal condition in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'Illegal condition' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, 2 )
                     ! no cycle, return to the calling routine...
                  else if( idid == -11 ) then
                     ! flag set to -2 in 'resid' (emergency exit!)
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "Emergency exit in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'Emergency exit' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, 2 )
                     ! no cycle, return to the calling routine...
                  else if( idid == -12 ) then
                     ! flag set to +3 in 'resid' (end condition!)
                     call PrintMessage( ROUTINE_NAME, "W",              &
                                        "End condition in the 'resid' user subroutine.", &
                                        "(the t_span vector may have been modified!)" )
                     ! must check that t_span is not tempo
                     if( t_span%status_temporary ) then
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "t_span cannot be a temporary mfArray, because it must be modified now!", &
                                           "'End condition' found in the 'resid'" )
                     end if
                     call msSet( t, t_span, 2 )
                     ! no cycle, return to the calling routine...
                  else if( idid == -13 ) then
                     ! a message has already been displayed by DDASSL
                     go to 99
                  else if( idid == -14 ) then
                     ! a message has already been displayed by DDASTP
                     go to 99
                  else if( idid == -15 ) then
                     ! a message has already been displayed by DDASTP
                     go to 99
                  else if( idid == -102 ) then
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "the DAE solver encountered the following problem:", &
                                        "failed to compute the initial vector yp_0" )
                     go to 99
                  else
                     if( print_progress ) then
                        ! avoid writing on the same line as progress bar
                        write(STDERR,*)
                     end if
                     write(STDERR,*) "(MUESLI mfDaeSolve:) error status: ", idid
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "the DAE solver encountered some problem!", &
                                        "[please examine the status value and the ref. manual]" )
                     go to 99
                  end if
               end if
            else
               idid = 3
            end if

            exit

         end do

         if( whole_y_out ) then
            out%double(1,:) = yy(:)
         else
            out%double(1,:) = yy(yindout)
         end if

      end if

      ! when IC are not known, the array 'yp_init_ddassl' is allocated
      ! by the solver DDASSL, so it must be deallocated for a future use
      ! in the same main program...
      if( allocated(yp_init_ddassl) ) then
         deallocate(yp_init_ddassl)
      end if

      if( monitor_y ) then
         close( unit_monitor_y )
         unit_monitor_y = -1
         deallocate( monitoryind )
         n_monitor_y = -1
         format_monitor_y = ""
      end if
      if( monitor_yp ) then
         close( unit_monitor_yp )
         unit_monitor_yp = -1
         deallocate( monitorypind )
         n_monitor_yp = -1
         format_monitor_yp = ""
      end if

      if( print_progress ) then
         ! summarize progression (comes from 'msPostProgress')
         if( disp_times ) then
            call system_clock( count=clock )
            left_time = dble(clock-clock_init)/clock_rate
            call sec_2_hms( left_time, hrs_1, min_1, sec_1 )
            write(string,"(A,I3,A,2(I2,A),A40)") "done. (time left = ", &
                  hrs_1, "h ", min_1, "m ", sec_1, "s)", " "
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_back_on_term( string(1:72) // char(0) )
#else
            call go_home_on_term()
            write(STDOUT,"(A)") string(1:72)
#endif
            write(STDOUT,"(A)")                                         &
            "........................................................................"
         else
            string = "done.  "
#if defined _INTEL_IFC
! BUG: write is done via C because of a bug in 'flush'
            call put_string_back_on_term( string(1:7) // char(0) )
#else
            call go_home_on_term()
            write(STDOUT,"(A)") string(1:7)
#endif
            write(STDOUT,"(A)") "......................................."
         end if
      end if

 99   continue

      if( allocated(dgeco_z) ) then
         deallocate(dgeco_z )
      end if

      if( MF_NUMERICAL_CHECK ) then
         call ddebd2_free2()
      end if

      out%status_temporary = .true.

      call msFreeArgs( t_span, y_0, yp_0 )
      call msAutoRelease( t_span, y_0, yp_0 )

      call mf_restore_fpe( )

#endif
   end function mfDaeSolve_JacUser
