! f90 include file

! Computes consistent Initial Conditions for the DAE solver.
! (cf. docs/numerics/cic.pdf)
!
! Comes from the 'decic' MATLAB-7.6 script (1.1.6.4, 2005/03/07)
!                             (idem up to MATLAB-7.11, at least)
! (Matlab-7.12 introduces only changes in messages' process)

! For dense  Jacobian matrix : 'dae_cic_dense'
!     banded    "       "    : 'dae_cic_band'
!     sparse    "       "    : 'dae_cic_sparse'

!_______________________________________________________________________
!
   subroutine dae_cic_sparse( t0, y0, y0_ind, yp0, yp0_ind,             &
                              resid, jac, flag0 )

      real(kind=MF_DOUBLE), intent(in)  :: t0
      real(kind=MF_DOUBLE)              :: y0(:), yp0(:)
      type(mfArray)                     :: y0_ind, yp0_ind
      integer,              intent(out) :: flag0

      interface
         subroutine resid( t, y, yprime, delta, flag )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
            real(kind=MF_DOUBLE), intent(out)    :: delta(*)
            integer,              intent(in out) :: flag
         end subroutine resid
      end interface

      ! effective shape for 'jacobian' matrix : (nrow,nrow)
      interface
         subroutine jac( t, y, yprime, cj, nrow, job, pd, ipd, jpd, nnz )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*), cj
            integer,              intent(in)     :: nrow, job
            real(kind=MF_DOUBLE), intent(out)    :: pd(*)
            integer,              intent(out)    :: ipd(*), jpd(*)
            integer,              intent(in out) :: nnz
            ! the CSC representation of the matrix is as follows:
            !   pd(1:nnz)    : values of the matrix entries
            !  ipd(1:nnz)    : contains the row indexes
            !  jpd(1:ncol+1) : the pointer to the beginning of the
            !                  columns, in arrays pd,ipd.
            !                (here, the matrix is square, the ncol=nrow)
            !
            ! when job=0, only the value of nnz must be returned
            !
            ! {pd,ipd} must contain all diagonal terms, even if they
            ! are null. (a test is made in SLATEC/DDASSL.F)
            !
            ! moreover, row indexes must be sorted in ascending order
            ! (constraint from UMFPACK)
         end subroutine jac
      end interface
      !------ API end ------
#ifdef _DEVLP

      ! limitation: this specific routine works with Jacobian matrix
      !             in sparse format;
      !         so: only '{mf|ms}DaeSolve_JacUserSP' can call this
      !             routine.

      ! tolerance used to control convergence is the default one:
      ! DAE_rtol_def divided by 1000 (as in decic of MATLAB)

      ! returns flag = 0 : normal exit
      !               -1 : returned flag in 'resid' has been set
      !                    to a non zero value
      !               -2 : bad number of fixed components
      !               -3 : DAE index perhaps greater than one
      !               -4 : convergence failure
      !               -5 : illegal call (y0_ind and yp0_ind must
      !                    be either empty, or of size = neq)
      !               -6 : 'jac' returned an invalid nnz value
      !                    (nnz <= 0) or invalid CSC structure
      !                    returned by 'jac'
      !               -7 : 'resid' returned invalid entries
      !                    (at least one is NaN)
      !               -8 : 'jac' returned invalid entries
      !                    (at least one is NaN)

      integer :: flag, neq, i, j, k, counter, chord, job, nnz, nnz2
      type(mfArray) :: dfdy, dfdyp, dy, dyp, free_y, free_yp
      type(mfArray) :: mf_tmp
      real(kind=MF_DOUBLE), allocatable :: res(:)
      real(kind=MF_DOUBLE) :: cj, resnorm0, resnorm, nrmv, nrmdv,       &
                              factor, threshold

      ! communication with some SLATEC routines
      real(kind=MF_DOUBLE) :: dt_min, dt_max
      integer :: nb_step, nb_resid
      common /slatec_daesolve_1/ dt_min, dt_max, nb_step, nb_resid

      integer :: nb_jac, nb_solv
      common /slatec_daesolve_2/ nb_jac, nb_solv

      real(kind=MF_DOUBLE) :: cic_cpu_time_0
      integer :: cic_nb_resid_0, cic_nb_jac_0, cic_nb_solv_0
      common /dae_cic_1/ cic_cpu_time_0, cic_nb_resid_0,                &
                         cic_nb_jac_0, cic_nb_solv_0

      real(kind=MF_DOUBLE) :: cic_cpu_time_init

      integer :: i_eqn_group, i_group_eqn, i_var_group, i_group_var

   !------ end of declarations -- execution starts hereafter  ------

      call cpu_time( cic_cpu_time_init )

      flag0 = 0 ! normal return

      neq = size(y0)

      if( size(y0_ind) /= 0 .and. size(y0_ind) /= neq ) then
         write(STDERR,*) "ERROR: Illegal call: y0_ind must be either empty,"
         write(STDERR,*) "       or of size equal to the number of equations!"
         flag0 = -5
         go to 99
      end if
      if( size(yp0_ind) /= 0 .and. size(yp0_ind) /= neq ) then
         write(STDERR,*) "ERROR: Illegal call: yp0_ind must be either empty,"
         write(STDERR,*) "       or of size equal to the number of equations!"
         flag0 = -5
         go to 99
      end if

      if( mfIsEmpty(y0_ind) ) then
         free_y = [ (i, i = 1, neq) ]
         free_y_full = .true.
      else
         call msAssign( free_y, mfFind( y0_ind == 0.0d0 ) )
         if( size(free_y) == neq ) then
            free_y_full = .true.
         else
            free_y_full = .false.
         end if
      end if
      if( mfIsEmpty(yp0_ind) ) then
         free_yp = [ (i, i = 1, neq) ]
         free_yp_full = .true.
      else
         call msAssign( free_yp, mfFind( yp0_ind == 0.0d0 ) )
         if( size(free_yp) == neq ) then
            free_yp_full = .true.
         else
            free_yp_full = .false.
         end if
      end if
      if( size(free_y)+size(free_yp) < neq ) then
         write(STDERR,"(1X,A,I0,A)") "ERROR: You cannot specify more than ", neq, " components"
         write(STDERR,"(1X,A)")      "       of y0 and yp0!"
         flag0 = -2
         go to 99
      end if

      allocate( res(neq) )

      flag = 0
      nb_resid = nb_resid + 1
      cic_nb_resid_0 = cic_nb_resid_0 + 1

      call mf_restore_fpe( )
      call resid( t0, y0, yp0, res, flag )
      call mf_save_and_disable_fpe( )

      if( MF_NUMERICAL_CHECK ) then
         ! res vector must not contain any NaN value
         do i = 1, neq
            if( isnan(res(i)) ) then
               print "(/,A)", "(MUESLI Daesolve:) [C.I.C.] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied RESID routine."
               print "(20X,A,I0)", "This occured in delta(i) for i = ", i
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,         &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",              &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               flag0 = -7
               go to 99
            end if
         end do
      endif

      if( flag /= 0 ) then
         flag0 = -1 ! pb in 'resid' call
         go to 99
      end if
      resnorm0 = mfNorm( mf(res) )
      ! update of rwesnorm0 for two reasons:
      !   first, increase the value of resnorm0 to be sure that
      !          new iterate values will be slightly less;
      !   second, to avoid too small initial value, choose a threshold
      !          value.
      ! FSF_G95 requires a greater increase for resnorm,
      ! due to roundoff errors during the call of 'resid'!
      ! The factor 1.2d0 should be ok for other compilers, but for the
      ! the former compilers, the factor 12.0d0 is absolutely required!
!### TODO 2: maintenant que je n'utilise plus FSF_G95, on pourrait baisser
!          la valeur du facteur à 1.2d0
      if( resnorm0 < sqrt(MF_EPS) ) resnorm0 = 12.0d0*resnorm0
      resnorm0 = max( resnorm0, 1.0d3*MF_EPS )

      ! getting nnz of jacobian sparse matrix
      job = 0
#if defined _GNU_GFC
      ! pointers need to be associated before calling routine jac()
      call msAssign( dfdy, mfSpAlloc(1,1) )
#endif

      call mf_restore_fpe( )
      call jac( t0, y0, yp0, cj, neq, job,                              &
                dfdy%a, dfdy%i, dfdy%j, nnz )
      call mf_save_and_disable_fpe( )

      if( MF_NUMERICAL_CHECK ) then
         ! check the validity of NNZ...
         ! avoid integer overflow
         if( nnz < 1 .or. dble(neq)**2 < dble(nnz) ) then
            write(STDERR,"(1X,A)")    "ERROR: 'jac' sparse routine returned an invalid nnz value"
            write(STDERR,"(1X,A,I0)") "       nnz = ", nnz
            flag0 = -6
            go to 99
         end if
      endif

      job = 1
      ! initialize the partial derivatives
      cj = 0.0d0
      call msAssign( dfdy, mfSpAlloc(neq,neq,nnz) )
      nb_jac = nb_jac + 1
      cic_nb_jac_0 = cic_nb_jac_0 + 1

      call mf_restore_fpe( )
      call jac( t0, y0, yp0, cj, neq, job,                              &
                dfdy%a, dfdy%i, dfdy%j, nnz )
      call mf_save_and_disable_fpe( )

      if( MF_NUMERICAL_CHECK ) then
         ! check the validity of the CSC vectors...
         ! 1) indices in i(:) must be in [1,neq]
         if( minval( dfdy%i(1:nnz) ) < 1 ) then
            write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
            write(STDERR,"(1X,A)") "       (at least one index in 'ipd' is < 1)"
            flag0 = -6
            go to 99
         end if
         if( maxval( dfdy%i(1:nnz) ) > neq ) then
            write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
            write(STDERR,"(1X,A)") "       (at least one index in 'ipd' is > neq)"
            flag0 = -6
            go to 99
         end if
         ! 2) indices in j(:) must be in increasing order
         do k = 2, neq+1
            if( dfdy%j(k) < dfdy%j(k-1) ) then
               write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
               write(STDERR,"(1X,A)") "       ('jpd' must have increasing indices)"
               flag0 = -6
               go to 99
            end if
         end do
         ! 3) indices in j(:) must be in [1,nnz]
         do k = 1, neq
            if( dfdy%j(k) < 1 .or. nnz < dfdy%j(k) ) then
               write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
               write(STDERR,"(1X,A)") "       ('jpd' must be in [1,nnz])"
               flag0 = -6
               go to 99
            end if
         end do
         ! 4) last index in j(:) must be exactly nnz+1
         if( dfdy%j(neq+1) /= nnz+1 ) then
            write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
            write(STDERR,"(1X,A)") "       (last value of 'jpd' must be equal to nnz+1)"
            flag0 = -6
            go to 99
         end if
         ! 5) all entries must be valid (not NaN)
         do k = 1, nnz
            if( isnan(dfdy%a(k)) ) then
               write(STDERR,"(1X,A)") "ERROR: 'jac' returned NaN value(s)"
               i = dfdy%i(k)
               ! find column
               do j = 1, neq-1
                  if( k < dfdy%j(j+1) ) exit
               end do
               print "(20X,A,I0,1X,I0)", "This occured in matrix for (i,j) = ", i, " ", j
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                  if( NAMED_VAR_PRESENCE ) then
                     call search_index_in_var_groups( k, i_var_group,   &
                                                         i_group_var )
                     print "(/,20X,A,A)", "Named variable is: ",        &
                                          trim(NAMED_EQN_PTR(i_var_group)%name)
                     print "(20X,A,I0)", "Variable number is: ", i_group_var
                  end if
               end if
               flag0 = -8
               go to 99
            end if
         end do
      endif
      dfdy%row_sorted = TRUE

      cj = 1.0d0
      call msAssign( mf_tmp, mfSpAlloc(neq,neq,nnz) )
      nb_jac = nb_jac + 1
      cic_nb_jac_0 = cic_nb_jac_0 + 1

      call mf_restore_fpe( )
      call jac( t0, y0, yp0, cj, neq, job,                              &
                mf_tmp%a, mf_tmp%i, mf_tmp%j, nnz )
      call mf_save_and_disable_fpe( )

      mf_tmp%row_sorted = TRUE
      call msAssign( dfdyp, mf_tmp - dfdy )
      ! Due to roundoff errors, some elements may be close to zero
      ! but have not been eliminated after the substraction...
      ! (Here, no need to keep the diagonal null elements)
      nnz2 = mfNnz(dfdyp)
      threshold = MF_EPS * maxval( abs( dfdyp%a(1:nnz2) ) )
      call msAssign( dfdyp, mfSpCut(dfdyp,threshold) )

      do counter = 1, 10

         sls_daecic_call_nb = 0

         do chord = 1, 2 ! orig. 3

            nb_solv = nb_solv + 1
            cic_nb_solv_0 = cic_nb_solv_0 + 1

            ! dy and dyp must be vectors of zeros before calling sls_sparse
            dy = mfZeros(neq,1)
            dyp = mfZeros(neq,1)

            call sls_sparse( res, dfdy, dfdyp, neq, free_y, free_yp,    &
                             dy, dyp, flag )
            if( flag /= 0 ) then
               flag0 = flag ! pb in 'sls' (-2 or -3)
               go to 99
            end if

            ! if the increments are too big, limit the change in norm
            ! to a factor of 2 -- trust region
            nrmv = max( mfDble(mfNorm(y0.vc.yp0)), DAE_atol_def )
            nrmdv = mfDble( mfNorm(dy.vc.dyp) )
            if( nrmdv > 2.0d0*nrmv ) then
               factor = 2.0d0*nrmv/nrmdv
               call msAssign( dy, factor*dy )
               call msAssign( dyp, factor*dyp )
               nrmdv = factor*nrmdv
            end if
            y0 = .t.mf(y0) + dy
            yp0 = .t.mf(yp0) + dyp
            flag = 0
            nb_resid = nb_resid + 1
            cic_nb_resid_0 = cic_nb_resid_0 + 1

            call mf_restore_fpe( )
            call resid( t0, y0, yp0, res, flag )
            call mf_save_and_disable_fpe( )

            if( flag /= 0 ) then
               flag0 = -1 ! pb in 'resid' call
               go to 99
            end if
            resnorm = mfNorm( mf(res) )

            ! Test for convergence. The norm of the residual must be
            ! no greater than the initial guess and the norm of the
            ! increments must be small in a relative sense.
            if( resnorm <= resnorm0 .and.                               &
               nrmdv <= 1.0d-3*DAE_rtol_def*nrmv ) then
               go to 99
            end if

         end do

         cj = 0.0d0
         nb_jac = nb_jac + 1
         cic_nb_jac_0 = cic_nb_jac_0 + 1

         call mf_restore_fpe( )
         call jac( t0, y0, yp0, cj, neq, job,                           &
                   dfdy%a, dfdy%i, dfdy%j, nnz )
         call mf_save_and_disable_fpe( )

         cj = 1.0d0
         nb_jac = nb_jac + 1
         cic_nb_jac_0 = cic_nb_jac_0 + 1

         call mf_restore_fpe( )
         call jac( t0, y0, yp0, cj, neq, job,                           &
                   mf_tmp%a, mf_tmp%i, mf_tmp%j, nnz )
         call mf_save_and_disable_fpe( )

         call msAssign( dfdyp, mf_tmp - dfdy )

         nnz2 = mfNnz(dfdyp)
         threshold = MF_EPS * maxval( abs( dfdyp%a(1:nnz2) ) )
         call msAssign( dfdyp, mfSpCut(dfdyp,threshold) )

      end do

      flag0 = -4 ! convergence failure

 99   continue

      call cpu_time( cic_cpu_time_0 )
      cic_cpu_time_0 = cic_cpu_time_0 - cic_cpu_time_init

      call msSilentRelease( dfdy, dfdyp, dy, dyp, free_y, free_yp )
      call msSilentRelease( mf_tmp )

#endif
   end subroutine dae_cic_sparse
!_______________________________________________________________________
!
   subroutine sls_sparse( res, dfdy, dfdyp, neq, free_y, free_yp,       &
                          dy, dyp, flag )

      real(kind=MF_DOUBLE) :: res(:)
      type(mfArray) :: dfdy, dfdyp
      type(mfArray) :: dy, dyp, free_y, free_yp
      integer :: neq, flag
      !------ API end ------
#ifdef _DEVLP

      ! Solve the underdetermined system
      !     0 = res + dfdyp*dyp + dfdy*dy
      ! A solution is obtained with as many components as possible
      ! of (transformed) dy and dyp set to zero.

      ! Sparse case:
      ! (from 2.6.0, msQR returns a permutation vector)
      !  if A(:,p) = Q*R then solution of A*x = b is
      !    x(p) = R\(Q'*b)        [MATLAB syntax]

      ! returns flag = 0 : normal exit
      !               -2 : bad number of fixed components
      !               -3 : DAE index perhaps greater than one

      integer :: free, rank, rankdef, Srank, nfree_yp, i
      type(mfArray) :: Q, R, p, d, S, w, w1p, wp

   !------ end of declarations -- execution starts hereafter  ------

      sls_daecic_call_nb = sls_daecic_call_nb + 1

      flag = 0 ! normal return
      free = size(free_y) + size(free_yp)

      ! ---- part A ----
      !
      if( mfIsEmpty(free_y) ) then
         ! Solve 0 = res + dfdyp*dyp
         call rrqr( dfdyp, rank, Q, R, p )
         rankdef = neq - rank
         if( rankdef > 0 ) then
            if( rankdef <= free ) then
               write(STDERR,"(/,2X,A)")    "ERROR: bad number of prescribed components!"
               write(STDERR,"(2X,A,I0,A)") "       advice: Try freeing ", rankdef, " fixed components."
               write(STDERR,"(2X,A)")      "(For a DAE system having " // achar(27) // &
                                           "[3mNd" // achar(27) // "[0m differential equations and " &
                                           // achar(27) // "[3mNa" // achar(27) // "[0m algebraic equations,"
               write(STDERR,"(2X,A)")      " you can only prescribed at most the " // achar(27) // &
                                           "[3mNd" // achar(27) // "[0m components of y0_ind corresponding to"
               write(STDERR,"(2X,A)")      " the differential equations; in all cases the " // achar(27) // &
                                           "[3mNa" // achar(27) // "[0m components of y0_ind corresp-"
               write(STDERR,"(2X,A)")      " onding to the algebraic equations must be left free.)"
               flag = -2
            else
               write(STDERR,"(/,2X,A)") "ERROR: Index of your DAE system may be greater than one."
               flag = -3
            end if
            go to 99
         end if
         call msAssign( d, - .t. mfMul(mf(res),Q) )
         call msAssign( dyp, mfRowPerm(mfLDiv(R,d),mfInvPerm(p)) )
         go to 99
      end if

      ! ---- part B ----
      !
      if( mfIsEmpty(free_yp) ) then
         ! Solve 0 = res + dfdy*dy
         call rrqr( dfdy, rank, Q, R, p )
         rankdef = neq - rank
         if( rankdef > 0 ) then
            if( rankdef <= free ) then
               write(STDERR,"(/,2X,A)")    "ERROR: bad number of prescribed components!"
               write(STDERR,"(2X,A,I0,A)") "       advice: Try freeing ", rankdef, " fixed components."
               write(STDERR,"(2X,A)")      "(For a DAE system having " // achar(27) // &
                                           "[3mNd" // achar(27) // "[0m differential equations and " &
                                           // achar(27) // "[3mNa" // achar(27) // "[0m algebraic equations,"
               write(STDERR,"(2X,A)")      " you can only prescribed at most the " // achar(27) // &
                                           "[3mNd" // achar(27) // "[0m components of y0_ind corresponding to"
               write(STDERR,"(2X,A)")      " the differential equations; in all cases the " // achar(27) // &
                                           "[3mNa" // achar(27) // "[0m components of y0_ind corresp-"
               write(STDERR,"(2X,A)")      " onding to the algebraic equations must be left free.)"
               flag = -2
            else
               write(STDERR,"(/,2X,A)") " ERROR: Index of your DAE system may be greater than one."
               flag = -3
            end if
            go to 99
         end if
         call msAssign( d, - .t. mfMul(mf(res),Q) )
         call msAssign( dyp, mfRowPerm(mfLDiv(R,d),mfInvPerm(p)) )
         go to 99
      end if

      ! ---- part C ----
      !
      if( sls_daecic_call_nb == 1 ) then
         ! Eliminate variables that are not free
         if( .not. free_y_full ) then
            call msAssign( dfdy, mfGet( dfdy, MF_ALL, free_y ) )
         end if
         if( .not. free_yp_full ) then
            call msAssign( dfdyp, mfGet( dfdyp, MF_ALL, free_yp ) )
         end if
      end if

      call rrqr( dfdyp, rank, Q, R, p )
      call msAssign( d, - .t. mfMul(mf(res),Q) )
      if( rank == neq ) then
         call msAssign( w, mfZeros(neq,1) )
         call msAssign( wp, mfRowPerm(mfLDiv(R,d),mfInvPerm(p)) )
      else
         call msAssign( S, mfMul(.t.Q,dfdy) )
         call msAssign( w, mfGet(S,rank+1.to.MF_END,MF_ALL) )
         call rrqr( w, Srank )
         rankdef = neq - (rank + Srank )
         if( rankdef > 0 ) then
            if( rankdef <= free ) then
               write(STDERR,"(/,2X,A)")    "ERROR: bad number of prescribed components!"
               write(STDERR,"(2X,A,I0,A)") "       advice: Try freeing ", rankdef, " fixed components."
               write(STDERR,"(2X,A)")      "(For a DAE system having " // achar(27) // &
                                           "[3mNd" // achar(27) // "[0m differential equations and " &
                                           // achar(27) // "[3mNa" // achar(27) // "[0m algebraic equations,"
               write(STDERR,"(2X,A)")      " you can only prescribed at most the " // achar(27) // &
                                           "[3mNd" // achar(27) // "[0m components of y0_ind corresponding to"
               write(STDERR,"(2X,A)")      " the differential equations; in all cases the " // achar(27) // &
                                           "[3mNa" // achar(27) // "[0m components of y0_ind corresp-"
               write(STDERR,"(2X,A)")      " onding to the algebraic equations must be left free.)"
               flag = -2
            else
               write(STDERR,"(/,2X,A)") "ERROR: Index of your DAE system may be greater than one."
               flag = -3
            end if
            go to 99
         end if
         call msAssign( w, mfLDiv( w, mfGet(d,rank+1.to.MF_END) ) )
         call msAssign( w1p, mfLDiv( mfGet(R,1.to.rank,1.to.rank),      &
                             mfGet(d,1.to.rank) - mfMul(mfGet(S,1.to.rank,MF_ALL),w) ) )
         nfree_yp = size(free_yp)
         call msAssign( wp, mfRowPerm(w1p.vc.mfZeros(nfree_yp-rank,1),mfInvPerm(p)) )
      end if

      call msSet( w, dy, free_y )
      ! dy must be a column vector
      if( dy%shape(1) == 1 ) then
         call msAssign( dy, .t. dy )
      end if

      call msSet( wp, dyp, free_yp )
      ! dyp must be a column vector
      if( dyp%shape(1) == 1 ) then
         call msAssign( dyp, .t. dyp )
      end if

 99   continue

      call msSilentRelease( Q, R, p, d, S )
      call msSilentRelease( w, w1p, wp )

#endif
   end subroutine sls_sparse
