! f90 include file

! Computes consistent Initial Conditions for the DAE solver.
! (cf. docs/numerics/cic.pdf)
!
! Comes from the 'decic' MATLAB-7.6 script (1.1.6.4, 2005/03/07)
!                             (idem up to MATLAB-7.11, at least)
! (Matlab-7.12 introduces only changes in messages' process)
!
! Inspecting in Matlab public sources: decic.m didn't change since 2013.

! For dense  Jacobian matrix : 'dae_cic_dense'
!     banded    "       "    : 'dae_cic_band'
!     sparse    "       "    : 'dae_cic_sparse'

!_______________________________________________________________________
!
   subroutine dae_cic_dense( t0, y0, y0_ind, yp0, yp0_ind,              &
                             resid, jac, flag0,                         &
                             check_jac, print_check_jac )

      use minpack

      real(kind=MF_DOUBLE), intent(in)  :: t0
      real(kind=MF_DOUBLE)              :: y0(:), yp0(:)
      type(mfArray)                     :: y0_ind, yp0_ind
      integer,              intent(out) :: flag0
      integer                           :: check_jac, print_check_jac

      interface
         subroutine resid( t, y, yprime, delta, flag )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
            real(kind=MF_DOUBLE), intent(out)    :: delta(*)
            integer,              intent(in out) :: flag
         end subroutine resid
      end interface

      ! effective shape for 'jacobian' matrix : [nrow,nrow] (i.e. ldjac=nrow)
      interface
         subroutine jac( t, y, yprime, jacobian, cj, ldjac )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)  :: t, y(*), yprime(*), cj
            integer,              intent(in)  :: ldjac
            real(kind=MF_DOUBLE), intent(out) :: jacobian(ldjac,*)
         end subroutine jac
      end interface
      !------ API end ------
#ifdef _DEVLP

      ! limitation: this specific routine works with Jacobian matrix
      !             in dense format;
      !         so: only '{mf|ms}DaeSolve_JacUser' can call this
      !             routine, when 'band' is not used.

      ! tolerance used to control convergence is the default one:
      ! DAE_rtol_def divided by 1000 (as in decic of MATLAB)

      ! returns flag = 0 : normal exit
      !               -1 : pb in 'resid' call
      !               -2 : bad number of fixed components
      !               -3 : DAE index perhaps greater than one
      !               -4 : convergence failure
      !               -5 : illegal call (y0_ind and yp0_ind must
      !                    be either empty, or of size = neq)
      !               -7 : 'resid' returned invalid entries
      !                    (at least one is NaN)
      !               -8 : 'jac' returned invalid entries
      !                    (at least one is NaN)

      integer :: flag, neq, i, k, counter, chord
      type(mfArray), target :: dfdy, dfdyp
      type(mfArray) :: dy, dyp, free_y, free_yp
      real(kind=MF_DOUBLE), pointer :: dfdy_ptr(:,:), dfdyp_ptr(:,:)
      real(kind=MF_DOUBLE), allocatable :: res(:)
      real(kind=MF_DOUBLE) :: cj, resnorm0, resnorm, nrmv, nrmdv,       &
                              factor

      ! communication with some SLATEC routines
      real(kind=MF_DOUBLE) :: dt_min, dt_max
      integer :: nb_step, nb_resid
      common /slatec_daesolve_1/ dt_min, dt_max, nb_step, nb_resid

      integer :: nb_jac, nb_solv
      common /slatec_daesolve_2/ nb_jac, nb_solv

      real(kind=MF_DOUBLE) :: cic_cpu_time_0
      integer :: cic_nb_resid_0, cic_nb_jac_0, cic_nb_solv_0
      common /dae_cic_1/ cic_cpu_time_0, cic_nb_resid_0,                &
                         cic_nb_jac_0, cic_nb_solv_0

      real(kind=MF_DOUBLE) :: cic_cpu_time_init

      real(kind=MF_DOUBLE), allocatable :: y0new(:), dy0(:), FTEMP(:),  &
                                       quality(:)
      real, allocatable :: err(:,:) ! single precision is sufficient

      logical :: box_constrained = .false.
      real(kind=MF_DOUBLE) :: ubounds(1)

      integer :: i_eqn_group, i_group_eqn, i_var_group, i_group_var
      integer :: free
      type(mfArray), pointer :: dfdy2, dfdyp2

   !------ end of declarations -- execution starts hereafter  ------

      call cpu_time( cic_cpu_time_init )

      flag0 = 0 ! normal return

      neq = size(y0)

      if( size(y0_ind) /= 0 .and. size(y0_ind) /= neq ) then
         write(STDERR,*) "ERROR: Illegal call: y0_ind must be either empty,"
         write(STDERR,*) "       or of size equal to the number of equations!"
         flag0 = -5
         go to 99
      end if
      if( size(yp0_ind) /= 0 .and. size(yp0_ind) /= neq ) then
         write(STDERR,*) "ERROR: Illegal call: yp0_ind must be either empty,"
         write(STDERR,*) "       or of size equal to the number of equations!"
         flag0 = -5
         go to 99
      end if

      if( mfIsEmpty(y0_ind) ) then
         free_y = [ (i, i = 1, neq) ]
         free_y_full = .true.
      else
         call msAssign( free_y, mfFind( y0_ind == 0.0d0 ) )
         if( size(free_y) == neq ) then
            free_y_full = .true.
         else
            free_y_full = .false.
         end if
      end if
      if( mfIsEmpty(yp0_ind) ) then
         free_yp = [ (i, i = 1, neq) ]
         free_yp_full = .true.
      else
         call msAssign( free_yp, mfFind( yp0_ind == 0.0d0 ) )
         if( size(free_yp) == neq ) then
            free_yp_full = .true.
         else
            free_yp_full = .false.
         end if
      end if

      free = size(free_y) + size(free_yp)
      if( free < neq ) then
         write(STDERR,"(1X,A,I0,A)") "ERROR: You cannot specify more than ", neq, " components"
         write(STDERR,"(1X,A)")      "       of y0 and/or yp0!"
         flag0 = -2
         go to 99
      end if

      allocate( res(neq) )

      flag = 0
      nb_resid = nb_resid + 1
      cic_nb_resid_0 = cic_nb_resid_0 + 1

      call mf_restore_fpe( )
      call resid( t0, y0, yp0, res, flag )
      call mf_save_and_disable_fpe( )

      if( MF_NUMERICAL_CHECK ) then
         ! res vector must not contain any NaN value
         do i = 1, neq
            if( isnan(res(i)) ) then
               print "(/,A)", "(MUESLI Daesolve:) [C.I.C.] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied RESID routine."
               print "(20X,A,I0)", "This occured in delta(i) for i = ", i
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               flag0 = -7
               go to 99
            end if
         end do
      endif

      if( flag /= 0 ) then
         flag0 = -1 ! pb in 'resid' call
         go to 99
      end if
      resnorm0 = mfNorm( mf(res) )
      ! update of resnorm0 for two reasons:
      !   first, increase the value of resnorm0 to be sure that
      !          new iterate values will be slightly less;
      !   second, to avoid too small initial value, choose a threshold
      !          value.
      ! FSF_G95 requires a greater increase for resnorm,
      ! due to roundoff errors during the call of 'resid'!
      ! The factor 1.2d0 should be ok for other compilers, but for the
      ! the former compilers, the factor 7.d0 is absolutely required!
      if( resnorm0 < sqrt(MF_EPS) ) resnorm0 = 12.0d0*resnorm0
      resnorm0 = max( resnorm0, 1.0d3*MF_EPS )

      ! initialize the partial derivatives
      cj = 0.0d0
      call msAssign( dfdy, mfZeros(neq,neq) )
      call msPointer( dfdy, dfdy_ptr, no_crc=.true. )
      nb_jac = nb_jac + 1
      cic_nb_jac_0 = cic_nb_jac_0 + 1

      call mf_restore_fpe( )
      call jac( t0, y0, yp0, dfdy_ptr, cj, neq )
      call mf_save_and_disable_fpe( )

      if( MF_NUMERICAL_CHECK ) then

         ! dfdy_ptr matrix must not contain any NaN value
         do i = 1, neq
         do k = 1, neq
            if( isnan(dfdy_ptr(i,k)) ) then
               print "(/,A)", "(MUESLI Daesolve:) [C.I.C.] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                              "user-supplied JAC routine."
               print "(20X,A,I0,1X,I0)", "This occured in matrix for (i,j) = ", i, k
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                  if( NAMED_VAR_PRESENCE ) then
                     call search_index_in_var_groups( k, i_var_group,   &
                                                         i_group_var )
                     print "(/,20X,A,A)", "Named variable is: ",        &
                                          trim(NAMED_EQN_PTR(i_var_group)%name)
                     print "(20X,A,I0)", "Variable number is: ", i_group_var
                  end if
               end if
               flag0 = -8
               go to 99
            end if
         end do
         end do

         if( check_jac == 1 ) then ! quick check of the jacobian

            allocate( y0new(neq), dy0(neq) )

            ! here, as cj = 0, we check only d.R/d.y
            ! (anyway, it's often the most difficult part to write)
            call chkder( neq, neq, y0, res, dfdy_ptr, y0new, dy0, FTEMP, 1, &
                         quality, box_constrained, ubounds )

            allocate( FTEMP(neq) )

            call mf_restore_fpe( )
            call resid( t0, y0new, yp0, FTEMP, flag )
            call mf_save_and_disable_fpe( )

            if( flag /= 0 ) then
               flag0 = -1 ! pb in 'resid' call
               go to 99
            end if

            allocate( quality(neq) )

            call chkder( neq, neq, y0, res, dfdy_ptr, y0new, dy0, FTEMP, 2, &
                         quality, box_constrained, ubounds )

            call post_chk_der( "DaeSolve", neq, neq, quality,           &
                               print_check_jac, time=t0 )

         else if( check_jac == 2 ) then ! full check of the jacobian

            allocate( err(neq,neq) )

            ! here we check only d.R/d.y
            ! (anyway, it's often the most difficult part to write)
            call full_chk_der_dae( neq, t0, y0, yp0, resid, res, dfdy_ptr, &
                                   err, flag )

            if( flag /= 0 ) then
               flag0 = -1 ! pb in 'resid' call
               go to 99
            end if

            call post_full_chk_der( "DaeSolve", neq, neq, err,          &
                                    print_check_jac, time=t0 )

         end if

      endif

      ! initialize the partial derivatives
      cj = 1.0d0
      call msAssign( dfdyp, mfZeros(neq,neq) )
      call msPointer( dfdyp, dfdyp_ptr, no_crc=.true. )
      nb_jac = nb_jac + 1
      cic_nb_jac_0 = cic_nb_jac_0 + 1

      call mf_restore_fpe( )
      call jac( t0, y0, yp0, dfdyp_ptr, cj, neq )
      call mf_save_and_disable_fpe( )

      call msAssign( dfdyp, dfdyp - dfdy )

      do counter = 1, 10

         sls_daecic_call_nb = 0

         do chord = 1, 2 ! orig. 3

            nb_solv = nb_solv + 1
            cic_nb_solv_0 = cic_nb_solv_0 + 1

            if( mfIsEmpty(free_y) ) then
               dy = mfZeros(neq,1)
               call sls_dense_A( res, dfdyp, neq, free,                 &
                                 dyp, flag )
            else if( mfIsEmpty(free_yp) ) then
               call sls_dense_B( res, dfdy, neq, free,                  &
                                 dy, flag )
               dyp = mfZeros(neq,1)
            else
               sls_daecic_call_nb = sls_daecic_call_nb + 1
               if( sls_daecic_call_nb == 1 ) then
                  ! Eliminate variables that are not free
                  if( .not. free_y_full ) then
                     allocate( dfdy2 )
                     call msAssign( dfdy2, mfGet( dfdy, MF_ALL, free_y ) )
                  else
                     dfdy2 => dfdy
                  end if
                  if( .not. free_yp_full ) then
                     allocate( dfdyp2 )
                     call msAssign( dfdyp2, mfGet( dfdyp, MF_ALL, free_yp ) )
                  else
                     dfdyp2 => dfdyp
                  end if
               end if
               ! dy and dyp must be vectors of zeros before calling sls_dense_C
               dy = mfZeros(neq,1)
               dyp = mfZeros(neq,1)
               call sls_dense_C( res, dfdy2, dfdyp2, neq, free_y, free_yp, &
                                 dy, dyp, flag )
            end if

            if( flag /= 0 ) then
               flag0 = flag ! pb in 'sls' (-2 or -3)
               go to 99
            end if

            ! if the increments are too big, limit the change in norm
            ! to a factor of 2 -- trust region
!### TODO 2:
! ci-dessous, 2 appels à la norm-2 pour des matrices, cela coûte cher (SVD),
! même si les matrices sont constituées de 2 lignes (?)
! => norm-1 ?, mfNormEst ?
            nrmv = max( mfDble(mfNorm(y0.hc.yp0)), DAE_atol_def )
            nrmdv = mfDble( mfNorm(dy.vc.dyp) )
            if( nrmdv > 2.0d0*nrmv ) then
               factor = 2.0d0*nrmv/nrmdv
               call msAssign( dy, factor*dy )
               call msAssign( dyp, factor*dyp )
               nrmdv = factor*nrmdv
            end if
            y0 = .t.mf(y0) + dy
            yp0 = .t.mf(yp0) + dyp
            flag = 0
            nb_resid = nb_resid + 1
            cic_nb_resid_0 = cic_nb_resid_0 + 1

            call mf_restore_fpe( )
            call resid( t0, y0, yp0, res, flag )
            call mf_save_and_disable_fpe( )

            if( flag /= 0 ) then
               flag0 = -1 ! pb in 'resid' call
               go to 99
            end if
            resnorm = mfNorm( mf(res) )

            ! Test for convergence. The norm of the residual must be
            ! no greater than the initial guess and the norm of the
            ! increments must be small in a relative sense.
            if( resnorm <= resnorm0 .and.                               &
               nrmdv <= 1.0d-3*DAE_rtol_def*nrmv ) then
               go to 99
            end if

         end do

         if( sls_daecic_call_nb >= 1 ) then
            if( .not. free_y_full ) then
               call msSilentRelease( dfdy2 )
               deallocate( dfdy2 )
            end if
            dfdy2 => null()
            if( .not. free_yp_full ) then
               call msSilentRelease( dfdyp2 )
               deallocate( dfdyp2 )
            end if
            dfdyp2 => null()
            ! to be sure not freeing twice at label 99
            sls_daecic_call_nb = 0
         end if

         cj = 0.0d0
         ! Set jacobian to zero before calling the user-supplied routine
         dfdy_ptr(:,:) = 0.0d0
         nb_jac = nb_jac + 1
         cic_nb_jac_0 = cic_nb_jac_0 + 1

         call mf_restore_fpe( )
         call jac( t0, y0, yp0, dfdy_ptr, cj, neq )
         call mf_save_and_disable_fpe( )

         if( MF_NUMERICAL_CHECK ) then

               ! dfdy_ptr matrix must not contain any NaN value
               do i = 1, neq
               do k = 1, neq
                  if( isnan(dfdy_ptr(i,k)) ) then
                     write(STDERR,"(1X,A)") "ERROR: 'jac' returned NaN value(s)"
                     flag0 = -8
                     go to 99
                  end if
               end do
               end do

               if( check_jac == 1 ) then ! quick check of the jacobian

                  allocate( y0new(neq), dy0(neq) )

                  ! here, as cj = 0, we check only d.R/d.y
                  ! (c'est souvent la partie la plus compliqué à écrire)
                  call chkder( neq, neq, y0, res, dfdy_ptr, y0new, dy0, FTEMP, 1, &
                               quality, box_constrained, ubounds )

                  allocate( FTEMP(neq) )

                  call mf_restore_fpe( )
                  call resid( t0, y0new, yp0, FTEMP, flag )
                  call mf_save_and_disable_fpe( )

                  if( flag /= 0 ) then
                     flag0 = -1 ! pb in 'resid' call
                     go to 99
                  end if

                  allocate( quality(neq) )

                  call chkder( neq, neq, y0, res, dfdy_ptr, y0new, dy0, FTEMP, 2, &
                               quality, box_constrained, ubounds )

                  call post_chk_der( "DaeSolve", neq, neq, quality,     &
                                     print_check_jac, time=t0 )

               else if( check_jac == 2 ) then ! full check of the jacobian

                  ! here we check only d.R/d.y
                  ! (anyway, it's often the most difficult part to write)
                  call full_chk_der_dae( neq, t0, y0, yp0, resid, res, dfdy_ptr, &
                                       err, flag )

                  if( flag /= 0 ) then
                     flag0 = -1 ! pb in 'resid' call
                     go to 99
                  end if

                  call post_full_chk_der( "DaeSolve", neq, neq, err,    &
                                          print_check_jac, time=t0 )

            end if

         endif

         cj = 1.0d0
         ! Set jacobian to zero before calling the user-supplied routine
         dfdyp_ptr(:,:) = 0.0d0
         nb_jac = nb_jac + 1
         cic_nb_jac_0 = cic_nb_jac_0 + 1

         call mf_restore_fpe( )
         call jac( t0, y0, yp0, dfdyp_ptr, cj, neq )
         call mf_save_and_disable_fpe( )

         call msAssign( dfdyp, dfdyp - dfdy )

      end do

      flag0 = -4 ! convergence failure

 99   continue

      call cpu_time( cic_cpu_time_0 )
      cic_cpu_time_0 = cic_cpu_time_0 - cic_cpu_time_init

      if( sls_daecic_call_nb >= 1 ) then
         if( .not. free_y_full ) then
            call msSilentRelease( dfdy2 )
            deallocate( dfdy2 )
         end if
         if( .not. free_yp_full ) then
            call msSilentRelease( dfdyp2 )
            deallocate( dfdyp2 )
         end if
      end if

      if( associated(dfdy_ptr) ) then
         call msFreePointer( dfdy, dfdy_ptr )
      end if
      if( associated(dfdyp_ptr) ) then
         call msFreePointer( dfdyp, dfdyp_ptr )
      end if
      call msSilentRelease( dfdy, dfdyp, dy, dyp, free_y, free_yp )

#endif
   end subroutine dae_cic_dense
!_______________________________________________________________________
!
   subroutine sls_dense_A( res, dfdyp, neq, free,                       &
                           dyp, flag )

      real(kind=MF_DOUBLE), intent(in)  :: res(:)
      type(mfArray)                     :: dfdyp
      type(mfArray)                     :: dyp
      integer,              intent(in)  :: neq, free
      integer,              intent(out) :: flag
      !------ API end ------
#ifdef _DEVLP

      ! ---- part A ----
      ! Solve: 0 = res + dfdyp*dyp
      !
      ! A solution is obtained with as many components as possible
      ! of (transformed) dyp set to zero.

      ! Dense case:
      ! (from 2.6.0, msQR returns a permutation vector)
      !  if A*p = Q*R then solution of A*x = b is
      !    x = p*(R\(Q'*b))                             [MATLAB syntax]
      !    x = RowPerm( R\(Q'*b), inv(p) )              [MUESLI syntax]

      ! returns flag = 0 : normal exit
      !               -2 : bad number of fixed components
      !               -3 : DAE index perhaps greater than one

      integer :: rank, rankdef
      type(mfArray) :: Q, R, p, d

   !------ end of declarations -- execution starts hereafter  ------

      flag = 0 ! normal return

      call rrqr( dfdyp, rank, Q, R, p )
      rankdef = neq - rank
      if( rankdef > 0 ) then
         if( rankdef <= free ) then
            write(STDERR,"(/,2X,A)")    "ERROR: bad number of prescribed components!"
            write(STDERR,"(2X,A,I0,A)") "       advice: Try freeing ", rankdef, " fixed components."
            write(STDERR,"(2X,A)")      "(For a DAE system having " // achar(27) // &
                                        "[3mNd" // achar(27) // "[0m differential equations and " &
                                        // achar(27) // "[3mNa" // achar(27) // "[0m algebraic equations,"
            write(STDERR,"(2X,A)")      " you can only prescribed at most the " // achar(27) // &
                                        "[3mNd" // achar(27) // "[0m components of y0_ind corresponding to"
            write(STDERR,"(2X,A)")      " the differential equations; in all cases the " // achar(27) // &
                                        "[3mNa" // achar(27) // "[0m components of y0_ind corresp-"
            write(STDERR,"(2X,A)")      " onding to the algebraic equations must be left free.)"
            flag = -2
         else
            write(STDERR,"(/,2X,A)") "ERROR: Index of your DAE system may be greater than one."
            flag = -3
         end if
         go to 99
      end if
      call msAssign( d, - .t. mfMul(mf(res),Q) )
      call msAssign( dyp, mfRowPerm(mfLDiv(R,d),mfInvPerm(p)) )

 99   continue

      call msSilentRelease( Q, R, p, d )

#endif
   end subroutine sls_dense_A
!_______________________________________________________________________
!
   subroutine sls_dense_B( res, dfdy, neq, free,                        &
                           dy, flag )

      real(kind=MF_DOUBLE), intent(in)  :: res(:)
      type(mfArray)                     :: dfdy
      type(mfArray)                     :: dy
      integer,              intent(in)  :: neq, free
      integer,              intent(out) :: flag
      !------ API end ------
#ifdef _DEVLP

      ! ---- part B ----
      ! Solve: 0 = res + dfdy*dy
      !
      ! A solution is obtained with as many components as possible
      ! of (transformed) dy set to zero.

      ! Dense case:
      ! (from 2.6.0, msQR returns a permutation vector)
      !  if A*p = Q*R then solution of A*x = b is
      !    x = p*(R\(Q'*b))                             [MATLAB syntax]
      !    x = RowPerm( R\(Q'*b), inv(p) )              [MUESLI syntax]

      ! returns flag = 0 : normal exit
      !               -2 : bad number of fixed components
      !               -3 : DAE index perhaps greater than one

      integer :: rank, rankdef
      type(mfArray) :: Q, R, p, d

   !------ end of declarations -- execution starts hereafter  ------

      flag = 0 ! normal return

      call rrqr( dfdy, rank, Q, R, p )
      rankdef = neq - rank
      if( rankdef > 0 ) then
         if( rankdef <= free ) then
            write(STDERR,"(2X,A)")      "ERROR: bad number of prescribed components!"
            write(STDERR,"(2X,A,I0,A)") "       advice: Try freeing ", rankdef, " fixed components."
            write(STDERR,"(2X,A)")      "(For a DAE system having " // achar(27) // &
                                        "[3mNd" // achar(27) // "[0m differential equations and " &
                                        // achar(27) // "[3mNa" // achar(27) // "[0m algebraic equations,"
            write(STDERR,"(2X,A)")      " you can only prescribed at most the " // achar(27) // &
                                        "[3mNd" // achar(27) // "[0m components of y0_ind corresponding to"
            write(STDERR,"(2X,A)")      " the differential equations; in all cases the " // achar(27) // &
                                        "[3mNa" // achar(27) // "[0m components of y0_ind corresp-"
            write(STDERR,"(2X,A)")      " onding to the algebraic equations must be left free.)"
            flag = -2
         else
            write(STDERR,*) " ERROR: Index of your DAE system may be greater than one."
            flag = -3
         end if
         go to 99
      end if
      call msAssign( d, - .t. mfMul(mf(res),Q) )
      call msAssign( dy, mfRowPerm(mfLDiv(R,d),mfInvPerm(p)) )

 99   continue

      call msSilentRelease( Q, R, p, d )

#endif
   end subroutine sls_dense_B
!_______________________________________________________________________
!
   subroutine sls_dense_C( res, dfdy, dfdyp, neq, free_y, free_yp,      &
                           dy, dyp, flag )

      real(kind=MF_DOUBLE), intent(in) :: res(:)
      type(mfArray)                    :: dfdy, dfdyp
      type(mfArray)                    :: dy, dyp, free_y, free_yp
      integer,              intent(in) :: neq
      integer,             intent(out) :: flag
      !------ API end ------
#ifdef _DEVLP

      ! ---- part C ----
      ! Solve the underdetermined system
      !                0 = res + dfdyp*dyp + dfdy*dy
      ! A solution is obtained with as many components as possible
      ! of (transformed) dy and dyp set to zero.
      !
      ! General case where prescribed components come from both y0 and yp_0

      ! Dense case:
      ! (from 2.6.0, msQR returns a permutation vector)
      !  if A*p = Q*R then solution of A*x = b is
      !    x = p*(R\(Q'*b))                             [MATLAB syntax]
      !    x = RowPerm( R\(Q'*b), inv(p) )              [MUESLI syntax]

      ! returns flag = 0 : normal exit
      !               -2 : bad number of fixed components
      !               -3 : DAE index perhaps greater than one

      integer :: free, rank, rankdef, Srank, nfree_yp
      type(mfArray) :: Q, R, p, d, S, w, w1p, wp

   !------ end of declarations -- execution starts hereafter  ------

      flag = 0 ! normal return
      free = size(free_y) + size(free_yp)

      call rrqr( dfdyp, rank, Q, R, p )
      call msAssign( d, - .t. mfMul(mf(res),Q) )
      if( rank == neq ) then
         call msAssign( w, mfZeros(neq,1) )
         call msAssign( wp, mfRowPerm(mfLDiv(R,d),mfInvPerm(p)) )
      else ! rank deficient
         call msAssign( S, mfMul(Q,dfdy,transp=1) )
         call msAssign( w, mfGet(S,rank+1.to.MF_END,MF_ALL) )
         call rrqr( w, Srank )
         rankdef = neq - (rank + Srank )
         if( rankdef > 0 ) then
            if( rankdef <= free ) then
               write(STDERR,"(2X,A)")      "ERROR: bad number of prescribed components!"
               write(STDERR,"(2X,A,I0,A)") "       advice: Try freeing ", rankdef, " fixed components."
            write(STDERR,"(2X,A)")      "(For a DAE system having " // achar(27) // &
                                        "[3mNd" // achar(27) // "[0m differential equations and " &
                                        // achar(27) // "[3mNa" // achar(27) // "[0m algebraic equations,"
            write(STDERR,"(2X,A)")      " you can only prescribed at most the " // achar(27) // &
                                        "[3mNd" // achar(27) // "[0m components of y0_ind corresponding to"
            write(STDERR,"(2X,A)")      " the differential equations; in all cases the " // achar(27) // &
                                        "[3mNa" // achar(27) // "[0m components of y0_ind corresp-"
            write(STDERR,"(2X,A)")      " onding to the algebraic equations must be left free.)"
               flag = -2
            else
               write(STDERR,*) " ERROR: Index of your DAE system may be greater than one."
               flag = -3
            end if
            go to 99
         end if
         call msAssign( w, mfLDiv( w, mfGet(d,rank+1.to.MF_END) ) )
         call msAssign( w1p, mfLDiv( mfGet(R,1.to.rank,1.to.rank),      &
                             mfGet(d,1.to.rank) - mfMul(mfGet(S,1.to.rank,MF_ALL),w) ) )
         nfree_yp = size(free_yp)
         call msAssign( wp, mfRowPerm(w1p.vc.mfZeros(nfree_yp-rank,1),mfInvPerm(p)) )
      end if

      call msSet( w, dy, free_y )
      ! dy must be a column vector
      if( dy%shape(1) == 1 ) then
         call msAssign( dy, .t. dy )
      end if

       call msSet( wp, dyp, free_yp )
      ! dyp must be a column vector
      if( dyp%shape(1) == 1 ) then
         call msAssign( dyp, .t. dyp )
      end if

 99   continue

      call msSilentRelease( Q, R, p, d, S )
      call msSilentRelease( w, w1p, wp )

#endif
   end subroutine sls_dense_C
!_______________________________________________________________________
!
   subroutine rrqr( A, rank, Q, R, p )

      type(mfArray), intent(in)       :: A
      integer,       intent(out)      :: rank
      type(mfArray), target, optional :: Q, R, p
      !------ API end ------
#ifdef _DEVLP

      ! valid for any matrix A(m,n) [dense/sparse; square/rect], with: m >= n

      type(mfArray), pointer :: QQ, RR, pp
      type(mfArray) :: rrqr_rank

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsScalar(A) ) then
         if( present(Q) ) then
            write(STDERR,*) "(MUESLI dae_cic/rrqr:) internal error:"
            write(STDERR,*) "   for a scalar, Q, R, p are not yet returned!"
            write(STDERR,*) "   (case not implemented)"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
         if( mfDble(mfNorm(A,1)) == 0.0d0 ) then
            rank = 0
         else
            rank = 1
         end if
         go to 99
      end if

      if( present(Q) ) then
         if( .not. present(R) .or. .not. present(p) ) then
            write(STDERR,*) "(MUESLI dae_cic/rrqr:) internal error:"
            write(STDERR,*) "   when Q is present, R and p must be also present!"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
         QQ => Q
         RR => R
         pp => p
      else
         allocate( QQ, RR, pp ) ! no_mem_trace !
      end if

      ! requesting full size factors R and Q is very important when neither
      ! free_y_full nor free_yp_full is true (mixed prescribed components).
      call msQR( mfOut(QQ,RR,pp,rrqr_rank), A, fullQ=.true. )
      rank = rrqr_rank

      if( .not. present(Q) ) then
         call msRelease(QQ,RR,pp)
         deallocate( QQ, RR, pp ) ! no_mem_trace !
      end if
      call msRelease(rrqr_rank)

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end subroutine rrqr
