! f90 include file

! Computes consistent Initial Conditions for the DAE solver.
! (cf. docs/numerics/cic.pdf)
!
! Comes from the 'decic' MATLAB-7.6 script (1.1.6.4, 2005/03/07)
!                             (idem up to MATLAB-7.11, at least)
! (Matlab-7.12 introduces only changes in messages' process)

! For dense  Jacobian matrix : 'dae_cic_dense'
!     banded    "       "    : 'dae_cic_band'
!     sparse    "       "    : 'dae_cic_sparse'

!_______________________________________________________________________
!
   subroutine dae_cic_band( t0, y0, y0_ind, yp0, yp0_ind,               &
                            resid, jac, band, flag0 )

      real(kind=MF_DOUBLE), intent(in)  :: t0
      real(kind=MF_DOUBLE)              :: y0(:), yp0(:)
      type(mfArray)                     :: y0_ind, yp0_ind, band
      integer,              intent(out) :: flag0

      interface
         subroutine resid( t, y, yprime, delta, flag )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)     :: t, y(*), yprime(*)
            real(kind=MF_DOUBLE), intent(out)    :: delta(*)
            integer,              intent(in out) :: flag
         end subroutine resid
      end interface

      ! effective shape for 'jacobian' matrix : [ldjac,nrow]
      interface
         subroutine jac( t, y, yprime, jacobian, cj, ldjac )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in)  :: t, y(*), yprime(*), cj
            integer,              intent(in)  :: ldjac
            real(kind=MF_DOUBLE), intent(out) :: jacobian(ldjac,*)
         end subroutine jac
      end interface
      !------ API end ------
#ifdef _DEVLP

      ! limitation: this specific routine works with Jacobian matrix
      !             in banded format;
      !         so: only '{mf|ms}DaeSolve_JacUser' can call this routine,
      !             when 'band' is used.

      ! Due to difficulties in applying the Matlab algorithm for banded
      ! jacobian (Rank Revealing QR should be used), this latter matrix is
      ! converted to sparse storage and most of computations in this routine
      ! comes from 'dae_cic_sparse'.

      ! tolerance used to control convergence is the default one:
      ! DAE_rtol_def divided by 1000 (as in decic of MATLAB)

      ! returns flag = 0 : normal exit
      !               -1 : pb in 'resid' call
      !               -2 : bad number of fixed components
      !               -3 : DAE index perhaps greater than one
      !               -4 : convergence failure
      !               -5 : illegal call (y0_ind and yp0_ind must
      !                    be either empty, or of size = neq)

      integer :: flag, neq, i, j, k, counter, chord, nnz, nnz2

      ! shape will be: [ldjac,neq] but logical shape is [neq,neq]
      real(kind=MF_DOUBLE), allocatable :: dfdy_band(:,:), dfdyp_band(:,:)

      type(mfArray) :: dfdy, dfdyp, dy, dyp, free_y, free_yp
      real(kind=MF_DOUBLE), allocatable :: res(:)
      real(kind=MF_DOUBLE) :: cj, resnorm0, resnorm, nrmv, nrmdv,       &
                              factor, threshold

      ! communication with some SLATEC routines
      real(kind=MF_DOUBLE) :: dt_min, dt_max
      integer :: nb_step, nb_resid
      common /slatec_daesolve_1/ dt_min, dt_max, nb_step, nb_resid

      integer :: nb_jac, nb_solv
      common /slatec_daesolve_2/ nb_jac, nb_solv

      real(kind=MF_DOUBLE) :: cic_cpu_time_0
      integer :: cic_nb_resid_0, cic_nb_jac_0, cic_nb_solv_0
      common /dae_cic_1/ cic_cpu_time_0, cic_nb_resid_0,                &
                         cic_nb_jac_0, cic_nb_solv_0

      real(kind=MF_DOUBLE) :: cic_cpu_time_init

      integer :: i_eqn_group, i_group_eqn, i_var_group, i_group_var

      integer :: ldjac, ML, MU, bandwidth

   !------ end of declarations -- execution starts hereafter  ------

      call cpu_time( cic_cpu_time_init )

      ML = mfGet( band, 1 )
      MU = mfGet( band, 2 )
      ldjac = 2*ML + MU + 1
      bandwidth = ml + 1 + mu

      flag0 = 0 ! normal return

      neq = size(y0)

      if( size(y0_ind) /= 0 .and. size(y0_ind) /= neq ) then
         write(STDERR,*) "ERROR: Illegal call: y0_ind must be either empty,"
         write(STDERR,*) "       or of size equal to the number of equations!"
         flag0 = -5
         go to 99
      end if
      if( size(yp0_ind) /= 0 .and. size(yp0_ind) /= neq ) then
         write(STDERR,*) "ERROR: Illegal call: yp0_ind must be either empty,"
         write(STDERR,*) "       or of size equal to the number of equations!"
         flag0 = -5
         go to 99
      end if

      if( mfIsEmpty(y0_ind) ) then
         free_y = [ (i, i = 1, neq) ]
         free_y_full = .true.
      else
         call msAssign( free_y, mfFind( y0_ind == 0.0d0 ) )
         if( size(free_y) == neq ) then
            free_y_full = .true.
         else
            free_y_full = .false.
         end if
      end if
      if( mfIsEmpty(yp0_ind) ) then
         free_yp = [ (i, i = 1, neq) ]
         free_yp_full = .true.
      else
         call msAssign( free_yp, mfFind( yp0_ind == 0.0d0 ) )
         if( size(free_yp) == neq ) then
            free_yp_full = .true.
         else
            free_yp_full = .false.
         end if
      end if
      if( size(free_y)+size(free_yp) < neq ) then
         write(STDERR,"(1X,A,I0,A)") "ERROR: You cannot specify more than ", neq, " components"
         write(STDERR,"(1X,A)")      "       of y0 and/or yp0!"
         flag0 = -2
         go to 99
      end if

      allocate( res(neq) )

      flag = 0
      nb_resid = nb_resid + 1
      cic_nb_resid_0 = cic_nb_resid_0 + 1

      call mf_restore_fpe( )
      call resid( t0, y0, yp0, res, flag )
      call mf_save_and_disable_fpe( )

      if( MF_NUMERICAL_CHECK ) then
         ! res vector must not contain any NaN value
         do i = 1, neq
            if( isnan(res(i)) ) then
               print "(/,A)", "(MUESLI Daesolve:) [C.I.C.] ERROR"
               print "(20X,A)", "A NaN value has been found after calling the", &
                                "user-supplied RESID routine."
               print "(20X,A,I0)", "This occured in delta(i) for i = ", i
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,         &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",              &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
               end if
               flag0 = -7
               go to 99
            end if
         end do
      endif

      if( flag /= 0 ) then
         flag0 = -1 ! pb in 'resid' call
         go to 99
      end if
      resnorm0 = mfNorm( mf(res) )
      ! update of resnorm0 for two reasons:
      !   first, increase the value of resnorm0 to be sure that
      !          new iterate values will be slightly less;
      !   second, to avoid too small initial value, choose a threshold
      !          value.
      ! FSF_G95 requires a greater increase for resnorm,
      ! due to roundoff errors during the call of 'resid'!
      ! The factor 1.2d0 should be ok for other compilers, but for the
      ! the former compilers, the factor 7.d0 is absolutely required!
!### TODO 2: maintenant que je n'utilise plus FSF_G95, on pourrait baisser
!          la valeur du facteur à 1.2d0
      if( resnorm0 < sqrt(MF_EPS) ) resnorm0 = 12.0d0*resnorm0
      resnorm0 = max( resnorm0, 1.0d3*MF_EPS )

      allocate( dfdy_band(ldjac,neq), dfdyp_band(ldjac,neq) )

      ! initialize the partial derivatives
      cj = 0.0d0
      nb_jac = nb_jac + 1
      cic_nb_jac_0 = cic_nb_jac_0 + 1
      dfdy_band = 0.0d0

      call mf_restore_fpe( )
      call jac( t0, y0, yp0, dfdy_band, cj, ldjac )
      call mf_save_and_disable_fpe( )

      cj = 1.0d0
      nb_jac = nb_jac + 1
      cic_nb_jac_0 = cic_nb_jac_0 + 1
      dfdyp_band = 0.0d0

      call mf_restore_fpe( )
      call jac( t0, y0, yp0, dfdyp_band, cj, ldjac )
      call mf_save_and_disable_fpe( )

      dfdyp_band = dfdyp_band - dfdy_band

      ! Convert Compressed Banded 'dfdy' to CSC storage.
      ! (the following allocation overestimates the true value; doesn't matter)
      nnz = neq*bandwidth
      call msAssign( dfdy, mfSpAlloc(neq,neq,nnz) )
      call sqbanded2csc( neq, ml, mu, dfdy_band,                        &
                         nnz, dfdy%a, dfdy%i, dfdy%j )

      if( MF_NUMERICAL_CHECK ) then
         ! check the validity of NNZ... (avoid integer overflow)
         if( nnz < 1 .or. dble(neq)**2 < dble(nnz) ) then
            write(STDERR,"(1X,A)")    "ERROR: after calling 'jac', the conversion of banded 'dfdy' to sparse"
            write(STDERR,"(1X,A)")    "       storage returned an invalid nnz value!"
            write(STDERR,"(1X,A,I0)") "       nnz = ", nnz
            flag0 = -6
            go to 99
         end if
      endif
      dfdy%row_sorted = TRUE

      ! Convert Compressed Banded 'dfdyp' to CSC storage.
      ! (Due to roundoff errors, some elements of dfdyp_band may be close
      !  to zero but have not been eliminated after the substraction...)
      threshold = MF_EPS * maxval( abs( dfdyp_band ) )
      call sqbanded2csc_nnz( neq, ml, mu, dfdyp_band, nnz2, threshold )
      call msAssign( dfdyp, mfSpAlloc(neq,neq,nnz2) )
      call sqbanded2csc( neq, ml, mu, dfdyp_band,                       &
                         nnz2, dfdyp%a, dfdyp%i, dfdyp%j, threshold )

      if( MF_NUMERICAL_CHECK ) then
         ! check the validity of NNZ2... (avoid integer overflow)
         if( nnz2 < 1 .or. dble(neq)**2 < dble(nnz2) ) then
            write(STDERR,"(1X,A)")    "ERROR: after calling 'jac', the conversion of banded 'dfdyp' to sparse"
            write(STDERR,"(1X,A)")    "       storage returned an invalid nnz value!"
            write(STDERR,"(1X,A,I0)") "       nnz = ", nnz2
            flag0 = -6
            go to 99
         end if
      endif
      dfdyp%row_sorted = TRUE

      if( MF_NUMERICAL_CHECK ) then
         ! check the validity of the CSC vectors... but only for dfdy
         ! (we can suppose that dfdyp has a similar structure)
         ! 1) indices in i(:) must be in [1,neq]
         if( minval( dfdy%i(1:nnz) ) < 1 ) then
            write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
            write(STDERR,"(1X,A)") "       (at least one index in 'ipd' is < 1)"
            flag0 = -6
            go to 99
         end if
         if( maxval( dfdy%i(1:nnz) ) > neq ) then
            write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
            write(STDERR,"(1X,A)") "       (at least one index in 'ipd' is > neq)"
            flag0 = -6
            go to 99
         end if
         ! 2) indices in j(:) must be in increasing order
         do k = 2, neq+1
            if( dfdy%j(k) < dfdy%j(k-1) ) then
               write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
               write(STDERR,"(1X,A)") "       ('jpd' must have increasing indices)"
               flag0 = -6
               go to 99
            end if
         end do
         ! 3) indices in j(:) must be in [1,nnz]
         do k = 1, neq
            if( dfdy%j(k) < 1 .or. nnz < dfdy%j(k) ) then
               write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
               write(STDERR,"(1X,A)") "       ('jpd' must be in [1,nnz])"
               flag0 = -6
               go to 99
            end if
         end do
         ! 4) last index in j(:) must be exactly nnz+1
         if( dfdy%j(neq+1) /= nnz+1 ) then
            write(STDERR,"(1X,A)") "ERROR: invalid CSC structure returned by 'jac'"
            write(STDERR,"(1X,A)") "       (last value of 'jpd' must be equal to nnz+1)"
            flag0 = -6
            go to 99
         end if
         ! 5) all entries must be valid (not NaN)
         do k = 1, nnz
            if( isnan(dfdy%a(k)) ) then
               write(STDERR,"(1X,A)") "ERROR: 'jac' returned NaN value(s)"
               i = dfdy%i(k)
               ! find column
               do j = 1, neq-1
                  if( k < dfdy%j(j+1) ) exit
               end do
               print "(20X,A,I0,1X,I0)", "This occured in matrix for (i,j) = ", i, " ", j
               if( NAMED_EQN_PRESENCE ) then
                  call search_index_in_eqn_groups( i, i_eqn_group,      &
                                                      i_group_eqn )
                  print "(/,20X,A,A)", "Named equation is: ",           &
                                       trim(NAMED_EQN_PTR(i_eqn_group)%name)
                  print "(20X,A,I0)", "Equation number is: ", i_group_eqn
                  if( NAMED_VAR_PRESENCE ) then
                     call search_index_in_var_groups( k, i_var_group,   &
                                                         i_group_var )
                     print "(/,20X,A,A)", "Named variable is: ",        &
                                          trim(NAMED_EQN_PTR(i_var_group)%name)
                     print "(20X,A,I0)", "Variable number is: ", i_group_var
                  end if
               end if
               flag0 = -8
               go to 99
            end if
         end do
      endif

      do counter = 1, 10

         sls_daecic_call_nb = 0

         do chord = 1, 2 ! orig. 3

            nb_solv = nb_solv + 1
            cic_nb_solv_0 = cic_nb_solv_0 + 1

            ! dy and dyp must be vectors of zeros before calling sls_sparse
            dy = mfZeros(neq,1)
            dyp = mfZeros(neq,1)

            call sls_sparse( res, dfdy, dfdyp, neq, free_y, free_yp,    &
                             dy, dyp, flag )
            if( flag /= 0 ) then
               flag0 = flag ! pb in 'sls' (-2 or -3)
               go to 99
            end if

            ! if the increments are too big, limit the change in norm
            ! to a factor of 2 -- trust region
            nrmv = max( mfDble(mfNorm(y0.vc.yp0)), DAE_atol_def )
            nrmdv = mfDble( mfNorm(dy.vc.dyp) )
            if( nrmdv > 2.0d0*nrmv ) then
               factor = 2.0d0*nrmv/nrmdv
               call msAssign( dy, factor*dy )
               call msAssign( dyp, factor*dyp )
               nrmdv = factor*nrmdv
            end if
            y0 = .t.mf(y0) + dy
            yp0 = .t.mf(yp0) + dyp
            flag = 0
            nb_resid = nb_resid + 1
            cic_nb_resid_0 = cic_nb_resid_0 + 1

            call mf_restore_fpe( )
            call resid( t0, y0, yp0, res, flag )
            call mf_save_and_disable_fpe( )

            if( flag /= 0 ) then
               flag0 = -1 ! pb in 'resid' call
               go to 99
            end if
            resnorm = mfNorm( mf(res) )

            ! Test for convergence. The norm of the residual must be
            ! no greater than the initial guess and the norm of the
            ! increments must be small in a relative sense.
            if( resnorm <= resnorm0 .and.                               &
               nrmdv <= 1.0d-3*DAE_rtol_def*nrmv ) then
               go to 99
            end if

         end do

         cj = 0.0d0
         nb_jac = nb_jac + 1
         cic_nb_jac_0 = cic_nb_jac_0 + 1
         dfdy_band = 0.0d0

         call mf_restore_fpe( )
         call jac( t0, y0, yp0, dfdy_band, cj, ldjac )
         call mf_save_and_disable_fpe( )

         cj = 1.0d0
         nb_jac = nb_jac + 1
         cic_nb_jac_0 = cic_nb_jac_0 + 1
         dfdyp_band = 0.0d0

         call mf_restore_fpe( )
         call jac( t0, y0, yp0, dfdyp_band, cj, ldjac )
         call mf_save_and_disable_fpe( )

         dfdyp_band = dfdyp_band - dfdy_band

         ! Convert Compressed Banded 'dfdy' to CSC storage.
         call sqbanded2csc( neq, ml, mu, dfdy_band,                     &
                            nnz, dfdy%a, dfdy%i, dfdy%j )
         ! (dfdy already tagged as row_sorted)

         ! Convert Compressed Banded 'dfdyp' to CSC storage.
         call sqbanded2csc( neq, ml, mu, dfdyp_band,                    &
                            nnz2, dfdyp%a, dfdyp%i, dfdyp%j, threshold )
         ! (dfdyp already tagged as row_sorted)

      end do

      flag0 = -4 ! convergence failure

 99   continue

      call cpu_time( cic_cpu_time_0 )
      cic_cpu_time_0 = cic_cpu_time_0 - cic_cpu_time_init

      call msSilentRelease( dfdy, dfdyp, dy, dyp, free_y, free_yp )

#endif
   end subroutine dae_cic_band
