! f90 include file

! Internally, the routines 'dqag' and 'dqags' of the SLATEC library
! are used.

!_______________________________________________________________________
!
   function mfQuad( fun, a, b, abs_tol, rel_tol, sing ) result( out )

      interface
         function fun( x ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      real(kind=MF_DOUBLE), intent(in)           :: a, b
      real(kind=MF_DOUBLE), intent(in), optional :: abs_tol, rel_tol
      logical,              intent(in), optional :: sing
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer, parameter :: limit = 100, key = 1, lenw = 4*limit
      real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw), res,  &
                              bound
      integer :: neval, ier, last, iwork(limit), inf_range
      logical :: singular

      real(kind=MF_DOUBLE), parameter :: DEFAULT_TOL = 1.0d-12

      character(len=*), parameter :: ROUTINE_NAME = "mfQuad"

   !------ end of declarations -- execution starts hereafter  ------

      if( isnan(a) .or. isnan(b) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'a' or 'b' argument cannot contain NaN!" )
         return
      end if

      if( a == MF_INF ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'a' cannot be +Inf, only -Inf or a finite value." )
         return
      else if( a == -MF_INF ) then
         if( b == MF_INF ) then
            inf_range = 2
         else if( b == -MF_INF ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'b' cannot be -Inf, only +Inf or a finite value." )
            return
         else
            inf_range = -1
         end if
      else
         if( b == MF_INF ) then
            inf_range = 1
            bound = a
         else if( b == -MF_INF ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'b' cannot be -Inf, only +Inf or a finite value." )
            return
         else
            inf_range = 0
         end if
      end if

      if( present(abs_tol) ) then
         if( abs_tol < 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'abs_tol' argument cannot be negative!" )
            return
         end if
         epsabs = abs_tol
      else
         epsabs = DEFAULT_TOL
      end if

      if( present(rel_tol) ) then
         if( rel_tol <= 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'rel_tol' argument must be strictly positive!" )
            return
         end if
         epsrel = rel_tol
         if( epsrel < DEFAULT_TOL ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "'rel_tol' argument too small",          &
                               "=> 'rel_tol' has been set to the default tolerance" )
            epsrel = DEFAULT_TOL
         end if
      else
         epsrel = DEFAULT_TOL
      end if

      if( present(sing) ) then
         if( inf_range /= 0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "the 'sing' argument cannot be present when the integral", &
                               "involves an infinite range!" )
            return
         end if
         singular = sing
      else
         singular = .false.
      end if

      if( singular ) then
         call dqags( fun, a, b, epsabs, epsrel, res,                    &
                     abserr, neval, ier,                                &
                     limit, lenw, last, iwork, work )
      else
         if( inf_range == 0 ) then
            call dqag( fun, a, b, epsabs, epsrel, key, res,             &
                       abserr, neval, ier,                              &
                       limit, lenw, last, iwork, work )
         else ! inf_range = 1, -1 or 2
            call dqagi( fun, bound, inf_range, epsabs, epsrel, res,     &
                        abserr, neval, ier,                             &
                        limit, lenw, last, iwork, work )
         end if
      end if
      ier = -ier

      if( ier /= 0 ) then
         ! invalid input
         if( ier == -6 ) then
            write(STDERR,*) "(MUESLI mfQuad:) internal error:"
            write(STDERR,*) "                 invalid input"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
         ! abnormal return
         res = MF_NAN
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "the method used didn't converge to the required precision!" )
      end if

      out = res

      out%status_temporary = .true.

#endif
   end function mfQuad
!_______________________________________________________________________
!
   subroutine msQuad( out, fun, a, b, abs_tol, rel_tol, sing )

      interface
         function fun( x ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      real(kind=MF_DOUBLE), intent(in)           :: a, b
      real(kind=MF_DOUBLE), intent(in), optional :: abs_tol, rel_tol
      logical,              intent(in), optional :: sing
      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      integer, parameter :: limit = 100, key = 1, lenw = 4*limit
      real(kind=MF_DOUBLE) :: epsabs, epsrel, qag_abserr, work(lenw),   &
                              res, bound
      integer :: qag_neval, qag_ier, last, iwork(limit), inf_range
      logical :: singular
      type(mfArray), pointer :: q => null(), abserr => null(),          &
                                neval => null(), status => null()
      logical :: no_pause

      real(kind=MF_DOUBLE), parameter :: DEFAULT_TOL = 1.0d-12

      character(len=*), parameter :: ROUTINE_NAME = "msQuad"

   !------ end of declarations -- execution starts hereafter  ------

      if( isnan(a) .or. isnan(b) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'a' or 'b' argument cannot contain NaN!" )
         return
      end if

      if( a == MF_INF ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'a' cannot be +Inf, only -Inf or a finite value." )
         return
      else if( a == -MF_INF ) then
         if( b == MF_INF ) then
            inf_range = 2
         else if( b == -MF_INF ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'b' cannot be -Inf, only +Inf or a finite value." )
            return
         else
            inf_range = -1
         end if
      else
         if( b == MF_INF ) then
            inf_range = 1
            bound = a
         else if( b == -MF_INF ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'b' cannot be -Inf, only +Inf or a finite value." )
            return
         else
            inf_range = 0
         end if
      end if

      call mf_save_and_disable_fpe( )

      ! 2 to 4 out-args must be specified
      ! 2 are optional, only the first can be equal to MF_NO_ARG
      if( out%n < 2 .or. 4 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to four output args required!",        &
                            "syntax is: call msQuad( mfOut(<out_arg>), fun, a, b [, abs_tol, rel_tol, sing] )", &
                            "    with <out_arg> equal to {q,abserr[,status,neval]}", &
                            "",                                         &
                            "Moreover, the output arg 'status' has not been set, because it may be not present!" )
         go to 99
      end if

      q => out%ptr1
      call msSilentRelease( q )
      abserr => out%ptr2
      call msSilentRelease( abserr )
      ! optional args
      if( out%arg_present(3) ) then
         status => out%ptr3
         call msSilentRelease( status )
         no_pause = .true.
      else
         allocate( status )
         no_pause = .false.
      end if
      if( out%arg_present(4) ) then
         neval => out%ptr4
         call msSilentRelease( neval )
      else
      end if

      q = MF_NAN

      if( present(abs_tol) ) then
         if( abs_tol < 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'abs_tol' argument cannot be negative!" )
            return
         end if
         epsabs = abs_tol
      else
         epsabs = DEFAULT_TOL
      end if

      if( present(rel_tol) ) then
         if( rel_tol <= 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'rel_tol' argument must be strictly positive!" )
            status = -6
            go to 99
         end if
         epsrel = rel_tol
         if( epsrel < DEFAULT_TOL ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "'rel_tol' argument too small!",         &
                               "=> 'rel_tol' has been set to the default tolerance" )
            epsrel = DEFAULT_TOL
         end if
      else
         epsrel = DEFAULT_TOL
      end if

      if( present(sing) ) then
         if( inf_range /= 0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "the 'sing' argument cannot be present when the integral", &
                               "involves an infinite range!" )
            return
         end if
         singular = sing
      else
         singular = .false.
      end if

      if( singular ) then
         call dqags( fun, a, b, epsabs, epsrel, res,                    &
                     qag_abserr, qag_neval, qag_ier,                    &
                     limit, lenw, last, iwork, work )
      else
         if( inf_range == 0 ) then
            call dqag( fun, a, b, epsabs, epsrel, key, res,             &
                       qag_abserr, qag_neval, qag_ier,                  &
                       limit, lenw, last, iwork, work )
         else ! inf_range = 1, -1 or 2
            call dqagi( fun, bound, inf_range, epsabs, epsrel, res,     &
                        qag_abserr, qag_neval, qag_ier,                 &
                        limit, lenw, last, iwork, work )
         end if
      end if
      qag_ier = -qag_ier

      if( qag_ier /= 0 ) then
         ! abnormal return
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "the method used didn't converge to the required precision!" )
      end if

      q = res

      abserr = qag_abserr

      if( out%arg_present(3) ) then
         status = qag_ier
      else
         deallocate( status )
      end if

      if( out%arg_present(4) ) then
         neval = qag_neval
      end if

 99   continue

      if( .not. out%arg_present(3) ) then
         if( associated(status) ) then
            deallocate( status )
         end if
      end if

      call mf_restore_fpe( )

#endif
   end subroutine msQuad
!_______________________________________________________________________
!
   function mfDblQuad_cte( fun, xa, xb, ya, yb, abs_tol, rel_tol )      &
   result( out )

      interface
         function fun( x, y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x, y
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      real(kind=MF_DOUBLE), intent(in)           :: xa, xb, ya, yb
      real(kind=MF_DOUBLE), intent(in), optional :: abs_tol, rel_tol
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer, parameter :: limit = 100, key = 1, lenw = 4*limit
      real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw), res
      integer :: neval, ier, last, iwork(limit)
      real(kind=MF_DOUBLE) :: epsabs_2, epsrel_2

      real(kind=MF_DOUBLE), parameter :: DEFAULT_TOL = 1.0d-12

      integer :: fun_inner_ier
      common /slatec_dqag/ fun_inner_ier

      character(len=*), parameter :: ROUTINE_NAME = "mfDblQuad"

   !------ end of declarations -- execution starts hereafter  ------

      if( present(abs_tol) ) then
         if( abs_tol < 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'abs_tol' argument cannot be negative!" )
            return
         end if
         epsabs = abs_tol
      else
         epsabs = DEFAULT_TOL
      end if

      if( present(rel_tol) ) then
         if( rel_tol <= 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'rel_tol' argument must be strictly positive!" )
            return
         end if
         epsrel = rel_tol
         if( epsrel < DEFAULT_TOL ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "'rel_tol' argument too small",          &
                               "=> 'rel_tol' has been set to the default tolerance" )
            epsrel = DEFAULT_TOL
         end if
      else
         epsrel = DEFAULT_TOL
      end if

      fun_inner_ier = 0
      epsabs_2 = epsabs/10.0d0
      epsrel_2 = epsrel/10.0d0
      call dqag( fun_inner, ya, yb, epsabs, epsrel, key, res,           &
                 abserr, neval, ier,                                    &
                 limit, lenw, last, iwork, work )

      if( fun_inner_ier /= 0 ) then
         ! abnormal return
         res = MF_NAN
         go to 99
      end if

      ier = -ier

      if( ier /= 0 ) then
         ! invalid input
         if( ier == -6 ) then
            write(STDERR,*) "(MUESLI mfDblQuad:) internal error:"
            write(STDERR,*) "                    invalid input"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
         ! abnormal return
         res = MF_NAN
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            " during integration along the y variable", &
                            "the method used didn't converge to the required precision!" )
      end if

 99   continue
      out = res

      out%status_temporary = .true.

   contains

      function fun_inner( y ) result( res )

         real(kind=MF_DOUBLE), intent(in) :: y
         real(kind=MF_DOUBLE) :: res

         integer, parameter :: limit = 100, key = 1, lenw = 4*limit
         real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw)
         integer :: neval, ier, last, iwork(limit)

         call dqag_2( fun, y, xa, xb, epsabs_2, epsrel_2, key, res,     &
                      abserr, neval, ier,                               &
                      limit, lenw, last, iwork, work )

         ier = -ier

         if( ier /= 0 ) then
            ! invalid input
            if( ier == -6 ) then
               write(STDERR,*) "(MUESLI mfDblQuad:) fun_inner: internal error:"
               write(STDERR,*) "                    invalid input"
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               stop
            end if
            ! abnormal return
            res = 0 ! not MF_NAN to avoid stopping dqag...
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               " during integration along the x variable", &
                               "the method used didn't converge to the required precision!" )
         end if

         ! this is a sticky variable to record any failure in dqag_2
         ! (avoid resetting fun_inner_ier to zero)
         if( ier /= 0 ) fun_inner_ier = ier

      end function fun_inner

#endif
   end function mfDblQuad_cte
!_______________________________________________________________________
!
   subroutine msDblQuad_cte( out, fun, xa, xb, ya, yb, abs_tol, rel_tol )

      interface
         function fun( x, y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x, y
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      real(kind=MF_DOUBLE), intent(in)           :: xa, xb, ya, yb
      real(kind=MF_DOUBLE), intent(in), optional :: abs_tol, rel_tol
      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      integer, parameter :: limit = 100, key = 1, lenw = 4*limit
      real(kind=MF_DOUBLE) :: epsabs, epsrel, qag_abserr, work(lenw), res
      integer :: qag_neval, qag_ier, last, iwork(limit)
      type(mfArray), pointer :: q => null(), abserr => null(),          &
                                neval => null(), status => null()
      logical :: no_pause
      real(kind=MF_DOUBLE) :: epsabs_2, epsrel_2
      integer :: neval_dummy

      real(kind=MF_DOUBLE), parameter :: DEFAULT_TOL = 1.0d-12

      integer :: fun_inner_ier
      common /slatec_dqag/ fun_inner_ier

      character(len=*), parameter :: ROUTINE_NAME = "msDblQuad"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      ! 2 to 4 out-args must be specified
      ! 2 are optional, only the first can be equal to MF_NO_ARG
      if( out%n < 2 .or. 4 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to four output args required!",       &
                            "syntax is: call msQuad( mfOut(<out_arg>), fun, a, b [, abs_tol, rel_tol, sing] )", &
                            "    with <out_arg> equal to {q,abserr[,status,neval]}", &
                            "",                                         &
                            "Moreover, the output arg 'status' has not been set, because it may be not present!" )
         go to 99
      end if

      q => out%ptr1
      call msSilentRelease( q )
      abserr => out%ptr2
      call msSilentRelease( abserr )
      ! optional args
      if( out%arg_present(3) ) then
         status => out%ptr3
         call msSilentRelease( status )
         no_pause = .true.
      else
         allocate( status )
         no_pause = .false.
      end if
      if( out%arg_present(4) ) then
         neval => out%ptr4
         call msSilentRelease( neval )
      else
      end if

      q = MF_NAN

      if( present(abs_tol) ) then
         if( abs_tol < 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'abs_tol' argument cannot be negative!" )
            return
         end if
         epsabs = abs_tol
      else
         epsabs = DEFAULT_TOL
      end if

      if( present(rel_tol) ) then
         if( rel_tol <= 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'rel_tol' argument must be strictly positive!" )
            status = -6
            go to 99
         end if
         epsrel = rel_tol
         if( epsrel < DEFAULT_TOL ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "'rel_tol' argument too small!",        &
                               "=> 'rel_tol' has been set to the default tolerance" )
            epsrel = DEFAULT_TOL
         end if
      else
         epsrel = DEFAULT_TOL
      end if

      fun_inner_ier = 0
      qag_neval = 0
      epsabs_2 = epsabs/10.0d0
      epsrel_2 = epsrel/10.0d0
      call dqag( fun_inner, ya, yb, epsabs, epsrel, key, res,           &
                 qag_abserr, neval_dummy, qag_ier,                      &
                 limit, lenw, last, iwork, work )

      if( fun_inner_ier /= 0 ) then
         ! abnormal return
         res = MF_NAN
         abserr = MF_NAN
         go to 98
      end if

      qag_ier = -qag_ier

      if( qag_ier /= 0 ) then
         ! abnormal return
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "the method used didn't converge to the required precision!" )
      end if

      abserr = qag_abserr

 98   continue

      q = res

      if( out%arg_present(3) ) then
         status = qag_ier
      else
         deallocate( status )
      end if

      if( out%arg_present(4) ) then
         neval = qag_neval
      end if

 99   continue

      if( .not. out%arg_present(3) ) then
         if( associated(status) ) then
            deallocate( status )
         end if
      end if

      call mf_restore_fpe( )

   contains

      function fun_inner( y ) result( res )

         real(kind=MF_DOUBLE), intent(in) :: y
         real(kind=MF_DOUBLE) :: res

         integer, parameter :: limit = 100, key = 1, lenw = 4*limit
         real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw)
         integer :: neval, ier, last, iwork(limit)

         call dqag_2( fun, y, xa, xb, epsabs_2, epsrel_2, key, res,     &
                      abserr, neval, ier,                               &
                      limit, lenw, last, iwork, work )

         qag_neval = qag_neval + neval
         ier = -ier

         if( ier /= 0 ) then
            ! invalid input
            if( ier == -6 ) then
               write(STDERR,*) "(MUESLI msDblQuad:) fun_inner: internal error:"
               write(STDERR,*) "                    invalid input"
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               stop
            end if
            ! abnormal return
            res = 0 ! not MF_NAN to avoid stopping dqag...
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               " during integration along the x variable", &
                               "the method used didn't converge to the required precision!" )
         end if

         ! this is a sticky variable to record any failure in dqag_2
         ! (avoid resetting fun_inner_ier to zero)
         if( ier /= 0 ) fun_inner_ier = ier

      end function fun_inner

#endif
   end subroutine msDblQuad_cte
!_______________________________________________________________________
!
   function mfDblQuad_funx( fun, fun_xa, fun_xb, ya, yb,                &
                            abs_tol, rel_tol )                          &
   result( out )

      interface
         function fun( x, y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x, y
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      interface
         function fun_xa( y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: y
            real(kind=MF_DOUBLE) :: res
         end function fun_xa
         function fun_xb( y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: y
            real(kind=MF_DOUBLE) :: res
         end function fun_xb
      end interface

      real(kind=MF_DOUBLE), intent(in)           :: ya, yb
      real(kind=MF_DOUBLE), intent(in), optional :: abs_tol, rel_tol
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer, parameter :: limit = 100, key = 1, lenw = 4*limit
      real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw), res
      integer :: neval, ier, last, iwork(limit)
      real(kind=MF_DOUBLE) :: epsabs_2, epsrel_2

      real(kind=MF_DOUBLE), parameter :: DEFAULT_TOL = 1.0d-12

      integer :: fun_inner_ier
      common /slatec_dqag/ fun_inner_ier

      character(len=*), parameter :: ROUTINE_NAME = "mfDblQuad"

   !------ end of declarations -- execution starts hereafter  ------

      if( present(abs_tol) ) then
         if( abs_tol < 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'abs_tol' argument cannot be negative!" )
            return
         end if
         epsabs = abs_tol
      else
         epsabs = DEFAULT_TOL
      end if

      if( present(rel_tol) ) then
         if( rel_tol <= 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'rel_tol' argument must be strictly positive!" )
            return
         end if
         epsrel = rel_tol
         if( epsrel < DEFAULT_TOL ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "'rel_tol' argument too small",          &
                               "=> 'rel_tol' has been set to the default tolerance" )
            epsrel = DEFAULT_TOL
         end if
      else
         epsrel = DEFAULT_TOL
      end if

      fun_inner_ier = 0
      epsabs_2 = epsabs/10.0d0
      epsrel_2 = epsrel/10.0d0
      call dqag( fun_inner, ya, yb, epsabs, epsrel, key, res,           &
                 abserr, neval, ier,                                    &
                 limit, lenw, last, iwork, work )

      if( fun_inner_ier /= 0 ) then
         ! abnormal return
         res = MF_NAN
         go to 99
      end if

      ier = -ier

      if( ier /= 0 ) then
         ! invalid input
         if( ier == -6 ) then
            write(STDERR,*) "(MUESLI mfDblQuad:) internal error:"
            write(STDERR,*) "                    invalid input"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
         ! abnormal return
         res = MF_NAN
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            " during integration along the y variable", &
                            "the method used didn't converge to the required precision!" )
      end if

 99   continue
      out = res

      out%status_temporary = .true.

   contains

      function fun_inner( y ) result( res )

         real(kind=MF_DOUBLE), intent(in) :: y
         real(kind=MF_DOUBLE) :: res

         integer, parameter :: limit = 100, key = 1, lenw = 4*limit
         real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw)
         integer :: neval, ier, last, iwork(limit)

         call dqag_2( fun, y, fun_xa(y), fun_xb(y), epsabs_2, epsrel_2, &
                      key, res, abserr, neval, ier,                     &
                      limit, lenw, last, iwork, work )

         ier = -ier

         if( ier /= 0 ) then
            ! invalid input
            if( ier == -6 ) then
               write(STDERR,*) "(MUESLI mfDblQuad:) fun_inner: internal error:"
               write(STDERR,*) "                    invalid input"
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               stop
            end if
            ! abnormal return
            res = 0 ! not MF_NAN to avoid stopping dqag...
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               " during integration along the x variable", &
                               "the method used didn't converge to the required precision!" )
         end if

         ! this is a sticky variable to record any failure in dqag_2
         ! (avoid resetting fun_inner_ier to zero)
         if( ier /= 0 ) fun_inner_ier = ier

      end function fun_inner

#endif
   end function mfDblQuad_funx
!_______________________________________________________________________
!
   subroutine msDblQuad_funx( out, fun, fun_xa, fun_xb, ya, yb,         &
                              abs_tol, rel_tol )

      interface
         function fun( x, y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x, y
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      interface
         function fun_xa( y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: y
            real(kind=MF_DOUBLE) :: res
         end function fun_xa
         function fun_xb( y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: y
            real(kind=MF_DOUBLE) :: res
         end function fun_xb
      end interface

      real(kind=MF_DOUBLE), intent(in)           :: ya, yb
      real(kind=MF_DOUBLE), intent(in), optional :: abs_tol, rel_tol
      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      integer, parameter :: limit = 100, key = 1, lenw = 4*limit
      real(kind=MF_DOUBLE) :: epsabs, epsrel, qag_abserr, work(lenw), res
      integer :: qag_neval, qag_ier, last, iwork(limit)
      type(mfArray), pointer :: q => null(), abserr => null(),          &
                                neval => null(), status => null()
      logical :: no_pause
      real(kind=MF_DOUBLE) :: epsabs_2, epsrel_2
      integer :: neval_dummy

      real(kind=MF_DOUBLE), parameter :: DEFAULT_TOL = 1.0d-12

      integer :: fun_inner_ier
      common /slatec_dqag/ fun_inner_ier

      character(len=*), parameter :: ROUTINE_NAME = "msDblQuad"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      ! 2 to 4 out-args must be specified
      ! 2 are optional, only the first can be equal to MF_NO_ARG
      if( out%n < 2 .or. 4 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to four output args required!",       &
                            "syntax is: call msQuad( mfOut(<out_arg>), fun, a, b [, abs_tol, rel_tol, sing] )", &
                            "    with <out_arg> equal to {q,abserr[,status,neval]}", &
                            "",                                         &
                            "Moreover, the output arg 'status' has not been set, because it may be not present!" )
         go to 99
      end if

      q => out%ptr1
      call msSilentRelease( q )
      abserr => out%ptr2
      call msSilentRelease( abserr )
      ! optional args
      if( out%arg_present(3) ) then
         status => out%ptr3
         call msSilentRelease( status )
         no_pause = .true.
      else
         allocate( status )
         no_pause = .false.
      end if
      if( out%arg_present(4) ) then
         neval => out%ptr4
         call msSilentRelease( neval )
      else
      end if

      q = MF_NAN

      if( present(abs_tol) ) then
         if( abs_tol < 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'abs_tol' argument cannot be negative!" )
            return
         end if
         epsabs = abs_tol
      else
         epsabs = DEFAULT_TOL
      end if

      if( present(rel_tol) ) then
         if( rel_tol <= 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'rel_tol' argument must be strictly positive!" )
            status = -6
            go to 99
         end if
         epsrel = rel_tol
         if( epsrel < DEFAULT_TOL ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "'rel_tol' argument too small!",        &
                               "=> 'rel_tol' has been set to the default tolerance" )
            epsrel = DEFAULT_TOL
         end if
      else
         epsrel = DEFAULT_TOL
      end if

      fun_inner_ier = 0
      qag_neval = 0
      epsabs_2 = epsabs/10.0d0
      epsrel_2 = epsrel/10.0d0
      call dqag( fun_inner, ya, yb, epsabs, epsrel, key, res,           &
                 qag_abserr, neval_dummy, qag_ier,                      &
                 limit, lenw, last, iwork, work )

      if( fun_inner_ier /= 0 ) then
         ! abnormal return
         res = MF_NAN
         abserr = MF_NAN
         go to 98
      end if

      qag_ier = -qag_ier

      if( qag_ier /= 0 ) then
         ! abnormal return
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "the method used didn't converge to the required precision!" )
      end if

      abserr = qag_abserr

 98   continue

      q = res

      if( out%arg_present(3) ) then
         status = qag_ier
      else
         deallocate( status )
      end if

      if( out%arg_present(4) ) then
         neval = qag_neval
      end if

 99   continue

      if( .not. out%arg_present(3) ) then
         if( associated(status) ) then
            deallocate( status )
         end if
      end if

      call mf_restore_fpe( )

   contains

      function fun_inner( y ) result( res )

         real(kind=MF_DOUBLE), intent(in) :: y
         real(kind=MF_DOUBLE) :: res

         integer, parameter :: limit = 100, key = 1, lenw = 4*limit
         real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw)
         integer :: neval, ier, last, iwork(limit)

         call dqag_2( fun, y, fun_xa(y), fun_xb(y), epsabs_2, epsrel_2, &
                      key, res, abserr, neval, ier,                     &
                      limit, lenw, last, iwork, work )

         qag_neval = qag_neval + neval
         ier = -ier

         if( ier /= 0 ) then
            ! invalid input
            if( ier == -6 ) then
               write(STDERR,*) "(MUESLI msDblQuad:) fun_inner: internal error:"
               write(STDERR,*) "                    invalid input"
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               stop
            end if
            ! abnormal return
            res = 0 ! not MF_NAN to avoid stopping dqag...
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               " during integration along the x variable", &
                               "the method used didn't converge to the required precision!" )
         end if

         ! this is a sticky variable to record any failure in dqag_2
         ! (avoid resetting fun_inner_ier to zero)
         if( ier /= 0 ) fun_inner_ier = ier

      end function fun_inner

#endif
   end subroutine msDblQuad_funx
!_______________________________________________________________________
!
   function mfDblQuad_funy( fun, xa, xb, fun_ya, fun_yb, fun_y,         &
                            abs_tol, rel_tol )                          &
   result( out )

      interface
         function fun( x, y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x, y
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      interface
         function fun_ya( x ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x
            real(kind=MF_DOUBLE) :: res
         end function fun_ya
         function fun_yb( x ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x
            real(kind=MF_DOUBLE) :: res
         end function fun_yb
      end interface

      real(kind=MF_DOUBLE), intent(in)           :: xa, xb
      logical :: fun_y
      real(kind=MF_DOUBLE), intent(in), optional :: abs_tol, rel_tol
      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer, parameter :: limit = 100, key = 1, lenw = 4*limit
      real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw), res
      integer :: neval, ier, last, iwork(limit)
      real(kind=MF_DOUBLE) :: epsabs_2, epsrel_2

      real(kind=MF_DOUBLE), parameter :: DEFAULT_TOL = 1.0d-12

      integer :: fun_inner_ier
      common /slatec_dqag/ fun_inner_ier

      character(len=*), parameter :: ROUTINE_NAME = "mfDblQuad"

   !------ end of declarations -- execution starts hereafter  ------

      if( present(abs_tol) ) then
         if( abs_tol < 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'abs_tol' argument cannot be negative!" )
            return
         end if
         epsabs = abs_tol
      else
         epsabs = DEFAULT_TOL
      end if

      if( present(rel_tol) ) then
         if( rel_tol <= 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'rel_tol' argument must be strictly positive!" )
            return
         end if
         epsrel = rel_tol
         if( epsrel < DEFAULT_TOL ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "'rel_tol' argument too small",          &
                               "=> 'rel_tol' has been set to the default tolerance" )
            epsrel = DEFAULT_TOL
         end if
      else
         epsrel = DEFAULT_TOL
      end if

      fun_inner_ier = 0
      epsabs_2 = epsabs/10.0d0
      epsrel_2 = epsrel/10.0d0
      call dqag( fun_inner, xa, xb, epsabs, epsrel, key, res,           &
                 abserr, neval, ier,                                    &
                 limit, lenw, last, iwork, work )

      if( fun_inner_ier /= 0 ) then
         ! abnormal return
         res = MF_NAN
         go to 99
      end if

      ier = -ier

      if( ier /= 0 ) then
         ! invalid input
         if( ier == -6 ) then
            write(STDERR,*) "(MUESLI mfDblQuad:) internal error:"
            write(STDERR,*) "                    invalid input"
            mf_message_displayed = .true.
            call muesli_trace( pause ="yes" )
            stop
         end if
         ! abnormal return
         res = MF_NAN
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            " during integration along the y variable", &
                            "the method used didn't converge to the required precision!" )
      end if

 99   continue
      out = res

      out%status_temporary = .true.

   contains

      function fun_inner( x ) result( res )

         real(kind=MF_DOUBLE), intent(in) :: x
         real(kind=MF_DOUBLE) :: res

         integer, parameter :: limit = 100, key = 1, lenw = 4*limit
         real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw)
         integer :: neval, ier, last, iwork(limit)

         call dqag_3( fun, x, fun_ya(x), fun_yb(x), epsabs_2, epsrel_2, &
                      key, res, abserr, neval, ier,                     &
                      limit, lenw, last, iwork, work )

         ier = -ier

         if( ier /= 0 ) then
            ! invalid input
            if( ier == -6 ) then
               write(STDERR,*) "(MUESLI mfDblQuad:) fun_inner: internal error:"
               write(STDERR,*) "                    invalid input"
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               stop
            end if
            ! abnormal return
            res = 0 ! not MF_NAN to avoid stopping dqag...
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               " during integration along the x variable", &
                               "the method used didn't converge to the required precision!" )
         end if

         ! this is a sticky variable to record any failure in dqag_3
         ! (avoid resetting fun_inner_ier to zero)
         if( ier /= 0 ) fun_inner_ier = ier

      end function fun_inner

#endif
   end function mfDblQuad_funy
!_______________________________________________________________________
!
   subroutine msDblQuad_funy( out, fun, xa, xb, fun_ya, fun_yb, fun_y,  &
                              abs_tol, rel_tol )

      interface
         function fun( x, y ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x, y
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      interface
         function fun_ya( x ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x
            real(kind=MF_DOUBLE) :: res
         end function fun_ya
         function fun_yb( x ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x
            real(kind=MF_DOUBLE) :: res
         end function fun_yb
      end interface

      real(kind=MF_DOUBLE), intent(in)           :: xa, xb
      logical :: fun_y
      real(kind=MF_DOUBLE), intent(in), optional :: abs_tol, rel_tol
      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      integer, parameter :: limit = 100, key = 1, lenw = 4*limit
      real(kind=MF_DOUBLE) :: epsabs, epsrel, qag_abserr, work(lenw), res
      integer :: qag_neval, qag_ier, last, iwork(limit)
      type(mfArray), pointer :: q => null(), abserr => null(),          &
                                neval => null(), status => null()
      logical :: no_pause
      real(kind=MF_DOUBLE) :: epsabs_2, epsrel_2
      integer :: neval_dummy

      real(kind=MF_DOUBLE), parameter :: DEFAULT_TOL = 1.0d-12

      integer :: fun_inner_ier
      common /slatec_dqag/ fun_inner_ier

      character(len=*), parameter :: ROUTINE_NAME = "msDblQuad"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      ! 2 to 4 out-args must be specified
      ! 2 are optional, only the first can be equal to MF_NO_ARG
      if( out%n < 2 .or. 4 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to four output args required!",       &
                            "syntax is: call msQuad( mfOut(<out_arg>), fun, a, b [, abs_tol, rel_tol, sing] )", &
                            "    with <out_arg> equal to {q,abserr[,status,neval]}", &
                            "",                                         &
                            "Moreover, the output arg 'status' has not been set, because it may be not present!" )
         go to 99
      end if

      q => out%ptr1
      call msSilentRelease( q )
      abserr => out%ptr2
      call msSilentRelease( abserr )
      ! optional args
      if( out%arg_present(3) ) then
         status => out%ptr3
         call msSilentRelease( status )
         no_pause = .true.
      else
         allocate( status )
         no_pause = .false.
      end if
      if( out%arg_present(4) ) then
         neval => out%ptr4
         call msSilentRelease( neval )
      else
      end if

      q = MF_NAN

      if( present(abs_tol) ) then
         if( abs_tol < 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'abs_tol' argument cannot be negative!" )
            return
         end if
         epsabs = abs_tol
      else
         epsabs = DEFAULT_TOL
      end if

      if( present(rel_tol) ) then
         if( rel_tol <= 0.0d0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'rel_tol' argument must be strictly positive!" )
            status = -6
            go to 99
         end if
         epsrel = rel_tol
         if( epsrel < DEFAULT_TOL ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "'rel_tol' argument too small!",        &
                               "=> 'rel_tol' has been set to the default tolerance" )
            epsrel = DEFAULT_TOL
         end if
      else
         epsrel = DEFAULT_TOL
      end if

      fun_inner_ier = 0
      qag_neval = 0
      epsabs_2 = epsabs/10.0d0
      epsrel_2 = epsrel/10.0d0
      call dqag( fun_inner, xa, xb, epsabs, epsrel, key, res,           &
                 qag_abserr, neval_dummy, qag_ier,                      &
                 limit, lenw, last, iwork, work )

      if( fun_inner_ier /= 0 ) then
         ! abnormal return
         res = MF_NAN
         abserr = MF_NAN
         go to 98
      end if

      qag_ier = -qag_ier

      if( qag_ier /= 0 ) then
         ! abnormal return
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "the method used didn't converge to the required precision!" )
      end if

      abserr = qag_abserr

 98   continue

      q = res

      if( out%arg_present(3) ) then
         status = qag_ier
      else
         deallocate( status )
      end if

      if( out%arg_present(4) ) then
         neval = qag_neval
      end if

 99   continue

      if( .not. out%arg_present(3) ) then
         if( associated(status) ) then
            deallocate( status )
         end if
      end if

      call mf_restore_fpe( )

   contains

      function fun_inner( x ) result( res )

         real(kind=MF_DOUBLE), intent(in) :: x
         real(kind=MF_DOUBLE) :: res

         integer, parameter :: limit = 100, key = 1, lenw = 4*limit
         real(kind=MF_DOUBLE) :: epsabs, epsrel, abserr, work(lenw)
         integer :: neval, ier, last, iwork(limit)

         call dqag_3( fun, x, fun_ya(x), fun_yb(x), epsabs_2, epsrel_2, &
                      key, res, abserr, neval, ier,                     &
                      limit, lenw, last, iwork, work )

         qag_neval = qag_neval + neval
         ier = -ier

         if( ier /= 0 ) then
            ! invalid input
            if( ier == -6 ) then
               write(STDERR,*) "(MUESLI msDblQuad:) fun_inner: internal error:"
               write(STDERR,*) "                    invalid input"
               mf_message_displayed = .true.
               call muesli_trace( pause ="yes" )
               stop
            end if
            ! abnormal return
            res = 0 ! not MF_NAN to avoid stopping dqag...
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               " during integration along the x variable", &
                               "the method used didn't converge to the required precision!" )
         end if

         ! this is a sticky variable to record any failure in dqag_2
         ! (avoid resetting fun_inner_ier to zero)
         if( ier /= 0 ) fun_inner_ier = ier

      end function fun_inner

#endif
   end subroutine msDblQuad_funy
