! f90 include file

!_______________________________________________________________________
!
   function mfLsqNonLin_JacDF( m, fcn, p0, n,                           &
                               options )                                &
   result( out )

      use minpack

      interface
         subroutine fcn( m, n, p, fvec, iflag )
            import :: MF_DOUBLE
            integer,              intent(in) :: m, n
            real(kind=MF_DOUBLE), intent(in) :: p(n)
            real(kind=MF_DOUBLE)             :: fvec(m)
            integer                          :: iflag
         end subroutine fcn
      end interface

      integer, intent(in) :: m, n
      type(mfArray)       :: p0
      type(mf_NL_Options), target, optional :: options

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Minimize the sum of the squares of m nonlinear functions
      ! depending on n parameters.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, p0(:) is an initial guess
      !
      ! Solved by the 'lmdif' routine of MINPACK

      type(mfArray), pointer :: tol => null(),                          &
                                epsfcn => null(),                       &
                                lower_bounds => null(),                 &
                                upper_bounds => null()
      type(mfArray), target :: empty
      integer :: max_iter
      logical :: print, print_using_transf, box_constrained,            &
                 print_sing_val

      integer :: info, iwork(n), lwork
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec(:), work(:)

      integer :: maxfev, mode, mp5n, nfev
      real(kind=MF_DOUBLE) :: ftol, gtol, xtol, factor, sing_jac_tol
      real(kind=MF_DOUBLE), parameter :: zero = 0.0d0

      integer :: i, niter
      real(kind=MF_DOUBLE), allocatable :: residue_hist(:), x_hist(:,:), &
                                           epsfcn2(:)

      real(kind=MF_DOUBLE), pointer :: lbounds(:) => null(),            &
                                       ubounds(:) => null()

      type(func_ptr), allocatable :: f_inv_transf(:)

      logical, parameter :: identifiability = .false.
      real(kind=MF_DOUBLE) :: sensitiv_ratio(1)

      character(len=*), parameter :: ROUTINE_NAME = "mfLsqNonLin"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( p0 )

      ! m must be greater or equal n
      if( m < n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "you must have m >= n!" )
         go to 99
      end if

      if( present(options) ) then
         tol      => options%tol
         max_iter  = options%max_iter
         epsfcn   => options%epsfcn
         print     = options%print
         print_using_transf = options%print_using_transf
         if( print_using_transf ) then
            allocate( f_inv_transf(n) )
            do i = 1, n
               f_inv_transf(i)%ptr => options%f_inv_transf(i)%ptr
            end do
         end if
         factor          = options%init_step_bound_factor
         box_constrained = options%box_constrained
         lower_bounds   => options%lower_bounds
         upper_bounds   => options%upper_bounds
         sing_jac_tol    = options%sing_jac_tol
         print_sing_val  = options%print_sing_val
      else
         tol      => empty
         max_iter  = 50
         epsfcn   => empty
         print     = .false.
         factor    = 1.0d2
         box_constrained = .false.
         lower_bounds   => empty
         upper_bounds   => empty
         sing_jac_tol    = epsilon(1.0d0)
         print_sing_val  = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( all(tol <= 0.0d0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'tol' argument must be strictly positive!" )
            go to 99
         end if
         if( size(tol) == 1 ) then
            ftol = tol
            xtol = tol
            gtol = zero
         else
            if( size(tol) == 3 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
               gtol = mfGet( tol, 3 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must be of length 3!" )
               go to 99
            end if
         end if
      else
         ftol = 1.0d-6
         xtol = 1.0d-6
         gtol = zero
      end if

      if( mfIsEmpty(p0) ) then

         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "initial guess 'p0' is empty: trying to use", &
                            "a zero vector!" )

         ! initial guess
         var = 0.0d0

      else

         ! 'p0' must be numeric
         if( .not. mfIsNumeric(p0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "mfArray 'p0' must be numeric!" )
            go to 99
         end if

         ! 'p0' must be dense
         if( mfIsSparse(p0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "mfArray 'p0' must have dense storage!" )
            go to 99
         end if

         ! 'p0' must be a scalar or a vector, according the value of n
         if( n == 1 ) then
            if( .not. mfIsScalar(p0) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'p0' must be a scalar!" )
               go to 99
            end if
         else if( n > 1 ) then
            if( .not. mfIsVector(p0) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'p0' must be a vector!" )
               go to 99
            end if
         else
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'n' cannot be negative or null!" )
            go to 99
         end if

         if( size(p0) /= n ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "size of 'p0' must be equal to n!" )
            go to 99
         end if

         ! initial guess
         var = p0

      end if

      if( box_constrained ) then

         if( n == 1 ) then
            ! case of a single parameter
            if( .not. mfIsScalar(lower_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "given the number of parameters (here, one),", &
                                  "'lower_bounds' option must be a scalar!" )
               go to 99
            end if
            if( .not. mfIsScalar(upper_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "given the number of parameters (here, one),", &
                                  "'upper_bounds' option must be a scalar!" )
               go to 99
            end if
         else
            ! more than one parameter
            if( lower_bounds%shape(1) /= 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'lower_bounds' option must be a row vector!" )
               go to 99
            end if
            if( size(lower_bounds) /= n ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'lower_bounds' vector must have the same", &
                                  "number of elements as the parameters!" )
               go to 99
            end if
            if( upper_bounds%shape(1) /= 1 ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'upper_bounds' option must be a row vector!" )
               go to 99
            end if
            if( size(upper_bounds) /= n ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'upper_bounds' vector must have the same", &
                                  "number of elements as the parameters!" )
               go to 99
            end if
         end if

         lbounds => lower_bounds%double(1,:)
         ubounds => upper_bounds%double(1,:)

      end if

      allocate( epsfcn2(n) )
      if( mfIsEmpty(epsfcn) ) then
         epsfcn2(:) = 0.0d0
      else
         if( mfIsScalar(epsfcn) ) then
            epsfcn2(:) = epsfcn%double(1,1)
         else
            if( mfIsVector(epsfcn) ) then
               if( size(epsfcn) /= n ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when 'epsfcn' is a vector,",      &
                                     "its length must be equal to n!" )
                  go to 99
               end if
               epsfcn2(:) = epsfcn
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'epsfcn' must be either a scalar,",  &
                                  "or a vector of length n!" )
               go to 99
            end if
         end if
      end if

      lwork = m*n + 5*n + m
      allocate( fvec(m), work(lwork) )
      allocate( residue_hist(max_iter) )
      allocate( x_hist(max_iter,n) )
      residue_hist(:) = MF_NAN
      x_hist(:,:) = MF_NAN

      maxfev = max_iter*(n + 1) - 2
      mode = 1
      mp5n = m + 5*n
      call lmdif( fcn, m, n, var, fvec, ftol, xtol, gtol, maxfev,       &
                  epsfcn2, work(1), mode, factor, info, nfev,           &
                  work(mp5n+1), iwork, work(n+1), work(2*n+1),          &
                  work(3*n+1), work(4*n+1), work(5*n+1),                &
                  max_iter, print, print_using_transf, f_inv_transf,    &
                  niter, residue_hist, x_hist,                          &
                  box_constrained, lbounds, ubounds,                    &
                  identifiability, sensitiv_ratio, sing_jac_tol,        &
                  print_sing_val )

      if( 1 <= info .and. info <= 3 ) then
         if( print ) then
            print "(2X,A)", "*** LMA algorithm has been terminated ***"
            select case( info )
               case( 1 )
                  print "(4X,A)", "-> ftol has been used as stopping criterion."
               case( 2 )
                  print "(4X,A)", "-> xtol has been used as stopping criterion."
               case( 3 )
                  print "(4X,A)", "-> both ftol and xtol have been used as stopping criterion."
            end select
         end if
      else
         ! some problem occured
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "please use instead the subroutine form (msLsqNonLin)", &
                            "to investigate the problem.",              &
                            "(Don't forget to use the 'status' output variable)" )
         go to 99
      end if

      out = var

      if( mf_phys_units ) then

         out%units(:) = p0%units(:)

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( p0 )
      call msAutoRelease( p0 )

      call mf_restore_fpe( )

#endif
   end function mfLsqNonLin_JacDF
!_______________________________________________________________________
!
   subroutine msLsqNonLin_JacDF( out, m, fcn, p0, n,                    &
                                 options )

      use minpack

      interface
         subroutine fcn( m, n, p, fvec, iflag )
            import :: MF_DOUBLE
            integer,              intent(in) :: m, n
            real(kind=MF_DOUBLE), intent(in) :: p(n)
            real(kind=MF_DOUBLE)             :: fvec(m)
            integer                          :: iflag
         end subroutine fcn
      end interface

      integer, intent(in) :: m, n
      type(mfArray)       :: p0
      type(mf_NL_Options), target, optional :: options

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Minimize the sum of the squares of m nonlinear functions
      ! depending on n parameters.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, p0(:) is an initial guess
      !
      ! returns also the norm of the residual
      !
      ! Solved by the 'lmdif' routine of MINPACK

      type(mfArray), pointer :: tol => null(),                          &
                                epsfcn => null(),                       &
                                lower_bounds => null(),                 &
                                upper_bounds => null()
      type(mfArray), target :: empty
      integer :: max_iter
      logical :: print, print_using_transf, box_constrained,            &
                 print_sing_val

      integer :: info, iwork(n), lwork
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec(:), work(:)

      integer :: maxfev, mode, mp5n, nfev
      real(kind=MF_DOUBLE) :: ftol, gtol, xtol, factor, sing_jac_tol
      real(kind=MF_DOUBLE), parameter :: zero = 0.0d0

      integer :: i, niter
      real(kind=MF_DOUBLE), allocatable :: residue_hist(:), x_hist(:,:), &
                                           epsfcn2(:)

      real(kind=MF_DOUBLE), pointer :: lbounds(:) => null(),            &
                                       ubounds(:) => null()

      type(mfArray), pointer :: p => null(), resnorm => null(),         &
                                status => null(), res_log => null(),    &
                                p_log => null(), ident => null()

      type(func_ptr), allocatable :: f_inv_transf(:)

      logical :: identifiability
      real(kind=MF_DOUBLE), allocatable :: sensitiv_ratio(:)

      character(len=*), parameter :: ROUTINE_NAME = "msLsqNonLin"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( p0 )

      ! 2 to 6 out-args must be specified
      ! 4 are optional, only the 3 firsts can be equal to MF_NO_ARG
      if( out%n < 2 .or. 6 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to five output args required!",       &
                            "syntax is : call msLsqNonLin( mfOut(p,resnorm[,status,res_log,p_log,ident]),", &
                            "                              m, fcn, p0, n [, tol, max_iter, epsfcn, print])" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, p0 ) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      p => out%ptr1
      resnorm => out%ptr2
      call msSilentRelease( p, resnorm )
      if( out%arg_present(3) ) then
         status => out%ptr3
         call msSilentRelease( status )
      end if
      if( out%arg_present(4) ) then
         res_log => out%ptr4
         call msSilentRelease( res_log )
      end if
      if( out%arg_present(5) ) then
         p_log => out%ptr5
         call msSilentRelease( p_log )
      end if
      if( out%arg_present(6) ) then
         ident => out%ptr6
         call msSilentRelease( ident )
         identifiability = .true.
      else
         identifiability = .false.
      end if

      ! m must be greater or equal n
      if( m < n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "you must have m >= n!" )
      end if

      if( present(options) ) then
         tol      => options%tol
         max_iter  = options%max_iter
         epsfcn   => options%epsfcn
         print     = options%print
         print_using_transf = options%print_using_transf
         if( print_using_transf ) then
            allocate( f_inv_transf(n) )
            do i = 1, n
               f_inv_transf(i)%ptr => options%f_inv_transf(i)%ptr
            end do
         end if
         factor          = options%init_step_bound_factor
         box_constrained = options%box_constrained
         lower_bounds   => options%lower_bounds
         upper_bounds   => options%upper_bounds
         sing_jac_tol    = options%sing_jac_tol
         print_sing_val  = options%print_sing_val
      else
         tol     => empty
         max_iter = 50
         epsfcn  => empty
         print    = .false.
         print_using_transf = .false.
         factor   = 1.0d2
         box_constrained = .false.
         lower_bounds   => empty
         upper_bounds   => empty
         sing_jac_tol    = epsilon(1.0d0)
         print_sing_val  = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( size(tol) == 1 ) then
            ftol = tol
            xtol = tol
            gtol = zero
         else
            if( size(tol) == 3 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
               gtol = mfGet( tol, 3 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must be of length 3!" )
               go to 99
            end if
         end if
      else
         ftol = 1.0d-6
         xtol = 1.0d-6
         gtol = zero
      end if

      if( mfIsEmpty(p0) ) then

         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "initial guess 'p0' is empty: trying to use", &
                            "a zero vector!" )

         ! initial guess
         var = 0.0d0

      else

         ! 'p0' must be numeric
         if( .not. mfIsNumeric(p0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "mfArray 'p0' must be numeric!" )
            go to 99
         end if

         ! 'p0' must be dense
         if( mfIsSparse(p0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "mfArray 'p0' must have dense storage!" )
            go to 99
         end if

         ! 'p0' must be a scalar or a vector, according the value of n
         if( n == 1 ) then
            if( .not. mfIsScalar(p0) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'p0' must be a scalar!" )
               go to 99
            end if
         else if( n > 1 ) then
            if( .not. mfIsVector(p0) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'p0' must be a vector!" )
               go to 99
            end if
         else
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'n' cannot be negative or null!" )
            go to 99
         end if

         if( size(p0) /= n ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "size of 'p0' must be equal to n!" )
            go to 99
         end if

         ! initial guess
         var = p0

      end if

      if( box_constrained ) then

         if( n == 1 ) then
            ! case of a single parameter
            if( .not. mfIsScalar(lower_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "given the number of parameters (here, one),", &
                                  "'lower_bounds' option must be a scalar!" )
               go to 99
            end if
            if( .not. mfIsScalar(upper_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "given the number of parameters (here, one),", &
                                  "'upper_bounds' option must be a scalar!" )
               go to 99
            end if
         else
            ! more than one parameter
            if( .not. mfIsVector(lower_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'lower_bounds' option must be a vector!" )
               go to 99
            end if
            if( size(lower_bounds) /= n ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'lower_bounds' vector must have the same", &
                                  "number of elements as the parameters!" )
               go to 99
            end if
            if( .not. mfIsVector(upper_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'upper_bounds' option must be a vector!" )
               go to 99
            end if
            if( size(upper_bounds) /= n ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'upper_bounds' vector must have the same", &
                                  "number of elements as the parameters!" )
               go to 99
            end if
         end if

         lbounds => lower_bounds%double(1,:)
         ubounds => upper_bounds%double(1,:)

      end if

      allocate( epsfcn2(n) )
      if( mfIsEmpty(epsfcn) ) then
         epsfcn2(:) = 0.0d0
      else
         if( mfIsScalar(epsfcn) ) then
            epsfcn2(:) = epsfcn%double(1,1)
         else
            if( mfIsVector(epsfcn) ) then
               if( size(epsfcn) /= n ) then
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "when 'epsfcn' is a vector,",      &
                                     "its length must be equal to n!" )
                  go to 99
               end if
               epsfcn2(:) = epsfcn
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'epsfcn' must be either a scalar,",  &
                                  "or a vector of length n!" )
               go to 99
            end if
         end if
      end if

      lwork = m*n + 5*n + m
      allocate( fvec(m), work(lwork) )
      allocate( residue_hist(max_iter) )
      allocate( x_hist(max_iter,n) )
      residue_hist(:) = MF_NAN
      x_hist(:,:) = MF_NAN

      if( identifiability ) then
         allocate( sensitiv_ratio(n) )
      end if

      maxfev = max_iter*(n + 1) - 2
      mode = 1
      mp5n = m + 5*n
      call lmdif( fcn, m, n, var, fvec, ftol, xtol, gtol, maxfev,       &
                  epsfcn2, work(1), mode, factor, info, nfev,           &
                  work(mp5n+1), iwork, work(n+1), work(2*n+1),          &
                  work(3*n+1), work(4*n+1), work(5*n+1),                &
                  max_iter, print, print_using_transf, f_inv_transf,    &
                  niter, residue_hist, x_hist,                          &
                  box_constrained, lbounds, ubounds,                    &
                  identifiability, sensitiv_ratio, sing_jac_tol,        &
                  print_sing_val )

      if( 1 <= info .and. info <= 3 ) then
         if( print ) then
            print "(2X,A)", "*** LMA algorithm has been terminated ***"
            select case( info )
               case( 1 )
                  print "(4X,A)", "-> ftol has been used as stopping criterion."
               case( 2 )
                  print "(4X,A)", "-> xtol has been used as stopping criterion."
               case( 3 )
                  print "(4X,A)", "-> both ftol and xtol have been used as stopping criterion."
            end select
         end if
      end if

      if( out%arg_present(3) ) then
         if( 1 <= info .and. info <= 3 ) then
            ! normal termination
            status = info
         else
            if( info == 0 ) then
               ! improper input parameters.
               status = -1
            else if( info < 0 ) then
               ! user has terminated execution (in fcn),
               ! or fdjac2 cannot compute the FD approx of the jacobian.
               status = info
            else
               ! other problem
               status = -info
            end if
            go to 99
         end if
      else ! cannot return status
         if( info < 1 .or. 3 < info ) then
            ! some problem occured
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "please use the 'status' output variable to", &
                               "investigate the problem." )
            go to 99
         end if
      end if

      if( out%arg_present(4) ) then
         call msAssign( res_log, .t. mf(residue_hist(1:niter)) )
      end if

      if( out%arg_present(5) ) then
         p_log = x_hist(1:niter,1:n)
      end if

      if( out%arg_present(6) ) then
         call msAssign( ident, .t. mf(sensitiv_ratio) )
      end if

      if( info < 1 .or. 3 < info ) then

         p = MF_NAN * mfOnes(1,n)
         resnorm = MF_NAN

      else

         p = var
         ! compute norm of residual, if required
         ! (mfNorm is not available within the current module!)
         resnorm = fvec
         call msAssign( resnorm, mfSqrt(mfSum(resnorm**2)) )

      end if

      if( mf_phys_units ) then

         p%units(:) = p0%units(:)

      end if

 99   continue

      call msFreeArgs( p0 )
      call msAutoRelease( p0 )

      call mf_restore_fpe( )

#endif
   end subroutine msLsqNonLin_JacDF
!_______________________________________________________________________
!
   function mfLsqNonLin_JacUser( m, fcn, p0, n,                         &
                                 options, jac )                         &
   result( out )

      use minpack

      interface
         subroutine fcn( m, n, p, fvec, iflag )
            import :: MF_DOUBLE
            integer,              intent(in) :: m, n
            real(kind=MF_DOUBLE), intent(in) :: p(n)
            real(kind=MF_DOUBLE)             :: fvec(m)
            integer                          :: iflag
         end subroutine fcn
         subroutine jac( m, n, p, jacobian )
            import :: MF_DOUBLE
            integer,              intent(in) :: m, n
            real(kind=MF_DOUBLE), intent(in) :: p(n)
            real(kind=MF_DOUBLE)             :: jacobian(m,n)
         end subroutine jac
      end interface

      integer, intent(in) :: m, n
      type(mfArray)       :: p0
      type(mf_NL_Options), target, optional :: options

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Minimize the sum of the squares of m nonlinear functions
      ! depending on n parameters.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, p0(:) is an initial guess
      !
      ! Solved by the 'lmder_mf' routine of MINPACK

      type(mfArray), pointer :: tol => null(),                          &
                                lower_bounds => null(),                 &
                                upper_bounds => null()
      type(mfArray), target :: empty
      integer :: max_iter
      logical :: print, print_using_transf, box_constrained,            &
                 print_sing_val
      integer :: check_jac, print_check_jac

      integer :: info, ipvt(n), lwork
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec(:), fjac(:,:), work(:)

      integer :: maxfev, mode, nfev, njev
      real(kind=MF_DOUBLE) :: ftol, gtol, xtol, factor, sing_jac_tol
      real(kind=MF_DOUBLE), parameter :: zero = 0.0d0

      integer :: i, niter
      real(kind=MF_DOUBLE), allocatable :: residue_hist(:), x_hist(:,:)

      real(kind=MF_DOUBLE), pointer :: lbounds(:) => null(),            &
                                       ubounds(:) => null()

      type(func_ptr), allocatable :: f_inv_transf(:)

      logical, parameter :: identifiability = .false.
      real(kind=MF_DOUBLE) :: sensitiv_ratio(1)

      character(len=*), parameter :: ROUTINE_NAME = "mfLsqNonLin"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( p0 )

      ! m must be greater or equal n
      if( m < n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "you must have m >= n!" )
      end if

      if( present(options) ) then
         tol            => options%tol
         max_iter        = options%max_iter
         print           = options%print
         check_jac       = options%check_jac
         print_check_jac = options%print_check_jac
         print_using_transf = options%print_using_transf
         if( print_using_transf ) then
            allocate( f_inv_transf(n) )
            do i = 1, n
               f_inv_transf(i)%ptr => options%f_inv_transf(i)%ptr
            end do
         end if
         factor          = options%init_step_bound_factor
         box_constrained = options%box_constrained
         lower_bounds   => options%lower_bounds
         upper_bounds   => options%upper_bounds
         sing_jac_tol    = options%sing_jac_tol
         print_sing_val  = options%print_sing_val
      else
         tol            => empty
         max_iter        = 50
         print           = .false.
         check_jac       = 0
         print_check_jac = 0
         factor          = 1.0d2
         box_constrained = .false.
         lower_bounds   => empty
         upper_bounds   => empty
         sing_jac_tol    = epsilon(1.0d0)
         print_sing_val  = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( size(tol) == 1 ) then
            ftol = tol
            xtol = tol
            gtol = zero
         else
            if( size(tol) == 3 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
               gtol = mfGet( tol, 3 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must be a length 3!" )
               go to 99
            end if
         end if
      else
         ftol = 1.0d-6
         xtol = 1.0d-6
         gtol = zero
      end if

      if( .not. MF_NUMERICAL_CHECK ) then
         if( check_jac > 0 ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "jacobian check cannot be done in the release mode!", &
                               "(it has been disabled)" )
         end if

         if( print_check_jac > 0 ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "jacobian check cannot be done in the release mode!", &
                               "(printing of the result of check has been disabled)" )
         end if
      end if

      if( mfIsEmpty(p0) ) then

         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "initial guess 'p0' is empty: trying to use", &
                            "a zero vector!" )

         ! initial guess
         var = 0.0d0

      else

         ! 'p0' must be numeric
         if( .not. mfIsNumeric(p0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "mfArray 'p0' must be numeric!" )
            go to 99
         end if

         ! 'p0' must be dense
         if( mfIsSparse(p0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "mfArray 'p0' must have dense storage!" )
            go to 99
         end if

         ! 'p0' must be a scalar or a vector, according the value of n
         if( n == 1 ) then
            if( .not. mfIsScalar(p0) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'p0' must be a scalar!" )
               go to 99
            end if
         else if( n > 1 ) then
            if( .not. mfIsVector(p0) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'p0' must be a vector!" )
               go to 99
            end if
         else
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'n' cannot be negative or null!" )
            go to 99
         end if

         if( size(p0) /= n ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "size of 'p0' must be equal to n!" )
            go to 99
         end if

         ! initial guess
         var = p0

      end if

      if( box_constrained ) then

         if( n == 1 ) then
            ! case of a single parameter
            if( .not. mfIsScalar(lower_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "given the number of parameters (here, one),", &
                                  "'lower_bounds' option must be a scalar!" )
               go to 99
            end if
            if( .not. mfIsScalar(upper_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "given the number of parameters (here, one),", &
                                  "'upper_bounds' option must be a scalar!" )
               go to 99
            end if
         else
            ! more than one parameter
            if( .not. mfIsVector(lower_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'lower_bounds' option must be a vector!" )
               go to 99
            end if
            if( size(lower_bounds) /= n ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'lower_bounds' vector must have the same", &
                                  "number of elements as the parameters!" )
               go to 99
            end if
            if( .not. mfIsVector(upper_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'upper_bounds' option must be a vector!" )
               go to 99
            end if
            if( size(upper_bounds) /= n ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'upper_bounds' vector must have the same", &
                                  "number of elements as the parameters!" )
               go to 99
            end if
         end if

         lbounds => lower_bounds%double(1,:)
         ubounds => upper_bounds%double(1,:)

      end if

      lwork = 5*n + m
      allocate( fvec(m), fjac(m,n), work(lwork) )
      allocate( residue_hist(max_iter) )
      allocate( x_hist(max_iter,n) )
      residue_hist(:) = MF_NAN
      x_hist(:,:) = MF_NAN

      maxfev = max_iter
      mode = 1
      call lmder_mf( fcn, jac, m, n, var, fvec, fjac, ftol,             &
                     xtol, gtol, maxfev, work(1), mode, factor,         &
                     info, nfev, njev, ipvt, work(n+1), work(2*n+1),    &
                     work(3*n+1), work(4*n+1), work(5*n+1),             &
                     max_iter, print, print_using_transf,               &
                     check_jac, print_check_jac,                        &
                     niter, residue_hist, x_hist,                       &
                     box_constrained, lbounds, ubounds,                 &
                     identifiability, sensitiv_ratio, sing_jac_tol,     &
                     print_sing_val )

      if( 1 <= info .and. info <= 3 ) then
         if( print ) then
            print "(2X,A)", "*** LMA algorithm has been terminated ***"
            select case( info )
               case( 1 )
                  print "(4X,A)", "-> ftol has been used as stopping criterion."
               case( 2 )
                  print "(4X,A)", "-> xtol has been used as stopping criterion."
               case( 3 )
                  print "(4X,A)", "-> both ftol and xtol have been used as stopping criterion."
            end select
         end if
      else
         ! some problem occured
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "please use instead the subroutine form (msLsqNonLin)", &
                            "to investigate the problem.",              &
                            "(Don't forget to use the 'status' output variable)" )
         go to 99
      end if

      out = var

      if( mf_phys_units ) then

         out%units(:) = p0%units(:)

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( p0 )
      call msAutoRelease( p0 )

      call mf_restore_fpe( )

#endif
   end function mfLsqNonLin_JacUser
!_______________________________________________________________________
!
   subroutine msLsqNonLin_JacUser( out, m, fcn, p0, n,                  &
                                   options, jac )

      use minpack

      interface
         subroutine fcn( m, n, p, fvec, iflag )
            import :: MF_DOUBLE
            integer,              intent(in) :: m, n
            real(kind=MF_DOUBLE), intent(in) :: p(n)
            real(kind=MF_DOUBLE)             :: fvec(m)
            integer                          :: iflag
         end subroutine fcn
         subroutine jac( m, n, p, jacobian )
            import :: MF_DOUBLE
            integer,              intent(in) :: m, n
            real(kind=MF_DOUBLE), intent(in) :: p(n)
            real(kind=MF_DOUBLE)             :: jacobian(m,n)
         end subroutine jac
      end interface

      integer, intent(in) :: m, n
      type(mfArray)       :: p0
      type(mf_NL_Options), target, optional :: options

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Minimize the sum of the squares of m nonlinear functions
      ! depending on n parameters.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, p0(:) is an initial guess
      !
      ! returns also the norm of the residual
      !
      ! Solved by the 'lmder_mf' routine of MINPACK

      type(mfArray), pointer :: tol => null(),                          &
                                lower_bounds => null(),                 &
                                upper_bounds => null()
      type(mfArray), target :: empty
      integer :: max_iter
      logical :: print, print_using_transf, box_constrained,            &
                 print_sing_val
      integer :: check_jac, print_check_jac

      integer :: info, ipvt(n), lwork
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec(:), fjac(:,:), work(:)

      integer :: maxfev, mode, nfev, njev
      real(kind=MF_DOUBLE) :: ftol, gtol, xtol, factor, sing_jac_tol
      real(kind=MF_DOUBLE), parameter :: zero = 0.0d0

      character(len=3) :: info_str
      integer :: i, niter
      real(kind=MF_DOUBLE), allocatable :: residue_hist(:), x_hist(:,:)

      real(kind=MF_DOUBLE), pointer :: lbounds(:) => null(),            &
                                       ubounds(:) => null()

      type(mfArray), pointer :: p => null(), resnorm => null(),         &
                                status => null(), res_log => null(),    &
                                p_log => null(), ident => null()

      type(func_ptr), allocatable :: f_inv_transf(:)

      logical :: identifiability
      real(kind=MF_DOUBLE), allocatable :: sensitiv_ratio(:)

      character(len=*), parameter :: ROUTINE_NAME = "msLsqNonLin"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( p0 )

      ! 2 to 6 out-args must be specified
      ! 4 are optional, only the 3 firsts can be equal to MF_NO_ARG
      if( out%n < 2 .or. 6 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to five output args required!",       &
                            "syntax is : call msLsqNonLin( mfOut(p,resnorm[,status,res_log,p_log,ident]),", &
                            "                              m, fcn, p0, n [, tol, max_iter, print, jac])" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, p0 ) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      p => out%ptr1
      resnorm => out%ptr2
      call msSilentRelease( p, resnorm )
      if( out%arg_present(3) ) then
         status => out%ptr3
         call msSilentRelease( status )
      end if
      if( out%arg_present(4) ) then
         res_log => out%ptr4
         call msSilentRelease( res_log )
      end if
      if( out%arg_present(5) ) then
         p_log => out%ptr5
         call msSilentRelease( p_log )
      end if
      if( out%arg_present(6) ) then
         ident => out%ptr6
         call msSilentRelease( ident )
         identifiability = .true.
      else
         identifiability = .false.
      end if

      ! m must be greater or equal n
      if( m < n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "you must have m >= n!" )
      end if

      if( present(options) ) then
         tol            => options%tol
         max_iter        = options%max_iter
         print           = options%print
         check_jac       = options%check_jac
         print_check_jac = options%print_check_jac
         print_using_transf = options%print_using_transf
         if( print_using_transf ) then
            allocate( f_inv_transf(n) )
            do i = 1, n
               f_inv_transf(i)%ptr => options%f_inv_transf(i)%ptr
            end do
         end if
         factor          = options%init_step_bound_factor
         box_constrained = options%box_constrained
         lower_bounds   => options%lower_bounds
         upper_bounds   => options%upper_bounds
         sing_jac_tol    = options%sing_jac_tol
         print_sing_val  = options%print_sing_val
      else
         tol            => empty
         max_iter        = 50
         print           = .false.
         check_jac       = 0
         print_check_jac = 0
         factor          = 1.0d2
         box_constrained = .false.
         lower_bounds   => empty
         upper_bounds   => empty
         sing_jac_tol    = epsilon(1.0d0)
         print_sing_val  = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( size(tol) == 1 ) then
            ftol = tol
            xtol = tol
            gtol = zero
         else
            if( size(tol) == 3 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
               gtol = mfGet( tol, 3 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must be a length 3!" )
               go to 99
            end if
         end if
      else
         ftol = 1.0d-6
         xtol = 1.0d-6
         gtol = zero
      end if

      if( .not. MF_NUMERICAL_CHECK ) then
         if( check_jac > 0 ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "jacobian check cannot be done in the release mode!", &
                               "(it has been disabled)" )
         end if

         if( print_check_jac > 0 ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "jacobian check cannot be done in the release mode!", &
                               "(printing of the result of check has been disabled)" )
         end if
      end if

      if( mfIsEmpty(p0) ) then

         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "initial guess 'p0' is empty: trying to use", &
                            "a zero vector!" )

         ! initial guess
         var = 0.0d0

      else

         ! 'p0' must be numeric
         if( .not. mfIsNumeric(p0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "mfArray 'p0' must be numeric!" )
            go to 99
         end if

         ! 'p0' must be dense
         if( mfIsSparse(p0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "mfArray 'p0' must have dense storage!" )
            go to 99
         end if

         ! 'p0' must be a scalar or a vector, according the value of n
         if( n == 1 ) then
            if( .not. mfIsScalar(p0) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'p0' must be a scalar!" )
               go to 99
            end if
         else if( n > 1 ) then
            if( .not. mfIsVector(p0) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'p0' must be a vector!" )
               go to 99
            end if
         else
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'n' cannot be negative or null!" )
            go to 99
         end if

         if( size(p0) /= n ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "size of 'p0' must be equal to n!" )
            go to 99
         end if

         ! initial guess
         var = p0

      end if

      if( box_constrained ) then

         if( n == 1 ) then
            ! case of a single parameter
            if( .not. mfIsScalar(lower_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "given the number of parameters (here, one),", &
                                  "'lower_bounds' option must be a scalar!" )
               go to 99
            end if
            if( .not. mfIsScalar(upper_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "given the number of parameters (here, one),", &
                                  "'upper_bounds' option must be a scalar!" )
               go to 99
            end if
         else
            ! more than one parameter
            if( .not. mfIsVector(lower_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'lower_bounds' option must be a vector!" )
               go to 99
            end if
            if( size(lower_bounds) /= n ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'lower_bounds' vector must have the same", &
                                  "number of elements as the parameters!" )
               go to 99
            end if
            if( .not. mfIsVector(upper_bounds) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'upper_bounds' option must be a vector!" )
               go to 99
            end if
            if( size(upper_bounds) /= n ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "'upper_bounds' vector must have the same", &
                                  "number of elements as the parameters!" )
               go to 99
            end if
         end if

         lbounds => lower_bounds%double(1,:)
         ubounds => upper_bounds%double(1,:)

      end if

      lwork = 5*n + m
      allocate( fvec(m), fjac(m,n), work(lwork) )

      allocate( residue_hist(max_iter) )
      allocate( x_hist(max_iter,n) )
      residue_hist(:) = MF_NAN
      x_hist(:,:) = MF_NAN

      if( identifiability ) then
         allocate( sensitiv_ratio(n) )
      end if

      maxfev = max_iter
      mode = 1
      call lmder_mf( fcn, jac, m, n, var, fvec, fjac, ftol,             &
                     xtol, gtol, maxfev, work(1), mode, factor,         &
                     info, nfev, njev, ipvt, work(n+1), work(2*n+1),    &
                     work(3*n+1), work(4*n+1), work(5*n+1),             &
                     max_iter, print, print_using_transf,               &
                     check_jac, print_check_jac,                        &
                     niter, residue_hist, x_hist,                       &
                     box_constrained, lbounds, ubounds,                 &
                     identifiability, sensitiv_ratio, sing_jac_tol,     &
                     print_sing_val )

      if( 1 <= info .and. info <= 3 ) then
         if( print ) then
            print "(2X,A)", "*** LMA algorithm has been terminated ***"
            select case( info )
               case( 1 )
                  print "(4X,A)", "-> ftol has been used as stopping criterion."
               case( 2 )
                  print "(4X,A)", "-> xtol has been used as stopping criterion."
               case( 3 )
                  print "(4X,A)", "-> both ftol and xtol have been used as stopping criterion."
            end select
         end if
      end if

      if( out%arg_present(3) ) then
         if( 1 <= info .and. info <= 3 ) then
            ! normal termination
            status = info
         else
            if( info == 0 ) then
               ! improper input parameters.
               status = -1
            else if( info < 0 ) then
               ! user has terminated execution (in fcn),
               ! or fdjac2 cannot compute the FD approx of the jacobian.
               status = info
            else
               ! other problem
               status = -info
            end if
            go to 99
         end if
      else ! cannot return status
         if( info < 1 .or. 3 < info ) then
            ! some problem occured
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "please use the 'status' output variable to", &
                               "investigate the problem." )
            go to 99
         end if
      end if

      if( out%arg_present(4) ) then
         res_log = .t. mf( residue_hist(1:niter) )
      end if

      if( out%arg_present(5) ) then
         p_log = x_hist(1:niter,1:n)
      end if

      if( out%arg_present(6) ) then
         ident = .t. mf( sensitiv_ratio )
      end if

      if( info < 1 .or. 3 < info ) then

         p = MF_NAN * mfOnes(1,n)
         resnorm = MF_NAN

      else

         p = var
         ! compute norm of residual, if required
         ! (mfNorm is not available within the current module!)
         resnorm = fvec
         call msAssign( resnorm, mfSqrt(mfSum(resnorm**2)) )

      end if

      if( mf_phys_units ) then

         p%units(:) = p0%units(:)

      end if

 99   continue

      call msFreeArgs( p0 )
      call msAutoRelease( p0 )

      call mf_restore_fpe( )

#endif
   end subroutine msLsqNonLin_JacUser
