! f90 include file

!_______________________________________________________________________
!
   function mfFZero( fun, x0, tol ) result( out )

      interface
         function fun( x ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      type(mfArray)                              :: x0
      real(kind=MF_DOUBLE), intent(in), optional :: tol

      type(mfArray)                              :: out
      !------ API end ------
#ifdef _DEVLP

      ! the case currently implemented is:
      !  'Root starting from an interval' [i.e. size(x0) = 2]

      real(kind=MF_DOUBLE) :: x1, x2, f1, f2, etol, val

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x0 )

      if( size(x0) < 2 .or. 2 < size(x0) ) then
         call PrintMessage( "mfFZero", "E",                             &
                            "'x0' must contain exactly two elements!" )
         go to 99
      end if

      x1 = mfGet(x0,1)
      f1 = fun(x1)
      if( f1 == 0.0d0 ) then
         go to 10
      end if

      x2 = mfGet(x0,2)
      f2 = fun(x2)
      if( f2 == 0.0d0 ) then
         x1 = x2
         go to 10
      end if

      ! fun applied to the two elements of x0 must have opposite sign
      if( f1*f2 > 0.0d0 ) then
         call PrintMessage( "mfFZero", "E",                             &
                            "'fun' applied to the two elements of 'x0'", &
                            "must have opposite sign!" )
         go to 99
      end if

      if( present(tol) ) then
         if( tol < 0.0d0 ) then
            call PrintMessage( "mfFZero", "W",                          &
                               "'tol' argument must be positive!",      &
                               "=> 'tol' has been set to twice the machine epsilon" )
            etol = 2.0d0*MF_EPS
         end if
         etol = tol
      else
         ! default tolerance
         etol = 2.0d0*MF_EPS
      end if

      ! calling the core routine
      call fzero_intv( fun, x1, x2, f1, f2, etol, val )

 10   out = x1

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x0 )
      call msAutoRelease( x0 )

      call mf_restore_fpe( )

#endif
   end function mfFZero
!_______________________________________________________________________
!
   subroutine msFZero( out, fun, x0, tol )

      interface
         function fun( x ) result( res )
            import :: MF_DOUBLE
            real(kind=MF_DOUBLE), intent(in) :: x
            real(kind=MF_DOUBLE) :: res
         end function fun
      end interface

      type(mfArray)                              :: x0
      real(kind=MF_DOUBLE), intent(in), optional :: tol

      type(mf_Out)                               :: out
      !------ API end ------
#ifdef _DEVLP

      ! driver routine for many cases

      real(kind=MF_DOUBLE) :: x1, x2, f1, f2, etol, val
      type(mfArray), pointer :: x, fval, status

   !------ end of declarations -- execution starts hereafter  ------

      ! 2 to 3 out-args must be specified
      if( out%n < 2 .or. 3 < out%n ) then
         call PrintMessage( "msFZero", "E",                             &
                            "two to three output args required!",       &
                            "syntax is: call msFZero( mfOut(<out_arg>), fun, x0 [, tol] )", &
                            "    with <out_arg> equal to {x,fval[,status]}" )
         return
      end if

      call mf_save_and_disable_fpe( )

      x => out%ptr1
      call msSilentRelease( x )
      fval => out%ptr2
      call msSilentRelease( fval )
      if( out%n == 3 ) then
         status => out%ptr3
         call msSilentRelease( status )
      end if

      call msInitArgs( x0 )

      ! 'x0' must be numeric
      if( .not. mfIsNumeric(x0) ) then
         call PrintMessage( "msFZero", "W",                             &
                            "mfArray 'x0' must be numeric!" )
         x = MF_NAN
         fval = MF_NAN
         if( out%n == 3 ) then
            status = -2 ! bad argument
         end if
         go to 99
      end if

      ! 'x0' must be dense
      if( mfIsSparse(x0) ) then
         call PrintMessage( "msFZero", "W",                             &
                            "mfArray 'x0' must have dense storage!" )
         x = MF_NAN
         fval = MF_NAN
         if( out%n == 3 ) then
            status = -2 ! bad argument
         end if
         go to 99
      end if

      if( size(x0) < 2 .or. 2 < size(x0) ) then
         call PrintMessage( "msFZero", "E",                             &
                            "'x0' must contain exactly two elements!" )
         x = MF_NAN
         fval = MF_NAN
         if( out%n == 3 ) then
            status = -2 ! bad argument
         end if
         go to 99
      end if

      x1 = mfGet(x0,1)
      f1 = fun(x1)
      if( f1 == 0.0d0 ) then
         go to 10
      end if

      x2 = mfGet(x0,2)
      f2 = fun(x2)
      if( f2 == 0.0d0 ) then
         x1 = x2
         fval = f2
         if( out%n == 3 ) then
            status = 0
         end if
         go to 10
      end if

      ! fun applied to the two elements of x0 must have opposite sign
      if( f1*f2 > 0.0d0 ) then
         call PrintMessage( "msFZero", "W",                             &
                            "'fun', applied to the two elements of 'x0'", &
                            "must have opposite sign!" )
         x = MF_NAN
         fval = MF_NAN
         if( out%n == 3 ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      if( present(tol) ) then
         if( tol < 0.0d0 ) then
            call PrintMessage( "msFZero", "W",                          &
                               "'tol' argument must be positive!",      &
                               "=> 'tol' has been set to twice the machine epsilon" )
            etol = 2.0d0*MF_EPS
         end if
         etol = tol
      else
         ! default tolerance
         etol = 2.0d0*MF_EPS
      end if

      ! calling the core routine
      call fzero_intv( fun, x1, x2, f1, f2, etol, val )
      fval = val
      if( out%n == 3 ) then
         status = 0
      end if

 10   continue

      x = x1

 99   continue

      call msFreeArgs( x0 )
      call msAutoRelease( x0 )

      call mf_restore_fpe( )

#endif
   end subroutine msFZero
!_______________________________________________________________________
!
   subroutine fzero_intv( fun, x1, x2, f1, f2, etol, val )

      real(kind=MF_DOUBLE) :: fun, x1, x2, f1, f2, etol, val
      !------ API end ------
#ifdef _DEVLP

      ! core routine:
      !   fun : user real function
      !   x1, x2 : interval range for the search

      ! we suppose that [x1,x2] is a good interval, i.e. it leads to
      ! opposite sign when the function 'fun' is applied to these two
      ! values.

      ! after convergence:
      !  - the searched value is returned in 'x1'
      !  - the function value is returned in 'val'

      ! Dekker algorithm, used in fzero of Matlab.
      ! (simplified version: see 'Numerical Computing with Matlab - C. Moler'
      !  at chapt. 4: Zeros and Roots)

      real(kind=MF_DOUBLE) :: a, b, c, fa, fb, fc, d, e
      real(kind=MF_DOUBLE) :: m, p, q, r, s, tol

   !------ end of declarations -- execution starts hereafter  ------

      a = x1
      b = x2
      fa = f1
      fb = f2
      c = a
      fc = fa
      d = b - c
      e = d

      do while( fb /= 0.0d0 )

         if( sign(1.0d0,fa) == sign(1.0d0,fb) ) then
            a = c
            fa = fc
            d = b -c
            e = d
         end if
         if( abs(fa) < abs(fb) ) then
            c = b
            b = a
            a = c
            fc = fb
            fb = fa
            fa = fc
         end if

         m = (a-b)/2.0d0
         tol = etol*max(abs(b),1.0d0)
         if( abs(m) <= tol .or. fb == 0.0d0 ) exit

         if( abs(e) < tol .or. abs(fc) <= abs(fb) ) then
            ! bisection
            d = m
            e = m
         else
            ! interpolation
            s = fb/fc
            if( a == c ) then
               ! linear interpolation (secant)
               p = 2.0d0*m*s
               q = 1.0d0 - s
            else
               ! inverse quadratic interpolation
               q = fc/fa
               r = fb/fa
               p = s*( 2.0d0*m*q*(q-r) - (b-c)*(r-1.0d0) )
               q = (q-1.0d0)*(r-1.0d0)*(s-1.0d0)
            end if
            if( p > 0.0d0 ) then
               q = -q
            else
               p = -p
            end if
            if( 2.0d0*p < 3.0d0*m*q - abs(tol*q) .and. p < abs(0.5d0*e*q) ) then
               e = d
               d = p/q
            else
               d = m
               e = m
            end if
         end if

         ! next point
         c = b
         fc = fb
         if( abs(d) > tol ) then
            b = b + d
         else
            b = b - sign(1.0d0,b-a)*tol
         end if
         fb = fun(b)

      end do

      x1 = b
      val = fb

#endif
   end subroutine fzero_intv
