! f90 include file

!_______________________________________________________________________
!
   function mfFSolve_JacDF( fcn, n, x0,                                 &
                            options )                                   &
   result( out )

      use minpack

      interface
         subroutine fcn( n, x, fvec, flag )
            import :: MF_DOUBLE
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: fvec(n)
            integer                          :: flag
         end subroutine fcn
      end interface

      integer, intent(in) :: n
      type(mfArray)       :: x0
      type(mf_NL_Options), target, optional :: options

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Find a zero of a system of n nonlinear functions of n variables.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, x0(:) is an initial guess
      !
      ! Solved by the 'hybrd' routine of MINPACK

      type(mfArray), pointer :: tol => null(),                          &
                                epsfcn => null()
      type(mfArray), target  :: empty
      integer                :: max_iter
      logical                :: print

      integer :: info, lwork
      real(kind=MF_DOUBLE) :: xtol, ftol, epsfcn2
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec(:), work(:)

      integer :: index, lr, maxfev, ml, mode, mu, nfev
      real(kind=MF_DOUBLE), parameter :: factor = 1.0d2
      character(len=256) :: format

      character(len=2) :: status_str

      character(len=*), parameter :: ROUTINE_NAME = "mfFSolve"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x0 )

      if( present(options) ) then
         tol     => options%tol
         max_iter = options%max_iter
         epsfcn  => options%epsfcn
         print    = options%print
      else
         tol     => empty
         max_iter = 50
         epsfcn  => empty
         print    = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( all(tol <= 0.0d0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'tol' argument must be strictly positive!" )
            go to 99
         end if
         if( size(tol) == 1 ) then
            xtol = tol
            ftol = tol
         else
            if( size(tol) == 2 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must", &
                                  "be of length 2!" )
               go to 99
            end if
         end if
      else
         xtol = 1.0d-9
         ftol = 1.0d-9
      end if

      if( mfIsEmpty(x0) ) then
         go to 99
      end if

      ! 'x0' must be numeric
      if( .not. mfIsNumeric(x0) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "mfArray 'x0' must be numeric!" )
         go to 99
      end if

      ! 'x0' must be dense
      if( mfIsSparse(x0) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "mfArray 'x0' must have dense storage!" )
         go to 99
      end if

      ! 'x0' must be a vector
      if( .not. mfIsVector(x0) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'x0' must be a vector!" )
         go to 99
      end if

      ! initial guess
      var = x0

      ! set format for optional output
      if( print ) then
         write(format,"(I0)") n
         format = "( '  *** iter = ', I0, ' ***', /, '  x components ='," &
                  // " /, " // trim(format) // "(2X,ES13.6), /,"        &
                  // " '  function norm = ', ES10.3, / )"
      end if

      if( mfIsEmpty(epsfcn) ) then
         epsfcn2 = 0.0d0
      else
         epsfcn2 = epsfcn%double(1,1)
      end if

      maxfev = max_iter*(n+1)
      ! ====================
      ! a priori, the jacobian is not banded, so the following initializations
      ! are used:
      ml = n - 1
      mu = n - 1
      ! for a banded jacobian matrix, the interface must be modified to take
      ! into account additional information from the user!
      ! ====================
      mode = 1
      lr = (n*(n+1))/2
      index = 6*n + lr
      ! lwork is the sum of the length of 8 working arrays:
      !   diag, qtf, wa1, wa2, wa3, wa4 (each of length n)
      !   r (length: lr)
      !   fjac (length: n^2)
      lwork = index + n**2
      allocate( fvec(n), work(lwork) )

      call hybrd( fcn, n, var, fvec, xtol, ftol, maxfev, ml, mu,        &
                  epsfcn2, work(1), mode, factor, info, nfev,           &
                  work(index+1), work(6*n+1), lr, work(n+1),            &
                  work(2*n+1), work(3*n+1), work(4*n+1), work(5*n+1),   &
                  max_iter, print, format )

      if( info == 5 ) info = 4

      ! info=1 and info=6 are both good
      if( info /= 1 .and. info /= 6 ) then
         ! some problem occured
         ! (returns the same status as the subroutine version)
         write(status_str,'(I0)') -(info+1)
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "[minpack]/hybrd routine failed.",          &
                            "for your information: status = " // status_str )
         go to 99
      end if

      out = var

      if( mf_phys_units ) then

         out%units(:) = x0%units(:)

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x0 )
      call msAutoRelease( x0 )

      call mf_restore_fpe( )

#endif
   end function mfFSolve_JacDF
!_______________________________________________________________________
!
   subroutine msFSolve_JacDF( out, fcn, n, x0,                          &
                              options )

      use minpack

      interface
         subroutine fcn( n, x, fvec, flag )
            import :: MF_DOUBLE
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: fvec(n)
            integer                          :: flag
         end subroutine fcn
      end interface

      integer, intent(in) :: n
      type(mfArray)       :: x0
      type(mf_NL_Options), target, optional :: options

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Find a zero of a system of n nonlinear functions of n variables.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, x0(:) is an initial guess
      !
      ! Solved by the 'hybrd' routine of MINPACK

      type(mfArray), pointer :: tol => null(),                          &
                                epsfcn => null()
      type(mfArray), target  :: empty
      integer                :: max_iter
      logical                :: print

      integer :: info, lwork
      real(kind=MF_DOUBLE) :: xtol, ftol, epsfcn2
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec_1(:), work(:)

      integer :: index, lr, maxfev, ml, mode, mu, nfev
      real(kind=MF_DOUBLE), parameter :: factor = 1.0d2
      character(len=256) :: format

      type(mfArray), pointer :: x, fvec, status

      character(len=*), parameter :: ROUTINE_NAME = "msFSolve"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x0 )

      ! 1 to 3 out-args must be specified
      ! 2 are optional, only the first one can be equal to MF_NO_ARG
      if( out%n < 1 .or. 3 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to three output args required!",      &
                            "syntax is: call msFSolve( mfOut(<out_arg>), fcn, n, x0 [, options, jac] )", &
                            "    with <out_arg> equal to {x[,fvec,status]}" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, x0 ) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      x => out%ptr1
      call msSilentRelease( x )
      if( out%arg_present(2) ) then
         fvec => out%ptr2
         call msSilentRelease( fvec )
      end if
      if( out%arg_present(3) ) then
         status => out%ptr3
         call msSilentRelease( status )
      end if

      if( present(options) ) then
         tol     => options%tol
         max_iter = options%max_iter
         epsfcn  => options%epsfcn
         print    = options%print
      else
         tol     => empty
         max_iter = 50
         epsfcn  => empty
         print    = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( all(tol <= 0.0d0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'tol' argument must be strictly positive!" )
            x = MF_NAN
            if( out%arg_present(2) ) then
               fvec = MF_NAN
            end if
            if( out%arg_present(3) ) then
               status = -2 ! bad argument
            end if
            go to 99
         end if
         if( size(tol) == 1 ) then
            xtol = tol
            ftol = tol
         else
            if( size(tol) == 2 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must", &
                                  "be of length 2!" )
               x = MF_NAN
               if( out%arg_present(2) ) then
                  fvec = MF_NAN
               end if
               if( out%arg_present(3) ) then
                  status = -2 ! bad argument
               end if
               go to 99
            end if
         end if
      else
         xtol = 1.0d-9
         ftol = 1.0d-9
      end if

      if( mfIsEmpty(x0) ) then
         go to 99
      end if

      ! 'x0' must be numeric
      if( .not. mfIsNumeric(x0) ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "mfArray 'x0' must be numeric!" )
         x = MF_NAN
         if( out%arg_present(2) ) then
            fvec = MF_NAN
         end if
         if( out%arg_present(3) ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      ! 'x0' must be dense
      if( mfIsSparse(x0) ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "mfArray 'x0' must have dense storage!" )
         x = MF_NAN
         if( out%arg_present(2) ) then
            fvec = MF_NAN
         end if
         if( out%arg_present(3) ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      ! 'x0' must be a vector
      if( .not. mfIsVector(x0) ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "'x0' must be a vector!" )
         x = MF_NAN
         if( out%arg_present(2) ) then
            fvec = MF_NAN
         end if
         if( out%arg_present(3) ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      ! initial guess
      var = x0

      ! set format for optional output
      if( print ) then
         write(format,"(I0)") n
         format = "( '  *** iter = ', I0, ' ***', /, '  x components ='," &
                  // " /, " // trim(format) // "(2X,ES13.6), /,"        &
                  // " '  function norm = ', ES10.3, / )"
      end if

      if( mfIsEmpty(epsfcn) ) then
         epsfcn2 = 0.0d0
      else
         epsfcn2 = epsfcn%double(1,1)
      end if

      maxfev = max_iter*(n+1)
      ! ====================
      ! a priori, the jacobian is not banded, so the following initializations
      ! are used:
      ml = n - 1
      mu = n - 1
      ! for a banded jacobian matrix, the interface must be modified to take
      ! into account additional information from the user!
      ! ====================
      mode = 1
      lr = (n*(n+1))/2
      index = 6*n + lr
      ! lwork is the sum of the length of 8 working arrays:
      !   diag, qtf, wa1, wa2, wa3, wa4 (each of length n)
      !   r (length: lr)
      !   fjac (length: n^2)
      lwork = index + n**2
      allocate( fvec_1(n), work(lwork) )

      call hybrd( fcn, n, var, fvec_1, xtol, ftol, maxfev, ml, mu,      &
                  epsfcn2, work(1), mode, factor, info, nfev,           &
                  work(index+1), work(6*n+1), lr, work(n+1),            &
                  work(2*n+1), work(3*n+1), work(4*n+1), work(5*n+1),   &
                  max_iter, print, format )

      if( info == 5 ) info = 4

      if( out%arg_present(3) ) then
         if( info == 1 ) then
            ! [minpack]/hybrd succeeded (xtol)
            status = 1
         else if( info == 6 ) then
            ! [minpack]/hybrj succeeded (ftol)
            status = 2
         else if( info < 0 ) then
            ! the user has terminated the execution via the 'flag'
            ! argument of the subroutine 'fcn'
            status = -10
         else
            ! [minpack]/hybrd failed
            status = -(info+1)
         end if
      end if

      if( info /= 1 .and. info /= 6 ) then

         x = MF_NAN * mfOnes(1,n)
         if( out%arg_present(2) ) then
            fvec = MF_NAN * mfOnes(1,n)
         end if

      else

         x = var
         if( out%arg_present(2) ) then
            fvec = fvec_1
         end if

      end if

      if( mf_phys_units ) then

         x%units(:) = x0%units(:)

      end if

 99   continue

      call msFreeArgs( x0 )
      call msAutoRelease( x0 )

      call mf_restore_fpe( )

#endif
   end subroutine msFSolve_JacDF
!_______________________________________________________________________
!
   function mfFSolve_JacUser( fcn, n, x0,                               &
                              options, jac )                            &
   result( out )

      use minpack

      interface
         subroutine fcn( n, x, fvec, flag )
            import :: MF_DOUBLE
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: fvec(n)
            integer                          :: flag
         end subroutine fcn
         subroutine jac( n, x, jacobian )
            import :: MF_DOUBLE
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: jacobian(n,n)
         end subroutine jac
      end interface

      integer, intent(in) :: n
      type(mfArray)       :: x0
      type(mf_NL_Options), target, optional :: options

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Find a zero of a system of n nonlinear functions of n variables.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, x0(:) is an initial guess
      !
      ! Solved by the 'hybrj_mf' routine of MINPACK

      type(mfArray), pointer :: tol => null()
      type(mfArray), target  :: empty
      integer                :: max_iter
      logical                :: print
      integer                :: check_jac, print_check_jac

      integer :: info, lwork
      real(kind=MF_DOUBLE) :: xtol, ftol
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec(:), fjac(:,:), work(:)

      integer :: lr, maxfev, mode, nfev, njev
      real(kind=MF_DOUBLE), parameter :: factor = 1.0d2
      character(len=256) :: format

      character(len=2) :: status_str

      character(len=*), parameter :: ROUTINE_NAME = "mfFSolve"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x0 )

      if( present(options) ) then
         tol            => options%tol
         max_iter        = options%max_iter
         print           = options%print
         check_jac       = options%check_jac
         print_check_jac = options%print_check_jac
      else
         tol            => empty
         max_iter        = 50
         print           = .false.
         check_jac       = 0
         print_check_jac = 0
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( all(tol <= 0.0d0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'tol' argument must be strictly positive!" )
            go to 99
         end if
         if( size(tol) == 1 ) then
            xtol = tol
            ftol = tol
         else
            if( size(tol) == 2 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must", &
                                  "be of length 2!" )
               go to 99
            end if
         end if
      else
         xtol = 1.0d-9
         ftol = 1.0d-9
      end if

      if( mfIsEmpty(x0) ) then
         go to 99
      end if

      ! 'x0' must be numeric
      if( .not. mfIsNumeric(x0) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "mfArray 'x0' must be numeric!" )
         go to 99
      end if

      ! 'x0' must be dense
      if( mfIsSparse(x0) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "mfArray 'x0' must have dense storage!" )
         go to 99
      end if

      ! 'x0' must be a vector
      if( .not. mfIsVector(x0) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'x0' must be a vector!" )
         go to 99
      end if

      ! initial guess
      var = x0

      ! set format for optional output
      if( print ) then
         write(format,"(I0)") n
         format = "( '  *** iter = ', I0, ' ***', /, '  x components ='," &
                  // " /, " // trim(format) // "(2X,ES13.6), /,"        &
                  // " '  function norm = ', ES10.3, / )"
      end if

      maxfev = max_iter
      mode = 1
      lr = (n*(n+1))/2
      ! lwork is the sum of the length of 7 working arrays:
      !   diag, qtf, wa1, wa2, wa3, wa4 (each of length n)
      !   r (length: lr)
      lwork = 6*n + lr
      allocate( fvec(n), fjac(n,n), work(lwork) )

      call hybrj_mf( fcn, jac, n, var, fvec, fjac, xtol, ftol, maxfev, work(1), &
                     mode, factor, info, nfev, njev, work(6*n+1), lr,   &
                     work(n+1), work(2*n+1), work(3*n+1), work(4*n+1),  &
                     work(5*n+1),                                       &
                     max_iter, print, check_jac, print_check_jac, format )

      if( info == 5 ) info = 4

      ! info=1 and info=6 are both good
      if( info /= 1 .and. info /= 6 ) then
         ! some problem occured
         ! (returns the same status as the subroutine version)
         write(status_str,'(I0)') -(info+1)
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "[minpack]/hybrj_mf routine failed.",       &
                            "for your information: info = " // status_str )
         go to 99
      end if

      out = var

      if( mf_phys_units ) then

         out%units(:) = x0%units(:)

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x0 )
      call msAutoRelease( x0 )

      call mf_restore_fpe( )

#endif
   end function mfFSolve_JacUser
!_______________________________________________________________________
!
   subroutine msFSolve_JacUser( out, fcn, n, x0,                        &
                                options, jac )

      use minpack

      interface
         subroutine fcn( n, x, fvec, flag )
            import :: MF_DOUBLE
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: fvec(n)
            integer                          :: flag
         end subroutine fcn
         subroutine jac( n, x, jacobian )
            import :: MF_DOUBLE
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: jacobian(n,n)
         end subroutine jac
      end interface

      integer, intent(in) :: n
      type(mfArray)       :: x0
      type(mf_NL_Options), target, optional :: options

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Find a zero of a system of n nonlinear functions of n variables.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, x0(:) is an initial guess
      !
      ! Solved by the 'hybrj_mf' routine of MINPACK

      type(mfArray), pointer :: tol => null()
      type(mfArray), target  :: empty
      integer                :: max_iter
      logical                :: print
      integer                :: check_jac, print_check_jac

      integer :: info, lwork
      real(kind=MF_DOUBLE) :: xtol, ftol
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec_1(:), fjac(:,:), work(:)

      integer :: lr, maxfev, mode, nfev, njev
      real(kind=MF_DOUBLE), parameter :: factor = 1.0d2
      character(len=256) :: format

      type(mfArray), pointer :: x, fvec, status

      character(len=*), parameter :: ROUTINE_NAME = "msFSolve"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x0 )

      ! 1 to 3 out-args must be specified
      ! 2 are optional, only the first one can be equal to MF_NO_ARG
      if( out%n < 1 .or. 3 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to three output args required!",      &
                            "syntax is: call msFSolve( mfOut(<out_arg>), fcn, n, x0 [, options, jac] )", &
                            "    with <out_arg> equal to {x[,fvec,status]}" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, x0 ) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      x => out%ptr1
      call msSilentRelease( x )
      if( out%arg_present(2) ) then
         fvec => out%ptr2
         call msSilentRelease( fvec )
      end if
      if( out%arg_present(3) ) then
         status => out%ptr3
         call msSilentRelease( status )
      end if

      if( present(options) ) then
         tol            => options%tol
         max_iter        = options%max_iter
         print           = options%print
         check_jac       = options%check_jac
         print_check_jac = options%print_check_jac
      else
         tol            => empty
         max_iter        = 50
         print           = .false.
         check_jac       = 0
         print_check_jac = 0
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( all(tol <= 0.0d0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'tol' argument must be strictly positive!" )
            x = MF_NAN
            if( out%arg_present(2) ) then
               fvec = MF_NAN
            end if
            if( out%arg_present(3) ) then
               status = -2 ! bad argument
            end if
            go to 99
         end if
         if( size(tol) == 1 ) then
            xtol = tol
            ftol = tol
         else
            if( size(tol) == 2 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must", &
                                  "be of length 2!" )
               x = MF_NAN
               if( out%arg_present(2) ) then
                  fvec = MF_NAN
               end if
               if( out%arg_present(3) ) then
                  status = -2 ! bad argument
               end if
               go to 99
            end if
         end if
      else
         xtol = 1.0d-9
         ftol = 1.0d-9
      end if

      if( mfIsEmpty(x0) ) then
         go to 99
      end if

      ! 'x0' must be numeric
      if( .not. mfIsNumeric(x0) ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "mfArray 'x0' must be numeric!" )
         x = MF_NAN
         if( out%arg_present(2) ) then
            fvec = MF_NAN
         end if
         if( out%arg_present(3) ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      ! 'x0' must be dense
      if( mfIsSparse(x0) ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "mfArray 'x0' must have dense storage!" )
         x = MF_NAN
         if( out%arg_present(2) ) then
            fvec = MF_NAN
         end if
         if( out%arg_present(3) ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      ! 'x0' must be a vector
      if( .not. mfIsVector(x0) ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "'x0' must be a vector!" )
         x = MF_NAN
         if( out%arg_present(2) ) then
            fvec = MF_NAN
         end if
         if( out%arg_present(3) ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      ! initial guess
      var = x0

      ! set format for optional output
      if( print ) then
         write(format,"(I0)") n
         format = "( '  *** iter = ', I0, ' ***', /, '  x components ='," &
                  // " /, " // trim(format) // "(2X,ES13.6), /,"        &
                  // " '  function norm = ', ES10.3, / )"
      end if

      maxfev = max_iter
      mode = 1
      lr = (n*(n+1))/2
      ! lwork is the sum of the length of 7 working arrays:
      !   diag, qtf, wa1, wa2, wa3, wa4 (each of length n)
      !   r (length: lr)
      lwork = 6*n + lr
      allocate( fvec_1(n), fjac(n,n), work(lwork) )

      call hybrj_mf( fcn, jac, n, var, fvec_1, fjac, xtol, ftol, maxfev, work(1), &
                     mode, factor, info, nfev, njev, work(6*n+1), lr,   &
                     work(n+1), work(2*n+1), work(3*n+1), work(4*n+1),  &
                     work(5*n+1),                                       &
                     max_iter, print, check_jac, print_check_jac, format )

      if( info == 5 ) info = 4

      if( out%arg_present(3) ) then
         if( info == 1 ) then
            ! [minpack]/hybrj_mf succeeded (xtol)
            status = 1
         else if( info == 6 ) then
            ! [minpack]/hybrj_mf succeeded (ftol)
            status = 2
         else if( info < 0 ) then
            ! the user has terminated the execution via the 'flag'
            ! argument of the subroutine 'fcn'
            status = -10
         else
            ! [minpack]/hybrj_mf failed
            status = -(info+1)
         end if
      end if

      if( info /= 1 .and. info /= 6 ) then

         x = MF_NAN * mfOnes(1,n)
         if( out%arg_present(2) ) then
            fvec = MF_NAN * mfOnes(1,n)
         end if

      else

         x = var
         if( out%arg_present(2) ) then
            fvec = fvec_1
         end if

      end if

      if( mf_phys_units ) then

         x%units(:) = x0%units(:)

      end if

 99   continue

      call msFreeArgs( x0 )
      call msAutoRelease( x0 )

      call mf_restore_fpe( )

#endif
   end subroutine msFSolve_JacUser
!_______________________________________________________________________
!
   function mfFSolve_JacUserSP( fcn, n, x0,                             &
                                options, jac, sparse )                  &
   result( out )

      use minpack

      interface
         subroutine fcn( n, x, fvec, flag )
            import :: MF_DOUBLE
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: fvec(n)
            integer                          :: flag
         end subroutine fcn
         subroutine jac( n, x, job, pd, ipd, jpd, nnz )
            import :: MF_DOUBLE
            integer,              intent(in) :: n, job
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: pd(*)
            integer                          :: ipd(*), jpd(*), nnz
         end subroutine jac
      end interface

      integer, intent(in) :: n
      type(mfArray)       :: x0
      type(mf_NL_Options), target, optional :: options
      logical, intent(in) :: sparse

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Find a zero of a system of n nonlinear functions of n variables.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, x0(:) is an initial guess
      !
      ! Solved by the 'hybrj_sp' routine of MINPACK

      type(mfArray), pointer :: tol => null()
      type(mfArray), target  :: empty
      integer                :: max_iter
      logical                :: print

      integer :: info, lwork
      real(kind=MF_DOUBLE) :: xtol, ftol
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec(:)

      integer :: maxfev, nfev, njev
      character(len=256) :: format

      character(len=2) :: status_str

      character(len=*), parameter :: ROUTINE_NAME = "mfFSolve"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x0 )

      if( present(options) ) then
         tol     => options%tol
         max_iter = options%max_iter
         print    = options%print
      else
         tol     => empty
         max_iter = 50
         print    = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( all(tol <= 0.0d0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'tol' argument must be strictly positive!" )
            go to 99
         end if
         if( size(tol) == 1 ) then
            xtol = tol
            ftol = tol
         else
            if( size(tol) == 2 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must", &
                                  "be of length 2!" )
               go to 99
            end if
         end if
      else
         xtol = 1.0d-9
         ftol = 1.0d-9
      end if

      if( mfIsEmpty(x0) ) then
         go to 99
      end if

      ! 'x0' must be numeric
      if( .not. mfIsNumeric(x0) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "mfArray 'x0' must be numeric!" )
         go to 99
      end if

      ! 'x0' must be dense
      if( mfIsSparse(x0) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "mfArray 'x0' must have dense storage!" )
         go to 99
      end if

      ! 'x0' must be a vector
      if( .not. mfIsVector(x0) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'x0' must be a vector!" )
         go to 99
      end if

      ! initial guess
      var = x0

      ! set format for optional output
      if( print ) then
         write(format,"(I0)") n
         format = "( '  *** iter = ', I0, ' ***', /, '  x components ='," &
                  // " /, " // trim(format) // "(2X,ES13.6), /,"        &
                  // " '  function norm = ', ES10.3, / )"
      end if

      maxfev = max_iter
      allocate( fvec(n) )

      call hybrj_sp( fcn, jac, n, var, fvec, xtol, ftol, maxfev,        &
                     info, nfev, njev, max_iter, print, format )

      if( info == 5 ) info = 4

      ! info=1 and info=6 are both good
      if( info /= 1 .and. info /= 6 ) then
         ! some problem occured
         ! (returns the same status as the subroutine version)
         write(status_str,'(I0)') -(info+1)
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "[minpack]/hybrj_sp routine failed.",       &
                            "for your information: info = " // status_str )
         go to 99
      end if

      out = var

      if( mf_phys_units ) then

         out%units(:) = x0%units(:)

      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( x0 )
      call msAutoRelease( x0 )

      call mf_restore_fpe( )

#endif
   end function mfFSolve_JacUserSP
!_______________________________________________________________________
!
   subroutine msFSolve_JacUserSP( out, fcn, n, x0,                      &
                                  options, jac, sparse )

      use minpack

      interface
         subroutine fcn( n, x, fvec, flag )
            import :: MF_DOUBLE
            integer,              intent(in) :: n
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: fvec(n)
            integer                          :: flag
         end subroutine fcn
         subroutine jac( n, x, job, pd, ipd, jpd, nnz )
            import :: MF_DOUBLE
            integer,              intent(in) :: n, job
            real(kind=MF_DOUBLE), intent(in) :: x(n)
            real(kind=MF_DOUBLE)             :: pd(*)
            integer                          :: ipd(*), jpd(*), nnz
         end subroutine jac
      end interface

      integer, intent(in) :: n
      type(mfArray)       :: x0
      type(mf_NL_Options), target, optional :: options
      logical, intent(in) :: sparse

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      ! Find a zero of a system of n nonlinear functions of n variables.
      !
      ! fcn : the user-supplied subroutine which calculate the functions.
      !
      ! at input, x0(:) is an initial guess
      !
      ! Solved by the 'hybrj_sp' routine of MINPACK

      type(mfArray), pointer :: tol => null()
      type(mfArray), target  :: empty
      integer                :: max_iter
      logical                :: print
      ! REUSE_SPJAC_STRUCT is a logical (global variable) defined in the
      ! 'minpack' module

      integer :: info, lwork
      real(kind=MF_DOUBLE) :: xtol, ftol
      real(kind=MF_DOUBLE) :: var(n)
      real(kind=MF_DOUBLE), allocatable :: fvec_1(:), fjac(:,:), work(:)

      integer :: lr, maxfev, mode, nfev, njev
      real(kind=MF_DOUBLE), parameter :: factor = 1.0d2
      character(len=256) :: format

      type(mfArray), pointer :: x, fvec, status

      character(len=*), parameter :: ROUTINE_NAME = "msFSolve"

   !------ end of declarations -- execution starts hereafter  ------

      call mf_save_and_disable_fpe( )

      call msInitArgs( x0 )

      ! 1 to 3 out-args must be specified
      ! 2 are optional, only the first one can be equal to MF_NO_ARG
      if( out%n < 1 .or. 3 < out%n ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "two to three output args required!",      &
                            "syntax is: call msFSolve( mfOut(<out_arg>), fcn, n, x0 [, options, jac, sparse] )", &
                            "    with <out_arg> equal to {x[,fvec,status]}" )
         go to 99
      end if

      ! check args of mfOut
      if( .not. args_mfout_ok( out, x0 ) ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      x => out%ptr1
      call msSilentRelease( x )
      if( out%arg_present(2) ) then
         fvec => out%ptr2
         call msSilentRelease( fvec )
      end if
      if( out%arg_present(3) ) then
         status => out%ptr3
         call msSilentRelease( status )
      end if

      if( present(options) ) then
         tol               => options%tol
         max_iter           = options%max_iter
         print              = options%print
         REUSE_SPJAC_STRUCT = options%reuse_spjac_struct
      else
         tol               => empty
         max_iter           = 50
         print              = .false.
         REUSE_SPJAC_STRUCT = .false.
      end if

      if( .not. mfIsEmpty(tol) ) then
         if( all(tol <= 0.0d0) ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'tol' argument must be strictly positive!" )
            x = MF_NAN
            if( out%arg_present(2) ) then
               fvec = MF_NAN
            end if
            if( out%arg_present(3) ) then
               status = -2 ! bad argument
            end if
            go to 99
         end if
         if( size(tol) == 1 ) then
            xtol = tol
            ftol = tol
         else
            if( size(tol) == 2 ) then
               ftol = mfGet( tol, 1 )
               xtol = mfGet( tol, 2 )
            else
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "when it is a vector mfArray, 'tol' must", &
                                  "be of length 2!" )
               x = MF_NAN
               if( out%arg_present(2) ) then
                  fvec = MF_NAN
               end if
               if( out%arg_present(3) ) then
                  status = -2 ! bad argument
               end if
               go to 99
            end if
         end if
      else
         xtol = 1.0d-9
         ftol = 1.0d-9
      end if

      if( mfIsEmpty(x0) ) then
         go to 99
      end if

      ! 'x0' must be numeric
      if( .not. mfIsNumeric(x0) ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "mfArray 'x0' must be numeric!" )
         x = MF_NAN
         if( out%arg_present(2) ) then
            fvec = MF_NAN
         end if
         if( out%arg_present(3) ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      ! 'x0' must be dense
      if( mfIsSparse(x0) ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "mfArray 'x0' must have dense storage!" )
         x = MF_NAN
         if( out%arg_present(2) ) then
            fvec = MF_NAN
         end if
         if( out%arg_present(3) ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      ! 'x0' must be a vector
      if( .not. mfIsVector(x0) ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "'x0' must be a vector!" )
         x = MF_NAN
         if( out%arg_present(2) ) then
            fvec = MF_NAN
         end if
         if( out%arg_present(3) ) then
            status = -1 ! bad initialization
         end if
         go to 99
      end if

      ! initial guess
      var = x0

      ! set format for optional output
      if( print ) then
         write(format,"(I0)") n
         format = "( '  *** iter = ', I0, ' ***', /, '  x components ='," &
                  // " /, " // trim(format) // "(2X,ES13.6), /,"        &
                  // " '  function norm = ', ES10.3, / )"
      end if

      maxfev = max_iter
      allocate( fvec_1(n) )

      call hybrj_sp( fcn, jac, n, var, fvec_1, xtol, ftol, maxfev,      &
                     info, nfev, njev, max_iter, print, format )

      if( info == 5 ) info = 4

      if( out%arg_present(3) ) then
         if( info == 1 ) then
            ! [minpack]/hybrj_sp succeeded (xtol)
            status = 1
         else if( info == 6 ) then
            ! [minpack]/hybrj_sp succeeded (ftol)
            status = 2
         else if( info < 0 ) then
            ! the user has terminated the execution via the 'flag'
            ! argument of the subroutine 'fcn'
            status = -10
         else
            ! [minpack]/hybrj_sp failed
            status = -(info+1)
         end if
      end if

      if( info /= 1 .and. info /= 6 ) then

         x = MF_NAN * mfOnes(1,n)
         if( out%arg_present(2) ) then
            fvec = MF_NAN * mfOnes(1,n)
         end if

      else

         x = var
         if( out%arg_present(2) ) then
            fvec = fvec_1
         end if

      end if

      if( mf_phys_units ) then

         x%units(:) = x0%units(:)

      end if

 99   continue

      call msFreeArgs( x0 )
      call msAutoRelease( x0 )

      call mf_restore_fpe( )

#endif
   end subroutine msFSolve_JacUserSP
