! f90 include file

!_______________________________________________________________________
!
   subroutine msSaveHDF5( filename, x, name,                            &
                          file_access, mfarray_overwrite, status )

      character(len=*), intent(in) :: filename
      type(mfArray) :: x
      character(len=*), intent(in), optional :: name
      character(len=*), intent(in), optional :: file_access
      logical, intent(in), optional :: mfarray_overwrite
      integer, intent(out), optional :: status
      !------ API end ------
#ifdef _DEVLP

      ! By default (or if file_access="trunc"), this routine
      ! creates a new HDF5 file (or overwrites a pre-existing one) and
      ! writes the mfArray 'x' in it.
      ! If the optional arg file_access="append", it tries to add
      ! in the existing HDF5 file a new group for 'x'. In the case
      ! where a group with the same name already exists, the group will
      ! be overwritten only if the other optional arg mfarray_overwrite
      ! is true; in the contrary, it will return a non zero value in
      ! the status flag.
      !
      ! The mfArray 'x' is written as one group (named from the
      ! optional arg 'name', or "untitled mfArray" by default),
      ! with all its MUESLI properties, stored as group attributes.
      !
      ! Attributes : MUESLI version
      !              data_type
      !              nrow
      !              ncol        (*)
      !              tril        (*)
      !              triu        (*)
      !              symm        (*)
      !              pos_def     (*)
      !              nnz         (**)
      !              row_sorted  (**)
      !
      !              (*)  (for all data type except permutation vector)
      !              (**) (only if data_type=sparse)
      !
      ! Physical units are added under the same group, beside the data;
      ! they are written if the mfArray is nondimensional, and if the
      ! MUESLI flag 'mf_phys_units' is TRUE.
      ! Physical units are saved in a vector of 'num_base_units'
      ! real exponents. Thus, the internal structure (using rational
      ! numbers) is not transfered.
      !
      ! Data is always compressed (dense or sparse matrix) with full
      ! chunk (i.e. chunk dims = data dims), and deflate_level=1

      integer :: len, ncol, nnz
      integer :: hdferr, i, nmembers
      integer(hid_t) :: file_id   ! HDF5 file identifier
      integer(hid_t) :: group_id  ! HDF5 group identifier
      integer(hid_t) :: attr_id   ! HDF5 attribute identifier
      integer(hid_t) :: atype_id  ! HDF5 attribute Dataspace identifier
      integer(hid_t) :: dset_id   ! HDF5 dataset identifier
      integer(hid_t) :: dspace_id ! HDF5 dataspace identifier
      integer(hid_t) :: dcpl_id   ! HDF5 dataset creation prop. list id
      integer :: rank
      integer :: rank_chunk
      integer :: deflate_level
      integer(hsize_t), dimension(2) :: data_dims ! dataset dimensions
      integer(hsize_t), dimension(2) :: chunk_dims ! chunk dimensions
      character(len=20) :: obj_name ! Buffer to hold object's name
      integer :: obj_type ! Type of the object
      character(len=32) :: groupname ! group name
      character(len=32) :: default_groupname = "untitled mfArray"
      character(len=6) :: file_access0
      logical :: mfarray_overwrite0, found
      real(kind=MF_DOUBLE) :: r_units(num_base_units)
      integer :: istatus, itmp
      character(len=8) :: muesli_version
      character(len=*), parameter :: ROUTINE_NAME = "msSaveHDF5"

   !------ end of declarations -- execution starts hereafter  ------

      if( len_trim(filename) == 0 ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "'filename' argument is an empty string!" )
         return
      end if

      call msInitArgs( x )

      ! warns if proposed extension is not '.h5'
      len = len_trim( filename )
      if( filename(len-2:len) /= ".h5" ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "your file should have the '.h5' extension" )
      end if

      ! initializes the HDF5 F90 interface
      call h5open_f( hdferr )
      if( hdferr /= 0 ) then
         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "cannot initialize the HDF5 F90 interface!" )
         go to 99
      end if

      if( present(name) ) then
         groupname = name
      else
         groupname = default_groupname
      end if

      if( present(file_access) ) then
         file_access0 = file_access
      else
         file_access0 = "trunc"
      end if

      if( present(mfarray_overwrite) ) then
         mfarray_overwrite0 = mfarray_overwrite
      else
         mfarray_overwrite0 = .false.
      end if

      if( present(status) ) then
         status = 0
      end if

      if( to_lower(file_access0) == "trunc" ) then
         ! creates a new file using default properties
         call h5fcreate_f( filename, H5F_ACC_TRUNC_F, file_id, hdferr )
      else
         ! opens an existing file
         call h5fopen_f( filename, H5F_ACC_RDWR_F, file_id, hdferr )

         ! detects if a group with the same name already exists
         ! (a direct inquiry function doesn't exist; we must iterate
         !  over the members of the root)

         ! gets number of members in the root group.
         call h5gn_members_f( file_id, "/", nmembers, hdferr )
         ! gets each group member's name and type
         found = .false.
         ! idx begins at 0 !
         do i = 0, nmembers-1
            call h5gget_obj_info_idx_f( file_id, "/", i, obj_name,      &
                                        obj_type, hdferr )
            if( obj_type == H5G_GROUP_F .and.                           &
                trim(obj_name) == trim(groupname) ) then
               found = .true.
               exit
            end if
         end do
         if( found ) then
            if( mfarray_overwrite0 ) then
! warning, in the current implementation of the HDF5 library
! freed space from objects is not re-used (?).
               call PrintMessage( "msSaveHDF5", "W",                    &
                                  "an object is being to be deleted.",  &
                                  "Actually, just the entry for the name will be unlinked.", &
                                  "(HDF5-1.6.x don't re-use the memory space;", &
                                  "see doc about the 'h5gunlink_f' routine)" )
               ! deletes the object (the group)
               call h5gunlink_f( file_id, trim(groupname), hdferr )
            else
               if( present(status) ) then
                  status = 1
               else
                  call PrintMessage( "msSaveHDF5", "E",                 &
                                     "don't write over an existing group!", &
                                     "If you want to force and overwrite, you should", &
                                     "use the 'mfarray_overwrite' optional arg." )
               end if
               go to 59
            end if
         end if
      end if

      ! can now safely create a group
      call h5gcreate_f( file_id, groupname, group_id, hdferr )

      ! creates scalar dataspace for the attribute.
      call h5screate_f( H5S_SCALAR_F, dspace_id, hdferr )

      ! BEGIN "MF_MUESLI_VERSION"
      ! creates datatype for the attribute.
      call h5tcopy_f( H5T_NATIVE_CHARACTER, atype_id, hdferr )
      ! (size of MF_MUESLI_VERSION : 8 characters => 8 bytes)
      call h5tset_size_f( atype_id, 8, hdferr )
      call h5acreate_f( group_id, "MF_MUESLI_VERSION", atype_id,        &
                        dspace_id, attr_id, hdferr )
      ! intermediate variable because MF_MUESLI_VERSION is often shorter
      ! than 8 characters.
      muesli_version = MF_MUESLI_VERSION
      call h5awrite_f( attr_id, atype_id, muesli_version, data_dims, hdferr )
      ! closes the attribute.
      call h5aclose_f( attr_id, hdferr )
      ! END  "MF_MUESLI_VERSION"

      ! BEGIN "DATA_TYPE"
      ! creates datatype for the attribute.
      call h5tcopy_f( H5T_NATIVE_INTEGER, atype_id, hdferr )
      ! (size of DATA_TYPE : 1 integer => 4 bytes)
      call h5tset_size_f( atype_id, 4, hdferr )
      call h5acreate_f( group_id, "DATA_TYPE", atype_id,                &
                        dspace_id, attr_id, hdferr )
      itmp = x%data_type
      call h5awrite_f( attr_id, atype_id,                               &
                       itmp, data_dims, hdferr )
      ! closes the attribute.
      call h5aclose_f( attr_id, hdferr )
      ! END "DATA_TYPE"

      ! BEGIN "NROW"
      ! creates datatype for the attribute.
      call h5tcopy_f( H5T_NATIVE_INTEGER, atype_id, hdferr )
      ! (size of NROW : 1 integer => 4 bytes)
      call h5tset_size_f( atype_id, 4, hdferr )
      call h5acreate_f( group_id, "NROW", atype_id,                     &
                        dspace_id, attr_id, hdferr )
      call h5awrite_f( attr_id, atype_id, x%shape(1), data_dims, hdferr )
      ! closes the attribute.
      call h5aclose_f( attr_id, hdferr )
      ! END "NROW"

      ! the following properties doesn't make sense for a permutation vector
      if( x%data_type /= MF_DT_PERM_VEC ) then

         ! "NCOL"
         ! creates datatype for the attribute.
         call h5tcopy_f( H5T_NATIVE_INTEGER, atype_id, hdferr )
         ! (size of NCOL : 1 integer => 4 bytes)
         call h5tset_size_f( atype_id, 4, hdferr )
         call h5acreate_f( group_id, "NCOL", atype_id,                  &
                           dspace_id, attr_id, hdferr )
         call h5awrite_f( attr_id, atype_id, x%shape(2), data_dims, hdferr )
         ! closes the attribute.
         call h5aclose_f( attr_id, hdferr )

         ! "TRIL"
         ! creates datatype for the attribute.
         call h5tcopy_f( H5T_NATIVE_INTEGER, atype_id, hdferr )
         ! (size of TRIL : 1 integer => 4 bytes)
         call h5tset_size_f( atype_id, 4, hdferr )
         call h5acreate_f( group_id, "TRIL", atype_id,                  &
                           dspace_id, attr_id, hdferr )
         if( x%prop%tril < UNKNOWN ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                              "mfArray cannot be stored in HDF5 because its 'tril'", &
                              "property is currently hidden!",         &
                              "[use of msPointer]" )
            go to 99
         end if
         itmp = x%prop%tril
         call h5awrite_f( attr_id, atype_id,                            &
                        itmp, data_dims, hdferr )
         ! closes the attribute.
         call h5aclose_f( attr_id, hdferr )

         ! "TRIU"
         ! creates datatype for the attribute.
         call h5tcopy_f( H5T_NATIVE_INTEGER, atype_id, hdferr )
         ! (size of TRIU : 1 integer => 4 bytes)
         call h5tset_size_f( atype_id, 4, hdferr )
         call h5acreate_f( group_id, "TRIU", atype_id,                  &
                           dspace_id, attr_id, hdferr )
         if( x%prop%triu < UNKNOWN ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                              "mfArray cannot be stored in HDF5 because its 'triu'", &
                              "property is currently hidden!",         &
                              "[use of msPointer]" )
            go to 99
         end if
         itmp = x%prop%triu
         call h5awrite_f( attr_id, atype_id,                            &
                        itmp, data_dims, hdferr )
         ! closes the attribute.
         call h5aclose_f( attr_id, hdferr )

         ! "SYMM"
         ! creates datatype for the attribute.
         call h5tcopy_f( H5T_NATIVE_INTEGER, atype_id, hdferr )
         ! (size of SYMM : 1 integer => 4 bytes)
         call h5tset_size_f( atype_id, 4, hdferr )
         call h5acreate_f( group_id, "SYMM", atype_id,                  &
                           dspace_id, attr_id, hdferr )
         if( x%prop%symm < UNKNOWN ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                              "mfArray cannot be stored in HDF5 because its 'symm'", &
                              "property is currently hidden!",         &
                              "[use of msPointer]" )
            go to 99
         end if
         itmp = x%prop%symm
         call h5awrite_f( attr_id, atype_id,                            &
                        itmp, data_dims, hdferr )
         ! closes the attribute.
         call h5aclose_f( attr_id, hdferr )

         ! "POS_DEF"
         ! creates datatype for the attribute.
         call h5tcopy_f( H5T_NATIVE_INTEGER, atype_id, hdferr )
         ! (size of POS_DEF : 1 integer => 4 bytes)
         call h5tset_size_f( atype_id, 4, hdferr )
         call h5acreate_f( group_id, "POS_DEF", atype_id,               &
                           dspace_id, attr_id, hdferr )
         if( x%prop%posd < UNKNOWN ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                              "mfArray cannot be stored in HDF5 because its 'pos_def'", &
                              "property is currently hidden!",         &
                              "[use of msPointer]" )
            go to 99
         end if
         itmp = x%prop%posd
         call h5awrite_f( attr_id, atype_id,                            &
                        itmp, data_dims, hdferr )
         ! closes the attribute.
         call h5aclose_f( attr_id, hdferr )

      end if

      if( mfIsSparse(x) ) then

         ! "ROW_SORTED"
         ! creates datatype for the attribute.
         call h5tcopy_f( H5T_NATIVE_INTEGER, atype_id, hdferr )
         ! (size of ROW_SORTED : 1 integer => 4 bytes)
         call h5tset_size_f( atype_id, 4, hdferr )
         call h5acreate_f( group_id, "ROW_SORTED", atype_id,            &
                           dspace_id, attr_id, hdferr )
         if( x%row_sorted < UNKNOWN ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "mfArray cannot be stored in HDF5 because its 'row_sorted'", &
                               "property is currently hidden!",        &
                               "[use of msPointer]" )
            go to 99
         end if
         itmp = x%row_sorted
         call h5awrite_f( attr_id, atype_id,                            &
                          itmp, data_dims, hdferr )
         ! closes the attribute.
         call h5aclose_f( attr_id, hdferr )

         ncol = x%shape(2)
         nnz = x%j(ncol+1) - 1

         ! "NNZ"
         ! creates datatype for the attribute.
         call h5tcopy_f( H5T_NATIVE_INTEGER, atype_id, hdferr )
         ! (size of ROW_SORTED : 1 integer => 4 bytes)
         call h5tset_size_f( atype_id, 4, hdferr )
         call h5acreate_f( group_id, "NNZ", atype_id,                   &
                           dspace_id, attr_id, hdferr )
         call h5awrite_f( attr_id, atype_id, nnz, data_dims, hdferr )
         ! closes the attribute.
         call h5aclose_f( attr_id, hdferr )

      end if

      ! closes the dataspace
      call h5sclose_f( dspace_id, hdferr )

      if( .not. mfIsSparse(x) ) then

         rank = 2

         ! gets some infos about the mfArray
         data_dims = x%shape

         ! creates the dataspace
!### TODO 2: copy detected by _g95_contiguous_array
         call h5screate_simple_f( rank, data_dims, dspace_id, hdferr )

         ! manages data compression
         call h5pcreate_f( H5P_DATASET_CREATE_F, dcpl_id, hdferr )
         rank_chunk = 2
         chunk_dims = data_dims
         call h5pset_chunk_f( dcpl_id, rank_chunk, chunk_dims, hdferr )
         deflate_level = 1
         call h5pset_deflate_f( dcpl_id, deflate_level, hdferr )

         if( x%data_type == MF_DT_DBLE ) then
            ! creates the dataset with previous defined properties
            call h5dcreate_f( group_id, "double", H5T_NATIVE_DOUBLE,    &
                              dspace_id, dset_id, hdferr, dcpl_id )

            ! writes the dataset
            call h5dwrite_f( dset_id, H5T_NATIVE_DOUBLE, x%double(1,1), &
                             data_dims, hdferr)
         else if( x%data_type == MF_DT_CMPLX ) then
            ! creates the dataset with previous defined properties
            call h5dcreate_f( group_id, "real", H5T_NATIVE_DOUBLE,      &
                              dspace_id, dset_id, hdferr, dcpl_id )

            ! writes the dataset
!### TODO 2: copy detected by _g95_contiguous_array
            call h5dwrite_f( dset_id, H5T_NATIVE_DOUBLE, real(x%cmplx), &
                             data_dims, hdferr)
            ! creates the dataset with previous defined properties
            call h5dcreate_f( group_id, "imag", H5T_NATIVE_DOUBLE,      &
                              dspace_id, dset_id, hdferr, dcpl_id )

            ! writes the dataset
!### TODO 2: copy detected by _g95_contiguous_array
            call h5dwrite_f( dset_id, H5T_NATIVE_DOUBLE, aimag(x%cmplx),&
                             data_dims, hdferr)
         else if( x%data_type /= MF_DT_PERM_VEC ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "unknown data type" )
            go to 99
         end if

      else ! sparse array

         rank = 1
         data_dims(1) = nnz

         ! creates the dataspace
         call h5screate_simple_f( rank, data_dims, dspace_id, hdferr )

         ! manages data compression
         call h5pcreate_f( H5P_DATASET_CREATE_F, dcpl_id, hdferr )
         rank_chunk = 1
         chunk_dims(1) = data_dims(1)
         call h5pset_chunk_f( dcpl_id, rank_chunk, chunk_dims, hdferr )
         deflate_level = 1
         call h5pset_deflate_f( dcpl_id, deflate_level, hdferr )

         if( x%data_type == MF_DT_SP_DBLE ) then
            ! creates the dataset with previous defined properties
            call h5dcreate_f( group_id, "a", H5T_NATIVE_DOUBLE,         &
                              dspace_id, dset_id, hdferr, dcpl_id )

            ! writes the dataset
            call h5dwrite_f( dset_id, H5T_NATIVE_DOUBLE, x%a(1),        &
                             data_dims, hdferr)
         else if( x%data_type == MF_DT_SP_CMPLX ) then
            ! creates the dataset with previous defined properties
            call h5dcreate_f( group_id, "real", H5T_NATIVE_DOUBLE,      &
                              dspace_id, dset_id, hdferr, dcpl_id )

            ! writes the dataset
!### TODO 2: copy detected by _g95_contiguous_array
            call h5dwrite_f( dset_id, H5T_NATIVE_DOUBLE, real(x%z),     &
                             data_dims, hdferr)

            ! creates the dataset with previous defined properties
            call h5dcreate_f( group_id, "imag", H5T_NATIVE_DOUBLE,      &
                              dspace_id, dset_id, hdferr, dcpl_id )

            ! writes the dataset
!### TODO 2: copy detected by _g95_contiguous_array
            call h5dwrite_f( dset_id, H5T_NATIVE_DOUBLE, aimag(x%z),    &
                             data_dims, hdferr)
         else
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "unknown data type" )
            go to 99
         end if

         ! creates the dataset with previous defined properties
         call h5dcreate_f( group_id, "i", H5T_NATIVE_INTEGER,           &
                           dspace_id, dset_id, hdferr, dcpl_id )

         ! writes the dataset
         call h5dwrite_f( dset_id, H5T_NATIVE_INTEGER, x%i(1),          &
                          data_dims, hdferr)

         ! closes the dataspace
         call h5sclose_f( dspace_id, hdferr )

         data_dims(1) = ncol+1

         ! creates the dataspace
!### TODO 2: copy detected by _g95_contiguous_array
         call h5screate_simple_f( rank, data_dims, dspace_id, hdferr )

         ! manages data compression
         call h5pcreate_f( H5P_DATASET_CREATE_F, dcpl_id, hdferr )
         rank_chunk = 1
         chunk_dims(1) = data_dims(1)
         call h5pset_chunk_f( dcpl_id, rank_chunk, chunk_dims, hdferr )
         deflate_level = 1
         call h5pset_deflate_f( dcpl_id, deflate_level, hdferr )

         ! creates the dataset with previous defined properties
         call h5dcreate_f( group_id, "j", H5T_NATIVE_INTEGER,           &
                           dspace_id, dset_id, hdferr, dcpl_id )

         ! writes the dataset
         call h5dwrite_f( dset_id, H5T_NATIVE_INTEGER, x%j(1),          &
                          data_dims, hdferr)

      end if

      if( x%data_type == MF_DT_PERM_VEC ) then

         rank = 1
         data_dims(1) = x%shape(1)

         ! creates the dataspace
!### TODO 2: copy detected by _g95_contiguous_array
         call h5screate_simple_f( rank, data_dims, dspace_id, hdferr )

         ! manages data compression
         call h5pcreate_f( H5P_DATASET_CREATE_F, dcpl_id, hdferr )
         rank_chunk = 1
         chunk_dims(1) = data_dims(1)
         call h5pset_chunk_f( dcpl_id, rank_chunk, chunk_dims, hdferr )
         deflate_level = 1
         call h5pset_deflate_f( dcpl_id, deflate_level, hdferr )

         ! creates the dataset with previous defined properties
         call h5dcreate_f( group_id, "i", H5T_NATIVE_INTEGER,           &
                           dspace_id, dset_id, hdferr, dcpl_id )

         ! writes the dataset
         call h5dwrite_f( dset_id, H5T_NATIVE_INTEGER, x%i(1),          &
                          data_dims, hdferr)

      end if

      ! closes the dataset
      call h5dclose_f( dset_id, hdferr )

      ! closes the dataspace
      call h5sclose_f( dspace_id, hdferr )

      ! physical units
      if( mf_phys_units ) then
         call verif_adim( x%units, status=istatus )
         if( istatus /= 0 ) then ! not dimensionless

            do i = 1, num_base_units
               r_units(i) = real( x%units(i)%num, kind=MF_DOUBLE ) /    &
                            real( x%units(i)%den, kind=MF_DOUBLE )
            end do

            rank = 1
            data_dims(1) = num_base_units

            ! creates the dataspace
            call h5screate_simple_f( rank, data_dims, dspace_id, hdferr )

            ! creates the dataset with default properties
            call h5dcreate_f( group_id, "units", H5T_NATIVE_DOUBLE,     &
                              dspace_id, dset_id, hdferr )

            ! writes the dataset
            call h5dwrite_f( dset_id, H5T_NATIVE_DOUBLE, r_units(1),    &
                             data_dims, hdferr)

            ! closes the dataset
            call h5dclose_f( dset_id, hdferr )

            ! closes the dataspace
            call h5sclose_f( dspace_id, hdferr )

         end if
      end if

      ! closes the group
      call h5gclose_f( group_id, hdferr )

 59   continue
      ! closes the file
      call h5fclose_f( file_id, hdferr )

      ! closes the HDF5's F90 interface
      call h5close_f( hdferr )
      if( hdferr /= 0 ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "cannot close the HDF5 F90 interface!" )
      end if

 99   continue

      call msFreeArgs( x )

      call msAutoRelease( x )

#endif
   end subroutine msSaveHDF5
