! f90 include file

!_______________________________________________________________________
!
   function mfLoadSparse( filename,                                     &
                          format, duplicated_entries )                  &
   result( out )

      character(len=*), intent(in)           :: filename
      character(len=3), intent(in), optional :: format
      character(len=*), intent(in), optional :: duplicated_entries
      type(mfArray)                          :: out
      !------ API end ------
#ifdef _DEVLP

      ! import a sparse structure from a file
      !
      ! supported format : MTX : (i,j,val)     [MTX=COO](*)
      !                    CSC : Compact Sparse Column
      !                    CSR : Compact Sparse Row
      !                    HBO : Harwell-Boeing
      !
      ! duplicated_entries : "added"       [default] (as in Matlab)
      !                      "ignored"
      !                      "replaced"

      ! (*) for the definition of the MTX format, see:
      !   docs/numerics/MMformat.ps

      ! 'row sorting' is always applied (whatever the value of the
      ! global variable MF_SP_AUTO_ROW_SORTED), because the processing
      ! of duplicated entries requires sorting.

      ! stored in order : pointer, indexes, values
      integer :: nrow, ncol, nnz, col_sorted
      integer :: i, n, unit, nmax, nzmax, job, nrhs, ierr
      integer :: n_diag, n_lower
      character(len=3) :: form
      character(len=7) :: kind
      character(len=3) :: format0
      character(len=2) :: guesol
      character(len=72) :: title
      character(len=8) :: key
      character(len=3) :: type
      ! work arrays
      integer,                 allocatable :: wk_i(:), wk_j(:)
      real(kind=MF_DOUBLE),    allocatable :: wk_a(:)
      complex(kind=MF_DOUBLE), allocatable :: wk_z(:)
      ! rhs : dummy vector -- should not be referenced inside 'readmt'
      real(kind=MF_DOUBLE) :: rhs(1)
      logical :: exist
      character(len=132) :: line
      character(len=32) :: head0, matrix, rep, field, symm
      character(len=32) :: item
      integer, allocatable :: irow(:), jcol(:)
      real(kind=MF_DOUBLE), allocatable :: val(:)
      real(kind=MF_DOUBLE) :: rtmp, itmp
      complex(kind=MF_DOUBLE), allocatable :: zval(:)
      integer :: mf_message_level_save
      character(len=8) :: dupl_entries
      character(len=*), parameter :: ROUTINE_NAME = "mfLoadSparse"

   !------ end of declarations -- execution starts hereafter  ------

      inquire( file=trim(filename), exist=exist )
      if( .not. exist ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "file not found! (broken link?)",           &
                            "file: " // trim(adjustl(filename)) )
         call msPause("As a last chance, you can try to fix the problem.")
         inquire( file=trim(adjustl(filename)), exist=exist )
         if( .not. exist ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "file not found!" )
            return
         end if
      end if

      call find_unit( unit )
      open( unit=unit, file=trim(adjustl(filename)) )

      if( present(format) ) then
         format0 = to_lower(format)
         go to 25
      else
         ! trying to auto-detect the sparse format
         read(unit,"(A)") line ! 1st line

         ! MTX ?
         if( to_lower(line(1:14)) == "%%matrixmarket" ) then
            format0 = "mtx"
            go to 20
         end if

         ! CSC or CSR ?
         read(line,*,end=5,err=5) nrow, ncol, i, form, kind
         select case ( to_lower(form) )
            case( "csc" )
               format0 = "csc"
               go to 20
            case( "csr" )
               format0 = "csr"
               go to 20
         end select

 5       continue

         ! HBO ?
         ! (4th line should contain 4 formats)
         ! read line number 4
         read(unit,"(A)") line ! 2nd line
         read(unit,"(A)") line ! 3rd line
         read(unit,"(A)") line ! 4th line
         do i = 1, 4
            call fileio_get_tok( line, item ) ! ith format
            n = len_trim(item)
            if( item(1:1) /= "(" .or. item(n:n) /= ")" ) then
               ! failed
               go to 10
            end if
         end do
         format0 = "hbo"
         go to 20

 10      continue

         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "file: " // trim(filename),                 &
                            "cannot detect this sparse format!" )
         go to 99

      end if

 20   continue
      call PrintMessage( ROUTINE_NAME, "I",                             &
                         "format auto-detected: """ // to_upper(format0) // """" )

 25   continue

      if( present(duplicated_entries) ) then
         if( .not. MF_SP_AUTO_ROW_SORTED ) then
            ! user chose not to sort columns' entries, but, the row
            ! sorting in each column is always required.
            call PrintMessage( ROUTINE_NAME, "I",                       &
                               "you chose, via the 'msSetAutoRowSorted' routine,", &
                               "not to sort columns' entries. However, 'duplicated_entries'", &
                               "requires sorting; so temporarily activating 'row sorting'" )
         end if
         dupl_entries = to_lower(duplicated_entries)
         if( dupl_entries /= "ignored" .and.                            &
             dupl_entries /= "added"   .and.                            &
             dupl_entries /= "replaced" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "duplicated_entries must one of:",       &
                               "'ignored', 'added' or 'replaced'" )
            return
         end if
      else
         dupl_entries = "added"
      end if

      rewind( unit )

      select case( format0 )
      case( "csc" )

         read(unit,*) nrow, ncol, out%row_sorted, form, kind
         if( out%row_sorted /= UNKNOWN .and. out%row_sorted /= TRUE .and. &
             out%row_sorted /= FALSE ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "bad TAG 'row_sorted'!" )
            go to 99
         end if
         if( to_lower(form) /= "csc" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "sparse format should be 'CSC'!" )
            go to 99
         end if
         if( to_lower(kind) /= "real" .and.                             &
             to_lower(kind) /= "complex" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'CSC': fourth tag should be 'real' or 'complex'!" )
            go to 99
         end if
         out%shape = [ nrow, ncol ]

         allocate( out%j(ncol+1) )

         read(unit,*) out%j
         nnz = out%j(ncol+1) - 1
         allocate( out%i(nnz) )

         read(unit,*) out%i
         if( to_lower(kind) == "real" ) then
            out%data_type = MF_DT_SP_DBLE
            allocate( out%a(nnz) )

            read(unit,*) out%a
         else
            out%data_type = MF_DT_SP_CMPLX
            allocate( out%z(nnz) )

            read(unit,*) out%z
         end if

         call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )

      case( "csr" )

         read(unit,*) nrow, ncol, col_sorted, form, kind
         if( to_lower(form) /= "csr" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "sparse format should be 'CSR'!" )
            go to 99
         end if
         if( to_lower(kind) /= "real" .and.                             &
             to_lower(kind) /= "complex" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "'CSR': fourth tag should be 'real' or 'complex'!" )
            go to 99
         end if
         out%shape = [ nrow, ncol ]

         allocate( wk_i(nrow+1) )

         read(unit,*) wk_i
         allocate( out%j(ncol+1) )

         nnz =  wk_i(nrow+1) - 1
         allocate( wk_j(nnz) )

         read(unit,*) wk_j
         allocate( out%i(nnz) )

         if( to_lower(kind) == "real" ) then
            out%data_type = MF_DT_SP_DBLE
            allocate( wk_a(nnz) )

            read(unit,*) wk_a
            allocate( out%a(nnz) )

            call csc_transp( ncol, nrow, wk_a, wk_j, wk_i,              &
                             out%a, out%i, out%j )
         else
            out%data_type = MF_DT_SP_CMPLX
            allocate( wk_z(nnz) )

            read(unit,*) wk_z
            allocate( out%z(nnz) )

            call csc_transp_cmplx( ncol, nrow, wk_z, wk_j, wk_i,        &
                                   out%z, out%i, out%j )
         end if

         ! after transposition, the matrix becomes row_sorted, whatever
         ! its col_sorted property
         out%row_sorted = TRUE

         call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )

      case( "hbo" )

         ! get infos on the matrix to be read
#ifndef _OPTIM
! Whereas the following assignment is not necessary, it is used in order
! to avoid "uninitialized value" by 'valgrind'.
         call msAssign( out, mfSpAlloc( 1, 1 ) )
#endif
         job = 0
         call readmt( nmax, nzmax, job, unit, out%a, out%i, out%j, rhs, &
                      nrhs, guesol, nrow, ncol, nnz, title,             &
                      key, type, ierr )
         if( ierr /= 0 ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "Cannot read HBO matrix." )
            go to 99
         end if

         rewind( unit )

         call msAssign( out, mfSpAlloc( nrow, ncol, nzmax=nnz ) )

         ! The only possibility to check if the HBO file contains also
         ! the RHS is to try to retrieve it...
         job = 3
         nrhs = 0 ! don't want the RHS actually
         call readmt( ncol, nnz, job, unit, out%a, out%i, out%j, rhs,   &
                      nrhs, guesol, nrow, ncol, nnz, title,             &
                      key, type, ierr )
         if( job == 2 ) then
            ! all is correct, the HBO file doesn't include the RHS
            ! (anyway we don't want it)
         else if( job == 3 ) then
            ! ierr should be 5 -- see readmt from fml_fileio
            if( ierr /= 5 ) then
               print *
               print *, "mfLoadSparse: internal error after calling 'readmt'"
               print *, "              (line 289 of mfLoadSparse.inc)"
               print *, "              ierr = ", ierr, " (instead of 5)"
               pause ! for debugging purpose
               stop
            end if
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "The HBO file contains also a Right-Hand-Side", &
                               "but it has been ignored." )
         else
            print *
            print *, "mfLoadSparse: internal error after calling 'readmt'"
            print *, "              (line 300 of mfLoadSparse.inc)"
            print *, "              on return: job = ", job
            pause ! for debugging purpose
            stop
         end if

         call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )

      case( "mtx" )

         ! read characteristics of matrix

         ! header must be conformant
         read(unit,"(A)") line

         call fileio_get_tok( line, head0 )
         head0 = to_lower(head0)
         if( trim(head0) /= "%%matrixmarket" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "bad header on line 1!",                 &
                               "head0 = '" // trim(head0) // "'" )
            go to 99
         end if

         call fileio_get_tok( line, matrix )
         matrix = to_lower(matrix)
         if( trim(matrix) /= "matrix" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "bad header on line 1!",                 &
                               "matrix = '" // trim(matrix) // "'",     &
                               "(currently, Muesli can read only matrix object)" )
            go to 99
         end if

         call fileio_get_tok( line, rep )
         rep = to_lower(rep)
         if( trim(rep) /= "coordinate" ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "bad header on line 1!",                 &
                               "rep = '" // trim(rep) // "'",           &
                               "(currently, Muesli can read only coordinate format)" )
            go to 99
         end if

         call fileio_get_tok( line, field )
         field = to_lower(field)
         if( trim(field) /= "real"    .and.                             &
             trim(field) /= "complex" .and.                             &
             trim(field) /= "pattern"  ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "bad header on line 1!",                 &
                               "field = '" // trim(field) // "'" )
            go to 99
         end if

         call fileio_get_tok( line, symm )
         symm = to_lower(symm)
         if( trim(symm) /= "symmetric"      .and.                       &
             trim(symm) /= "hermitian"      .and.                       &
             trim(symm) /= "skew-symmetric" .and.                       &
             trim(symm) /= "general"  ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "bad header on line 1!",                 &
                               "symm = '" // trim(symm) // "'" )
            go to 99
         end if

         read(unit,"(A)") line
         do while( line(1:1) == "%" )
            read(unit,"(A)") line
         end do

         read(line,*) nrow, ncol, nnz
         out%shape = [ nrow, ncol ]

         allocate( irow(nnz), jcol(nnz) )

         if( field == "real" ) then
            allocate( val(nnz) )
            out%data_type = MF_DT_SP_DBLE
            do i = 1, nnz
               read(unit,*) irow(i), jcol(i), val(i)
            end do
            if( trim(symm) == "general" ) then
               nzmax = nnz
               allocate( out%a(nzmax) )
               allocate( out%i(nzmax) )
               allocate( out%j(ncol+1) )
               call coo2csc( ncol, nnz, val, irow, jcol, out%a, out%i, out%j )
               go to 98
            end if
         else if( field == "complex" ) then
            allocate( zval(nnz) )
            out%data_type = MF_DT_SP_CMPLX
            do i = 1, nnz
               read(unit,*) irow(i), jcol(i), rtmp, itmp
               zval(i) = cmplx( rtmp, itmp,kind=MF_DOUBLE )
            end do
            if( trim(symm) == "general" ) then
               nzmax = nnz
               allocate( out%z(nzmax) )
               allocate( out%i(nzmax) )
               allocate( out%j(ncol+1) )
               call coo2csc_cmplx( ncol, nnz, zval, irow, jcol, out%z, out%i, out%j )
               go to 98
            end if
         else
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "field = 'pattern' is not (yet) supported!" )
            go to 99
         end if

         ! de-activate warning for using .t. operator for complex matrices
         mf_message_level_save = mf_message_level
         mf_message_level = 0

         ! If pattern symmetric, skew-symmetric or hermitian, duplicate
         ! the lower triangular part and modify entries as appropriate:
         if( trim(symm) == "symmetric" ) then
            ! computing true nzmax
            n_diag = 0
            n_lower = 0
            do i = 1, nnz
               if( irow(i) > jcol(i) ) then
                  ! lower part of the matrix
                  n_lower = n_lower + 1
               else if( irow(i) < jcol(i) ) then
                  ! upper part of the matrix
                  ! WARNING this is out of the MTX specification
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "For this particular pattern, only the lower part of the matrix", &
                                     "must be defined!",                &
                                     "(the current MTX object doesn't follow the format specifications)")
                  go to 99
               else ! diagonal element
                  n_diag = n_diag + 1
               end if
            end do
            nzmax = n_diag + 2*n_lower
            allocate( out%i(nzmax) )
            allocate( out%j(ncol+1) )
            if( field == "real" ) then
               allocate( out%a(nzmax) )
               call coo_symm2csc( ncol, nnz, val, irow, jcol,           &
                                  out%a, out%i, out%j )
            else ! field = "complex"
               allocate( out%z(nzmax) )
               call coo_symm2csc_cmplx( ncol, nnz, zval, irow, jcol,    &
                                        out%z, out%i, out%j )
            end if
            out%prop%symm = TRUE
         else if( trim(symm) == "hermitian" ) then
            ! computing true nzmax
            n_diag = 0
            n_lower = 0
            do i = 1, nnz
               if( irow(i) > jcol(i) ) then
                  ! lower part of the matrix
                  n_lower = n_lower + 1
               else if( irow(i) < jcol(i) ) then
                  ! upper part of the matrix
                  ! WARNING this is out of the MTX specification
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "For this particular pattern, only the lower part of the matrix", &
                                     "must be defined!",                &
                                     "(the current MTX object doesn't follow the format specifications)")
                  go to 99
               else ! diagonal element
                  n_diag = n_diag + 1
               end if
            end do
            nzmax = n_diag + 2*n_lower
            allocate( out%i(nzmax) )
            allocate( out%j(ncol+1) )
            if( field == "real" ) then
               ! particular case (not academic): a real and hermitian
               ! matrix is simply symmetric!
               allocate( out%a(nzmax) )
               call coo_symm2csc( ncol, nnz, val, irow, jcol,           &
                                  out%a, out%i, out%j )
            else ! field = "complex"
               allocate( out%z(nzmax) )
               call coo_herm2csc_cmplx( ncol, nnz, zval, irow, jcol,    &
                                        out%z, out%i, out%j )
            end if
            out%prop%symm = TRUE
         else if( trim(symm) == "skew-symmetric" ) then
            ! computing true nzmax
            n_diag = 0
            n_lower = 0
            do i = 1, nnz
               if( irow(i) > jcol(i) ) then
                  ! lower part of the matrix
                  n_lower = n_lower + 1
               else if( irow(i) < jcol(i) ) then
                  ! upper part of the matrix
                  ! WARNING this is out of the MTX specification
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "For this particular pattern, only the lower part of the matrix", &
                                     "must be defined!",                &
                                     "(the current MTX object doesn't follow the format specifications)")
                  go to 99
               else ! diagonal element
                  n_diag = n_diag + 1
               end if
            end do
            ! No diagonal element can be present
            if( n_diag /= 0 ) then
               ! Simple warning, because diagonal elements will be ignored...
               call PrintMessage( ROUTINE_NAME, "W",                    &
                                  "For a skew-symmetric pattern, all diagonal elements of the matrix", &
                                  "must be null! They will be ignored!", &
                                  "(the current MTX object doesn't follow the format specifications)")
            end if
            nzmax = 2*n_lower
            allocate( out%i(nzmax) )
            allocate( out%j(ncol+1) )
            if( field == "real" ) then
               allocate( out%a(nzmax) )
               call coo_skew2csc( ncol, nnz, val, irow, jcol,           &
                                  out%a, out%i, out%j )
            else ! field = "complex"
               allocate( out%z(nzmax) )
               call coo_skew2csc_cmplx( ncol, nnz, zval, irow, jcol,    &
                                        out%z, out%i, out%j )
            end if
            out%prop%symm = FALSE
         end if

         ! restore warning level
         mf_message_level = mf_message_level_save

 98      continue

         call process_dupl_entries( out, dupl_entries, ROUTINE_NAME )
         ! the matrix 'out' is now "row sorted".

      case default

         call PrintMessage( ROUTINE_NAME, "E",                          &
                            "unknown requested format!" )

      end select

      out%status_temporary = .true.

 99   continue

      close(unit)

#endif
   end function mfLoadSparse
!_______________________________________________________________________
!
   subroutine fileio_get_tok( line, token )

      character(len=*) :: line, token
      !------ API end ------
#ifdef _DEVLP

      ! extract one non-blank token from 'line'
      !
      ! warn: 'line' is modified !

      integer :: i

      if( trim(line) == "" ) then
         token = ""
         return
      end if

      line = adjustl(line)
      i = index( line, " " )
      token= line(1:i-1)
      line = line(i:)

#endif
   end subroutine fileio_get_tok
