! f90 include file

!_______________________________________________________________________
!
   function mfLoadAscii( filename,                                      &
                         ieee, comment, rect, row_max, col_max )        &
   result( out )

      character(len=*), intent(in)           :: filename
      logical,          intent(in), optional :: ieee
      character(len=1), intent(in), optional :: comment
      logical,          intent(in), optional :: rect
      integer,          intent(in), optional :: row_max, col_max

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      ! * if 'ieee'==.true. then the current routine is able to read
      !   ASCII files containing special IEEE values, as
      !   "NaN", "Inf" and "-Inf" (but the read will be slower for
      !   some compilers which doesn't support natively the read of
      !   these special IEEE values).
      !
      ! * if 'comment' is present, commented or blank lines are
      !   supported (they must begin by the character stored in this
      !   argument). Comments can follow data on a line...
      !   These lines are simply ignored and cannot be used as
      !   zone-separators.
      !
      ! * if 'rect'==.false. then non rectangular data can be read;
      !   Numerical values are stored in a long column vector.

      !-----------------------------

#if defined _INTEL_IFC | defined _GNU_GFC
! the following return status codes seem standard...
      integer, parameter :: End_Of_File   = -1,                         &
                            End_Of_Record = -2
#else
- '(MUESLI mfLoadAscii:) compiler not defined!'
#endif

! F2003 introduces the following intrinsic functions:
! 'is_iostat_end' and 'is_iostat_eor'
! (ok for: GNU gfortran>=4.3, INTEL-ifort)
!
! F2003 introduces also the 'ISO_FORTRAN_ENV' module, which defines
! the parameter constants: 'iostat_eof' and 'iostat_eor'
! (ok for: GNU gfortran>=4.3, INTEL-ifort)
!
      integer :: i, j, unit, ni, nj, iostat, ipos, status, ntot
      character :: c
      integer, parameter :: len_max = 1000000
      logical :: in_separator, ieee_l, exist, rect_l
      character(len=12) :: iostat_str, i_str, j_str

      type(varying_string) :: line
      character(len=*), parameter :: ROUTINE_NAME = "mfLoadAscii"

   !------ end of declarations -- execution starts hereafter  ------

      inquire( file=trim(adjustl(filename)), exist=exist )
      if( .not. exist ) then
         call PrintMessage( ROUTINE_NAME, "W",                          &
                            "file not found! (broken link?)",           &
                            'file: "' // trim(adjustl(filename)) // '"' )
         call msPause("As a last chance, you can try to fix the problem.")
         inquire( file=trim(adjustl(filename)), exist=exist )
         if( .not. exist ) then
            call PrintMessage( ROUTINE_NAME, "E",                       &
                               "file not found!" )
            return
         end if
      end if

      call find_unit( unit )

      ! reading must be done in free format!
      ! (because the file may have been created by another program)

      if( present(ieee) ) then
         ieee_l = ieee
      else
         ieee_l = .false.
      end if

      if( present(rect) ) then
         rect_l = rect
      else
         rect_l = .true.
      end if

      if( present(col_max) ) then
         if( col_max <= 0 ) then
            call PrintMessage( ROUTINE_NAME, "W",                       &
                               "'col_max' <= 0!",                       &
                               "(output will be empty)" )
            out = MF_EMPTY
            close(unit)
            return
         end if
      end if

      ! the RECL parameter must be set because lines may be very long.
      open( unit=unit, recl=len_max, status="old", file=trim(adjustl(filename)) )

      if( rect_l ) then

         ! 3 passes for reading rectangular data:
         !  first and second determine the array size to be allocated;
         !  third reads data

         ! first pass: only the first line (containing real values)
         !             is read
         ! if the optional argument 'comment' is present, the first
         ! non blank character must be checked
         if( present(comment) ) then
            do
               ! getting the whole line
               call get( unit, line, iostat=iostat )
               if( iostat > 0 ) then
                  write(iostat_str,*) iostat
                  call PrintMessage( ROUTINE_NAME, "E",                 &
                                     "cannot read line!",               &
                                     "file: " // trim(adjustl(filename)), &
                                     "iostat = " // trim(adjustl(iostat_str)) )
                  return
               end if

               line = adjustl(line)
               if( len_trim(line) /= 0 ) then
                  if( .not. VS_elem_is_same(line,1,comment) ) exit
               end if
            end do
         else ! no comments within data lines
            ! getting the whole line
            call get( unit, line, iostat=iostat )
            if( iostat > 0 )then
               write(iostat_str,*) iostat
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "cannot read first line!",            &
                                  "file: " // trim(adjustl(filename)),  &
                                  "iostat = " // trim(adjustl(iostat_str)) )
               return
            else if( iostat == End_Of_File ) then
               ! file is empty
               out = MF_EMPTY
               close(unit)
               return
            end if
            line = adjustl(line)
         end if
         nj = 0
         in_separator = .true.
         do i = 1, len_trim(line)
            ! separators may be blanks or tabulations, in arbitrary number...
            if( VS_elem_is_blank(line,i) ) then
               if( .not. in_separator ) then
                  in_separator = .true.
               end if
            else
               ! first non-blank character found... must take into account
               ! a possible comment
               if( present(comment) ) then
                  if( VS_elem_is_same(line,i,comment) ) exit
               end if
               if( in_separator ) then
                  nj = nj + 1
                  in_separator = .false.
               end if
            end if
         end do

         if( nj == 0 ) then
            out = MF_EMPTY
            close(unit)
            return
         end if

         ! here, the number of columns is known... but it can be limited
         ! by the user himself:
         if( present(col_max) ) then
            if( col_max > nj ) then
               call PrintMessage( ROUTINE_NAME, "W",                    &
                                  "'col_max' > actual number of columns!", &
                                  "(argument ignored)" )
            end if
            nj = min( nj, col_max )
         end if

         rewind(unit)
         out%shape(2) = nj

         ! second pass: all lines are read
         ni = 0
         if( present(comment) ) then
            do
               ! getting the whole line
               call get( unit, line, iostat=iostat )
!!#if defined _GNU_GFC
!!               if( is_iostat_end(iostat) ) exit
!!#else
               if( iostat == End_Of_File ) exit
!!#endif
               line = adjustl(line)
               if( len_trim(line) /= 0 ) then
                  if( .not. VS_elem_is_same(line,1,comment) ) then
                     ni = ni + 1
                     if( present(row_max) ) then
                        if( ni == row_max ) exit
                     end if
                  end if
               end if
            end do
         else
            do
               read( unit, "(A)", iostat=iostat ) c
               if( iostat < 0 ) exit
               ni = ni + 1
               if( present(row_max) ) then
                  if( ni == row_max ) exit
               end if
            end do
         end if

         rewind(unit)
         out%shape(1) = ni

         ! data from an ASCII file are supposed to be always real...
         ! but they may be stored as characters. Ex. "NaN", "Inf"
         out%data_type = MF_DT_DBLE
         allocate( out%double(ni, nj) )

         if( present(comment) ) then

            if( ieee_l ) then
#if defined _INTEL_IFC | defined _GNU_GFC
! the compiler can read "NaN" and "Inf" values : good !
               i = 1
               do
                  read(unit,*,iostat=iostat) ( out%double(i,j), j = 1, nj )
                  if( iostat == 0 ) then
                     i = i + 1
                     if( i > ni ) exit
                  end if
               end do
#else
               i = 1
               do
                  ! getting the whole line
                  call get( unit, line, iostat=iostat )
!!#if defined _GNU_GFC
!!               if( is_iostat_end(iostat) ) exit
!!#else
                  if( iostat == End_Of_File ) exit
!!#endif
                  line = adjustl(line)
                  if( len_trim(line) == 0 ) cycle
                     if( VS_elem_is_same(line,1,comment) ) cycle
                  ipos = 1
                  do j = 1, nj
                     call extract_ieee_val_from_VS( line, ipos,         &
                                                    out%double(i,j),    &
                                                    status )
                     if( status /= 0 ) then
                        write(i_str,*) i
                        write(j_str,*) j
                        call PrintMessage( ROUTINE_NAME, "I",           &
                                           "raw line: " // trim(adjustl(i_str)), &
                                           "raw col: " // trim(adjustl(j_str)) )
                        write(iostat_str,*) iostat
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "data cannot be read as a rectangular matrix!", &
                                           "file: " // trim(adjustl(filename)), &
                                           "iostat = " // trim(adjustl(iostat_str)) )
                        out = MF_EMPTY
                        exit
                     end if
                  end do
                  i = i + 1
               end do
#endif
            else ! no ieee
               i = 1
               do
                  ! getting the whole line
                  call get( unit, line, iostat=iostat )
!!#if defined _GNU_GFC
!!               if( is_iostat_end(iostat) ) exit
!!#else
                  if( iostat == End_Of_File ) exit
!!#endif
                  line = adjustl(line)
                  if( len_trim(line) == 0 ) cycle
                  if( VS_elem_is_same(line,1,comment) ) cycle
                  call read_val_vec_in_VS( line, out%double(i,1:nj), status )
#if defined _INTEL_IFC | defined _GNU_GFC
! the compiler can read "NaN" and "Inf" values, so we must add some test
                  if( .not. all(mf_isfinite(out%double(i,1:nj))) ) then
                     write(i_str,*) i
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "raw line: " // trim(adjustl(i_str)) )
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "non numeric value found!",     &
                                        'file: "' // trim(adjustl(filename)) // '"', &
                                        "(try to use the 'ieee' argument)" )
                     out = MF_EMPTY
                     exit
                  end if
#endif
                  if( status < 0 ) then
                     write(iostat_str,*) iostat
                     write(i_str,*) i
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "raw line: " // trim(adjustl(i_str)) )
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "data cannot be read as a rectangular matrix!", &
                                        'file: "' // trim(adjustl(filename)) // '"', &
                                        "iostat = " // trim(adjustl(iostat_str)) )
                     out = MF_EMPTY
                     exit
                  else if( status > 0 ) then
                     write(i_str,*) i
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "raw line: " // trim(adjustl(i_str)) )
                     write(iostat_str,*) iostat
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "non numeric value found!",     &
                                        'file: "' // trim(adjustl(filename)) // '"', &
                                        "iostat = " // trim(adjustl(iostat_str)), &
                                        "(try to use the 'ieee' argument)" )
                     out = MF_EMPTY
                     exit
                  end if
                  i = i + 1
               end do
            end if

         else ! no comment

            if( ieee_l ) then
#if defined _INTEL_IFC | defined _GNU_GFC
! the compiler can read "NaN" and "Inf" values, good !
               do i = 1, ni
                  read(unit,*,iostat=iostat) ( out%double(i,j), j = 1, nj )
                  if( iostat < 0 ) then
                     write(i_str,*) i
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "raw line: " // trim(adjustl(i_str)) )
                     write(iostat_str,*) iostat
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "data cannot be read as a rectangular matrix!", &
                                        'file: "' // trim(adjustl(filename)) // '"', &
                                        "iostat = " // trim(adjustl(iostat_str)) )
                     out = MF_EMPTY
                     exit
                  else if( iostat > 0 ) then
                     write(i_str,*) i
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "raw line: " // trim(adjustl(i_str)) )
                     write(iostat_str,*) iostat
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "non numeric value found!",     &
                                        'file: "' // trim(adjustl(filename)) // '"', &
                                        "iostat = " // trim(adjustl(iostat_str)), &
                                        "(try to use the 'ieee' argument)" )
                     out = MF_EMPTY
                     exit
                  end if
               end do
#else
               do i = 1, ni
                  ! getting the whole line
                  call get( unit, line, iostat=iostat )
                  ipos = 1
                  do j = 1, nj
                     call extract_ieee_val_from_VS( line, ipos,         &
                                                    out%double(i,j),    &
                                                    status )
                     if( status < 0 ) then
                        write(i_str,*) i
                        write(j_str,*) j
                        call PrintMessage( ROUTINE_NAME, "I",           &
                                           "raw line: " // trim(adjustl(i_str)), &
                                           "raw col: " // trim(adjustl(j_str)) )
                        write(iostat_str,*) iostat
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "data cannot be read as a rectangular matrix!", &
                                           'file: "' // trim(adjustl(filename)) // '"', &
                                           "iostat = " // trim(adjustl(iostat_str)) )
                        out = MF_EMPTY
                        exit
                     else if( status > 0 ) then
                        write(i_str,*) i
                        write(j_str,*) j
                        call PrintMessage( ROUTINE_NAME, "I",           &
                                           "raw line: " // trim(adjustl(i_str)), &
                                           "raw col: " // trim(adjustl(j_str)) )
                        write(iostat_str,*) iostat
                        call PrintMessage( ROUTINE_NAME, "E",           &
                                           "non numeric value found!",  &
                                           'file: "' // trim(adjustl(filename)) // '"', &
                                           "iostat = " // trim(adjustl(iostat_str)), &
                                           "(try to use the 'ieee' argument)" )
                        out = MF_EMPTY
                        exit
                     end if
                  end do
               end do
#endif
            else ! no ieee
               do i = 1, ni
                  read(unit,*,iostat=iostat) ( out%double(i,j), j = 1, nj )
                  if( iostat < 0 ) then
                     write(i_str,*) i
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "raw line: " // trim(adjustl(i_str)) )
                     write(iostat_str,*) iostat
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "data cannot be read as a rectangular matrix!", &
                                        'file: "' // trim(adjustl(filename)) // '"', &
                                        "iostat = " // trim(adjustl(iostat_str)) )
                     out = MF_EMPTY
                     exit
                  else if( iostat > 0 ) then
                     write(i_str,*) i
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "raw line: " // trim(adjustl(i_str)) )
                     write(iostat_str,*) iostat
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "non numeric value found!",     &
                                        'file: "' // trim(adjustl(filename)) // '"', &
                                        "iostat = " // trim(adjustl(iostat_str)), &
                                        "(try to use the 'ieee' argument)" )
                     out = MF_EMPTY
                     exit
                  end if
#if defined _INTEL_IFC | defined _GNU_GFC
! the compiler can read "NaN" and "Inf" values, so we must add some test
                  if( .not. all(mf_isfinite(out%double(i,1:nj))) ) then
                     write(i_str,*) i
                     call PrintMessage( ROUTINE_NAME, "I",              &
                                        "raw line: " // trim(adjustl(i_str)) )
                     call PrintMessage( ROUTINE_NAME, "E",              &
                                        "non numeric value found!",     &
                                        'file: "' // trim(adjustl(filename)) // '"', &
                                        "(try to use the 'ieee' argument)" )
                     out = MF_EMPTY
                     exit
                  end if
#else
                  ! nothing -- if special IEEE value are present in the file,
                  ! the previous read would have failed.
#endif
               end do
            end if

         end if

      else ! non rectangular case

         ! 2 passes are required to read non rectangular data:
         !   the first one is used to find the size of the array to be
         !     allocated
         !   during the second one, data is read

         ! first pass: number of values is determined
         ! (if the optional argument 'comment' is present, the first
         ! non blank character must be checked)
         ntot = 0
         do
            ! getting the whole line
            call get( unit, line, iostat=iostat )
            if( iostat > 0 )then
               write(iostat_str,*) iostat
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "cannot read line!",                  &
                                  'file: "' // trim(adjustl(filename)) // '"',  &
                                  "iostat = " // trim(adjustl(iostat_str)) )
               return
            end if
            if( iostat == End_Of_File ) exit
            line = adjustl(line)
            if( present(comment) ) then
               if( len_trim(line) == 0 ) cycle
               if( VS_elem_is_same(line,1,comment) ) cycle
            end if
            nj = 0
            in_separator = .true.
            do i = 1, len_trim(line)
               if( VS_elem_is_blank(line,i) ) then
                  if( .not. in_separator ) then
                     in_separator = .true.
                  end if
               else
                  if( in_separator ) then
                     if( present(comment) ) then
                        if( VS_elem_is_same(line,i,comment) ) exit
                     end if
                     nj = nj + 1
                     in_separator = .false.
                  end if
               end if
            end do
            ntot = ntot + nj
         end do

         if( ntot == 0 ) then
            out = MF_EMPTY
            close(unit)
            return
         end if

         rewind(unit)
         out%shape(1) = ntot
         out%shape(2) = 1

         ! data from an ASCII file are supposed to be always real...
         ! but they may be stored as characters. Ex. "NaN", "Inf"
         out%data_type = MF_DT_DBLE
         allocate( out%double(ntot, 1) )

         ! second pass: all lines are read
         ipos = 1
         do
            ! getting the whole line
            call get( unit, line, iostat=iostat )
            if( iostat == End_Of_File ) exit
            line = adjustl(line)
            if( present(comment) ) then
               if( len_trim(line) == 0 ) cycle
               if( VS_elem_is_same(line,1,comment) ) cycle
               call read_vals_in_VS( line, out%double(:,1),             &
                                     ipos, status, comment )
            else
               call read_vals_in_VS( line, out%double(:,1),             &
                                     ipos, status )
            end if
         end do

         if( .not. ieee_l ) then
            if( .not. all(mf_isfinite(out%double(:,:))) ) then
               call PrintMessage( ROUTINE_NAME, "E",                    &
                                  "non numeric value found!",           &
                                  'file: "' // trim(adjustl(filename)) // '"', &
                                  "(try to use the 'ieee' argument)" )
               out = MF_EMPTY
            end if
         end if

      end if

      close(unit)

      out%status_temporary = .true.

   contains
      !-----------------------------------------------------------------
      subroutine extract_ieee_val_from_VS( string, ipos, val, status )

         type(varying_string), intent(in) :: string
         integer :: ipos
         real(kind=MF_DOUBLE) :: val
         integer :: status

         ! extracts one IEEE value from position 'ipos'
         !
         ! status = 0 if extraction is ok
         !        = End_Of_Record if no more item in string
         !        = 1 if a non numeric value (other than inf or nan) is
         !          found

         integer :: i, i_max, iostat, ipos_save
         character(len=4) :: ieee_val
         character(len=25) :: double_str

         i_max = len(string)
         if( ipos > i_max ) then
            ! nothing to be read, no more item in 'string'
            status = End_Of_Record
            return
         end if

         ! first character must not be blank/TAB
         do
            if( .not. VS_elem_is_blank(string,ipos) ) then
               exit
            else
               ipos = ipos + 1
            end if
         end do

         ! find next blank/TAB or End-of-Line
         i = ipos + 1
         do
            if( i > i_max ) then
               exit
            else
               if( .not. VS_elem_is_blank(string,i) ) then
                  i = i + 1
               else
                  exit
               end if
            end if
         end do

         double_str = extract(string,ipos,i-1)
         ipos_save = ipos
         ipos = i

         ! reading of a substring
         read( double_str, *, iostat=iostat ) val
         if( iostat /= 0 ) then
            ! IEEE particular case
#if defined _GNU_GFC
! bug work around ("char_auto" has been declared 'public'
!                  in 'iso_varying_string')
            ieee_val = to_lower( char_auto( extract(string,ipos_save,i-1) ) )
#else
            ieee_val = to_lower( char(extract(string,ipos_save,i-1)) )
#endif
            if( ieee_val == "nan" ) then
               val = MF_NAN
            else if( ieee_val == "inf" .or. ieee_val == "infi"          &
                                       .or. ieee_val == "+inf" ) then
               ! added "infi" because ordinary prints by INTEL-ifort
               ! is "Infinity"
               ! added "+inf" because ordinary prints by GNU-gfortran
               ! is "+Infinity"
               val = MF_INF
            else if( ieee_val == "-inf" ) then
               val = -MF_INF
            else
               status = 1
               return
            end if
         end if

         status = 0

      end subroutine extract_ieee_val_from_VS
      !-----------------------------------------------------------------
      subroutine read_val_vec_in_VS( string, val_vec, status )

         type(varying_string), intent(in) :: string
         real(kind=MF_DOUBLE) :: val_vec(:)
         integer :: status

         ! extracts a vector of finite reals from string
         !
         ! status = 0 if extraction is ok
         !        = End_Of_Record if no more item in string
         !        = 1 if a non numeric value is found

         integer :: nval, i, k, ipos, i_max
         character(len=25) :: double_str

         nval = size(val_vec)
         if( nval == 0 ) then
            status = 0
            return
         end if

         ipos = 1
         i_max = len(string)

         do k = 1, nval

            ! first character must not be blank/TAB
            do
               if( ipos > i_max ) then
                  status = End_Of_Record
                  return
               end if
               if( .not. VS_elem_is_blank(string,ipos) ) then
                  exit
               else
                  ipos = ipos + 1
               end if
            end do

            ! find next blank/TAB or End-of-Line
            i = ipos + 1
            do
               if( i > i_max ) then
                  exit
               end if
               if( .not. VS_elem_is_blank(string,i) ) then
                  i = i + 1
               else
                  exit
               end if
            end do

            double_str = extract(string,ipos,i-1)

            ! reading a substring
            read( double_str, *, iostat=iostat ) val_vec(k)
            if( iostat /= 0 ) then
               status = 1
               return
            end if

            ipos = i

         end do

         status = 0

      end subroutine read_val_vec_in_VS
      !-----------------------------------------------------------------
      subroutine read_vals_in_VS( string, vec, ipos, status,            &
                                  comment )

         type(varying_string), intent(in)            :: string
         real(kind=MF_DOUBLE)                        :: vec(:)
         integer                                     :: ipos, status
         character,            intent(in), optional  :: comment

         ! extracts all possible finite reals from string
         !
         ! status = 0 if extraction is ok
         !        = 1 if a non numeric value is found

         integer :: i1, i2, i_max
         character(len=25) :: double_str
         logical :: commented

         if( present(comment) ) then
            commented = .true.
         else
            commented = .false.
         end if

         i_max = len(string)
         status = 0

         i1 = 1
         do

            ! first character must not be blank/TAB
            do
               if( i1 > i_max ) then
                  return
               end if
               if( .not. VS_elem_is_blank(string,i1) ) then
                  exit
               else
                  i1 = i1 + 1
               end if
            end do

            if( commented ) then
               if( VS_elem_is_same(string,i1,comment) ) return
            end if

            ! find next blank/TAB or End-of-Line
            i2 = i1 + 1
            do
               if( i2 > i_max ) then
                  exit
               end if
               if( .not. VS_elem_is_blank(string,i2) ) then
                  i2 = i2 + 1
               else
                  exit
               end if
            end do

            double_str = extract(string,i1,i2-1)

            ! lecture d'une sous-chaîne
            read( double_str, *, iostat=iostat ) vec(ipos)
            ipos = ipos + 1
            if( iostat /= 0 ) then
               status = 1
               return
            end if

            i1 = i2

         end do

      end subroutine read_vals_in_VS
      !-----------------------------------------------------------------
#endif
   end function mfLoadAscii
