! f90 include file

!_______________________________________________________________________
!
   function mfGradient_real8( F, h, dim, location ) result( out )

      type(mfArray),        intent(in)           :: F
      real(kind=MF_DOUBLE), intent(in)           :: h
      integer,              intent(in), optional :: dim
      character(len=*),     intent(in), optional :: location

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: m, n, i, j, dim0
      logical :: centered
      real(kind=MF_DOUBLE) :: two_h
      real(kind=MF_DOUBLE), allocatable :: Grad(:,:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( F )

      if( F%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfGradient", "E",                          &
                            "'F' must be dense, real!" )
         go to 99
      end if

      ! checking that F is allocated
      if( mfIsEmpty(F) ) then
         call PrintMessage( "mfGradient", "E",                          &
                            "'F' empty!" )
         go to 99
      end if

      m = F%shape(1)
      n = F%shape(2)

      if( present(dim) ) then
         dim0 = dim
      else
         if( m == 1 ) then
            dim0 = 2
         else if( n == 1 ) then
            dim0 = 1
         else
            dim0 = 1
         end if
      end if

      if( dim0 == 1 ) then
         if( m < 3 ) then
            call PrintMessage( "mfGradient", "E",                       &
                               "first dimension must be at least 3!" )
            go to 99
         end if
      else ! dim0 = 2
         if( n < 3 ) then
            call PrintMessage( "mfGradient", "E",                       &
                               "second dimension must be at least 3!" )
            go to 99
         end if
      end if

      centered = .false.
      if( present(location) ) then
         if( location == "centered" ) then
            centered = .true.
         end if
      end if

      out%data_type = MF_DT_DBLE

      allocate( Grad(m,n) )

      two_h = 2.0d0*h

      ! use a second order formula O(h^2)

      if( dim0 == 1 ) then
         ! boundary point: end-point formula
         Grad(1,:) = ( -3*F%double(1,:) + 4*F%double(2,:) - F%double(3,:) ) &
                     / two_h
         ! internal points: centered difference
         do i = 2, m-1
            Grad(i,:) = ( F%double(i+1,:) - F%double(i-1,:) ) / two_h
         end do
         ! boundary point: end-point formula
         Grad(m,:) = ( F%double(m-2,:) - 4*F%double(m-1,:) + 3*F%double(m,:) ) &
                     / two_h
      else ! dim0 = 2
         ! boundary point: end-point formula
         Grad(:,1) = ( -3*F%double(:,1) + 4*F%double(:,2) - F%double(:,3) ) &
                     / two_h
         ! internal points: centered difference
         do j = 2, n-1
            Grad(:,j) = ( F%double(:,j+1) - F%double(:,j-1) ) / two_h
         end do
         ! boundary point: end-point formula
         Grad(:,n) = ( F%double(:,n-2) - 4*F%double(:,n-1) + 3*F%double(:,n) ) &
                     / two_h
      end if

      if( centered ) then
         if( dim0 == 1 ) then
            out%shape = [ m-1, n ]
            allocate( out%double(m-1,n) )
            out%double(:,:) = ( Grad(1:m-1,:) + Grad(2:m,:) ) / 2.0d0
         else ! dim0 = 2
            out%shape = [ m, n-1 ]
            allocate( out%double(m,n-1) )
            out%double(:,:) = ( Grad(:,1:n-1) + Grad(:,2:n) ) / 2.0d0
         end if
      else
         if( dim0 == 1 ) then
            out%shape = [ m, n ]
            allocate( out%double(m,n) )
            out%double(:,:) = Grad(:,:)
         else ! dim0 = 2
            out%shape = [ m, n ]
            allocate( out%double(m,n) )
            out%double(:,:) = Grad(:,:)
         end if
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( F )
      call msAutoRelease( F )

#endif
   end function mfGradient_real8
!_______________________________________________________________________
!
   function mfGradient_mfArray( F, h, dim, location ) result( out )

      type(mfArray),    intent(in)           :: F
      type(mfArray),    intent(in)           :: h
      integer,          intent(in), optional :: dim
      character(len=*), intent(in), optional :: location

      type(mfArray) :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, status

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( F, h )

      if( present(dim) ) then
         call msAssign( out, mfGradient_real8(F,mfDble(h),dim,location) )
      else
         call msAssign( out, mfGradient_real8(F,mfDble(h),location=location) )
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         do i = 1, num_base_units
            call rational_sub( F%units(i), h%units(i),                  &
                               out%units(i), status )
            if( status == 1 ) then
               call PrintMessage( "mfGradient", "E",                    &
                                  "in processing physical units:",      &
                                  "integer overflow!" )
               go to 99
            else if( status == -1 ) then
               call PrintMessage( "mfGradient", "E",                    &
                                  "in processing physical units:",      &
                                  "Please report this bug to: Edouard.Canot@univ-rennes.fr" )
               go to 99
            end if
         end do
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( F, h )
      call msAutoRelease( F, h )

#endif
   end function mfGradient_mfArray
!_______________________________________________________________________
!
   subroutine msGradient( out, F, hi, hj, location )

      type(mfArray),        intent(in)           :: F
      real(kind=MF_DOUBLE), intent(in)           :: hi, hj
      character(len=*),     intent(in), optional :: location

      type(mf_Out) :: out
      !------ API end ------
#ifdef _DEVLP

      type(mfArray), pointer :: Fi, Fj
      integer :: m, n, i, j
      logical :: centered
      real(kind=MF_DOUBLE) :: two_hi, two_hj
      real(kind=MF_DOUBLE), allocatable :: Grad1(:), Grad2(:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( F )

      if( F%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msGradient", "E",                          &
                            "'F' must be dense, real!" )
         go to 99
      end if

      ! 2 output args must be specified
      if( out%n /= 2 ) then
         call PrintMessage( "msGradient", "E",                          &
                            "two output args required!",               &
                            "syntax is : call msGradient ( mfOut(Fi,Fj), F, hi, hj[, location] )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, F ) ) then
         call PrintMessage( "msGradient", "E",                          &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      Fi => out%ptr1
      call msSilentRelease( Fi )
      Fj => out%ptr2
      call msSilentRelease( Fj )

      ! checking that F is allocated
      if( mfIsEmpty(F) ) then
         call PrintMessage( "msGradient", "E",                          &
                            "'F' empty!" )
         go to 99
      end if

      m = F%shape(1)
      n = F%shape(2)

      if( m < 3 .or. n < 3 ) then
         call PrintMessage( "msGradient", "E",                          &
                            "'F' must be a matrix, of shape: at least 3x3!" )
         go to 99
      end if

      centered = .false.
      if( present(location) ) then
         if( location == "centered" ) then
            centered = .true.
         end if
      end if

      Fj%data_type = MF_DT_DBLE
      Fi%data_type = MF_DT_DBLE

      two_hj = 2.0d0*hj
      two_hi = 2.0d0*hi

      ! use a second order formula O(h^2) for all points and all cases

      if( centered ) then

         Fj%shape = [ m-1, n-1 ]
         allocate( Fj%double(m-1,n-1) ) ! x = j

         ! temporary vectors
         allocate( Grad1(m), Grad2(m) )

         ! compute first the whole line of Fj at nodes, along j = 1
         Grad1(:) = ( -3*F%double(:,1) + 4*F%double(:,2) - F%double(:,3) ) &
                    / two_hj
         do j = 1, n-2
            ! compute first the whole line of Fj at nodes, along j+1
            Grad2(:) = ( F%double(:,j+2) - F%double(:,j) ) / two_hj
            do i = 1, m-1
               ! here, Fj must be computed at the center of a square
               ! for which Fj is known at its four corners
               Fj%double(i,j) =  ( Grad1(i) + Grad1(i+1)                &
                                 + Grad2(i) + Grad2(i+1) ) / 4
            end do
            Grad1(:) = Grad2(:)
         end do
         ! last whole line of Fj at nodes, along j = n
         Grad2(:) = ( F%double(:,n-2) - 4*F%double(:,n-1) + 3*F%double(:,n) ) &
                    / two_hj
         j = n-1
         do i = 1, m-1
            Fj%double(i,j) = ( Grad1(i) + Grad1(i+1)                    &
                             + Grad2(i) + Grad2(i+1) ) / 4
         end do

         Fi%shape = [ m-1, n-1 ]
         allocate( Fi%double(m-1,n-1) ) ! y = i

         ! update shape of temporary vectors
         if( n /= m ) then
            deallocate( Grad1, Grad2 )
            allocate( Grad1(n), Grad2(n) )
         end if

         ! compute first the whole line of Fi at nodes, along i = 1
         Grad1(:) = ( 3*F%double(1,:) - 4*F%double(2,:) + F%double(3,:) ) &
                    / two_hi
         do i = 1, m-2
            ! compute first the whole line of Fi at nodes, along i+1
            Grad2(:) = ( F%double(i,:) - F%double(i+2,:) ) / two_hi
            do j = 1, n-1
               ! here, Fi must be computed at the center of a square
               ! for which Fi is known at its four corners
               Fi%double(i,j) = -( Grad1(j) + Grad1(j+1)                &
                                 + Grad2(j) + Grad2(j+1) ) / 4
            end do
            Grad1(:) = Grad2(:)
         end do
         ! last whole line of Fj at nodes, along i = m
         Grad2(:) = ( -F%double(m-2,:) + 4*F%double(m-1,:) - 3*F%double(m,:) ) &
                    / two_hi
         i = m-1
         do j = 1, n-1
            Fi%double(i,j) = -( Grad1(j) + Grad1(j+1)                   &
                              + Grad2(j) + Grad2(j+1) ) / 4
         end do

      else

         Fj%shape = F%shape
         allocate( Fj%double(m,n) ) ! x = j

         ! boundary point: end-point formula
         Fj%double(:,1) = ( -3*F%double(:,1) + 4*F%double(:,2) - F%double(:,3) ) &
                          / two_hj
         ! internal points: centered difference
         do j = 2, n-1
            Fj%double(:,j) = ( F%double(:,j+1) - F%double(:,j-1) ) / two_hj
         end do
         ! boundary point: end-point formula
         Fj%double(:,n) = ( F%double(:,n-2) - 4*F%double(:,n-1) + 3*F%double(:,n) ) &
                          / two_hj

         Fi%shape = F%shape
         allocate( Fi%double(m,n) ) ! y = i

         ! boundary point: end-point formula
         Fi%double(1,:) = -( 3*F%double(1,:) - 4*F%double(2,:) + F%double(3,:) ) &
                           / two_hi
         ! internal points: centered difference
         do i = 2, m-1
            Fi%double(i,:) = -( F%double(i-1,:) - F%double(i+1,:) ) / two_hi
         end do
         ! boundary point: end-point formula
         Fi%double(m,:) = -( -F%double(m-2,:) + 4*F%double(m-1,:) - 3*F%double(m,:) ) &
                           / two_hi

      end if

 99   continue

      call msFreeArgs( F )
      call msAutoRelease( F )

#endif
   end subroutine msGradient
