! f90 include file

!_______________________________________________________________________
!
   function mfFourierSin( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------
#ifdef _DEVLP

      ! Discrete Fourier-sine transformation
      !
      ! To be valid, there exist some constraints about the data
      ! vector of length n:
      !   data is supposed to be periodic on [-n,n], equally spaced and
      !   antisymmetric at points 1 and n; this implies that
      !                     y(1) = y(n) = 0

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfFourierSin", "E",                        &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfFourierSin", "E",                        &
                            "data must be real!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! works on columns (as in Matlab)
            do j = 1, A%shape(2)
               call fourier_sin( A%double(:,j), out%double(:,j) )
            end do
         else if( dim == 2 ) then
            ! works on rows
            do i = 1, A%shape(1)
               call fourier_sin( A%double(i,:), out%double(i,:) )
            end do
         else
            call PrintMessage( "mfFourierSin", "E",                     &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
            ! vector
            if( A%shape(1) == 1 ) then
               call fourier_sin( A%double(1,:), out%double(1,:) )
            else if( A%shape(2) == 1 ) then
               call fourier_sin( A%double(:,1), out%double(:,1) )
            end if
         else
            ! matrix case -> works on columns
            do j = 1, A%shape(2)
               call fourier_sin( A%double(:,j), out%double(:,j) )
            end do
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfFourierSin
!_______________________________________________________________________
!
   subroutine fourier_sin( y, y_chap )

      real(kind=MF_DOUBLE), intent(in) :: y(:)
      real(kind=MF_DOUBLE) :: y_chap(:)
      !------ API end ------
#ifdef _DEVLP

      ! compute the Fourier modes of the function y
      ! known in n+1 equidistant points.
      !
      !   assumption over y :
      !               anti-symmetric w.r.t. the point 1
      !           and anti-symmetric w.r.t. the point n+1
      !
      !   procedure :
      !     decompose y in sine-serie

      integer :: i, k, n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(y) - 1

      !-- Sine-Fourier transformation.

      do k = 1, n+1

         y_chap(k) = y(1) + y(n+1)*sin(MF_PI*(k-1))
         do i = 2, n
            y_chap(k) = y_chap(k) + y(i)*( sin(MF_PI*(i-1)*(k-1)/dble(n)) &
                                         - sin(MF_PI*(2*n-i+1)*(k-1)/dble(n)) )
         enddo

         y_chap(k) = y_chap(k)/n

      enddo

      y_chap(1) = y_chap(1)/2.0d0
      y_chap(n+1) = y_chap(n+1)/2.0d0

#endif
   end subroutine fourier_sin
!_______________________________________________________________________
!
   function mfInvFourierSin( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfInvFourierSin", "E",                     &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfInvFourierSin", "E",                     &
                            "data must be real!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! works on columns (as in Matlab)
            do j = 1, A%shape(2)
               call inv_fourier_sin( A%double(:,j), out%double(:,j) )
            end do
         else if( dim == 2 ) then
            ! works on rows
            do i = 1, A%shape(1)
               call inv_fourier_sin( A%double(i,:), out%double(i,:) )
            end do
         else
            call PrintMessage( "mfInvFourierSin", "E",                  &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
            ! vector
            if( A%shape(1) == 1 ) then
               call inv_fourier_sin( A%double(1,:), out%double(1,:) )
            else if( A%shape(2) == 1 ) then
               call inv_fourier_sin( A%double(:,1), out%double(:,1) )
            end if
         else
            ! matrix case -> works on columns
            do j = 1, A%shape(2)
               call inv_fourier_sin( A%double(:,j), out%double(:,j) )
            end do
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfInvFourierSin
!_______________________________________________________________________
!
   subroutine inv_fourier_sin( y_chap, y )

      real(kind=MF_DOUBLE), intent(in) :: y_chap(:)
      real(kind=MF_DOUBLE) :: y(:)
      !------ API end ------
#ifdef _DEVLP

      ! compute the function from the Fourier modes obtained
      ! by 'fourier_sin'

      integer :: i, k, n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(y) - 1

      !-- Sine-Fourier inverse transformation.

      do i = 1, n+1
         y(i) = 0.0d0
         do k = 1, n+1
            y(i) = y(i) + y_chap(k)*sin(MF_PI*(k-1)*(i-1)/dble(n))
         end do
      end do

#endif
   end subroutine inv_fourier_sin
