! f90 include file

!_______________________________________________________________________
!
   function mfFourierCos( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------
#ifdef _DEVLP

      ! Discrete Fourier-cosine transformation
      !
      ! To be valid, there exist some constraints about the data
      ! vector of length n:
      !   data is supposed to be periodic on [-n,n], with an
      !   implicit symmetry at point 1; data must be also equally
      !   spaced.

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfFourierCos", "E",                        &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfFourierCos", "E",                        &
                            "data must be real!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! works on columns (as in Matlab)
            do j = 1, A%shape(2)
               call fourier_cos( A%double(:,j), out%double(:,j) )
            end do
         else if( dim == 2 ) then
            ! works on rows
            do i = 1, A%shape(1)
               call fourier_cos( A%double(i,:), out%double(i,:) )
            end do
         else
            call PrintMessage( "mfFourierCos", "E",                     &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else ! dim is not present
         if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
            ! vector
            if( A%shape(1) == 1 ) then
               call fourier_cos( A%double(1,:), out%double(1,:) )
            else if( A%shape(2) == 1 ) then
               call fourier_cos( A%double(:,1), out%double(:,1) )
            end if
         else
            ! matrix case -> works on columns
            do j = 1, A%shape(2)
               call fourier_cos( A%double(:,j), out%double(:,j) )
            end do
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfFourierCos
!_______________________________________________________________________
!
   subroutine fourier_cos( y, y_chap )

      real(kind=MF_DOUBLE), intent(in) :: y(:)
      real(kind=MF_DOUBLE) :: y_chap(:)
      !------ API end ------
#ifdef _DEVLP

      ! compute the Fourier modes of the function y
      ! known in n+1 equidistant points.
      !
      !   assumption over y :
      !                  symmetric w.r.t. the point 1
      !              and symmetric w.r.t. the point n+1
      !
      !   procedure :
      !     decompose y in cosine-serie

      integer :: i, k, n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(y) - 1

      !-- Cosine-Fourier transformation.

      do k = 1, n+1

         y_chap(k) = y(1) + y(n+1)*cos(MF_PI*(k-1))
         do i = 2, n
            y_chap(k) = y_chap(k) + y(i)*( cos(MF_PI*(i-1)*(k-1)/dble(n)) &
                                         + cos(MF_PI*(2*n-i+1)*(k-1)/dble(n)) )
         end do

         y_chap(k) = y_chap(k)/n

      end do

      y_chap(1) = y_chap(1)/2.0d0
      y_chap(n+1) = y_chap(n+1)/2.0d0

#endif
   end subroutine fourier_cos
!_______________________________________________________________________
!
   function mfInvFourierCos( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfInvFourierCos", "E",                     &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfInvFourierCos", "E",                     &
                            "data must be real!" )
         go to 99
      end if

      out%data_type = A%data_type
      out%shape = A%shape
      allocate( out%double(out%shape(1),out%shape(2)) )

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! works on columns (as in Matlab)
            do j = 1, A%shape(2)
               call inv_fourier_cos( A%double(:,j), out%double(:,j) )
            end do
         else if( dim == 2 ) then
            ! works on rows
            do i = 1, A%shape(1)
               call inv_fourier_cos( A%double(i,:), out%double(i,:) )
            end do
         else
            call PrintMessage( "mfInvFourierCos", "E",                  &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
            ! vector
            if( A%shape(1) == 1 ) then
               call inv_fourier_cos( A%double(1,:), out%double(1,:) )
            else if( A%shape(2) == 1 ) then
               call inv_fourier_cos( A%double(:,1), out%double(:,1) )
            end if
         else
            ! matrix case -> works on columns
            do j = 1, A%shape(2)
               call inv_fourier_cos( A%double(:,j), out%double(:,j) )
            end do
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfInvFourierCos
!_______________________________________________________________________
!
   subroutine inv_fourier_cos( y_chap, y )

      real(kind=MF_DOUBLE), intent(in) :: y_chap(:)
      real(kind=MF_DOUBLE) :: y(:)
      !------ API end ------
#ifdef _DEVLP

      ! compute the function from the Fourier modes obtained
      ! by 'fourier_cos'

      integer :: i, k, n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(y) - 1

      !-- Cosine-Fourier inverse transformation.

      do i = 1, n+1
         y(i) = 0.0d0
         do k = 1, n+1
            y(i) = y(i) + y_chap(k)*cos(MF_PI*(k-1)*(i-1)/dble(n))
         end do
      end do

#endif
   end subroutine inv_fourier_cos
