! f90 include file

!_______________________________________________________________________
!
   function mfMean( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfMean", "E",                              &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfMean", "E",                              &
                            "cannot be applied to boolean matrices!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfMean", "E",                              &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      out%data_type = A%data_type
      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! processing by column (as in Matlab)
            out%shape = [ 1, a%shape(2) ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do j = 1, A%shape(2)
                  out%double(1,j) = sum(A%double(:,j))/A%shape(1)
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do j = 1, A%shape(2)
                  out%cmplx(1,j) = sum(A%cmplx(:,j))/A%shape(1)
               end do
            end if
         else if( dim == 2 ) then
            ! processing by row
            out%shape = [ A%shape(1), 1 ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do i = 1, A%shape(1)
                  out%double(i,1) = sum(A%double(i,:))/A%shape(2)
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do i = 1, A%shape(1)
                  out%cmplx(i,1) = sum(a%cmplx(i,:))/A%shape(2)
               end do
            end if
         else
            call PrintMessage( "mfMean", "E",                           &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
            ! vector
            out%shape = [ 1, 1 ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               if( A%shape(1) == 1 ) then
                  out%double(1,1) = sum(A%double(1,:))/A%shape(2)
               else if( A%shape(2) == 1 ) then
                  out%double(1,1) = sum(A%double(:,1))/A%shape(1)
               end if
            else if( A%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               if( A%shape(1) == 1 ) then
                  out%cmplx(1,1) = sum(A%cmplx(1,:))/A%shape(2)
               else if( A%shape(2) == 1 ) then
                  out%cmplx(1,1) = sum(A%cmplx(:,1))/A%shape(1)
               end if
            end if
         else
            ! matrix -> processing by column
            out%shape = [ 1, A%shape(2) ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do j = 1, A%shape(2)
                  out%double(1,j) = sum(A%double(:,j))/A%shape(1)
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               allocate( out%cmplx(out%shape(1),out%shape(2)) )

               do j = 1, A%shape(2)
                  out%cmplx(1,j) = sum(A%cmplx(:,j))/A%shape(1)
               end do
            end if
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfMean
!_______________________________________________________________________
!
   function mfMedian( a, dim ) result( out )

      type(mfArray)                 :: a
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( a )

      if( mfIsEmpty(a) ) then
         go to 99
      end if

      if( mfIsSparse(a) ) then
         call PrintMessage( "mfMedian", "E",                            &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( a%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfMedian", "E",                            &
                            "arg must be a real mfArray!" )
         go to 99
      end if

      out%data_type = a%data_type

      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! works on columns (as in Matlab)
            out%shape = [ 1, a%shape(2) ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            do j = 1, a%shape(2)
               call median( a%double(:,j), out%double(1,j) )
            end do
         else if( dim == 2 ) then
            ! works on rows
            out%shape = [ a%shape(1), 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            do i = 1, a%shape(1)
               call median( a%double(i,:), out%double(i,1) )
            end do
         else
            call PrintMessage( "mfMedian", "E",                         &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( a%shape(1) == 1 .or. a%shape(2) == 1 ) then
            ! A is a vector
            out%shape = [ 1, 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( a%shape(1) == 1 ) then
               call median( a%double(1,:), out%double(1,1) )
            else if( a%shape(2) == 1 ) then
               call median( a%double(:,1), out%double(1,1) )
            end if
         else
            ! matrix -> works on column
            out%shape = [ 1, a%shape(2) ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            do j = 1, a%shape(2)
               call median( a%double(:,j), out%double(1,j) )
            end do
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = a%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( a )
      call msAutoRelease( a )

#endif
   end function mfMedian
!_______________________________________________________________________
!
   subroutine median( vec, res )

      real(kind=MF_DOUBLE), intent(in) :: vec(:)
      real(kind=MF_DOUBLE)             :: res
      !------ API end ------
#ifdef _DEVLP

      real(kind=MF_DOUBLE), allocatable :: v(:)
      integer :: n

   !------ end of declarations -- execution starts hereafter  ------

      n = size(vec)
      if( n == 1 ) then
         res = vec(1)
      else if( n == 2 ) then
         res = (vec(1)+vec(2))/2.0d0
      else
         allocate( v(n) )

         v(:) = vec(:)
         call quick_sort( "asc", v )
         if( mod(n,2) == 0 ) then ! n est pair
            res = ( v(n/2) + v(n/2+1) )/2.0d0
         else ! n is odd
            res = v( (1+n)/2 )
         end if
      end if

#endif
   end subroutine median
!_______________________________________________________________________
!
   function mfVar( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, j, m, n
      real(kind=MF_DOUBLE) :: x_bar
      complex(kind=MF_DOUBLE) :: z_bar

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfVar", "E",                               &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfVar", "E",                               &
                            "cannot be applied to boolean matrices!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfVar", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)

      out%data_type = MF_DT_DBLE
      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! processing by column (as in Matlab)
            out%shape = [ 1, n ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               do j = 1, n
                  x_bar = sum(A%double(:,j))/m
                  out%double(1,j) = sum( abs(A%double(:,j)-x_bar)**2 )/m
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               do j = 1, n
                  z_bar = sum(A%cmplx(:,j))/m
                  out%double(1,j) = sum( abs(A%cmplx(:,j)-x_bar)**2 )/m
               end do
            end if
         else if( dim == 2 ) then
            ! processing by row
            out%shape = [ m, 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               do i = 1, m
                  x_bar = sum(A%double(i,:))/n
                  out%double(i,1) = sum( abs(A%double(i,:)-x_bar)**2 )/n
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               do i = 1, m
                  z_bar = sum(a%cmplx(i,:))/n
                  out%double(i,1) = sum( abs(A%cmplx(i,:)-x_bar)**2 )/m
               end do
            end if
         else
            call PrintMessage( "mfVar", "E",                            &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( m == 1 .or. n == 1 ) then
            ! vector
            out%shape = [ 1, 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               if( m == 1 ) then
                  x_bar = sum(A%double(1,:))/n
                  out%double(1,1) = sum( abs(A%double(1,:)-x_bar)**2 )/n
               else if( n == 1 ) then
                  x_bar = sum(A%double(:,1))/m
                  out%double(1,1) = sum( abs(A%double(:,1)-x_bar)**2 )/m
               end if
            else if( A%data_type == MF_DT_CMPLX ) then
               if( m == 1 ) then
                  z_bar = sum(A%cmplx(1,:))/n
                  out%double(1,1) = sum( abs(A%cmplx(1,:)-x_bar)**2 )/m
               else if( n == 1 ) then
                  z_bar = sum(A%cmplx(:,1))/m
                  out%double(1,1) = sum( abs(A%cmplx(:,1)-x_bar)**2 )/m
               end if
            end if
         else
            ! matrix -> processing by column
            out%shape = [ 1, n ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               do j = 1, n
                  x_bar = sum(A%double(:,j))/m
                  out%double(1,j) = sum( abs(A%double(:,j)-x_bar)**2 )/m
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               do j = 1, n
                  z_bar = sum(A%cmplx(:,j))/m
                  out%double(1,j) = sum( abs(A%cmplx(:,j)-x_bar)**2 )/m
               end do
            end if
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfVar
!_______________________________________________________________________
!
   function mfStd( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, j, m, n
      real(kind=MF_DOUBLE) :: x_bar
      complex(kind=MF_DOUBLE) :: z_bar

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfStd", "E",                               &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( A%data_type == MF_DT_BOOL ) then
         call PrintMessage( "mfStd", "E",                               &
                            "cannot be applied to boolean matrices!" )
         go to 99
      end if

      if( mfIsPerm(a) ) then
         call PrintMessage( "mfStd", "E",                               &
                            "cannot be applied to a permutation vector!" )
         go to 99
      end if

      m = A%shape(1)
      n = A%shape(2)

      out%data_type = MF_DT_DBLE
      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! processing by column (as in Matlab)
            out%shape = [ 1, n ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               do j = 1, n
                  x_bar = sum(A%double(:,j))/m
                  out%double(1,j) = sqrt( sum( abs(A%double(:,j)-x_bar)**2 )/m )
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               do j = 1, n
                  z_bar = sum(A%cmplx(:,j))/m
                  out%double(1,j) = sqrt( sum( abs(A%cmplx(:,j)-x_bar)**2 )/m )
               end do
            end if
         else if( dim == 2 ) then
            ! processing by row
            out%shape = [ m, 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               do i = 1, m
                  x_bar = sum(A%double(i,:))/n
                  out%double(i,1) = sqrt( sum( abs(A%double(i,:)-x_bar)**2 )/n )
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               do i = 1, m
                  z_bar = sum(a%cmplx(i,:))/n
                  out%double(i,1) = sqrt( sum( abs(A%cmplx(i,:)-x_bar)**2 )/m )
               end do
            end if
         else
            call PrintMessage( "mfStd", "E",                            &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( m == 1 .or. n == 1 ) then
            ! vector
            out%shape = [ 1, 1 ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               if( m == 1 ) then
                  x_bar = sum(A%double(1,:))/n
                  out%double(1,1) = sqrt( sum( abs(A%double(1,:)-x_bar)**2 )/n )
               else if( n == 1 ) then
                  x_bar = sum(A%double(:,1))/m
                  out%double(1,1) = sqrt( sum( abs(A%double(:,1)-x_bar)**2 )/m )
               end if
            else if( A%data_type == MF_DT_CMPLX ) then
               if( m == 1 ) then
                  z_bar = sum(A%cmplx(1,:))/n
                  out%double(1,1) = sqrt( sum( abs(A%cmplx(1,:)-x_bar)**2 )/m )
               else if( n == 1 ) then
                  z_bar = sum(A%cmplx(:,1))/m
                  out%double(1,1) = sqrt( sum( abs(A%cmplx(:,1)-x_bar)**2 )/m )
               end if
            end if
         else
            ! matrix -> processing by column
            out%shape = [ 1, n ]
            allocate( out%double(out%shape(1),out%shape(2)) )

            if( A%data_type == MF_DT_DBLE ) then
               do j = 1, n
                  x_bar = sum(A%double(:,j))/m
                  out%double(1,j) = sqrt( sum( abs(A%double(:,j)-x_bar)**2 )/m )
               end do
            else if( A%data_type == MF_DT_CMPLX ) then
               do j = 1, n
                  z_bar = sum(A%cmplx(:,j))/m
                  out%double(1,j) = sqrt( sum( abs(A%cmplx(:,j)-x_bar)**2 )/m )
               end do
            end if
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfStd
!_______________________________________________________________________
!
   function mfRMS( A, dim ) result( out )

      type(mfArray)                 :: A
      integer, intent(in), optional :: dim
      type(mfArray)                 :: out
      !------ API end ------
#ifdef _DEVLP

      integer :: i, j

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( mfIsSparse(A) ) then
         call PrintMessage( "mfRMS", "E",                               &
                            "sparse matrices not handled!" )
         go to 99
      end if

      if( .not. mfIsReal(A) ) then
         call PrintMessage( "mfRMS", "E",                               &
                            "mfArray must be real!" )
         go to 99
      end if

      out%data_type = A%data_type
      if( present(dim) ) then
         ! A is always considered as a matrix
         if( dim == 1 ) then
            ! processing by column (as in Matlab)
            out%shape = [ 1, a%shape(2) ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do j = 1, A%shape(2)
                  out%double(1,j) = sqrt(sum(A%double(:,j)**2)/A%shape(1))
               end do
            end if
         else if( dim == 2 ) then
            ! processing by row
            out%shape = [ A%shape(1), 1 ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do i = 1, A%shape(1)
                  out%double(i,1) = sqrt(sum(A%double(i,:)**2)/A%shape(2))
               end do
            end if
         else
            call PrintMessage( "mfRMS", "E",                            &
                               "dim must be equal to 1 or 2!" )
            go to 99
         end if
      else
         if( A%shape(1) == 1 .or. A%shape(2) == 1 ) then
            ! vector
            out%shape = [ 1, 1 ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               if( A%shape(1) == 1 ) then
                  out%double(1,1) = sqrt(sum(A%double(1,:)**2)/A%shape(2))
               else if( A%shape(2) == 1 ) then
                  out%double(1,1) = sqrt(sum(A%double(:,1)**2)/A%shape(1))
               end if
            end if
         else
            ! matrix -> processing by column
            out%shape = [ 1, A%shape(2) ]
            if( A%data_type == MF_DT_DBLE ) then
               allocate( out%double(out%shape(1),out%shape(2)) )

               do j = 1, A%shape(2)
                  out%double(1,j) = sqrt(sum(A%double(:,j)**2)/A%shape(1))
               end do
            end if
         end if
      end if

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

#endif
   end function mfRMS
!_______________________________________________________________________
!
   function mfQuantile( v, p ) result( q )

      type(mfArray)                    :: v
      real(kind=MF_DOUBLE), intent(in) :: p
      type(mfArray)                    :: q
      !------ API end ------
#ifdef _DEVLP

      ! Should return exactly the same value as Matlab (or Octave),
      ! i.e. using the type 6 of Quantile Definition of Hyndman & Fan.

      real(kind=MF_DOUBLE), allocatable :: x2(:), v2(:)
      real(kind=MF_DOUBLE) :: quantile, t
      integer :: i, n, dim

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( v )

      if( mfIsEmpty(v) ) then
         go to 99
      end if

      if( size(v,1) == 1 ) then
         dim = 2
      else if( size(v,2) == 1 ) then
         dim = 1
      else
         call PrintMessage( "mfQuantile", "E",                          &
                            "'v' must be a vector!" )
         go to 99
      end if

      if( p < 0.0d0 .or. 1.0d0 < p ) then
         call PrintMessage( "mfQuantile", "E",                          &
                            "'p' is out-of-range! (it must be in [0,1]" )
         go to 99
      end if

      if( v%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfQuantile", "E",                          &
                            "'v' must be a real vector" )
         go to 99
      end if

      q%data_type = v%data_type
      q%shape = [ 1, 1 ]
      allocate( q%double(q%shape(1),q%shape(2)) )

      n = size(v)

      allocate( x2(n) )
      x2 = [ ( dble(i), i = 1, n ) ]
      x2 = ( x2 - 0.5d0 ) / n ! as in Matlab

      allocate( v2(n) )
      v2 = mfSort( v )

      ! Bornes
      if( p <= x2(1) ) then
         q = v2(1)
         return
      else if( p >= x2(n) ) then
         q = v2(n)
         return
      end if

      do i = 1, n-1
         if( p <= x2(i+1) ) then
            t = ( p - x2(i) )/( x2(i+1) - x2(i) )
            quantile = v2(i) + t*( v2(i+1) - v2(i) )
            exit
         end if
      end do

      ! Linear interpolation
      q%double(1,1) = quantile

      q%prop%symm = FALSE

      if( mf_phys_units ) then
         q%units(:) = v%units(:)
      end if

      q%status_temporary = .true.

 99   continue

      call msFreeArgs( v )
      call msAutoRelease( v )

#endif
   end function mfQuantile
