! f90 include file

!_______________________________________________________________________
!
   function mfSort_vec( A, mode ) result( out )

      type(mfArray)                          :: A
      character(len=*), intent(in), optional :: mode
      type(mfArray)                          :: out
      !------ API end ------
#ifdef _DEVLP

      ! this routine accepts a complex vector: sorting is done first
      ! on module, then on phase-angle.

      ! also accepts NaNs : these special IEEE values are moved
      ! at the end of the sorted vector (as in Matlab).

      ! sort is done by default by increasing value ("ascending")
      ! or by decreasing value ("descending"); only the three first
      ! characters of the optional 'mode' are tested.

      type(mfArray) :: nans
      real(kind=MF_DOUBLE), pointer :: tmp(:)
      integer, allocatable :: ind(:)
      integer :: i, nb_NaNs, n
      character(len=3) :: mode_sort

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( .not. mfIsNumeric(A) ) then
         call PrintMessage( "mfSort", "E",                              &
                            "numeric array required!" )
         go to 99
      end if

      if( A%shape(1) /= 1 .and. A%shape(2) /= 1 ) then
         call PrintMessage( "mfSort", "E",                              &
                            "mfArray 'A' should be a vector!",          &
                            "(please specify 'dim' for a matrix)")
         go to 99
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            mode_sort = "asc"
         else if( to_lower(mode(1:3)) == "des" ) then
            mode_sort = "des"
         else
            call PrintMessage( "mfSort", "E",                           &
                               "bad arg value for 'mode' (must be 'asc' or 'des'!" )
            go to 99
         end if
      else
         mode_sort = "asc"
      end if

      call msAssign( nans, mfIsNaN(A) )
      nb_NaNs = mfCount( nans )
      n = size(A)
      if( nb_NaNs == n ) then
         out = A
         go to 98
      end if

      if( A%data_type == MF_DT_DBLE ) then

         out%data_type = MF_DT_DBLE
         out%shape = A%shape
         allocate( out%double(A%shape(1),A%shape(2)) )

         if( nb_NaNs >= 1 ) then
            if( A%shape(1) == 1 ) then ! row vector
               call msAssign( out, mfGet( A, mfFind(.not. nans) ) .hc.  &
                                   MF_NAN*mfOnes(1,nb_NaNs) )
            else                       ! col vector
               call msAssign( out, mfGet( A, mfFind(.not. nans) ) .vc.  &
                                   MF_NAN*mfOnes(nb_NaNs,1) )
            end if
         else
            out%double(:,:) = A%double(:,:)
         end if

         if( A%shape(1) == 1 ) then ! row vector
            tmp => out%double(1,1:n-nb_NaNs)
         else                       ! col vector
            tmp => out%double(1:n-nb_NaNs,1)
         end if

         call quick_sort( mode_sort, tmp )

      else ! MF_DT_CMPLX

         out%data_type = MF_DT_CMPLX
         out%shape = A%shape
         allocate( out%cmplx(A%shape(1),A%shape(2)) )

         if( A%shape(1) == 1 ) then ! row vector

            if( nb_NaNs >= 1 ) then
               allocate( tmp(n-nb_NaNs) )

               tmp(:) = mfAbs( mfGet( A, mfFind(.not. nans) ) )
               allocate( ind(n-nb_NaNs) )

               ind(:) = [ (i,i=1,n-nb_NaNs) ]
               call quick_sort( mode_sort, tmp, ind )

               call msAssign( out, mfGet( mfGet(A,mfFind(.not. nans)), ind ) )
               ! sorting also according to phase-angle
               call sort_cmplx_by_phase_angle( mode_sort, tmp, out%cmplx(1,:) )
               deallocate( tmp )
               call msAssign( out, out .hc. (MF_NAN*mfOnes(1,nb_NaNs))*MF_I )
            else
               allocate( tmp(n) )

               tmp(:) = abs( A%cmplx(1,:) )
               allocate( ind(n) )

               ind(:) = [ (i,i=1,n) ]
               call quick_sort( mode_sort, tmp, ind )

               out%cmplx(1,:) = A%cmplx(1,ind(:))
               ! sorting also according to phase-angle
               call sort_cmplx_by_phase_angle( mode_sort, tmp, out%cmplx(1,:) )
               deallocate( tmp )
            end if

         else                       ! col vector

            if( nb_NaNs >= 1 ) then
               allocate( tmp(n-nb_NaNs) )

               tmp(:) = mfAbs( mfGet( A, mfFind(.not. nans) ) )
               allocate( ind(n-nb_NaNs) )

               ind(:) = [ (i,i=1,n-nb_NaNs) ]
               call quick_sort( mode_sort, tmp, ind )

               call msAssign( out, mfGet( mfGet(A,mfFind(.not. nans)), ind ) )
               ! sorting also according to phase-angle
               call sort_cmplx_by_phase_angle( mode_sort, tmp, out%cmplx(:,1) )
               deallocate( tmp )
               call msAssign( out, out .vc. (MF_NAN*mfOnes(nb_NaNs,1))*MF_I )
            else
               allocate( tmp(A%shape(1)) )

               tmp(:) = abs( A%cmplx(:,1) )
               allocate( ind(A%shape(1)) )

               ind(:) = [ (i,i=1,A%shape(1)) ]
               call quick_sort( mode_sort, tmp, ind )

               out%cmplx(:,1) = A%cmplx(ind(:),1)
               ! sorting also according to phase-angle
               call sort_cmplx_by_phase_angle( mode_sort, tmp, out%cmplx(:,1) )
               deallocate( tmp )
            end if

         end if

      end if

 98   continue

      out%prop%symm = FALSE

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )
      call msAutoRelease( A )

      call msSilentRelease( nans )

#endif
   end function mfSort_vec
!_______________________________________________________________________
!
   function mfSort_matrix( A, dim, mode ) result( out )

      type(mfArray)                          :: A
      integer,          intent(in)           :: dim
      character(len=*), intent(in), optional :: mode
      type(mfArray)                          :: out
      !------ API end ------
#ifdef _DEVLP

      real(kind=MF_DOUBLE), pointer :: tmp(:)
      integer :: i, j
      character(len=3) :: mode_sort

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfSort", "E",                              &
                            "real array required!" )
         go to 99
      end if

      if( Any(mfIsNaN(A)) ) then
         call PrintMessage( "mfSort", "E",                              &
                            "NaNs are not allowed when sorting matrices!" )
         go to 99
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            mode_sort = "asc"
         else if( to_lower(mode(1:3)) == "des" ) then
            mode_sort = "des"
         else
            call PrintMessage( "mfSort", "E",                           &
                               "bad arg value for 'mode' (must be 'asc' or 'des'!" )
            go to 99
         end if
      else
         mode_sort = "asc"
      end if

      out%data_type = MF_DT_DBLE
      out%shape = A%shape
      allocate( out%double(A%shape(1),A%shape(2)) )

      out%double(:,:) = A%double(:,:)
      if( dim == 1 ) then
         do j = 1, A%shape(2)
            tmp => out%double(:,j)
            call quick_sort( mode_sort, tmp )
         end do
      else if( dim == 2 ) then
         do i = 1, A%shape(1)
            tmp => out%double(i,:)
            call quick_sort( mode_sort, tmp )
         end do
      else
         call PrintMessage( "mfSort", "E",                              &
                            "dim must be equal to 1 or 2!" )
         go to 99
      end if

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfSort_matrix
!_______________________________________________________________________
!
   subroutine msSort_vec( A, mode )

      type(mfArray)                          :: A
      character(len=*), intent(in), optional :: mode
      !------ API end ------
#ifdef _DEVLP

      ! in-place version of 'mfSort_vec'.
      !
      ! A cannot be tempo. A: real only.

      type(mfArray) :: nans
      real(kind=MF_DOUBLE), pointer :: tmp(:)
      integer :: i, k, nb_NaNs, n
      character(len=3) :: mode_sort

   !------ end of declarations -- execution starts hereafter  ------

      if( A%status_temporary ) then
         call PrintMessage( "msSort", "E",                              &
                            "mfArray 'A' cannot be tempo!" )
         return
      end if

      if( mfIsEmpty(A) ) then
         return
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msSort", "E",                              &
                            "in-place sorting requires real values only!" )
         return
      end if

      if( A%shape(1) /= 1 .and. A%shape(2) /= 1 ) then
         call PrintMessage( "msSort", "E",                              &
                            "mfArray 'A' should be a vector!",          &
                            "(please specify 'dim' for a matrix)")
         return
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            mode_sort = "asc"
         else if( to_lower(mode(1:3)) == "des" ) then
            mode_sort = "des"
         else
            call PrintMessage( "msSort", "E",                           &
                               "bad arg value for 'mode' (must be 'asc' or 'des'!" )
            return
         end if
      else
         mode_sort = "asc"
      end if

      call msAssign( nans, mfIsNaN(A) )
      nb_NaNs = mfCount( nans )
      n = size(A)
      if( nb_NaNs == n ) then
         go to 98
      end if

      if( nb_NaNs >= 1 ) then
         ! shift NaNs values at the end of the vector
         if( A%shape(1) == 1 ) then ! row vector
            k = 0
            do i = 1, n
               if( nans%double(1,i) == 0.0d0 ) cycle
               k = k + 1
               ! swap 'i' and 'n-k+1' elements
               A%double(1,i) = A%double(1,n-k+1)
               A%double(1,n-k+1) = MF_NAN
            end do
         else                       ! col vector
            k = 0
            do i = 1, n
               if( nans%double(i,1) == 0.0d0 ) cycle
               k = k + 1
               ! swap 'i' and 'n-k+1' elements
               A%double(i,1) = A%double(n-k+1,1)
               A%double(n-k+1,1) = MF_NAN
            end do
         end if
      end if

      if( A%shape(1) == 1 ) then ! row vector
         tmp => A%double(1,1:n-nb_NaNs)
      else                       ! col vector
         tmp => A%double(1:n-nb_NaNs,1)
      end if

      call quick_sort( mode_sort, tmp )

 98   continue

      A%prop%symm = FALSE

 99   continue

      call msSilentRelease( nans )

#endif
   end subroutine msSort_vec
!_______________________________________________________________________
!
   subroutine msSort_matrix( A, dim, mode )

      type(mfArray)                          :: A
      integer,          intent(in)           :: dim
      character(len=*), intent(in), optional :: mode
      !------ API end ------
#ifdef _DEVLP

      ! in-place version of 'mfSort_matrix'.
      !
      ! A cannot be tempo.

      real(kind=MF_DOUBLE), pointer :: tmp(:)
      integer :: i, j
      character(len=3) :: mode_sort

   !------ end of declarations -- execution starts hereafter  ------

      if( A%status_temporary ) then
         call PrintMessage( "msSort", "E",                              &
                            "mfArray 'A' cannot be tempo!" )
         return
      end if

      if( mfIsEmpty(A) ) then
         return
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msSort", "E",                              &
                            "real array required!" )
         return
      end if

      if( Any(mfIsNaN(A)) ) then
         call PrintMessage( "msSort", "E",                              &
                            "NaNs are not allowed when sorting matrices!" )
         return
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            mode_sort = "asc"
         else if( to_lower(mode(1:3)) == "des" ) then
            mode_sort = "des"
         else
            call PrintMessage( "msSort", "E",                           &
                               "bad arg value for 'mode' (must be 'asc' or 'des'!" )
            return
         end if
      else
         mode_sort = "asc"
      end if

      if( dim == 1 ) then
         do j = 1, A%shape(2)
            tmp => A%double(:,j)
            call quick_sort( mode_sort, tmp )
         end do
      else if( dim == 2 ) then
         do i = 1, A%shape(1)
            tmp => A%double(i,:)
            call quick_sort( mode_sort, tmp )
         end do
      else
         call PrintMessage( "msSort", "E",                              &
                            "dim must be equal to 1 or 2!" )
         return
      end if

#endif
   end subroutine msSort_matrix
!_______________________________________________________________________
!
   subroutine msSort_vec_out( out, A, mode )

      type(mfArray)                          :: A
      character(len=*), intent(in), optional :: mode
      type(mf_Out)                           :: out
      !------ API end ------
#ifdef _DEVLP

      ! this routine accepts a complex vector: sorting is done first
      ! on module, then on phase-angle.

      ! also accepts NaNs : these special IEEE values are moved
      ! at the end of the sorted vector (as in Matlab).

      ! sort is done by default by increasing value ("ascending")
      ! or by decreasing value ("descending"); only the three first
      ! characters of the optional 'mode' are tested.

      type(mfArray), pointer :: va, ia

      type(mfArray) :: nans
      real(kind=MF_DOUBLE), allocatable :: tmp(:)
      real(kind=MF_DOUBLE), pointer :: ind(:)
      integer :: i, nb_NaNs, n
      character(len=3) :: mode_sort
      integer, allocatable :: ind_2(:)

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! we must have two output arguments
      if( out%n /= 2 ) then
         call PrintMessage( "msSort", "E",                              &
                            "two output args required!",                &
                            "syntax is : call msSort ( mfOut(v,i), a )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msSort", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      va => out%ptr1
      ia => out%ptr2
!### Warning : what happens if 'va' and 'A' share the same address ?
! (for example, a destructive sort can be chosen!)
! is code below ok ?
      if( a%data_type == MF_DT_DBLE ) then
         if( same_address(va,a) == 0 ) then
            call msSilentRelease( va )
         end if
      else
         call msSilentRelease( va )
      end if
      call msSilentRelease( ia )

      if( mfIsEqual(A,MF_EMPTY) ) then
         go to 99
      end if

      if( .not. mfIsNumeric(A) ) then
         call PrintMessage( "msSort", "E",                              &
                            "numeric array required!" )
         go to 99
      end if

      if( A%shape(1) /= 1 .and. A%shape(2) /= 1 ) then
         call PrintMessage( "msSort", "E",                              &
                            "mfArray 'A' must be a vector!",            &
                            "(please specify 'dim' for a matrix)" )
         go to 99
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            mode_sort = "asc"
         else if( to_lower(mode(1:3)) == "des" ) then
            mode_sort = "des"
         else
            call PrintMessage( "msSort", "E",                           &
                               "bad arg value for 'mode' (must be 'asc' or 'des'!" )
            go to 99
         end if
      else
         mode_sort = "asc"
      end if

      call msAssign( nans, mfIsNaN(A) )
      nb_NaNs = mfCount( nans )
      n = size(A)
      if( nb_NaNs == n ) then
         va = A
         if( a%shape(1) == 1 ) then ! row vector
            call msAssign( ia, mfColon( 1, n ) )
         else                       ! col vector
            call msAssign( ia, .t. mfColon( 1, n ) )
         end if
         go to 98
      end if

      if( A%data_type == MF_DT_DBLE ) then
         if( nb_NaNs >= 1 ) then
            call msAssign( ia, mfFind(.not. nans) .hc. mfFind(nans) )
            if( A%shape(1) == 1 ) then ! row vector
               call msAssign( va, mfGet( A, mfFind(.not. nans) ) .hc.   &
                                  MF_NAN*mfOnes(1,nb_NaNs) )
            else                       ! col vector
               call msAssign( ia, .t. ia )
               call msAssign( va, mfGet( A, mfFind(.not. nans) ) .vc.   &
                                  MF_NAN*mfOnes(nb_NaNs,1) )
            end if
         else ! no NaNs
            va = A
         end if
      else ! MF_DT_CMPLX
         if( nb_NaNs >= 1 ) then
            call msAssign( ia, mfFind(.not. nans) .hc. mfFind(nans) )
            if( A%shape(1) == 1 ) then ! row vector
               call msAssign( va, mfAbs( mfGet( A, mfFind(.not. nans) ) ) .hc. &
                                  MF_NAN*mfOnes(1,nb_NaNs) )
            else                       ! col vector
               call msAssign( ia, .t. ia )
               call msAssign( va, mfAbs( mfGet( A, mfFind(.not. nans) ) ) .vc. &
                                  MF_NAN*mfOnes(nb_NaNs,1) )
            end if
         else ! no NaNs
            call msAssign( va, mfAbs(A) )
         end if
      end if

      if( nb_NaNs >= 1 ) then
         allocate( tmp(n-nb_NaNs) )
         if( A%shape(1) == 1 ) then
            tmp =  va%double(1,1:n-nb_NaNs)
            ind => ia%double(1,1:n-nb_NaNs)
         else if( A%shape(2) == 1 ) then
            tmp =  va%double(1:n-nb_NaNs,1)
            ind => ia%double(1:n-nb_NaNs,1)
         end if
      else ! no NaNs
         allocate( tmp(n) )
         ia = [ (i,i=1,n) ]
         if( A%shape(1) == 1 ) then
            tmp =  va%double(1,:)
            ind => ia%double(1,:)
         else if( A%shape(2) == 1 ) then
            tmp =  va%double(:,1)
            call msAssign( ia, .t. ia )
            ind => ia%double(:,1)
         end if
      end if

      call quick_sort( mode_sort, tmp, ind )

      if( A%data_type == MF_DT_DBLE ) then
         if( nb_NaNs >= 1 ) then
            if( A%shape(1) == 1 ) then ! row vector
               va%double(1,1:n-nb_NaNs) = tmp
            else                       ! col vector
               va%double(1:n-nb_NaNs,1) = tmp
            end if
         else ! no NaNs
            if( A%shape(1) == 1 ) then ! row vector
               va%double(1,:) = tmp
            else                       ! col vector
               va%double(:,1) = tmp
            end if
         end if
      else ! MF_DT_CMPLX
         if( A%shape(1) == 1 ) then ! row vector
            if( nb_NaNs >= 1 ) then
               call msAssign( va, mfGet(A,nint(ind)) )
               ! sorting also according to phase-angle
               allocate( ind_2(n-nb_NaNs) )
               ind_2 = nint(ind)
               call sort_cmplx_by_phase_angle_2( mode_sort, tmp,        &
                                                 va%cmplx(1,:), ind_2 )
               ind = ind_2
               call msAssign( va, va .hc. (MF_NAN*mfOnes(1,nb_NaNs))*MF_I )
            else
               call msAssign( va, mfGet(A,nint(ind)) )
               ! sorting also according to phase-angle
               allocate( ind_2(n) )
               ind_2 = nint(ind)
               call sort_cmplx_by_phase_angle_2( mode_sort, tmp,        &
                                                 va%cmplx(1,:), ind_2 )
               ind = ind_2
            end if
         else                       ! col vector
            if( nb_NaNs >= 1 ) then
               call msAssign( va, mfGet(A,nint(ind)) )
               ! sorting also according to phase-angle
               allocate( ind_2(n-nb_NaNs) )
               ind_2 = nint(ind)
               call sort_cmplx_by_phase_angle_2( mode_sort, tmp,        &
                                                 va%cmplx(:,1), ind_2 )
               ind = ind_2
               call msAssign( va, va .vc. (MF_NAN*mfOnes(nb_NaNs,1))*MF_I )
            else
               call msAssign( va, mfGet(A,nint(ind)) )
               ! sorting also according to phase-angle
               allocate( ind_2(n) )
               ind_2 = nint(ind)
               call sort_cmplx_by_phase_angle_2( mode_sort, tmp,        &
                                                 va%cmplx(:,1), ind_2 )
               ind = ind_2
            end if
         end if
      end if

 98   continue

      va%prop%symm = FALSE
      ia%prop%symm = FALSE

      if( mf_phys_units ) then
         va%units(:) = a%units(:)
      end if

 99   continue

      call msFreeArgs( a )
      call msAutoRelease( a )

      call msSilentRelease( nans )

#endif
   end subroutine msSort_vec_out
!_______________________________________________________________________
!
   subroutine msSort_matrix_out( out, A, dim, mode )

      type(mfArray)                          :: A
      integer,          intent(in)           :: dim
      character(len=*), intent(in), optional :: mode
      type(mf_Out)                           :: out
      !------ API end ------
#ifdef _DEVLP

      type(mfArray), pointer :: va, ia

      real(kind=MF_DOUBLE), pointer :: tmp(:), ind(:)
      integer :: i, j
      character(len=3) :: mode_sort

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! we must have two output arguments
      if( out%n /= 2 ) then
         call PrintMessage( "msSort", "E",                              &
                            "two output args required!",                &
                            "syntax is : call msSort ( mfOut(v,i), A, dim )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, A ) ) then
         call PrintMessage( "msSort", "E",                              &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      va => out%ptr1
      ia => out%ptr2
      call msSilentRelease( va, ia )

      if( mfIsEqual(a,MF_EMPTY) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msSort", "E",                              &
                            "real array required!" )
         go to 99
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            mode_sort = "asc"
         else if( to_lower(mode(1:3)) == "des" ) then
            mode_sort = "des"
         else
            call PrintMessage( "msSort", "E",                           &
                               "bad arg value for 'mode' (must be 'asc' or 'des'!" )
            go to 99
         end if
      else
         mode_sort = "asc"
      end if

      va%data_type = MF_DT_DBLE
      ia%data_type = MF_DT_DBLE
      va%shape = A%shape
      ia%shape = A%shape
      allocate( va%double(A%shape(1),A%shape(2)) )

      allocate( ia%double(A%shape(1),A%shape(2)) )

      va%double(:,:) = A%double(:,:)
      if( dim == 1 ) then
         do j = 1, A%shape(2)
            tmp => va%double(:,j)
            ia%double(:,j) = [ (i,i=1,A%shape(1)) ]
            ind => ia%double(:,j)
            call quick_sort( mode_sort, tmp, ind )
         end do
      else if( dim == 2 ) then
         do i = 1, A%shape(1)
            tmp => va%double(i,:)
            ia%double(i,:) = [ (j,j=1,A%shape(2)) ]
            ind => ia%double(i,:)
            call quick_sort( mode_sort, tmp, ind )
         end do
      else
         call PrintMessage( "msSort", "E",                              &
                            "dim must be equal to 1 or 2!" )
         go to 99
      end if

      va%prop%symm = FALSE
      ia%prop%symm = FALSE

      if( mf_phys_units ) then
         va%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msSort_matrix_out
!_______________________________________________________________________
!
   function mfSortRows_mode( A, col, mode ) result( out )

      type(mfArray)                          :: A
      integer,          intent(in)           :: col
      character(len=*), intent(in), optional :: mode
      type(mfArray)                          :: out
      !------ API end ------
#ifdef _DEVLP

      real(kind=MF_DOUBLE), allocatable :: tmp(:), ind(:)
      integer :: i
      character(len=3) :: mode_sort

      ! sort the first column and apply the sort to other columns

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfSortRows", "E",                          &
                            "real array required!" )
         go to 99
      end if

      if( .not. mfIsMatrix(A) ) then
         call PrintMessage( "mfSortRows", "E",                          &
                            "'A' be a rank-2 mfArray!" )
         go to 99
      end if

      if( col < 1 .or. A%shape(2) < col ) then
         call PrintMessage( "mfSortRows", "E",                          &
                            "'col' is out-of-range!" )
         go to 99
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            mode_sort = "asc"
         else if( to_lower(mode(1:3)) == "des" ) then
            mode_sort = "des"
         else
            call PrintMessage( "mfSortRows", "E",                       &
                               "bad arg value for 'mode' (must be 'asc' or 'des'!" )
            go to 99
         end if
      else
         mode_sort = "asc"
      end if

      allocate( tmp(A%shape(1)) )
      tmp = A%double(:,col)
      allocate( ind(A%shape(1)) )
      ind = [ (i,i=1,A%shape(1)) ]
      call quick_sort( mode_sort, tmp, ind )

      out%data_type = MF_DT_DBLE
      out%shape = A%shape
      allocate( out%double(A%shape(1),A%shape(2)) )

      do i = 1, A%shape(1)
         out%double(i,:) = A%double(nint(ind(i)),:)
      end do

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfSortRows_mode
!_______________________________________________________________________
!
   function mfSortRows_cols( A, cols ) result( out )

      type(mfArray)                      :: A
      integer,      intent(in), optional :: cols(:)
      type(mfArray)                      :: out
      !------ API end ------
#ifdef _DEVLP

      real(kind=MF_DOUBLE), allocatable :: tmp(:,:)
      integer :: i, j, nrow, ncol
      character(len=10) :: str

      ! Sort the rows of A by columns whose numbering is given in the
      ! integer vector 'cols'.
      ! The elements of 'cols' cannot be zero. Positive (resp. negative)
      ! elements in 'cols' indicate an ascending (resp. descending) sort.

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "mfSortRows", "E",                          &
                            "real array required!" )
         go to 99
      end if

      if( .not. mfIsMatrix(A) ) then
         call PrintMessage( "mfSortRows", "E",                          &
                            "'A' be a rank-2 mfArray!" )
         go to 99
      end if

      nrow = size(A,1)
      ncol = size(A,2)

      if( present(cols) ) then
         size_col_ind = size(cols)
         if( size_col_ind < 1 .or. ncol < size_col_ind ) then
            call PrintMessage( "mfSortRows", "E",                       &
                               "bad size for the 'cols' argument!" )
            go to 99
         end if
         ! Check that element in 'cols' have an absolute value in [1,ncol]
         do i = 1, size_col_ind
            j = abs(cols(i))
            if( j < 1 .or. ncol < j ) then
               write(str,"(I0)") i
               call PrintMessage( "mfSortRows", "E",                    &
                                  "bad value for " // trim(str)         &
                                  // "th element of cols!" )
               go to 99
            end if
         end do
         allocate( col_ind(size_col_ind) )
         col_ind(:) = cols(:)
      else
         allocate( col_ind(ncol) )
         col_ind(:) = [ (i, i = 1, ncol) ]
      end if

      allocate( tmp(A%shape(1),A%shape(2)) )
      tmp = A%double(:,:)
      call quick_sortrows( tmp ) ! uses col_ind(:) in internal routines

      deallocate( col_ind )

      out%data_type = MF_DT_DBLE
      out%shape = A%shape
      allocate( out%double(A%shape(1),A%shape(2)) )

      out%double(:,:) = tmp(:,:)

      if( mf_phys_units ) then
         out%units(:) = A%units(:)
      end if

      out%status_temporary = .true.

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end function mfSortRows_cols
!_______________________________________________________________________
!
   subroutine msSortRows_mode( A, col, mode )

      type(mfArray)                          :: A
      integer,          intent(in)           :: col
      character(len=*), intent(in), optional :: mode
      !------ API end ------
#ifdef _DEVLP

      ! in-place version of 'mfSortRows_mode'.
      !
      ! A cannot be tempo. A: real only.

      integer :: dir

      ! sort the first column and apply the sort to other columns

   !------ end of declarations -- execution starts hereafter  ------

      if( A%status_temporary ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "mfArray 'A' cannot be tempo!" )
         return
      end if

      if( mfIsEmpty(A) ) then
         return
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "real array required!" )
         return
      end if

      if( .not. mfIsMatrix(A) ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "'A' be a rank-2 mfArray!" )
         return
      end if

      if( col < 1 .or. A%shape(2) < col ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "'col' is out-of-range!" )
         return
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            dir = 1
         else if( to_lower(mode(1:3)) == "des" ) then
            dir = -1
         else
            call PrintMessage( "msSortRows", "E",                       &
                               "bad arg value for 'mode' (must be 'asc' or 'des'!" )
            return
         end if
      else
         dir = 1
      end if

      allocate( col_ind(1) )
      col_ind = dir*col

      call quick_sortrows( A%double ) ! uses col_ind(:) in internal routines

      deallocate( col_ind )

#endif
   end subroutine msSortRows_mode
!_______________________________________________________________________
!
   subroutine msSortRows_cols( A, cols )

      type(mfArray)                      :: A
      integer,      intent(in), optional :: cols(:)
      !------ API end ------
#ifdef _DEVLP

      integer :: i, j, nrow, ncol
      character(len=10) :: str

      ! Sort the rows of A by columns whose numbering is given in the
      ! integer vector 'cols'.
      ! The elements of 'cols' cannot be zero. Positive (resp. negative)
      ! elements in 'cols' indicate an ascending (resp. descending) sort.

   !------ end of declarations -- execution starts hereafter  ------

      if( A%status_temporary ) then
         call PrintMessage( "msSortRows", "E",                              &
                            "mfArray 'A' cannot be tempo!" )
         return
      end if

      if( mfIsEmpty(A) ) then
         return
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "real array required!" )
         return
      end if

      if( .not. mfIsMatrix(A) ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "'A' be a rank-2 mfArray!" )
         return
      end if

      nrow = size(A,1)
      ncol = size(A,2)

      if( present(cols) ) then
         size_col_ind = size(cols)
         if( size_col_ind < 1 .or. ncol < size_col_ind ) then
            call PrintMessage( "msSortRows", "E",                       &
                               "bad size for the 'cols' argument!" )
            return
         end if
         ! Check that element in 'cols' have an absolute value in [1,ncol]
         do i = 1, size_col_ind
            j = abs(cols(i))
            if( j < 1 .or. ncol < j ) then
               write(str,"(I0)") i
               call PrintMessage( "msSortRows", "E",                    &
                                  "bad value for " // trim(str)         &
                                  // "th element of cols!" )
               return
            end if
         end do
         allocate( col_ind(size_col_ind) )
         col_ind(:) = cols(:)
      else
         allocate( col_ind(ncol) )
         col_ind(:) = [ (i, i = 1, ncol) ]
      end if

      call quick_sortrows( A%double ) ! uses col_ind(:) in internal routines

      deallocate( col_ind )

#endif
   end subroutine msSortRows_cols
!_______________________________________________________________________
!
   subroutine msSortRows_mode_out( out, A, col, mode )

      type(mfArray)                          :: A
      integer,          intent(in)           :: col
      character(len=*), intent(in), optional :: mode
      type(mf_Out)                           :: out
      !------ API end ------
#ifdef _DEVLP

      ! in-place version of 'mfSortRows_mode'.
      !
      ! A cannot be tempo. A: real only.

      type(mfArray), pointer :: va, ia

      integer :: i, dir

      ! sort the first column and apply the sort to other columns

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! we must have two output arguments
      if( out%n /= 2 ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "two output args required!",                &
                            "syntax is : call msSort ( mfOut(v,i), a )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      va => out%ptr1
      ia => out%ptr2
!### Warning : what happens if 'va' and 'A' share the same address ?
! (for example, a destructive sort can be chosen!)
! is code below ok ?
      if( a%data_type == MF_DT_DBLE ) then
         if( same_address(va,a) == 0 ) then
            call msSilentRelease( va )
         end if
      else
         call msSilentRelease( va )
      end if
      call msSilentRelease( ia )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "real array required!" )
         go to 99
      end if

      if( .not. mfIsMatrix(A) ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "'A' be a rank-2 mfArray!" )
         go to 99
      end if

      if( col < 1 .or. A%shape(2) < col ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "'col' is out-of-range!" )
         go to 99
      end if

      if( present(mode) ) then
         if( to_lower(mode(1:3)) == "asc" ) then
            dir = 1
         else if( to_lower(mode(1:3)) == "des" ) then
            dir = -1
         else
            call PrintMessage( "msSortRows", "E",                       &
                               "bad arg value for 'mode' (must be 'asc' or 'des'!" )
            go to 99
         end if
      else
         dir = 1
      end if

      allocate( col_ind(1) )
      col_ind = dir*col

      va%data_type = MF_DT_DBLE
      va%shape = A%shape
      allocate( va%double(A%shape(1),A%shape(2)) )
      va%double = A%double

      ia%data_type = MF_DT_DBLE
      ia%shape = [ A%shape(1), 1 ]
      allocate( ia%double(A%shape(1),1) )
      ia%double(:,1) = [ (i,i=1,A%shape(1)) ]

      call quick_sortrows( va%double, ia%double(:,1) ) ! uses col_ind(:) in internal routines

      deallocate( col_ind )

      if( mf_phys_units ) then
         va%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msSortRows_mode_out
!_______________________________________________________________________
!
   subroutine msSortRows_cols_out( out, A, cols )

      type(mfArray)                      :: A
      integer,      intent(in), optional :: cols(:)
      type(mf_Out)                       :: out
      !------ API end ------
#ifdef _DEVLP

      ! in-place version of 'mfSortRows_mode'.
      !
      ! A cannot be tempo. A: real only.

      type(mfArray), pointer :: va, ia

      integer :: i, j, nrow, ncol
      character(len=10) :: str

      ! sort the first column and apply the sort to other columns

   !------ end of declarations -- execution starts hereafter  ------

      call msInitArgs( A )

      ! we must have two output arguments
      if( out%n /= 2 ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "two output args required!",                &
                            "syntax is : call msSort ( mfOut(v,i), a )" )
         go to 99
      end if

      ! internal check for all mfOut args
      if( .not. args_mfout_ok( out, a ) ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "output arguments cannot be tempo, or cannot share",&
                            "same memory as another input argument." )
         go to 99
      end if

      va => out%ptr1
      ia => out%ptr2
!### Warning : what happens if 'va' and 'A' share the same address ?
! (for example, a destructive sort can be chosen!)
! is code below ok ?
      if( a%data_type == MF_DT_DBLE ) then
         if( same_address(va,a) == 0 ) then
            call msSilentRelease( va )
         end if
      else
         call msSilentRelease( va )
      end if
      call msSilentRelease( ia )

      if( mfIsEmpty(A) ) then
         go to 99
      end if

      if( A%data_type /= MF_DT_DBLE ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "real array required!" )
         go to 99
      end if

      if( .not. mfIsMatrix(A) ) then
         call PrintMessage( "msSortRows", "E",                          &
                            "'A' be a rank-2 mfArray!" )
         go to 99
      end if

      nrow = size(A,1)
      ncol = size(A,2)

      if( present(cols) ) then
         size_col_ind = size(cols)
         if( size_col_ind < 1 .or. ncol < size_col_ind ) then
            call PrintMessage( "msSortRows", "E",                       &
                               "bad size for the 'cols' argument!" )
            return
         end if
         ! Check that element in 'cols' have an absolute value in [1,ncol]
         do i = 1, size_col_ind
            j = abs(cols(i))
            if( j < 1 .or. ncol < j ) then
               write(str,"(I0)") i
               call PrintMessage( "msSortRows", "E",                    &
                                  "bad value for " // trim(str)         &
                                  // "th element of cols!" )
               return
            end if
         end do
         allocate( col_ind(size_col_ind) )
         col_ind(:) = cols(:)
      else
         allocate( col_ind(ncol) )
         col_ind(:) = [ (i, i = 1, ncol) ]
      end if

      va%data_type = MF_DT_DBLE
      va%shape = A%shape
      allocate( va%double(A%shape(1),A%shape(2)) )
      va%double = A%double

      ia%data_type = MF_DT_DBLE
      ia%shape = [ A%shape(1), 1 ]
      allocate( ia%double(A%shape(1),1) )
      ia%double(:,1) = [ (i,i=1,A%shape(1)) ]

      call quick_sortrows( va%double, ia%double(:,1) ) ! uses col_ind(:) in internal routines

      deallocate( col_ind )

      if( mf_phys_units ) then
         va%units(:) = A%units(:)
      end if

 99   continue

      call msFreeArgs( A )

      call msAutoRelease( A )

#endif
   end subroutine msSortRows_cols_out
